/*
 * @parseRockColumnSymbolsUtility.java Version 1.1 08/02/2012
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseRockColumnSymbolsListStruct;
import parse.parseRockColumnSymbolsStruct;

import lith.lithology.lithologyColorsStruct;
import lith.lithology.lithologyGroupsStruct;

import iqstrat.iqstratRemarkListStruct;

import lith.rock.rockColumnListStruct;
import lith.rock.rockColumnStruct;

/** Class parseRockColumnSymbolsUtility
 *  <p> This Class will provide basic utilities for the rock column parser
 *      data structures.
 *
 *  @version 1.1 08/02/2012
 *  @author  John R. Victorine
 */

public class parseRockColumnSymbolsUtility
{
  public static final int _NONE      = lith.rock.rockColumnStruct._NONE;
  public static final int _TRACE     = lith.rock.rockColumnStruct._TRACE;
  public static final int _SLIGHTLY  = lith.rock.rockColumnStruct._SLIGHTLY;
  public static final int _NORMAL    = lith.rock.rockColumnStruct._NORMAL;
  public static final int _MANY      = lith.rock.rockColumnStruct._MANY;
  public static final int _VERY      = lith.rock.rockColumnStruct._VERY;

  public static final String AMOUNT[] = lith.rock.rockColumnStruct.AMOUNT;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /*
   *  <p> This method will return the color number identifier from the
   *      character that is passed in.
   * @param  c    = Character
   * @param  st   = lithology colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( char c, lithologyColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.colors != null)
      {
        for (int i=0; i<st.colors.length; i++)
        {
          if (c == st.colors[i]) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getColor()
   *  <p> This method will return the color number identifier from the
   *      color name that is passed in.
   * @param  str  = Color Name
   * @param  st   = lithology colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( String str, lithologyColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.sColors != null)
      {
        for (int i=1; i<st.sColors.length; i++)
        {
          if (str.equals(st.sColors[i])) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getRGB()
   * <p> This method will return the RGB Array of colors
   * @param  icolor = The lithology color to return
   * @param  st     = lithology colors data structure
   * @return RGB    = Array of Red Green Blue Color numbers
   */

  public static int[] getRGB( int icolor, lithologyColorsStruct st  )
  {
    int RGB[] = { 255, 255, 255 };

    if (icolor != -1)
    {
      if (st != null)
      {
        if (st.iRGB != null)
        {
          if (icolor < st.iRGB.length)
          {
            for (int j=0; j<3; j++)
              RGB[j] = st.iRGB[icolor][j];
          }
        }
      }
    }

    return ( RGB );
  }

  /** Method getGroupID( int iColumn, String str, lithologyGroupsStruct st )
   *  <p> This method will return the ID number of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  str     = String to comparison string
   * @param  st      = lithology group data structure
   * @return iGroup  = Group id
   */

  public static int getGroupID( int iColumn,
                                String str,
                                lithologyGroupsStruct st )
  {
    int iGroup = -1;

    if (st != null)
    {
      for (int i=0; i<st.iGroups; i++)
      {
        if (st.sGroups[i][iColumn].equals(str))
          iGroup = i;
      }
    }

    return (iGroup);
  }

  /** Method getGroup( int iColumn, String str, lithologyGroupsStruct st )
   *  <p> This method will return the string of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  iGroup  = Group ID
   * @param  st      = lithology group data structure
   * @return sGroup  = Group String
   */

  public static String getGroup( int iColumn,
                                 int iGroup,
                                 lithologyGroupsStruct st )
  {
    String sGroup = "";

    if (st != null)
    {
      if (iGroup > -1)
      {
        sGroup = new String(st.sGroups[iGroup][iColumn]);
      }
    }

    return (sGroup);
  }

  /** Method getColor()
   * <p> This method will retrieve the color for a specific lithology
   * @param  iSymbol = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return iRGB    = array of RGB Colors to color image
   */

  public static int[] getColor( int iSymbol, parseRockColumnSymbolsListStruct st )
  {
    int iRGB[] = { 255, 255, 255 };

    if (st != null)
    {
      if (iSymbol > -1)
      {
        for (int j=0; j<3; j++)
          iRGB[j] = st.stItem[iSymbol].iRGB[j];
      }
    }

    return (iRGB);
  }

  /** Method getColor()
   * <p> This method will retrieve the color for a specific lithology
   * @param  str  = Color Name
   * @param  st   = Lithology Symbols List Data Structure
   * @return iRGB = array of RGB Colors to color image
   */

  public static int[] getColor( String str, parseRockColumnSymbolsListStruct st )
  {
    int iRGB[] = { 255, 255, 255 };

    if (st != null)
    {
	  for (int i=0; i<st.iCount; i++)
	  {
        if (str.equals(st.stItem[i].sID))
        {
          for (int j=0; j<3; j++)
            iRGB[j] = st.stItem[i].iRGB[j];
        }
	  }
    }

    return (iRGB);
  }

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = Texture Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parseRockColumnSymbolsListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseRockColumnSymbolsListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getSymbol()
   * <p> This method will return the symbol
   * @param  sID     = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static String[][] getSymbol(String sID, parseRockColumnSymbolsListStruct st)
  {
	int    i,j;
	int    iSymbol    = -1;
    String symbol[][] = null;

    if (st != null)
    {
	  for (i=0; i<st.iCount; i++)
	  {
		if (sID.equals(st.stItem[i].sID))
		  iSymbol = i;
	  }

	  if (iSymbol > -1)
	  {
        if (st.stItem[iSymbol].symbol != null)
        {
          symbol = new String[st.stItem[iSymbol].symbol.length][2];
          for (i=0; i<st.stItem[iSymbol].symbol.length; i++)
          {
            for (j=0; j<2; j++)
              symbol[i][j] = st.stItem[iSymbol].symbol[i][j];
		  }
        }
      }
    }

    return (symbol);
  }

  /** Method getCaps()
   * <p> This method will return the symbol
   * @param  sID     = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static String[][] getCaps(String sID, parseRockColumnSymbolsListStruct st)
  {
	int    i,j;
	int    iSymbol    = -1;
    String symbol[][] = null;

    if (st != null)
    {
	  for (i=0; i<st.iCount; i++)
	  {
		if (sID.equals(st.stItem[i].sID))
		  iSymbol = i;
	  }

	  if (iSymbol > -1)
	  {
        if (st.stItem[iSymbol].caps != null)
        {
          symbol = new String[st.stItem[iSymbol].caps.length][3];
          for (i=0; i<st.stItem[iSymbol].caps.length; i++)
          {
            for (j=0; j<st.stItem[iSymbol].iCaps; j++)
              symbol[i][j] = "";
		  }

          for (i=0; i<st.stItem[iSymbol].caps.length; i++)
          {
            for (j=0; j<st.stItem[iSymbol].iCaps; j++)
              symbol[i][j] = st.stItem[iSymbol].caps[i][j];
		  }
        }
      }
    }

    return (symbol);
  }

  /** Method getCapType()
   * <p> This method will return the symbol
   * @param  sID     = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static int[] getCapType(String sID, parseRockColumnSymbolsListStruct st)
  {
	int    i,j;
	int    iSymbol  = -1;
    int    iTypes[] = null;

    if (st != null)
    {
	  for (i=0; i<st.iCount; i++)
	  {
		if (sID.equals(st.stItem[i].sID))
		  iSymbol = i;
	  }

	  if (iSymbol > -1)
	  {
        if (st.stItem[iSymbol].iCaps > 0)
        {
          iTypes = new int[3];
          for (j=0; j<3; j++)
            iTypes[j] = parse.parseRockColumnSymbolsStruct._NONE;

          for (j=0; j<st.stItem[iSymbol].iCaps; j++)
            iTypes[j] = st.stItem[iSymbol].iTypes[j];
        }
      }
    }

    return (iTypes);
  }

  /** Method getLithology()
   * <p> This method will check the Lithology key words, phrases to see if any
   *     matches the default lithology key words or phrases and return the row
   *     id if it does.
   *
   * @param  str = the string passed in.
   * @param  st  = Lithology Symbols List Data Structure
   * @return sID = the lithology identifier.
   */

  public static String getLithology( String str, parseRockColumnSymbolsListStruct st )
  {
    String  sID   = "";
    String  sWord = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int j=0; j<st.stItem[i].iThesaurus; j++)
        {
          if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
            sID = new String( st.stItem[i].sID );
        }
      }
    }

    return (sID);
  }

  /** Method getLithology()
   * <p> This method will check the Lithology key words, phrases to see if any
   *     matches the default lithology key words or phrases and return the row
   *     id if it does.
   *
   * @param  str   = the string passed in.
   * @param  sType = the type of data
   * @param  st    = Lithology Symbols List Data Structure
   * @return sID   = the lithology identifier.
   */

  public static String getLithology( String str, String sType, parseRockColumnSymbolsListStruct st )
  {
    String  sID   = "";
    String  sWord = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int j=0; j<st.stItem[i].iThesaurus; j++)
        {
          if ((sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase())) &&
              (sType.equals(st.stItem[i].sType.toLowerCase())))
          {
            sID = new String( st.stItem[i].sID );
		  }
        }
      }
    }

    return (sID);
  }

  /** Method getID()
   * <p> This method will retrieve the ID order number within the list of lithologies
   *
   * @param  sID = the string passed in.
   * @param  st  = Lithology Symbols List Data Structure
   * @return id  = the lithology order number.
   */

  public static int getID( String sID, parseRockColumnSymbolsListStruct st )
  {
    int    id = -1;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if (sID.equals(st.stItem[i].sID))
          id = i;
      }
    }

    return (id);
  }

  /* ======================================================================== *
   * --------------------------- PARSE METHODS ------------------------------ *
   * ======================================================================== */

  /** Method parse()
   * <p> This method will parse the Remarks string for Rock Column Lithology.
   * @param  stList     = Remarks List Data Structure
   * @param  stSymbols  = Parse Rock Column Lithology Symbols Data List Structure
   * @return stTexture  = Rock Column Data List Structure
   */

  public static rockColumnListStruct parse(
	              iqstratRemarkListStruct          stList,
                  parseRockColumnSymbolsListStruct stSymbols)
  {
    rockColumnListStruct stColumn   = null;
    String               sKEY       = cmn.cmnString.UniqueName();
    String               str        = "";
    String               tokens[]   = null;
    String               sDelimiter = new String("[ /,;:.=-]+");
    rockColumnStruct     stText     = null;
    double               diff       = 1000.0;
    double               dtemp      = 0.0;
    int                  icheck     = 1;
    String               sGrains[]  = null;

    if (stList != null)
    {
	  stColumn = new rockColumnListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
		if (stList.stItem[i].depthEnd > stList.stItem[i].depthStart)
		{
		  icheck=0;
		}
	  }

      if (icheck==1)
      {
        for (int i=0; i<stList.iCount-1; i++)
        {
		  dtemp = stList.stItem[i+1].depthStart - stList.stItem[i].depthStart;

		  if ((dtemp < diff) && (dtemp > 0.0))
		  {
			diff = dtemp;
		  }
		}

        for (int i=0; i<stList.iCount-1; i++)
        {
		  dtemp = stList.stItem[i+1].depthStart - stList.stItem[i].depthStart;

		  if (dtemp > diff)
		    stList.stItem[i].depthEnd = stList.stItem[i].depthStart + diff;
		  else
		    stList.stItem[i].depthEnd = stList.stItem[i].depthStart + dtemp;
		}

		stList.stItem[stList.iCount-1].depthEnd = stList.stItem[stList.iCount-1].depthStart + diff;
	  }

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        // Copy the List of Terms to a another Array of Words for parsing out grain size
        // and textures.

        if (tokens != null)
        {
		  if (tokens.length > 0)
		  {
			sGrains = new String[tokens.length];
			for (int j=0; j<tokens.length; j++)
			{
			  sGrains[j] = new String(tokens[j]);
			}
		  }
		}

        stText = parseColumn( stList.stItem[i].depthStart,
                              stList.stItem[i].depthEnd,
                              tokens, stSymbols );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stColumn    = lith.rock.rockColumnUtility.add( stText, stColumn );
        }

        tokens  = null;
        sGrains = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
	}

    return (stColumn);
  }

  /** Method parse()
   * <p> This method will parse the Remarks string for Rock Column Lithology.
   * @param  stList     = Remarks List Data Structure
   * @param  stSymbols  = Parse Rock Column Lithology Symbols Data List Structure
   * @return stTexture  = Rock Column Data List Structure
   */

  public static rockColumnListStruct parse(
	              iqstratRemarkListStruct          stList,
                  parseRockColumnSymbolsListStruct stSymbols,
                  parseGrainsTextureListStruct     stGrains )
  {
    rockColumnListStruct stColumn   = null;
    String               sKEY       = cmn.cmnString.UniqueName();
    String               str        = "";
    String               tokens[]   = null;
    String               sDelimiter = new String("[ /,;:.=-]+");
    rockColumnStruct     stText     = null;
    double               diff       = 1000.0;
    double               dtemp      = 0.0;
    int                  icheck     = 1;
    String               sGrains[]  = null;
    String               sTXTR[]    = null;

    if (stList != null)
    {
	  stColumn = new rockColumnListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
		if (stList.stItem[i].depthEnd > stList.stItem[i].depthStart)
		{
		  icheck=0;
		}
	  }

      if (icheck==1)
      {
        for (int i=0; i<stList.iCount-1; i++)
        {
		  dtemp = stList.stItem[i+1].depthStart - stList.stItem[i].depthStart;

		  if ((dtemp < diff) && (dtemp > 0.0))
		  {
			diff = dtemp;
		  }
		}

        for (int i=0; i<stList.iCount-1; i++)
        {
		  dtemp = stList.stItem[i+1].depthStart - stList.stItem[i].depthStart;

		  if (dtemp > diff)
		    stList.stItem[i].depthEnd = stList.stItem[i].depthStart + diff;
		  else
		    stList.stItem[i].depthEnd = stList.stItem[i].depthStart + dtemp;
		}

		stList.stItem[stList.iCount-1].depthEnd = stList.stItem[stList.iCount-1].depthStart + diff;
	  }

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        // Copy the List of Terms to a another Array of Words for parsing out grain size
        // and textures.

        if (tokens != null)
        {
		  if (tokens.length > 0)
		  {
			sGrains = new String[tokens.length];
			for (int j=0; j<tokens.length; j++)
			{
			  sGrains[j] = new String(tokens[j]);
			}
		  }
		}

        if (tokens != null)
        {
		  if (tokens.length > 0)
		  {
			sTXTR = new String[tokens.length];
			for (int j=0; j<tokens.length; j++)
			{
			  sTXTR[j] = new String(tokens[j]);
			}
		  }
		}

        stText = parseColumn( stList.stItem[i].depthStart,
                              stList.stItem[i].depthEnd,
                              tokens, stSymbols );

        stText = parse.parseGrainsTextureUtility.parseColumn( stText, sGrains, stGrains );

        stText = parse.parseGrainsTextureUtility.getTexture( stText, sTXTR, stGrains );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stColumn    = lith.rock.rockColumnUtility.add( stText, stColumn );
        }

        tokens  = null;
        sGrains = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
	}
//lith.rock.rockColumnUtility.print(stColumn);

    return (stColumn);
  }

  /** Method parseColumn()
   * <p> This method will parse the text string into rock column lithology
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  sData      = array of rock column string phrase or word
   * @param  stSymbols  = Rock Column Lithology Symbols Data List Structure
   * @return st         = rock texture data structure
   */

  public static  rockColumnStruct parseColumn(
                     double depthStart, double depthEnd, String sData[],
                     parseRockColumnSymbolsListStruct stSymbols )
  {
    int i, j, k, m, n, p;
    parseRockColumnSymbolsStruct stRock    = null;
    rockColumnStruct st         = null;
    int              iContinue  = 0;
    int              iNext      = 0;
    int              iCount     = 0;
    int              iTotal     = 30;
    String           sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                    {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                    {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                    {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                    {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                    {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int              iRow[]     = { -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,
                                    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1 };
    String           sTemp[]    = { "", "" };
    String           sWord      = "";

    String           sPrimary      = "";
    String           sCycle        = "";   // sedimentary, igneous, etc
    String           sCycle1       = "";   // sedimentary, igneous, etc
    int              iSecondary    = 0;    // Total Number of Secondary Lithology
    int              iAmount[]     = null; // Amount of Secondary Lithology using above identifiers.
    String           secondary[]   = null; // Array of Secondary Lithology ID's
//    int              iDescriptor   = 0;    // Total Number of Descriptors
//    int              iDAmount[]    = null; // Amount of descriptor using above identifiers.
//    String           sDescriptor[] = null; // Array of Descriptors ID's
    String           stmp          = "";
    int              id            = 0;

    if (stRock != null)
      stRock.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stSymbols.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stSymbols.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stSymbols.sAbbrev[j][1]);
      }
    }

    // Find Rock Column Words up to 30

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stSymbols ))
      {
        if (iCount < iTotal)
        {
          sTemp = getKeyword( sData[j], stSymbols );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iRow[iCount]      = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("B")) || (sWords[i][1].equals("L")))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stSymbols ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
	}

/*
    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("M")) || (sWords[i][1].equals("Mt")))
      {
        k = i+1;
        m = i+2;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stSymbols ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");

            if (m < iCount)
            {
		      if (sWords[m][1].equals("T"))
		      {
                sWords[i][0] = new String(sWords[i][0] + " " + sWords[m][0]);
                sWords[i][1] = new String(sWords[m][1]);

                sWords[m][0] = new String("");
                sWords[m][1] = new String("");
			  }
			}
          }
        }
      }

      if (sWords[i][1].equals("Da"))
      {
	    if ((i > 0) && (i<iCount-1))
	 	{
          k = i-1;
          m = i+1;

		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")) &&
		      (!sWords[k][1].equals("B")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i == 0)
		{
          m = i+1;
		  if ((!sWords[m][1].equals("L")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i==iCount-1)
		{
          k = i-1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[k][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
      }

      if (sWords[i][1].equals("B"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stSymbols ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }
*/
    // Identify the main lithology type

    iSecondary  = 0;             // Total Number of Secondary Lithology
    iAmount     = new int[5];    // Amount of Secondary Lithology using above identifiers.
    secondary   = new String[5]; // Array of Secondary Lithology ID's
/*
    iDescriptor = 0;             // Total Number of Descriptors
    iDAmount    = new int[5];    // Amount of descriptor using above identifiers.
    sDescriptor = new String[5]; // Array of Descriptors ID's
*/
    for (i=0; i<5; i++)
    {
//	  iAmount[i]    = _NORMAL;
	  iAmount[i]    = 0;
	  secondary[i]   = "";

//	  iDAmount[i]    = _NORMAL;
//	  sDescriptor[i] = "";
	}

    for (i=0; i<iCount; i++)
    {
	  iNext = 0;
      if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")))
      {
		if (sPrimary.length() == 0)
	    {
		  if (isThesarus( sWords[i][0], "1st", stSymbols ))
		  {
            sPrimary = new String( getLithology( sWords[i][0], "1st", stSymbols ) );
            iContinue = 1;
		  }
		  else if (isThesarus( sWords[i][0], "both", stSymbols ))
		  {
            sPrimary = new String( getLithology( sWords[i][0], "both", stSymbols ) );
            iContinue = 1;
		  }

		  if (iContinue == 1)
		  {
			id = getID( sPrimary, stSymbols );
			if (id > 0) { sCycle = new String( stSymbols.stItem[id].sCycle ); }
		  }

		}
		else if (sCycle.equals("sed"))
		{
		  if (isThesarus( sWords[i][0], "2nd", stSymbols ))
		    stmp = new String( getLithology( sWords[i][0], "2nd", stSymbols ) );
		  else if (isThesarus( sWords[i][0], "both", stSymbols ))
		    stmp = new String( getLithology( sWords[i][0], "both", stSymbols ) );

          if (stmp.length() > 2)
          {
			id = getID( stmp, stSymbols );
			if (id > 0) { sCycle1 = new String( stSymbols.stItem[id].sCycle ); }

			if (sCycle1.equals("sed"))
			{
              if (sPrimary.length() > 2)
              {
		        if (sPrimary.substring(0,3).equals(stmp.substring(0,3)))
		        {
		          iNext++;
		        }
		        else if (sPrimary.equals("CO3-LS_DOL"))
		        {
  		          if (sPrimary.substring(7,10).equals(stmp.substring(0,3)))
		          {
		            iNext++;
		          }
				}
		      }

		      if (iSecondary > 0)
		      {
		        for (n=0; n<iSecondary; n++)
		        {
		          if (secondary[n].substring(0,3).equals(stmp.substring(0,3)))
			      {
			        iNext++;
			      }
		          else if (sPrimary.equals("CO3-LS_DOL"))
		          {
  		            if (sPrimary.substring(7,10).equals(stmp.substring(0,3)))
		            {
		              iNext++;
		            }
			  	  }
			    }
		      }
		    }

  		    if (iNext == 0)
            {
		      if (iSecondary < 4)
			  {
			    id = getID( stmp, stSymbols );
			    if (id > 0) { iAmount[iSecondary] = stSymbols.stItem[id].iOrder; }
			    secondary[iSecondary] = new String( stmp );
			    iSecondary++;
		      }
	        }
          }

  		  stmp = new String("");
/*
		  if (isThesarus( sWords[i][0], "2nd", stSymbols ))
		    sWord = new String( getLithology( sWords[i][0], "2nd", stSymbols ) );
		  else if (isThesarus( sWords[i][0], "both", stSymbols ))
		    sWord = new String( getLithology( sWords[i][0], "both", stSymbols ) );

          if ((sPrimary.length() > 2) && (sWord.length() > 2))
          {
		    if (!sWord.substring(0,3).equals(sPrimary.substring(0,3)))
		    {
		      if (iSecondary < 4)
			  {
				id = getID( sWord, stSymbols );
				if (id > 0) { iAmount[iSecondary] = stSymbols.stItem[id].iOrder; }
			    secondary[iSecondary] = new String( sWord );
			    iSecondary++;
		  	  }
		    }
	      }
*/
		}
	  }
	}

    if ((iContinue == 1) && (sCycle.equals("sed")))
    {
	  for (i=0; i<iCount; i++)
	  {
        iNext = 0;
	    if (sWords[i][1].equals("D"))
	    {
		  if (isThesarus( sWords[i][0], "2nd", stSymbols ))
		    stmp = new String( getLithology( sWords[i][0], "2nd", stSymbols ) );
		  else if (isThesarus( sWords[i][0], "both", stSymbols ))
		    stmp = new String( getLithology( sWords[i][0], "both", stSymbols ) );

          if (stmp.length() > 2)
          {
			id = getID( stmp, stSymbols );
			if (id > 0) { sCycle1 = new String( stSymbols.stItem[id].sCycle ); }

			if (sCycle1.equals("sed"))
			{
              if (sPrimary.length() > 2)
              {
		        if (sPrimary.substring(0,3).equals(stmp.substring(0,3)))
		        {
		          iNext++;
		        }
		      }

		      if (iSecondary > 0)
		      {
		        for (n=0; n<iSecondary; n++)
		        {
		          if (secondary[n].substring(0,3).equals(stmp.substring(0,3)))
			      {
			        iNext++;
			      }
			    }
		      }
			}

  		    if (iNext == 0)
            {
		      if (iSecondary < 4)
			  {
				id = getID( stmp, stSymbols );
				if (id > 0) { iAmount[iSecondary] = stSymbols.stItem[id].iOrder; }
			    secondary[iSecondary] = new String( stmp );
		  	    iSecondary++;
		      }
	        }

		  }
  		  stmp = new String("");
/*
		  if (isThesarus( sWords[i][0], stSymbols ))
		  {
			stmp = new String( getLithology( sWords[i][0], stSymbols ) );


            if (sPrimary.length() > 2)
            {
			  if (sPrimary.substring(0,3).equals(stmp.substring(0,3)))
			  {
			    iNext++;
			  }
			}

		    if (iSecondary > 0)
			{
			  for (n=0; n<iSecondary; n++)
			  {
				if (secondary[n].substring(0,3).equals(stmp.substring(0,3)))
				{
				  iNext++;
				}
			  }
			}

			if (iNext == 0)
			{
			  if (iDescriptor < 2)
			  {
                sDescriptor[iDescriptor] = new String( stmp );
                iDescriptor++;
			  }
			}
			stmp = new String("");
		  }
*/
		}
  	  }
	}

    if (iContinue == 1)
    {
      st = new rockColumnStruct();

      st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                  ((int)depthStart)); // Create a Unique KEY
      st.depthStart = depthStart; // Starting Depth
      st.depthEnd   = depthEnd;   // Ending Depth

      st.iRGB        = getColor( sPrimary, stSymbols );

      st.sPrimary    = new String( sPrimary ); // Primary Lithology Identifier

//System.out.println(st.sKEY+" "+st.depthStart+" "+st.depthEnd+" "+st.sPrimary);

      st.iSecondary  = iSecondary;    // Total Number of Secondary Lithology
      if (st.iSecondary > 0)
      {
        st.iAmount   = new int[st.iSecondary];    // Amount of Secondary Lithology
        st.secondary = new String[st.iSecondary]; // Array of Secondary Lithology ID's

        secondary = bubbleSort( iSecondary, iAmount, secondary ); // Sort by symbol mask id

        for (i=0; i<st.iSecondary; i++)
        {
//  		  st.iAmount[i]   = iAmount[i];
		  st.secondary[i] = new String( secondary[i] );
	    }
	  }
/*
      st.iDescriptor   = iDescriptor; // Total Number of Descriptors
      if (st.iDescriptor > 0)
      {
        st.iDAmount    = new int[st.iDescriptor];    // Amount of descriptor.
        st.sDescriptor = new String[st.iDescriptor]; // Array of Descriptors ID's

        for (i=0; i<st.iDescriptor; i++)
        {
          st.iDAmount[i]    = iDAmount[i];               // Amount of descriptor.
          st.sDescriptor[i] = new String( sDescriptor[i] ); // Array of Descriptors ID's
	    }
	  }
*/
    }

    return (st);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  iRows  = The total number of lithologies
   * @param  iOrder = the order of plot
   * @param  sID    = the lithology ID
   * @return st     = sorted data list structure
   */

  public static String[] bubbleSort( int iRows, int iOrder[], String sID[] )
  {
    boolean swappedOnPrevRun = true;
    int     iTemp = 0;
    String  sTemp = "";

    if (sID != null)
    {
      if (iRows > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<iRows-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(iOrder[i] > iOrder[i+1])
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              sTemp = new String(sID[i]);
              iTemp = iOrder[i];

              // set element i+1 to where i used to be

              sID[i]    = new String(sID[i+1]);
              iOrder[i] = iOrder[i+1];

              // release the old i from temp into i+1 slot

              sID[i+1]    = new String(sTemp);
              iOrder[i+1] = iTemp;
            }
          }
        }
      }
    }

    return (sID);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isThesarus()
   * <p> This method will determine if the string comes from the Thesaurus Terms
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isThesarus( String str, parseRockColumnSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
		for (int j=0; j<st.stItem[i].iThesaurus; j++)
		{
          if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
            bValue = true;
	    }
      }
    }

    return (bValue);
  }

  /** Method isThesarus()
   * <p> This method will determine if the string comes from the Thesaurus Terms
   * @param  str    = string or string phrase being compared.
   * @param  sType  = Type of lithology
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isThesarus( String str, String sType, parseRockColumnSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
		for (int j=0; j<st.stItem[i].iThesaurus; j++)
		{
          if ((sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase())) &&
              (sType.equals(st.stItem[i].sType.toLowerCase())))
          {
            bValue = true;
		  }
	    }
      }
    }

    return (bValue);
  }


  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseRockColumnSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified()
   *  <p> This method will determine if the string is part of descriptor set
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isModified( String str, parseRockColumnSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if (sWord.equals(st.sModifier[i][1].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified()
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseRockColumnSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseRockColumnSymbolsListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseRockColumnSymbolsListStruct copyList(
                                             parseRockColumnSymbolsListStruct stOld )
  {
    parseRockColumnSymbolsListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseRockColumnSymbolsListStruct();

      // Lithology Colors

      stNew.stForeground = copyColor( stOld.stForeground ); // foreground colors
      stNew.stBackground = copyColor( stOld.stBackground ); // background colors

      // Lithology Groups

      stNew.stGroups     = copyGroup( stOld.stGroups );     // Lithology Groups

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Lithology Symbols

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseRockColumnSymbolsStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseRockColumnSymbolsStruct copy(parseRockColumnSymbolsStruct stOld)
  {
    parseRockColumnSymbolsStruct stNew = new parseRockColumnSymbolsStruct();
    int i=0;
    int j=0;

    stNew.sID      = new String( stOld.sID );    // ID for Lithology Symbol
    stNew.iGroup   = stOld.iGroup;               // Group Number
    stNew.sGroup   = new String( stOld.sGroup ); // The group name
    stNew.sName    = new String( stOld.sName );  // Name
    stNew.sAbrev   = new String( stOld.sAbrev ); // Abbreviation

    stNew.sType    = new String( stOld.sType );  // 1st (primary) lithology, etc.
    stNew.sCycle   = new String( stOld.sCycle ); // sed (sedimentary), etc.
    stNew.iOrder   = stOld.iOrder;               // order of symbol to be plotted
    stNew.iWeather = stOld.iWeather;             // # times horizontal

    // Background color

    for (i=0; i<3; i++)
      stNew.iRGB[i]   = stOld.iRGB[i];

    // Plot Symbol for lithology default is an empty or None symbol

    stNew.iRows      = stOld.iRows;
    stNew.iColumns   = stOld.iColumns;
    if (stNew.iRows > 0)
    {
      stNew.symbol     = new String[stNew.iRows][2];
      for (i=0; i<stNew.iRows; i++)
      {
        for (j=0; j<2; j++)
        {
          stNew.symbol[i][j] = new String( stOld.symbol[i][j] );
        }
      }
    }

    // Plot Cap for lithology default is an empty or None symbol

    stNew.iCapRows  = stOld.iCapRows;
    stNew.iCaps     = stOld.iCaps;

    if (stNew.iCapRows > 0)
    {
	  stNew.iTypes = new int[3];

      for (j=0; j<3; j++)
      {
        stNew.iTypes[j] = stOld.iTypes[j];
      }

      stNew.caps     = new String[stNew.iRows][3];
      for (i=0; i<stNew.iCapRows; i++)
      {
        for (j=0; j<3; j++)
        {
          stNew.caps[i][j] = new String( stOld.caps[i][j] );
        }
      }
    }

    // Thesaurus for searching and parsing geologist field notes or comments

    stNew.iThesaurus   = stOld.iThesaurus;
    if (stNew.iThesaurus > 0)
    {
      stNew.sThesaurus   = new String[stNew.iThesaurus];
      stNew.iType        = new int[stNew.iThesaurus];
      for (i=0; i<stNew.iThesaurus; i++)
      {
        stNew.sThesaurus[i] = new String( stOld.sThesaurus[i] );
        stNew.iType[i]      = stOld.iType[i];
	  }
    }

    return (stNew);
  }

  /** Method copyGroup( lithologyGroupsStruct stOld )
   * <p> This method will copy the group data structure
   * @param  stOld = The old group data structure
   * @return stNew = The new group data structure
   */

  public static lithologyGroupsStruct copyGroup( lithologyGroupsStruct stOld )
  {
    lithologyGroupsStruct stNew = new lithologyGroupsStruct();

    if (stOld != null)
    {
      stNew.iGroups = stOld.iGroups;
      stNew.sGroups = new String[stNew.iGroups][2];

      for (int i=0; i<stNew.iGroups; i++)
      {
        for (int j=0; j<2; j++)
        {
          stNew.sGroups[i][j] = new String( stOld.sGroups[i][j] );
        }
      }
    }

    return (stNew);
  }

  /** Method copyColor( lithologyGroupsStruct stOld )
   * <p> This method will copy the color data structure
   * @param  stOld = The old color data structure
   * @return stNew = The new color data structure
   */

  public static lithologyColorsStruct copyColor( lithologyColorsStruct stOld )
  {
    lithologyColorsStruct stNew = new lithologyColorsStruct();
    int len = 0;
    int i   = 0;
    int j   = 0;

    if (stOld != null)
    {
      stNew.iColors   = stOld.iColors;    // total number of records

      if (stOld.colors != null)
      {
        len = stOld.colors.length;
        stNew.colors = new char[len]; // Character Symbol

        for (i=0; i<len; i++)
          stNew.colors[i] = stOld.colors[i];
      }

      if (stOld.sColors != null)
      {
        len = stOld.sColors.length;
        stNew.sColors = new String[len]; // Color description

        for (i=0; i<len; i++)
          stNew.sColors[i] = new String( stOld.sColors[i] );
      }

      if (stOld.iRGB != null)
      {
        stNew.iRGB = new int[len][3]; // rgb values

        for (i=0; i<len; i++)
        {
          for (j=0; j<3; j++)
            stNew.iRGB[i][j] = stOld.iRGB[i][j];
        }
      }
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print()
   * <p> This method will print the lithology symbols structure
   * @param  st = Lithology Symbols data list structure
   */

  public static void print( parseRockColumnSymbolsListStruct st )
  {
    System.out.println(" ------------ Foreground Colors ------------- \n");
    print( st.stForeground );

    System.out.println("\n ------------ Background Colors ------------- \n");
    print( st.stBackground );

    System.out.println("\n ------------ Lithology Groups ------------- \n");
    print( st.stGroups );

    System.out.println("\n ------------ Lithologies ------------- \n");
    for (int i=0; i<st.iCount; i++)
    {
      print( st.stItem[i] );
    }
  }

  /** Method print()
   * <p> This method will print the color data structure
   * @param  st = Lithology colors data structure
   */

  public static void print( lithologyColorsStruct st )
  {
    int i=0;

    if (st != null)
    {
      if (st.iColors > 0)
      {
        if (st.colors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.colors[i]  + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
        else if (st.sColors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.sColors[i] + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
      }
    }
  }

  /** Method print()
   * <p> This method will print the group data structure
   * @param  st = Lithology Groups data structure
   */

  public static void print( lithologyGroupsStruct st )
  {
    if (st != null)
    {
      if (st.iGroups > 0)
      {
        for (int i=0; i<st.iGroups; i++)
        {
          System.out.println( i + " " + st.sGroups[i][0] +
                              " <-> " + st.sGroups[i][1] );
        }
      }
    }
  }

  /** Method print()
   * <p> This method will print the symbols data structure
   * @param  st = Lithology symbols data structure
   */

  public static void print( parseRockColumnSymbolsStruct st )
  {
    int i = 0;

    if (st != null)
    {
      System.out.println(st.sID    + " " + // ID for Lithology Symbol
                         st.iGroup + " " + // Group Number
                         st.sGroup + " " + // The group the lithology belongs
                         st.sName  + " " + // Name or Description of lithology
                         st.sAbrev + " " + // Abbreviation of lithology
                         st.sType  + " " + // Type of Lithology
                         st.iWeather );    // Weathering

      System.out.println("Color " +
                         st.iRGB[0] + " " +
                         st.iRGB[1] + " " +
                         st.iRGB[2]);

      if (st.symbol != null)
      {
        for (i=0; i<st.iRows; i++)
        {
          System.out.println(st.symbol[i][0] + " " + st.symbol[i][1]);
        }
      }

      if (st.iThesaurus > 0)
      {
        for (i=0; i<st.iThesaurus; i++)
        {
          System.out.print(st.sThesaurus[i] + " ");
        }
        System.out.println("\n");
      }
    }
  }
}

/*
 *  @version 1.1 08/02/2012
 *  @author  John Victorine
 */
