/*
 * @parsePHIUtility.java Version 1.1 10/14/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parsePHIListStruct;
import parse.parsePHIStruct;

import iqstrat.iqstratRemarkListStruct;

import rock.phi.phiListStruct;
import rock.phi.phiStruct;

import horizon.bio.bioStratListStruct;

/** Class parsePHIUtility
 *  <p> This Class will provide basic utilities for the Porosity Types parser
 *      data structures.
 *
 *  @version 1.1 10/14/2009
 *  @author  John R. Victorine
 */

public class parsePHIUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = Porosity Type Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parsePHIListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Porosity Type Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parsePHIListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getPHIStruct()
   * <p> This method will return the Porosity Type data structure
   * @param  iLevel = MAIN (Porosity Type) or SUB (Porosity Type Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Porosity Type Name or the Descriptor Name
   * @param  st     = The Parse Porosity Type List Data Structure
   * @return stT    = parse Porosity Type data structure
   */

  public static parsePHIStruct getPHIStruct(
      int iLevel, int iRow, String str, parsePHIListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parsePHIStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getPHIRow()
   * <p> This method will return the Porosity Type data structure
   * @param  iLevel = MAIN (Porosity Type) or SUB (Porosity Type Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Porosity Type Name or the Descriptor Name
   * @param  st     = The Parse Porosity Type List Data Structure
   * @return stT    = parse Porosity Type data structure
   */

  public static int getPHIRow(
           int iLevel, int iRow, String str, parsePHIListStruct st )
  {
    int iPorosity = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              iPorosity = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              iPorosity = i;
            }
          }
          break;
      }
    }

    return (iPorosity);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into Porosity
   *     Type list data structure
   * @param  stList    = Remarks List Data Structure
   * @param  stPorosities = Porosity Type Thesaurus
   * @return stPorosity  = Rock Porosity Type Data List Structure
   */

  public static phiListStruct parse(
                    iqstratRemarkListStruct stList,
                    parsePHIListStruct   stPorosities )
  {
    phiListStruct stPorosity   = null;
    String        sKEY       = cmn.cmnString.UniqueName();
    String        str        = "";
    String        tokens[]   = null;
//    String           sDelimiter = new String("[,-;.:= ]+");
    String        sDelimiter = new String("[ /,;:.=-]+");
    phiStruct     stText     = null;

    if (stList != null)
    {
      stPorosity = new phiListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parsePHIs( stList.stItem[i].depthStart,
                            stList.stItem[i].depthEnd,
                            tokens, stPorosities );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stPorosity  = rock.phi.phiUtility.add( stText, stPorosity );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stPorosity);
  }

  /** Method parsePHIs()
   * <p> This method will parse the text string into rock Porosity Type
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  sData        = the Porosity Type string phrase or word
   * @param  stPorosities = Porosity Type Thesaurus
   * @return st           = rock Porosity Type data structure
   */

  public static  phiStruct parsePHIs(
                     double depthStart, double depthEnd, String sData[],
                     parsePHIListStruct stPorosities )
  {
    int i, j, k, m, m0, m1, m2, p, q;
    int            iPHI[]     = { -1, -1, -1, -1, -1, -1, -1, -1 };
    phiStruct      st         = null;
    int            iCount     = 0;
    int            iValue     = 0;
    double         dValue[]   = { 0.0, 0.0, 0.0 };
    String         sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                  {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                  {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int            iRow[]     = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int            iOrder[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String         sTemp[]    = {"", ""};
    int            iPHITypes  = -1;
    int            iDescrip   = -1;
    int            iContinue  = 0;
    int            iNext      = 0;
    parsePHIStruct stPorosity = null;
    String         sID        = "";
    String         sName      = "";
    int            iNew       = 0;
    int            iFound     = 0;

    if (stPorosity != null)
      stPorosity.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stPorosities.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stPorosities.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stPorosities.sAbbrev[j][1]);
      }
    }

    iCount = 0;
    for (i=0; i<sData.length; i++)
    {
	  if ((sData[i].equals("porosity")) && (iCount < sData.length))
	  {
		if (iCount < 8)
		{
	      iPHI[iCount] = i;
	      iContinue    = 1;
	      iCount++;
	    }
	  }
    }

    if (iContinue > 0)
    {
      iContinue = 0;

      // Find Porosity Type Words up to 15

      for (j=0; j<sData.length; j++)
      {
        if (isKeyword( sData[j], stPorosities ))
        {
          if (iCount < 15)
          {
            sTemp             = getKeyword( sData[j], stPorosities );
            sWords[iCount][0] = new String(sTemp[0]);
            sWords[iCount][1] = new String(sTemp[1]);
            iOrder[iCount]    = j;
            iCount++;
          }
        }
      }

      for (i=0; i<iCount; i++)
      {
        if (sWords[i][1].equals("V"))
        {
  		  if (iValue < 3)
		  {
		    if (sWords[i][0].equals("excellent"))
		      dValue[iValue] = 25.0;
		    if (sWords[i][0].equals("very good"))
		      dValue[iValue] = 18.0;
		    if (sWords[i][0].equals("good"))
		      dValue[iValue] = 13.0;
		    if (sWords[i][0].equals("fair"))
		      dValue[iValue] = 7.0;
		    if (sWords[i][0].equals("poor"))
		      dValue[iValue] = 3.0;
		    if (sWords[i][0].equals("very poor"))
		      dValue[iValue] = 1.0;
		    if (sWords[i][0].equals("trace"))
		      dValue[iValue] = 0.7;

		    iValue++;
		  }
        }
      }

      // Change a Modifier to a descriptor

      for (i=0; i<iCount; i++)
      {
        if (sWords[i][1].equals("M"))
        {
          k = i+1;
          if (k<iCount)
          {
            if (isModified( sWords[i][0], sWords[k][0], stPorosities ))
            {
              sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
              sWords[i][1] = new String(sWords[k][1]);
              iOrder[i]    = iOrder[k];

              sWords[k][0] = new String("");
              sWords[k][1] = new String("");
            }
          }
        }
      }

      // Identify the main prosity  type

      for (i=0; i<iCount; i++)
      {
        if (sWords[i][1].equals("P"))
        {
		  iFound = 0;
		  for (j=0; j<iPHI.length; j++)
		  {
		    if ((iPHI[j]-iOrder[i] < 5) && (iPHI[j]-iOrder[i] > 0))
		      iFound = 1;
	      }

	      if (iFound == 1)
	      {
            iPHITypes  = getPHIRow( _MAIN, -1, sWords[i][0], stPorosities );

            if (iPHITypes > -1)
            {
              iContinue    = 1;
              iRow[i]      = iPHITypes;
              sWords[i][1] = new String("X");

            }
          }
        }
      }

      // Modify porosity with a descriptor

      if (iContinue == 1)
      {
        for (m=0; m<iCount; m++)
        {
          m0 = m+1;
          m1 = m+2;
          if (m0 > iCount) m0 = iCount;
          if (m1 > iCount) m1 = iCount;
          m2 = m-1;
          if (m2 < 0) m2 = 0;

          if (iRow[m] > -1)
          {
            stPorosity = stPorosities.stItem[iRow[m]];

            sID      = new String( stPorosity.sID);
            sName    = new String( stPorosity.sPhrase );

            for (i=0; i<m; i++)
            {
              if (sWords[i][1].equals("D"))
              {
                iDescrip = getPHIRow(
                               _SUB, iRow[m], sWords[i][0], stPorosities );

                if (iDescrip > -1)
                {
                  sWords[i][1] = new String("X");
                  stPorosity     = stPorosities.stItem[iRow[m]].descriptors[iDescrip];

                  sID          = new String( stPorosity.sID);
                  sName        = new String( stPorosity.sPhrase + stPorosities.stItem[iRow[m]].sPhrase );
                }
              }
            }

            if (iNew == 0)
            {
              iNew = 1;
              st   = new phiStruct();

              st.sKEY = new String(cmn.cmnString.UniqueName() +
                                 ( (int) depthStart)); // Create a Unique KEY

              st.depthStart   = depthStart;         // Starting Depth
              st.depthEnd     = depthEnd;           // Ending Depth

              if (iValue > 0)
              {
			    for (q=0; q<iValue; q++)
			    {
				  st.dPHI = st.dPHI + dValue[q];
				}

				st.dPHI = st.dPHI / iValue;
			  }

			  iValue = 0;
            }

            if (st != null)
            {
              st    = rock.phi.phiUtility.addPHI(sID, "", sName, st);

              sID   = new String("");
              sName = new String("");
            }
          }
        }
      }


      if (iValue > 0)
      {
        iNew = 1;
        st   = new phiStruct();

        st.sKEY = new String(cmn.cmnString.UniqueName() +
                           ( (int) depthStart)); // Create a Unique KEY

        st.depthStart   = depthStart;         // Starting Depth
        st.depthEnd     = depthEnd;           // Ending Depth

        if (iValue > 0)
        {
		  for (q=0; q<iValue; q++)
		  {
		    st.dPHI = st.dPHI + dValue[q];
		  }

		  st.dPHI = st.dPHI / iValue;
		}

		iValue = 0;
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Porosity Types Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parsePHIListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parsePHIListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Porosity Types Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parsePHIListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parsePHIListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parsePHIListStruct copyList( parsePHIListStruct stOld )
  {
    parsePHIListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parsePHIListStruct();

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Porosity Types Dictionary

      stNew.iPorosity  = stOld.iPorosity;
      stNew.stPorosity = new parsePHIStruct[stOld.iPorosity];

      for (i=0; i<stOld.iPorosity; i++)
      {
        stNew.stPorosity[i] = copy( stOld.stPorosity[i] );
      }

      // Porosity Types

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parsePHIStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parsePHIStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parsePHIStruct copy( parsePHIStruct stOld )
  {
    int i=0;
    int j=0;
    parsePHIStruct stNew = new parsePHIStruct();

    // Fossils Lithology

    stNew.sID     = new String( stOld.sID );        // String ID for Mask
    stNew.sPhrase = new String( stOld.sPhrase );    // Phrase

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parsePHIStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parsePHIListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parsePHIListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print( parsePHIStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parsePHIStruct st )
  {
    if (st != null)
    {
      System.out.println( st.sID     + " <" + // String ID for Mask Symbol
                          st.sPhrase + "> ");

      if (st.descriptors != null)
      {
        System.out.println( " ---- Descriptors ---- ");

        for (int i=0; i<st.iCount; i++)
        {
          print(st.descriptors[i]);
        }

        System.out.println( " ---- End Descriptors ---- ");
      }
    }
  }
}