/*
 * @parseMineralsUtility.java Version 1.1 12/11/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseMineralsListStruct;
import parse.parseMineralsStruct;
import parse.parseTextureListStruct;
import parse.parseTextureStruct;

//import kgs.kgsMeasSectListStruct;
import iqstrat.iqstratRemarkListStruct;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;

/** Class parseMineralsUtility
 *  <p> This Class will provide basic utilities for the standard
 *      well logging mineral data.
 *
 *  @version 1.1 12/11/2009
 *  @author  John R. Victorine
 */

public class parseMineralsUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * --------------------------- PARSE METHODS ------------------------------ *
   * ======================================================================== */

  /** Method parse()
   * <p> This method will parse the Measured Section into rock texture list
   *     data structure
   * @param  stList      = Remarks List Data Structure
   * @param  stSymbols   = Lithology Symbols Data List Structure
   * @param  stTextures  = Texture Thesaurus
   * @param  stMinerals  = Common Minerals Data & Thesaurus
   * @return stLithology = Rock Lithology Data List Structure
   */

  public static lithologyListStruct parse( iqstratRemarkListStruct    stList,
                                           lithologySymbolsListStruct stSymbols,
                                           parseTextureListStruct  stTextures,
                                           parseMineralsListStruct stMinerals )
  {
    lithologyListStruct st          = null;
    int                 iUnit       = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT; //_SCALE_10TH_FOOT; //_SCALE_1_FOOT;
    double              dUnit       = 0.1;//1.0;
    double              depth       = 0.0;
    double              depthStart  = 0.0;
    double              depthEnd    = 0.0;
    String              sKEY        = cmn.cmnString.UniqueName();
    String              str         = "";
    String              tokens[]    = null;
//    String              sDelimiter  = new String("[,-;.:= ]+");
    String            sDelimiter = new String("[ /,;:.=-]+");
    lithologyStruct     stLithology = null;

    if (stList != null)
    {
      st = new lithologyListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        if (i == 0)
        {
          depthStart = stList.stItem[i].depthStart;
          depthEnd   = stList.stItem[i].depthEnd;
        }

        if (depthStart > stList.stItem[i].depthStart)
          depthStart = stList.stItem[i].depthStart;

        if (depthEnd < stList.stItem[i].depthEnd)
          depthEnd   = stList.stItem[i].depthEnd;

        depth = Math.abs( stList.stItem[i].depthEnd -
                          stList.stItem[i].depthStart );

        iUnit = lith.lithology.lithologyUtility.getUnit( iUnit, depth );
        dUnit = iqstrat.iqstratTracksStruct._SCALE[iUnit];
/*
        if (dUnit > iqstrat.iqstratTracksStruct._SCALE[
                      lith.lithology.lithologyUtility.getUnit( depth )])
        {
          iUnit = lith.lithology.lithologyUtility.getUnit( depth );
          dUnit = iqstrat.iqstratTracksStruct._SCALE[iUnit];
        }
*/
      }

      st.depthStart = depthStart;
      st.depthEnd   = depthEnd;
      st.iUnit      = iUnit;
//System.out.println("LITHOLOGY: "+iUnit+" "+dUnit);

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stLithology = parseMinerals( stList.stItem[i].depthStart,
                                     stList.stItem[i].depthEnd,
                                     tokens, stSymbols, stTextures, stMinerals);

        if (stLithology != null)
        {
          stLithology.sKEY = new String( sKEY + "_" + i );
          st = lith.lithology.lithologyUtility.add( stLithology, st );
        }

        tokens = null;

        if (stLithology != null)
          stLithology.delete();
        stLithology = null;
      }

      stList.delete();
      stList = null;
    }

    return (st);
  }

  /** Method parseMinerals()
   * <p> This method will parse the text string into rock texture
   * @param  depthStart  = the starting depth
   * @param  depthEnd    = the ending depth
   * @param  sData       = the texture string phrase or word
   * @param  stSymbols   = Lithology Symbols Data List Structure
   * @param  stTextures  = Texture Thesaurus
   * @param  stMinerals  = Common Minerals Data & Thesaurus
   * @return stLithology = Rock Lithology Data Structure
   */

  public static lithologyStruct parseMinerals(
                                       double depthStart,
                                       double depthEnd,
                                       String sData[],
                                       lithologySymbolsListStruct stSymbols,
                                       parseTextureListStruct     stTextures,
                                       parseMineralsListStruct    stMinerals )
  {
	int i,j,k, m, i1;
    lithologyStruct     st          = null;
    parseMineralsStruct stMineral   = null;
    parseTextureStruct  stTexture   = null;
    int                 iCount      = 0;
    String              sWords[][]  = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                        {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                                        {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int                 iRow[]      = { -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1 };
    String              sTemp[]     = { "", "" };
    int                 iTextures[] = { -1, -1 };
    int                 iTexture    = -1;
    int                 iDescrip    = -1;
    int                 iContinue   = 0;
    int                 iNext       = 0;
    String              sMineral    = "";
    String              sPrimary    = "";
    String              secondary   = "";
    int                 iOther      = 0;
    int                 iMaximum    = 3;
    int                 iMaxWords   = 7;
    String              sOther[]    = { "", "", "", "", "", "", "", "", "", "" };
    double              dRhomaa[]   = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
    double              dUmaa[]     = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
    double              dGR[]       = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
    double              dPer[]      = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
    double              dPercent    = 60.0; //0.6;
    double              dTotal      = 0.0;
    double              dTemp       = 0.0;
    int                 iSymbol[][] = { { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 },
                                        { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } };
    int                 iSecond     = -1;

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stTextures.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stTextures.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stTextures.sAbbrev[j][1]);
      }
    }

    // Find Texture Words up to 15

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseTextureUtility.isKeyword( sData[j], stTextures ))
      {
        if (iCount < 15)
        {
          sTemp = parse.parseTextureUtility.getKeyword( sData[j], stTextures );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iRow[iCount]      = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("M")) || (sWords[i][1].equals("Mt")))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified(
                sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iRow[k]      = -1;

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
/*
      if (sWords[i][1].equals("Da"))
      {
		if ((i > 0) && (i<iCount-1))
		{
          k = i-1;
          m = i+1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")))
		  {
		    sWords[i][1] = new String("L");
		  }
		}
		else if (i<iCount-1)
		{
          m = i+1;
		  if (!sWords[m][1].equals("L"))
		  {
		    sWords[i][1] = new String("L");
		  }
		}
		else if (i > 0)
		{
          k = i-1;
		  if (!sWords[k][1].equals("L"))
		  {
		    sWords[i][1] = new String("L");
		  }
		}
      }
*/
      if (sWords[i][1].equals("Da"))
      {
	    if ((i > 0) && (i<iCount-1))
		{
          k = i-1;
          m = i+1;

		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")) &&
		      (!sWords[k][1].equals("B")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i == 0)
		{
          m = i+1;
		  if ((!sWords[m][1].equals("L")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i==iCount-1)
		{
          k = i-1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[k][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
      }

      if (sWords[i][1].equals("B"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified(
			    sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main lithology type

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")))
      {
        iTexture  = parse.parseTextureUtility.getTextureRow(
                      _MAIN, -1, sWords[i][0], stTextures );

        if (iTexture > -1)
        {
          iContinue = 1;
          stTexture = stTextures.stItem[iTexture];

          if (sPrimary.length() == 0)
          {
            sPrimary     = new String( stTexture.sPrimary.toLowerCase() );
            dPercent     = 60.0;
            dPer[0]      = 100.0;

            if (stTexture.sSecondary.length() > 0)
            {
			  iSecond    = 0;
              secondary  = new String( stTexture.sSecondary.toLowerCase() );
              dPercent   = 20.0;
              dPer[0]    = 60.0;
              dPer[1]    = 40.0;
		    }

            iTextures[0] = iTexture;
          }

          if (secondary.length() == 0)
          {
            if ((!stTexture.sPrimary.toLowerCase().equals(sPrimary)) &&
                (stTexture.sPrimary.toLowerCase().length() > 0))
            {
			  iSecond      = 1;
              iTextures[1] = iTexture;
              secondary    = new String( stTexture.sPrimary.toLowerCase() );
              dPercent     = 40.0;//80.0;
              dPer[0]      = 80.0;
              dPer[1]      = 20.0;
              if (i > 0)
              {
				i1 = i-1;
                if (sWords[i1][0].equals("slightly"))
                {
                  dPercent     = 50.0;//90.0;
                  dPer[0]      = 90.0;
                  dPer[1]      = 10.0;
				}
				else if (sWords[i1][0].equals("very"))
                {
                  dPercent     = 30.0;//70.0;
                  dPer[0]      = 70.0;
                  dPer[1]      = 30.0;
				}
				else if (sWords[i1][0].equals("extremely"))
                {
                  dPercent     = 20.0;//70.0;
                  dPer[0]      = 60.0;
                  dPer[1]      = 40.0;
				}
		      }
		    }
		  }
        }
      }
//System.out.println("A "+iSecond + " "+depthStart+" "+depthEnd+" "+sPrimary+" "+secondary );
    }
//System.out.println("B "+iSecond + " "+depthStart+" "+depthEnd+" "+sPrimary+" "+secondary );

    // Get the Rhomaa-Umaa & Gamma Ray Values and the Symbols for the Primary &
    // Secondary Minerals.

    for (j=0; j<stMinerals.iCount; j++)
    {
      sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

      // Set Primary Mineral Rhomaa, Umaa and GR data Values

      if (sPrimary.equals(sMineral))
      {
        iSymbol[0][0]  = stMinerals.stItem[j].id;
        dRhomaa[0]     = stMinerals.stItem[j].dRhomaa;
        dUmaa[0]       = stMinerals.stItem[j].dUmaa;
        dGR[0]         = stMinerals.stItem[j].dGR;

        if ((stTexture.dGR > 0.0) && (stTexture.sSecondary.length() == 0))
          dGR[0] = stTexture.dGR;
      }

      // Set Secondary Mineral Rhomaa, Umaa and GR data Values

      if (secondary.equals(sMineral))
      {
        iSymbol[1][0] = stMinerals.stItem[j].id;
        dRhomaa[1]    = stMinerals.stItem[j].dRhomaa;
        dUmaa[1]      = stMinerals.stItem[j].dUmaa;
        dGR[1]        = stMinerals.stItem[j].dGR;

        if (stTexture.dGR > 0.0)
          dGR[1] = stTexture.dGR;
      }
    }

    // Get other lithologies from the descriptors

    if (iContinue == 1)
    {
      if (secondary.length() > 0)
      {
        iMaximum = 2;
      }

      for (i=0; i<iCount; i++)
      {
        if (iOther < iMaximum)
        {
          if (sWords[i][1].equals("D"))
          {
			if (iTextures[0] > -1)
			{
			  iTexture = iTextures[0];
              iDescrip = parse.parseTextureUtility.getTextureRow(
                           _SUB, iTexture, sWords[i][0], stTextures);

              if (iDescrip == -1)
              {
			    if (iTextures[1] > -1)
			    {
  			      iTexture = iTextures[1];
                  iDescrip = parse.parseTextureUtility.getTextureRow(
                               _SUB, iTexture, sWords[i][0], stTextures);
				}
              }

              if (iDescrip > -1)
              {
                stTexture = stTextures.stItem[iTexture].descriptors[iDescrip];

                iNext = 0;
                for (int n=0; n<iOther; n++)
                {
                  if (stTexture.sOther.toLowerCase().equals(sOther[n]))
                    iNext = 1;
                }

                if (sPrimary.equals(stTexture.sOther.toLowerCase()))
                  iNext = 1;

                if (secondary.equals(stTexture.sOther.toLowerCase()))
                  iNext = 1;

                if (iNext == 0)
                {
                  dTemp = stTexture.dPercent;
                  if (dTemp + dTotal <= dPercent)
                  {
                    dTotal         = dTotal + dTemp;
                    dPer[iOther+2] = dTemp;
                    sOther[iOther] = new String(stTexture.sOther.toLowerCase());
                    dGR[iOther+2]  = stTexture.dGR;
                    iOther++;
                  }
                }
              }
		    }
          }
        }
	  }

	  if (iOther == 0)
	  {
        for (i=0; i<iCount; i++)
        {
          if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")))
          {
            iTexture  = parse.parseTextureUtility.getTextureRow(
                          _MAIN, -1, sWords[i][0], stTextures );

            if (iTexture > -1)
            {
              iContinue = 1;
              stTexture = stTextures.stItem[iTexture];

              iNext = 0;
              if (sPrimary.equals(stTexture.sPrimary.toLowerCase()))
                iNext = 1;

              if (secondary.equals(stTexture.sPrimary.toLowerCase()))
                iNext = 1;

              if (iNext == 0)
              {
                dTemp = stTexture.dPercent;
                if (dTemp + dTotal <= dPercent)
                {
                  dTotal         = dTotal + dTemp;
                  dPer[iOther+2] = dTemp;
                  if (i>1)
                  {
					if (sWords[i-1].equals("slightly"))
					  dPer[iOther+2] = 10.0;
					else if (sWords[i-1].equals("very"))
					  dPer[iOther+2] = 30.0;
				  }
                  sOther[iOther] = new String(stTexture.sPrimary.toLowerCase());
                  dGR[iOther+2]  = stTexture.dGR;
                  iOther++;
                }
              }
            }
          }
        }
	  }

      // Get the Rhomaa-Umaa & Gamma Ray Values and the Symbols for the
      // descriptors to the lithogy.

      if (iOther > 0)
      {
        for (i=0; i<iOther; i++)
        {
          for (j=0; j<stMinerals.iCount; j++)
          {
            sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

            // Set Primary Mineral Rhomaa, Umaa and GR data Values

            if (sOther[i].equals(sMineral))
            {
              iSymbol[i+2][0] = stMinerals.stItem[j].id;
              dRhomaa[i+2]    = stMinerals.stItem[j].dRhomaa;
              dUmaa[i+2]      = stMinerals.stItem[j].dUmaa;
              if (dGR[i+2] == 0.0)
                dGR[i+2]      = stMinerals.stItem[j].dGR;
            }
          }
        }
      }

      if (iSecond == 1)
      {
        for (i=0; i<iOther; i++)
        {
          dPer[0] = dPer[0] - dPer[i+2];
        }
	  }
	  else if (iSecond == 0)
	  {
        if (dTotal == 20.0) //0.2)
        {
          dPer[0] = dPer[0] - 10.0; //0.1;
          dPer[1] = dPer[1] - 10.0; //0.1;
        }
        else if (dTotal == 10.0) //0.1)
          dPer[1] = dPer[1] - 10.0; //0.1;
	  }
	  else
	  {
        for (i=0; i<iOther; i++)
        {
          dPer[0] = dPer[0] - dPer[i+2];
        }
	  }
/*
      if (dPer[1] > 0)
      {
        if (dTotal == 20.0) //0.2)
        {
          dPer[0] = dPer[0] - 10.0; //0.1;
          dPer[1] = dPer[1] - 10.0; //0.1;
        }
        else if (dTotal == 10.0) //0.1)
          dPer[1] = dPer[1] - 10.0; //0.1;
      }
      else
      {
        for (i=0; i<iOther; i++)
        {
          dPer[0] = dPer[0] - dPer[i+2];
        }
      }
*/
    }

    if (iContinue == 1)
    {
      st            = new lithologyStruct();

      st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                  ((int)depthStart)); // Create a Unique KEY
      st.depthStart = depthStart; // Starting Depth
      st.depthEnd   = depthEnd;   // Ending Depth
//System.out.println("-- "+depthStart+" "+depthEnd+" <"+sPrimary+"><"+secondary+"><"+sOther[0]+"><"+sOther[1]+"><"+sOther[2]+"> "+dPer[0]+" "+dPer[1]+" "+dPer[2]+" "+dPer[3]+" "+dPer[4]);

      iCount   = 0;

      for (i=0; i<5; i++)
      {
        iSymbol[i][1] = (int) (dPer[i]/10.0);
      }

      iSymbol = Sort(5, 2, iSymbol);

      for (i=0; i<5; i++)
      {
        st.dRhomaa = st.dRhomaa + dPer[i] * dRhomaa[i]/100.0;
        st.dUmaa   = st.dUmaa   + dPer[i] * dUmaa[i]/100.0;
        st.dGR     = st.dGR     + dPer[i] * dGR[i]/100.0;

        if (iSymbol[i][1] > 0)
        {
          for (j=0; j<iSymbol[i][1]; j++)
          {
            if (iCount < 10)
            {
              st.iLithology[iCount] = iSymbol[i][0];
              iCount++;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method parseMinerals()
   * <p> This method will parse the text string into rock texture
   * @param  depthStart  = the starting depth
   * @param  depthEnd    = the ending depth
   * @param  sData       = the texture string phrase or word
   * @param  stSymbols   = Lithology Symbols Data List Structure
   * @param  stTextures  = Texture Thesaurus
   * @param  stMinerals  = Common Minerals Data & Thesaurus
   * @return stLithology = Rock Lithology Data Structure
   *

  public static lithologyStruct parseMinerals(
                                       double depthStart,
                                       double depthEnd,
                                       String sData[],
                                       lithologySymbolsListStruct stSymbols,
                                       parseTextureListStruct     stTextures,
                                       parseMineralsListStruct    stMinerals )
  {
    lithologyStruct     st        = null;
    parseMineralsStruct stMineral = null;
    parseTextureStruct  stTexture = null;

    int    i           = 0;
    int    j           = 0;
    int    k           = 0;
    int    pRow        = -1;
    int    mRow        = -1;
    int    iCount      = 0;
    String sWords[][]  = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                           {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                           {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int    iRow        = -1;
//    int    iRow[]     = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String sTemp[]     = {"", ""};
    int    iTexture    = -1;
    int    iDescrip    = -1;
    int    iContinue   = 0;
    int    iNext       = 0;
    String sMineral    = "";
    String sPrimary    = "";
    String secondary   = "";
    int    iOther      = 0;
    int    iMaximum    = 3;
    int    iMaxWords   = 7;
    String sOther[]    = { "", "", "" };
    double dRhomaa[]   = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dUmaa[]     = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dGR[]       = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dPer[]      = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dPercent    = 60.0; //0.6;
    double dTotal      = 0.0;
    double dTemp       = 0.0;
    int    iSymbol[][] = { { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } };

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stTextures.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stTextures.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stTextures.sAbbrev[j][1]);
      }
    }

    // Find Texture Words up to 15

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseTextureUtility.isKeyword( sData[j], stTextures ))
      {
        if (iCount < 15)
        {
          sTemp = parse.parseTextureUtility.getKeyword( sData[j], stTextures );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified(
                sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }

      if (sWords[i][1].equals("B"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified(
                sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }

    }

    // Identify the main lithology type

    for (i=0; i<iCount; i++)
    {
      if (sPrimary.length() == 0)
      {
        if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")))
        {
          iTexture  = parse.parseTextureUtility.getTextureRow(
                        _MAIN, -1, sWords[i][0], stTextures );
          iContinue = 1;

          if (iTexture > -1)
          {
            iRow = i;

            pRow = iRow + iMaxWords;
            if (pRow > iCount) pRow = iCount;

            mRow = iRow - iMaxWords;
            if (mRow < 0) mRow = 0;

            stTexture = stTextures.stItem[iTexture];

            sPrimary  = new String( stTexture.sPrimary.toLowerCase() );
            secondary = new String( stTexture.sSecondary.toLowerCase() );
          }
        }
      }
    }

    // Get the Rhomaa-Umaa & Gamma Ray Values and the Symbols for the Primary &
    // Secondary Minerals.

    for (j=0; j<stMinerals.iCount; j++)
    {
      sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

      // Set Primary Mineral Rhomaa, Umaa and GR data Values

      if (sPrimary.equals(sMineral))
      {
        dPercent       = 60.0; //0.6;
        dPer[0]        = 100.0; //1.0;
        iSymbol[0][0]  = stMinerals.stItem[j].id;
        dRhomaa[0]     = stMinerals.stItem[j].dRhomaa;
        dUmaa[0]       = stMinerals.stItem[j].dUmaa;
        dGR[0]         = stMinerals.stItem[j].dGR;

        if ((stTexture.dGR > 0.0) && (stTexture.sSecondary.length() == 0))
          dGR[0] = stTexture.dGR;
      }

      // Set Secondary Mineral Rhomaa, Umaa and GR data Values

      if (secondary.equals(sMineral))
      {
        dPercent      = 20.0; //0.2;
        dPer[0]       = 60.0; //0.6;
        dPer[1]       = 40.0; //0.4;
        iSymbol[1][0] = stMinerals.stItem[j].id;
        dRhomaa[1]    = stMinerals.stItem[j].dRhomaa;
        dUmaa[1]      = stMinerals.stItem[j].dUmaa;
        dGR[1]        = stMinerals.stItem[j].dGR;

        if (stTexture.dGR > 0.0)
          dGR[1] = stTexture.dGR;
      }
    }

    // Modify Litholgy with a descriptors

    if (iContinue == 1)
    {
      if (secondary.length() > 0)
      {
        iMaximum = 2;
      }

      for (i=mRow; i<pRow; i++)
      {
        if (iOther < iMaximum)
        {
          if ((sWords[i][1].equals("D")) || (sWords[i][1].equals("M")))
          {
            if (iTexture > -1)
            {
              iDescrip = parse.parseTextureUtility.getTextureRow(
                           _SUB, iTexture, sWords[i][0], stTextures);

              if (iDescrip > -1)
              {
                stTexture = stTextures.stItem[iTexture].descriptors[iDescrip];

                iNext = 0;
                for (int n=0; n<iOther; n++)
                {
                  if (stTexture.sOther.toLowerCase().equals(sOther[n]))
                    iNext = 1;
                }

                if (iNext == 0)
                {
                  if (!sPrimary.equals(stTexture.sOther.toLowerCase()))
                  {
                    dTemp = stTexture.dPercent;
                    if (dTemp + dTotal <= dPercent)
                    {
                      dTotal         = dTotal + dTemp;
                      dPer[iOther+2] = dTemp;
                      sOther[iOther] = new String(stTexture.sOther.toLowerCase());
                      dGR[iOther+2]  = stTexture.dGR;
                      iOther++;
                    }
                  }
                  else
                  {
                    if (stTexture.dGR > 0.0)
                      dGR[0] = stTexture.dGR;
                  }
                }
              }
            }
          }
        }
      }

      // Get the Rhomaa-Umaa & Gamma Ray Values and the Symbols for the
      // descriptors to the lithogy.

      if (iOther > 0)
      {
        for (i=0; i<iOther; i++)
        {
          for (j=0; j<stMinerals.iCount; j++)
          {
            sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

            // Set Primary Mineral Rhomaa, Umaa and GR data Values

            if (sOther[i].equals(sMineral))
            {
              iSymbol[i+2][0] = stMinerals.stItem[j].id;
              dRhomaa[i+2]    = stMinerals.stItem[j].dRhomaa;
              dUmaa[i+2]      = stMinerals.stItem[j].dUmaa;
              if (dGR[i+2] == 0.0)
                dGR[i+2]      = stMinerals.stItem[j].dGR;
            }
          }
        }
      }

      if (dPer[1] > 0)
      {
        if (dTotal == 20.0) //0.2)
        {
          dPer[0] = dPer[0] - 10.0; //0.1;
          dPer[1] = dPer[1] - 10.0; //0.1;
        }
        else if (dTotal == 10.0) //0.1)
          dPer[1] = dPer[1] - 10.0; //0.1;
      }
      else
      {
        for (i=0; i<iOther; i++)
        {
          dPer[0] = dPer[0] - dPer[i+2];
        }
      }
    }

    if (iContinue == 1)
    {
      st            = new lithologyStruct();

      st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                  ((int)depthStart)); // Create a Unique KEY
      st.depthStart = depthStart; // Starting Depth
      st.depthEnd   = depthEnd;   // Ending Depth
//System.out.println(depthStart+" "+depthEnd+" <"+sPrimary+"><"+secondary+"><"+sOther[0]+"><"+sOther[1]+"><"+sOther[2]+"> "+dPer[0]+" "+dPer[1]+" "+dPer[2]+" "+dPer[3]+" "+dPer[4]);

      iCount   = 0;

      for (i=0; i<5; i++)
      {
//        iSymbol[i][1] = (int) (10.0 * dPer[i]);
        iSymbol[i][1] = (int) (dPer[i]/10.0);
      }

      iSymbol = Sort(5, 2, iSymbol);

      for (i=0; i<5; i++)
      {
//        iSymbol[i][1] = (int) (10.0 * dPer[i]);

        st.dRhomaa = st.dRhomaa + dPer[i] * dRhomaa[i]/100.0;
        st.dUmaa   = st.dUmaa   + dPer[i] * dUmaa[i]/100.0;
        st.dGR     = st.dGR     + dPer[i] * dGR[i]/100.0;

        if (iSymbol[i][1] > 0)
        {
          for (j=0; j<iSymbol[i][1]; j++)
          {
            if (iCount < 10)
            {
              st.iLithology[iCount] = iSymbol[i][0];
              iCount++;
            }
          }
        }
      }
    }

    return (st);
  }
*/
  /** Method Sort()
   * <p> This method will sort in ascending percent order (lowest to highest)
   * @param  iRows = the total number of rows
   * @param  iCols = the toal number of columns
   * @param  iSymbol  = list of mineral symbols
   * @return iSymbol  = list of mineral symbols ordered by minerals
   */

  public static int[][] Sort(int iRows, int iCols, int iSymbol[][])
  {
    int i,j=0;
    boolean swappedOnPrevRun = true;
    int     iTemp[]          = null;
    int     iTotal           = 12;
    int     i1               = 0;
    int     i2               = 1;
    int     iOrder[][]       = { { 10, 0 },  // Smectite (Clay Symbol)
                                 { 12, 1 },  // Illite (Shale Symbol)
                                 { 43, 2 },  // Peat
                                 { 46, 3 },  // Lignite
                                 { 48, 4 },  // Bituminous Coal
                                 { 49, 5 },  // Anthracite
                                 {  8, 6 },  // Silt
                                 {  7, 7 },  // Sandstone
                                 { 95, 8 },  // Feldspar
                                 { 19, 9 },  // Calcite (Limestone)
                                 { 22, 10 }, // Dolomite
                                 { 52, 11 }, // Chert
                                 { 34, 12 }, // Halite
                                 { 35, 13 }, // Sylvite
                                 { 32, 14 }, // Anhydrite
                                 { 31, 15 }, // Gypsum
    };

    iTemp = new int[iCols];
    for (j=0; j<iCols; j++)
        iTemp[j] = 0;

    if (iSymbol != null)
    {
      while(swappedOnPrevRun)
      {
        // this variable keeps track of whether to continue sorting or exit

        swappedOnPrevRun = false;

        // loop through every element in the array, except for the last one

        for(i=0; i<iRows-1; i++)
        {
          // if current element is greater than the next swap the two elements

          i1 = iTotal;
          i2 = iTotal;

          for (int k=0; k<iTotal; k++)
          {
            if (iSymbol[i][0] == iOrder[k][0])
              i1 = iOrder[k][1];

            if (iSymbol[i+1][0] == iOrder[k][0])
              i2 = iOrder[k][1];
          }

          if (i1 > i2)
          {
            // we don't want the loop to end just yet, we're not done

            swappedOnPrevRun = true;

            // store element i in a temporary variable

            for (j=0; j<iCols; j++) iTemp[j] = iSymbol[i][j];

            // set element i+1 to where i used to be

            for (j=0; j<iCols; j++) iSymbol[i][j] = iSymbol[i+1][j];

            // release the old i from temp into i+1 slot

            for (j=0; j<iCols; j++) iSymbol[i+1][j] = iTemp[j];
          }
        }
      }
    }

    return (iSymbol);
  }

  /* ======================================================================== *
   * --------------------------- COMPUTE METHODS ---------------------------- *
   * ======================================================================== */

  /** Method copyList( parseMineralsListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  st = the Minerals Data List Structure
   * @return st = the Minerals Data with Rhomaa-Umaa List structure
   */

  public static parseMineralsListStruct computeRhomaaUmaa(
                                            parseMineralsListStruct st )
  {
    double dDPHI = 0.0;
    double dNPHI = 0.0;
    double dphi  = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        dDPHI  = math.mathLAS.computePHI( st.stItem[i].dRHOB,
                                          math.mathLAS._DENSITY,
                                          math.mathLAS._CALCITE,
                                          math.mathLAS._FRESH );
//        dDPHI = lith.math.lithMath.computeDPHI(
//          lith.math.lithMath._CALCITE,
//          lith.math.lithMath._FRESH_H2O,
//          st.stItem[i].dRHOB );

        dNPHI = 0.0;
        if (st.stItem[i].dCNL > 0.0)
          dNPHI = st.stItem[i].dCNL / 100.0;
        else
          dNPHI = st.stItem[i].dCNLCalc / 100.0;

        dphi = (dDPHI + dNPHI) / 2.0;

        st.stItem[i].dRhomaa = math.mathLAS.computeRhomaa(
            lith.math.lithMath._FRESH_H2O, st.stItem[i].dRHOB, dphi );

        st.stItem[i].dUmaa = math.mathLAS.computeUmaa( st.stItem[i].dU, dphi );
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseMineralsListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseMineralsListStruct copyList( parseMineralsListStruct stOld )
  {
    parseMineralsListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseMineralsListStruct();

      // Minerals

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseMineralsStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseMineralsStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseMineralsStruct copy( parseMineralsStruct stOld )
  {
    int i=0;
    int j=0;
    parseMineralsStruct stNew = new parseMineralsStruct();

    // Mineral Identification

    stNew.sName       = new String( stOld.sName );        // Name of mineral
    stNew.sGroup      = new String( stOld.sGroup );       // Mineral Group
    stNew.sClass      = new String( stOld.sClass );       // Mineral Class
    stNew.sFormula    = new String( stOld.sFormula );     // Mineral Formula
    stNew.sOccurrence = new String( stOld.sOccurrence );  // Mineral Occurrence

    stNew.id          = stOld.id;  // Plot Symbol

    // Schlumberger Measured & Calculated Data

    stNew.dMolWt     = stOld.dMolWt;     // Molecular weight
    stNew.dRHOB      = stOld.dRHOB;      // Bulk Density
    stNew.dRHOE      = stOld.dRHOE;      // Electron Density (gm/cc)
    stNew.dRHOL      = stOld.dRHOL;      // Litho Density, calculated (gm/cc)
    stNew.dPECalc    = stOld.dPECalc;    // Photoelectric calc (barns/electron)
    stNew.dPE        = stOld.dPE;        // Photoelectric meas (barns/electron)
    stNew.dU         = stOld.dU;         // Vol Photoelectric Factor (barns/cc)
    stNew.dSigmaCalc = stOld.dSigmaCalc; // Neutron Cross-section, calculated
    stNew.dSigma     = stOld.dSigma;     // Neutron Cross-section, measured
    stNew.dLs        = stOld.dLs;        // Neutron slowing-down length, calc
    stNew.dLd        = stOld.dLd;        // Neutron diffusion length, calculated
    stNew.dCNLCalc   = stOld.dCNLCalc;   // Epithermal CNL porosity, calculated
    stNew.dCNL       = stOld.dCNL;       // Epithermal CNL porosity, measured
    stNew.dCNL_TCalc = stOld.dCNL_TCalc; // Thermal CNL porosity, calculated
    stNew.dCNL_T     = stOld.dCNL_T;     // Thermal CNL porosity, measured
    stNew.dBulk      = stOld.dBulk;      // Bulk Modulus, calculated
    stNew.dShear     = stOld.dShear;     // Shear Modulus, calculated
    stNew.dt_comp    = stOld.dt_comp;    // Delta-T Compressional (us/m), calc
    stNew.dt_shear   = stOld.dt_shear;   // Delta-T Shear (us/m), calculated
    stNew.dPoisson   = stOld.dPoisson;   // Poisson's Ratio, calculated
    stNew.dEPT       = stOld.dEPT;       // EPT Propagation time, calculated

    // Computed Values for Colorlith and Compositional Analysis

    stNew.dRhomaa    = stOld.dRhomaa;    // Grain Density (Rhomaa)
    stNew.dUmaa      = stOld.dUmaa;      // Photoelectric Factor (Umaa)
    stNew.dGR        = stOld.dGR;        // Gamma Ray (API)

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseMineralsListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parseMineralsListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print( parseMineralsStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parseMineralsStruct st )
  {
    if (st != null)
    {
      System.out.println( st.sName  + " " + // Name or Description of parse
                          st.sGroup + " " + // Abbreviation of parse
                          st.sClass + " " + // Lithology Plot Symbol
                          st.sFormula );    // Mineral Formula

      // Schlumberger Measured & Calculated Data

      if ( st.dMolWt > 0.0 )     System.out.print( " Mol wt="   + st.dMolWt );
      if ( st.dRHOB > 0.0 )      System.out.print( " Rhob="     + st.dRHOB );
      if ( st.dRHOE > 0.0 )      System.out.print( " Rhoe="     + st.dRHOE );
      if ( st.dRHOL > 0.0 )      System.out.print( " Rhol="     + st.dRHOL );
      if ( st.dPECalc > 0.0 )    System.out.print( " PE calc="  + st.dPECalc );
      if ( st.dPE > 0.0 )        System.out.print( " PE="       + st.dPE );
      if ( st.dU > 0.0 )         System.out.print( " U="        + st.dU );
      if ( st.dSigmaCalc > 0.0 ) System.out.print( " Sigma calc=" + st.dSigmaCalc );
      if ( st.dSigma > 0.0 )     System.out.print( " Sigma="    + st.dSigma );
      if ( st.dLs > 0.0 )        System.out.print( " N Slow Down=" + st.dLs );
      if ( st.dLd > 0.0 )        System.out.print( " N Diff="   + st.dLd );
      if ( st.dCNLCalc > 0.0 )   System.out.print( " CNL calc=" + st.dCNLCalc );
      if ( st.dCNL > 0.0 )       System.out.print( " CNL="      + st.dCNL );
      if ( st.dCNL_TCalc > 0.0 ) System.out.print( " Therm CNL calc=" + st.dCNL_TCalc );
      if ( st.dCNL_T > 0.0 )     System.out.print( " Therm CNL=" + st.dCNL_T );
      if ( st.dBulk > 0.0 )      System.out.print( " Bulk="     + st.dBulk );
      if ( st.dShear > 0.0 )     System.out.print( " Shear="    + st.dShear );
      if ( st.dt_comp > 0.0 )    System.out.print( " DT Comp="  + st.dt_comp );
      if ( st.dt_shear > 0.0 )   System.out.print( " DT Shear=" + st.dt_shear );
      if ( st.dPoisson > 0.0 )   System.out.print( " PR="       + st.dPoisson );
      if ( st.dEPT > 0.0 )       System.out.print( " EPT="      + st.dEPT );

      System.out.println("\n");

      System.out.println( "Computed: "            +
                          "Rhomaa=" + st.dRhomaa  + " " + // Density
                          "Umaa="   + st.dUmaa    + " " + // Photoelectric factor
                          "GR="     + st.dGR );           // gamma ray
    }
  }
}