/*
 * @parseGrainsTextureUtility.java Version 1.1 08/02/2012
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseGrainsTextureListStruct;
import parse.parseGrainsTextureStruct;

import lith.rock.rockColumnListStruct;
import lith.rock.rockColumnStruct;

/** Class parseGrainsTextureUtility
 *  <p> This Class will provide basic utilities for the rock column parser
 *      data structures.
 *
 *  @version 1.1 08/02/2012
 *  @author  John R. Victorine
 */

public class parseGrainsTextureUtility
{
  public static final int _MAIN = 0;
  public static final int _DESC = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  id   = identifier to retrieve
   * @param  str  = Abbreviation
   * @param  st   = Texture Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(int id, String str, parseGrainsTextureListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
	  switch (id)
	  {
		case _MAIN:
          for (int i=0; i<st.iAbbrev_main; i++)
          {
            if (str.equals(st.sAbbrev_main[i][0].toLowerCase()))
            {
              word = new String( st.sAbbrev_main[i][1] );
            }
          }
          break;

		case _DESC:
          for (int i=0; i<st.iAbbrev_desc; i++)
          {
            if (str.equals(st.sAbbrev_desc[i][0].toLowerCase()))
            {
              word = new String( st.sAbbrev_desc[i][1] );
            }
          }
          break;
	  }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseGrainsTextureListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method whereIsGrain()
   *  <p> This method will determine if a grain/texture is present and where in the array
   *      it then gets the position number.
   *  @param  words    = array of words.
   *  @param  stGrains = Parse Grains & Texture Data List Structure
   *  @return loc      = location in the array of the main texture term.
   *

  public static int whereIsGrain( String words[], parseGrainsTextureListStruct st )
  {
	int    i,j;
	String str   = "";
	String sKey  = "";
	String sAbrv = "";
	int    loc   = -1;
	int    ikey  = -1;
	int    iabrv = -1;

	if ((words != null) && (st != null))
	{
	  if (words.length > 0)
	  {
		for (i=0; i<words.length; i++)
		{
		  str = words[i].toLowerCase();

     	  if (ikey > -1)
		  {
		    for (j=0; j<st.iKeywords; i++)
		    {
			  if (st.sKeywords[j][1].equals("G"))
			  {
  		  	    sKey = st.sKeywords[j][0].toLowerCase();

			    if (sKey.equals(str))
			    {
				  ikey = i;
				}
			  }
			}
		  }

		  if (iabrv > -1)
		  {
            for (j=0; j<st.iAbbrev_main; j++)
            {
			  sKey = st.sAbbrev_main[j][0].toLowerCase();

              if (str.equals( sKey) )
              {
                iabrv = i;
//System.out.println(iabrv+" "+str+" "+sKey);
			  }
            }
          }
		}
//System.out.println(iabrv+" "+ikey);

		if (iabrv < ikey)
		  loc = iabrv;
		else
		  loc = ikey;
	  }
	}

	return (loc);
  }
*/
  /** Method getTextures()
   *  <p> This method will retrieve the textures from the word array.
   *  @param  loc      = location of the Grain/Texture Main Term.
   *  @param  words    = array of words.
   *  @param  stGrains = Parse Grains & Texture Data List Structure
   */

  public static String[] getTextures( int loc,
                                      String words[],
                                      parseGrainsTextureListStruct st )
  {
	int    i,j;
    int    iTotal   = 8;
    String sWords[] = { "", "", "", "", "", "", "", "" };
	String str      = "";
    String sKey     = "";
    int    iStart   = 0;
    int    iEnd     = 0;
    int    icnt     = 0;

	if ((words != null) && (st != null))
	{
	  if (words.length > 0)
	  {
		for (i=0; i<words.length; i++)
		{
		  str = new String(words[i].toLowerCase());

          for (j=0; j<st.iAbbrev_desc; j++)
          {
		    sKey = new String(st.sAbbrev_desc[j][0].toLowerCase());

            if (str.equals(sKey) )
            {
              words[i] = new String(st.sAbbrev_desc[j][1]);
//System.out.print(	loc + " " +i +" "+words[i]);
		    }
		  }
//System.out.println("");
	    }

		for (i=0; i<words.length; i++)
		{
		  str = words[i].toLowerCase();

		  for (j=0; j<st.iKeywords; j++)
		  {
			if (st.sKeywords[j][1].equals("M"))
			{
  		      sKey = st.sKeywords[j][0].toLowerCase();

              if (i < words.length-1)
              {
			    if (sKey.equals(str))
			    {
                  words[i]   = new String(words[i] + " " + words[i+1]);
                  words[i+1] = new String("");
			    }
			  }
			}
		  }
		}

        if ((loc-7) > -1)
  	      iStart = loc-7;

  	    iEnd = words.length;
  	    if ((loc+7) < words.length)
  	      iEnd = loc + 7;

		for (i=iStart; i<iEnd; i++)
		{
		  str = words[i].toLowerCase();

		  for (j=0; j<st.iKeywords; j++)
		  {
			if (st.sKeywords[j][1].equals("D"))
			{
  		      sKey = st.sKeywords[j][0].toLowerCase();

              if (sKey.equals(str))
              {
				if (icnt < iTotal)
				{
				  sWords[icnt] = new String(str);
				  icnt++;
				}
			  }
			}
		  }
		}
	  }
	}

	return (sWords);
  }

  /** Method mapTexture()
   *  <p> This method will take the list of words and map the grain/texture to
   *      the grain column.
   *  @param sWords  = array of grain descriptors
   *  @param igrains = array of grain location indicator
   */

  public static int[] mapTexture( String sWords[] )
  {
	int igrains[]  = { -1, -1, -1, -1, -1, -1, -1, -1 };

    if (sWords != null)
    {
      for (int i=0; i<sWords.length; i++)
      {
		for (int j=0; j<lith.rock.rockColumnStruct._TOTAL; j++)
		{
		  if (sWords[i].equals(lith.rock.rockColumnStruct.MAP_STR[j]))
		  {
			igrains[j] = 1;
		  }
		}
      }
	}

	return (igrains);
  }

  /** Method whereIsGrain()
   *  <p> This method will determine if a grain/texture is present and where in the array
   *      it then gets the position number.
   *  @param  words    = array of words.
   *  @param  stGrains = Parse Grains & Texture Data List Structure
   *  @return loc      = location in the array of the main texture term.
   */

  public static int whereIsGrain( String words[], parseGrainsTextureListStruct st )
  {
	int loc = -1;
	int j,k,m;
	int iFound = 0;

	for (k=0; k<words.length; k++)
	{
	  if (iFound == 0)
	  {
		for (j=0; j<st.iKeywords; j++)
		{
		  if (st.sKeywords[j][1].equals("G"))
		  {
            if (st.sKeywords[j][0].toLowerCase().equals(words[k].toLowerCase()))
            {
		      loc = k;
		      iFound = 1;
		    }
		  }
	    }

	    if (iFound == 0)
	    {
	      for (m=0; m<st.iAbbrev_main; m++)
	      {
            if (st.sAbbrev_main[m][0].toLowerCase().equals(words[k].toLowerCase()))
            {
		      loc = k;
		      iFound = 1;
		    }
		  }
		}
	  }
	}

    return (loc);
  }

  /** Method fillgrains()
   *  <p> This method will fill the grains so there are no spaces between grainsize
   *  @param  igrains = array of grain size indicators
   *  @return igrains = array of grain size indicators
   */

  public static int[] fillgrains( int igrains[] )
  {
	int min = -1;
	int max = -1;
	int ifound = 0;

	if (igrains != null)
	{
	  for (int i=0; i<igrains.length; i++)
	  {
		if (ifound == 0)
		{
		  if (igrains[i] > -1)
		  {
			min = i;
			max = i;
			ifound = 1;
		  }
		}
		else
		{
		  if (igrains[i] > -1)
		  {
			if (max < i)
			  max = i;
		  }
		}
	  }

	  if ((max > min) && (min != -1) && (max != -1))
	  {
		for (int i=min; i<=max; i++)
		{
		  igrains[i] = 1;
		}
	  }
	}

	return (igrains);
  }

  /** Method getTexture()
   *  <p> This method will retrieve the textures from the word array and force
   *      by lithology name.
   *  @param  st       = rock texture data structure
   *  @param  words    = array of words.
   *  @param  stGrains = Parse Grains & Texture Data List Structure
   *  @return st       = rock texture data structure
   */

  public static rockColumnStruct getTexture( rockColumnStruct stRock,
                                             String words[],
                                             parseGrainsTextureListStruct st )
  {
	int    j,k,m;
	int    iFound = 0;
	int    iAbbrv = 0;
	String sFound = "";

    if ((stRock != null) &&  (st != null))
    {
	  for (k=0; k<words.length; k++)
	  {
	    if (iFound == 0)
	    {
	  	  for (j=0; j<st.iKeywords; j++)
		  {
		    if (st.sKeywords[j][1].equals("G"))
		    {
              if ((st.sKeywords[j][0].toLowerCase().equals(words[k].toLowerCase())) &&
                  ((!st.sKeywords[j][0].toLowerCase().equals("crystalline")) &&
                   (!st.sKeywords[j][0].toLowerCase().equals("crystals")) &&
                   (!st.sKeywords[j][0].toLowerCase().equals("sucrose")) &&
                   (!st.sKeywords[j][0].toLowerCase().equals("grain"))))
              {
//		      System.out.println("G: "+words[k]);
		        sFound = new String(words[k]);
		        iFound = 1;
		      }
		    }
	      }

	      if (iFound == 0)
	      {
	        for (m=0; m<st.iAbbrev_main; m++)
	        {
              if ((st.sAbbrev_main[m][0].toLowerCase().equals(words[k].toLowerCase())) &&
                  ((!st.sAbbrev_main[m][1].toLowerCase().equals("crystalline")) &&
                   (!st.sAbbrev_main[m][1].toLowerCase().equals("crystals")) &&
                   (!st.sAbbrev_main[m][1].toLowerCase().equals("sucrose")) &&
                   (!st.sAbbrev_main[m][1].toLowerCase().equals("grain"))))
              {
//		      System.out.println("Abbrev: "+words[k]);
		        sFound = new String(words[k]);
		        iFound = 1;
		        iAbbrv = 1;
		      }
		    }
		  }
	    }
	  }

	  if (iAbbrv == 1)
	  {
	    for (m=0; m<st.iAbbrev_main; m++)
	    {
          if (st.sAbbrev_main[m][0].toLowerCase().equals(sFound.toLowerCase()))
          {
	  	    sFound = new String(st.sAbbrev_main[m][1].toLowerCase());
//		  System.out.println("Abbrev: "+sFound + " " + stRock.sPrimary);
		  }
	    }
  	  }

      if (stRock.sPrimary != null)
      {
		if (stRock.sPrimary.equals("CLY"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_1;
		if (stRock.sPrimary.equals("SH"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_2;
		if (stRock.sPrimary.equals("SLT"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;

		if (stRock.sPrimary.equals("DOL-MDST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;
	    if (stRock.sPrimary.equals("DOL-WKST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_1;
	    if (stRock.sPrimary.equals("DOL-PKST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_2;
	    if (stRock.sPrimary.equals("DOL-GRST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_3;
	    if (stRock.sPrimary.equals("DOL-BDST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_E_CRS;

		if (stRock.sPrimary.equals("CO3-MDST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;
	    if (stRock.sPrimary.equals("CO3-WKST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_1;
	    if (stRock.sPrimary.equals("CO3-PKST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_2;
	    if (stRock.sPrimary.equals("CO3-GRST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_3;
	    if (stRock.sPrimary.equals("CO3-BDST"))
	        stRock.iTexture = lith.rock.rockColumnStruct._T_E_CRS;

        if (stRock.sPrimary.equals("DOL"))
        {
          if (sFound.equals("mudstone"))
          {
	        stRock.sPrimary = new String("DOL-MDST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;
	      }
	      else if (sFound.equals("wackestone"))
          {
	        stRock.sPrimary = new String("DOL-WKST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_1;
	      }
	      else if (sFound.equals("packstone"))
          {
	        stRock.sPrimary = new String("DOL-PKST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_2;
	      }
	      else if (sFound.equals("grainstone"))
          {
	        stRock.sPrimary = new String("DOL-GRST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_3;
	      }
	      else if (sFound.equals("boundstone"))
          {
	        stRock.sPrimary = new String("DOL-BDST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_E_CRS;
	      }
        }

        if (stRock.sPrimary.equals("CO3"))
        {
          if (sFound.equals("mudstone"))
          {
	        stRock.sPrimary = new String("CO3-MDST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;
	      }
	      else if (sFound.equals("wackestone"))
          {
	        stRock.sPrimary = new String("CO3-WKST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_1;
	      }
	      else if (sFound.equals("packstone"))
          {
	        stRock.sPrimary = new String("CO3-PKST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_2;
	      }
	      else if (sFound.equals("grainstone"))
          {
	        stRock.sPrimary = new String("CO3-GRST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_3;
	      }
	      else if (sFound.equals("boundstone"))
          {
	        stRock.sPrimary = new String("CO3-BDST");
	        stRock.iTexture = lith.rock.rockColumnStruct._T_E_CRS;
	      }
        }

        if (stRock.sPrimary.equals("CO3-LS_DOL"))
        {
          if (sFound.equals("mudstone"))
          {
	        stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_3;
	      }
	      else if (sFound.equals("wackestone"))
          {
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_1;
	      }
	      else if (sFound.equals("packstone"))
          {
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_2;
	      }
	      else if (sFound.equals("grainstone"))
          {
	        stRock.iTexture = lith.rock.rockColumnStruct._T_GRN_3;
	      }
	      else if (sFound.equals("boundstone"))
          {
	        stRock.iTexture = lith.rock.rockColumnStruct._T_E_CRS;
	      }
        }

        if (stRock.sPrimary.equals("CO3-3"))
          stRock.iTexture = lith.rock.rockColumnStruct._T_MUD_2;
	  }
    }


	return (stRock);
  }

  /* ======================================================================== *
   * --------------------------- PARSE METHODS ------------------------------ *
   * ======================================================================== */

  /** Method parseColumn()
   * <p> This method will parse the text string into rock column lithology
   * @param  st         = rock texture data structure
   * @param  sData      = array of rock column string phrase or word
   * @param  stGrains   = Parse Grains & Texture Data List Structure
   * @return st         = rock texture data structure
   */

  public static  rockColumnStruct parseColumn( rockColumnStruct st,
                                               String sData[],
                                               parseGrainsTextureListStruct stGrains )
  {
	int    i,j;
	int    loc      = -1;
    String sWords[] = null;
    int    iFound   = 0;

    if ((st != null) && (sData != null) && (stGrains != null))
    {
      // Where is the Grain/Texture String in the array of words.

      loc = whereIsGrain( sData, stGrains );

//if ((st.depthStart>4039) && (st.depthEnd < 4100))
//{
//System.out.println(st.depthStart+" "+st.depthEnd+" "+loc);
//}

      if (loc > -1)
      {
        sWords     = getTextures( loc, sData, stGrains );
        st.igrains = mapTexture( sWords );

		if (st.sPrimary.equals("CLY"))
	        st.igrains[lith.rock.rockColumnStruct._CRYPTO] = 1;
		if (st.sPrimary.equals("SH"))
	        st.igrains[lith.rock.rockColumnStruct._CRYPTO] = 1;
		if (st.sPrimary.equals("SLT"))
	        st.igrains[lith.rock.rockColumnStruct._MICRO] = 1;

        st.igrains = fillgrains( st.igrains );
//if ((st.depthStart>4039) && (st.depthEnd < 4100))
//{
//for (int p=0; p<8; p++)
//System.out.print(st.igrains[p]+", ");
//System.out.println("");
//}
		for (j=0; j<lith.rock.rockColumnStruct._TOTAL; j++)
		{
		  if (iFound == 0)
		  {
			if (st.igrains[j] > 0)
			{
			  st.iTexture = lith.rock.rockColumnStruct.MAP[j];
			  iFound++;
			}
		  }
		}
	  }

	  if (st.iTexture == -1)
	  {
		for (i=0; i<stGrains.iCount; i++)
		{
		  if (st.sPrimary.equals(stGrains.stItem[i].sID))
		  {
			st.iTexture = stGrains.stItem[i].iWeather;
		  }
		}

        if (st.iTexture > -1)
        {
		  st.igrains[lith.rock.rockColumnStruct.MAP2[st.iTexture-1]] = 1;
		}
	  }
	}

    return (st);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  iRows  = The total number of lithologies
   * @param  iOrder = the order of plot
   * @param  sID    = the lithology ID
   * @return st     = sorted data list structure
   */

  public static String[] bubbleSort( int iRows, int iOrder[], String sID[] )
  {
    boolean swappedOnPrevRun = true;
    int     iTemp = 0;
    String  sTemp = "";

    if (sID != null)
    {
      if (iRows > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<iRows-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(iOrder[i] > iOrder[i+1])
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              sTemp = new String(sID[i]);
              iTemp = iOrder[i];

              // set element i+1 to where i used to be

              sID[i]    = new String(sID[i+1]);
              iOrder[i] = iOrder[i+1];

              // release the old i from temp into i+1 slot

              sID[i+1]    = new String(sTemp);
              iOrder[i+1] = iTemp;
            }
          }
        }
      }
    }

    return (sID);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseGrainsTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified()
   *  <p> This method will determine if the string is part of descriptor set
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified( String str, parseGrainsTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if (sWord.equals(st.sModifier[i][1].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified()
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseGrainsTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseGrainsTextureListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseGrainsTextureListStruct copyList(
                                             parseGrainsTextureListStruct stOld )
  {
    parseGrainsTextureListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseGrainsTextureListStruct();

      // Abbreviations

      stNew.iAbbrev_main = stOld.iAbbrev_main;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev_main = new String[stNew.iAbbrev_main][2];

      for (i=0; i<stNew.iAbbrev_main; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev_main[i][j] = new String(stOld.sAbbrev_main[i][j]);
      }

      stNew.iAbbrev_desc = stOld.iAbbrev_desc;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev_desc = new String[stNew.iAbbrev_desc][2];

      for (i=0; i<stNew.iAbbrev_desc; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev_desc[i][j] = new String(stOld.sAbbrev_desc[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Lithology Symbols

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseGrainsTextureStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseGrainsTextureStruct copy(parseGrainsTextureStruct stOld)
  {
    parseGrainsTextureStruct stNew = new parseGrainsTextureStruct();
    int i=0;
    int j=0;

    stNew.sID      = new String( stOld.sID );    // ID for Lithology Symbol
    stNew.iGroup   = stOld.iGroup;               // Group Number
    stNew.sGroup   = new String( stOld.sGroup ); // The group name
    stNew.sName    = new String( stOld.sName );  // Name
    stNew.sAbrev   = new String( stOld.sAbrev ); // Abbreviation

    stNew.sCycle   = new String( stOld.sCycle ); // sed (sedimentary), etc.
    stNew.iWeather = stOld.iWeather;             // # times horizontal

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print()
   * <p> This method will print the lithology symbols structure
   * @param  st = Lithology Symbols data list structure
   */

  public static void print( parseGrainsTextureListStruct st )
  {
    System.out.println("\n ------------ Abbreviation-Main ------------- \n");

    for (int i=0; i<st.iAbbrev_main; i++)
    {
	  System.out.println(st.sAbbrev_main[i][0]+" "+st.sAbbrev_main[i][1]);
	}

    System.out.println("\n ------------ Abbreviation-Descriptor ------------- \n");

    for (int i=0; i<st.iAbbrev_desc; i++)
    {
	  System.out.println(st.sAbbrev_desc[i][0]+" "+st.sAbbrev_desc[i][1]);
	}

    System.out.println("\n ------------ Modifiers ------------- \n");

    for (int i=0; i<st.iModifiers; i++)
    {
	  System.out.println(st.sModifier[i][0]+" "+st.sModifier[i][1]+" "+st.sModifier[i][2]);
	}

    System.out.println("\n ------------ Keywords ------------- \n");

    for (int i=0; i<st.iKeywords; i++)
    {
	  System.out.println(st.sKeywords[i][0]+" "+st.sKeywords[i][1]);
	}

    System.out.println("\n ------------ Lithologies ------------- \n");
    for (int i=0; i<st.iCount; i++)
    {
      print( st.stItem[i] );
    }
  }

  /** Method print()
   * <p> This method will print the symbols data structure
   * @param  st = Lithology symbols data structure
   */

  public static void print( parseGrainsTextureStruct st )
  {
    int i = 0;

    if (st != null)
    {
      System.out.println(st.sID    + " " + // ID for Lithology Symbol
                         st.iGroup + " " + // Group Number
                         st.sGroup + " " + // The group the lithology belongs
                         st.sName  + " " + // Name or Description of lithology
                         st.sAbrev + " " + // Abbreviation of lithology
                         st.iWeather );    // Weathering
    }
  }
}

/*
 *  @version 1.1 08/02/2012
 *  @author  John Victorine
 */
