/*
 * @parseGeneraUtility.java Version 1.1 10/19/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseFossilListStruct;
import parse.parseFossilStruct;

import iqstrat.iqstratRemarkListStruct;

import rock.fossil.fossilListStruct;
import rock.fossil.fossilStruct;

import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;

/** Class parseGeneraUtility
 *  <p> This Class will provide basic utilities for retrieving
 *      the Genera & species from measured section reports
 *
 *  @version 1.1 10/19/2011
 *  @author  John R. Victorine
 */

public class parseGeneraUtility
{
  public static final int    iAbbrev = 14;
  public static final String abbrev[][] =
  {
    { "(s)",      "sparse" },
    { "(c)",      "common" },
    { "(a)",      "abundant" },
    { "(p)",      "profuse" },

    { "many",     "common" },
    { "numerous", "abundant" },
    { "rare",     "sparse" },
    { "few",      "sparse" },

    { "fragment", "sparse" },
    { "fragments","sparse" },

    { "sparse",   "sparse" },
    { "common",   "common" },
    { "abundant", "abundant" },
    { "profuse",  "profuse" },
  };

  public static final int _NONE   = 0;
  public static final int _GENERA = 1;
  public static final int _NUMBER = 2;

  /** Method parse()
   * <p> This method will parse the Measured Section into fossil list
   *     data structure
   * @param  stList    = Remarks List Data Structure
   * @param  stKSBio   = Kansas Fossils Scientific Name
   * @return stFossil  = Rock fossil Data List Structure
   */

  public static bioStratListStruct parse(
                    iqstratRemarkListStruct stList,
                    bioStratListStruct      stKSBio )
  {
    bioStratListStruct stFossil   = null;
    String             sKEY       = cmn.cmnString.UniqueName();
    String             str        = "";
    String             tokens[]   = null;
    String             sDelimiter = new String("[,;:|]+");

    if (stList != null)
    {
      stFossil = new bioStratListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str      = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str      = new String( str.replace('\n', ' ') );
        str      = new String( str.replaceAll(" and ", ", ") );

        tokens   = str.split( sDelimiter );

        stFossil = parseFossils( stFossil,
                                 i,
                                 stList.stItem[i].depthStart,
                                 stList.stItem[i].depthEnd,
                                 tokens,
                                 stKSBio );

        tokens   = null;
        str      = new String("");
      }

      stList.delete();
      stList = null;
    }

    stFossil = horizon.bio.bioStratUtility.bubbleSort( stFossil );

//horizon.bio.bioStratUtility.printSpecies( stFossil );
    return (stFossil);
  }

  /** Method parseFossils()
   * <p> This method will parse the text string into rock fossil
   * @param  stFossil     = the fossil list data structure
   * @param  iBed         = Bed Number just to make sure the species KEY is unique
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  sData        = the fossil string phrase or word
   * @param  stKSBio      = Kansas Fossils Scientific Names Lookup
   * @return st           = rock fossil data structure
   */

  public static  bioStratListStruct parseFossils(
	                 bioStratListStruct stFossil,
	                 int    iBed,
                     double depthStart,
                     double depthEnd,
                     String sData[],
                     bioStratListStruct stKSBio )
  {
	int i, j, k, m;
    String         str        = "";
    String         tokens[]   = null;
    String         sDelimiter = new String("[ ]+");
    bioStratStruct stText     = null;

    int            iContinue  = 0;
    int            iNext      = 0;
    int            iSpecies   = -1;

    String         sKEY       = "0";
    String         species    = "";
    int            iAbundance = horizon.bio.bioStratStruct._NONE;
    String         sTerms[]   = null;
    int            iTerms[]   = null;
    String         sGenera    = "";
    String         sAbbrev    = "";
    int            iPosition  = -1;

	if (stKSBio != null)
	{
      for (i=0; i<sData.length; i++)
      {
        sKEY = new String( cmn.cmnString.UniqueName() +
                           "_" + iBed + "_" + stFossil.iCount );

	    str    = new String( sData[i].trim() );
        tokens = str.split( sDelimiter );

        // BUILD and INITIALIZE the Group of text being tested

        sTerms = new String[tokens.length];
        iTerms = new int[tokens.length];

        for (j=0; j<tokens.length; j++)
        {
	  	  sTerms[j] = new String( tokens[j].trim() );
	 	  iTerms[j] = _NONE;
        }

        // CHECK for abundance and assign a value if it exists

        for (j=0; j<sTerms.length; j++)
        {
		  for (k=0; k<iAbbrev; k++)
		  {
		    // Search and replace any abbreviation that represents a
		    // fossil quantity term and set the Abundance inticator.

		    if (abbrev[k][0].equals(sTerms[j].toLowerCase()))
		    {
		  	  sTerms[j]  = new String( abbrev[k][1] );
			  iTerms[j]  = _NUMBER;
			  iAbundance = getAbundance( sTerms[j] );
		    }
		  }
	    }

        // REMOVE '(', ')' and '?' from each String

        for (j=0; j<sTerms.length; j++)
        {
		  sTerms[j] = removeChars( sTerms[j] );
        }

	    // CHECK if any of the terms are in the Kansas Fossil Genera List

	    iContinue  = 0;
	    iNext      = 0;
	    iPosition  = -1;

	    for (j=0; j<sTerms.length; j++)
	    {
	      if (sTerms[j].equals(sAbbrev))
	      {
			sTerms[j] = new String(sGenera);
		  }

	  	  // IF this term a Kansas Fossil with a recognizable Genera Term THEN

		  if (horizon.bio.bioStratUtility.exists(sTerms[j], stKSBio))
		  {
            if (iPosition == -1) { iPosition  = j; }
    	    iTerms[j]  = _GENERA;
    	    sGenera    = new String(sTerms[j]);
    	    sAbbrev    = new String(sGenera.substring(0,1) + ".");

	        iNext  = horizon.bio.bioStratUtility.getPosition( sTerms[j], stKSBio );
	        if (iNext > -1)
	        {
	          stText = horizon.bio.bioStratUtility.copy( stKSBio.stItem[iNext] );
            }

	        // CHECK if an abundance has been set.

            iNext = 0;

	        for (k=0; k<sTerms.length; k++)
	        {
		  	  if (iTerms[k] == _NUMBER)
		  	    iNext = iAbundance;
		    }

		    // IF the abundance has not been set THEN set to "Present".

		    if (iNext == horizon.bio.bioStratStruct._NONE)
		      iAbundance = getAbundance("present");
		  }
	    }

        iContinue  = 0;
        for (k=0; k<sTerms.length; k++)
        {
		  if (iTerms[k] == _GENERA) { iContinue++; }
		}

        if (sTerms.length > iPosition+3) { iContinue = 0; }

	    // IF this group of words contains a Genera Term THEN

	    if (iContinue == 1)
	    {
		  iContinue = 0;
		  iSpecies  = -1;

          for (k=0; k<sTerms.length; k++)
          {
		    if (iTerms[k] == _GENERA)
		    {
		  	  species = new String( sTerms[k] );

		  	  m = k+1;
		  	  if (m < sTerms.length)
		  	  {
  		        if ((iTerms[m] != _NUMBER) &&
  		            (sTerms[m].length() > 1) &&
  		            (!sTerms[m].equals("and")) &&
  		            (!sTerms[m].equals("rare"))   &&
  		            (!sTerms[m].equals("spines")) && (!sTerms[m].equals("spine")) &&
  		            (!sTerms[m].equals("stems"))  && (!sTerms[m].equals("stem"))  &&
  		            (!sTerms[m].equals("plates")) && (!sTerms[m].equals("plate"))  &&
  		            (!sTerms[m].equals("hooks"))  && (!sTerms[m].equals("hook"))  &&
  		            (!sTerms[m].equals("bones"))  && (!sTerms[m].equals("bone"))  &&
  		            (!sTerms[m].equals("teeth"))  && (!sTerms[m].equals("tooth")) &&
  		            (!sTerms[m].equals("scales")) && (!sTerms[m].equals("scale")) &&
  		            (!sTerms[m].equals("sp"))     && (!sTerms[m].equals("sp."))   &&
  		            (!sTerms[m].equals("vertebra")) &&
		            (!cmn.cmnString.isNumeric(sTerms[m])))
		        {
			      iContinue = 1;
			      species = new String( species + " " + sTerms[m] );
		        }
		      }
		    }
		  }

		  if (iContinue == 0)
		  {
  		    species = new String( species + " (sp)" );
		  }

		  if (species.length() > 0)
		  {
		    iSpecies = horizon.bio.bioStratUtility.isSpecies( species, stFossil );

		    if (iSpecies > -1)
		    {
			  stFossil.stItem[iSpecies] =
			    horizon.bio.bioStratUtility.addDepth(
			  	  stFossil.stItem[iSpecies],
                  cmn.cmnString.UniqueName() +
                    "_" + iBed + "_" + stFossil.stItem[iSpecies].iRows,
                  iAbundance,
                  Math.round(depthStart*100.0)/100.0,
                  Math.round(depthEnd*100.0)/100.0 );
		    }
		    else
		    {
              stText.sKEY          = new String( sKEY );
              stText.sName         = new String( species );

              stText.iRows         = 1;
              stText.sKEYa         = new String[1];
              stText.iAbundance    = new int[1];
              stText.depthStart    = new double[1];
              stText.depthEnd      = new double[1];

              stText.sKEYa[0]      = new String( sKEY );
              stText.iAbundance[0] = iAbundance;
              stText.depthStart[0] = Math.round(depthStart*100.0)/100.0;
              stText.depthEnd[0]   = Math.round(depthStart*100.0)/100.0;
              if (depthEnd > depthStart)
                stText.depthEnd[0] = Math.round(depthEnd*100.0)/100.0;

			  stFossil = horizon.bio.bioStratUtility.add( stText, stFossil );
		    }
		  }
	    }

        sKEY       = new String("0");
		iAbundance = horizon.bio.bioStratStruct._NONE;
		species    = new String("");

        if (stText != null)
      	  stText.delete();
      	stText = null;

	    sTerms = null;
	    iTerms = null;
	    tokens = null;
	  }
    }

    return (stFossil);
  }

  /** Method removeChars()
   * <p> This method will remove the characer from the string
   * @param  str  = The string to be converted
   * @return sNew = The text being converted
   */

  public static String removeChars( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if ((c[i] != '(') && (c[i] != '?') && (c[i] != ')'))
        {
          sNew = new String(sNew + c[i]);
	    }
      }
    }

    return (sNew);
  }

  /** Method getAbundance()
   *  <p> This method will determine if the term represents a quantity term
   *  @param  sTerm = the term being tested
   *  @return iNo   = the number of fossils
   */

  public static int getAbundance( String sTerm )
  {
	int iNo = horizon.bio.bioStratStruct._NONE;

	if (sTerm.equals("sparse"))
	  iNo = horizon.bio.bioStratStruct._SPARSE;
    if (sTerm.equals("present"))
	  iNo = horizon.bio.bioStratStruct._PRESENT;
    if (sTerm.equals("common"))
	  iNo = horizon.bio.bioStratStruct._COMMON;
    if (sTerm.equals("abundant"))
	  iNo = horizon.bio.bioStratStruct._ABUNDANT;
    if (sTerm.equals("profuse"))
	  iNo = horizon.bio.bioStratStruct._PROFUSE;

	return (iNo);
  }
}