/*
 * @parseFossilUtility.java Version 1.1 10/14/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseFossilListStruct;
import parse.parseFossilStruct;

//import kgs.kgsMeasSectListStruct;
import iqstrat.iqstratRemarkListStruct;

import rock.fossil.fossilListStruct;
import rock.fossil.fossilStruct;

import horizon.bio.bioStratListStruct;

/** Class parseFossilUtility
 *  <p> This Class will provide basic utilities for the fossils parser
 *      data structures.
 *
 *  @version 1.1 10/14/2009
 *  @author  John R. Victorine
 */

public class parseFossilUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = Fossils Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parseFossilListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Fossils Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseFossilListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getFossilsStruct()
   * <p> This method will return the fossil data structure
   * @param  iLevel = MAIN (Fossil) or SUB (Fossil Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Fossil Name or the Descriptor Name
   * @param  st     = The Parse fossil List Data Structure
   * @return stT    = parse fossil data structure
   */

  public static parseFossilStruct getFossilsStruct(
      int iLevel, int iRow, String str, parseFossilListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parseFossilStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getFossilsRow()
   * <p> This method will return the fossil data structure
   * @param  iLevel = MAIN (Fossil) or SUB (Fossil Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Fossil Name or the Descriptor Name
   * @param  st     = The Parse fossil List Data Structure
   * @return stT    = parse fossil data structure
   */

  public static int getFossilsRow(
           int iLevel, int iRow, String str, parseFossilListStruct st )
  {
    int iFossils = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              iFossils = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              iFossils = i;
            }
          }
          break;
      }
    }

    return (iFossils);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into fossil list
   *     data structure
   * @param  stList    = Remarks List Data Structure
   * @param  stFossils = Fossils Thesaurus
   * @param  stBio     = Fossils Scientific Name
   * @param  stKSBio   = Kansas Fossils Scientific Name
   * @return stFossil  = Rock fossil Data List Structure
   */

  public static fossilListStruct parse(
                    iqstratRemarkListStruct stList,
                    parseFossilListStruct   stFossils,
                    bioStratListStruct      stBio,
                    bioStratListStruct      stKSBio )
  {
    fossilListStruct stFossil   = null;
    String           sKEY       = cmn.cmnString.UniqueName();
    String           str        = "";
    String           tokens[]   = null;
//    String           sDelimiter = new String("[,-;.:= ]+");
//    String            sDelimiter = new String("[ /,;:.=-]+");
    String            sDelimiter = new String("[ /,;:.=-?]+");
    fossilStruct     stText     = null;

    if (stList != null)
    {
      stFossil = new fossilListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parseFossils( stList.stItem[i].depthStart,
                               stList.stItem[i].depthEnd,
                               tokens, stFossils, stBio, stKSBio );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stFossil    = rock.fossil.fossilUtility.add( stText, stFossil );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stFossil);
  }

  /** Method parseFossils()
   * <p> This method will parse the text string into rock fossil
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  sData        = the fossil string phrase or word
   * @param  stFossils    = Fossils Thesaurus
   * @param  stBio        = Fossils Scientific Name
   * @param  stKSBio      = Kansas Fossils Scientific Name
   * @return st           = rock fossil data structure
   */

  public static  fossilStruct parseFossils(
                     double depthStart, double depthEnd, String sData[],
                     parseFossilListStruct stFossils,
                     bioStratListStruct    stBio,
                     bioStratListStruct    stKSBio )
  {
    int i, j, k, m, m0, m1, m2, p;
    fossilStruct    st         = null;
    int                  iCount     = 0;
    String sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int iRow[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int iOrder[] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String               sTemp[]     = {"", ""};
    int                  iFossils  = -1;
    int                  iDescrip    = -1;
    int                  iContinue   = 0;
    int                  iNext       = 0;
    parseFossilStruct stFossil = null;
    String               sID         = "";
    String               sName       = "";
    String               sMod        = "";
    String               sEnv        = "";
    int                  iNew        = 0;
    int                  iFound      = 0;

    if (stFossil != null)
      stFossil.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stFossils.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stFossils.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stFossils.sAbbrev[j][1]);
      }

      sData[i] = new String(
		  horizon.bio.bioStratUtility.convert(sData[i], stBio) );

      sData[i] = new String(
		  horizon.bio.bioStratUtility.convert(sData[i], stKSBio) );
    }

    // Find Fossils Words up to 25

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stFossils ))
      {
        if (iCount < 25)
        {
          sTemp             = getKeyword( sData[j], stFossils );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iOrder[iCount]    = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stFossils ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Is there a Environment Descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("E"))
      {
        if (sWords[i][0].equals("fresh water"))
          sEnv = new String( "Fresh Water" );
        if (sWords[i][0].equals("brackish water"))
          sEnv = new String( "Brackish Water" );
        if (sWords[i][0].equals("marine"))
          sEnv = new String( "Marine" );
      }
    }

    // Identify the main fossil type

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("F"))
      {
        iFossils  = getFossilsRow( _MAIN, -1, sWords[i][0], stFossils );

        if (iFossils > -1)
        {
          iContinue    = 1;
          iRow[i]      = iFossils;
          sWords[i][1] = new String("X");
        }
      }
    }

    // Modify fossil with a descriptor

    if (iContinue == 1)
    {
      for (m=0; m<iCount; m++)
      {
        m0 = m+1;
        m1 = m+2;
        if (m0 >= iCount) m0 = iCount-1;
        if (m1 >= iCount) m1 = iCount-1;
        m2 = m-1;
        if (m2 < 0) m2 = 0;

        if (iRow[m] > -1)
        {
          stFossil = stFossils.stItem[iRow[m]];

          iFound = 0;
          if (sWords[m0][0].equals("fragments"))
          {
            iFound = 1;
            sMod = new String( "Broken" );
            sWords[m0][1] = new String("X");
          }

          sID      = new String( stFossil.sID);
          sName    = new String( stFossil.sPhrase );

          // Add a Modification to the fossil "Abundant", "Rare" or "Broken"

          for (i=0; i<m0; i++)
          {
            if (sWords[i][1].equals("A"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0))
              {
                iFound = 1;
                sMod = new String("Abundant");
                sWords[i][1] = new String("X");
              }
            }

            if (sWords[i][1].equals("T"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0))
              {
                iFound = 1;
                sMod = new String("Rare");
                sWords[i][1] = new String("X");
              }
            }

            if (sWords[i][1].equals("B"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0) &&
                  (!sWords[i][0].equals("fragments")))
              {
                iFound = 1;
                sMod = new String( "Broken" );
                sWords[i][1] = new String("X");
              }
            }
          }

          for (i=0; i<m; i++)
          {
            if (sWords[i][1].equals("D"))
            {
              iDescrip = getFossilsRow(
                               _SUB, iRow[m], sWords[i][0], stFossils );

              if (iDescrip > -1)
              {
                sWords[i][1] = new String("X");
                stFossil     = stFossils.stItem[iRow[m]].descriptors[iDescrip];

                sID          = new String( stFossil.sID);
                sName        = new String( stFossil.sPhrase );
              }
            }
          }

          if (iNew == 0)
          {
            iNew = 1;
            st   = new fossilStruct();

            st.sKEY = new String(cmn.cmnString.UniqueName() +
                                 ( (int) depthStart)); // Create a Unique KEY

            st.depthStart   = depthStart;         // Starting Depth
            st.depthEnd     = depthEnd;           // Ending Depth
            st.sEnvironment = new String( sEnv ); // Environment
          }

          if (st != null)
          {
            st    = rock.fossil.fossilUtility.addFossil(sID, sMod, sName, st);

            sID   = new String("");
            sMod  = new String("");
            sName = new String("");
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Fossils Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseFossilListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parseFossilListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Fossils Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseFossilListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseFossilListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseFossilListStruct copyList( parseFossilListStruct stOld )
  {
    parseFossilListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseFossilListStruct();

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Fossils Dictionary

      stNew.iFossil  = stOld.iFossil;
      stNew.stFossil = new parseFossilStruct[stOld.iFossil];

      for (i=0; i<stOld.iFossil; i++)
      {
        stNew.stFossil[i] = copy( stOld.stFossil[i] );
      }

      // Fossils

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseFossilStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseFossilStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseFossilStruct copy( parseFossilStruct stOld )
  {
    int i=0;
    int j=0;
    parseFossilStruct stNew = new parseFossilStruct();

    // Fossils Lithology

    stNew.sID     = new String( stOld.sID );        // String ID for Mask
    stNew.sPhrase = new String( stOld.sPhrase );    // Phrase

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parseFossilStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseFossilListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parseFossilListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print( parseFossilStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parseFossilStruct st )
  {
    if (st != null)
    {
      System.out.println( st.sID     + " <" + // String ID for Mask Symbol
                          st.sPhrase + "> ");

      if (st.descriptors != null)
      {
        System.out.println( " ---- Descriptors ---- ");

        for (int i=0; i<st.iCount; i++)
        {
          print(st.descriptors[i]);
        }

        System.out.println( " ---- End Descriptors ---- ");
      }
    }
  }
}