/*
 * @parseDepEnvUtility.java Version 1.1 11/01/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseDepEnvListStruct;
import parse.parseDepEnvStruct;

import parse.parseTextureListStruct;
import parse.parseColorListStruct;
import parse.parseFossilListStruct;

import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratStruct;

import horizon.env.envListStruct;
import horizon.env.envStruct;
import horizon.bio.bioStratListStruct;

/** Class parseDepEnvUtility
 *  <p> This Class will provide basic utilities for the Depositional Environment parser
 *      data structures.
 *
 *  @version 1.1 11/01/2011
 *  @author  John R. Victorine
 */

public class parseDepEnvUtility
{
  public static final int _MAIN  = 0;
  public static final int _SUB   = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseDepEnvListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getDepEnvStruct()
   * <p> This method will return the texture data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse texture List Data Structure
   * @return stT    = parse texture data structure
   */

  public static parseDepEnvStruct getDepEnvStruct(
      int iLevel, int iRow, String str, parseDepEnvListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parseDepEnvStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sName.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[iRow].descriptors[i].sName.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getDepEnvRow()
   * <p> This method will return the depositional environment data structure
   * @param  iLevel = MAIN or SUB (Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The depositional environment Name or the Descriptor Name
   * @param  st     = The Parse depositional environment List Data Structure
   * @return stT    = parse texture data structure
   */

  public static int getDepEnvRow(
           int iLevel, int iRow, String str, parseDepEnvListStruct st )
  {
    int iDepEnv = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sName.toLowerCase()) )
            {
              iDepEnv = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals( st.stItem[iRow].descriptors[i].sName.toLowerCase()) )
            {
              iDepEnv = i;
            }
          }
          break;
      }
    }

    return (iDepEnv);
  }

  /* ======================================================================== *
   * --------------------------- PARSE METHODS ------------------------------ *
   * ======================================================================== */

  /** Method parse()
   * <p> This method will parse the Remarks data list structure for
   *     Depositional Environment terms.
   * @param  stList    = Remarks List Data Structure
   * @param  stDepEnvs = Depositional Environment Thesaurus
   * @return stDepEnv  = Depositional Environment Data List Structure
   */

  public static envListStruct parse( iqstratRemarkListStruct stList,
                                     iqstratStruct stStruct )
  {
    envListStruct     stDepEnv    = null;
    String            sKEY        = cmn.cmnString.UniqueName();
    String            str         = "";
    String            tokens[]    = null;
    String            tokensG[]   = null;
    String            sDelimiter  = new String("[ /,;:.=-]+");
    String            sDelimiter1 = new String("[,;:|]+");
    envStruct         stText      = null;
    String            sData[]     = null;
    String            sGenera     = "";

    if (stList != null)
    {
      stDepEnv = new envListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str     = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str     = new String( str.replace('\n', ' ') );

        sGenera = new String( str );

        tokens  = str.split( sDelimiter );

        // Search Lithology/Texture Terms

        sData = new String[tokens.length];
        for (int j=0; j<tokens.length; j++)
        {
          sData[j] = new String( tokens[j] );
	    }

        stText = new envStruct();
        stText = parseTexture( stText, sData, stStruct.stParseTexture );

        // Search for Rock Color Terms

        sData = new String[tokens.length];
        for (int j=0; j<tokens.length; j++)
          sData[j] = new String( tokens[j] );

        stText = parseColor( stText, sData, stStruct.stParseColors );

        // Search for General Fossil Terms

        sData = new String[tokens.length];
        for (int j=0; j<tokens.length; j++)
          sData[j] = new String( tokens[j] );

        stText = parseFossils( stText, sData,
                               stStruct.stParseFossils,
                               stStruct.stBio,
                               stStruct.stKSBio );

        // Search for Fossil Genera Terms

        tokensG = sGenera.split( sDelimiter1 );

        stText  = parseGenera( stText, tokensG, stStruct.stKSBio );

        tokensG = null;

        // Search for Depositional Environment

        stText = parseDepEnv( stList.stItem[i].depthStart,
                              stList.stItem[i].depthEnd,
                              stText,
                              stStruct.stDepEnv );
//horizon.env.envUtility.print(stText);

        if (stText != null)
        {
          stText.sKEY     = new String( sKEY + "_" + i );
//          stText.sDescrip = new String( stList.stItem[i].sText );
          stDepEnv        = horizon.env.envUtility.add(stText, stDepEnv );
        }

        tokens = null;
        sData  = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stDepEnv);
  }

  /** Method parseTexture()
   * <p> This method will parse the text string into rock texture
   * @param  st         = the depositional environment data structure
   * @param  sData      = the texture string phrase or word
   * @param  stTextures = Texture Thesaurus
   * @return st         = the depositional environment data structure
   */

  public static  envStruct parseTexture(
                     envStruct st, String sData[],
                     parseTextureListStruct stTextures )
  {
    int i, j, k, m, p;
    int                iCount     = 0;
    int                iTotal     = 10;
    String             sWords[][] = { {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""}};
    String             sTemp[]    = {"", ""};
    int                iTexture  = -1;
    int                iDescrip  = -1;
    int                iContinue = 0;
    int                iNext     = 0;
    parseTextureStruct stTexture = null;
    String             sLithology = "";
    int                iSet       = 0;
    int                iSet_a     = 0;

    if (stTexture != null)
      stTexture.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stTextures.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stTextures.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stTextures.sAbbrev[j][1]);
      }
    }

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseTextureUtility.isKeyword( sData[j], stTextures ))
      {
		iTotal++;
      }
    }

    sWords = new String[iTotal][2];

    for (i=0; i<iTotal; i++)
    {
	  for (j=0; j<2; j++)
	    sWords[i][j] = "";
	}

    // Find Texture Words

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseTextureUtility.isKeyword( sData[j], stTextures ))
      {
        if (iCount < iTotal)
        {
          sTemp             = parse.parseTextureUtility.getKeyword( sData[j], stTextures );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("M")) || (sWords[i][1].equals("Mt")))
      {
        k = i+1;
        m = i+2;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified( sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");

            if (m < iCount)
            {
		      if (sWords[m][1].equals("T"))
		      {
                sWords[i][0] = new String(sWords[i][0] + " " + sWords[m][0]);
                sWords[i][1] = new String(sWords[m][1]);

                sWords[m][0] = new String("");
                sWords[m][1] = new String("");
			  }
			}
          }
        }
      }

      if (sWords[i][1].equals("Da"))
      {
	    if ((i > 0) && (i<iCount-1))
	 	{
          k = i-1;
          m = i+1;

		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")) &&
		      (!sWords[k][1].equals("B")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i == 0)
		{
          m = i+1;
		  if ((!sWords[m][1].equals("L")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i==iCount-1)
		{
          k = i-1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[k][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
      }

      if (sWords[i][1].equals("B"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseTextureUtility.isModified( sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main lithology type

    for (i=0; i<iCount; i++)
    {
      if (sLithology.length() == 0)
      {
        if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")) &&
            (iContinue == 0))
        {
          iTexture  = parse.parseTextureUtility.getTextureRow( _MAIN, -1, sWords[i][0], stTextures );
          iContinue = 1;

          if (iTexture > -1)
          {
            stTexture  = stTextures.stItem[iTexture];
            sLithology = new String( stTexture.sName );
          }
        }
      }
    }

    // Modify Litholgy with a descriptor

    if (iContinue == 1)
    {
      iNext    = 0;
      iSet_a   = 0;

      for (i=0; i<iCount; i++)
      {
		iSet = 0;

        if ((iSet_a == 0) && (iSet == 0) &&
            ((sWords[i][1].equals("D")) ))
        {
          if (iTexture > -1)
          {
            iDescrip = parse.parseTextureUtility.getTextureRow(_SUB, iTexture, sWords[i][0], stTextures);

            if ((iDescrip > -1) && (iNext == 0))
            {
              iNext = 1;
              stTexture  = stTextures.stItem[iTexture].descriptors[iDescrip];
              if (sWords[i][1].equals("D"))  //JRV
              {
				if ((!stTexture.sName.equals("nodular")) && (!stTexture.sName.equals("massive")))
                  sLithology = new String( stTexture.sName + " " + sLithology);
			  }
            }
          }
        }
      }
    }

    if (iContinue == 1)
    {
      if (st != null)
      {
		if ((!sLithology.equals("coal"))        &&
		    (!sLithology.equals("peat"))        &&
//		    (!sLithology.equals("calcareous"))  &&
		    (!sLithology.equals("anhydrite"))   &&
		    (!sLithology.equals("salt"))        &&
		    (!sLithology.equals("gypsum"))      &&
		    (!sLithology.equals("gypsiferous")) &&
		    (!sLithology.equals("phosphatic"))  &&
		    (!sLithology.equals("chert"))       &&
		    (!sLithology.equals("cherty"))      &&
		    (!sLithology.equals("chalky")) )
        {
          st.iData[horizon.env.envStruct._L] = 1;
		  st = horizon.env.envUtility.addDataItem(
	  		  st.iLith, sLithology, horizon.env.envStruct._L, st );
	    }

        for (i=0; i<iTotal; i++)
        {
		  if (sWords[i][0].toLowerCase().equals("coal"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "coal", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("peat"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "peat", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("calcareous"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "calcareous", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("anhydrite"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "anhydrite", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("salt"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "salt", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("gypsum"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "gypsum", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("gypsiferous"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "gypsiferous", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("cherty"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "cherty", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("chalky"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "chalky", horizon.env.envStruct._Lt, st );
		  }

		  if (sWords[i][0].toLowerCase().equals("chert"))
		  {
			if (!horizon.env.envUtility.doesMatch(
  			      horizon.env.envStruct._Lt, "chert", st ))
            {
			  if (st.iData[horizon.env.envStruct._Lt] == -1)
                st.iData[horizon.env.envStruct._Lt] = 1;
              else
                st.iData[horizon.env.envStruct._Lt]++;

  		      st = horizon.env.envUtility.addDataItem(
			        st.iLith_T, "chert", horizon.env.envStruct._Lt, st );

			  if (st.iData[horizon.env.envStruct._L] == -1)
                st.iData[horizon.env.envStruct._L] = 1;
              else
                st.iData[horizon.env.envStruct._L]++;

  		      st = horizon.env.envUtility.addDataItem(
			      st.iLith, "chert", horizon.env.envStruct._L, st );
			}
		  }

		  if (sWords[i][0].toLowerCase().equals("phosphatic"))
		  {
			if (st.iData[horizon.env.envStruct._Lt] == -1)
              st.iData[horizon.env.envStruct._Lt] = 1;
            else
              st.iData[horizon.env.envStruct._Lt]++;

  		    st = horizon.env.envUtility.addDataItem(
			      st.iLith_T, "phosphatic", horizon.env.envStruct._Lt, st );
		  }

	    }
	  }
    }
//horizon.env.envUtility.print(st);

    return (st);
  }

  /** Method parseColor()
   * <p> This method will parse the text string into rock color
   * @param  st       = the depositional environment data structure
   * @param  sData    = the color string phrase or word
   * @param  stColors = Color Thesaurus
   * @return st       = the depositional environment data structure
   */

  public static  envStruct parseColor(
                     envStruct st, String sData[],
                     parseColorListStruct stColors )
  {
    int i, j, k, m, p;
    int                  iCount     = 0;
    String sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int iRow[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int iOrder[] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String               sTemp[]     = {"", ""};
    int                  iColor      = -1;
    int                  iDescrip    = -1;
    int                  iContinue   = 0;
    int                  iNext       = 0;
    parseColorStruct     stColor     = null;
    String               sID         = "";
    String               sName       = "";
    int                  iNew        = 0;
    int                  iHue        = 0;
    int                  iTotal      = 0;

    if (stColor != null)
      stColor.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stColors.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stColors.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stColors.sAbbrev[j][1]);
      }
    }

    // Find Color Words up to 10

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseColorUtility.isKeyword( sData[j], stColors ))
      {
        if (iCount < 15)
        {
          sTemp             = parse.parseColorUtility.getKeyword( sData[j], stColors );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iOrder[iCount]    = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseColorUtility.isModified( sWords[i][0], sWords[k][0], stColors ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }

      if ((sWords[i][1].equals("B")) || (sWords[i][1].equals("Q")))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseColorUtility.isModified( sWords[i][0], sWords[k][0], stColors ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("P");
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main color type

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("P")) || (sWords[i][1].equals("B")))
      {
        iColor  = parse.parseColorUtility.getColorRow( _MAIN, -1, sWords[i][0], stColors );

        if (iColor > -1)
        {
          iContinue    = 1;
          iRow[i]      = iColor;
          sWords[i][1] = new String("X");
        }
      }
    }

    // Modify Color with a descriptor

    if (iContinue == 1)
    {
      for (m=0; m<iCount; m++)
      {
        if (iRow[m] > -1)
        {
          stColor = stColors.stItem[iRow[m]];
          sID     = new String( stColor.sID );

          iNext = 0;
          for (i=0; i<m; i++)
          {
            if ( (sWords[i][1].equals("D")) || (sWords[i][1].equals("M")))
            {
              iDescrip = parse.parseColorUtility.getColorRow(_SUB, iRow[m], sWords[i][0], stColors);

              if ( (iDescrip > -1) && (iNext == 0))
              {
                iNext = 1;
                sWords[i][1] = new String("X");
                stColor = stColors.stItem[iRow[m]].descriptors[iDescrip];
                sID = new String(stColor.sID);
              }
            }
          }

          if (st != null)
          {
            if (iTotal < 3)
            {
              if (stColors.iColors > 0)
              {
                for (j=0; j<stColors.iColors; j++)
                {
                  if (stColors.stColors[j].sID.equals(sID))
                  {
                    iColor = j;
                  }
                }

			    if (st.iData[horizon.env.envStruct._C] == -1)
                  st.iData[horizon.env.envStruct._C] = 1;
                else
                  st.iData[horizon.env.envStruct._C]++;

		        st = horizon.env.envUtility.addDataItem(
		    	  st.iColor, stColors.stColors[iColor].sPhrase, horizon.env.envStruct._C, st );

                iTotal++;  // Total Number of Colors for section
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method parseFossils()
   * <p> This method will parse the text string into rock fossil
   * @param  st         = the depositional environment data structure
   * @param  sData      = the fossil string phrase or word
   * @param  stFossils  = Fossils Thesaurus
   * @param  stBio      = Fossils Scientific Name
   * @param  stKSBio    = Kansas Fossils Scientific Name
   * @return st         = the depositional environment data structure
   */

  public static  envStruct parseFossils(
                     envStruct st, String sData[],
                     parseFossilListStruct stFossils,
                     bioStratListStruct    stBio,
                     bioStratListStruct    stKSBio )
  {
    int i, j, k, m, m0, m1, m2, p;
//    fossilStruct    st         = null;
    int                  iCount     = 0;
    String sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int iRow[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int iOrder[] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String               sTemp[]     = {"", ""};
    int                  iFossils    = -1;
    int                  iDescrip    = -1;
    int                  iContinue   = 0;
    int                  iNext       = 0;
    parseFossilStruct    stFossil    = null;
    String               sID         = "";
    String               sName       = "";
    String               sMod        = "";
    String               sEnv        = "";
    int                  iNew        = 0;
    int                  iFound      = 0;

    if (stFossil != null)
      stFossil.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stFossils.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stFossils.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stFossils.sAbbrev[j][1]);
      }

      sData[i] = new String(
		  horizon.bio.bioStratUtility.convert(sData[i], stBio) );

      sData[i] = new String(
		  horizon.bio.bioStratUtility.convert(sData[i], stKSBio) );
    }

    // Find Fossils Words up to 25

    for (j=0; j<sData.length; j++)
    {
      if (parse.parseFossilUtility.isKeyword( sData[j], stFossils ))
      {
        if (iCount < 25)
        {
          sTemp             = parse.parseFossilUtility.getKeyword( sData[j], stFossils );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iOrder[iCount]    = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (parse.parseFossilUtility.isModified( sWords[i][0], sWords[k][0], stFossils ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Is there a Environment Descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("E"))
      {
        if (sWords[i][0].equals("fresh water"))
          sEnv = new String( "Fresh Water" );
        if (sWords[i][0].equals("brackish water"))
          sEnv = new String( "Brackish Water" );
        if (sWords[i][0].equals("marine"))
          sEnv = new String( "Marine" );
      }
    }

    // Identify the main fossil type

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("F"))
      {
        iFossils = parse.parseFossilUtility.getFossilsRow(
			         _MAIN, -1, sWords[i][0], stFossils );

        if (iFossils > -1)
        {
          iContinue    = 1;
          iRow[i]      = iFossils;
          sWords[i][1] = new String("X");
        }
      }
    }

    // Modify fossil with a descriptor

    if (iContinue == 1)
    {
      for (m=0; m<iCount; m++)
      {
        m0 = m+1;
        m1 = m+2;
        if (m0 >= iCount) m0 = iCount-1;
        if (m1 >= iCount) m1 = iCount-1;
        m2 = m-1;
        if (m2 < 0) m2 = 0;

        if (iRow[m] > -1)
        {
          stFossil = stFossils.stItem[iRow[m]];

          iFound = 0;
          if (sWords[m0][0].equals("fragments"))
          {
            iFound = 1;
            sMod = new String( "Broken" );
            sWords[m0][1] = new String("X");
          }

          sID      = new String( stFossil.sID);
          sName    = new String( stFossil.sPhrase );

          // Add a Modification to the fossil "Abundant", "Rare" or "Broken"

          for (i=0; i<m0; i++)
          {
            if (sWords[i][1].equals("A"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0))
              {
                iFound = 1;
                sMod = new String("Abundant");
                sWords[i][1] = new String("X");
              }
            }

            if (sWords[i][1].equals("T"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0))
              {
                iFound = 1;
                sMod = new String("Rare");
                sWords[i][1] = new String("X");
              }
            }

            if (sWords[i][1].equals("B"))
            {
              if ((iOrder[m2] == iOrder[m]-1) && (iFound == 0) &&
                  (!sWords[i][0].equals("fragments")))
              {
                iFound = 1;
                sMod = new String( "Broken" );
                sWords[i][1] = new String("X");
              }
            }
          }

          for (i=0; i<m; i++)
          {
            if (sWords[i][1].equals("D"))
            {
              iDescrip = parse.parseFossilUtility.getFossilsRow(
                           _SUB, iRow[m], sWords[i][0], stFossils );

              if (iDescrip > -1)
              {
                sWords[i][1] = new String("X");
                stFossil     = stFossils.stItem[iRow[m]].descriptors[iDescrip];

                sID          = new String( stFossil.sID);
                sName        = new String( stFossil.sPhrase );
              }
            }
          }

          if (st != null)
          {
			if (!horizon.env.envUtility.doesMatch(
			      horizon.env.envStruct._F, sName, st ))
            {
 	          if (st.iData[horizon.env.envStruct._F] == -1)
                st.iData[horizon.env.envStruct._F] = 1;
              else
                st.iData[horizon.env.envStruct._F]++;

		      st = horizon.env.envUtility.addDataItem(
		      	    st.iFossil, sName, horizon.env.envStruct._F, st );
		    }
            sID   = new String("");
            sMod  = new String("");
            sName = new String("");
          }
        }
      }
    }

    return (st);
  }

  /** Method parseGenera()
   * <p> This method will parse the text string into rock fossil
   * @param  st           = the depositional environment data structure
   * @param  sData        = the fossil string phrase or word
   * @param  stKSBio      = Kansas Fossils Scientific Names Lookup
   * @return st           = the depositional environment data structure
   */

  public static envStruct parseGenera( envStruct st,
                                       String sData[],
                                       bioStratListStruct stKSBio )
  {
	int i, j, k, m;
    String         str        = "";
    String         tokens[]   = null;
    String         sDelimiter = new String("[ ]+");
    String         sTerms[]   = null;
    int            iTerms[]   = null;
    String         sGenera    = "";
    int            iPosition  = -1;

	if (stKSBio != null)
	{
      for (i=0; i<sData.length; i++)
      {
	    str    = new String( sData[i].trim() );
        tokens = str.split( sDelimiter );

        // BUILD and INITIALIZE the Group of text being tested

        sTerms = new String[tokens.length];
        iTerms = new int[tokens.length];

        for (j=0; j<tokens.length; j++)
        {
	  	  sTerms[j] = new String( tokens[j].trim() );
	 	  iTerms[j] = parse.parseGeneraUtility._NONE;
        }

        // REMOVE '(', ')' and '?' from each String

        for (j=0; j<sTerms.length; j++)
        {
		  sTerms[j] = parse.parseGeneraUtility.removeChars( sTerms[j] );
        }

	    // CHECK if any of the terms are in the Kansas Fossil Genera List

	    iPosition  = -1;

	    for (j=0; j<sTerms.length; j++)
	    {
	  	  // IF this term a Kansas Fossil with a recognizable Genera Term THEN

		  if (horizon.bio.bioStratUtility.exists(sTerms[j], stKSBio))
		  {
            if (iPosition == -1) { iPosition  = j; }
    	    iTerms[j]  = parse.parseGeneraUtility._GENERA;
    	    sGenera    = new String(sTerms[j]);

 	        if (st.iData[horizon.env.envStruct._G] == -1)
              st.iData[horizon.env.envStruct._G] = 1;
            else
              st.iData[horizon.env.envStruct._G]++;

		    st = horizon.env.envUtility.addDataItem(
		    	  st.iGenus, sGenera, horizon.env.envStruct._G, st );

		    if (sGenera.toLowerCase().equals("carbonita"))
		    {
		  	  if (st.iData[horizon.env.envStruct._Gt] == -1)
                st.iData[horizon.env.envStruct._Gt] = 1;
              else
                st.iData[horizon.env.envStruct._Gt]++;

  		      st = horizon.env.envUtility.addDataItem(
			        st.iLith_T, "carbonita", horizon.env.envStruct._Gt, st );
		    }

		    if (sGenera.toLowerCase().equals("darwinula"))
		    {
		  	  if (st.iData[horizon.env.envStruct._Gt] == -1)
                st.iData[horizon.env.envStruct._Gt] = 1;
              else
                st.iData[horizon.env.envStruct._Gt]++;

  		      st = horizon.env.envUtility.addDataItem(
			        st.iLith_T, "darwinula", horizon.env.envStruct._Gt, st );
		    }

		    if (sGenera.toLowerCase().equals("geisina"))
		    {
		  	  if (st.iData[horizon.env.envStruct._Gt] == -1)
                st.iData[horizon.env.envStruct._Gt] = 1;
              else
                st.iData[horizon.env.envStruct._Gt]++;

  		      st = horizon.env.envUtility.addDataItem(
			        st.iLith_T, "geisina", horizon.env.envStruct._Gt, st );
		    }
		  }
	    }

	    sTerms = null;
	    iTerms = null;
	    tokens = null;
	  }
    }

    return (st);
  }

  /** Method parseDepEnv()
   * <p> This method will parse the text string into Depositional Environment
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  st         = Depositional Environment data structure
   * @param  stDepEnv   = Depositional Environment Lookup List Structure
   * @return st         = Depositional Environment data structure
   */

  public static  envStruct parseDepEnv( double        depthStart,
                                        double        depthEnd,
                                        envStruct     st,
                                        envListStruct stDepEnv )
  {
	int     i,j, k;
	int     iFound      = -1;
	int     iData[]     = { -1, -1, -1, -1, -1, -1 };
	int     iDataTest[] = { -1, -1, -1, -1, -1, -1 };
	int     iHold       = -1;
	boolean bFail       = false;
	int     iStart      = 0;
	int     iTotal      = 0;
	int     iTotalT     = 0;
	String  sDescrip    = "";

	if ((st != null) && (stDepEnv != null))
	{
	  for (i=0; i<stDepEnv.iCount; i++)
	  {
		// Does any of the Lithologies found match the lithologies that would
		// determine the environment by themselves.

	    if (iFound == -1)
	    {
  		  for (j=0; j<st.iLith_T; j++)
		  {
			if ((st.sLith_T[j].toLowerCase().equals("coal"))       ||
			    (st.sLith_T[j].toLowerCase().equals("peat"))       ||
			    (st.sLith_T[j].toLowerCase().equals("phosphatic")) ||
			    (st.sLith_T[j].toLowerCase().equals("anhydrite"))  ||
			    (st.sLith_T[j].toLowerCase().equals("salt"))       ||
			    (st.sLith_T[j].toLowerCase().equals("gypsum"))     ||
			    (st.sLith_T[j].toLowerCase().equals("gypsiferous")) )
			{
              if (horizon.env.envUtility.doesMatch(
			        horizon.env.envStruct._Lt, st.sLith_T[j], stDepEnv.stItem[i] ))
              {
		  	    iFound = i;
                if (iHold > -1)
                  iHold = -1;
		      }
		    }
	      }
	    }

		// Does any of the Fossil Genera found match the Fresh Water Genera that would
		// determine the environment by themselves.

        if (iFound == -1)
	    {
		  for (j=0; j<st.iGenus_T; j++)
		  {
            if (horizon.env.envUtility.doesMatch(
			      horizon.env.envStruct._Gt, st.sGenus_T[j], stDepEnv.stItem[i] ))
            {
		  	  iFound = i;
              if (iHold > -1)
                iHold = -1;
		    }
	      }
	    }

        if (iFound == -1)
        {
		  for (j=0; j<6; j++)
		  {
			if (stDepEnv.stItem[i].iData[j] > 0)
			  iDataTest[j] = 0;
		    else
		      iDataTest[j] = stDepEnv.stItem[i].iData[j];
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._L] > -1)
		  {
  		    for (j=0; j<st.iLith; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._L, st.sLith[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._L] == -1)
				  iDataTest[horizon.env.envStruct._L] = 1;
				else
				  iDataTest[horizon.env.envStruct._L]++;
			  }
		    }
	      }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._C] > -1)
		  {
  		    for (j=0; j<st.iColor; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._C, st.sColor[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._C] == -1)
				  iDataTest[horizon.env.envStruct._C] = 1;
				else
				  iDataTest[horizon.env.envStruct._C]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._F] > -1)
		  {
  		    for (j=0; j<st.iFossil; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._F, st.sFossil[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._F] == -1)
				  iDataTest[horizon.env.envStruct._F] = 1;
				else
				  iDataTest[horizon.env.envStruct._F]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._G] > -1)
		  {
  		    for (j=0; j<st.iGenus; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._G, st.sGenus[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._G] == -1)
				  iDataTest[horizon.env.envStruct._G] = 1;
				else
				  iDataTest[horizon.env.envStruct._G]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._Gt] > -1)
		  {
  		    for (j=0; j<st.iGenus; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._Gt, st.sGenus[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._Gt] == -1)
				  iDataTest[horizon.env.envStruct._Gt] = 1;
				else
				  iDataTest[horizon.env.envStruct._Gt]++;
			  }
		    }
	      }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._Lt] > -1)
		  {
  		    for (j=0; j<st.iLith_T; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._Lt, st.sLith_T[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._Lt] == -1)
				  iDataTest[horizon.env.envStruct._Lt] = 1;
				else
				  iDataTest[horizon.env.envStruct._Lt]++;
			  }
		    }
		  }

  	      bFail = false;
//System.out.print("["+i+"] ");
		  for (k=0; k<6; k++)
		  {
//System.out.print(iDataTest[k]+" ");
			switch (iDataTest[k])
			{
			  case -1:
			   break;
			  case 0:
			   bFail = true;
			   break;
			}
		  }
//System.out.println(bFail+"");

		  if (!bFail)
		  {
//System.out.println("? "+bFail);
			if (iStart == 0)
			{
  			  iHold = i;
  		      for (k=0; k<6; k++)
		      {
				iData[k]     = iDataTest[k];
				iDataTest[k] = -1;
			  }
		  	  iStart++;
		    }
		    else
		    {
  		      for (k=0; k<6; k++)
		      {
				if (iDataTest[k] > -1)
				{
				  iTotalT = iTotalT + iDataTest[k];
				}

				if (iData[k] > -1)
				{
				  iTotal  = iTotal + iData[k];
				}
			  }

			  if (iTotalT > iTotal)
			  {
  			    iHold = i;
  		        for (k=0; k<6; k++)
		        {
				  iData[k] = iDataTest[k];
			    }
			  }

  		      for (k=0; k<6; k++)
		      {
				iDataTest[k] = -1;
			  }
			}
		  }
		}
	  }

      if (iHold > -1)
        iFound = iHold;

      if (iFound > -1)
        sDescrip = getDescription(st);

      st.delete();
      st = null;

	  if (iFound > -1)
	  {
		st = horizon.env.envUtility.copyEnv( stDepEnv.stItem[iFound] );
		st.depthStart = depthStart;
		st.depthEnd   = depthEnd;
		st.sDescrip   = new String( sDescrip );
	  }
	}
//horizon.env.envUtility.print(st);

	return (st);
  }

  public static String getDescription( envStruct st )
  {
	int i,j, k;
	String sLevel = "";
	String str    = ".";
	int    len    = 0;
	String out    = "Notes: ";

    if (st != null)
    {
      for (k=0; k<6; k++)
      {
        if (st.iData[k] > -1)
        {
	      len = 0;
	      switch (k)
	      {
            case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
              len = st.iLith_T;
              break;
            case horizon.env.envStruct._L:  // Total number of Lithologies found
              len = st.iLith;
              break;
            case horizon.env.envStruct._C:  // Total number of Colors found
              len = st.iColor;
              break;
            case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
              len = st.iFossil;
              break;
            case horizon.env.envStruct._Gt: // Total number of Unique Genera found
              len = st.iGenus_T;
              break;
            case horizon.env.envStruct._G:  // Total number of Genera found
              len = st.iGenus;
              break;
          }

	      for (j=0; j<len; j++)
	      {
  	        switch (k)
	        {
              case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
                str = new String( st.sLith_T[j] );
                break;
              case horizon.env.envStruct._L:  // Total number of Lithologies found
                str = new String( st.sLith[j] );
                break;
              case horizon.env.envStruct._C:  // Total number of Colors found
                str = new String( st.sColor[j] );
                break;
              case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
                str = new String( st.sFossil[j] );
                break;
              case horizon.env.envStruct._Gt: // Total number of Unique Genera found
                str = new String( st.sGenus_T[j] );
                break;
              case horizon.env.envStruct._G:  // Total number of Genera found
                str = new String( st.sGenus[j] );
                break;
            }

			if (j == 0)
			{
			  switch (k)
			  {
                case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
   		          out = new String( out + "{ Lithology (t): " + str );
                  break;
                case horizon.env.envStruct._L:  // Total number of Lithologies found
   		          out = new String( out + "{ Lithology: " + str );
                  break;
                case horizon.env.envStruct._C:  // Total number of Colors found
   		          out = new String( out + "{ Color: " + str );
                  break;
                case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
   		          out = new String( out + "{ Fossil: " + str );
                  break;
                case horizon.env.envStruct._Gt: // Total number of Unique Genera found
   		          out = new String( out + "{ Genera (t): " + str );
                  break;
                case horizon.env.envStruct._G:  // Total number of Genera found
   		          out = new String( out + "{ Genera: " + str );
                  break;
              }
			}
			else
			{
	          out = new String( out + ", " + str );
			}
		  }

          out = new String( out + " }");
  		}
	  }
	}

	return (out);
  }

  /** Method parseDepEnv()
   * <p> This method will parse the text string into Depositional Environment
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  st         = Depositional Environment data structure
   * @param  stDepEnv   = Depositional Environment Lookup List Structure
   * @return st         = Depositional Environment data structure
   *

  public static  envStruct parseDepEnv( double        depthStart,
                                        double        depthEnd,
                                        envStruct     st,
                                        envListStruct stDepEnv )
  {
	int     i,j, k;
	int     iFound      = -1;
	int     iHold       = -1;
	boolean bFail       = false;
	int     iData[]     = { -1, -1, -1, -1, -1, -1 };
	int     iDataTest[] = { -1, -1, -1, -1, -1, -1 };
	int     iStart      = 0;
	int     iTotal      = 0;
	int     iTotalT     = 0;

	if ((st != null) && (stDepEnv != null))
	{
//horizon.env.envUtility.print(st);
	  for (i=0; i<stDepEnv.iCount; i++)
	  {
		// Does any of the Lithologies found match the lithologies that would
		// determine the environment by themselves.

	    if (iFound == -1)
	    {
  		  for (j=0; j<st.iLith_T; j++)
		  {
            if (horizon.env.envUtility.doesMatch(
			      horizon.env.envStruct._Lt, st.sLith_T[j], stDepEnv.stItem[i] ))
            {
		  	  iFound = i;
		    }
	      }
	    }
//System.out.println("A "+" "+iFound);

		// Does any of the Fossil Genera found match the Fresh Water Genera that would
		// determine the environment by themselves.

        if (iFound == -1)
	    {
		  for (j=0; j<st.iGenus_T; j++)
		  {
            if (horizon.env.envUtility.doesMatch(
			      horizon.env.envStruct._Gt, st.sGenus_T[j], stDepEnv.stItem[i] ))
            {
		  	  iFound = i;
		    }
	      }
	    }
//System.out.println("B "+" "+iFound);

        if (iFound == -1)
        {
		  iDataTest = stDepEnv.stItem[i].iData;

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._L] > -1)
		  {
  		    for (j=0; j<st.iLith; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._L, st.sLith[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._L] == -1)
				  iDataTest[horizon.env.envStruct._L] = 1;
				else
				  iDataTest[horizon.env.envStruct._L]++;
			  }
		    }
	      }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._C] > -1)
		  {
  		    for (j=0; j<st.iColor; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._C, st.sColor[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._C] == -1)
				  iDataTest[horizon.env.envStruct._C] = 1;
				else
				  iDataTest[horizon.env.envStruct._C]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._F] > -1)
		  {
  		    for (j=0; j<st.iFossil; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._F, st.sFossil[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._F] == -1)
				  iDataTest[horizon.env.envStruct._F] = 1;
				else
				  iDataTest[horizon.env.envStruct._F]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._G] > -1)
		  {
  		    for (j=0; j<st.iGenus; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._G, st.sGenus[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._G] == -1)
				  iDataTest[horizon.env.envStruct._G] = 1;
				else
				  iDataTest[horizon.env.envStruct._G]++;
			  }
		    }
		  }

		  if (stDepEnv.stItem[i].iData[horizon.env.envStruct._Gt] > -1)
		  {
  		    for (j=0; j<st.iGenus; j++)
		    {
              if (horizon.env.envUtility.doesMatch(
  			        horizon.env.envStruct._Gt, st.sGenus[j], stDepEnv.stItem[i] ))
  			  {
				if (iDataTest[horizon.env.envStruct._Gt] == -1)
				  iDataTest[horizon.env.envStruct._Gt] = 1;
				else
				  iDataTest[horizon.env.envStruct._Gt]++;
			  }
		    }
		  }

		  for (k=0; k<6; k++)
		  {
			switch (iDataTest[k])
			{
			  case -1:
			   break;
			  case 0:
			   bFail = true;
			   break;
			}
		  }

		  if (!bFail)
		  {
			if (iStart == 0)
			{
  			  iHold = i;
  		      for (k=0; k<6; k++)
		      {
				iData[k]     = iDataTest[k];
				iDataTest[k] = -1;
			  }
		  	  iStart++;
		    }
		    else
		    {
  		      for (k=0; k<6; k++)
		      {
				if (iDataTest[k] > -1)
				{
				  iTotalT = iTotalT + iDataTest[k];
				}

				if (iData[k] > -1)
				{
				  iTotal  = iTotal + iData[k];
				}
			  }

			  if (iTotalT > iTotal)
			  {
  			    iHold = i;
  		        for (k=0; k<6; k++)
		        {
				  iData[k] = iDataTest[k];
			    }
			  }

  		      for (k=0; k<6; k++)
		      {
				iDataTest[k] = -1;
			  }
			}
		  }
	    }
	  }

      if (iHold > -1)
        iFound = iHold;
//System.out.println("C "+" "+iFound);

      st.delete();
      st = null;

	  if (iFound > -1)
	  {
		st = horizon.env.envUtility.copyEnv( stDepEnv.stItem[iFound] );
		st.depthStart = depthStart;
		st.depthEnd   = depthEnd;
	  }
	}
//horizon.env.envUtility.print(st);

	return (st);
  }
*/

  /** Method parse()
   * <p> This method will parse the Remarks data list structure for
   *     Depositional Environment terms.
   * @param  stList    = Remarks List Data Structure
   * @param  stDepEnvs = Depositional Environment Thesaurus
   * @return stDepEnv  = Depositional Environment Data List Structure
   *

  public static envListStruct parse( iqstratRemarkListStruct stList,
                                     parseDepEnvListStruct   stDepEnvs )
  {
    envListStruct     stDepEnv  = null;
    String            sKEY       = cmn.cmnString.UniqueName();
    String            str        = "";
    String            tokens[]   = null;
    String            sDelimiter = new String("[ /,;:.=-]+");
    envStruct         stText     = null;

    if (stList != null)
    {
      stDepEnv = new envListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parseDepEnv( stList.stItem[i].depthStart,
                              stList.stItem[i].depthEnd, //" ",
                              tokens, stDepEnvs );

        if (stText != null)
        {
          stText.sKEY     = new String( sKEY + "_" + i );
          stText.sDescrip = new String( stList.stItem[i].sText );
          stDepEnv        = horizon.env.envUtility.add(stText, stDepEnv );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stDepEnv);
  }
*/
  /** Method parse()
   * <p> This method will parse the Remarks string for Depositional Environment.
   * @param  iRow       = the row number from source
   * @param  st         = Depositional Environment Data Structure
   * @param  stDepEnv   = Depositional Environment Data List Structure
   * @param  stDepEnvs  = Depositional Environment Thesaurus
   * @return stDepEnv   = Depositional Environment Data List Structure
   *

  public static envListStruct parse( int                   iRow,
                                     envStruct             st,
                                     envListStruct         stDepEnv,
                                     parseDepEnvListStruct stDepEnvs )
  {
    String    sKEY       = cmn.cmnString.UniqueName();
    String    str        = "";
    String    tokens[]   = null;
    String    sDelimiter = new String("[ /,;:.=-]+");
    envStruct stText     = null;

    if (st != null)
    {
	  if (st.sDescrip.length() > 0)
	  {
        str    = new String( st.sDescrip.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parseDepEnv( st.depthStart,
                              st.depthEnd,
//                              st.sLabel,
                              tokens,
                              stDepEnvs );

        if (stText != null)
        {
          stText.sKEY     = new String( sKEY + "_" + iRow );
//          stText.sLabel   = new String( st.sLabel );
          stText.sDescrip = new String( st.sDescrip ); //sText );

//          if (st.iFrom > -4)
//	      {
//            stText.iFrom = st.iFrom;
//            stText.iTo   = st.iTo;
//		  }

          stDepEnv = horizon.env.envUtility.add(stText, stDepEnv );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }
    }

    return (stDepEnv);
  }
*/
  /** Method parseDepEnv()
   * <p> This method will parse the text string into Depositional Environment
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  sData      = the texture string phrase or word
   * @param  stDepEnvs  = Depositional Environment Thesaurus
   * @return st         = Depositional Environment data structure
   *

  public static  envStruct parseDepEnv( double                depthStart,
                                        double                depthEnd,
//                                        String                sLabel,
                                        String                sData[],
                                        parseDepEnvListStruct stDepEnvs )
  {
    int i, j, k, m, p;
    envStruct         st         = null;
    int               iCount     = 0;
    int               iTotal     = 0;
    String            sWords[][] = null;
    String            sTemp[]    = {"", ""};
    int               iDepEnv    = -1;
    int               iDescrip   = -1;
    int               iContinue  = 0;
    int               iFound     = 0;
    int               iSet       = 0;
    int               iNext      = 0;
    parseDepEnvStruct stDepEnv   = null;
    String            sDepEnv    = "";
    String            sTerm      = "T";
//    int               id         = 0;
//    int               iSymbol    = 0;
//    int               iRGB[]     = { 255, 255, 255 };
    int               iLevel[]   = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int               itmp[]     = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int               itmp2[]    = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int               iType      = horizon.env.envStruct._NONE;
    String            sType      = "";
    String            sGroup     = "";
    String            sGroup2    = "";
    double            depth      = 0.0;
    int               iFossil    = 0;
    int               iFactor    = 0;
    int               iMax       = 0;

    if (stDepEnv != null)
      stDepEnv.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stDepEnvs.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stDepEnvs.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stDepEnvs.sAbbrev[j][1]);
      }
    }

    // Find Depositional Environment Words

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stDepEnvs ))
      {
        iCount++;
      }
    }

    iTotal = iCount;
    iCount = 0;
    sWords = new String[iTotal][2];

    if (iTotal > 0)
    {
      // Find Depositional Environment Words

      for (j=0; j<sData.length; j++)
      {
        if (isKeyword( sData[j], stDepEnvs ))
        {
          if (iCount < iTotal)
          {
            sTemp             = getKeyword( sData[j], stDepEnvs );
            sWords[iCount][0] = new String(sTemp[0]);
            sWords[iCount][1] = new String(sTemp[1]);
            iCount++;
          }
        }
      }

      // Change a Modifier to a descriptor

      for (i=0; i<iCount; i++)
      {
        if (sWords[i][1].equals("M"))
        {
          k = i+1;

          if (k<iCount)
          {
            if (isModified( sWords[i][0], sWords[k][0], stDepEnvs ))
            {
              sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
              sWords[i][1] = new String(sWords[k][1]);

              sWords[k][0] = new String("");
              sWords[k][1] = new String("");
            }
          }
        }
      }

      // Change an [A]ll term to an [E]nvironment term
      // IF a [T]ype or [E]nvironment term is NOT present
      //    before the [E]nvironment[d]escriptor  term is detected.

      for (i=0; i<iCount; i++)
      {
        if ((sWords[i][1].equals("T")) || (sWords[i][1].equals("E")))
        {
		  iFound = 1;
	    }

	    if (sWords[i][1].equals("Ed"))
	    {
	  	  if (iFound != 1) { sWords[i][1] = new String("E"); }
	    }
	  }

      // Identify if [E]nvironment term is present
      // IF [E]nvironment term is present then use it as primary
      // Default is [T]ype term

      for (i=0; i<iCount; i++)
      {
	    if (sWords[i][1].equals("E"))
	      sTerm = new String("E");
      }

      // Check to see if fossils are present and use as primary

      for (i=0; i<iCount; i++)
      {
        if (sWords[i][1].equals("G"))
        {
          iDepEnv  = getDepEnvRow( _MAIN, -1, sWords[i][0], stDepEnvs );

          if (iDepEnv > -1)
          {
            stDepEnv   = stDepEnvs.stItem[iDepEnv];

            for (j=0; j<itmp.length; j++)
              itmp2[j] = 0;

            if (iFossil == 0)
            {
              sDepEnv    = new String( stDepEnv.sName );
              sType      = new String( stDepEnv.sType );
              sGroup     = new String( stDepEnv.sGroup );
              sGroup2    = new String( stDepEnv.sGroup2 );

              iType      = stDepEnv.iType;
		      itmp[0]    = stDepEnv.icontl;    // Continental
		      itmp[1]    = stDepEnv.icstl;     // Coastal
		      itmp[2]    = stDepEnv.ifrsh;     // Fresh Water
		      itmp[3]    = stDepEnv.ibrcksh;   // Brackish Water
		      itmp[4]    = stDepEnv.ishore;    // Shoreface
		      itmp[5]    = stDepEnv.ioff_tran; // Off shore trasition
		      itmp[6]    = stDepEnv.ioffshr;   // Off Shore
		      itmp[7]    = stDepEnv.ishlf_m;   // Middle Neritic
		      itmp[8]    = stDepEnv.ishlf_o;   // Outer Neritic
		      itmp[9]    = stDepEnv.ibthyl_u;  // Upper Bathyal
		      itmp[10]   = stDepEnv.ibthyl_m;  // Middle Bathyal
		      itmp[11]   = stDepEnv.ibthyl_l;  // Lower Bathyal
	  	      itmp[12]   = stDepEnv.iabyssl;   // Abyssal

	  	      iFossil   = 1;
	  	      iContinue = 1;
		    }
		    else
		    {
		      itmp2[0]   = stDepEnv.icontl;    // Continental
		      itmp2[1]   = stDepEnv.icstl;     // Coastal
		      itmp2[2]   = stDepEnv.ifrsh;     // Fresh Water
		      itmp2[3]   = stDepEnv.ibrcksh;   // Brackish Water
		      itmp2[4]   = stDepEnv.ishore;    // Shoreface
		      itmp2[5]   = stDepEnv.ioff_tran; // Off shore trasition
		      itmp2[6]   = stDepEnv.ioffshr;   // Off Shore
		      itmp2[7]   = stDepEnv.ishlf_m;   // Middle Neritic
		      itmp2[8]   = stDepEnv.ishlf_o;   // Outer Neritic
		      itmp2[9]   = stDepEnv.ibthyl_u;  // Upper Bathyal
		      itmp2[10]  = stDepEnv.ibthyl_m;  // Middle Bathyal
		      itmp2[11]  = stDepEnv.ibthyl_l;  // Lower Bathyal
	  	      itmp2[12]  = stDepEnv.iabyssl;   // Abyssal

              for (j=0; j<itmp.length; j++)
              {
//				iFactor = itmp[j] * itmp2[j];

//				if ((iFactor == 0) && (itmp2[j] > 0))
//				{
//				  itmp2[j] = 0;
//				}

				itmp[j] = itmp[j] + itmp2[j];
			  }
			}
		  }
	    }
      }

      if (iFossil == 1)
      {
        for (j=0; j<itmp.length; j++)
        {
		  if (itmp[j] > iMax)
		  {
			iType = j;
		    sType = new String( horizon.env.envStruct._ENV[iType] );
		    iMax  = itmp[j];
		  }
		}

        for (j=0; j<itmp.length; j++)
        {
		  itmp[j]   = ( 5 * itmp[j] ) / iMax;
          iLevel[j] = itmp[j];
	    }
	  }

      // Identify the main Depositional Environment type

      if (iFossil == 0)
      {
        for (i=0; i<iCount; i++)
        {
          if (sDepEnv.length() == 0)
          {
            if ((sWords[i][1].equals(sTerm)) && (iContinue == 0))
            {
              iDepEnv  = getDepEnvRow( _MAIN, -1, sWords[i][0], stDepEnvs );
              iContinue = 1;

              if (iDepEnv > -1)
              {
                stDepEnv   = stDepEnvs.stItem[iDepEnv];

                sDepEnv    = new String( stDepEnv.sName );
                sType      = new String( stDepEnv.sType );
                sGroup     = new String( stDepEnv.sGroup );
                sGroup2    = new String( stDepEnv.sGroup2 );

                iType      = stDepEnv.iType;
		        iLevel[0]  = stDepEnv.icontl;    // Continental
		        iLevel[1]  = stDepEnv.icstl;     // Coastal
		        iLevel[2]  = stDepEnv.ifrsh;     // Fresh Water
		        iLevel[3]  = stDepEnv.ibrcksh;   // Brackish Water
		        iLevel[4]  = stDepEnv.ishore;    // Shoreface
		        iLevel[5]  = stDepEnv.ioff_tran; // Off shore trasition
		        iLevel[6]  = stDepEnv.ioffshr;   // Off Shore
		        iLevel[7]  = stDepEnv.ishlf_m;   // Middle Neritic
		        iLevel[8]  = stDepEnv.ishlf_o;   // Outer Neritic
		        iLevel[9]  = stDepEnv.ibthyl_u;  // Upper Bathyal
		        iLevel[10] = stDepEnv.ibthyl_m;  // Middle Bathyal
		        iLevel[11] = stDepEnv.ibthyl_l;  // Lower Bathyal
		        iLevel[12] = stDepEnv.iabyssl;   // Abyssal
              }
            }
          }
        }
      }

      // Modify Primary Depositional Environment with a descriptor

      if (iContinue == 1)
      {
        iNext   = 0;

        for (i=0; i<iCount; i++)
        {
	  	  iSet = 0;

          if ((sWords[i][1].equals("D")) && (iSet == 0))
          {
            if (iDepEnv > -1)
            {
              iDescrip = getDepEnvRow(_SUB, iDepEnv, sWords[i][0], stDepEnvs);

              if ((iDescrip > -1) && (iNext == 0))
              {
                iSet       = 1;
                stDepEnv   = stDepEnvs.stItem[iDepEnv].descriptors[iDescrip];

                sDepEnv    = new String( sDepEnv + " (" + stDepEnv.sName + ")" );
                sType      = new String( stDepEnv.sType );
                sGroup     = new String( stDepEnv.sGroup );
                sGroup2    = new String( stDepEnv.sGroup2 );

                iType      = stDepEnv.iType;
		        iLevel[0]  = stDepEnv.icontl;    // Continental
		        iLevel[1]  = stDepEnv.icstl;     // Coastal
		        iLevel[2]  = stDepEnv.ifrsh;     // Fresh Water
		        iLevel[3]  = stDepEnv.ibrcksh;   // Brackish Water
		        iLevel[4]  = stDepEnv.ishore;    // Shoreface
		        iLevel[5]  = stDepEnv.ioff_tran; // Off shore trasition
		        iLevel[6]  = stDepEnv.ioffshr;   // Off Shore
		        iLevel[7]  = stDepEnv.ishlf_m;   // Middle Neritic
		        iLevel[8]  = stDepEnv.ishlf_o;   // Outer Neritic
		        iLevel[9]  = stDepEnv.ibthyl_u;  // Upper Bathyal
		        iLevel[10] = stDepEnv.ibthyl_m;  // Middle Bathyal
		        iLevel[11] = stDepEnv.ibthyl_l;  // Lower Bathyal
		        iLevel[12] = stDepEnv.iabyssl;   // Abyssal

                iNext    = 1;
		      }
            }
	      }
        }
      }
    }

    if (iContinue == 1)
    {
      st = new envStruct();

      st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                  ((int)depthStart)); // Create a Unique KEY
      st.depthStart = depthStart; // Starting Depth
      st.depthEnd   = depthEnd;   // Ending Depth
      if (st.depthEnd <= st.depthStart) { st.depthEnd = st.depthStart + 1.0; }
      depth         = Math.abs(st.depthEnd - st.depthStart);
	  st.dBed       = (double) ((int) (depth*1000.0)) / 1000.0;

      st.sDepEnv    = new String( sDepEnv );
      st.iType      = iType;
      st.sType      = new String( sType );
      st.sGroup     = new String( sGroup );
      st.sGroup2    = new String( sGroup2 );
	  st.icontl     = iLevel[0];  // Continental
	  st.icstl      = iLevel[1];  // Coastal
	  st.ifrsh      = iLevel[2];  // Fresh Water
	  st.ibrcksh    = iLevel[3];  // Brackish Water
	  st.ishore     = iLevel[4];  // Shoreface
	  st.ioff_tran  = iLevel[5];  // Off shore trasition
	  st.ioffshr    = iLevel[6];  // Off Shore
	  st.ishlf_m    = iLevel[7];  // Middle Neritic
	  st.ishlf_o    = iLevel[8];  // Outer Neritic
	  st.ibthyl_u   = iLevel[9];  // Upper Bathyal
	  st.ibthyl_m   = iLevel[10]; // Middle Bathyal
	  st.ibthyl_l   = iLevel[11]; // Lower Bathyal
	  st.iabyssl    = iLevel[12]; // Abyssal
    }

    sWords = null;

    return (st);
  }
*/
  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseDepEnvListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String str, parseDepEnvListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified( String str, parseDepEnvListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if (sWord.equals(st.sModifier[i][1].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parseDepEnvListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseDepEnvListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseDepEnvListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseDepEnvListStruct copyList( parseDepEnvListStruct stOld )
  {
    parseDepEnvListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseDepEnvListStruct();

      // Colors

      stNew.iColors = stOld.iColors;             // total number of records
      stNew.sColors = new String[stNew.iColors]; // Color description
      stNew.iRGB    = new int[stNew.iColors][3]; // rgb values

      for (i=0; i<stNew.iColors; i++)
      {
        stNew.sColors[i] = new String(stOld.sColors[i]);
        for (j=0; j<3; j++)
          stNew.iRGB[i][j] = stOld.iRGB[i][j];
      }

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Lithologies

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseDepEnvStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseDepEnvStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseDepEnvStruct copy( parseDepEnvStruct stOld )
  {
    int i=0;
    int j=0;
    parseDepEnvStruct stNew = new parseDepEnvStruct();

    // Depositional Environment

    stNew.sName  = new String( stOld.sName );       // Name

    stNew.sType      = new String( stOld.sType );   // Type
    stNew.iType      = stOld.iType;                 // Type
    stNew.sGroup     = new String( stOld.sGroup );  // Group of Type
    stNew.sGroup2    = new String( stOld.sGroup2 ); // Group of Group

    // Level variables

    stNew.icstl      = stOld.icstl;     // Coastal
    stNew.ifrsh      = stOld.ifrsh;     // Fresh Water
    stNew.ibrcksh    = stOld.ibrcksh;   // Brackish Water
    stNew.ishore     = stOld.ishore;    // Shoreface
    stNew.ioff_tran  = stOld.ioff_tran; // Off shore trasition
    stNew.ioffshr    = stOld.ioffshr;   // Off Shore
    stNew.ishlf_m    = stOld.ishlf_m;   // Middle Neritic
    stNew.ishlf_o    = stOld.ishlf_o;   // Outer Neritic
    stNew.ibthyl_u   = stOld.ibthyl_u;  // Upper Bathyal
    stNew.ibthyl_m   = stOld.ibthyl_m;  // Middle Bathyal
    stNew.ibthyl_l   = stOld.ibthyl_l;  // Lower Bathyal
    stNew.iabyssl    = stOld.iabyssl;   // Abyssal

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parseDepEnvStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseDepEnvListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  id = (0) Main or (1) descriptor
   * @param  st = Mask Symbols data list structure
   */

  public static void print( int id, parseDepEnvListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(id, st.stItem[i]);
      }
    }
  }

  /** Method print( int id, parseDepEnvStruct st )
   * <p> This method will print the symbols data structure
   * @param  id = (0) Main or (1) descriptor
   * @param  st = Mask symbols data structure
   */

  public static void print( int id, parseDepEnvStruct st )
  {
    if (st != null)
    {
	  if (id == 0)
	  {
        System.out.println( st.sType   + ": " +
                            st.sGroup  + " " + // Group of Type
                            st.sGroup2 + " " + // Group of Group
                            st.sName   + " " + // Name or Description of parse
                            " Level: {"  +
                            st.icontl    + " " +  // Continental
                            st.icstl     + " " +  // Coastal
                            st.ifrsh     + " " +  // Fresh Water
                            st.ibrcksh   + " " +  // Brackish Water
                            st.ishore    + " " +  // Shoreface
                            st.ioff_tran + " " +  // Off shore trasition
                            st.ioffshr   + " " +  // Off Shore
                            st.ishlf_m   + " " +  // Middle Neritic
                            st.ishlf_o   + " " +  // Outer Neritic
                            st.ibthyl_u  + " " +  // Upper Bathyal
                            st.ibthyl_m  + " " +  // Middle Bathyal
                            st.ibthyl_l  + " " +  // Lower Bathyal
                            st.iabyssl   + "}" ); // Abyssal
      }
      else
      {
        System.out.println( " --- "   +
			                st.sType  + ": " +
                            st.sGroup  + " " + // Group of Type
                            st.sGroup2 + " " + // Group of Group
                            st.sName  + " " + // Name or Description of parse
                            " Level: {"  +
                            st.icontl    + " " +  // Continental
                            st.icstl     + " " +  // Coastal
                            st.ifrsh     + " " +  // Fresh Water
                            st.ibrcksh   + " " +  // Brackish Water
                            st.ishore    + " " +  // Shoreface
                            st.ioff_tran + " " +  // Off shore trasition
                            st.ioffshr   + " " +  // Off Shore
                            st.ishlf_m   + " " +  // Middle Neritic
                            st.ishlf_o   + " " +  // Outer Neritic
                            st.ibthyl_u  + " " +  // Upper Bathyal
                            st.ibthyl_m  + " " +  // Middle Bathyal
                            st.ibthyl_l  + " " +  // Lower Bathyal
                            st.iabyssl   + "}" ); // Abyssal
	  }

      if (st.descriptors != null)
      {
        for (int i=0; i<st.iCount; i++)
        {
          print(1, st.descriptors[i]);
        }
      }
    }
  }
}

/*
 *  @version 1.1 11/02/2011
 *  @author  John Victorine
 */
