/*
 * @parseColorUtility.java Version 1.1 10/12/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseColorListStruct;
import parse.parseColorStruct;

//import kgs.kgsMeasSectListStruct;
import iqstrat.iqstratStruct;
import iqstrat.iqstratRemarkListStruct;

import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;

/** Class parseColorUtility
 *  <p> This Class will provide basic utilities for the color parser
 *      data structures.
 *
 *  @version 1.1 10/12/2009
 *  @author  John R. Victorine
 */

public class parseColorUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = Color Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parseColorListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Color Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseColorListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getColorStruct()
   * <p> This method will return the color data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse color List Data Structure
   * @return stT    = parse color data structure
   */

  public static parseColorStruct getColorStruct(
      int iLevel, int iRow, String str, parseColorListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parseColorStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getColorRow()
   * <p> This method will return the color data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse color List Data Structure
   * @return stT    = parse color data structure
   */

  public static int getColorRow(
           int iLevel, int iRow, String str, parseColorListStruct st )
  {
    int iColor = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              iColor = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              iColor = i;
            }
          }
          break;
      }
    }

    return (iColor);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into rock color list
   *     data structure
   * @param  stList   = Remarks List Data Structure
   * @param  stStruct = Global IQSTRAT Data Structure
   * @return stColor  = Rock color Data List Structure
   */

  public static rockColorListStruct parse(
                    iqstratRemarkListStruct stList,
                    iqstratStruct           stStruct )
//                    parseColorListStruct    stColors )
  {
    rockColorListStruct  stColor     = null;
    String               sKEY        = cmn.cmnString.UniqueName();
    String               str         = "";
    String               tokens[]    = null;
    String               sDelimiter  = new String("[ /,;:.=-]+");
    rockColorStruct      stText      = null;
    parseColorListStruct stColors    = stStruct.stParseColors;

    if (stList != null)
    {
      stColor = new rockColorListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        if (rock.color.rockColorChartUtility.checkPhrase(
              str, stStruct.stRockColor ))
        {
          stText = rock.color.rockColorChartUtility.parseColor(
                               stList.stItem[i].depthStart,
                               stList.stItem[i].depthEnd,
                               str, stStruct.stRockColor );
        }
        else if (rock.color.rockColorChartUtility.checkPhrase(
                   str, stStruct.stSoilColor ))
        {
          stText = rock.color.rockColorChartUtility.parseColor(
                               stList.stItem[i].depthStart,
                               stList.stItem[i].depthEnd,
                               str, stStruct.stSoilColor );
        }
        else
        {
          tokens = str.split( sDelimiter );
          stText = parseColor( stList.stItem[i].depthStart,
                               stList.stItem[i].depthEnd,
                               tokens, stStruct.stParseColors ); //stColors );
        }

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stColor     = rock.color.rockColorUtility.add( stText, stColor );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stColor);
  }

  /** Method parseColor()
   * <p> This method will parse the text string into rock color
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  sData        = the color string phrase or word
   * @param  stColors = Color Thesaurus
   * @return st           = rock color data structure
   */

  public static  rockColorStruct parseColor(
                     double depthStart, double depthEnd, String sData[],
                     parseColorListStruct stColors )
  {
    int i, j, k, m, p;
    rockColorStruct      st         = null;
    int                  iCount     = 0;
    String sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int iRow[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int iOrder[] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String               sTemp[]     = {"", ""};
    int                  iColor      = -1;
    int                  iDescrip    = -1;
    int                  iContinue   = 0;
    int                  iNext       = 0;
    parseColorStruct     stColor     = null;
    String               sID         = "";
    String               sName       = "";
    int                  iNew        = 0;
    int                  iHue        = 0;

    if (stColor != null)
      stColor.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stColors.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stColors.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stColors.sAbbrev[j][1]);
      }
    }

    // Find Color Words up to 10

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stColors ))
      {
        if (iCount < 15)
        {
          sTemp             = getKeyword( sData[j], stColors );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iOrder[iCount]    = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stColors ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }

      if ((sWords[i][1].equals("B")) || (sWords[i][1].equals("Q")))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stColors ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("P");
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main color type

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("P")) || (sWords[i][1].equals("B")))
      {
        iColor  = getColorRow( _MAIN, -1, sWords[i][0], stColors );

        if (iColor > -1)
        {
          iContinue    = 1;
          iRow[i]      = iColor;
          sWords[i][1] = new String("X");
        }
      }
    }

    // Modify Color with a descriptor

    if (iContinue == 1)
    {
      for (m=0; m<iCount; m++)
      {
        if (iRow[m] > -1)
        {
          stColor = stColors.stItem[iRow[m]];
          sID     = new String( stColor.sID );

          iNext = 0;
          for (i=0; i<m; i++)
          {
            if ( (sWords[i][1].equals("D")) || (sWords[i][1].equals("M")))
            {
              iDescrip = getColorRow(_SUB, iRow[m], sWords[i][0], stColors);

              if ( (iDescrip > -1) && (iNext == 0))
              {
                iNext = 1;
                sWords[i][1] = new String("X");
                stColor = stColors.stItem[iRow[m]].descriptors[iDescrip];
                sID = new String(stColor.sID);
              }
            }
          }

          if (iNew == 0)
          {
            iNew = 1;
            st = new rockColorStruct();

            st.sKEY = new String(cmn.cmnString.UniqueName() +
                                 ( (int) depthStart)); // Create a Unique KEY
            st.depthStart = depthStart; // Starting Depth
            st.depthEnd   = depthEnd;   // Ending Depth
            st.iTotal     = 0;
          }

          if (st != null)
          {
            if (st.iTotal < 3)
            {
              if (stColors.iColors > 0)
              {
                for (j=0; j<stColors.iColors; j++)
                {
                  if (stColors.stColors[j].sID.equals(sID))
                  {
                    iColor = j;
                  }
                }

                for (j=0; j<3; j++)
                {
                  st.iRGB[st.iTotal][j] = stColors.stColors[iColor].iRGB[j];
                }

                st.sColor[st.iTotal] =
                    new String(stColors.stColors[iColor].sPhrase);
                st.sMunsell[st.iTotal] =
                    new String(stColors.stColors[iColor].sMunsell);
                st.sRGBHex[st.iTotal] =
                    new String(stColors.stColors[iColor].sHex);

                st.iTotal++;  // Total Number of Colors for section
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Color Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseColorListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parseColorListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Color Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseColorListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseColorListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseColorListStruct copyList( parseColorListStruct stOld )
  {
    parseColorListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseColorListStruct();

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // NBS/ISCC Dictionary

      stNew.iColors  = stOld.iColors;
      stNew.stColors = new parseColorStruct[stOld.iColors];

      for (i=0; i<stOld.iColors; i++)
      {
        stNew.stColors[i] = copy( stOld.stColors[i] );
      }

      // Colors

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseColorStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseColorStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseColorStruct copy( parseColorStruct stOld )
  {
    int i=0;
    int j=0;
    parseColorStruct stNew = new parseColorStruct();

    // Color Lithology

    stNew.sID     = new String( stOld.sID );          // String ID for Color
    stNew.sPhrase = new String( stOld.sPhrase );      // Phrase

    stNew.iRGB = new int[3];                          // RGB Color
    for (i=0; i<3; i++)
      stNew.iRGB[i]  = stOld.iRGB[i];

    stNew.sHex       = new String( stOld.sHex );      // RGB Hex Color

    stNew.sMunsell   = new String( stOld.sMunsell );  // Munsel Equivalent Code

    stNew.iRockColor = stOld.iRockColor;              // Rock Color Indicator
    stNew.iSoilColor = stOld.iSoilColor;              // Soil Color Indicator

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parseColorStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseColorListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parseColorListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }

      for (i=0; i<st.iColors; i++)
      {
        print(st.stColors[i]);
      }
    }
  }

  /** Method print( parseColorStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parseColorStruct st )
  {
    if (st != null)
    {
      System.out.println( st.sID      + " <" + // String ID for Mask Symbol
                          st.sPhrase  + "> " +
                          st.iRGB[0]  + " " +
                          st.iRGB[1]  + " " +
                          st.iRGB[2]  + " " +
                          st.sMunsell + " " +
                          st.sHex);

      if (st.descriptors != null)
      {
        System.out.println( " ---- Descriptors ---- ");

        for (int i=0; i<st.iCount; i++)
        {
          print(st.descriptors[i]);
        }

        System.out.println( " ---- End Descriptors ---- ");
      }
    }
  }
}