/*
 * @ReadRockColumnSymbolsXMLFile.java Version 1.0 08/01/2012
 *
 * Copyright (c) 2012 John R. Victorine
 * All Rights Reserved.
 */

package parse.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import lith.lithology.lithologyColorsStruct;
import lith.lithology.lithologyGroupsStruct;
import parse.parseRockColumnSymbolsListStruct;
import parse.parseRockColumnSymbolsStruct;

/** Class ReadRockColumnSymbolsXMLFile()
 * <p> This Class will parse the Rock Lithology XML Data Stream
 *     directly to the Rock Lithology List Data Structure.
 *
 *  @version 1.1 08/01/2012
 *  @author  John R. Victorine
 */

public class ReadRockColumnSymbolsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  public static final int _NON    = parse.parseRockColumnSymbolsStruct._NONE; // None
  public static final int _TOP    = parse.parseRockColumnSymbolsStruct._TOP;
  public static final int _CENTER = parse.parseRockColumnSymbolsStruct._CENTER;
  public static final int _BASE   = parse.parseRockColumnSymbolsStruct._BASE;
  public static final int _ROW_l  = parse.parseRockColumnSymbolsStruct._ROW_l;
  public static final int _ROW_r  = parse.parseRockColumnSymbolsStruct._ROW_r;

  // **************************************************************************************

  // DTD terms

  /* ------------------------------------------------------------------------------------ *
   * <!DOCTYPE lithology [
   * <!ELEMENT lithology (background, foreground, groups, modifiers, abbreviation, rocks, keywords)+>
   * ------------------------------------------------------------------------------------ */

  public static final String ROOTNAME     = "lithology";    // ROOT NAME

  public static final String RECORDS      = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * -- Rock Column Background Colors
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT background (color*)>
   * <!ATTLIST background records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String BACKGROUND   = "background";   // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * -- Rock Column Foreground Colors
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT foreground (color*)>
   * <!ATTLIST foreground records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String FOREGROUND   = "foreground";   // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Color
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT color    EMPTY>
   * <!ATTLIST color    symbol  CDATA #IMPLIED
   *                    r       CDATA #IMPLIED
   *                    g       CDATA #IMPLIED
   *                    b       CDATA #IMPLIED>
   * ------------------------------------------------------------------------------------ */

  public static final String COLOR        = "color";        // ELEMENT

  public static final String SYMBOL       = "symbol";       // Symbol Text
  public static final String RED          = "r";            // Red
  public static final String GREEN        = "g";            // Green
  public static final String BLUE         = "b";            // BLue

  /* ------------------------------------------------------------------------------------ *
   * -- Rock Column Groups
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT groups   (group*)>
   * <!ATTLIST groups   records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String GROUPS       = "groups";       // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Rock Column Group
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT group    EMPTY>
   * <!ATTLIST group    mnemonic  CDATA #REQUIRED
   *                    desc      CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String GROUP        = "group";        // ELEMENT

  public static final String MNEMONIC     = "mnemonic";     // Mnemonic
  public static final String DESCRIPTION  = "desc";         // Description

  /* ------------------------------------------------------------------------------------ *
   * -- Modifiers to Rock Description
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT modifiers (modifier*)>
   * <!ATTLIST modifiers records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String MODIFIERS    = "modifiers";    // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Modifier to Rock Description
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT modifier EMPTY>
   * <!ATTLIST modifier id     CDATA #REQUIRED
   *                    word   CDATA #REQUIRED
   *                    modify CDATA #REQUIRED
   *                    phrase CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String MODIFIER     = "modifier";     // ELEMENT

  public static final String ID           = "id";           // Lithology ID
  public static final String WORD         = "word";         // ELEMENT
  public static final String MODIFY       = "modify";       // Modifies
  public static final String PHRASE       = "phrase";       // Word Phrase

  /* ------------------------------------------------------------------------------------ *
   * -- Abbreviations
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT abbreviation (words*)>
   * <!ATTLIST abbreviation records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String ABBREVIATION = "abbreviation"; // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Abbreviation Word list
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT words    EMPTY>
   * <!ATTLIST words    abbrev CDATA #REQUIRED
   *                    word   CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  //    WORDS (DEFINED)                   = "words";        // ELEMENT

  public static final String ABBREV       = "abbrev";       // Abbreviation used
  //    WORD  (DEFINED)                   = "word";         // Modifier

  /* ------------------------------------------------------------------------------------ *
   * -- List of Valid Rocks
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT rocks    (rock*)>
   * <!ATTLIST rocks    records CDATA #REQUIRED
   *                    type    CDATA #REQUIRED
   *                    rows    CDATA #REQUIRED
   *                    columns CDATA #REQUIRED
   *                    words   CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String ROCKS        = "rocks";        // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute
  public static final String TYPE         = "type";         // Type of Symbol text
  public static final String ROWS         = "rows";         // Number of Rows
  public static final String COLUMNS      = "columns";      // Number of Columns
  public static final String WORDS        = "words";        // Maximum Thesaurus

  /* ------------------------------------------------------------------------------------ *
   * .... Rock Column
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT rock     (column, symbols?, caps?, thesaurus?)>
   * ------------------------------------------------------------------------------------ */

  public static final String ROCK         = "rock";         // ELEMENT

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Information Data
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT column   EMPTY>
   * <!ATTLIST column   id       CDATA #REQUIRED
   *                    group    CDATA #IMPLIED
   *                    name     CDATA #REQUIRED
   *                    color    CDATA #IMPLIED
   *                    abrv     CDATA #IMPLIED
   *                    wthr     CDATA #IMPLIED>
   * ------------------------------------------------------------------------------------ */

  public static final String COLUMN       = "column";       // ELEMENT

  //    ID     (DEFINED)                  = "id";           // Rock Column ID
  //    GROUP  (DEFINED)                  = "group";        // Group Identifier
  public static final String NAME         = "name";         // Name of Rock
  //    COLOR  (DEFINED)                  = "color";        // Background color
  public static final String ABREV        = "abrv";         // Abbreviation
  //    TYPE                              = "type";         // 1st (primary), etc.
  public static final String CYCLE        = "cycle";        // sed (sedimentary), etc.
  public static final String ORDER        = "order";        // order of plotting
  public static final String WEATHER      = "wthr";         // Weathering Factor

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Symbols
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT symbols  (symbol*)>
   * <!ATTLIST symbols  records CDATA #REQUIRED
   *                    columns CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String SYMBOLS      = "symbols";      // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute
  //    COLUMNS (DEFINED)                 = "columns";      // Number of Columns

  /* ------------------------------------------------------------------------------------ *
   * .. -- .... Rock Column Symbol
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT symbol   EMPTY>
   * <!ATTLIST symbol   row     CDATA #IMPLIED
   *                    l       CDATA #IMPLIED
   *                    r       CDATA #IMPLIED
   *                    c       CDATA #IMPLIED>
   * ------------------------------------------------------------------------------------ */

  //    SYMBOL  (DEFINED)                 = "symbol";       // ELEMENT

  public static final String ROW          = "row";          // Row Number
  public static final String LEFT         = "l";            // Left Image Mask
  public static final String RIGHT        = "r";            // Right Image Mask
  public static final String CENTER       = "c";            // Center Image Mask

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Termination List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT caps     (cap*)>
   * <!ATTLIST caps     records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String CAPS         = "caps";         // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute
  //    COLUMNS (DEFINED)                 = "columns";      // Number of Columns

  /* ------------------------------------------------------------------------------------ *
   * .. -- .... Rock Column Termination  Symbol
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT cap      EMPTY>
   * <!ATTLIST cap      row     CDATA #IMPLIED
   *                    l       CDATA #IMPLIED
   *                    r       CDATA #IMPLIED
   *                    t       CDATA #IMPLIED
   *                    b       CDATA #IMPLIED
   *                    c       CDATA #IMPLIED>
   * ------------------------------------------------------------------------------------ */

  public static final String CAP          = "cap";          // ELEMENT

  //    ROW     (DEFINED)                 = "row";          // Row Number
  //    LEFT    (DEFINED)                 = "l";            // Left Image Mask
  //    RIGHT   (DEFINED)                 = "r";            // Right Image Mask
  public static final String TOP          = "t";            // Top Image Mask
  public static final String BASE         = "b";            // Base Image Mask
  //    CENTER  (DEFINED)                 = "c";            // Center Image Mask

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Thesaurus List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT thesaurus (word*)>
   * <!ATTLIST thesaurus records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String THESAURUS    = "thesaurus";    // ELEMENT

  /* ------------------------------------------------------------------------------------ *
   * -- Valid Keywords List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT keywords (word*)>
   * <!ATTLIST keywords records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String KEYWORDS     = "keywords";     // ELEMENT

  /* ------------------------------------------------------------------------------------ *
   * .... Word List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT word     EMPTY>
   * <!ATTLIST word     phrase CDATA #IMPLIED
   *                    type   CDATA #IMPLIED>]>
   * ------------------------------------------------------------------------------------ */

  //    WORD   (DEFINED)                  = "word";         // ELEMENT

  //    PHRASE (DEFINED)                  = "phrase";       // Word Phrase
  //    TYPE   (DEFINED)                  = "type";         // Type of Symbol text

  // **************************************************************************************

  // ELEMENT Identifiers

  private static final int _NONE       = 0;

  private static final int _BACKGROUND = 1;  // Read Background Colors
  private static final int _FOREGROUND = 2;  // Read Foreground Colors
  private static final int _COLOR      = 3;  // Individual Color indicator

  private static final int _GROUPS     = 4;  // Read Groups
  private static final int _GROUP      = 5;  // Individual Group indicator

  private static final int _MODIFIERS  = 6;  // modifiers
  private static final int _MODIFIER   = 7;  // modifier

  private static final int _ABBREV     = 8;  // abbreviation
  private static final int _WORDS      = 9;  // Abbreviation Words

  private static final int _ROCKS      = 10; // Read Rock Type
  private static final int _ROCK       = 11; // Individual Rock Type indicator
  private static final int _COLUMN     = 12; // Rock Type Column
  private static final int _SYMBOLS    = 13; // Read Rock Column Symbols
  private static final int _SYMBOL     = 14; // Individual Rock Column Symbol Line
  private static final int _CAPS       = 15; // Read Rock Column Symbols Terminator
  private static final int _CAP        = 16; // Individual Rock Column Symbol Line Terminator

  private static final int _THESAURUS  = 17; // Read Rock Column Thesaurus
  private static final int _WORD       = 18; // Individual Rock Column Word

  private static final int _KEYWORDS   = 19; // keywords
  private static final int _KEYWORD    = 20; // keywords Word

  // Global Variables

  private int                              iRows    = 0;
  private int                              iLines   = 0;
  private int                              iColumns = 0;
  private int                              iCount   = -1;
  private int                              iRow     = 0;
  private int                              iData    = _NONE;
  private int                              iColor   = _NONE;
  private parseRockColumnSymbolsListStruct stList   = new parseRockColumnSymbolsListStruct();

  private int                              iError   = 0;
  private String                           sError   = "";

  /** Method ReadRockColumnSymbolsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadRockColumnSymbolsXMLFile() { iType = FILE; }

  /** Method ReadRockColumnSymbolsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadRockColumnSymbolsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public parseRockColumnSymbolsListStruct Process(String filename)
  {
    Read(filename);

//    stList = lith.lithology.parseRockColumnSymbolsUtility.setHierarchy(stList);
//    stList = lith.lithology.parseRockColumnSymbolsUtility.bubbleSort(stList);

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadRockColumnSymbolsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadRockColumnSymbolsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name
    int    m     = 0;
    int    n     = 0;
    int    iCol  = 0;

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    if (eName.equals(ROOTNAME))     // ROOT NAME Element
    {
      iRow  = -1;
      iData = _NONE;
	}

    if (eName.equals(BACKGROUND))   // Background Color Element
    {
      iRow                = -1;
      iData               = _BACKGROUND;
      iColor              = _BACKGROUND;
      stList.stBackground = new lithologyColorsStruct();
	}

    if (eName.equals(FOREGROUND))   // Foreground Color Element
    {
      iRow                = -1;
      iData               = _FOREGROUND;
      iColor              = _FOREGROUND;
      stList.stForeground = new lithologyColorsStruct();
	}

    if (eName.equals(COLOR))        // Color Element
    {
      iData = _COLOR;
      iRow++;
	}

    if (eName.equals(GROUPS))       // Groups Element
    {
      iRow            = -1;
      iColor          = _NONE;
      iData           = _GROUPS;
      stList.stGroups = new lithologyGroupsStruct();
	}

    if (eName.equals(GROUP))        // Group Element
    {
      iData  = _GROUP;
      iRow++;
	}

    if (eName.equals(MODIFIERS))    // Modifier Element
    {
      iData  = _MODIFIERS;
      iRow   = -1;
	}

    if (eName.equals(MODIFIER))     // Modifier Element
    {
      iData  = _MODIFIER;
      iRow++;
	}

    if (eName.equals(ABBREVIATION)) // Abbreviation Terms Element
    {
      iData  = _ABBREV;
      iRow   = -1;
	}

    if (eName.equals(WORDS))        // Words Element
    {
      iData  = _WORDS;
      iRow++;
	}

    if (eName.equals(ROCKS))        // Rocks Element
    {
      iData  = _ROCKS;
      iRow   = -1;
	}

    if (eName.equals(ROCK))         // Rock Element
    {
      iData  = _ROCK;
      iRow++;
      stList.stItem[iRow] = new parseRockColumnSymbolsStruct();
	}

    if (eName.equals(COLUMN))       // Column Element
    {
      iData  = _COLUMN;
	}

    if (eName.equals(SYMBOLS))      // Rock Column Symbols Element
    {
      iData  = _SYMBOLS;
      iCount = -1;
	}

    if (eName.equals(SYMBOL))       // Rock Column Symbol Element
    {
      iData  = _SYMBOL;
	}

    if (eName.equals(CAPS))         // Rock Column End Caps Element
    {
      iData  = _CAPS;
      iCount = -1;
	}

    if (eName.equals(CAP))          // Rock Column End Cap Element
    {
      iData  = _CAP;
	}

    if (eName.equals(THESAURUS))    // Rock Column Thesaurus Terms Element
    {
      iData  = _THESAURUS;
      iCount = -1;
	}

    if (eName.equals(KEYWORDS))     // Valid Rock Column Keywords
    {
      iData  = _KEYWORDS;
      iCount = -1;
	}

    if (eName.equals(WORD))         // Keyword Element
    {
      if (iData == _THESAURUS)
        iData = _WORD;
      else if (iData == _KEYWORDS)
        iData  = _KEYWORD;
      iCount++;
	}

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        if (iData == _CAPS)
        {
          if (aName.equals(COLUMNS))
          {
		    iCol = cmn.cmnString.stringToInt(sOut.trim());
            stList.stItem[iRow].iCaps  = iCol;  // Total number of columns
            stList.stItem[iRow].iTypes = new int[3];
            if (iCol == 2)
            {
              stList.stItem[iRow].iTypes[0] = _ROW_l;
              stList.stItem[iRow].iTypes[1] = _ROW_r;
              stList.stItem[iRow].iTypes[1] = _NON;
		    }
            else
            {
              stList.stItem[iRow].iTypes[0] = _TOP;
              stList.stItem[iRow].iTypes[1] = _BASE;
              stList.stItem[iRow].iTypes[1] = _CENTER;
		    }
		  }
		}

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
	      switch (iData)
	      {
            case _NONE:
              break;

            case _BACKGROUND:  // Read Background Colors
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stBackground.iColors = iRows;
              stList.stBackground.sColors = new String[iRows];
              stList.stBackground.iRGB    = new int[iRows][3];

              // Initialize the Color Variables

              for (m=0; m<iRows; m++)
              {
                stList.stBackground.sColors[m] = new String("");
                for (n=0; n<3; n++)
                {
                  stList.stBackground.iRGB[m][n] = 255;
                }
              }
              break;
            case _FOREGROUND:  // Read Foreground Colors
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stForeground.iColors = iRows;
              stList.stForeground.colors  = new char[iRows];
              stList.stForeground.iRGB    = new int[iRows][3];

              // Initialize the Color Variables

              for (m=0; m<iRows; m++)
              {
                stList.stForeground.colors[m] = ' ';
                for (n=0; n<3; n++)
                {
                  stList.stForeground.iRGB[m][n] = 255;
                }
              }
              break;
            case _COLOR:       // Individual Color indicator
              break;

            case _GROUPS:      // Read Groups
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stGroups.iGroups = iRows;
              stList.stGroups.sGroups = new String[iRows][2];

              // Initialize the Group Variables

              for (m=0; m<iRows; m++)
              {
                for (n=0; n<2; n++)
                {
                  stList.stGroups.sGroups[m][n] = new String("");
                }
              }
              break;
            case _GROUP:       // Individual Group indicator
              break;

            case _MODIFIERS:   // modifiers
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iModifiers = iRows;
              stList.sModifier  = new String[iRows][3];
              break;
            case _MODIFIER:    // modifier
              break;

            case _ABBREV:      // abbreviation
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iAbbrev = iRows;
              stList.sAbbrev = new String[iRows][2];
              break;
            case _WORDS:       // Abbreviation Words
              break;

            case _ROCKS:       // Read Rock Type
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iCount = iRows;
              stList.stItem = new parseRockColumnSymbolsStruct[iRows];
              break;
            case _ROCK:        // Individual Rock Type indicator
              break;
            case _COLUMN:      // Rock Type Column
              break;
            case _SYMBOLS:     // Read Rock Column Symbols
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stItem[iRow].iRows  = iRows;
              stList.stItem[iRow].symbol = new String[iRows][2];

              // Initialize the Symbol Variable

              for (m=0; m<iRows; m++)
              {
                for (n=0; n<2; n++)
                {
                  stList.stItem[iRow].symbol[m][n] = new String("....................");
                }
              }
              break;
            case _SYMBOL:      // Individual Rock Column Symbol Line
              break;
            case _CAPS:
              // Read Rock Column Symbols Terminator
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stItem[iRow].iCapRows = iRows;
              stList.stItem[iRow].caps     = new String[iRows][3];

              // Initialize the Symbol Variable

              for (m=0; m<iRows; m++)
              {
                for (n=0; n<3; n++)
                {
                  stList.stItem[iRow].caps[m][n] = new String("");
                }
              }
              break;
            case _CAP:         // Individual Rock Column Symbol Line Terminator
              break;

            case _THESAURUS:   // Read Rock Column Thesaurus
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stItem[iRow].iThesaurus = iRows;
              stList.stItem[iRow].sThesaurus = new String[iRows];
              stList.stItem[iRow].iType      = new int[iRows];

              // Initialize the Thesaurus Variable

              for (m=0; m<iRows; m++)
              {
                stList.stItem[iRow].sThesaurus[m] = new String("");
                stList.stItem[iRow].iType[m]      = 5;
              }
              break;
            case _WORD:        // Individual Rock Column Word
              break;

            case _KEYWORDS:    // keywords
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iKeywords = iRows;
              stList.sKeywords = new String[iRows][2];
              break;
            case _KEYWORD:     // keywords Word
              break;
		  }
        }

	    switch (iData)
	    {
          case _NONE:
            break;

          case _BACKGROUND:  // Read Background Colors
          case _FOREGROUND:  // Read Foreground Colors
            break;
          case _COLOR:       // Individual Color indicator
            switch (iColor)
            {
              case _FOREGROUND:  // Read Foreground Colors
                stList.stForeground =
                    ParseData( aName, sOut.trim(), stList.stForeground );
                break;

              case _BACKGROUND:  // Read Background Colors
                stList.stBackground =
                    ParseData( aName, sOut.trim(), stList.stBackground );
                break;
            }
            break;

          case _GROUPS:      // Read Groups
            break;
          case _GROUP:       // Individual Group indicator
            stList.stGroups = ParseData( aName, sOut.trim(), stList.stGroups );
            break;

          case _MODIFIERS:   // modifiers
            break;
          case _MODIFIER:    // modifier
            parseModifier( aName, sOut.trim() );
            break;

          case _ABBREV:      // abbreviation
            break;
          case _WORDS:       // Abbreviation Words
            parseAbbreviation( aName, sOut.trim() );
            break;

          case _ROCKS:       // Read Rock Type
/*
            if (aName.equals(ROWS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                iLines = cmn.cmnString.stringToInt(sOut.trim());
            }

            if (aName.equals(COLUMNS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                iColumns = cmn.cmnString.stringToInt(sOut.trim());
            }
*/
            break;
          case _ROCK:        // Individual Rock Type indicator
            break;
          case _COLUMN:      // Rock Type Column
            stList.stItem[iRow] =
                ParseData( aName, sOut.trim(), stList.stItem[iRow] );
            break;
          case _SYMBOLS:     // Read Rock Column Symbols
            if (aName.equals(COLUMNS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                stList.stItem[iRow].iColumns = cmn.cmnString.stringToInt(sOut.trim());
            }
            break;
          case _SYMBOL:      // Individual Rock Column Symbol Line
            stList.stItem[iRow] =
                ParseSymbol( aName, sOut.trim(), stList.stItem[iRow] );
            break;
          case _CAPS:        // Read Rock Column Symbols Terminator
            if (aName.equals(COLUMNS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                stList.stItem[iRow].iCaps = cmn.cmnString.stringToInt(sOut.trim());
            }
            break;
          case _CAP:         // Individual Rock Column Symbol Line Terminator
            stList.stItem[iRow] =
                ParseCap( aName, sOut.trim(), stList.stItem[iRow] );
            break;

          case _THESAURUS:   // Read Rock Column Thesaurus
            break;
          case _WORD:        // Individual Rock Column Word
            if (aName.equals(PHRASE))
            {
              stList.stItem[iRow].sThesaurus[iCount] = new String(sOut.trim());
            }
            if (aName.equals(TYPE))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                stList.stItem[iRow].iType[iCount] = cmn.cmnString.stringToInt(sOut.trim());
			}
            break;

          case _KEYWORDS:    // keywords
            break;
          case _KEYWORD:     // keywords Word
            parseKeywords( aName, sOut.trim() );
            break;
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData( String sIdentifier, String sData, lithologyColorsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = Color Data Structure
   * @return st          = Color Data Structure
   */

  private lithologyColorsStruct ParseData( String sIdentifier,
                                           String sData,
                                           lithologyColorsStruct st )
  {
    char c[] = null;

    if (sIdentifier.equals(SYMBOL))      // Symbol Text
    {
      switch (iColor)
      {
        case _BACKGROUND:  // Read Background Colors
          st.sColors[iRow] = new String(sData);
          break;
        case _FOREGROUND:  // Read Foreground Colors
          c = sData.toCharArray();
          if (c.length > 0) st.colors[iRow] = c[0];
          break;
      }
    }

    if (sIdentifier.equals(RED))           // Red
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][0] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(GREEN))          // Green
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][1] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(BLUE))          // BLue
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][2] = cmn.cmnString.stringToInt(sData);
      }
    }

    return (st);
  }

  /** ParseData( String sIdentifier, String sData, lithologyGroupsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = Group Data Structure
   * @return st          = Group Data Structure
   */

  private lithologyGroupsStruct ParseData( String sIdentifier,
                                           String sData,
                                           lithologyGroupsStruct st )
  {
    if (sIdentifier.equals(MNEMONIC))
    {
      st.sGroups[iRow][lith.lithology.lithologyGroupsStruct._MNEMONIC] =
          new String(sData);
    }

    if (sIdentifier.equals(DESCRIPTION))
    {
      st.sGroups[iRow][lith.lithology.lithologyGroupsStruct._DESCRIPTION] =
          new String(sData);
    }

    return (st);
  }

  /** ParseData( String sIdentifier, String sData, parseRockColumnSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = lithology Data Structure
   * @return st          = lithology Data Structure
   */

  private parseRockColumnSymbolsStruct ParseData( String sIdentifier,
                                                  String sData,
                                                  parseRockColumnSymbolsStruct st)
  {
    String sTemp  = "";
    int    iColor = -1;

    if (sIdentifier.equals(ID)) { st.sID = new String( sData);  }

    if (sIdentifier.equals(GROUP))      // Group Mnemonic
    {
      if (stList.stGroups != null)
      {
        st.iGroup = lith.lithology.lithologySymbolsUtility.getGroupID(
                    lith.lithology.lithologyGroupsStruct._MNEMONIC,
                    sData,
                    stList.stGroups );

        st.sGroup = new String(
            lith.lithology.lithologySymbolsUtility.getGroup(
                    lith.lithology.lithologyGroupsStruct._DESCRIPTION,
                    st.iGroup,
                    stList.stGroups ));
      }
    }

    if (sIdentifier.equals(COLOR))
    {
      iColor  = lith.lithology.lithologySymbolsUtility.getColor(
                  sData, stList.stBackground );
      st.iRGB = lith.lithology.lithologySymbolsUtility.getRGB(
                  iColor, stList.stBackground );
    }

    if (sIdentifier.equals(ABREV))    { st.sAbrev    = new String(sData); }
    if (sIdentifier.equals(NAME))     { st.sName     = new String(sData); } //System.out.println(sData);}
    if (sIdentifier.equals(TYPE))     { st.sType     = new String(sData); }
    if (sIdentifier.equals(CYCLE))    { st.sCycle    = new String(sData); }

    if (sIdentifier.equals(ORDER))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iOrder = cmn.cmnString.stringToInt(sData);
	}


    if (sIdentifier.equals(WEATHER))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iWeather = cmn.cmnString.stringToInt(sData);
	}

    return (st);
  }

  /** ParseSymbol( String sIdentifier, String sData, parseRockColumnSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = lithology Data Structure
   * @return st          = lithology Data Structure
   */

  private parseRockColumnSymbolsStruct ParseSymbol( String sIdentifier,
                                                    String sData,
                                                    parseRockColumnSymbolsStruct st)
  {

    if (sIdentifier.equals(ROW))   // Rows Number
    {
//      st.iRows    = iLines;
//      st.iColumns = iColumns;

      if (cmn.cmnString.isNumeric(sData))
        iCount = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(LEFT))  // Left Image
    {
      st.symbol[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(RIGHT)) // right Image
    {
      st.symbol[iCount][1] = new String(sData);
    }

    return (st);
  }

  /** ParseCap( String sIdentifier, String sData, parseRockColumnSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = lithology Data Structure
   * @return st          = lithology Data Structure
   */

  private parseRockColumnSymbolsStruct ParseCap( String sIdentifier,
                                                 String sData,
                                                 parseRockColumnSymbolsStruct st)
  {
    if (sIdentifier.equals(ROW))   // Rows Number
    {
//      st.iCapRows = iLines;
//      st.iCaps    = iColumns;

      if (cmn.cmnString.isNumeric(sData))
        iCount = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(LEFT))  // Left Image
    {
      st.caps[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(RIGHT)) // right Image
    {
      st.caps[iCount][1] = new String(sData);
    }

    if (sIdentifier.equals(TOP))  // Top Image
    {
      st.caps[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(BASE)) // Base Image
    {
      st.caps[iCount][1] = new String(sData);
    }

    if (sIdentifier.equals(CENTER)) // Center Image
    {
      st.caps[iCount][2] = new String(sData);
    }

    return (st);
  }

  /** parseModifier( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseModifier( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ID))  {}   // Modifier ID
    if (sIdentifier.equals(WORD))   // Modifier
    {
      if (iRow > -1)
        stList.sModifier[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(MODIFY)) // Modifies
    {
      if (iRow > -1)
        stList.sModifier[iRow][1] = new String(sData);
    }

    if (sIdentifier.equals(PHRASE)) // Resultant Phrase
    {
      if (iRow > -1)
        stList.sModifier[iRow][2] = new String(sData);
    }
  }

  /** parseAbbreviation( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseAbbreviation( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ABBREV)) // Abbreviation
    {
      if (iRow > -1)
        stList.sAbbrev[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(WORD))   // Word or Phrase
    {
      if (iRow > -1)
        stList.sAbbrev[iRow][1] = new String(sData);
    }
  }

  /** parseKeywords( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseKeywords( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(PHRASE)) // Abbreviation
    {
      if (iCount > -1)
        stList.sKeywords[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(TYPE))   // Word or Phrase
    {
      if (iCount > -1)
        stList.sKeywords[iCount][1] = new String(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}

/*
 *  @version 1.1 08/01/2012
 *  @author  John Victorine
 */
