/*
 * @ReadParseMineralsXMLFile.java Version 1.0 12/11/2009
 *
 * Copyright (c) 2009 John R. Victorine
 * All Rights Reserved.
 */

package parse.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import parse.parseMineralsListStruct;
import parse.parseMineralsStruct;

/** Class ReadParseMineralsXMLFile()
 * <p> This Class will parse the Texture Thesaurus XML Data Stream
 *     directly to the Textures Thesaurus List Data Structure.
 *
 *  @version 1.1 12/11/2009
 *  @author  John R. Victorine
 */

public class ReadParseMineralsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  // DTD terms

  public static final String ROOTNAME  = "minerals";   // ROOT NAME
  public static final String RECORDS   = "records";   // Attribute

  public static final String DATA      = "data";    // ELEMENT

  public static final String MINERAL   = "mineral"; // ELEMENT

  public static final String ID        = "id";      // Symbol ID
  public static final String NAME      = "name";    // Mineral Name
  public static final String GROUP     = "group";   // Mineral Group
  public static final String MIN_CLASS = "class";   // Mineral Class
  public static final String FORMULA   = "formula"; // Mineral Formula
  public static final String OCCURANCE = "occ";     // Mineral Occurance

  public static final String  MOL_WT   = "mol_wt";
  public static final String  RHOB     = "rhob";
  public static final String  RHOE     = "rhoe";
  public static final String  RHOL     = "rhol";
  public static final String  GR       = "gr";
  public static final String  PE_C     = "pe_calc";
  public static final String  PE_M     = "pe";
  public static final String  U        = "u";
  public static final String  SIGMA_C  = "sigma_calc";
  public static final String  SIGMA_M  = "sigma_meas";
  public static final String  LS       = "ls";
  public static final String  LD       = "ld";
  public static final String  CNL_C    = "cnl_calc";
  public static final String  CNL_M    = "cnl_meas";
  public static final String  CNL_T_C  = "cnl_therm_calc";
  public static final String  CNL_T_M  = "cnl_therm_meas";
  public static final String  BMOD     = "bmod";
  public static final String  SMOD     = "smod";
  public static final String  DT_COMP  = "dt_comp";  public static final String  DT_SHEAR = "dt_shear";
  public static final String  PR       = "pr";
  public static final String  EPT      = "ept";

  private static final int _NONE       = 0; // texture
  private static final int _DATA       = 1; // data
  private static final int _MINERAL    = 2; // mineral

  private int                     iData  = _NONE;

  private int                     iRows  = 0;
  private int                     iCount = -1;
  private int                     iRow   = 0;
  private parseMineralsListStruct stList = new parseMineralsListStruct();

  private int                     iError = 0;
  private String                  sError = "";

  /** Method ReadParseMineralsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadParseMineralsXMLFile() { iType = FILE; }

  /** Method ReadParseMineralsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadParseMineralsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public parseMineralsListStruct Process(String filename)
  {
    Read(filename);
    stList = parse.parseMineralsUtility.computeRhomaaUmaa( stList );

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadParseMineralsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadParseMineralsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name
    int    m = 0;
    int    n = 0;

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    if (eName.equals(MINERAL))
    {
      iData = _MINERAL; // mineral
      iCount++;
      stList.stItem[iCount] = new parseMineralsStruct();
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows         = cmn.cmnString.stringToInt(sOut.trim());
          stList.iCount = iRows;
          stList.stItem = new parseMineralsStruct[iRows];
        }

        if (iCount > -1)
        {
          stList.stItem[iCount] =
                ParseData( aName, sOut.trim(), stList.stItem[iCount] );
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData( String sIdentifier, String sData, parseMineralsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = parse Data Structure
   * @return st          = parse Data Structure
   */

  private parseMineralsStruct ParseData( String sIdentifier,
                                         String sData,
                                         parseMineralsStruct st)
  {
    String sTemp  = "";

    if (sIdentifier.equals(ID))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.id = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(NAME))      { st.sName  = new String(sData); }
    if (sIdentifier.equals(GROUP))     { st.sGroup  = new String(sData); }
    if (sIdentifier.equals(MIN_CLASS)) { st.sClass  = new String(sData); }
    if (sIdentifier.equals(FORMULA))   { st.sFormula  = new String(sData); }
    if (sIdentifier.equals(OCCURANCE)) { st.sOccurrence  = new String(sData); }

    if (sIdentifier.equals(MOL_WT))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dMolWt = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(RHOB))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dRHOB = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(RHOE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dRHOE = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(RHOL))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dRHOL = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(GR))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dGR = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(PE_C))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dPECalc = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(PE_M))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dPE = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(U))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dU = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(SIGMA_C))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dSigmaCalc = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(SIGMA_M))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dSigma = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LS))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dLs = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LD))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dLd = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(CNL_C))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dCNLCalc = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(CNL_M))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dCNL = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(CNL_T_C))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dCNL_TCalc = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(CNL_T_M))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dCNL_T = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(BMOD))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dBulk = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(SMOD))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dShear = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(DT_COMP))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dt_comp = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(DT_SHEAR))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dt_shear = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(PR))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dPoisson = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(EPT))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dEPT = cmn.cmnString.stringToDouble(sData);
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
