/*
 * @ReadParseDepEnvXMLFile.java Version 1.0 11/02/2011
 *
 * Copyright (c) 2011 John R. Victorine
 * All Rights Reserved.
 */

package parse.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import parse.parseDepEnvListStruct;
import parse.parseDepEnvStruct;

/** Class ReadParseDepEnvXMLFile()
 * <p> This Class will parse the Texture Thesaurus XML Data Stream
 *     directly to the Textures Thesaurus List Data Structure.
 *
 *  @version 1.1 11/02/2011
 *  @author  John R. Victorine
 */

public class ReadParseDepEnvXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  // DTD terms

  public static final String ROOTNAME     = "texture";   // ROOT NAME
  public static final String RECORDS      = "records";   // Attribute

  public static final String BACKGROUND   = "background"; // ELEMENT

  public static final String COLOR        = "color";      // ELEMENT

  public static final String RED          = "r";          // Red
  public static final String GREEN        = "g";          // Green
  public static final String BLUE         = "b";          // BLue

  public static final String MODIFIERS    = "modifiers";  // ELEMENT

  public static final String MODIFIER     = "modifier";   // ELEMENT

  public static final String ID           = "id";     // Modifier ID
  public static final String WORD         = "word";   // Modifier
  public static final String MODIFY       = "modify"; // Modifies
  public static final String PHRASE       = "phrase"; // Resultant Phrase

  public static final String ABBREVIATION = "abbreviation";   // ELEMENT

  public static final String WORDS        = "words";   // Modifier
  public static final String ABBREV       = "abbrev";  // Abbreviation used

  public static final String DATA         = "data";    // ELEMENT

  public static final String ENVIRON      = "environ"; // ELEMENT

  public static final String NAME         = "name";    // Description of Texture
  public static final String LEVEL        = "level";   // Energy Value
  public static final String LEVEL2       = "level2";  // Energy Value 2
  public static final String CONTINENTAL  = "contl";   // Continental
  public static final String COASTAL      = "cstl";    // Coastal
  public static final String FRESH        = "frsh";    // Fresh Water
  public static final String BRACKISH     = "brcksh";  // Brackish Water
  public static final String SHOREFACE    = "shore";   // Shoreface
  public static final String OFF_SHORE_T  = "off_tran"; // Off shore trasition
  public static final String OFF_SHORE    = "offshr";  // Off Shore
  public static final String NERITIC_M    = "shlf_m";  // Middle Neritic
  public static final String NERITIC_O    = "shlf_o";  // Outer Neritic
  public static final String BATHYAL_U    = "bthyl_u"; // Upper Bathyal
  public static final String BATHYAL_M    = "bthyl_m"; // Middle Bathyal
  public static final String BATHYAL_L    = "bthyl_l"; // Lower Bathyal
  public static final String ABYSSAL      = "abyssl";  // Abyssal
  public static final String TYPE         = "type";    // Type of Dep Environment
  public static final String GROUP        = "group";   // Group within the Type
  public static final String GROUP2       = "group2";  // Group within the Group

  public static final String DESCRIPTORS  = "descriptors"; // ELEMENT

  public static final String DESCRIPTOR   = "descriptor"; // ELEMENT


  public static final String KEYWORDS     = "keywords";   // ELEMENT

                                                // ELEMENT
  private static final int _NONE          = 0;  // texture
  private static final int _BG            = 1;  // background
  private static final int _COLOR         = 2;  // background
  private static final int _MODIFIERS     = 3;  // modifiers
  private static final int _MODIFIER      = 4;  // modifier
  private static final int _ABBREV        = 5;  // abbreviation
  private static final int _WORDS         = 6;  // abbreviation Words
  private static final int _DATA          = 7;  // data
  private static final int _ENVIRON       = 8;  // environ
  private static final int _DESCRIPTORS   = 9;  // descriptors
  private static final int _DESCRIPTOR    = 10; // descriptor
  private static final int _KEYWORDS      = 11; // keywords
  private static final int _WORD          = 12; // keywords Word

  private int                    iData    = _NONE;
  private int                    iLevel   = _NONE;

  private int                    iRows    = 0;
  private int                    iCount   = -1;
  private int                    iRow     = 0;
  private parseDepEnvListStruct stList   = new parseDepEnvListStruct();

  private int                    iError   = 0;
  private String                 sError   = "";

  /** Method ReadParseDepEnvXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadParseDepEnvXMLFile() { iType = FILE; }

  /** Method ReadParseDepEnvXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadParseDepEnvXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public parseDepEnvListStruct Process(String filename)
  {
    Read(filename);
    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadParseDepEnvXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadParseDepEnvXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name
    int    m = 0;
    int    n = 0;

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    /* ---------------------------- *
     * Texture Background Color
     * ---------------------------- */

    if (eName.equals(BACKGROUND))
    {
      iData  = _BG; // background
      iRow = -1;
    }

    if (eName.equals(COLOR))
    {
      iData = _COLOR;  // color
      iRow++;
    }

    /* ---------------------------- *
     * Modifiers to Descriptors
     * ---------------------------- */

    if (eName.equals(MODIFIERS))
    {
      iData = _MODIFIERS; // modifiers
      iRow = -1;
    }

    if (eName.equals(MODIFIER))
    {
      iData = _MODIFIER; // modifier
      iRow++;
    }

    /* ---------------------------- *
     * Abbreviated Terms
     * ---------------------------- */

    if (eName.equals(ABBREVIATION))
    {
      iData = _ABBREV; // abbreviation
      iRow = -1;
    }

    if (eName.equals(WORDS))
    {
      iData = _WORDS; // abbreviation Words
      iRow++;
    }

    /* ---------------------------- *
     * Mineral Data
     * ---------------------------- */

    if (eName.equals(DATA))
    {
      iData = _DATA; // data
      iCount = -1;
    }

    if (eName.equals(ENVIRON))
    {
      iData = _ENVIRON; // environ
      iCount++;
      stList.stItem[iCount] = new parseDepEnvStruct();
    }

    /* ---------------------------- *
     * Descriptors of Minerals
     * ---------------------------- */

    if (eName.equals(DESCRIPTORS))
    {
      iData = _DESCRIPTORS; // descriptors
      iRow = -1;
    }

    if (eName.equals(DESCRIPTOR))
    {
      iData = _DESCRIPTOR; // descriptor
      iRow++;
      stList.stItem[iCount].descriptors[iRow] = new parseDepEnvStruct();
    }

    /* ---------------------------- *
     * Texture Keywords
     * ---------------------------- */

    if (eName.equals(KEYWORDS))
    {
      iData = _KEYWORDS; // keywords
      iRow = -1;
    }

    if (eName.equals(WORD))
    {
      iData = _WORD; // keywords Word
      iRow++;
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _BG:          // background
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iColors = iRows;
              stList.sColors = new String[iRows];
              stList.iRGB    = new int[iRows][3];
              break;
            case _COLOR:       // background
              break;
            case _MODIFIERS:   // modifiers
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iModifiers = iRows;
              stList.sModifier  = new String[iRows][3];
              break;
            case _MODIFIER:    // modifier
              break;
            case _ABBREV:      // abbreviation
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iAbbrev = iRows;
              stList.sAbbrev = new String[iRows][2];
              break;
            case _WORDS:       // abbreviation Words
              break;
            case _DATA:        // data
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iCount = iRows;
              stList.stItem = new parseDepEnvStruct[iRows];
              break;
            case _ENVIRON:     // environ
              break;
            case _DESCRIPTORS: // descriptors
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              if (iCount > -1)
              {
                if (stList.stItem[iCount] != null)
                {
                  stList.stItem[iCount].iCount = iRows;
                  stList.stItem[iCount].descriptors =
                      new parseDepEnvStruct[iRows];
                }
              }
              break;
            case _DESCRIPTOR:  // descriptor
              break;
            case _KEYWORDS:    // keywords
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iKeywords = iRows;
              stList.sKeywords = new String[iRows][2];
              break;
            case _WORD:        // keywords Word
              break;
          }
        }

        switch (iData)
        {
          case _NONE:
            break;

          case _BG:          // background
            break;
          case _COLOR:       // background
            parseColor( aName, sOut.trim() );
            break;
          case _MODIFIERS:   // modifiers
            break;
          case _MODIFIER:    // modifier
            parseModifier( aName, sOut.trim() );
            break;
          case _ABBREV:      // abbreviation
            break;
          case _WORDS:       // abbreviation Words
            parseAbbreviation( aName, sOut.trim() );
            break;
          case _DATA:        // data
            break;
          case _ENVIRON:     // environ
            stList.stItem[iCount] =
                ParseData( aName, sOut.trim(), stList.stItem[iCount] );
            break;
          case _DESCRIPTORS: // descriptors
            break;
          case _DESCRIPTOR:  // descriptor
            stList.stItem[iCount].descriptors[iRow] =
                ParseData( aName, sOut.trim(),
                           stList.stItem[iCount].descriptors[iRow] );
            break;
          case _KEYWORDS:    // keywords
            break;
          case _WORD:        // keywords Word
            parseKeywords( aName, sOut.trim() );
            break;
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** parseColor( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseColor( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(NAME))      // Color Name
    {
      if (iRow > -1)
        stList.sColors[iRow] = new String(sData);
    }

    if (sIdentifier.equals(RED))           // Red
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          stList.iRGB[iRow][0] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(GREEN))          // Green
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          stList.iRGB[iRow][1] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(BLUE))          // Blue
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          stList.iRGB[iRow][2] = cmn.cmnString.stringToInt(sData);
      }
    }
  }

  /** parseModifier( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseModifier( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(WORD))   // Modifier
    {
      if (iRow > -1)
        stList.sModifier[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(MODIFY)) // Modifies
    {
      if (iRow > -1)
        stList.sModifier[iRow][1] = new String(sData);
    }

    if (sIdentifier.equals(PHRASE)) // Resultant Phrase
    {
      if (iRow > -1)
        stList.sModifier[iRow][2] = new String(sData);
    }
  }

  /** parseAbbreviation( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseAbbreviation( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ABBREV)) // Abbreviation
    {
      if (iRow > -1)
        stList.sAbbrev[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(WORD))   // Word or Phrase
    {
      if (iRow > -1)
        stList.sAbbrev[iRow][1] = new String(sData);
    }
  }

  /** parseKeywords( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseKeywords( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(PHRASE)) // Abbreviation
    {
      if (iRow > -1)
        stList.sKeywords[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(TYPE))   // Word or Phrase
    {
      if (iRow > -1)
        stList.sKeywords[iRow][1] = new String(sData);
    }
  }

  /** ParseData( String sIdentifier, String sData, parseDepEnvStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = parse Data Structure
   * @return st          = parse Data Structure
   */

  private parseDepEnvStruct ParseData( String sIdentifier,
                                       String sData,
                                       parseDepEnvStruct st)
  {
    if (sIdentifier.equals(NAME))     { st.sName   = new String(sData); }
    if (sIdentifier.equals(TYPE))     { st.sType   = new String(sData); }
    if (sIdentifier.equals(GROUP))    { st.sGroup  = new String(sData); }
    if (sIdentifier.equals(GROUP2))   { st.sGroup2 = new String(sData); }

    if (sIdentifier.equals(CONTINENTAL))  // Continental
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.icontl = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(COASTAL))   // Coastal
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.icstl = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(FRESH))    // Fresh Water
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ifrsh = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(BRACKISH))  // Brackish Water
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ibrcksh = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(SHOREFACE))   // Shoreface
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ishore = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(OFF_SHORE_T)) // Off shore trasition
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ioff_tran = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(OFF_SHORE))  // Off Shore
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ioffshr = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(NERITIC_M))  // Middle Neritic
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ishlf_m = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(NERITIC_O))  // Outer Neritic
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ishlf_o = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(BATHYAL_U)) // Upper Bathyal
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ibthyl_u = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(BATHYAL_M)) // Middle Bathyal
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ibthyl_m = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(BATHYAL_L)) // Lower Bathyal
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.ibthyl_l = cmn.cmnString.stringToInt(sData);
	  }
    }

    if (sIdentifier.equals(ABYSSAL))  // Abyssal
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iabyssl = cmn.cmnString.stringToInt(sData);
	  }
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}

/*
 *  @version 1.1 11/02/2011
 *  @author  John Victorine
 */
