/*
 * @ReadGrainsTexturesXMLFile.java Version 1.0 02/03/2016
 *
 * Copyright (c) 2016 John R. Victorine
 * All Rights Reserved.
 */

package parse.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import parse.parseGrainsTextureListStruct;
import parse.parseGrainsTextureStruct;

/** Class ReadGrainsTexturesXMLFile()
 * <p> This Class will parse the Rock Lithology XML Data Stream
 *     directly to the Rock Lithology List Data Structure.
 *
 *  @version 1.1 02/03/2016
 *  @author  John R. Victorine
 */

public class ReadGrainsTexturesXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  // **************************************************************************************

  // DTD terms

  /* ------------------------------------------------------------------------------------ *
   * <!DOCTYPE grains [
   * <!ELEMENT grains (modifiers, abbrev_main, abbrev_desc, rocks, keywords)+>
   * ------------------------------------------------------------------------------------ */

  public static final String ROOTNAME     = "grains ";      // ROOT NAME

  public static final String RECORDS      = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * -- Modifiers to Rock Description
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT modifiers (modifier*)>
   * <!ATTLIST modifiers records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String MODIFIERS    = "modifiers";    // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Modifier to Rock Description
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT modifier EMPTY>
   * <!ATTLIST modifier word   CDATA #REQUIRED
   *                    modify CDATA #REQUIRED
   *                    phrase CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String MODIFIER     = "modifier";     // ELEMENT

  public static final String ID           = "id";           // Lithology ID
  public static final String WORD         = "word";         // ELEMENT
  public static final String MODIFY       = "modify";       // Modifies
  public static final String PHRASE       = "phrase";       // Word Phrase

  /* ------------------------------------------------------------------------------------ *
   * -- Abbreviations Main
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT abbreviation (words*)>
   * <!ATTLIST abbreviation records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String ABBREV_MAIN = "abbrev_main"; // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Abbreviation Main Word list
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT words    EMPTY>
   * <!ATTLIST words    abbrev CDATA #REQUIRED
   *                    word   CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String WORDS        = "words";        // ELEMENT

  public static final String ABBREV       = "abbrev";       // Abbreviation used
  //    WORD  (DEFINED)                   = "word";         // Modifier

  /* ------------------------------------------------------------------------------------ *
   * -- Abbreviations Descriptor
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT abbreviation (words*)>
   * <!ATTLIST abbreviation records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String ABBREV_DESC = "abbrev_desc";   // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .... Abbreviation Descriptor Word list
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT words    EMPTY>
   * <!ATTLIST words    abbrev CDATA #REQUIRED
   *                    word   CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  //    WORDS (DEFINED)                   = "words";        // ELEMENT

  // public static final String ABBREV    = "abbrev";       // Abbreviation used
  //    WORD  (DEFINED)                   = "word";         // Modifier

  /* ------------------------------------------------------------------------------------ *
   * -- List of Valid Rocks
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT rocks    (rock*)>
   * <!ATTLIST rocks    records CDATA #REQUIRED
   *                    type    CDATA #REQUIRED
   *                    rows    CDATA #REQUIRED
   *                    columns CDATA #REQUIRED
   *                    words   CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String ROCKS        = "rocks";        // ELEMENT

  //    RECORDS (DEFINED)                 = "records";      // Attribute

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Information Data
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT column   EMPTY>
   * <!ATTLIST column   id       CDATA #REQUIRED
   *                    group    CDATA #IMPLIED
   *                    name     CDATA #REQUIRED
   *                    abrv     CDATA #IMPLIED
   *                    cycle    CDATA #IMPLIED
   *                    wthr     CDATA #IMPLIED>
   * ------------------------------------------------------------------------------------ */

  public static final String COLUMN       = "column";       // ELEMENT

  //    ID     (DEFINED)                  = "id";           // Rock Column ID
  //    GROUP  (DEFINED)                  = "group";        // Group Identifier
  public static final String NAME         = "name";         // Name of Rock
  public static final String ABREV        = "abrv";         // Abbreviation
  public static final String CYCLE        = "cycle";        // sed (sedimentary), etc.
  public static final String WEATHER      = "wthr";         // Weathering Factor

  /* ------------------------------------------------------------------------------------ *
   * .. -- .. Rock Column Thesaurus List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT thesaurus (word*)>
   * <!ATTLIST thesaurus records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String THESAURUS    = "thesaurus";    // ELEMENT

  /* ------------------------------------------------------------------------------------ *
   * -- Valid Keywords List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT keywords (word*)>
   * <!ATTLIST keywords records CDATA #REQUIRED>
   * ------------------------------------------------------------------------------------ */

  public static final String KEYWORDS     = "keywords";     // ELEMENT

  /* ------------------------------------------------------------------------------------ *
   * .... Word List
   * ------------------------------------------------------------------------------------ *
   * <!ELEMENT word     EMPTY>
   * <!ATTLIST word     phrase CDATA #IMPLIED
   *                    type   CDATA #IMPLIED>]>
   * ------------------------------------------------------------------------------------ */

  //    WORD   (DEFINED)                  = "word";         // ELEMENT

  //    PHRASE (DEFINED)                  = "phrase";       // Word Phrase
  public static final String TYPE         = "type";         // Type of Symbol text

  // **************************************************************************************

  // ELEMENT Identifiers

  private static final int _NONE        = 0;

  private static final int _MODIFIERS   = 1;  // modifiers
  private static final int _MODIFIER    = 2;  // modifier

  private static final int _ABBREV_MAIN = 3;  // abbreviation
  private static final int _WORDS_MAIN  = 4;  // Abbreviation Words

  private static final int _ABBREV_DESC = 5;  // abbreviation
  private static final int _WORDS_DESC  = 6;  // Abbreviation Words

  private static final int _ROCKS       = 7; // Read Rock Type
  private static final int _COLUMN      = 8; // Rock Type Column

  private static final int _KEYWORDS    = 9; // keywords
  private static final int _KEYWORD     = 10; // keywords Word

  // Global Variables

  private int                          iRows    = 0;
  private int                          iCount   = -1;
  private int                          iRow     = 0;
  private int                          iData    = _NONE;
  private int                          iCNTR    = _NONE;
  private parseGrainsTextureListStruct stList   = new parseGrainsTextureListStruct();

  private int                          iError   = 0;
  private String                       sError   = "";

  /** Method ReadGrainsTexturesXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadGrainsTexturesXMLFile() { iType = FILE; }

  /** Method ReadGrainsTexturesXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadGrainsTexturesXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public parseGrainsTextureListStruct Process(String filename)
  {
    Read(filename);

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadGrainsTexturesXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadGrainsTexturesXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name
    int    m     = 0;
    int    n     = 0;
    int    iCol  = 0;

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    if (eName.equals(ROOTNAME))     // ROOT NAME Element
    {
      iRow  = -1;
      iData = _NONE;
	}

    if (eName.equals(MODIFIERS))    // Modifier Element
    {
      iData  = _MODIFIERS;
      iRow   = -1;
	}

    if (eName.equals(MODIFIER))     // Modifier Element
    {
      iData  = _MODIFIER;
      iRow++;
	}

    if (eName.equals(ABBREV_MAIN)) // Abbreviation Terms Element
    {
      iData  = _ABBREV_MAIN;
      iCNTR  = _ABBREV_MAIN;
      iRow   = -1;
	}

    if (eName.equals(ABBREV_DESC)) // Abbreviation Terms Element
    {
      iData  = _ABBREV_DESC;
      iCNTR  = _ABBREV_DESC;
      iRow   = -1;
	}

    if (eName.equals(WORDS))       // Words Element
    {
	  if (iCNTR == _ABBREV_MAIN)
        iData  = _WORDS_MAIN;
      else
        iData  = _WORDS_DESC;
      iRow++;
	}

    if (eName.equals(ROCKS))        // Rocks Element
    {
      iData  = _ROCKS;
      iRow   = -1;
	}

    if (eName.equals(COLUMN))       // Column Element
    {
      iData  = _COLUMN;
      iRow++;
      stList.stItem[iRow] = new parseGrainsTextureStruct();
	}

    if (eName.equals(KEYWORDS))     // Valid Rock Column Keywords
    {
      iData  = _KEYWORDS;
      iCount = -1;
	}

    if (eName.equals(WORD))         // Keyword Element
    {
      iData  = _KEYWORD;
      iCount++;
	}

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
	      switch (iData)
	      {
            case _NONE:
              break;

            case _MODIFIERS:   // modifiers
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iModifiers = iRows;
              stList.sModifier  = new String[iRows][3];
              break;
            case _MODIFIER:    // modifier
              break;

            case _ABBREV_MAIN:      // abbreviation Main
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iAbbrev_main = iRows;
              stList.sAbbrev_main = new String[iRows][2];
              break;
            case _WORDS_MAIN:       // Abbreviation Main Words
              break;

            case _ABBREV_DESC:      // abbreviation Descriptor
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iAbbrev_desc = iRows;
              stList.sAbbrev_desc = new String[iRows][2];
              break;
            case _WORDS_DESC:       // Abbreviation Descriptor Words
              break;

            case _ROCKS:       // Read Rock Type
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iCount = iRows;
              stList.stItem = new parseGrainsTextureStruct[iRows];
              break;
            case _COLUMN:      // Rock Type Column
              break;

            case _KEYWORDS:    // keywords
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iKeywords = iRows;
              stList.sKeywords = new String[iRows][2];
              break;
            case _KEYWORD:     // keywords Word
              break;
		  }
        }

	    switch (iData)
	    {
          case _NONE:
            break;

          case _MODIFIERS:   // modifiers
            break;
          case _MODIFIER:    // modifier
            parseModifier( aName, sOut.trim() );
            break;

          case _ABBREV_MAIN:      // abbreviation
            break;
          case _WORDS_MAIN:       // Abbreviation Words
            parseAbbreviation( aName, sOut.trim() );
            break;

          case _ABBREV_DESC:      // abbreviation
            break;
          case _WORDS_DESC:       // Abbreviation Words
            parseAbbrev_desc( aName, sOut.trim() );
            break;

          case _ROCKS:       // Read Rock Type
            break;
          case _COLUMN:      // Rock Type Column
            stList.stItem[iRow] =
                ParseData( aName, sOut.trim(), stList.stItem[iRow] );
            break;

          case _KEYWORDS:    // keywords
            break;
          case _KEYWORD:     // keywords Word
            parseKeywords( aName, sOut.trim() );
            break;
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData( String sIdentifier, String sData, parseGrainsTextureStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = lithology Data Structure
   * @return st          = lithology Data Structure
   */

  private parseGrainsTextureStruct ParseData( String sIdentifier,
                                              String sData,
                                              parseGrainsTextureStruct st)
  {
    String sTemp  = "";
    int    iColor = -1;

    if (sIdentifier.equals(ID))    { st.sID       = new String( sData);  }
    if (sIdentifier.equals(ABREV)) { st.sAbrev    = new String(sData); }
    if (sIdentifier.equals(NAME))  { st.sName     = new String(sData); }
    if (sIdentifier.equals(CYCLE)) { st.sCycle    = new String(sData); }

    if (sIdentifier.equals(WEATHER))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iWeather = cmn.cmnString.stringToInt(sData);
	}

    return (st);
  }

  /** parseModifier( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseModifier( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ID))  {}   // Modifier ID
    if (sIdentifier.equals(WORD))   // Modifier
    {
      if (iRow > -1)
        stList.sModifier[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(MODIFY)) // Modifies
    {
      if (iRow > -1)
        stList.sModifier[iRow][1] = new String(sData);
    }

    if (sIdentifier.equals(PHRASE)) // Resultant Phrase
    {
      if (iRow > -1)
        stList.sModifier[iRow][2] = new String(sData);
    }
  }

  /** parseAbbreviation( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseAbbreviation( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ABBREV)) // Abbreviation
    {
      if (iRow > -1)
        stList.sAbbrev_main[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(WORD))   // Word or Phrase
    {
      if (iRow > -1)
        stList.sAbbrev_main[iRow][1] = new String(sData);
    }
  }

  /** parseAbbrev_desc( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseAbbrev_desc( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(ABBREV)) // Abbreviation
    {
      if (iRow > -1)
        stList.sAbbrev_desc[iRow][0] = new String(sData);
    }

    if (sIdentifier.equals(WORD))   // Word or Phrase
    {
      if (iRow > -1)
        stList.sAbbrev_desc[iRow][1] = new String(sData);
    }
  }

  /** parseKeywords( String sIdentifier, String sData )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   */

  private void parseKeywords( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(PHRASE)) // Abbreviation
    {
      if (iCount > -1)
        stList.sKeywords[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(TYPE))   // Word or Phrase
    {
      if (iCount > -1)
        stList.sKeywords[iCount][1] = new String(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}

/*
 *  @version 1.1 02/03/2016
 *  @author  John Victorine
 */
