/*
 * @mathLAS.java Version 1.1 06/03/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package math;

/** Class mathLAS
 *  <p> This Class provides methods that will compute one curve value from
 *      another.
 *
 *  @version 1.0 06/03/2010
 *  @author  John R. Victorine
 */

public class mathLAS
{
  // Log Curve Type

  public static final int _DENSITY  = 0;
  public static final int _SONIC    = 1;
  public static final int _U        = 2;

  // Grain Matrix Values

  public static final int _QUARTZ   = 0;
  public static final int _CALCITE  = 1;
  public static final int _DOLOMITE = 2;

  // Grain Matrix Value Array GRAIN X TYPE

  public static final double MATRIX[][] = {{ 2.65, 55.5,  4.8 }, // Quartz
                                           { 2.71, 47.5, 13.8 }, // Calcite
                                           { 2.87, 43.5,  9.0}}; // Dolomite

  // Fluid Values

  public static final int _FRESH    = 0;
  public static final int _SALT     = 1;
  public static final int _OIL      = 2;

  // Grain Fluid Value Array FLUID X TYPE
                                         // pb      dt      Uf
  public static final double FLUID[][] = {{ 1.0,   189.0, 0.398 }, // Fresh Water
                                          { 1.1,   185.0, 1.36 },  // Salty Water
                                          { 0.985, 204.5, 0.136 }};  // Oil Mud

  /** Method computeLogMinMax( double dCurve[], double dNull )
   * <p> This method will return an array holding the minimum and maximum
   *     curve value that is not a LAS File Null Value
   * @param  dCurve = array of Log Curve values
   * @param  dNull  = LAS File Null Value
   * @return data   = array holding the Minimum and Maximum Curve Values
   */

  public static double[] computeLogMinMax( double dCurve[], double dNull )
  {
    double data[] = { 0.0, 0.0 };
    int    iStart = 0;
    int    len    = dCurve.length;
    int    iTemp  = 0;
    double dLogL  = 0.1;
    double dLogU  = 1000.0;

    for (int i=0; i<len; i++)
    {
      if (dCurve[i] != dNull)
      {
	    if (dCurve[i] < dLogL)
	    {
		  if (dLogL > 0.01)
		    dLogL = 0.01;
	    }
	  }
    }

    for (int i=0; i<len; i++)
    {
      if (dCurve[i] != dNull)
      {
	    if (dCurve[i] > dLogU)
	    {
		  if (dLogU < 10000.0)
		    dLogU = dLogU * 10.0;
        }
	  }
    }

    data[0] = dLogL;
    data[1] = dLogU;

    return (data);
  }

  /** Method computeCurveMinMax( double dCurve[], double dNull )
   * <p> This method will return an array holding the minimum and maximum
   *     curve value that is not a LAS File Null Value
   * @param  dCurve = array of Log Curve values
   * @param  dNull  = LAS File Null Value
   * @return data   = array holding the Minimum and Maximum Curve Values
   */

  public static double[] computeCurveMinMax( double dCurve[], double dNull )
  {
    double data[] = { 0.0, 0.0 };
    int    iStart = 0;
    int    len    = dCurve.length;
    int    iTemp  = 0;

    for (int i=0; i<len; i++)
    {
      if ((iStart == 0) && (dCurve[i] != dNull))
      {
        iStart  = 1;
        data[0] = dCurve[i];
        data[1] = dCurve[i];
      }
      else if (dCurve[i] != dNull)
      {
        if ((data[0] > dCurve[i]) && (dCurve[i] != 0.0))
          data[0] = dCurve[i];

        if (data[1] < dCurve[i])
          data[1] = dCurve[i];
      }
    }

    iTemp   = (int) (1000.0 * data[0]);
    data[0] = (double) iTemp / 1000.0;

    iTemp   = (int) (1000.0 * data[1]);
    data[1] = (double) iTemp / 1000.0;

    return (data);
  }

  /* ================================================================== *
   * ------------------ SPECTRAL GAMMA RAY METHODS -------------------- *
   * ================================================================== */

  /** Method convertPotassium( int iRows, double dK[], double dNull )
   * <p> This method will convert the Postassium value from decimal to percent
   * @param  iRows = total number of values
   * @param  dK    = array of Potassium Curve values
   * @param  dNull = LAS File Null Value
   * @return dK    = array of Potassium Curve values corrected
   */

  public static double[] convertPotassium( int iRows, double dK[], double dNull )
  {
    int i     = 0;
    int idata = 0;

    // Check to see if the Potassium is percent or fractions

    for (i=0; i<iRows; i++)
    {
      if (dK[i] > 1.0)
        idata = 1;
    }

    // If the Potassium is in fractions then convert to
    // percent for computation

    if (idata == 0)
    {
      for (i=0; i<iRows; i++)
      {
        if (dK[i] != dNull)
          dK[i] = dK[i] * 100.0;
      }
    }

    return (dK);
  }

  /** Method computeGR( int iRows,
   *                    int iGR, double dGR[],
   *                    int iK,  double dK[],
   *                    int iTh, double dTh[],
   *                    int iU,  double dU[],
   *                    double dNull )
   * <p> Compute the Gamma Ray Curve Value from the Spectral Gamma Ray Values.
   * @param  iRows = total number of values
   * @param  iGR   = Gamma Ray Curve Present Indicator
   * @param  dGR   = Array of Computed Gamma Ray Values
   * @param  iK    = Potassium Curve Present Indicator
   * @param  dK    = Array of Potassium Curve Values
   * @param  iTh   = Thorium Curve Present Indicator
   * @param  dTh   = Array of Thorium Curve Values
   * @param  iU    = Uranium Curve Present Indicator
   * @param  dU    = Array of Uranimum Curve Values
   * @param  dNull = LAS File Null Value
   * @return dGR   = Array of Computed Gamma Ray Values
   */

  public static double[] computeGR( int iRows,
                                    int iGR, double dGR[],
                                    int iK,  double dK[],
                                    int iTh, double dTh[],
                                    int iU,  double dU[],
                                    double dNull )
  {
    if (iRows > 0)
    {
      /* IF Computed Gamma Ray does not exist and or
       *    Gamma Ray does not exist
       *   Uranium:    1.0 ppm =  8.0   (8.09) API
       *   Thorium:    1.0 ppm =  4.0   (3.93) API
       *   Potassium:  1.0 %   = 16.0   (16.32) API
       */

      if (iGR == -1)
      {
        if ((iTh > -1) && (iU > -1) && (iK > -1))
        {
          for (int i=0; i<iRows; i++)
          {
            dGR[i] = dNull;
            if ((dU[i] != dNull) && (dTh[i] != dNull) && (dK[i] != dNull))
            {
              dGR[i] = 8.0 * dU[i] + 4.0 * dTh[i] + 16.0 * dK[i];
            }
          }
        }
      }
    }

    return (dGR);
  }

  /** Method computeCGR( int iRows,
   *                     int iGR, double dGR[],
   *                     int iK,  double dK[],
   *                     int iTh, double dTh[],
   *                     int iU,  double dU[],
   *                     double dNull )
   * <p> Compute the CGR Curve Value from the Spectral Gamma Ray Values.
   * @param  iRows = total number of values
   * @param  iCGR  = Gamma Ray Curve Present Indicator
   * @param  dCGR  = Array of Computed Gamma Ray Values
   * @param  iK    = Potassium Curve Present Indicator
   * @param  dK    = Array of Potassium Curve Values
   * @param  iTh   = Thorium Curve Present Indicator
   * @param  dTh   = Array of Thorium Curve Values
   * @param  dNull = LAS File Null Value
   * @return dGR   = Array of Computed Gamma Ray Values
   */

  public static double[] computeCGR( int iRows,
                                     int iCGR, double dCGR[],
                                     int iK,  double dK[],
                                     int iTh, double dTh[],
                                     double dNull )
  {
    if (iRows > 0)
    {
      /* IF Computed Gamma Ray does not exist and or
       *    Gamma Ray does not exist
       *   Thorium:    1.0 ppm =  4.0   (3.93) API
       *   Potassium:  1.0 %   = 16.0   (16.32) API
       */

      if (iCGR == -1)
      {
        if ((iTh > -1) && (iK > -1))
        {
          for (int i=0; i<iRows; i++)
          {
            dCGR[i] = dNull;
            if ((dTh[i] != dNull) && (dK[i] != dNull))
            {
              dCGR[i] = 4.0 * dTh[i] + 16.0 * dK[i];
            }
          }
        }
      }
    }

    return (dCGR);
  }

  /* ================================================================== *
   * ---------------------- GAMMA RAY METHODS ------------------------- *
   * ================================================================== */

  /* ------------------------------------------------------------------ *
   * -------------- GAMMA RAY COUNTS TO GAMMA RAY API ----------------- *
   * ------------------------------------------------------------------ */

  /** Method computeGR()
   * <p> This method will compute the Gamma Ray from the the Gamma Ray Counts
   * @param  dGRN     = Gamma Ray Counts
   * @param  dNull    = Log File Null Value
   * @param  dGRmax   = Maximum Gamma Ray Value in API
   * @param  dGRmin   = Minimum Gamma Ray Value in API
   * @return dGR      = Gamma Ray
   */

  public static double[] computeGR( double dGRN[], double dNull,
                                    double dGRmax, double dGRmin )
  {
    int i=0;
    double dGR[]   = null;
    int    len     = 0;
    int    iStart  = 0;
    double dMax    = 0.0;
    double dMin    = 0.0;

    if (dGRN != null)
    {
      len = dGRN.length;
      dGR = new double[len];

      for (i=0; i<len; i++)
      {
        if ((iStart == 0) && (dGRN[i] != dNull))
        {
          iStart = 1;
          dMax   = dGRN[i];
          dMin   = dGRN[i];
        }
        else
        {
          if (dMax < dGRN[i])
            dMax = dGRN[i];

          if (dMin > dGRN[i])
            dMin = dGRN[i];

          if (dMin < 0.0) dMin = 0.0;
        }
      }

      for (i=0; i<len; i++)
      {
        dGR[i] = dGRmax - (dGRmax - dGRmin) * (dMax - dGRN[i] ) / (dMax - dMin);
      }
    }

    return (dGR);
  }

  /** Method computeGR()
   * <p> This method will compute the Gamma Ray from the the Gamma Ray Counts
   * @param  dGRN     = Gamma Ray Counts
   * @param  dNull    = Log File Null Value
   * @param  dMax     = Maximum Gamma Ray Count Value
   * @param  dMin     = Minimum Gamma Ray Count Value
   * @param  dGRmax   = Maximum Gamma Ray Value in API
   * @param  dGRmin   = Minimum Gamma Ray Value in API
   * @return dGR      = Gamma Ray
   */

  public static double[] computeGR( double dGRN[], double dNull,
                                    double dMax,    double dMin,
                                    double dGRmax, double dGRmin )
  {
    int i=0;
    double dGR[]   = null;
    int    len     = 0;

    if (dGRN != null)
    {
      len = dGRN.length;
      dGR = new double[len];

      for (i=0; i<len; i++)
      {
        dGR[i] = dGRmax - (dGRmax - dGRmin) * (dMax - dGRN[i] ) / (dMax - dMin);
      }
    }

    return (dGR);
  }

  /* ================================================================== *
   * ------------------- NEUTRON POROSITY METHODS --------------------- *
   * ================================================================== */

  /* ------------------------------------------------------------------ *
   * -------------- NEUTRON COUNTS TO NEUTRON POROSITY ---------------- *
   * ------------------------------------------------------------------ */

  /** Method computeNPHI()
   * <p> This method will compute the Neutron Porosity from the the Neutron
   *     Counts
   * @param  dNEUT    = Neutron Counts
   * @param  dNull    = Log File Null Value
   * @param  PHIHigh  = High Porosity Point Value
   * @param  PHILow   = Low Porosity Point Value
   * @return dNPHI    = Neutron Porosity
   */

  public static double[] computeNPHI( double dNEUT[], double dNull,
                                      double PHIHigh, double PHILow )
  {
    int i=0;
    double dNPHI[] = null;
    int    len     = 0;
    double dSlope  = 0.0;
    double dIntcpt = 0.0;
    int    iStart  = 0;
    double dMax    = 0.0;
    double dMin    = 0.0;

    if (dNEUT != null)
    {
      len   = dNEUT.length;
      dNPHI = new double[len];

      for (i=0; i<len; i++)
      {
        if ((iStart == 0) && (dNEUT[i] != dNull))
        {
          iStart = 1;
          dMax   = dNEUT[i];
          dMin   = dNEUT[i];
        }
        else
        {
          if (dMax < dNEUT[i])
            dMax = dNEUT[i];

          if (dMin > dNEUT[i])
            dMin = dNEUT[i];

          if (dMin < 0.0) dMin = 0.0;
        }
      }

      for (i=0; i<len; i++)
      {
        dSlope   = ( Math.log(PHIHigh/PHILow)/Math.log(10.0) ) / (dMin-dMax);
        dIntcpt  = PHIHigh * Math.pow(10.0, dSlope * dMin );
        dNPHI[i] = dIntcpt * Math.pow(10.0, dSlope * dNEUT[i]);
      }
    }

    return (dNPHI);
  }

  /** Method computeNPHI()
   * <p> This method will compute the Neutron Porosity from the the Neutron
   *     Counts
   * @param  dNEUT    = Neutron Counts
   * @param  dNull    = Log File Null Value
   * @param  dMax     = Maximum Neutron Count Value
   * @param  dMin     = Minimum Neutron Count Value
   * @param  PHIHigh  = High Porosity Point Value
   * @param  PHILow   = Low Porosity Point Value
   * @return dNPHI    = Neutron Porosity
   */

  public static double[] computeNPHI( double dNEUT[], double dNull,
                                      double dMax,    double dMin,
                                      double PHIHigh, double PHILow )
  {
    int i=0;
    double dNPHI[] = null;
    int    len     = 0;
    double dSlope  = 0.0;
    double dIntcpt = 0.0;

    if (dNEUT != null)
    {
      len   = dNEUT.length;
      dNPHI = new double[len];

      for (i=0; i<len; i++)
      {
        dSlope   = ( Math.log(PHIHigh/PHILow)/Math.log(10.0) ) / (dMin-dMax);
        dIntcpt  = PHIHigh * Math.pow(10.0, dSlope * dMin );
        dNPHI[i] = dIntcpt * Math.pow(10.0, dSlope * dNEUT[i]);
      }
    }

    return (dNPHI);
  }

  /* ================================================================== *
   * ----------------------- POROSITY METHODS ------------------------- *
   * ================================================================== */

  /** Method convertPorosity( int iRows, double data[], double dNull )
   * <p> This method will convert the Porosity value from percent to decimal
   * @param  iRows = total number of values
   * @param  dPHI  = array of Porosity Curve values
   * @param  dNull = LAS File Null Value
   * @return dPHI  = array of Porosity Curve values corrected
   */

  public static double[] convertPorosity( int iRows, double dPHI[], double dNull )
  {
    int i     = 0;
    int idata = 0;

    // Check to see if the Porosity in Percent or Fractions.

    for (i=0; i<iRows; i++)
    {
      if (dPHI[i] > 2.0)
        idata = 1;
    }

    // If the Porosity is in Percent then convert to
    // fractions for computation

    if (idata > 0)
    {
      for (i=0; i<iRows; i++)
      {
        if (dPHI[i] != dNull)
          dPHI[i] = dPHI[i] / 100.0;
      }
    }

    return (dPHI);
  }

  /** Method computePHI( int    iRows,
   *                     double dLog[],
   *                     int    iType,
   *                     int    iMatrix,
   *                     int    iFluid,
   *                     double dNull )
   * <p> This method will compute the Porosity from Log Curve
   * @param  iRows   = total number of values
   * @param  dLog    = Log Curve Value
   * @param  iType   = Log Type (0=Density; 1=Sonic)
   * @param  iMatrix = Matrix Identifier
   * @param  iFluid  = Fluid Identifier
   * @param  dNull   = LAS File Null Value
   * @return dPHI    = Porosity
   */

  public static double[] computePHI( int    iRows,
                                     double dLog[],
                                     int    iType,
                                     int    iMatrix,
                                     int    iFluid,
                                     double dNull )
  {
    double dPHI[]  = null;
    double dMatrix = MATRIX[iMatrix][iType];
    double dFluid  = FLUID[iFluid][iType];

    if (iRows > 0)
    {
      dPHI = new double[iRows];

      for (int i=0; i<iRows; i++)
      {
        dPHI[i] = dNull;
        if (dLog[i] > dNull)
          dPHI[i] = ( dMatrix - dLog[i] ) / ( dMatrix - dFluid );
      }
    }

    return (dPHI);
  }

  /** Method computeLOG( int    iRows,
   *                     double dLog[],
   *                     int    iType,
   *                     int    iMatrix,
   *                     int    iFluid,
   *                     double dNull )
   * <p> This method will compute the Log Curve from Porosity
   * @param  iRows   = total number of values
   * @param  dPHI    = Porosity
   * @param  iType   = Log Type (0=Density; 1=Sonic)
   * @param  iMatrix = Matrix Identifier
   * @param  iFluid  = Fluid Identifier
   * @param  dNull = LAS File Null Value
   * @return dLog    = Log Curve Value
   */

  public static double[] computeLOG( int    iRows,
                                     double dPHI[],
                                     int    iType,
                                     int    iMatrix,
                                     int    iFluid,
                                     double dNull )
  {
    double dLog[]  = null;
    double dMatrix = MATRIX[iMatrix][iType];
    double dFluid  = FLUID[iFluid][iType];

    if (iRows > 0)
    {
      dLog = new double[iRows];

      for (int i=0; i<iRows; i++)
      {
        dLog[i] = dNull;
        if (dPHI[i] > dNull)
          dLog[i] = dMatrix - dPHI[i] * ( dMatrix - dFluid );
      }
    }

    return (dLog);
  }

  /** Method computePHI( int    iRows,
   *                     double dLog[],
   *                     int    iType,
   *                     int    iMatrix,
   *                     int    iFluid,
   *                     double dNull )
   * <p> This method will compute the Porosity from Log Curve
   * @param  dLog    = Log Curve Value
   * @param  iType   = Log Type (0=Density; 1=Sonic)
   * @param  iMatrix = Matrix Identifier
   * @param  iFluid  = Fluid Identifier
   * @return dPHI    = Porosity
   */

  public static double computePHI( double dLog,
                                   int    iType,
                                   int    iMatrix,
                                   int    iFluid )
  {
    double dPHI    = 0.0;
    double dMatrix = MATRIX[iMatrix][iType];
    double dFluid  = FLUID[iFluid][iType];

    dPHI = ( dMatrix - dLog ) / ( dMatrix - dFluid );

    return (dPHI);
  }

  /** Method computeAvgPHI()
   * <p> This method will compute the average porosity from the Neutron Porosity
   *     and the Density Porosity
   *     dPHI = (dNPHI + dDPHI) / 2.0
   *
   * @param  dNPHI = Neutron Porosity (in decimal value)
   * @param  dDPHI = Density Porosity (in decimal value)
   * @return dPHI  = Average Porosity
   */

  public static double computeAvgPHI(double dNPHI, double dDPHI)
  {
    double dPHI = 0.0;

    dPHI = (dNPHI + dDPHI) / 2.0;
//    dPHI = Math.sqrt((dNPHI*dNPHI + dDPHI*dDPHI)/2);

    return (dPHI);
  }

  /* ================================================================== *
   * ---------------------- Rhomaa-Umaa METHODS ----------------------- *
   * ================================================================== */

  /** Method computeRHOMAA( int fluid, double dRHOB, double dPHI )
   * <p> This method will compute the Rhomaa Value from log curves
   *     NOTE: Need Porosity and Bulk Density Log Curves
   *     RHOmaa = (RHOB - RHOf * PHI) / ( 1 - PHI )
   * @param  fluid   = Identifier of type of fluid to use
   * @param  dRHOB   = Bulk density log value
   * @param  dPHI    = Porosity
   * @return dRhomaa = Rhomaa Value
   */

  public static double computeRhomaa( int fluid, double dRHOB, double dPHI )
  {
    double dRhomaa = 0.0;
    double dFluid  = FLUID[fluid][_DENSITY];

    dRhomaa = (dRHOB - dFluid * dPHI) / ( 1.0 - dPHI );

    return (dRhomaa);
  }

  /** Method computeUMAA()
   * <p> This method will compute the Umaa Value from log curves
   *     NOTE: Need Porosity, Bulk Density and Photoelectric factor log curves
   *     Umaa = ((Pe * RHOB) - Uf * PHI) / ( 1 - PHI )
   * @param  fluid   = Identifier of type of fluid to use
   * @param  dRHOB   = Bulk density log value
   * @param  dPE     = Photoelectric Factor log value
   * @param  dPHI    = Porosity
   * @return dUmaa = Rhomaa Value
   */

  public static double computeUmaa( int fluid, double dRHOB,
                                    double dPE, double dPHI )
  {
    double dUmaa = 0.0;
    double dFluid  = FLUID[fluid][_U];

    dUmaa = ( (dPE * dRHOB) - (dFluid * dPHI) ) / ( 1.0 - dPHI );

    return (dUmaa);
  }

  /** Method computeUMAA()
   * <p> This method will compute the Umaa Value from log curves
   *     NOTE: Need Porosity, and Photoelectric factor log curves
   *     Umaa = dU / ( 1 - PHI )
   * @param  dU      = Photoelectric Factor
   * @param  dPHI    = Porosity
   * @return dUmaa = Rhomaa Value
   */

  public static double computeUmaa( double dU, double dPHI )
  {
    double dUmaa = 0.0;

    dUmaa = dU  / ( 1.0 - dPHI );

    return (dUmaa);
  }

  /** Method computeDTMAA( int fluid, double dRHOB, double dPHI )
   * <p> This method will compute the Rhomaa Value from log curves
   *     NOTE: Need Porosity and Bulk Density Log Curves
   *     RHOmaa = (RHOB - RHOf * PHI) / ( 1 - PHI )
   * @param  fluid   = Identifier of type of fluid to use
   * @param  dDT     = Acoustic transit time log value
   * @param  dPHI    = Porosity
   * @return dDT     = DTmaa Value
   */

  public static double computeDTMAA( int fluid, double dDT, double dPHI )
  {
    double dDTmaa = 0.0;
    double dFluid  = FLUID[fluid][_SONIC];

    dDTmaa = (dDT - dFluid * dPHI) / ( 1.0 - dPHI );

    return (dDTmaa);
  }

  /** Method computeRhomaaUmaa()
   * <p> This method will convert the raw log curves to Rhomaa-Umaa Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dRhob = Array of Bulk Density Values
   * @param  dPE   = Array of Photoelectric factor Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Rhomaa-Umaa Values
   */

  public static double[][] computeRhomaaUmaa( int    fluid,
                                              int    iRows,
                                              double dNPHI[],
                                              double dRhob[],
                                              double dPE[],
                                              double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dU    = 0.0;
    double dDPHI = 0.0;
    double dPHI  = 0.0;
    double dOut[][] = null;

    if (iRows > 0)
    {
      dOut = new double[2][iRows];
      for (j=0; j<2; j++)
      {
        for (i=0; i<iRows; i++)
        {
          dOut[j][i] = dNull;
        }
      }

      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dRhob[i] > dNull) && (dPE[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dDPHI      = computePHI( dRhob[i], _DENSITY, _CALCITE, fluid );
          dPHI       = computeAvgPHI( dNPHI[i], dDPHI);
//          dDPHI = (RHO_MATRIX - dRhob[i]) / (RHO_MATRIX - RHO[iH2O]);
//          dPHI  = (dNPHI[i] + dDPHI) / 2.0;

          dOut[0][i] = computeRhomaa( fluid, dRhob[i], dPHI );
          dOut[1][i] = computeUmaa( fluid, dRhob[i], dPE[i], dPHI );
//          dOut[0][i] = (dRhob[i] - dPHI * RHO[iH2O])/(1.0-dPHI);
//          dOut[1][i] = ((dPE[i] * dRhob[i]) - (dPHI * U[iH2O]))/(1.0-dPHI);
        }
      }
    }

    return (dOut);
  }

  /** Method computeRhomaa()
   * <p> This method will convert the raw log curves to Rhomaa-Umaa Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dRhob = Array of Bulk Density Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Rhomaa-Umaa Values
   */

  public static double[] computeRhomaa( int    fluid,
                                        int    iRows,
                                        double dNPHI[],
                                        double dRhob[],
                                        double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dU    = 0.0;
    double dDPHI = 0.0;
    double dPHI  = 0.0;
    double dOut[] = null;

    if (iRows > 0)
    {
      dOut = new double[iRows];
      for (i=0; i<iRows; i++)
      {
        dOut[i] = dNull;
      }

      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dRhob[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dDPHI   = computePHI( dRhob[i], _DENSITY, _CALCITE, fluid );
          dPHI    = computeAvgPHI( dNPHI[i], dDPHI);

          dOut[i] = computeRhomaa( fluid, dRhob[i], dPHI );
        }
      }
    }

    return (dOut);
  }

  /** Method computeDTmaa()
   * <p> This method will convert the raw log curves to DTmaa Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dRhob = Array of Bulk Density Values
   * @param  dDT   = Array of Acoustic transit time Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Rhomaa-Umaa Values
   */

  public static double[] computeDTmaa( int    fluid,
                                       int    iRows,
                                       double dNPHI[],
                                       double dRhob[],
                                       double dDT[],
                                       double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dU    = 0.0;
    double dDPHI = 0.0;
    double dPHI  = 0.0;
    double dOut[] = null;

    if (iRows > 0)
    {
      dOut = new double[iRows];
      for (i=0; i<iRows; i++)
      {
        dOut[i] = dNull;
      }

      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dRhob[i] > dNull) && (dDT[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dDPHI   = computePHI( dRhob[i], _DENSITY, _CALCITE, fluid );
          dPHI    = computeAvgPHI( dNPHI[i], dDPHI);

          dOut[i] = computeDTMAA( fluid, dDT[i], dPHI );
        }
      }
    }

    return (dOut);
  }

  /** Method computeDTmaa()
   * <p> This method will convert the raw log curves to DTmaa Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dDT   = Array of Acoustic transit time Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Rhomaa-Umaa Values
   */

  public static double[] computeDTmaa( int    fluid,
                                       int    iRows,
                                       double dNPHI[],
                                       double dDT[],
                                       double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dU    = 0.0;
    double dPHI  = 0.0;
    double dOut[] = null;

    if (iRows > 0)
    {
      dOut = new double[iRows];
      for (i=0; i<iRows; i++)
      {
        dOut[i] = dNull;
      }

      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dDT[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dPHI    = dNPHI[i];

          dOut[i] = computeDTMAA( fluid, dDT[i], dPHI );
        }
      }
    }

    return (dOut);
  }

  /** Method computePHIt()
   * <p> This method will convert the raw log curves to PHIt Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dRhob = Array of Bulk Density Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Rhomaa-Umaa Values
   */

  public static double[] computePHIt( int    fluid,
                                      int    iRows,
                                      double dNPHI[],
                                      double dRhob[],
                                      double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dU    = 0.0;
    double dDPHI = 0.0;
    double dPHI  = 0.0;
    double dOut[] = null;

    if (iRows > 0)
    {
      dOut = new double[iRows];
      for (i=0; i<iRows; i++)
      {
        dOut[i] = dNull;
      }

      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dRhob[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dDPHI   = computePHI( dRhob[i], _DENSITY, _CALCITE, fluid );
          dOut[i] = computeAvgPHI( dNPHI[i], dDPHI);
        }
      }
    }

    return (dOut);
  }

  /** Method computePorosityDifference()
   * <p> This method will convert the raw log curves to colorlith Curves for
   *     plotting.
   * @param  fluid = Identifier for Salt or Fresh Water
   * @param  iRows = The total number of records
   * @param  dNPHI = Array of Neutron Porosity Values
   * @param  dRhob = Array of Bulk Density Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Porosity Difference Values
   */

  public static double[][] computePorosityDifference( int    fluid,
                                                      int    iRows,
                                                      double dNPHI[],
                                                      double dRhob[],
                                                      double dNull )
  {
    int    i = 0;
    int    j = 0;
    double dDPHI    = 0.0;
    double dOut[][] = null;

    dOut = new double[2][iRows];
    for (j=0; j<2; j++)
    {
      for (i=0; i<iRows; i++)
      {
        dOut[j][i] = dNull;
      }
    }

    if (iRows > 0)
    {
      for (i=0; i<iRows; i++)
      {
        if ((dNPHI[i] > dNull) && (dRhob[i] > dNull))
        {
          if (dNPHI[i] == 1.0)
            dNPHI[i] = .99;

          dDPHI  = computePHI( dRhob[i], _DENSITY, _CALCITE, fluid );

          dOut[0][i] = dNPHI[i];         // Neutron Porosity
          dOut[1][i] = dNPHI[i] - dDPHI; // Porosity Difference
        }
      }
    }

    return (dOut);
  }

  /** Method computeSpectralGR()
   * <p> This method will convert the raw log curves to colorlith Curves for
   *     plotting.
   * @param  iRows = The total number of records
   * @param  dTh   = Array of Thorium Values
   * @param  dU    = Array of Uranium Values
   * @param  dK    = Array of Potassium Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Spectral Gamma Ray Values
   */

  public static double[][] computeSpectralGR( int    iRows,
                                              double dTh[],
                                              double dU[],
                                              double dK[],
                                              double dNull)
  {
    int    i = 0;
    int    j = 0;
    double dOut[][] = null;

    dOut = new double[2][iRows];
    for (j=0; j<2; j++)
    {
      for (i=0; i<iRows; i++)
      {
        dOut[j][i] = dNull;
      }
    }

    for (i=0; i<iRows; i++)
    {
      if ((dTh[i] > dNull) && (dU[i] > dNull) && (dK[i] > dNull))
      {
        if (dU[i] == 0.0) dU[i] = .0001;
        if (dK[i] == 0.0) dK[i] = .0001;

        dOut[0][i] = dTh[i] / dU[i]; // Thorium-Uranium Ratio
        dOut[1][i] = dTh[i] / dK[i]; // Thorium-Potassium Ratio
      }
    }

    return (dOut);
  }

  /* ================================================================== *
   * ---------------------- TEMPERATURE METHODS ----------------------- *
   * ================================================================== */

  /** Method computeTempDiff()
   * <p> This method will compute the temperature difference
   * @param  iRows = The total number of records
   * @param  dTemp = Array of Thorium Values
   * @param  dNull = Empty data value
   * @return dOut  = Array of Spectral Gamma Ray Values
   */

  public static double[] computeTempDiff( int iRows, double dTemp[], double dNull )
  {
    int    i = 0;
    double dtmp1  = 0;
    double dtmp2  = 0;
    double dOut[] = null;

    dOut = new double[iRows];
    for (i=0; i<iRows; i++)
    {
      dOut[i] = dNull;
    }

    for (i=0; i<iRows-1; i++)
    {
	  dtmp1 = dTemp[i];
	  dtmp2 = dTemp[i+1];

      if ((dtmp1 > dNull) && (dtmp2 > dNull))
      {
        dOut[i] = dtmp2 - dtmp1; // Temperature Difference
      }
    }

    return (dOut);
  }
}

/*
 *  @version 1.0 06/03/2010
 *  @author  John Victorine
 */
