/*
 * @maskPlotSymbol.java Version 1.1 07/20/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package mask.plot;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;
import javax.swing.*;

import mask.maskSymbolsListStruct;

/** Class maskPlotSymbol
 *  <p> This method will draw symbols from a text mask
 *
 *  @version 1.1 07/20/2009
 *  @author  John R. Victorine
 */

public class maskPlotSymbol extends Canvas
{
  private maskSymbolsListStruct stSymbols = null;

  /** CONSTRUCTOR maskPlotSymbol()
   *  <p> This is the constructor for this class.
   *  @param st = Mask Symbols List Data Structure
   */

  public maskPlotSymbol( maskSymbolsListStruct st ) { this.stSymbols = st; }

  /** Method close()
   * <p> This method will force java to deallocate memory
   */

  public void close() { stSymbols = null; }

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getOrder()
   * <p> This method will return the 0 or 1 depending if the number passed in is
   *     even or odd
   * @param  iNo    = the number passed in
   * @return iOrder = 0 or 1 depending if the iNo is even or odd respectively
   */

  public int getOrder( int iNo )
  {
    int iOrder = 0;

    if (iNo > 1)
    {
      while (iNo > 1)
        iNo = iNo - 2;
    }
    iOrder = iNo;

    return (iOrder);
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawCell()
   * <p> This method will draw the symbol cell.
   *  @param g        = Graphics Handle;
   *  @param iMask    = the mask identifier
   *  @param iType    = Symbols or Modifiers
   *  @param iOrder   = the order number (right or left text array to plot)
   *  @param iRows    = total number of rows
   *  @param iColumns = total number of columns
   *  @param iXo      = initial X starting point on plot
   *  @param iYo      = initial Y starting point on plot
   */

  public void drawCell( Graphics g, int iMask, int iType, int iOrder,
                        int iRows, int iColumns, int iXo, int iYo)
  {
    drawBackground( g, iMask, iXo, iYo, iColumns, iRows);
    drawSymbol( g, iRows, getOrder(iOrder),
                mask.maskSymbolsUtility.getSymbol( iMask, iType, stSymbols ),
                iXo, iYo );
  }

  /** Method drawBackground()
   * <p> This method will draw and fill a rectangular block
   *  @param g     = Graphics Handle;
   *  @param iMask = The Mask Identifier
   *  @param iXp   = X Starting Position
   *  @param iYp   = Y Starting Position
   *  @param iWide = Width  of block
   *  @param iHigh = Height of block
   */

  public void drawBackground( Graphics g,
                              int iMask,
                              int iXp,   int iYp,
                              int iWide, int iHigh )
  {
    int iColor[] = { 255, 255, 255 };

    if (stSymbols != null)
      iColor = stSymbols.stItem[iMask].iRGB;

    g.setColor( new Color(iColor[0], iColor[1], iColor[2]) );
    g.fillRect( iXp, iYp, iWide, iHigh );
  }

  /** Method drawBackground()
   * <p> This method will draw and fill a rectangular block
   *  @param g          = Graphics Parameter
   *  @param iRed       = Red Value for RGB Color
   *  @param iGreen     = Green Value for RGB Color
   *  @param iBlue      = Blue Value for RGB Color
   *  @param iXp        = X Starting Position
   *  @param iYp        = Y Starting Position
   *  @param iWide      = Width  of block
   *  @param iHigh      = Height of block
   */

  public static void drawBackground( Graphics g,
                                     int iRed, int iGreen, int iBlue,
                                     int iXp,   int iYp,
                                     int iWide, int iHigh)
  {
    g.setColor( new Color(iRed, iGreen, iBlue) );
    g.fillRect( iXp, iYp, iWide, iHigh );
  }

  /** Method drawSymbol()
   * <p> This method will convert the String Array to an image and paint it
   *  @param g      = Graphics Handle;
   *  @param iRows  = total number of rows in the text array
   *  @param iOrder = the order number (right or left text array to plot)
   *  @param symbol = the text array mask
   *  @param iXo    = initial X starting point on plot
   *  @param iYo    = initial Y starting point on plot
   */

  public void drawSymbol( Graphics g, int iRows, int iOrder,
                          String symbol[][], int iXo, int iYo )
  {
    int  i      = 0;
    int  j      = 0;
    int  iRGB[] = null;
    int  iColor = 0;
    char c[]    = null;

    for (i=0; i<iRows; i++)
    {
      if (symbol != null)
        c = symbol[i][iOrder].toCharArray();

      for (j=0; j<c.length; j++)
      {
        if ((stSymbols != null) && (c != null))
          iColor = mask.maskSymbolsUtility.getColor(
                     c[j], stSymbols.stForeground );

        if (iColor > -1)
        {
          if (stSymbols != null)
            iRGB = mask.maskSymbolsUtility.getRGB(
                     iColor, stSymbols.stForeground );
          g.setColor( new Color(iRGB[0], iRGB[1], iRGB[2]) );
          g.fillRect(iXo+j, iYo+i, 1, 1);
        }
      }
    }
  }
}