/*
 * @maskSymbolsUtility.java Version 1.1 07/20/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package mask;

import mask.maskColorsStruct;
import mask.maskGroupsStruct;
import mask.maskSymbolsListStruct;
import mask.maskSymbolsStruct;

/** Class maskSymbolsUtility
 *  <p> This Class will provide basic utilities for the Standard Mask
 *      data structures.
 *
 *  @version 1.1 07/20/2009
 *  @author  John R. Victorine
 */

public class maskSymbolsUtility
{
  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getColor( char c, maskColorsStruct st )
   *  <p> This method will return the color number identifier from the
   *      character that is passed in.
   * @param  c    = Character
   * @param  st   = mask colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( char c, maskColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.colors != null)
      {
        for (int i=1; i<st.colors.length; i++)
        {
          if (c == st.colors[i]) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getColor( String str, maskColorsStruct st )
   *  <p> This method will return the color number identifier from the
   *      color name that is passed in.
   * @param  str  = Color Name
   * @param  st   = mask colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( String str, maskColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.sColors != null)
      {
        for (int i=1; i<st.sColors.length; i++)
        {
          if (str.equals(st.sColors[i])) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getRGB( int icolor, maskColorsStruct st )
   * <p> This method will return the RGB Array of colors
   * @param  icolor = The mask color to return
   * @param  st     = mask colors data structure
   * @return RGB    = Array of Red Green Blue Color numbers
   */

  public static int[] getRGB( int icolor, maskColorsStruct st )
  {
    int RGB[] = { 255, 255, 255 };

    if (icolor != -1)
    {
      if (st != null)
      {
        if (st.iRGB != null)
        {
          if (icolor < st.iRGB.length)
          {
            for (int j=0; j<3; j++)
              RGB[j] = st.iRGB[icolor][j];
          }
        }
      }
    }

    return ( RGB );
  }

  /** Method getGroupID( int iColumn, String str, maskGroupsStruct st )
   *  <p> This method will return the ID number of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  str     = String to comparison string
   * @param  st      = mask group data structure
   * @return iGroup  = Group id
   */

  public static int getGroupID( int iColumn, String str, maskGroupsStruct st )
  {
    int iGroup = -1;

    if (st != null)
    {
      for (int i=0; i<st.iGroups; i++)
      {
        if (st.sGroups[i][iColumn].equals(str))
          iGroup = i;
      }
    }

    return (iGroup);
  }

  /** Method getGroup( int iColumn, int iGroup, maskGroupsStruct st )
   *  <p> This method will return the string of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  iGroup  = Group ID
   * @param  st      = mask group data structure
   * @return sGroup  = Group String
   */

  public static String getGroup( int iColumn, int iGroup, maskGroupsStruct st )
  {
    String sGroup = "";

    if (st != null)
    {
      if (iGroup > -1)
      {
        sGroup = new String(st.sGroups[iGroup][iColumn]);
      }
    }

    return (sGroup);
  }

  /** Method getSymbol( int iSymbol, maskSymbolsListStruct st )
   * <p> This method will return the symbol
   * @param  iSymbol = Symbol identifier to retrieve symbol mask
   * @param  iType   = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st      = Mask Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static String[][] getSymbol( int iSymbol, int iType,
                                      maskSymbolsListStruct st )
  {
    String symbol[][] = null;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          if (st.stItem[iSymbol].symbol != null)
          {
            symbol = new String[st.stItem[iSymbol].symbol.length][2];
            for (int i=0; i<st.stItem[iSymbol].symbol.length; i++)
            {
              for (int j=0; j<2; j++)
                symbol[i][j] = st.stItem[iSymbol].symbol[i][j];
            }
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          if (st.stModifiers[iSymbol].symbol != null)
          {
            symbol = new String[st.stModifiers[iSymbol].symbol.length][2];
            for (int i=0; i<st.stModifiers[iSymbol].symbol.length; i++)
            {
              for (int j=0; j<2; j++)
                symbol[i][j] = st.stModifiers[iSymbol].symbol[i][j];
            }
          }
          break;
      }
    }

    return (symbol);
  }

  /** Method getSymbol( int iSymbol, maskSymbolsListStruct st )
   * <p> This method will return the symbol
   * @param  sID     = Symbol identifier to retrieve symbol mask
   * @param  iType   = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st      = Mask Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static String[][] getSymbol( String sID, int iType,
                                      maskSymbolsListStruct st )
  {
    int    iSymbol    = -1;
    String symbol[][] = null;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (int k=0; k<st.iCount; k++)
          {
            if (sID.equals(st.stItem[k].sID))
              iSymbol = k;
          }

          if (iSymbol != -1)
          {
            if (st.stItem[iSymbol].symbol != null)
            {
              symbol = new String[st.stItem[iSymbol].symbol.length][2];
              for (int i=0; i<st.stItem[iSymbol].symbol.length; i++)
              {
                for (int j=0; j<2; j++)
                  symbol[i][j] = st.stItem[iSymbol].symbol[i][j];
              }
            }
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          for (int k=0; k<st.iModifiers; k++)
          {
            if (sID.equals(st.stModifiers[k].sID))
              iSymbol = k;
          }

          if (iSymbol != -1)
          {
            if (st.stModifiers[iSymbol].symbol != null)
            {
              symbol = new String[st.stModifiers[iSymbol].symbol.length][2];
              for (int i=0; i<st.stModifiers[iSymbol].symbol.length; i++)
              {
                for (int j=0; j<2; j++)
                  symbol[i][j] = st.stModifiers[iSymbol].symbol[i][j];
              }
            }
          }
          break;
      }
    }

    return (symbol);
  }

  /** Method getSymbolID( int iSymbol, maskSymbolsListStruct st )
   * <p> This method will return the symbol
   * @param  sID     = Symbol identifier to retrieve symbol mask
   * @param  iType   = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st      = Mask Symbols List Data Structure
   * @return iSymbol = Symbol ID
   */

  public static int getSymbolID( String sID, int iType,
                                 maskSymbolsListStruct st )
  {
    int    iSymbol    = -1;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (int k=0; k<st.iCount; k++)
          {
            if (sID.equals(st.stItem[k].sID))
              iSymbol = k;
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          for (int k=0; k<st.iModifiers; k++)
          {
            if (sID.equals(st.stModifiers[k].sID))
              iSymbol = k;
          }
          break;
      }
    }

    return (iSymbol);
  }

  /** Method getSymbolIDByName( String sName, int iType,
   *                            maskSymbolsListStruct st )
   * <p> This method will return the symbol
   * @param  sName   = Name to retrieve symbol mask
   * @param  iType   = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st      = Mask Symbols List Data Structure
   * @return iSymbol = Symbol ID
   */

  public static int getSymbolIDByName( String sName, int iType,
                                       maskSymbolsListStruct st )
  {
    int    iSymbol    = -1;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (int k=0; k<st.iCount; k++)
          {
            if (sName.toLowerCase().equals(st.stItem[k].sName.toLowerCase()))
              iSymbol = k;
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          for (int k=0; k<st.iModifiers; k++)
          {
            if (sName.toLowerCase().equals(st.stModifiers[k].sName.toLowerCase()))
              iSymbol = k;
          }
          break;
      }
    }

    return (iSymbol);
  }

  /** Method getMask( String str, int iType, maskSymbolsListStruct st )
   * <p> This method will check the Mask key words, phrases to see if any
   *     matches the default mask key words or phrases and return the row
   *     id if it does.
   *
   * @param  str   = the string passed in.
   * @param  iType = Retrieve from SYMBOL or MODIIFER
   * @param  st    = Mask Symbols List Data Structure
   * @return iMask = the mask identifier.
   */

  public static int getMask( String str, int iType, maskSymbolsListStruct st )
  {
    int     i     = 0;
    int     j     = 0;
    int     iMask = -1;
    String  sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (i=0; i<st.iCount; i++)
          {
            for (j=0; j<st.stItem[i].iThesaurus; j++)
            {
              if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
                iMask = st.stItem[i].id;
            }
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          if (st.iModifiers > 0)
          {
            for (i=0; i<st.iModifiers; i++)
            {
              for (j=0; j<st.stModifiers[i].iThesaurus; j++)
              {
                if (sWord.equals(st.stModifiers[i].sThesaurus[j].toLowerCase()))
                  iMask = st.stModifiers[i].id;
              }
            }
          }
          break;
      }
    }

    return (iMask);
  }

  /** Method getMask( String str, int iType, maskSymbolsListStruct st )
   * <p> This method will check the Mask key words, phrases to see if any
   *     matches the default mask key words or phrases and return the row
   *     id if it does.
   *
   * @param  iData = Just a value to force this function recognition
   * @param  str   = the string passed in.
   * @param  iType = Retrieve from SYMBOL or MODIIFER
   * @param  st    = Mask Symbols List Data Structure
   * @return stMask = the mask data structure.
   */

  public static maskSymbolsStruct getMask(
      int iData, String str, int iType, maskSymbolsListStruct st )
  {
    int     i     = 0;
    int     j     = 0;
    int     iMask = -1;
    String  sWord = new String(str.toLowerCase());
    maskSymbolsStruct stMask = null;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (i=0; i<st.iCount; i++)
          {
            for (j=0; j<st.stItem[i].iThesaurus; j++)
            {
              if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
                stMask = st.stItem[i];
            }
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          if (st.iModifiers > 0)
          {
            for (i=0; i<st.iModifiers; i++)
            {
              for (j=0; j<st.stModifiers[i].iThesaurus; j++)
              {
                if (sWord.equals(st.stModifiers[i].sThesaurus[j].toLowerCase()))
                  stMask = st.stModifiers[i];
              }
            }
          }
          break;
      }
    }

    return (stMask);
  }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method setHierarchy( int iType, maskSymbolsListStruct st )
   * <p> This method will set the Hierararchy id from the group number and order
   * @param  iType = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st    = mask symbol data list Structure
   * @return st    = mask symbol data list Structure
   */

  public static maskSymbolsListStruct setHierarchy(
                                        int iType, maskSymbolsListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (i=0; i<st.iCount; i++)
          {
            st.stItem[i].iHierarchy =
                100*st.stItem[i].iGroup + st.stItem[i].iOrder;
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          if (st.iModifiers > 0)
          {
            for (i=0; i<st.iModifiers; i++)
            {
              st.stModifiers[i].iHierarchy =
                  100*st.stModifiers[i].iGroup + st.stModifiers[i].iOrder;
            }
          }
          break;
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isLegend( int iLegend, maskSymbolsListStruct st )
   * <p> This method will return true it should be included in legend and false
   *     it should not be included in legend
   * @param  iLegend = the id of the mask
   * @param  st      = mask symbols list data structure
   * @return bValue  = true or false
   */

  public static boolean isLegend( int iLegend, maskSymbolsListStruct st )
  {
    boolean bValue = false;

    if (st != null)
    {
      if (st.stItem[iLegend].iExtend == mask.maskSymbolsStruct.ON)
        bValue = true;
    }

    return (bValue);
  }

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = mask symbols list data structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, maskSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isPhrase()
   * <p> This method will determine if the string comes from the PHRASE
   * @param  str    = string or string phrase being compared.
   * @param  iType  = Retrieve SYMBOL or MODIIFER Symbol
   * @param  st     = mask symbols list data structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isPhrase( String str, int iType,
                                  maskSymbolsListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());
    int     i      = 0;
    int     j      = 0;

    if (st != null)
    {
      switch (iType)
      {
        case mask.maskSymbolsListStruct._SYMBOLS:
          for (i=0; i<st.iCount; i++)
          {
            for (j=0; j<st.stItem[i].iThesaurus; j++)
            {
              if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
              bValue = true;
            }
          }
          break;

        case mask.maskSymbolsListStruct._MODIFIERS:
          if (st.iModifiers > 0)
          {
            for (i=0; i<st.iModifiers; i++)
            {
              for (j=0; j<st.stModifiers[i].iThesaurus; j++)
              {
                if (sWord.equals(st.stModifiers[i].sThesaurus[j].toLowerCase()))
                bValue = true;
              }
            }
          }
          break;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( maskSymbolsListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static maskSymbolsListStruct copyList( maskSymbolsListStruct stOld )
  {
    maskSymbolsListStruct stNew = null;
    int i = 0;

    if (stOld != null)
    {
      stNew = new maskSymbolsListStruct();

      // Mask Colors

      stNew.stForeground = copyColor( stOld.stForeground ); // foreground colors
      stNew.stBackground = copyColor( stOld.stBackground ); // background colors

      // Mask Groups

      stNew.stGroups = copyGroup( stOld.stGroups ); // Mask Groups

      // Modifier Symbols

      if (stOld.iModifiers > 0)
      {
        stNew.iModifiers = stOld.iModifiers;
        stNew.stModifiers = new maskSymbolsStruct[stOld.iModifiers];

        for (i=0; i<stOld.iModifiers; i++)
        {
          stNew.stModifiers[i] = copy( stOld.stModifiers[i] );
        }
      }

      // Mask Symbols

      stNew.iCount = stOld.iCount;
      stNew.stItem = new maskSymbolsStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }

      // Keywords

      if (stOld.iKeywords > 0)
      {
        stNew.iKeywords = stOld.iKeywords;
        stNew.sKeywords = new String[stOld.iKeywords];
        for (i=0; i<stOld.iKeywords; i++)
          stNew.sKeywords[i] = new String( stOld.sKeywords[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( maskSymbolsStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static maskSymbolsStruct copy( maskSymbolsStruct stOld )
  {
    maskSymbolsStruct stNew = new maskSymbolsStruct();
    int i=0;
    int j=0;

    stNew.iHierarchy = stOld.iHierarchy;               // Hierarchy
    stNew.iShow      = stOld.iShow;                    // Default Legend param
    stNew.iExtend    = stOld.iExtend;                  // Extended Legend param
    stNew.id         = stOld.id;                       // ID for Mask Symbol
    stNew.sID        = new String( stOld.sID );        // String ID for Mask
    stNew.iOrder     = stOld.iOrder;                   // Order within the group
    stNew.iGroup     = stOld.iGroup;                   // Group Number
    stNew.sGroup     = new String( stOld.sGroup );     // The group name
    stNew.sName      = new String( stOld.sName );      // Name
    stNew.sAbrev     = new String( stOld.sAbrev );     // Abbreviation
    stNew.sMnemonic  = new String( stOld.sMnemonic );  // Mnemonic

    // Misc variables

    stNew.sType      = new String( stOld.sType );      // Type of Mask
    stNew.size       = new String( stOld.size );       // Size Description
    stNew.dMinimum   = stOld.dMinimum;                 // Minimum Size
    stNew.dMaximum   = stOld.dMaximum;                 // Maximum Size

    // Background color

    for (i=0; i<3; i++)
      stNew.iRGB[i]   = stOld.iRGB[i];

    // Plot Symbol for mask default is an empty or None symbol

    stNew.iRows      = stOld.iRows;
    stNew.iColumns   = stOld.iColumns;
    if (stNew.iRows > 0)
    {
      stNew.symbol     = new String[stNew.iRows][2];
      for (i=0; i<stNew.iRows; i++)
      {
        for (j=0; j<2; j++)
        {
          stNew.symbol[i][j] = new String( stOld.symbol[i][j] );
        }
      }
    }

    // Lithology

    for (i=0; i<10; i++)
      stNew.iLithology[i] = stOld.iLithology[1];

    // Thesaurus for searching and parsing geologist field notes or comments

    stNew.iThesaurus   = stOld.iThesaurus;
    if (stNew.iThesaurus > 0)
    {
      stNew.sThesaurus   = new String[stNew.iThesaurus];
      for (i=0; i<stNew.iThesaurus; i++)
        stNew.sThesaurus[i] = stOld.sThesaurus[i];
    }

    return (stNew);
  }

  /** Method copyGroup( maskGroupsStruct stOld )
   * <p> This method will copy the group data structure
   * @param  stOld = The old group data structure
   * @return stNew = The new group data structure
   */

  public static maskGroupsStruct copyGroup( maskGroupsStruct stOld )
  {
    maskGroupsStruct stNew = new maskGroupsStruct();

    if (stOld != null)
    {
      stNew.iGroups = stOld.iGroups;
      stNew.sGroups = new String[stNew.iGroups][2];

      for (int i=0; i<stNew.iGroups; i++)
      {
        for (int j=0; j<2; j++)
        {
          stNew.sGroups[i][j] = new String( stOld.sGroups[i][j] );
        }
      }
    }

    return (stNew);
  }

  /** Method copyColor( maskGroupsStruct stOld )
   * <p> This method will copy the color data structure
   * @param  stOld = The old color data structure
   * @return stNew = The new color data structure
   */

  public static maskColorsStruct copyColor( maskColorsStruct stOld )
  {
    maskColorsStruct stNew = new maskColorsStruct();
    int len = 0;
    int i   = 0;
    int j   = 0;

    if (stOld != null)
    {
      stNew.iColors   = stOld.iColors;    // total number of records

      if (stOld.colors != null)
      {
        len = stOld.colors.length;
        stNew.colors = new char[len]; // Character Symbol

        for (i=0; i<len; i++)
          stNew.colors[i] = stOld.colors[i];
      }

      if (stOld.sColors != null)
      {
        len = stOld.sColors.length;
        stNew.sColors = new String[len]; // Color description

        for (i=0; i<len; i++)
          stNew.sColors[i] = new String( stOld.sColors[i] );
      }

      if (stOld.iRGB != null)
      {
        stNew.iRGB = new int[len][3]; // rgb values

        for (i=0; i<len; i++)
        {
          for (j=0; j<3; j++)
            stNew.iRGB[i][j] = stOld.iRGB[i][j];
        }
      }
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- SORT METHODS ------------------------------ *
   * ======================================================================== */

  /** Method bubbleSort( maskSymbolsListStruct st )
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = Mask Symbols data list structure
   * @return st = sorted Mask Symbols data list structure
   */

  public static maskSymbolsListStruct bubbleSort( maskSymbolsListStruct st )
  {
    boolean swappedOnPrevRun  = true;
    maskSymbolsStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].iHierarchy > st.stItem[i+1].iHierarchy)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( maskSymbolsListStruct st )
   * <p> This method will print the mask symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( maskSymbolsListStruct st )
  {
    int i=0;

    System.out.println(" ------------ Foreground Colors ------------- \n");
    print( st.stForeground );

    System.out.println("\n ------------ Background Colors ------------- \n");
    print( st.stBackground );

    if (st.stGroups != null)
    {
      System.out.println("\n ------------ Mask Groups ------------- \n");
      print(st.stGroups);
    }

    if (st.iModifiers > 0)
    {
      System.out.println("\n ------------ Modifiers ------------- \n");
      for (i=0; i<st.iModifiers; i++)
      {
        print( st.stModifiers[i] );
      }
    }

    System.out.println("\n ------------ Mask ------------- \n");
    for (i=0; i<st.iCount; i++)
    {
      print( st.stItem[i] );
    }

    if (st.iKeywords > 0)
    {
      System.out.println("\n ------------ Keywords ------------- \n");

      for (i=0; i<st.iCount; i++)
      {
        System.out.print( st.sKeywords[i] + ", " );
      }
      System.out.println("\n");
    }
  }

  /** Method print( maskColorsStruct st )
   * <p> This method will print the color data structure
   * @param  st = Mask colors data structure
   */

  public static void print( maskColorsStruct st )
  {
    int i=0;

    if (st != null)
    {
      if (st.iColors > 0)
      {
        if (st.colors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.colors[i]  + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
        else if (st.sColors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.sColors[i] + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
      }
    }
  }

  /** Method print( maskGroupsStruct st )
   * <p> This method will print the group data structure
   * @param  st = Mask Groups data structure
   */

  public static void print( maskGroupsStruct st )
  {
    if (st != null)
    {
      if (st.iGroups > 0)
      {
        for (int i=0; i<st.iGroups; i++)
        {
          System.out.println( i + " " + st.sGroups[i][0] +
                              " <-> " + st.sGroups[i][1] );
        }
      }
    }
  }

  /** Method print( maskSymbolsStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( maskSymbolsStruct st )
  {
    int i = 0;

    if (st != null)
    {
      System.out.println( st.iHierarchy + " " + // Hierarchy within the list
                          st.id         + " " + // ID for Mask Symbol
                          st.sID        + " " + // String ID for Mask Symbol
                          st.iOrder     + " " + // Order within the group
                          st.iGroup     + " " + // Group Number
                          st.sGroup     + " " + // The group the mask belongs
                          st.sName      + " " + // Name or Description of mask
                          st.sAbrev     + " " + // Abbreviation of mask
                          st.sMnemonic );       // Mnemonic of mask

      System.out.println( "Misc "     +
                          st.sType    + " " +   // Type of Mask
                          st.size     + " " +   // Size Description
                          st.dMinimum + " " +   // Minimum Size
                          st.dMaximum );        // Maximum Size

      System.out.println( "Color "   +
                          st.iRGB[0] + " " +    // Background Color Red Number
                          st.iRGB[1] + " " +    // Background Color Green Number
                          st.iRGB[2]);          // Background Color Blue Number

      if (st.symbol != null)
      {
        for (i=0; i<st.iRows; i++)
        {
          System.out.println(st.symbol[i][0] + " " + st.symbol[i][1]);
        }
      }

      if (st.iThesaurus > 0)
      {
        for (i=0; i<st.iThesaurus; i++)
        {
          System.out.print(st.sThesaurus[i] + " ");
        }
        System.out.println("\n");
      }
    }
  }
}