/*
 * @ReadMaskSymbolsXMLFile.java Version 1.0 07/20/2009
 *
 * Copyright (c) 2009 John R. Victorine
 * All Rights Reserved.
 */

package mask.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import mask.maskColorsStruct;
import mask.maskGroupsStruct;
import mask.maskSymbolsListStruct;
import mask.maskSymbolsStruct;

/** Class ReadMaskSymbolsXMLFile()
 * <p> This Class will parse the Rock Lithology XML Data Stream
 *     directly to the Rock Lithology List Data Structure.
 *
 *  @version 1.1 07/20/2009
 *  @author  John R. Victorine
 */

public class ReadMaskSymbolsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  // DTD terms

  public static final String ROOTNAME    = "symbol_data";   // ROOT NAME
  public static final String RECORDS     = "records";   // Attribute

  public static final String BACKGROUND  = "background"; // ELEMENT
  public static final String FOREGROUND  = "foreground"; // ELEMENT

  public static final String COLOR       = "color";      // ELEMENT

  public static final String SYMBOL      = "symbol";     // Symbol Text
  public static final String RED         = "r";          // Red
  public static final String GREEN       = "g";          // Green
  public static final String BLUE        = "b";          // BLue

  public static final String GROUPS      = "groups";     // ELEMENT

  public static final String GROUP       = "group";      // ELEMENT

  public static final String MNEMONIC    = "mnemonic";   // Mnemonic
  public static final String DESCRIPTION = "desc";       // Description

  public static final String MODIFIERS   = "modifiers";  // ELEMENT
  public static final String MASKS       = "masks";      // ELEMENT

  public static final String TYPE        = "type";       // Type of Symbol text
  public static final String ROWS        = "rows";       // Number of Rows
  public static final String WORDS       = "words";      // Maximum Thesaurus

  public static final String DATA        = "data";       // ELEMENT

  public static final String COLUMNS     = "columns";    // ELEMENT
  public static final String COLUMN      = "column";     // ELEMENT

  public static final String ID          = "id";         // Mask ID
  public static final String SID         = "sid";        // Mask ID
  public static final String SHOW        = "show";       // To Add to Legend
  public static final String ORDER       = "order";      // Order within group
  public static final String SIZE        = "size";       // Size description
  public static final String MINIMUM     = "min";        // Minimum Size
  public static final String MAXIMUM     = "max";        // Maximum Size
  public static final String ABREV       = "abrv";       // Abbreviation
  public static final String NAME        = "name";       // Name of Lithology

  public static final String SYMBOLS     = "symbols";    // ELEMENT
  public static final String ROW         = "row";        // Row Number
  public static final String LEFT        = "l";          // Left Image Mask
  public static final String RIGHT       = "r";          // Right Image Mask
  public static final String CENTER      = "c";          // Center Image Mask

  public static final String THESAURUS   = "thesaurus";  // ELEMENT
  public static final String KEYWORDS    = "keywords";   // ELEMENT

  public static final String WORD        = "word";       // ELEMENT

  public static final String PHRASE      = "phrase";     // Word Phrase

  public static final String LITHOLOGY   = "lith";   // ELEMENT
  public static final String _1          = "o1";     // Lithology - 0  to 10 %
  public static final String _2          = "o2";     // Lithology - 10 to 20 %
  public static final String _3          = "o3";     // Lithology - 20 to 30 %
  public static final String _4          = "o4";     // Lithology - 30 to 40 %
  public static final String _5          = "o5";     // Lithology - 40 to 50 %
  public static final String _6          = "o6";     // Lithology - 50 to 60 %
  public static final String _7          = "o7";     // Lithology - 60 to 70 %
  public static final String _8          = "o8";     // Lithology - 70 to 80 %
  public static final String _9          = "o9";     // Lithology - 80 to 90 %
  public static final String _10         = "o10";    // Lithology - 90 to 100 %

  // ELEMENT Identifiers

  private static final int _NONE       = 0;
  private static final int _BACKGROUND = 1;  // Read Background Colors
  private static final int _FOREGROUND = 2;  // Read Foreground Colors
  private static final int _COLOR      = 3;  // Individual Color indicator
  private static final int _GROUPS     = 4;  // Read Groups
  private static final int _GROUP      = 5;  // Individual Group indicator
  private static final int _MASKS      = 6;  // Read Masks
  private static final int _DATA       = 7;  // Individual Mask indicator
  private static final int _COLUMN     = 8;  // Mask Column
  private static final int _SYMBOLS    = 9;  // Read Mask Symbols
  private static final int _SYMBOL     = 10; // Individual Mask Symbol Line
  private static final int _THESAURUS  = 11; // Read Mask Thesaurus
  private static final int _WORD       = 12; // Individual Mask Word
  private static final int _LITHOLOGY  = 13; // Lithology indicator
  private static final int _MODIFIERS  = 14; // Read Masks
  private static final int _KEYWORDS   = 15; // List of Keywords

  private int                   iRows    = 0;
  private int                   iLines   = 0;
  private int                   iColumns = 0;
  private int                   iCount   = -1;
  private int                   iRow     = 0;
  private int                   iData    = _NONE;
  private int                   iColor   = _NONE;
  private int                   iSymbol  = _NONE;
  private int                   iWord    = _NONE;
  private maskSymbolsListStruct stList   = new maskSymbolsListStruct();

  private int                   iError   = 0;
  private String                sError   = "";

  /** Method ReadMaskSymbolsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadMaskSymbolsXMLFile() { iType = FILE; }

  /** Method ReadMaskSymbolsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadMaskSymbolsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public maskSymbolsListStruct Process(String filename)
  {
    Read(filename);
    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadMaskSymbolsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadMaskSymbolsXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name
    int    m = 0;
    int    n = 0;

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    if (eName.equals(BACKGROUND)) // Loading Lithology Background Color
    {
      iRow                = -1;
      iData               = _BACKGROUND;
      iColor              = _BACKGROUND;
      stList.stBackground = new maskColorsStruct();
    }

    if (eName.equals(FOREGROUND)) // Loading Lithology Foreground Color
    {
      iRow                = -1;
      iData               = _FOREGROUND;
      iColor              = _FOREGROUND;
      stList.stForeground = new maskColorsStruct();
    }

    if (eName.equals(COLOR))
    {
      iData = _COLOR;
      iRow++;
    }

    if (eName.equals(GROUPS))
    {
      iRow            = -1;
      iColor          = _NONE;
      iData           = _GROUPS;
      stList.stGroups = new maskGroupsStruct();
    }

    if (eName.equals(GROUP))
    {
      iData = _GROUP;
      iRow++;
    }

    if (eName.equals(MASKS))
    {
      iRow    = -1;
      iSymbol = _MASKS;
      iData   = _MASKS;
      iWord   = _MASKS;
    }

    if (eName.equals(MODIFIERS))
    {
      iRow    = -1;
      iSymbol = _MODIFIERS;
      iData   = _MODIFIERS;
      iWord   = _MODIFIERS;
    }

    if (eName.equals(DATA))
    {
      iData = _DATA;
      iRow++;

      switch (iSymbol)
      {
        case _MASKS:
          stList.stItem[iRow] = new maskSymbolsStruct();
          break;

        case _MODIFIERS:
          stList.stModifiers[iRow] = new maskSymbolsStruct();
          break;
      }
    }

    if (eName.equals(COLUMN)) { iData = _COLUMN; }

    if (eName.equals(SYMBOLS))
    {
      iData  = _SYMBOLS;
      iCount = -1;
    }

    if (eName.equals(SYMBOL)) { iData = _SYMBOL; }

    if (eName.equals(THESAURUS))
    {
      iData = _THESAURUS;
      iCount = -1;
    }

    if (eName.equals(WORD))
    {
      iData = _WORD;
      iCount++;
    }

    if (eName.equals(LITHOLOGY))
    {
      iData = _LITHOLOGY;
    }

    if (eName.equals(KEYWORDS))
    {
      iRow  = -1;
      iData = _KEYWORDS;
      iWord = _KEYWORDS;
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _BACKGROUND:  // Read Background Colors
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stBackground.iColors = iRows;
              stList.stBackground.sColors = new String[iRows];
              stList.stBackground.iRGB    = new int[iRows][3];

              // Initialize the Color Variables

              for (m=0; m<iRows; m++)
              {
                stList.stBackground.sColors[m] = new String("");
                for (n=0; n<3; n++)
                {
                  stList.stBackground.iRGB[m][n] = 255;
                }
              }
              break;

            case _FOREGROUND:  // Read Foreground Colors
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stForeground.iColors = iRows;
              stList.stForeground.colors  = new char[iRows];
              stList.stForeground.iRGB    = new int[iRows][3];

              // Initialize the Color Variables

              for (m=0; m<iRows; m++)
              {
                stList.stForeground.colors[m] = ' ';
                for (n=0; n<3; n++)
                {
                  stList.stForeground.iRGB[m][n] = 255;
                }
              }
              break;

            case _GROUPS:      // Read Groups
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stGroups.iGroups = iRows;
              stList.stGroups.sGroups = new String[iRows][2];

              // Initialize the Group Variables

              for (m=0; m<iRows; m++)
              {
                for (n=0; n<2; n++)
                {
                  stList.stGroups.sGroups[m][n] = new String("");
                }
              }
              break;

            case _MASKS:    // Read Masks
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iCount = iRows;
              stList.stItem = new maskSymbolsStruct[iRows];
              break;

            case _MODIFIERS:    // Read Modifiers
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iModifiers = iRows;
              stList.stModifiers = new maskSymbolsStruct[iRows];
              break;

            case _SYMBOLS:     // Read Mineral Symbols
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              switch (iSymbol)
              {
                case _MASKS:
                  stList.stItem[iRow].symbol = new String[iRows][2];

                  // Initialize the Symbol Variable

                  for (m=0; m<iRows; m++)
                  {
                    for (n=0; n<2; n++)
                    {
                      stList.stItem[iRow].symbol[m][n] =
                          new String("..........");
                    }
                  }
                  break;

                case _MODIFIERS:
                  stList.stModifiers[iRow].symbol = new String[iRows][2];

                  // Initialize the Symbol Variable

                  for (m=0; m<iRows; m++)
                  {
                    for (n=0; n<2; n++)
                    {
                      stList.stModifiers[iRow].symbol[m][n] =
                          new String("..........");
                    }
                  }
                  break;
              }
              break;

            case _THESAURUS:   // Read Mineral Thesaurus
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              switch (iSymbol)
              {
                case _MASKS:
                  stList.stItem[iRow].iThesaurus = iRows;
                  stList.stItem[iRow].sThesaurus = new String[iRows];

                  // Initialize the Thesaurus Variable

                  for (m=0; m<iRows; m++)
                  {
                    stList.stItem[iRow].sThesaurus[m] = new String("");
                  }
                  break;

                case _MODIFIERS:
                  stList.stModifiers[iRow].iThesaurus = iRows;
                  stList.stModifiers[iRow].sThesaurus = new String[iRows];

                  // Initialize the Thesaurus Variable

                  for (m=0; m<iRows; m++)
                  {
                    stList.stModifiers[iRow].sThesaurus[m] = new String("");
                  }
                  break;
              }
              break;

            case _KEYWORDS:
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.iKeywords = iRows;
              stList.sKeywords = new String[iRows];
              break;

            case _COLOR:       // Individual Color indicator
            case _GROUP:       // Individual Group indicator
            case _DATA:        // Individual Mask indicator
            case _COLUMN:      // Mask Column
            case _SYMBOL:      // Individual Mask Symbol Line
            case _WORD:        // Individual Mask Word
            case _LITHOLOGY:   // Lithology
              break;
          }
        }

        switch (iData)
        {
          case _COLOR:       // Individual Color indicator
            switch (iColor)
            {
              case _FOREGROUND:  // Read Foreground Colors
                stList.stForeground =
                    ParseData( aName, sOut.trim(), stList.stForeground );
                break;

              case _BACKGROUND:  // Read Background Colors
                stList.stBackground =
                    ParseData( aName, sOut.trim(), stList.stBackground );
                break;
            }
            break;

          case _GROUP:  // Read each Group names
            stList.stGroups = ParseData( aName, sOut.trim(), stList.stGroups );
            break;

          case _MASKS:
          case _MODIFIERS:
            if (aName.equals(ROWS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                iLines = cmn.cmnString.stringToInt(sOut.trim());
            }

            if (aName.equals(COLUMNS))
            {
              if (cmn.cmnString.isNumeric(sOut.trim()))
                iColumns = cmn.cmnString.stringToInt(sOut.trim());
            }
            break;

          case _COLUMN:      // Mineral Column
            switch (iSymbol)
            {
              case _MASKS:
                stList.stItem[iRow] =
                  ParseData( aName, sOut.trim(), stList.stItem[iRow] );
                break;

              case _MODIFIERS:
                stList.stModifiers[iRow] =
                  ParseData( aName, sOut.trim(), stList.stModifiers[iRow] );
                break;
            }
            break;

          case _SYMBOL:      // Individual Mineral Symbol Line
            switch (iSymbol)
            {
              case _MASKS:
                stList.stItem[iRow] =
                  ParseSymbol( aName, sOut.trim(), stList.stItem[iRow] );
                break;

              case _MODIFIERS:
                stList.stModifiers[iRow] =
                  ParseSymbol( aName, sOut.trim(), stList.stModifiers[iRow] );
                break;
            }
            break;

          case _LITHOLOGY: // Default Lithology Distribution
            switch (iSymbol)
            {
              case _MASKS:
                stList.stItem[iRow] =
                  parseLithology( aName, sOut.trim(), stList.stItem[iRow] );
                break;

              case _MODIFIERS:
                stList.stModifiers[iRow] =
                  parseLithology( aName, sOut.trim(), stList.stModifiers[iRow] );
                break;
            }
            break;

          case _WORD:        // Individual Mask Word
            switch (iWord)
            {
              case _MASKS:
                if (aName.equals(PHRASE))
                  stList.stItem[iRow].sThesaurus[iCount] =
                      new String(sOut.trim());
                break;

              case _MODIFIERS:
                if (aName.equals(PHRASE))
                  stList.stModifiers[iRow].sThesaurus[iCount] =
                      new String(sOut.trim());
                break;

              case _KEYWORDS:
                if (aName.equals(PHRASE))
                {
                  iRow++;
                  stList.sKeywords[iRow] = new String(sOut.trim());
                }
                break;
            }
            break;
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData( String sIdentifier, String sData, maskColorsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = Color Data Structure
   * @return st          = Color Data Structure
   */

  private maskColorsStruct ParseData( String sIdentifier,
                                      String sData,
                                      maskColorsStruct st )
  {
    char c[] = null;

    if (sIdentifier.equals(SYMBOL))      // Symbol Text
    {
      switch (iColor)
      {
        case _BACKGROUND:  // Read Background Colors
          st.sColors[iRow] = new String(sData);
          break;
        case _FOREGROUND:  // Read Foreground Colors
          c = sData.toCharArray();
          if (c.length > 0) st.colors[iRow] = c[0];
          break;
      }
    }

    if (sIdentifier.equals(RED))           // Red
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][0] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(GREEN))          // Green
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][1] = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(BLUE))          // BLue
    {
      if (iRow > -1)
      {
        if (cmn.cmnString.isNumeric(sData))
          st.iRGB[iRow][2] = cmn.cmnString.stringToInt(sData);
      }
    }

    return (st);
  }

  /** ParseData( String sIdentifier, String sData, maskGroupsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = Group Data Structure
   * @return st          = Group Data Structure
   */

  private maskGroupsStruct ParseData( String sIdentifier,
                                      String sData,
                                      maskGroupsStruct st )
  {
    if (sIdentifier.equals(MNEMONIC))
    {
      st.sGroups[iRow][mask.maskGroupsStruct._MNEMONIC] =
          new String(sData);
    }

    if (sIdentifier.equals(DESCRIPTION))
    {
      st.sGroups[iRow][mask.maskGroupsStruct._DESCRIPTION] =
          new String(sData);
    }

    return (st);
  }

  /** ParseData( String sIdentifier, String sData, maskSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = mask Data Structure
   * @return st          = mask Data Structure
   */

  private maskSymbolsStruct ParseData( String sIdentifier,
                                       String sData,
                                       maskSymbolsStruct st)
  {
    String sTemp  = "";
    int    iColor = -1;

    if (sIdentifier.equals(ID)) // Lithology ID
    {
      if (cmn.cmnString.isNumeric(sData))
        st.id = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(SID))
    {
      st.sID = new String(sData);
      if (st.id == 0) st.id = iRow;
    }

    if (sIdentifier.equals(ROWS))
    {
      if (cmn.cmnString.isNumeric(sData))
        iLines = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(COLUMNS))
    {
      if (cmn.cmnString.isNumeric(sData))
        iColumns = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(SHOW))  // To Add to Legend
    {
      if (sData.equals("yes"))
      {
        st.iShow   = mask.maskSymbolsStruct.ON;
        st.iExtend = mask.maskSymbolsStruct.ON;
      }
    }

    if (sIdentifier.equals(ORDER))     // Order within group
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iOrder = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(GROUP))      // Group Mnemonic
    {
      if (stList.stGroups != null)
      {
        st.iGroup = mask.maskSymbolsUtility.getGroupID(
                    mask.maskGroupsStruct._MNEMONIC,
                    sData,
                    stList.stGroups );

        st.sGroup = new String(
            mask.maskSymbolsUtility.getGroup(
                    mask.maskGroupsStruct._DESCRIPTION,
                    st.iGroup,
                    stList.stGroups ));
      }
    }

    if (sIdentifier.equals(COLOR))
    {
      iColor  = mask.maskSymbolsUtility.getColor(
                  sData, stList.stBackground );
      st.iRGB = mask.maskSymbolsUtility.getRGB(
                  iColor, stList.stBackground );
    }

    if (sIdentifier.equals(TYPE)) { st.sType = new String(sData); }
    if (sIdentifier.equals(SIZE)) { st.size  = new String(sData); }
    if (sIdentifier.equals(MINIMUM))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dMinimum = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(MAXIMUM))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dMaximum = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(MNEMONIC)) { st.sMnemonic = new String(sData); }
    if (sIdentifier.equals(ABREV))    { st.sAbrev    = new String(sData); }
    if (sIdentifier.equals(NAME))     { st.sName     = new String(sData); }

    return (st);
  }

  /** ParseSymbol( String sIdentifier, String sData, maskSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = mask Data Structure
   * @return st          = mask Data Structure
   */

  private maskSymbolsStruct ParseSymbol( String sIdentifier,
                                         String sData,
                                         maskSymbolsStruct st)
  {
    if (sIdentifier.equals(ROW))   // Rows Number
    {
      st.iRows    = iLines;
      st.iColumns = iColumns;

      if (cmn.cmnString.isNumeric(sData))
        iCount = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(CENTER))  // Center Image
    {
      st.symbol[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(LEFT))  // Left Image
    {
      st.symbol[iCount][0] = new String(sData);
    }

    if (sIdentifier.equals(RIGHT)) // right Image
    {
      st.symbol[iCount][1] = new String(sData);
    }

    return (st);
  }

  /** parseLithology( String sIdentifier, String sData, maskSymbolsStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Data String
   * @param  st          = mask Data Structure
   * @return st          = mask Data Structure
   */

  private maskSymbolsStruct parseLithology( String sIdentifier,
                                            String sData,
                                            maskSymbolsStruct st)
  {
    if (sIdentifier.equals(_1))      // Lithology - 0  to 10 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[0] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_2))      // Lithology - 10 to 20 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[1] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_3))      // Lithology - 20 to 30 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[2] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_4))      // Lithology - 30 to 40 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[3] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_5))      // Lithology - 40 to 50 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[4] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_6))      // Lithology - 50 to 60 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[5] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_7))      // Lithology - 60 to 70 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[6] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_8))      // Lithology - 70 to 80 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[7] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_9))      // Lithology - 80 to 90 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[8] = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(_10))     // Lithology - 90 to 100 %
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iLithology[9] = cmn.cmnString.stringToInt(sData);
      }
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
