/*
 * @textureUtility.java Version 1.1 09/14/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.texture;

import iqstrat.iqstratHeadersStruct;
import lith.texture.textureListStruct;
import lith.texture.textureStruct;

import lith.lithology.lithologySymbolsListStruct;

import las.lasFileDataStruct;

import mask.maskSymbolsStruct;
import mask.maskSymbolsListStruct;

/** Class textureUtility
 *  <p> This Class will provide basic utilities for the Rock Texture
 *      data structures.
 *
 *  @version 1.1 09/14/2007
 *  @author  John R. Victorine
 */

public class textureUtility
{
  public static final double GAMMA_MAX  = 150.0;
  public static final double GAMMA_MIN  = 0.0;
  public static final double CLEAN      = 60.0;

  /** Method buildNew()
   * <p> This method will create a new Rock Texture Structure based on the
   *     depth range.
   * @param  iUnit      = Depth Units, 0-Feet and 1-Inches;  Default is Feet
   * @param  depthStart = The Starting Depth of data
   * @param  depthEnd   = The Ending Depth of Data
   * @return st         = The new Rock Texture List Data Struture
   */

  public static textureListStruct buildNew( int iUnit,
                                            double depthStart,
                                            double depthEnd )
  {
    textureListStruct st = null;
    int    iTotal  = 0;
    int    iCount  = 0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dScale  = 0.0;

    iCount = 1+ (int) (iqstrat.iqstratTracksStruct._SCALE_FACTOR[iUnit] *
                    Math.abs( depthEnd - depthStart ));
    dScale = iqstrat.iqstratTracksStruct._SCALE[iUnit];

    st            = new textureListStruct();
    st.iCount     = iCount;
    st.iUnit      = iUnit;
    st.depthStart = depthStart;
    st.depthEnd   = depthEnd;
    st.stItem     = new textureStruct[iCount];

    for (int i=0; i<iCount; i++)
    {
      if (iTotal < iCount)
      {
        depth1 = depthStart + i * dScale;
        depth2 = depthStart + (i+1) * dScale;

        st.stItem[iTotal] = new textureStruct();
        st.stItem[iTotal].depthStart = getDepth(iUnit, depth1);
        st.stItem[iTotal].depthEnd   = getDepth(iUnit, depth2);

        iTotal++;
      }
    }

    st.iCount = iTotal;

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock texture data list structure
   * @return  st = the Rock texture data list structure
   */

  public static textureListStruct computeDepthRange(textureListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }

      st.depthStart = Math.floor(st.depthStart);
      st.depthEnd   = Math.ceil(st.depthEnd);
    }

    return (st);
  }

  /** Method getUnit()
   * <p> This method will determine the smallest division of the data and set
   *     the spacing accordingly.
   * @param  depth = the unit depth
   * @return iUnit = The smallest unit
   */

  public static int getUnit(double depth)
  {
    int iUnit = iqstrat.iqstratTracksStruct._SCALE_1_FOOT;

    if (depth <= 0.5)     iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
    if (depth <= 0.16667) iUnit = iqstrat.iqstratTracksStruct._SCALE_2_INCH;
    if (depth <= 0.08333) iUnit = iqstrat.iqstratTracksStruct._SCALE_1_INCH;
    if (depth <= 0.04167) iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_INCH;
    if (depth <= 0.25)    iUnit = iqstrat.iqstratTracksStruct._SCALE_QUARTER_FOOT;
    if (depth <= 0.1)     iUnit = iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT;

    return (iUnit);
  }

  /** Method getUnit()
   * <p> This method will determine the smallest division of the data and set
   *     the spacing accordingly.
   * @param  iUnit = the scale factor the depth is being divided into
   * @param  depth = the unit depth
   * @return iUnit = The smallest unit
   */

  public static int getUnit( int iUnit, double depth )
  {
    int    iTail  = (int) depth;
    int    iTail1 = iTail;
    double dTemp  = 0.0;
    double dTail  = Math.abs( depth - (double) iTail );

    if (dTail == 0.0) dTail = 0.5;

    if (dTail <= 0.5)     iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
    if (dTail <= 0.16667) iUnit = iqstrat.iqstratTracksStruct._SCALE_2_INCH;
//    if (dTail <= 0.08333) iUnit = iqstrat.iqstratTracksStruct._SCALE_1_INCH;
//    if (dTail <= 0.04167) iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_INCH;
//    if (dTail <= 0.25)    iUnit = iqstrat.iqstratTracksStruct._SCALE_QUARTER_FOOT;
    if (dTail <= 0.1)     iUnit = iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT;

    return (iUnit);
  }

  /** Method getDepth()
   * <p> This method will return the depth nearest the scale depth
   * @param  iUnit = the scale factor the depth is being divided into
   * @param  depth = the depth being modified
   * @return depth = the new computed depth
   */

  public static double getDepth(int iUnit, double depth)
  {
    int    iTail  = (int) depth;
    int    iTail1 = iTail;
    double dTemp  = 0.0;
    double dTail  = Math.abs( depth - (double) iTail );
    int    iNeg   = 1;

    if (depth < 0.0)
    {
      iTail1 = (int) Math.abs( depth );
      if (iTail1 == 0) iTail1 = 1;
      iNeg = iTail / iTail1;
    }

    switch(iUnit)
    {
      case iqstrat.iqstratTracksStruct._SCALE_HALF_INCH: // each cell 1/2 in deep
        if (dTail <= .020835)                        dTemp = 0.0;     // 0 in
        if ((dTail > .020835) && (dTail <= .0625))   dTemp = 0.04167; // .5 in
        if ((dTail > .0625)   && (dTail <= .104165)) dTemp = 0.08333; // 1 in
        if ((dTail > .104165) && (dTail <= .145835)) dTemp = 0.125;   // 1.5 in
        if ((dTail > .145835) && (dTail <= .1875))   dTemp = 0.16667; // 2 in
        if ((dTail > .1875)   && (dTail <= .229165)) dTemp = 0.20833; // 2.5 in
        if ((dTail > .229165) && (dTail <= .270835)) dTemp = 0.25;    // 3.0 in
        if ((dTail > .270835) && (dTail <= .3125))   dTemp = 0.29167; // 3.5 in
        if ((dTail > .3125)   && (dTail <= .354165)) dTemp = 0.33333; // 4 in
        if ((dTail > .354165) && (dTail <= .395835)) dTemp = 0.375;   // 4.5 in
        if ((dTail > .395835) && (dTail <= .4375))   dTemp = 0.41667; // 5 in
        if ((dTail > .4375)   && (dTail <= .479165)) dTemp = 0.45833; // 5.5 in
        if ((dTail > .479165) && (dTail <= .520835)) dTemp = 0.5;     // 6.0 in
        if ((dTail > .520835) && (dTail <= .5625))   dTemp = 0.54167; // 6.5 in
        if ((dTail > .5625)   && (dTail <= .604165)) dTemp = 0.58333; // 7.0 in
        if ((dTail > .604165) && (dTail <= .645835)) dTemp = 0.625;   // 7.5 in
        if ((dTail > .645835) && (dTail <= .6875))   dTemp = 0.66667; // 8.0 in
        if ((dTail > .6875)   && (dTail <= .729165)) dTemp = 0.70833; // 8.5 in
        if ((dTail > .729165) && (dTail <= .790835)) dTemp = 0.75;    // 9.0 in
        if ((dTail > .790835) && (dTail <= .8125))   dTemp = 0.79167; // 9.5 in
        if ((dTail > .8125)   && (dTail <= .854165)) dTemp = 0.83333; // 10 in
        if ((dTail > .854165) && (dTail <= .915835)) dTemp = 0.875;   // 10.5 in
        if ((dTail > .915835) && (dTail <= .9375))   dTemp = 0.91667; // 11 in
        if ((dTail > .9375)   && (dTail <= .979165)) dTemp = 0.95833; // 11.5 in
        if (dTail >  .979165)                        dTemp = 1.0;     // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_1_INCH:    // each cell 1 in deep
        if (dTail <= .04167)                       dTemp = 0.0;           // 0 in
        if ((dTail > .04167) && (dTail <= .125))   dTemp = 0.08333; // 1 in
        if ((dTail > .125)   && (dTail <= .20833)) dTemp = 0.16667; // 2 in
        if ((dTail > .20833) && (dTail <= .29167)) dTemp = 0.25;    // 3 in
        if ((dTail > .29167) && (dTail <= .375))   dTemp = 0.33333; // 4 in
        if ((dTail > .375)   && (dTail <= .45833)) dTemp = 0.41667; // 5 in
        if ((dTail > .45833) && (dTail <= .54167)) dTemp = 0.5;     // 6 in
        if ((dTail > .54167) && (dTail <= .625))   dTemp = 0.58333; // 7 in
        if ((dTail > .625)   && (dTail <= .70833)) dTemp = 0.66667; // 8 in
        if ((dTail > .70833) && (dTail <= .79167)) dTemp = 0.75;    // 9 in
        if ((dTail > .79167) && (dTail <= .875))   dTemp = 0.83333; // 10 in
        if ((dTail > .875)   && (dTail <= .95833)) dTemp = 0.91667; // 11 in
        if (dTail >  .95833)                       dTemp = 1.0;     // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_2_INCH:    // each cell 2 in deep
        if (dTail <= .08333)                       dTemp = 0.0;            // 0 in
        if ((dTail > .08333) && (dTail <= .25))    dTemp = 0.16667;  // 2 in
        if ((dTail > .25)    && (dTail <= .41667)) dTemp = 0.33333;  // 4 in
        if ((dTail > .41667) && (dTail <= .58333)) dTemp = 0.5;      // 6 in
        if ((dTail > .58333) && (dTail <= .75))    dTemp = 0.66667;  // 8 in
        if ((dTail > .75)    && (dTail <= .91667)) dTemp = 0.83333;  // 10 in
        if (dTail >  .91667)                       dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT:
        if (dTail <= .05)                       dTemp = 0.0;      // 0 ft
        if ((dTail > .05)  && (dTail <= .105))  dTemp = 0.1;      // 3 in
        if ((dTail > .105) && (dTail <= .205))  dTemp = 0.2;      // 3 in
        if ((dTail > .205) && (dTail <= .305))  dTemp = 0.3;      // 3 in
        if ((dTail > .305) && (dTail <= .405))  dTemp = 0.4;      // 3 in
        if ((dTail > .405) && (dTail <= .505))  dTemp = 0.5;      // 3 in
        if ((dTail > .505) && (dTail <= .605))  dTemp = 0.6;      // 3 in
        if ((dTail > .605) && (dTail <= .705))  dTemp = 0.7;      // 3 in
        if ((dTail > .705) && (dTail <= .805))  dTemp = 0.8;      // 3 in
        if ((dTail > .805) && (dTail <= .905))  dTemp = 0.9;      // 3 in
        if (dTail >  .905)                      dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_QUARTER_FOOT:
        if (dTail <= .125)                     dTemp = 0.0;      // 0 in
        if ((dTail > .125) && (dTail <= .375)) dTemp = 0.25;     // 3 in
        if ((dTail > .375) && (dTail <= .625)) dTemp = 0.5;      // 6 in
        if ((dTail > .625) && (dTail <= .875)) dTemp = 0.75;     // 9 in
        if (dTail >  .75)                      dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT: // each cell 1/2 ft deep
        if (dTail <= .25)                     dTemp = 0.0;      // 0 in
        if ((dTail > .25) && (dTail <= .75))  dTemp = 0.5;      // 6 in
        if (dTail >  .75)                     dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_1_FOOT:    // each cell 1 ft deep
        if (dTail <= .5) dTemp = 0.0;
        if (dTail >  .5) dTemp = 1.0;
        break;
    }

//    depth = (double) iTail + dTemp;
    depth = (double) iNeg * (iTail1 + dTemp);

    return (depth);
  }

  /** Method fillEmptySpaces()
   * <p> This method will fill empty spaces within the depth range
   * @param  st = The Rock Texture List Data Structure
   * @return st = The Rock Texture List Data Structure
   */

  public static textureListStruct fillEmptySpaces(textureListStruct st)
  {
    int i = 0;
    int j = 0;
    int iFilled1 = 0;
    int iFilled2 = 0;
    int iFilled3 = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount-2; i++)
      {
        iFilled1 = 0;
//        for (j=0; j<10; j++)
//        {
//          if (st.stItem[i].iLithology[j] != lith.plot.lithPlotSymbol._NO_TEXTURE)
          if (st.stItem[i].iTexture != lith.texture.textureTypesStruct._TEXTURE_NONE)
            iFilled1 = 1;
//        }

        iFilled2 = 0;
        iFilled3 = 0;
        if (iFilled1 != 0)
        {
//          for (j=0; j<10; j++)
//          {
//            if (st.stItem[i+1].iLithology[j] != lith.plot.lithPlotSymbol._NO_TEXTURE)
            if (st.stItem[i+1].iTexture != lith.texture.textureTypesStruct._TEXTURE_NONE)
              iFilled2 = 1;
//          }

//          for (j=0; j<10; j++)
//          {
//            if (st.stItem[i+2].iLithology[j] != lith.plot.lithPlotSymbol._NO_TEXTURE)
            if (st.stItem[i+2].iTexture != lith.texture.textureTypesStruct._TEXTURE_NONE)
              iFilled3 = 1;
//          }

          if ((iFilled2 == 0) && (iFilled3 == 1))
          {
            st.stItem[i+1] = copy(st.stItem[i]);
          }
        }
      }
    }

    return (st);
  }

  /** Method addHeaderData()
   * <p> This method will add the header data to the Texture data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Texture Block List Data Structure
   * @return st       = The Rock Texture Block List Data Structure
   */

  public static textureListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                textureListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method addRecord( textureStruct st, textureListStruct stList)
   * <p> This method will add a row to the Texture List data structure
   * @param  st     = Rock Texture data structure to be added
   * @param  stList = the old Rock Texture List data structure.
   * @return stList = the new Rock Texture List data structure.
   */

  public static textureListStruct addRecord( textureStruct st,
                                             textureListStruct stList)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if ((st != null) && (stList != null))
    {
      depthStart = st.depthStart;
      depthEnd   = st.depthEnd;

      for (int i=0; i<stList.iCount; i++)
      {
        if ((depthStart >= stList.stItem[i].depthStart) &&
            (depthStart <  stList.stItem[i].depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
        else if ((depthEnd > stList.stItem[i].depthStart) &&
                 (depthEnd <= stList.stItem[i].depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
        else if ((stList.stItem[i].depthStart >= depthStart) &&
                 (stList.stItem[i].depthEnd <= depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
      }
    }

    return (stList);
  }

  /** Method add()
   * <p> This method will add a Texture to an existing list
   * @param  stNew = The Rock Texture Data Structure
   * @param  st    = The Old Rock Texture List Data Structure
   * @return st    = The new Rock Texture List.
   */

  public static textureListStruct add(textureStruct stNew, textureListStruct st)
  {
    int i = 0;
    int    iCount   = 0;
    textureListStruct stTemp = new textureListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new textureStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp.depthStart = st.depthStart;
      stTemp.depthEnd   = st.depthEnd;
      stTemp.iUnit      = st.iUnit;

      if (st.iCount > 0)
        stTemp.stItem = new textureStruct[st.iCount+1];

      for (i=0; i<st.iCount; i++)
      {
        stTemp.stItem[iCount] = new textureStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (stNew != null)
    {
      stTemp.stItem[iCount] = new textureStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new textureListStruct();
    st.iCount = iCount;
    st.stItem = new textureStruct[st.iCount];

    st.depthStart = stTemp.depthStart;
    st.depthEnd   = stTemp.depthEnd;
    st.iUnit      = stTemp.iUnit;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<st.iCount; i++)
    {
      st.stItem[i] = new textureStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Texture in an existing list
   * @param  stNew = The New Rock Texture Data Structure
   * @param  st   = The Old Rock Texture List Data Structure
   * @return st   = The new Rock Texture List.
   */

  public static textureListStruct modify( textureStruct stNew,
                                          textureListStruct st)
  {
    int i = 0;
    int    iCount   = 0;
    textureListStruct stTemp = new textureListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new textureStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp.depthStart = st.depthStart;
      stTemp.depthEnd   = st.depthEnd;
      stTemp.iUnit      = st.iUnit;

      if (st.iCount > 0)
        stTemp.stItem = new textureStruct[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(stNew.sKEY))
        {
          stTemp.stItem[iCount] = new textureStruct();
          stTemp.stItem[iCount] = copy(stNew);
          iCount++;
        }
        else
        {
          stTemp.stItem[iCount] = new textureStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new textureListStruct();
      st.iCount = iCount;
      st.stItem = new textureStruct[st.iCount];

      st.depthStart = stTemp.depthStart;
      st.depthEnd   = stTemp.depthEnd;
      st.iUnit      = stTemp.iUnit;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<st.iCount; i++)
      {
        st.stItem[i] = new textureStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Texture from an existing list
   * @param  sKEY = Unique Identifier for depth range
   * @param  st   = The Old Rock Texture List Data Structure
   * @return st   = The new Rock Texture List.
   */

  public static textureListStruct remove(String sKEY, textureListStruct st)
  {
    int i = 0;
    int    iCount   = 0;
    textureListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount < 2)
      {
        st.iCount = 0;
        st.delete();
        st = null;
      }
      else
      {
        stTemp        = new textureListStruct();
        stTemp.stItem = new textureStruct[st.iCount-1];

        stTemp.depthStart = st.depthStart;
        stTemp.depthEnd   = st.depthEnd;
        stTemp.iUnit      = st.iUnit;

        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i] != null)
          {
            if (!st.stItem[i].sKEY.equals(sKEY))
            {
              stTemp.stItem[iCount] = new textureStruct();
              stTemp.stItem[iCount] = copy(st.stItem[i]);
              iCount++;
            }
          }
        }

        st.delete();
        st = null;

        st        = new textureListStruct();
        st.iCount = iCount;
        st.stItem = new textureStruct[st.iCount];

        st.depthStart = stTemp.depthStart;
        st.depthEnd   = stTemp.depthEnd;
        st.iUnit      = stTemp.iUnit;

        st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
        st.source   = new String( source );    // Source of Data
        st.sCreated = new String( sCreated );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          st.stItem[i] = new textureStruct();
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        if (stTemp != null)
          stTemp.delete();
        stTemp = null;
      }
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static textureListStruct copyList(textureListStruct stOld)
  {
    textureListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new textureListStruct();

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth
      stNew.iUnit      = stOld.iUnit;

      // LAS 3 Dependent Variable

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new textureStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static textureStruct copy(textureStruct stOld)
  {
    int i=0;
    int j=0;
    textureStruct stNew = new textureStruct();

    if (stOld != null)
    {
      stNew.sKEY        = new String(stOld.sKEY);

      stNew.depthStart  = stOld.depthStart; // Starting Depth
      stNew.depthEnd    = stOld.depthEnd;   // Ending Depth

      /* ---------------------------------------------------- *
       * --------------------- ROCK TYPE -------------------- *
       * ---------------------------------------------------- */

      stNew.iType = stOld.iType;
      stNew.sType = new String(stOld.sType);

      /* ---------------------------------------------------- *
       * -------------------- ROCK TEXTURE ------------------ *
       * ---------------------------------------------------- */

      stNew.iEnergy  = stOld.iEnergy;
      stNew.iRGB[0]  = stOld.iRGB[0];
      stNew.iRGB[1]  = stOld.iRGB[1];
      stNew.iRGB[2]  = stOld.iRGB[2];
      stNew.id       = stOld.id;
      stNew.iSymbol  = stOld.iSymbol;

      stNew.iTexture = stOld.iTexture;
      stNew.sTexture = new String(stOld.sTexture);

      /* ---------------------------------------------------- *
       * ------------------ ROCK COMPOSITION ---------------- *
       * ---------------------------------------------------- */

      stNew.iComposition = stOld.iComposition;
      stNew.sComposition = new String(stOld.sComposition);

      /* .................................................... *
       * ............... Siliclastic Variables .............. *
       * .................................................... */

      // Clay / Silt matrix

      stNew.dClay        = stOld.dClay;  // Clay / Fine Silt

      // QFR Content Values

      stNew.dQuartz      = stOld.dQuartz;    // Quartz
      stNew.dFeldspar    = stOld.dFeldspar;  // Feldspar
      stNew.dRock        = stOld.dRock;      // Rock Fragments

      // VMS Content Values

      stNew.dVolcanic    = stOld.dVolcanic;     // Volcanic Rocks
      stNew.dMetamorphic = stOld.dMetamorphic;  // Metamorphic Rocks
      stNew.dSedimentary = stOld.dSedimentary;  // Sedimentary Rocks

      // Sedimentary Rock Content Values

      stNew.dChert       = stOld.dChert;      // Chert
      stNew.dCarbonate   = stOld.dCarbonate;  // Carbonate
      stNew.dSilicate    = stOld.dSilicate;   // Silicate

      /* .................................................... *
       * ....... Carbonate Allochemical Rock Variables ...... *
       * .................................................... */

      // Allochemical Rocks

      stNew.sOther    = new String(stOld.sOther); // Other descriptor names

      // Allochem Type

      stNew.iAlloType = stOld.iAlloType;
      stNew.sAlloType = new String(stOld.sAlloType); // Allochem Type

      // Matrix Type

      stNew.iMatrix   = stOld.iMatrix;
      stNew.sMatrix   = new String(stOld.sMatrix);  // Matrix

      // Prefix

      stNew.sPrefix   = new String(stOld.sPrefix);  // Prefix

      // Percent number of Allochems

      stNew.dIntraclasts = stOld.dIntraclasts; // Intraclasts % Compostion Value
      stNew.dOoids       = stOld.dOoids;       // Ooids % Compostion Value
      stNew.dBioclasts   = stOld.dBioclasts;   // Bioclasts % Compostion Value
      stNew.dPeloids     = stOld.dPeloids;     // Peloids % Compostion Value

      /* ---------------------------------------------------- *
       * --------------------- GRAIN SIZE ------------------- *
       * ---------------------------------------------------- */

      stNew.iData     = stOld.iData;
      stNew.iRows     = stOld.iRows;

      stNew.iDesCols  = stOld.iDesCols;
      stNew.sDesc     = new String[stNew.iRows][stNew.iDesCols];

      stNew.iColumns  = stNew.iColumns;
      stNew.data      = new double[stNew.iRows][stNew.iColumns];

      for (i=0; i<stNew.iRows; i++)
      {
        for (j=0; j<stNew.iDesCols; j++)
        {
          stNew.sDesc[i][j] = new String(stOld.sDesc[i][j]);
        }

        for (j=0; j<stNew.iColumns; j++)
        {
          stNew.data[i][j] = stOld.data[i][j];
        }
      }

      stNew.iClass   = stOld.iClass;
      stNew.dSize    = stOld.dSize;
      stNew.dPhi     = stOld.dPhi;
      stNew.sClass   = new String(stOld.sClass);

      /* .................................................... *
       * .............. Grain Sorting Variables ............. *
       * .................................................... */

      stNew.iSorted  = stOld.iSorted;
      stNew.dSorted  = stOld.dSorted;
      stNew.sorted   = new String(stOld.sorted);

      /* .................................................... *
       * ............. Grain Skewness Variables ............. *
       * .................................................... */

      stNew.iSkewed  = stOld.iSkewed;
      stNew.dSkewed  = stOld.dSkewed;
      stNew.skewed   = new String(stOld.skewed);

      for (i=0; i<lith.texture.textureStruct.TOTAL; i++)
        stNew.iLithology[i] = stOld.iLithology[i];
    }

    return (stNew);
  }

  /** Method computeLASPlotTexture()
   * <p> This method will compute the texture from the digital LAS File.
   * @param  stLAS = The LAS File Data Structure
   * @param  st    = the texture list data structure
   * @param  stSymbols = Lithology Symbols List Data Structure
   * @return st    = the texture list data structure
   */

  public static textureListStruct computeLASPlotTexture(
                   lasFileDataStruct stLAS, textureListStruct st,
                   lithologySymbolsListStruct stSymbols)
  {
    int    i           = 0;
    int    k           = 0;
    int    m           = 0;
    int    n           = 0;
    int    p           = 0;
    int    iLithology  = 0;
    int    iLith       = 0;
    int    iCell       = 0;
    double depth1      = 0.0;
    double depth2      = 0.0;
    double dLith[]     = {0.0, 0.0, 0.0, 0.0, 0.0};
    double dSum        = 0.0;
    double dShaly      = iqstrat.iqstratShaleStruct.SHALY;
    double dShale      = iqstrat.iqstratShaleStruct.SHALE;
    double dHotShale   = iqstrat.iqstratShaleStruct.SHALE_HOT;

    int    len         = 0;
    int    iMin[]      = null;
    int    iMin_Lith[] = null;
    int    iMin_Col[]  = null;
    double dIN[]       = null;
    double dMat[][]    = null;

    if (stLAS != null)
    {
      if (stLAS.depthStep == 2.0)
      {
        st = buildNew(iqstrat.iqstratTracksStruct._SCALE_2_FOOT,
                      stLAS.depthStart, stLAS.depthEnd);
      }
      else if (stLAS.depthStep == 1.0)
      {
        st = buildNew(iqstrat.iqstratTracksStruct._SCALE_1_FOOT,
                      stLAS.depthStart, stLAS.depthEnd);
      }

      for (i=0; i<stLAS.iRows-1; i++)
      {
        depth1 = stLAS.depths[i];
        depth2 = stLAS.depths[i+1];

        if (st != null)
        {
          for (k=0; k<st.iCount; k++)
          {
            if (st.stItem[k].depthStart == depth1)
              iLith = k;
            else if (( st.stItem[k].depthStart > depth1 ) &&
                     ( st.stItem[k].depthStart < depth2 ))
              iLith = k;
          }
        }

        double dGamma = stLAS.dGR[i];

        if (dGamma > stLAS.dNull)
        {
          if (dGamma > GAMMA_MAX)
            dGamma = GAMMA_MAX;
          if (dGamma < GAMMA_MIN)
            dGamma = GAMMA_MIN;

          dShaly    = iqstrat.iqstratShaleStruct.SHALY;
          dShale    = iqstrat.iqstratShaleStruct.SHALE;
          dHotShale = iqstrat.iqstratShaleStruct.SHALE_HOT;

          if (stLAS.stList != null)
          {
            for (n=0; n<stLAS.stList.iCount; n++)
            {
              if ((depth1 >= stLAS.stList.stItem[n].depthStart) &&
                  (depth1 <= stLAS.stList.stItem[n].depthEnd))
              {
                dShaly    = stLAS.stList.stItem[n].dShaly;
                dShale    = stLAS.stList.stItem[n].dShale;
                dHotShale = stLAS.stList.stItem[n].dHotShale;
              }
            }
          }

          iLithology = lith.math.lithMathRules.getLithology(
                       math.mathLAS._FRESH,
                       dShaly,
                       dShale,
                       dHotShale,
                       dGamma,
                       stLAS.dNPHI[i],
                       stLAS.dRHOB[i],
                       stLAS.dPE[i],
                       stLAS.dDT[i],
                       stLAS.dNull);

          if ((iLithology == lith.lithology.lithologySymbolsStruct._DOLOMITE)   ||
              (iLithology == lith.lithology.lithologySymbolsStruct._LIMESTONE)  ||
              (iLithology == lith.lithology.lithologySymbolsStruct._SANDSTONE)  ||
              (iLithology == lith.lithology.lithologySymbolsStruct._ARKOSE)     ||
              (iLithology == lith.lithology.lithologySymbolsStruct._ANHYDRITE)  ||
              (iLithology == lith.lithology.lithologySymbolsStruct._GYPSUM)     ||
//              (iLithology == lith.lithology.lithologySymbolsStruct._HALITE)     ||
//              (iLithology == lith.lithology.lithologySymbolsStruct._COAL)       ||
              (iLithology == lith.lithology.lithologySymbolsStruct._SHALY_CARB) ||
              (iLithology == lith.lithology.lithologySymbolsStruct._SHALY_DOLO) ||
              (iLithology == lith.lithology.lithologySymbolsStruct._SHALY_SAND) ||
//              (iLithology == lith.lithology.lithologySymbolsStruct._SHALE_HOT)  ||
              (iLithology == lith.lithology.lithologySymbolsStruct._SHALE))
          {

            if (iLithology == lith.lithology.lithologySymbolsStruct._ANHYDRITE)
            {
              iMin      = lith.math.lithMath._ANHY;
              iMin_Lith = lith.math.lithMath._ANHY_LITH;
            }
            else if (iLithology == lith.lithology.lithologySymbolsStruct._GYPSUM)
            {
              iMin      = lith.math.lithMath._GYP;
              iMin_Lith = lith.math.lithMath._GYP_LITH;
            }
            else
            {
              iMin      = lith.math.lithMath._RU;
              iMin_Lith = lith.math.lithMath._RU_LITH;
            }

            if (stLAS.iPE != -1)
            {
              iMin_Col  = lith.math.lithMath._RU_COL;

              dIN = lith.math.lithMath.getRU( lith.math.lithMath._CALCITE,
                                              lith.math.lithMath._FRESH_H2O,
                                              stLAS.dRHOB[i],
                                              stLAS.dPE[i],
                                              stLAS.dNPHI[i]);
            }
            else
            {
              iMin_Col  = lith.math.lithMath._PD_COL;

              dIN = lith.math.lithMath.getPD( lith.math.lithMath._CALCITE,
                                              lith.math.lithMath._FRESH_H2O,
                                              stLAS.dRHOB[i],
                                              stLAS.dNPHI[i]);
            }

            for (n=0; n<3; n++) dLith[0] = 0.0;

            dMat  = lith.math.lithMath.getMatrix(iMin, iMin_Col);
            dLith = lith.math.lithMath.composition( dIN, dMat );

            dSum = 0.0;
            for (n=0; n<3; n++)
            {
              if (dLith[n] < 0.0)
                dLith[n] = 0.0;

              dSum = dSum + dLith[n];
            }

            int iDolomite  = (int) (10.0 * dLith[0] / dSum);
            int iLimestone = (int) (10.0 * dLith[1] / dSum);
            int iSand      = (int) (10.0 * dLith[2] / dSum);

            if (dGamma >= dShale)
            {
              if ((iLithology == lith.lithology.lithologySymbolsStruct._SHALE) ||
                  (iLithology == lith.lithology.lithologySymbolsStruct._SHALY_SAND))
              {
                st.stItem[iLith].iEnergy = 2;
                st.stItem[iLith].iRGB =
                    lith.texture.textureConstants.SIO2_COLOR[1];
                st.stItem[iLith].id =
                    lith.texture.textureConstants.SIO2_TEXTURE[1][0];

                st.stItem[iLith].iType =
                    lith.texture.textureConstants.SANDSTONE;
                st.stItem[iLith].iTexture = lith.texture.textureConstants._CLAY;
              }

              if (iLithology == lith.lithology.lithologySymbolsStruct._ARKOSE)
              {

                st.stItem[iLith].iEnergy = 6;
                st.stItem[iLith].iRGB =
                    lith.texture.textureConstants.SIO2_COLOR[7];
                st.stItem[iLith].id =
                    lith.lithology.lithologySymbolsStruct._ARKOSE;

                st.stItem[iLith].iType =
                    lith.texture.textureConstants.SANDSTONE;
                st.stItem[iLith].iTexture =
                    lith.texture.textureConstants._MEDIUM;
              }
            }
            else if (dGamma < dShale)
            {
              if ((iLithology != lith.lithology.lithologySymbolsStruct._HALITE) ||
                  (iLithology != lith.lithology.lithologySymbolsStruct._COAL))
              {
                if ((iLithology == lith.lithology.lithologySymbolsStruct._ANHYDRITE) ||
                    (iLithology == lith.lithology.lithologySymbolsStruct._GYPSUM))
                  st.stItem[iLith].iType =
                      lith.texture.textureConstants.EVAPORITE;
                else
                  st.stItem[iLith].iType =
                      lith.texture.textureConstants.SANDSTONE;

                if (iDolomite+iLimestone > iSand)
                {
                  st.stItem[iLith].iType =
                      lith.texture.textureConstants.CARBONATE;
                }

                iCell = (int) (10.0 * (dShale - dGamma) / dShale);

                switch (st.stItem[iLith].iType)
                {
                  case lith.texture.textureConstants.SANDSTONE:
                    for (m=lith.texture.textureTypesStruct.SiO2_COUNT; m>0; m--)
                    {
                      if (iCell <= lith.texture.textureConstants.SiO2_CELLS[m])
                      {
                        st.stItem[iLith].iTexture = m;
                      }
                    }

                    iCell = lith.texture.textureConstants.SiO2_CELLS[
                              st.stItem[iLith].iTexture];
                    st.stItem[iLith].iEnergy = iCell;
                    st.stItem[iLith].iRGB =
                      lith.texture.textureConstants.SIO2_COLOR[iCell-1];
                    st.stItem[iLith].id =
                      lith.texture.textureConstants.SIO2_TEXTURE[iCell-1][0];

                    break;

                  case lith.texture.textureConstants.CARBONATE:
                    for (m=lith.texture.textureTypesStruct.CaCO3_COUNT; m>-1; m--)
                    {
                      if (iCell <= lith.texture.textureConstants.CaCO3_CELLS[m])
                      {
                        st.stItem[iLith].iTexture = m;
                      }
                    }

                    iCell = lith.texture.textureConstants.CaCO3_CELLS[
                              st.stItem[iLith].iTexture];
                    st.stItem[iLith].iEnergy = iCell;
                    if (iLimestone >= iDolomite)
                    {
                      st.stItem[iLith].iRGB =
                        lith.texture.textureConstants.LIMESTONE_COLOR[iCell-1];
                      st.stItem[iLith].id =
                        lith.texture.textureConstants.LIMESTONE_TEXTURE[iCell-1][0];
                    }
                    else
                    {
                      st.stItem[iLith].iRGB =
                        lith.texture.textureConstants.DOLOMITE_COLOR[iCell-1];
                      st.stItem[iLith].id =
                        lith.texture.textureConstants.DOLOMITE_TEXTURE[iCell-1][0];
                    }

                    break;

                  case lith.texture.textureConstants.EVAPORITE:
                    st.stItem[iLith].id = iLithology;
                    st.stItem[iLith].iEnergy = iCell+1;

                    st.stItem[iLith].iType =
                        lith.texture.textureConstants.EVAPORITE;
                    st.stItem[iLith].iTexture =
                        lith.texture.textureConstants.EVAPORITE;
                    break;
                }
              }
            }

            for (p=0; p<stSymbols.iCount; p++)
            {
              if (st.stItem[iLith].id == stSymbols.stItem[p].id)
              {
                st.stItem[iLith].iSymbol = p;

                if ((st.stItem[iLith].id ==
                      lith.lithology.lithologySymbolsStruct._ARKOSE) ||
                    (st.stItem[iLith].id ==
                      lith.lithology.lithologySymbolsStruct._ANHYDRITE) ||
                    (st.stItem[iLith].id ==
                      lith.lithology.lithologySymbolsStruct._GYPSUM))
                {
                  st.stItem[iLith].iSymbol = p;
                  st.stItem[iLith].iRGB = stSymbols.stItem[p].iRGB;
                }
              }
            }
          }
          else
          {
            iCell = (int) (10.0 * (dShale - dGamma) / dShale);
            st.stItem[iLith].id = iLithology;
            st.stItem[iLith].iEnergy = iCell+1;

            st.stItem[iLith].iType = lith.texture.textureConstants.OTHER;

            if (iLithology == lith.lithology.lithologySymbolsStruct._HALITE)
            {
              st.stItem[iLith].iType =
                  lith.texture.textureConstants.EVAPORITE;
              st.stItem[iLith].iTexture =
                  lith.texture.textureConstants.EVAPORITE;
            }
            else if (iLithology == lith.lithology.lithologySymbolsStruct._COAL)
            {
              st.stItem[iLith].iType =
                  lith.texture.textureConstants.ORGANIC;
              st.stItem[iLith].iTexture =
                  lith.texture.textureConstants.ORGANIC;
            }

            for (p=0; p<stSymbols.iCount; p++)
            {
              if (st.stItem[iLith].id == stSymbols.stItem[p].id)
              {
                st.stItem[iLith].iSymbol = p;
                st.stItem[iLith].iRGB    = stSymbols.stItem[p].iRGB;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method setLegend( textureListStruct stList )
   * <p> This method will set the legend based on the lithologies selected
   *     along with the default legend
   * @param  stList = Lithology List Data Structure
   * @return st     = lithology symbol data list Structure
   */

  public static textureListStruct setLegend( textureListStruct stList )
  {
    textureListStruct st     = null;
    int               iFound = -1;

    if (stList != null)
    {
      if (stList.iCount > 0)
      {
        st = new textureListStruct();

        for (int i=0; i<stList.iCount; i++)
        {
          iFound = -1;
          for (int j=0; j<st.iCount; j++)
          {
            if (stList.stItem[i].sTexture.equals(st.stItem[j].sTexture))
            {
              iFound = 1;
            }
          }

          if (iFound == -1)
          {
            st = add( stList.stItem[i], st );
          }
        }
      }
    }

    return (st);
  }

  /** Method legendSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static textureListStruct legendSort(textureListStruct st)
  {
    boolean swappedOnPrevRun = true;
    textureStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if((10 * st.stItem[i].iType   + st.stItem[i].iEnergy) >
               (10 * st.stItem[i+1].iType + st.stItem[i+1].iEnergy))
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static textureListStruct bubbleSort(textureListStruct st)
  {
    boolean swappedOnPrevRun = true;
    textureStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method getOrder()
   * <p> This method will determine the order of the Allochems from percentages
   * @param  dIntra = % Intraclasts
   * @param  dOO    = % Oolites
   * @param  dBio   = % Fossils
   * @param  dPel   = % Pellets
   * @return data   = allochems ordered by percent
   */

  public static double[][] getOrder( double dIntra, double dOO,
                                     double dBio,   double dPel )
  {
    double data[][] = { { 0.0, dIntra },   // Intraclasts
                        { 1.0, dOO },      // Oolites
                        { 2.0, dBio },     // Fossils
                        { 3.0, dPel }};    // Pellets

    data = Sort(4, 2, 1, data);

    return (data);
  }

  /** Method Sort()
   * <p> This method will sort in ascending percent order (lowest to highest)
   * @param  iRows = the total number of rows
   * @param  iCols = the toal number of columns
   * @param  iSort = Sort on Column
   * @param  data  = data list of percentage allochems
   * @return data  = data list of percentage allochems ordered by percent
   */

  public static double[][] Sort(int iRows, int iCols, int iSort, double data[][])
  {
    int j=0;
    boolean swappedOnPrevRun = true;
    double dTemp[]           = null;

    dTemp = new double[iCols];
    for (j=0; j<iCols; j++) dTemp[j] = 0.0;

    if (data != null)
    {
      while(swappedOnPrevRun)
      {
        // this variable keeps track of whether to continue sorting or exit

        swappedOnPrevRun = false;

        // loop through every element in the array, except for the last one

        for(int i=0; i<iRows-1; i++)
        {
          // if current element is greater than the next swap the two elements

          if(data[i][iSort] > data[i+1][iSort])
          {
            // we don't want the loop to end just yet, we're not done

            swappedOnPrevRun = true;

            // store element i in a temporary variable

            for (j=0; j<iCols; j++) dTemp[j] = data[i][j];

            // set element i+1 to where i used to be

            for (j=0; j<iCols; j++) data[i][j] = data[i+1][j];

            // release the old i from temp into i+1 slot

            for (j=0; j<iCols; j++) data[i+1][j] = dTemp[j];
          }
        }
      }
    }

    return (data);
  }

  /** Method print()
   * <p> This method will print the data structure
   * @param st = The Rock Column Litholog List Data Structure.
   */

  public static void print(textureListStruct st)
  {
    System.out.println("-------------------------------------------------");
    if (st != null)
    {
      System.out.println("TEXTURE: "+st.iUnit+" "+st.depthStart +" "+st.depthEnd);

      for (int k=0; k<st.iCount; k++)
      {
        if (st.stItem[k] != null)
        {
          System.out.print(k + " " + st.stItem[k].sKEY       + " " +
                                     st.stItem[k].depthStart + " " +
                                     st.stItem[k].depthEnd   + " " +
                                     st.stItem[k].iEnergy    + " " +
                                     st.stItem[k].sType      + " " +
                                     st.stItem[k].sTexture);
          System.out.println("");
        }
      }
    }
    System.out.println("-------------------------------------------------");
  }
}