/*
 * @textureTypesStruct.java Version 1.1 07/30/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.texture;

/** Class textureTypesStruct
 *  <p> This Class holds the Texture Data.
 *
 *  @version 1.1 07/30/2008
 *  @author  John R. Victorine
 */

public class textureTypesStruct
{
  // Type of sediment

  public static final int  NONE      = -1;
  public static final int  SANDSTONE = 0;
  public static final int  CARBONATE = 1;

  // Rock Texture

  public static final int _TEXTURE_NONE = -1;

  /* =================================================================== *
   * --------------------- SILICLASTIC ROCK TEXTURE -------------------- *
   * =================================================================== */

  public static final int _CLAY        = 0;
  public static final int _SILT        = 1;
  public static final int _VERY_FINE   = 2;
  public static final int _FINE        = 3;
  public static final int _MEDIUM      = 4;
  public static final int _COARSE      = 5;
  public static final int _VERY_COARSE = 6;
  public static final int _GRANULE     = 7;
  public static final int _PEBBLE      = 8;
  public static final int _COBBLE      = 9;
  public static final int _BOULDER     = 10;
  public static final int _TOTAL_SILICLASTIC = 11;

  public static final String SiO2[][] =
  {
  //  Type       Size Class       Abbreviation
  { "Mud",    "Clay",               "cl",   "< 1/256" },
  { "Mud",    "Silt",               "slt",  "1/256 - 1/16 mm" },
  { "Sand",   "Sand - Very Fine",   "*f",   "1/16 - 1/8 mm" },
  { "Sand",   "Sand - Fine",        "f",    "1/8 - 1/4 mm" },
  { "Sand",   "Sand - Medium",      "m",    "1/4 - 1/2 mm" },
  { "Sand",   "Sand - Coarse",      "crs",  "1/2 - 1 mm" } ,
  { "Sand",   "Sand - Very Coarse", "*crs", "1 - 2 mm" },
  { "Gravel", "Granules",           "gran", "2 - 4 mm" },
  { "Gravel", "Pebbles",            "pbl",  "4 - 64 mm" },
  { "Gravel", "Cobbles",            "cbl",  "64 - 256 mm" },
  { "Gravel", "Boulders",           "bld",  "> 256 mm" }
  };

  public static final double SIZE[][] =
  {
    {   0.0,       0.0039065 }, // < 1/256
    {   0.0039065, 0.0625 },    //   1/256 - 1/16 mm
    {   0.0625,    0.125 },     //   1/16 - 1/8 mm
    {   0.125,     0.25 },      //   1/8 - 1/4 mm
    {   0.25,      0.5 },       //   1/4 - 1/2 mm
    {   0.5,       1.0 },       //   1/2 - 1 mm
    {   1.0,       2.0 },       //  1 -   2 mm
    {   2.0,       4.0 },       //  2 -   4 mm
    {   4.0,      64.0 },       //  4 -  64 mm
    {  64.0,     256.0 },       // 64 - 256 mm
    { 256.0,   99999.0 },       // > 256 mm
  };

  //                                     cly slt vf f  m  c vc  gra peb cob bou
  public static final int SiO2_CELLS[] = {2,  4, 5, 6, 7, 8, 9, 10, 12, 14, 16 };
  public static final int SiO2_COUNT   = 9;

  /* =================================================================== *
   * ---------------------- CARBONATE ROCK TEXTURE --------------------- *
   * =================================================================== */

  public static final int _MUDSTONE    = 0;
  public static final int _WACKESTONE  = 1;
  public static final int _PACKSTONE   = 2;
  public static final int _GRAINSTONE  = 3;
  public static final int _BOUNDSTONE  = 4;
  public static final int _CRYSTALLINE = 5;
  public static final int _FLOATSTONE  = 6;
  public static final int _RUDSTONE_M  = 7;
  public static final int _RUDSTONE_S  = 8;
  public static final int _FRAMESTONE  = 9;
  public static final int _BINDSTONE   = 10;
  public static final int _BAFFLESTONE = 11;
  public static final int _TOTAL_CARBONATE = 12;

  public static final String CaCO3[][] =
  {
  //  Type                Size Class                Abbreviation
    { "Mud Supported",    "Mudstone",                "mdst"   },
    { "Mud Supported",    "Wackestone",              "wkst"   },
    { "Grain Supported",  "Packstone",               "pkst"   },
    { "Grain Supported",  "Grainstone",              "grst"   },
    { "Bound",            "Boundstone",              "bdst"   },
    { "Diagenetic",       "Crystalline",             "xln"    },
    { "Mud Supported",    "Floatstone",              "flst"   },
    { "Grain Supported",  "Rudstone (contains mud)", "rdmst"  },
    { "Grain Supported",  "Rudstone (lacks mud)",    "rdgst"  },
    { "Bound",            "Framestone",              "fmst"   },
    { "Bound",            "Bindstone",               "bnst"   },
    { "Bound",            "Bafflestone",             "bfst"   },
  };

  public static final int CaCO3_CELLS[] =
      { 3, 5, 7, 9, 12, 13,  5, 7, 9, 12, 12, 12 };
  public static final int CaCO3_COUNT   = 5;

  /* =================================================================== *
   * ------------------- SILICLASTIC ROCK COMPOSITION ------------------ *
   * =================================================================== */

  // Symbol ID from Symbols XML File

  public static final int QUARTZ_ARENITE          = 0;
  public static final int SUBARKOSE               = 1;
  public static final int SUBLITHARENITE          = 2;

  public static final int ARKOSE                  = 3;
  public static final int LITHIC_ARENITE          = 4;
  public static final int FELDSPATHIC_LITHARENITE = 5;
  public static final int LITHARENITE             = 6;

  public static final int VOLCANIC_ARENITE        = 7;
  public static final int PHYLLARENITE            = 8;
  public static final int SEDARENITE              = 9;

  public static final int CHERT_ARENITE           = 10;
  public static final int CALCLITHITE             = 11;
  public static final int SANDSTONE_ARENITE       = 12;

  public static final int QUARTZ_WACKE            = 13;
  public static final int FELDSPATHIC_GRAYWACKE   = 14;
  public static final int LITHIC_GRAYWACKE        = 15;

  public static final int MUDSTONE                = 16;

  public static final int _ROCK_CLAN_TOTAL        = 17;

  public static final String ROCK_CLAN[] =
  {
    // ARENITE ROCKS < 15 % Clay Matrix
    // QFR Rock Triangle
    "Quartzarenite",
    "Subarkose",
    "Sublitharenite",
    "Arkose",
    "Lithic arenite",
    "Feldspathic Litharenite",
    "Litharenite",
    // VMS Rock Triangle
    "Volcanic arenite",
    "Phyllarenite",
    "Sedarenite",
    // Sedimentary Rock Triangle
    "Chert-arenite",
    "Calclithite",
    "Sandstone-arenite",
    // WACKE ROCKS 15 to 75 % Clay Matrix
    "Quartz wacke",
    "Feldspathic graywacke",
    "lithic graywacke",
    // MUDSTONES  > 75 % Clay Matrix
    "mudstone"
  };

  /* =================================================================== *
   * ------------------------ ALLOCHEMICAL ROCKS ----------------------- *
   * =================================================================== */

  // Symbol ID from Symbols XML File

  public static final int _C_TYPE_I     = 11;
  public static final int _C_TYPE_II    = 12;
  public static final int _C_TYPE_III   = 13;
  public static final int _C_TYPE_IIIa  = 28;
  public static final int _C_TYPE_IV    = 27;

  public static final int _INTRA_I    = 15;
  public static final int _OOID_I     = 16;
  public static final int _BIO_I      = 17;
  public static final int _PELOID_I   = 18;

  public static final int _INTRA_II   = 19;
  public static final int _OOID_II    = 20;
  public static final int _BIO_II     = 21;
  public static final int _PELOID_II  = 22;

  public static final int _INTRA_III  = 23;
  public static final int _OOID_III   = 24;
  public static final int _BIO_III    = 25;
  public static final int _PELOID_III = 26;

  // Allochemical Rock Type

  public static final int   _TYPE_NONE = -1;
  public static final int   _TYPE_I    = 0;
  public static final int   _TYPE_II   = 1;
  public static final int   _TYPE_III  = 2;
  public static final int   _TYPE_IIIa = 3;
  public static final int   _TYPE_IV   = 4;

  public static final String ALLOCHEM_ROCK[] = { "I", "II", "III", "III", "IV" };

  // Allochemical Particles

  public static final int   _ALLO_NONE  = -1;
  public static final int   _ALLO_INTRA = 0;
  public static final int   _ALLO_OO    = 1;
  public static final int   _ALLO_BIO   = 2;
  public static final int   _ALLO_PEL   = 3;

  public static final String ALLOCHEMS[] = { "intra", "oo", "bio", "pel" };

  // Allochemical Matrix

  public static final int   _MATRIX_NONE       = -1;
  public static final int   _MATRIX_SPARITE    = 0;
  public static final int   _MATRIX_MICRITE    = 1;
  public static final int   _MATRIX_DISMICRITE = 2;
  public static final int   _MATRIX_BIOLITHE   = 3;

  public static final String ALLOCHEM_MATRIX[] =
      { "sparite", "micrite", "dismicrite", "biolithite" };

  // Allochemical Names

  public static final String ALLOCHEM_NAMES[][] =
  {
  //     Type I,         Type II,      Type III  (1-10% Allochems)
    { "Intrasparite", "Intramicrite", "Intraclast bearing micrite" }, // Intra
    { "Oosparite",    "Oomicrite",    "Oolite bearing micrite" },     // oo
    { "Biosparite",   "Biomicrite",   "Fossiliferous micrite" },      // bio
    { "Pelsparite",   "Pelmicrite",   "Pelletiferous micrite" }       // pel
  };

  /* =================================================================== *
   * -------------------------- ROCK GRAIN SIZE ------------------------ *
   * =================================================================== */

  // Sediment sorting

  public static final int _SORTED_NONE            = -1;
  public static final int _VERY_WELL_SORTED       = 0;
  public static final int _WELL_SORTED            = 1;
  public static final int _MODERATELY_WELL_SORTED = 2;
  public static final int _MODERATELY_SORTED      = 3;
  public static final int _POORLY_SORTED          = 4;
  public static final int _VERY_POORLY_SORTED     = 5;
  public static final int _EXTEMELY_POORLY_SORTED = 6;
  public static final int _TOTAL_SORTED           = 7;

  public static final double SORTING[][] = {
    { 0.0,  0.35 },  // very well sorted
    { 0.35, 0.5 },   // well sorted
    { 0.5,  0.71 },  // moderately well sorted
    { 0.71, 1.0 },   // moderately sorted
    { 1.0,  2.0 },   // poorly sorted
    { 2.0,  4.0 },   // very poorly sorted
    { 4.0,  99999.0 } // extremely poorly sorted
  };

  public static final String SORTING_TEXT[] = {
    "very well sorted",
    "well sorted",
    "moderately well sorted",
    "moderately sorted",
    "poorly sorted",
    "very poorly sorted",
    "extremely poorly sorted"
  };

  // Sediment Skewness

  public static final int _SKEWNESS_NONE          = -1;
  public static final int _STRONGLY_FINE_SKEWED   = 0;
  public static final int _FINE_SKEWED            = 1;
  public static final int _NEAR_SYMMETRICAL       = 2;
  public static final int _COARSE_SKEWED          = 3;
  public static final int _STRONGLY_COARSE_SKEWED = 4;
  public static final int _TOTAL_SKEWED           = 5;

  public static final double SKEWNESS[][] = {
    {  0.3,   1.0 },  // strongly fine skewed
    {  0.1,   0.3 },  // fine skewed
    { -0.1,   0.1 },  // near symmetrical
    { -0.3,  -0.1 },  // coarse skewed
    { -1.0,  -0.3 },  // strongly coarse skewed
  };

  public static final String SKEWNESS_TEXT[] = {
    "strongly fine skewed",
    "fine skewed",
    "near symmetrical",
    "coarse skewed",
    "strongly coarse skewed"
  };

  /* =================================================================== *
   * ---------------------------- GET METHODS -------------------------- *
   * =================================================================== */

  /** Method getGrainSize(double dPHI)
   * <p> This method will return the grain size identifier depending on the
   *     grains PHI Value
   * @param  dSize  = The average size of the grain
   * @return ivalue = grain size identifier
   */

  public static int getGrainSize(double dSize)
  {
    int ivalue = _TEXTURE_NONE;

    if (dSize != 9999.0)
    {
      for (int i=0; i<lith.texture.textureTypesStruct._TOTAL_SILICLASTIC; i++)
      {
        if ((dSize >= lith.texture.textureTypesStruct.SIZE[i][0]) &&
            (dSize < lith.texture.textureTypesStruct.SIZE[i][1]))
        {
          ivalue = i;
        }
      }
    }

    return (ivalue);
  }

  /** Method getSorting()
   * <p> This method will use the Inclusive Graphic Standard Deviation
   *     to retrive the sorting identifier value according to Folk (pg 42).
   *
   *     It was computed by the following formula:
   *
   *       sigma = [ Phi(@84%) - Phi(@16%) ]/4 - [ Phi(@95%) - Phi(@5%) ]/6.6
   *
   *     "The standard deviation here is a measure of the spread of PHI units
   *      of the sample, therefore the symbol PHI must always be attached to
   *      the value for sigma"
   *
   *     sigma under 0.35 PHI         - very well sorted
   *                 0.35 to 0.50 PHI - well sorted
   *                 0.50 to 0.71 PHI - moderately well sorted
   *                 0.71 to 1.0 PHI  - moderately sorted
   *                 1.0  to 2.0 PHI  - poorly sorted
   *                 2.0  to 4.0 PHI  - very poorly sorted
   *           over  4.0 PHI          - extremely pooly sorted
   *
   * @param  dSigma = sigma in Phi Units
   * @return ivalue = sorting identifier
   */

  public static int getSorting(double dSigma)
  {
    int ivalue = _SKEWNESS_NONE;

    if (dSigma != 9999.0)
    {
      for (int i=0; i<lith.texture.textureTypesStruct._TOTAL_SORTED; i++)
      {
        if ((dSigma >= lith.texture.textureTypesStruct.SORTING[i][0]) &&
            (dSigma < lith.texture.textureTypesStruct.SORTING[i][1]))
        {
          ivalue = i;
        }
      }
    }

    return (ivalue);
  }

  /** Method getSkewness()
   * <p> This method will use the Inclusive Graphic Skewness to retrieve the
   *     skewness identifier value according to Folk (pg 44).
   *
   *     It was computed by the following formula:
   *
   *     Sk = [Phi(@16%) + Phi(@84%) - 2*Phi(@50%)]/2*[Phi(@84%)-Phi(@16%)] +
   *          [Phi(@5%)  + Phi(@95%) - 2*Phi(@50%)]/2*[Phi(@95%)-Phi(@5%)]
   *
   *     "The method includes 90% of the curve"
   *
   *     skewness    1.00 to  0.30     - strongly fine-skewed
   *                 0.30 to  0.10     - fine-skewed
   *                 0.10 to -0.10     - near-symmetrical
   *                -0.10 to -0.30     - coarse-skewed
   *                -0.30 to -1.00     - strongly coarse-skewed
   *
   *      NOTE: This method uses interperpolation to determine the values at
   *            the %PHI Values.
   *
   * @param  dSk    = Graphic Skewness.
   * @return ivalue = skewness identifier
   */

  public static int getSkewness(double dSk)
  {
    int ivalue = _SKEWNESS_NONE;

    if (dSk != 9999.0)
    {
      for (int i=0; i<lith.texture.textureTypesStruct._TOTAL_SKEWED; i++)
      {
        if ((dSk >= lith.texture.textureTypesStruct.SKEWNESS[i][0]) &&
            (dSk < lith.texture.textureTypesStruct.SKEWNESS[i][1]))
        {
          ivalue = i;
        }
      }
    }

    return (ivalue);
  }

  /* =================================================================== *
   * ---------------------------- OLD METHODS -------------------------- *
   * =================================================================== */

  // Rock Texture

  public static final int _NO_TEXTURE  = 0;
  public static final int _SiO2_GRAVEL = 1;
  public static final int _SiO2_COARSE = 2;
  public static final int _SiO2_MEDIUM = 3;
  public static final int _SiO2_FINE   = 4;
  public static final int _SiO2_SILT   = 5;
  public static final int _SiO2_SHALE  = 6;
  public static final int _CaCO3_C     = 7;
  public static final int _CaCO3_B     = 8;
  public static final int _CaCO3_G     = 9;
  public static final int _CaCO3_P     = 10;
  public static final int _CaCO3_W     = 11;
  public static final int _CaCO3_M     = 12;

  public static final String NAME[][] =
  {
    // Mnemonic       Name          Abbreviation
    { "NO_TEXTURE",  "No Texture",  "N/A"   },
    { "SiO2_GRAVEL", "Sand-Gravel", "s grv" },
    { "SiO2_COARSE", "Sand-Course", "s crs" },
    { "SiO2_MEDIUM", "Sand-Medium", "s med" },
    { "SiO2_FINE",   "Sand-Fine",   "s fn"  },
    { "SiO2_SILT",   "Silt",        "slt"   },
    { "SiO2_SHALE",  "Shale",       "sh"    },
    { "CaCO3_C",     "Crystaline",  "xln"   },
    { "CaCO3_B",     "Boundstone",  "bdst"  },
    { "CaCO3_G",     "Grainstone",  "grst"  },
    { "CaCO3_P",     "Packstone",   "pkst"  },
    { "CaCO3_W",     "Wackestone",  "wkst"  },
    { "CaCO3_M",     "Mudstone",    "mdst"  }
  };

  public static final int COLUMN[] = { 0, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1 };
}