/*
 * @texturePlotTrack.java Version 1.1 09/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.texture.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

import las.lasFileDataStruct;

import lith.texture.textureListStruct;
import lith.texture.textureStruct;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.plot.lithologyPlotSymbol;

/** CLASS texturePlotTrack
 *  <p> This Class will create the Colorlith Plots.
 *
 *  @version 1.1 09/05/2007
 *  @author  John R. Victorine
 */

public class texturePlotTrack extends Canvas
{
  // Input Variables

  private int     iUnits     = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
  private int     iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double  depthStart = 0.0;   // Starting Depth
  private double  depthEnd   = 0.0;   // Ending Depth
  private int     iScale     = -1;    // Plot Scale

  // Global Variables

  public static final double GAMMA_MAX  = 150.0;
  public static final double GAMMA_MIN  = 0.0;
  public static final double CLEAN      = 60.0;

  private lasFileDataStruct stLASFileData = null; // LAS File Data

  private double dStart    = 0.0;
  private double dEnd      = 0.0;

  private textureListStruct   stLASTexture = null;
//  private textureListStruct   stTexture    = null;

  private lithologySymbolsListStruct stLithology = null;
  private lithologyPlotSymbol        plotLith    = null;

  private int iFocus = 0;
  private int iLASTexture = iqstrat.iqstratTracksStruct._LAS_TEXTURE;
//  private int iTexture    = iqstrat.iqstratTracksStruct._ROCK_TEXTURE;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR texturePlotTrack( int iDataType,
   *                                double depthStart,
   *                                double depthEnd,
   *                                int iScale,
   *                                int iUnits )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param iUnits      = The scale factor for building texture cells
   *  @param stLithology = Texture Symbols List Data Structure
   */

  public texturePlotTrack( int iDataType,
                           double depthStart,
                           double depthEnd,
                           int iScale,
                           int iUnits,
                           lithologySymbolsListStruct stLithology )
  {
    this.iDataType   = iDataType;
    this.iUnits      = iUnits;

    this.stLithology = stLithology;
    plotLith = new lithologyPlotSymbol( stLithology );

    setPlotHeight(iScale, depthEnd, depthStart);

    builidDataStructure();

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stLASFileData = null;
    stLithology   = null;

    if (plotLith != null)
      plotLith.close();
    plotLith = null;

    if (stLASTexture != null)
      stLASTexture.delete();
    stLASTexture = null;
//    stTexture    = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getTexture()
   * <p> This method will get the rock column data
   * @return stTexture = LAS Texture Column determined by Gamma Ray.
   */

  public textureListStruct getTexture() { return (stLASTexture); }

  /** Method getRockTexture()
   * <p> This method will get the rock column data
   * @return stTexture = Rock Texture Column determined by user from rock data.
   *

  public textureListStruct getRockTexture() { return (stTexture); }
*/
  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setLASFileData( lasFileDataStruct stLASFileData )
   * <p> This method will set the LAS File Data Structure
   * @param stLASFileData - LAS File Data Structure
   */

  public void setLASFileData( lasFileDataStruct stLASFileData )
  {
    builidDataStructure();

    this.stLASFileData = stLASFileData;
    stLASTexture = lith.texture.textureUtility.computeLASPlotTexture(
        stLASFileData, stLASTexture, stLithology);
    stLASTexture = lith.texture.textureUtility.fillEmptySpaces(stLASTexture);
  }

  /** Method setRockTexture( textureListStruct stTexture )
   * <p> This method will set the rock texture that will be shown
   * @param stTexture = Rock texture determined by user from rock data.
   *

  public void setRockTexture( textureListStruct stTexture )
  {
    this.stTexture = stTexture;
  }
*/
  /* =============================================================== *
   * --------------------------- Compute --------------------------- *
   * =============================================================== */

  /** Method buildDataStructure()
   * <p> This method will allocate space for the depth range that is passed in
   *     the total number of records will be based on 2 * the depth difference
   */

  private void builidDataStructure()
  {
    if (stLASTexture != null)
      stLASTexture.delete();
    stLASTexture = null;

    stLASTexture = lith.texture.textureUtility.buildNew(
        iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT, depthStart, depthEnd);
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawTextureGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the LAS Texture Grid
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTextureGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i          = 0;
    int    j          = 0;
    int    k          = 0;
    int    n          = 0;
    int    iY1        = 0;
    int    iY2        = 0;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/14;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iR         = 0;
    int    iG         = 0;
    int    iB         = 0;
    int    id         = 0;
    int    iCells     = 0;
    int    iRows      = 0;
    int    iColumns   = 0;
    int    iRGB[]     = { 255, 255, 255 };
    int    iSymbol   = -1;
    String symbol[][]  = null;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    if (iFocus == 0)
      g.drawString("Rock Texture by LAS",
                   iStartTrack + iIncrement, LABELSTART+10);
    else
      g.drawString("Rock Texture",
                   iStartTrack + 3*iIncrement, LABELSTART+10);

    g.setColor(new Color(0, 229, 229));
    g.fillRect(iStartTrack, PLOT_TITLES-65, iWidthTrack, 10);

    iRows    = 5;
    iColumns = 10;
    id       = lith.lithology.lithologySymbolsStruct._LIMESTONE;

    for (int p=0; p<stLithology.iCount; p++)
    {
      if (id == stLithology.stItem[p].id)
      {
        iSymbol = p;
      }
    }

    iRGB   = stLithology.stItem[iSymbol].iRGB;
    symbol = lith.lithology.lithologySymbolsUtility.getSymbol(
                 iSymbol, stLithology);

    for (j=0; j<14; j++)
    {
      int inc = iWidthTrack/12;

      plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                          iStartTrack + j*inc, PLOT_TITLES-65,
                          iColumns, iRows);

      plotLith.drawLithologySymbolMask(
         g, iRows, plotLith.getOrder(0), symbol,
         iStartTrack + j*inc, PLOT_TITLES-65 );
      plotLith.drawLithologySymbolMask(
         g, iRows, plotLith.getOrder(1), symbol,
         iStartTrack + j*inc, PLOT_TITLES-60 );
    }

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawString("Carbonate", iStartTrack + 4*iIncrement,      PLOT_TITLES-55);

    for (k=6; k>-1; k--)
    {
      iRows    = 5;
      iColumns = 10;
      iCells   = lith.texture.textureConstants.CaCO3_CELLS[k];
      iRGB     = lith.texture.textureConstants.LIMESTONE_COLOR[iCells-1];

      for (n=0; n<iCells; n++)
      {
        plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                            iStartTrack+n*10, PLOT_TITLES-44,
                            iIncrement, 11 );
      }

      g.setColor(Color.black);
      g.drawLine(iStartTrack+iCells*10, PLOT_TITLES-44,
                 iStartTrack+iCells*10, PLOT_TITLES-33);
    }

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawLine(iStartTrack+50, PLOT_TITLES-55,
               iStartTrack+50, PLOT_TITLES-33);
    g.drawLine(iStartTrack+90, PLOT_TITLES-55,
               iStartTrack+90, PLOT_TITLES-33);
    g.drawLine(iStartTrack+120, PLOT_TITLES-55,
               iStartTrack+120, PLOT_TITLES-33);

    g.drawString("Mud",   iStartTrack + 2*iIncrement,   PLOT_TITLES-44);
    g.drawString("Grain", iStartTrack + 6*iIncrement,   PLOT_TITLES-44);

    g.drawString("M",     iStartTrack+ iIncrement,      PLOT_TITLES-33);
    g.drawString("W",     iStartTrack+ 4*iIncrement,    PLOT_TITLES-33);
    g.drawString("P",     iStartTrack+ 6*iIncrement,    PLOT_TITLES-33);
    g.drawString("G",     iStartTrack+ 8*iIncrement,    PLOT_TITLES-33);
    g.drawString("B",     iStartTrack+ 10*iIncrement,   PLOT_TITLES-33);
    g.drawString("C",     iStartTrack+ 12*iIncrement,   PLOT_TITLES-33);

    g.setColor(new Color(255, 255, 0));
    g.fillRect(iStartTrack, PLOT_TITLES-32, iWidthTrack, 10);

    iRows    = 5;
    iColumns = 10;
//    id       = lith.lithology.lithologySymbolsStruct._SAND;
    id       = lith.lithology.lithologySymbolsStruct._SANDSTONE;

    for (int p=0; p<stLithology.iCount; p++)
    {
      if (id == stLithology.stItem[p].id)
      {
        iSymbol = p;
      }
    }

    iRGB   = stLithology.stItem[iSymbol].iRGB;
    symbol = lith.lithology.lithologySymbolsUtility.getSymbol(
                 iSymbol, stLithology);

    for (j=0; j<13; j++)
    {
      int inc = iWidthTrack/12;

      plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                          iStartTrack + j*inc, PLOT_TITLES-27,
                          iColumns, iRows);

      plotLith.drawLithologySymbolMask(
         g, iRows, plotLith.getOrder(0), symbol,
         iStartTrack + j*inc, PLOT_TITLES-32 );
      plotLith.drawLithologySymbolMask(
         g, iRows, plotLith.getOrder(1), symbol,
         iStartTrack + j*inc, PLOT_TITLES-27 );
    }

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawString("Siliciclastic", iStartTrack + 4*iIncrement, PLOT_TITLES-22);

    for (k=9; k>-1; k--)
    {
      iRows    = 5;
      iColumns = 10;
      iCells   = lith.texture.textureConstants.SiO2_CELLS[k];
      iRGB     = lith.texture.textureConstants.SIO2_COLOR[iCells-1];

      for (n=0; n<iCells; n++)
      {
        plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                            iStartTrack+n*10, PLOT_TITLES-11,
                            iIncrement, 11 );
      }

      g.setColor(Color.black);
      g.drawLine(iStartTrack+iCells*10, PLOT_TITLES-11,
                 iStartTrack+iCells*10, PLOT_TITLES);
    }

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawLine(iStartTrack+40, PLOT_TITLES-22,
               iStartTrack+40, PLOT_TITLES);
    g.drawLine(iStartTrack+90, PLOT_TITLES-22,
               iStartTrack+90, PLOT_TITLES);

    g.drawString("Mud",    iStartTrack + iIncrement,     PLOT_TITLES-11);
    g.drawString("Sand",   iStartTrack + 5*iIncrement+2, PLOT_TITLES-11);
    g.drawString("Gravel", iStartTrack + 10*iIncrement+2, PLOT_TITLES-11);

    g.drawString("Clay",   iStartTrack,                  PLOT_TITLES);
    g.drawString("Silt",   iStartTrack + 2*iIncrement+2, PLOT_TITLES);
    g.drawString("v",      iStartTrack + 4*iIncrement+2, PLOT_TITLES);
    g.drawString("f",      iStartTrack + 5*iIncrement+3, PLOT_TITLES);
    g.drawString("m",      iStartTrack + 6*iIncrement+2, PLOT_TITLES);
    g.drawString("c",      iStartTrack + 7*iIncrement+3, PLOT_TITLES);
    g.drawString("C",      iStartTrack + 8*iIncrement+1, PLOT_TITLES);
    g.drawString("g",      iStartTrack + 9*iIncrement+2, PLOT_TITLES);
    g.drawString("peb",    iStartTrack + 10*iIncrement+2,PLOT_TITLES);
    g.drawString("cob",    iStartTrack + 12*iIncrement+2,PLOT_TITLES);

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
//    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawRockTextureGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the Rock Texture Grid
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   *

  public void drawRockTextureGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i          = 0;
    int    j          = 0;
    int    iY1        = 0;
    int    iY2        = 0;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/14;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    String str        = "";

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if (iY2 >= PLOT_TITLES)
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawString( "Rock Lithology/Texture",
                  iStartTrack + iIncrement, LABELSTART+10 );

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.drawString( ">> Increasing Energy >>",
                  iStartTrack + (3*iIncrement)/2, PLOT_TITLES-14 );

    fsb  = new Font("Monospaced", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=1; i<8; i++)
    {
      j = 2*i;
      str = new String("" + j);
      if (i<10) str = new String(" " + j);
      g.drawString( str, iStartTrack + (2*i-1)*iIncrement-iIncrement/2, PLOT_TITLES );
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
  }
*/
  /** Method drawTexture( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method draw the texture from the Gamma Ray after determining if
   *     it is a carbonate or a siliclastic rock.
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTexture( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int               k          = 0;
    int               n          = 0;
    int               iPlus      = 0;
    int               iCount     = 0;
    int               iDepth1    = 0;
    int               iDepth2    = 0;
    int               iDepth     = 0;
    int               iGroup     = 0;
    double            depth1     = 0.0;
    double            depth2     = 0.0;
    double            diff       = 0;
    double            dRatio     = 0.0;
    int               iCells     = 0;
    int               iRows      = 0;
    int               iColumns   = 0;
    int               iRGB[]     = { 255, 255, 255 };
    int               iSymbol   = -1;
    String            symbol[][] = null;
    textureListStruct st         = null;
    int               iNew       = 0;
    int               iSegment   = 0;

//    if (iFocus == 0)
//      if (iLASTexture == iqstrat.iqstratTracksStruct._LAS_TEXTURE)
        st = stLASTexture;

//    if (iFocus == 1)
//      if (iTexture == iqstrat.iqstratTracksStruct._ROCK_TEXTURE)
//        st = stTexture;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        if (st.stItem[0] != null)
        {
          depth1 = st.stItem[0].depthStart;
          depth2 = st.stItem[0].depthEnd;
          diff   = depth2-depth1;

          iPlus = (int) (( 5.0/diff ) *
                         ( iqstrat.iqstratTracksStruct.SCALE[iScale] / 100.0 ));
          if (iPlus < 1) iPlus = 1;

          iCount = -1;
          for (k=0; k<st.iCount; k++)
          {
            if (st.stItem[k] != null)
            {
              depth1 = st.stItem[k].depthStart;
              depth2 = depth1 + diff*iPlus;

              for (int i=1; i<iPlus; i++)
                if (k<st.iCount-1)
                  k++;

              dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                              (depthStart - depthEnd);
              iDepth1 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 = iHeight - (int) dRatio;

              dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                              (depthStart - depthEnd);
              iDepth2 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth2 = iHeight - (int) dRatio;

              iDepth = Math.abs( iDepth2 - iDepth1 );

              if (iNew == 0)
              {
                iSegment = iDepth;
                iNew = 1;
              }

              if (iDepth > iSegment) iSegment = iDepth;

              if (iDepth < iSegment)
              {
                iDepth  = iSegment;
                iDepth2 = iDepth1 + iSegment;
              }

//System.out.println(depth1 +" "+iDepth1+" - "+depth2+" "+iDepth2+" "+iDepth);

              if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                  (iDepth1 < iHeight)      && (iDepth2 < iHeight))
              {
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iDepth1 =  iDepth2;

                 iGroup = iDepth / 5;
                 if (iGroup == 0) iGroup = 1;

                 iRows = 5;
                 iColumns = 10;
                 iRGB     = st.stItem[k].iRGB;
                 iCells   = st.stItem[k].iEnergy;
                 symbol   = lith.lithology.lithologySymbolsUtility.getSymbol(
                              st.stItem[k].iSymbol, stLithology);

                 if (st.stItem[k].iSymbol > -1)
                 {
                   for (n=0; n<iGroup; n++)
                   {
                     iCount++;
                     for (int i=0; i<iCells; i++)
                     {
                       plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                           iStartTrack+i*iColumns, iDepth1 + n*5,
                                           iColumns, iRows);

                       plotLith.drawLithologySymbolMask(
                          g, iRows, plotLith.getOrder(iCount), symbol,
                          iStartTrack+i*iColumns, iDepth1 + n*5 );

                     }
                   }
                 }
              }
            }
          }
        }
      }
    }
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track that was selected
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw(Graphics g, int iSelected, int iStartTrack, int iWidthTrack)
  {
    int iColor = -1;

    switch (iSelected)
    {
      case iqstrat.iqstratTracksStruct._LAS_TEXTURE:
        iColor = iqstrat.iqstratTracksStruct._SRC_LAS;

        g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
        g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

        iFocus = 0;
        drawTextureGrid(g, iStartTrack, iWidthTrack);
        drawTexture(g, iStartTrack, iWidthTrack);
        break;
//      case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
//        iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
//
//        g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
//                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
//                             iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
//        g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

//        iFocus = 1;
//        drawRockTextureGrid(g, iStartTrack, iWidthTrack);
//        drawTexture(g, iStartTrack, iWidthTrack);
//        break;
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}
