/*
 * @rockColumnUtility.java Version 1.1 08/03/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.rock;

import lith.rock.rockColumnListStruct;
import lith.rock.rockColumnStruct;

/** Class rockColumnUtility
 *  <p> This Class will provide basic utilities for the Rock Lithology
 *      data structures.
 *
 *  @version 1.1 08/03/2012
 *  @author  John R. Victorine
 */

public class rockColumnUtility
{
  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock texture data list structure
   * @return  st = the Rock texture data list structure
   */

  public static rockColumnListStruct computeDepthRange(rockColumnListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }

      st.depthStart = Math.floor(st.depthStart);
      st.depthEnd   = Math.ceil(st.depthEnd);
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add a Lithology to an existing list
   * @param  stNew = The Rock Lithology Data Structure
   * @param  st    = The Old Rock Lithology List Data Structure
   * @return st    = The new Rock Lithology List.
   */

  public static rockColumnListStruct add( rockColumnStruct     stNew,
                                          rockColumnListStruct st )
  {
    int i = 0;
    int iCount      = 0;
    rockColumnListStruct stTemp = new rockColumnListStruct();

    stTemp.stItem = new rockColumnStruct[1];

    if (st != null)
    {
      if (st.iCount > 0)
        stTemp.stItem = new rockColumnStruct[st.iCount+1];

      for (i=0; i<st.iCount; i++)
      {
        stTemp.stItem[iCount] = new rockColumnStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (stNew != null)
    {
      stTemp.stItem[iCount] = new rockColumnStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new rockColumnListStruct();
    st.iCount = iCount;
    st.stItem = new rockColumnStruct[st.iCount];

    for (i=0; i<st.iCount; i++)
    {
      st.stItem[i] = new rockColumnStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Lithology in an existing list
   * @param  stNew = The New Rock Lithology Data Structure
   * @param  st   = The Old Rock Lithology List Data Structure
   * @return st   = The new Rock Lithology List.
   */

  public static rockColumnListStruct modify( rockColumnStruct     stNew,
                                             rockColumnListStruct st )
  {
    int i = 0;
    int iCount      = 0;
    rockColumnListStruct stTemp = new rockColumnListStruct();

    stTemp.stItem = new rockColumnStruct[1];

    if (st != null)
    {
      if (st.iCount > 0)
        stTemp.stItem = new rockColumnStruct[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(stNew.sKEY))
        {
          stTemp.stItem[iCount] = new rockColumnStruct();
          stTemp.stItem[iCount] = copy(stNew);
          iCount++;
        }
        else
        {
          stTemp.stItem[iCount] = new rockColumnStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new rockColumnListStruct();
      st.iCount = iCount;
      st.stItem = new rockColumnStruct[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
        st.stItem[i] = new rockColumnStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Lithology from an existing list
   * @param  sKEY = Unique Identifier for depth range
   * @param  st   = The Old Rock Lithology List Data Structure
   * @return st   = The new Rock Lithology List.
   */

  public static rockColumnListStruct remove( String sKEY, rockColumnListStruct st )
  {
    int i = 0;
    int iCount      = 0;
    rockColumnListStruct stTemp = null;

    if (st != null)
    {
      if (st.iCount < 2)
      {
        st.iCount = 0;
        st.delete();
        st = null;
      }
      else
      {
        stTemp        = new rockColumnListStruct();
        stTemp.stItem = new rockColumnStruct[st.iCount-1];

        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i] != null)
          {
            if (!st.stItem[i].sKEY.equals(sKEY))
            {
              stTemp.stItem[iCount] = new rockColumnStruct();
              stTemp.stItem[iCount] = copy(st.stItem[i]);
              iCount++;
            }
          }
        }

        st.delete();
        st = null;

        st        = new rockColumnListStruct();
        st.iCount = iCount;
        st.stItem = new rockColumnStruct[st.iCount];

        for (i=0; i<st.iCount; i++)
        {
          st.stItem[i] = new rockColumnStruct();
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        if (stTemp != null)
          stTemp.delete();
        stTemp = null;
      }
    }

    return (st);
  }

  /** Method copyList( rockColumnListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockColumnListStruct copyList( rockColumnListStruct stOld )
  {
    rockColumnListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new rockColumnListStruct();

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new rockColumnStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockColumnListStruct transfer(rockColumnListStruct stOld)
  {
    rockColumnListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static rockColumnStruct copy( rockColumnStruct stOld )
  {
	int i = 0;
    rockColumnStruct stNew = new rockColumnStruct();

    stNew.sKEY        = new String(stOld.sKEY);

    stNew.depthStart  = stOld.depthStart; // Starting Depth
    stNew.depthEnd    = stOld.depthEnd;   // Ending Depth

    for (i=0; i<3; i++)
    {
	  stNew.iRGB[i] = stOld.iRGB[i];
	}

    stNew.sPrimary    = new String( stOld.sPrimary ); // Primary Lithology Identifier

    stNew.iSecondary  = stOld.iSecondary;    // Total Number of Secondary Lithology
    if (stNew.iSecondary > 0)
    {
      stNew.iAmount   = new int[stNew.iSecondary];    // Amount of Secondary Lithology
      stNew.secondary = new String[stNew.iSecondary]; // Array of Secondary Lithology ID's

      for (i=0; i<stNew.iSecondary; i++)
      {
		stNew.iAmount[i]   = stOld.iAmount[i];
		stNew.secondary[i] = new String( stOld.secondary[i] );
	  }
	}

    stNew.iTexture = stOld.iTexture;
//    stNew.igrains  = new int[rock.rockColumnStruct._TOTAL];

    for (i=0; i<lith.rock.rockColumnStruct._TOTAL; i++)
    {
	  stNew.igrains[i] = stOld.igrains[i];
	}

    return (stNew);
  }

  /** Method getData()
   *  <p> This method will determine if the average bed thickness through out the column is
   *      greater than or less than the minimum sampling depth by scale.  If the average
   *      bed thickness is less than the scale minimum thickness then resample the data
   *      at the minimum thickness
   */

  public static rockColumnListStruct getData(rockColumnListStruct st, int iScale)
  {
	int i,j,k,m;
	rockColumnListStruct stNew    = null;
	double               dStart   = 0.0;
	double               dEnd     = 0.0;
	double               depth    = 0.0;
	double               dAverage = 0.0;
	int                  iStart   = 0;
	int                  iTotal   = 0;
	double               dTest    = iqstrat.iqstratTracksStruct.SCALE[iScale] / 10.0;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (k=0; k<st.iCount; k++)
        {
          depth  = st.stItem[k].depthEnd - st.stItem[k].depthStart;

          if (depth > 0.0)
          {
			if (iStart == 0)
			{
			  dStart = st.stItem[k].depthStart;
			  dEnd   = st.stItem[k].depthEnd;
			  iStart++;
			}

			if (dEnd < st.stItem[k].depthEnd) dEnd = st.stItem[k].depthEnd;

            dAverage = dAverage + depth;
            iTotal++;
	      }
	    }

	    dAverage = Math.round(dAverage) / iTotal;

	    if (dAverage > dTest)
	    {
		  stNew = copyList( st );
	    }
	    else
	    {
		  iTotal = 1 + (int) ( Math.round(dEnd - dStart) / dTest );
//System.out.println(dStart +" "+dEnd+" "+dTest+" "+iTotal);

		  stNew  = new rockColumnListStruct();
		  stNew.stItem = new rockColumnStruct[iTotal];

		  for (i=0; i<iTotal; i++)
		  {
		    stNew.stItem[i]            = new rockColumnStruct();
		    stNew.iCount               = iTotal;
            stNew.stItem[i].sKEY       = new String( cmn.cmnString.UniqueName() + "_" + i );
            stNew.stItem[i].depthStart = dStart + i * dTest;
		    stNew.stItem[i].depthEnd   = dStart + (i+1) * dTest;
//System.out.println(stNew.stItem[i].sKEY+" "+stNew.stItem[i].depthStart+" "+stNew.stItem[i].depthEnd);
		  }

		  for (i=0; i<iTotal; i++)
		  {
            for (k=0; k<st.iCount; k++)
            {
			  if ((stNew.stItem[i].depthStart >= st.stItem[k].depthStart) &&
			      (stNew.stItem[i].depthStart < st.stItem[k].depthEnd))
			  {
    			for (j=0; j<3; j++)
    			{
				  stNew.stItem[i].iRGB[j] = st.stItem[k].iRGB[j];
				}

    			stNew.stItem[i].sPrimary    = new String( st.stItem[k].sPrimary );

    			stNew.stItem[i].iSecondary  = st.stItem[k].iSecondary;
    			if (stNew.stItem[i].iSecondary > 0)
    			{
    			  stNew.stItem[i].iAmount   = new int[stNew.stItem[i].iSecondary];
    			  stNew.stItem[i].secondary = new String[stNew.stItem[i].iSecondary];

    			  for (j=0; j<stNew.stItem[i].iSecondary; j++)
    			  {
					stNew.stItem[i].iAmount[j]   = st.stItem[k].iAmount[j];
					stNew.stItem[i].secondary[j] = new String( st.stItem[k].secondary[j] );
				  }
				}

                stNew.stItem[i].iTexture = st.stItem[k].iTexture;
//    stNew.igrains  = new int[rock.rockColumnStruct._TOTAL];

                for (m=0; m<lith.rock.rockColumnStruct._TOTAL; m++)
                {
	              stNew.stItem[i].igrains[m] = st.stItem[k].igrains[m];
  	            }
/*
    			stNew.stItem[i].iDescriptor   = st.stItem[k].iDescriptor;
    			if (stNew.stItem[i].iDescriptor > 0)
    			{
    			  stNew.stItem[i].iDAmount    = new int[stNew.stItem[i].iDescriptor];
    			  stNew.stItem[i].sDescriptor = new String[stNew.stItem[i].iDescriptor];

    			  for (j=0; j<stNew.stItem[i].iDescriptor; j++)
    			  {
    			    stNew.stItem[i].iDAmount[j]    = st.stItem[k].iDAmount[j];
    			    stNew.stItem[i].sDescriptor[j] = new String( st.stItem[k].sDescriptor[j] );
				  }
				}
*/
			  }
		    }
		  }
  	    }
	  }
	}

	return (stNew);
  }

  /** Method print()
   *  <p> This method will print the contents of the data list structure
   */

  public static void print( rockColumnListStruct st)
  {
	int j = 0;
	if (st != null)
	{
	  for (int i=0; i<st.iCount; i++)
	  {
		System.out.println( st.stItem[i].sKEY       + " " +
		                    st.stItem[i].depthStart + " " +
		                    st.stItem[i].depthEnd);
		System.out.println( " -- Color " + st.stItem[i].iRGB[0] + " " +
		                                   st.stItem[i].iRGB[1] + " " +
		                                   st.stItem[i].iRGB[2]);
		System.out.println( " -- Primary " + st.stItem[i].sPrimary );
		if (st.stItem[i].iSecondary > 0)
		{
		  System.out.println( " -- Secondary " );
		  for (j=0; j<st.stItem[i].iSecondary; j++)
		  {
		    System.out.println( " .. Secondary " + st.stItem[i].secondary[j] );
		  }
		}

		System.out.println( " -- Texture= " + st.stItem[i].iTexture );
		for (j=0; j<lith.rock.rockColumnStruct._TOTAL; j++)
		{
		  System.out.print( st.stItem[i].igrains[j] + ", ");
		}
        System.out.println("");
	  }
	}
  }
}

/*
 *  @version 1.1 08/03/2012
 *  @author  John Victorine
 */
