/*
 * @rockColumnPlotTrack.java Version 1.1 08/07/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.rock.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

import lith.rock.rockColumnListStruct;
import lith.rock.rockColumnStruct;
import lith.rock.plot.rockColumnPlotSymbol;

import parse.parseRockColumnSymbolsListStruct;

public class rockColumnPlotTrack extends Canvas
{
  // Input Variables

  private int    iUnits     = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private rockColumnListStruct             stList    = null;
  private rockColumnListStruct             stColumn  = null;

  private parseRockColumnSymbolsListStruct stSymbols = null;
  private rockColumnPlotSymbol             plotLith  = null;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR rockColumnPlotTrack()
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param stSymbols  = Texture Symbols List Data Structure
   */

  public rockColumnPlotTrack( int    iDataType,
                              double depthStart,
                              double depthEnd,
                              int    iScale,
                              parseRockColumnSymbolsListStruct stSymbols )
  {
    this.iDataType = iDataType;

    this.stSymbols = stSymbols;
    plotLith       = new rockColumnPlotSymbol( stSymbols );

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	stList    = null;
    stSymbols = null;

	if (stColumn != null)
      stColumn.delete();
    stColumn  = null;

    if (plotLith != null)
      plotLith.close();
    plotLith = null;
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;

	if (stColumn != null)
      stColumn.delete();
    stColumn  = null;

    stColumn = lith.rock.rockColumnUtility.getData( stList, iScale );
  }

  /** Method setRockColumn( rockColumnListStruct st )
   * <p> This method will set the rock Column that will be shown
   * @param st = Rock column determined by user from rock data.
   */

  public void setRockColumn( rockColumnListStruct st )
  {
	this.stList = st;
    stColumn = lith.rock.rockColumnUtility.getData( this.stList, iScale );
//lith.rock.rockColumnUtility.print(st);
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack,  int iWidthTrack )
  {
	int    i,j;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/10;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iY1        = 0;
    int    iY2        = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight);

//    g.setColor(Color.lightGray);
//    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iHeight);

    g.setColor(Color.black);

    g.drawString("Lithology",   iStartTrack+3*iIncrement, LABELSTART+14);
    g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+28);

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
  }

  /** Method drawColumn( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method draw the texture from the Gamma Ray after determining if
   *     it is a carbonate or a siliclastic rock.
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawColumn( Graphics g, int iStartTrack, int iWidthTrack )
  {
	int i,j,k,m,n;
    rockColumnListStruct st         = null;
    double               depth1     = 0.0;
    double               depth2     = 0.0;
    double               diff       = 0;
    double               dRatio     = 0.0;
    int                  iCount     = 0;
    int                  iDepth1    = 0;
    int                  iDepth2    = 0;
    int                  iDepth     = 0;
    int                  iDiff      = 0;
    String               sID        = "";
    int                  iLith      = -1;
    String               sIDd       = "";
    int                  iLithd     = -1;
    int                  iRows      = 0;
    int                  iCells     = 0;
    int                  iCellY     = 0;
    int                  iCellX     = 0;
    int                  iCellYd    = 0;
    int                  iCellXd    = 0;
    int                  iRGB[]     = { 255, 255, 255 };
    int                  iRGBd[]    = { 255, 255, 255 };
//    int                  iTypes[]   = { -1, -1, -1 };
    int                  iOrder     = -1;
    int                  iPlot      = 0;
    int                  iPlotd     = 0;
    int                  iContinue  = 0;

    st = stColumn;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (k=0; k<st.iCount; k++)
        {
          depth1 = st.stItem[k].depthStart;
          depth2 = st.stItem[k].depthEnd;

          dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                          (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                          (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs( iDepth2 - iDepth1 );

//          if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
//              (iDepth1 < iHeight)      && (iDepth2 < iHeight))
          {
  		    sID    = st.stItem[k].sPrimary;
            iLith  = parse.parseRockColumnSymbolsUtility.getID( sID, stSymbols );

//            iTypes = parse.parseRockColumnSymbolsUtility.getCapType( sID, stSymbols );

            if (iLith > -1)
            {
              iCells = stSymbols.stItem[iLith].iWeather;
              iCellY = stSymbols.stItem[iLith].iRows;
              iCellX = stSymbols.stItem[iLith].iColumns;
              iRGB   = stSymbols.stItem[iLith].iRGB;
              iPlot  = stSymbols.stItem[iLith].iOrder;

			  iRows  = iDepth / iCellY;
              iDiff  = iDepth - iRows *iCellY;

              for (i=0; i<iRows; i++)
              {
                if ((iDepth1+i*iCellY >= PLOT_TITLES) && (iDepth1+i*iCellY < iHeight))
                {
                  for (j=0; j<iCells; j++)
                  {
                    plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                        iStartTrack+j*iCellX, iDepth1 + i*iCellY,
                                        iCellX, iCellY);
                    plotLith.drawSymbolMask(
                           g, iCellY, plotLith.getOrder(iCount),
                           parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                           iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			      }
/*
                  if (iTypes != null)
                  {
                    if (iTypes[0] != -1)
                    {
				      iOrder = plotLith.getOrder(iCount);
//				      if (iTypes[2] != -1)
//				        iOrder = 2;

                      plotLith.drawSymbolMask(
                             g, iCellY, iOrder, iRGB,
                             parse.parseRockColumnSymbolsUtility.getCaps(sID, stSymbols),
                             iStartTrack+iCells*iCellX, iDepth1 + i*iCellY );
//                             iStartTrack+j*iCellX, iDepth1 + i*iCellY );
				    }
			      }
*/
   	              if (st.stItem[k].iSecondary > 0)
			      {
			        for (m=0; m<st.stItem[k].iSecondary; m++)
			        {
  		              sIDd    = st.stItem[k].secondary[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );

                      if (iLithd > -1)
                      {
                        iCellYd = stSymbols.stItem[iLithd].iRows;
                        iCellXd = stSymbols.stItem[iLithd].iColumns;
                        iPlotd  = stSymbols.stItem[iLithd].iOrder;
                        if ((iPlot == 0) && ((iPlotd == 1) || (iPlotd ==2)))
                          iRGBd = stSymbols.stItem[iLithd].iRGB;
                        else
                          iRGBd = iRGB;

                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;
//						  if ((st.stItem[k].iSecondary > 1) && (st.stItem[k].iDescriptor > 0))
//						  {
//						    if (m != plotLith.getOrder(j+iCount))
//						      iContinue = 0;
//						  }

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iCellYd, plotLith.getOrder(iCount), iRGBd,
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellXd, iDepth1 + i*iCellYd );
			            }
			          }
			        }
			  	  }

                  if (iPlot > 3) //Force a replot of lithologies that have special symbols
                  {
                    for (j=0; j<iCells; j++)
                    {
                      plotLith.drawSymbolMask(
                             g, iCellY, plotLith.getOrder(iCount),
                             parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                             iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			        }
			      }
/*
   	              if (st.stItem[k].iDescriptor > 0)
			      {
			        for (m=0; m<st.stItem[k].iDescriptor; m++)
			        {
  		              sIDd    = st.stItem[k].sDescriptor[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );
                      iCellYd = stSymbols.stItem[iLithd].iRows;
                      iCellXd = stSymbols.stItem[iLithd].iColumns;
                      iPlotd  = stSymbols.stItem[iLithd].iOrder;

                      if (iLithd > -1)
                      {
                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;
//						  if ((st.stItem[k].iSecondary > 0) && (st.stItem[k].iDescriptor > 1))
//						  {
//						    if (m == plotLith.getOrder(j+iCount))
//						      iContinue = 0;
//						  }

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iCellYd, plotLith.getOrder(iCount),
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellXd, iDepth1 + i*iCellYd );
			            }
			          }
			        }
			  	  }
*/
			      iCount++;
			    }
			  }

              if ((iDepth1+iRows*iCellY >= PLOT_TITLES) && (iDepth1+iRows*iCellY < iHeight))
              {
			    if (iDiff > 0)
			    {
                  for (j=0; j<iCells; j++)
                  {
                    plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                        iStartTrack+j*iCellX, iDepth1 + iRows*iCellY,
                                        iCellX, iDiff);
                    plotLith.drawSymbolMask(
                           g, iDiff, plotLith.getOrder(iCount),
                           parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                           iStartTrack+j*iCellX, iDepth1 + iRows*iCellY );
			      }
/*
                  if (iTypes != null)
                  {
                    if (iTypes[0] != -1)
                    {
				      iOrder = plotLith.getOrder(iCount);
				      if (iTypes[2] != -1)
				        iOrder = 2;

                      plotLith.drawSymbolMask(
                             g, iDiff, iOrder, iRGB,
                             parse.parseRockColumnSymbolsUtility.getCaps(sID, stSymbols),
                             iStartTrack+iCells*iCellX, iDepth1 + iRows*iCellY );
				    }
				  }
*/
   	              if (st.stItem[k].iSecondary > 0)
			      {
			        for (m=0; m<st.stItem[k].iSecondary; m++)
			        {
  		              sIDd    = st.stItem[k].secondary[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );

                      if (iLithd > -1)
                      {
                        iCellYd = stSymbols.stItem[iLithd].iRows;
                        iCellXd = stSymbols.stItem[iLithd].iColumns;
                        iPlotd  = stSymbols.stItem[iLithd].iOrder;
                        if ((iPlot == 0) && ((iPlotd == 1) || (iPlotd ==2)))
                          iRGBd = stSymbols.stItem[iLithd].iRGB;
                        else
                          iRGBd = iRGB;

                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;
//						  if ((st.stItem[k].iSecondary > 1) && (st.stItem[k].iDescriptor > 0))
//						  {
//						    if (m != plotLith.getOrder(j+iCount))
//						      iContinue = 0;
//						  }

                          if (iDiff > iCellYd) iDiff = iCellYd;

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iDiff, plotLith.getOrder(iCount), iRGBd,
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellXd, iDepth1 + iRows*iCellYd );
			            }
			          }
			        }
			  	  }

                  if (iPlot > 3) // Force a replot of lithologies that have special symbols
                  {
                    for (j=0; j<iCells; j++)
                    {
                      plotLith.drawSymbolMask(
                             g, iDiff, plotLith.getOrder(iCount),
                             parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                             iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			        }
			      }
/*
   	              if (st.stItem[k].iDescriptor > 0)
			      {
			        for (m=0; m<st.stItem[k].iDescriptor; m++)
			        {
  		              sIDd    = st.stItem[k].sDescriptor[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );

                      if (iLithd > -1)
                      {
                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;
//						  if ((st.stItem[k].iSecondary > 0) && (st.stItem[k].iDescriptor > 1))
//						  {
//						    if (m == plotLith.getOrder(j+iCount))
//						      iContinue = 0;
//						  }

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iDiff, plotLith.getOrder(iCount),
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellX, iDepth1 + iRows*iCellY );
			            }
			          }
			        }
			  	  }
*/
			    }
		      }
			}
		  }
	    }
	  }
	}
  }

  /** Method drawTextureGrid( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTextureGrid( Graphics g, int iStartTrack,  int iWidthTrack )
  {
	int    i,j;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/14;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iY1        = 0;
    int    iY2        = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight);

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=2*iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

//    g.drawString("Carbonate", iStartTrack + 4*iIncrement,      PLOT_TITLES-55);

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(new Color(0,157,118));
    g.fillRect(iStartTrack, PLOT_TITLES-44, 60, 11);
    g.setColor(new Color(105,156,160));
    g.fillRect(iStartTrack+6*iIncrement, PLOT_TITLES-44, 20, 11);
    g.setColor(new Color(0,180,204));
    g.fillRect(iStartTrack+8*iIncrement, PLOT_TITLES-44, 20, 11);
    g.setColor(new Color(128,255,255));
    g.fillRect(iStartTrack+10*iIncrement, PLOT_TITLES-44, 20, 11);
    g.setColor(new Color(186,132,195));
    g.fillRect(iStartTrack+12*iIncrement, PLOT_TITLES-44, 20, 11);

    g.setColor(Color.black);

    g.drawLine(iStartTrack+60, PLOT_TITLES-44,
               iStartTrack+60, PLOT_TITLES-33);
    g.drawLine(iStartTrack+80, PLOT_TITLES-55,
               iStartTrack+80, PLOT_TITLES-33);
    g.drawLine(iStartTrack+100, PLOT_TITLES-44,
               iStartTrack+100, PLOT_TITLES-33);
    g.drawLine(iStartTrack+120, PLOT_TITLES-55,
               iStartTrack+120, PLOT_TITLES-33);

    g.drawString("Mud",   iStartTrack + 5*iIncrement,   PLOT_TITLES-44);
    g.drawString("Grain", iStartTrack + 9*iIncrement,   PLOT_TITLES-44);

    g.drawString("Mdst", iStartTrack+  2*iIncrement,     PLOT_TITLES-33);
    g.drawString("Wk",   iStartTrack+  6*iIncrement+2,   PLOT_TITLES-33);
    g.drawString("Pk",   iStartTrack+  8*iIncrement+2,   PLOT_TITLES-33);
    g.drawString("Gn",   iStartTrack+ 10*iIncrement+2,   PLOT_TITLES-33);
    g.drawString("Bd",   iStartTrack+ 12*iIncrement+2,   PLOT_TITLES-33);

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(new Color(127,127,127));
    g.fillRect(iStartTrack,              PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(190,190,190));
    g.fillRect(iStartTrack+2*iIncrement, PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(229,229,229));
    g.fillRect(iStartTrack+4*iIncrement, PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(255,204,51));
    g.fillRect(iStartTrack+6*iIncrement, PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(255,230,0));
    g.fillRect(iStartTrack+8*iIncrement, PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(255,255,52));
    g.fillRect(iStartTrack+10*iIncrement, PLOT_TITLES-11, 20, 11);
    g.setColor(new Color(204,255,0));
    g.fillRect(iStartTrack+12*iIncrement, PLOT_TITLES-11, 20, 11);

    g.setColor(Color.black);

    g.drawLine(iStartTrack+20,  PLOT_TITLES-11,
               iStartTrack+20,  PLOT_TITLES);
    g.drawLine(iStartTrack+40,  PLOT_TITLES-11,
               iStartTrack+40,  PLOT_TITLES);
    g.drawLine(iStartTrack+60,  PLOT_TITLES-22,
               iStartTrack+60,  PLOT_TITLES);
    g.drawLine(iStartTrack+80,  PLOT_TITLES-11,
               iStartTrack+80,  PLOT_TITLES);
    g.drawLine(iStartTrack+100, PLOT_TITLES-11,
               iStartTrack+100, PLOT_TITLES);
    g.drawLine(iStartTrack+120, PLOT_TITLES-22,
               iStartTrack+120, PLOT_TITLES);

    g.setColor(Color.black);
    g.drawString("Mud",    iStartTrack +  2*iIncrement,   PLOT_TITLES-11);
    g.drawString("Sand",   iStartTrack +  8*iIncrement,   PLOT_TITLES-11);

    g.drawString("Clay",   iStartTrack,                   PLOT_TITLES);
    g.drawString("Shle",   iStartTrack +  2*iIncrement,   PLOT_TITLES);
    g.drawString("Silt",   iStartTrack +  4*iIncrement+2, PLOT_TITLES);
    g.drawString("Fn",     iStartTrack +  6*iIncrement+2, PLOT_TITLES);
    g.drawString("Med",    iStartTrack +  8*iIncrement+2, PLOT_TITLES);
    g.drawString("Crs",    iStartTrack + 10*iIncrement+2, PLOT_TITLES);
    g.drawString("Grv",    iStartTrack + 12*iIncrement+2, PLOT_TITLES);

//    g.setColor(Color.lightGray);
//    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iHeight);

    g.setColor(Color.black);

    g.drawString("Texture",   iStartTrack+3*iIncrement, LABELSTART+14);
//    g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+28);

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
  }

  /** Method drawColumn( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method draw the texture from the Gamma Ray after determining if
   *     it is a carbonate or a siliclastic rock.
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Plot Track selected
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawColumn( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
  {
	int i,j,k,m,n;
    rockColumnListStruct st         = null;
    double               depth1     = 0.0;
    double               depth2     = 0.0;
    double               diff       = 0;
    double               dRatio     = 0.0;
    int                  iCount     = 0;
    int                  iDepth1    = 0;
    int                  iDepth2    = 0;
    int                  iDepth     = 0;
    int                  iDiff      = 0;
    String               sID        = "";
    int                  iLith      = -1;
    String               sIDd       = "";
    int                  iLithd     = -1;
    int                  iRows      = 0;
    int                  iCells     = 0;
    int                  iCellY     = 0;
    int                  iCellX     = 0;
    int                  iCellYd    = 0;
    int                  iCellXd    = 0;
    int                  iRGB[]     = { 255, 255, 255 };
    int                  iRGBd[]    = { 255, 255, 255 };
    int                  iOrder     = -1;
    int                  iPlot      = 0;
    int                  iPlotd     = 0;
    int                  iContinue  = 0;

    st = stColumn;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (k=0; k<st.iCount; k++)
        {
          depth1 = st.stItem[k].depthStart;
          depth2 = st.stItem[k].depthEnd;

          dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                          (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                          (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs( iDepth2 - iDepth1 );

//          if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
//              (iDepth1 < iHeight)      && (iDepth2 < iHeight))
          {
  		    sID    = st.stItem[k].sPrimary;
            iLith  = parse.parseRockColumnSymbolsUtility.getID( sID, stSymbols );

            if (iLith > -1)
            {
			  if (iSelected == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION)
                iCells = stSymbols.stItem[iLith].iWeather;
              else
                iCells = st.stItem[k].iTexture;

              iCellY = stSymbols.stItem[iLith].iRows;
              iCellX = stSymbols.stItem[iLith].iColumns;
              iRGB   = stSymbols.stItem[iLith].iRGB;
              iPlot  = stSymbols.stItem[iLith].iOrder;

			  iRows  = iDepth / iCellY;
              iDiff  = iDepth - iRows *iCellY;

              for (i=0; i<iRows; i++)
              {
                if ((iDepth1+i*iCellY >= PLOT_TITLES) && (iDepth1+i*iCellY < iHeight))
                {
                  for (j=0; j<iCells; j++)
                  {
                    plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                        iStartTrack+j*iCellX, iDepth1 + i*iCellY,
                                        iCellX, iCellY);
                    plotLith.drawSymbolMask(
                           g, iCellY, plotLith.getOrder(iCount),
                           parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                           iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			      }

   	              if (st.stItem[k].iSecondary > 0)
			      {
			        for (m=0; m<st.stItem[k].iSecondary; m++)
			        {
  		              sIDd    = st.stItem[k].secondary[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );

                      if (iLithd > -1)
                      {
                        iCellYd = stSymbols.stItem[iLithd].iRows;
                        iCellXd = stSymbols.stItem[iLithd].iColumns;
                        iPlotd  = stSymbols.stItem[iLithd].iOrder;
                        if ((iPlot == 0) && ((iPlotd == 1) || (iPlotd ==2)))
                          iRGBd = stSymbols.stItem[iLithd].iRGB;
                        else
                          iRGBd = iRGB;

                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iCellYd, plotLith.getOrder(iCount), iRGBd,
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellXd, iDepth1 + i*iCellYd );
			            }
			          }
			        }
			  	  }

                  if (iPlot > 3) //Force a replot of lithologies that have special symbols
                  {
                    for (j=0; j<iCells; j++)
                    {
                      plotLith.drawSymbolMask(
                             g, iCellY, plotLith.getOrder(iCount),
                             parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                             iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			        }
			      }

			      iCount++;
			    }
			  }

              if ((iDepth1+iRows*iCellY >= PLOT_TITLES) && (iDepth1+iRows*iCellY < iHeight))
              {
			    if (iDiff > 0)
			    {
                  for (j=0; j<iCells; j++)
                  {
                    plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                        iStartTrack+j*iCellX, iDepth1 + iRows*iCellY,
                                        iCellX, iDiff);
                    plotLith.drawSymbolMask(
                           g, iDiff, plotLith.getOrder(iCount),
                           parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                           iStartTrack+j*iCellX, iDepth1 + iRows*iCellY );
			      }

   	              if (st.stItem[k].iSecondary > 0)
			      {
			        for (m=0; m<st.stItem[k].iSecondary; m++)
			        {
  		              sIDd    = st.stItem[k].secondary[m];
                      iLithd  = parse.parseRockColumnSymbolsUtility.getID( sIDd, stSymbols );

                      if (iLithd > -1)
                      {
                        iCellYd = stSymbols.stItem[iLithd].iRows;
                        iCellXd = stSymbols.stItem[iLithd].iColumns;
                        iPlotd  = stSymbols.stItem[iLithd].iOrder;
                        if ((iPlot == 0) && ((iPlotd == 1) || (iPlotd ==2)))
                          iRGBd = stSymbols.stItem[iLithd].iRGB;
                        else
                          iRGBd = iRGB;

                        for (j=0; j<iCells; j++)
                        {
						  iContinue = 1;

                          if (iDiff > iCellYd) iDiff = iCellYd;

						  if (iContinue == 1)
                            plotLith.drawSymbolMask(
                                 g, iDiff, plotLith.getOrder(iCount), iRGBd,
                                 parse.parseRockColumnSymbolsUtility.getSymbol(sIDd, stSymbols),
                                 iStartTrack+j*iCellXd, iDepth1 + iRows*iCellYd );
			            }
			          }
			        }
			  	  }

                  if (iPlot > 3) // Force a replot of lithologies that have special symbols
                  {
                    for (j=0; j<iCells; j++)
                    {
                      plotLith.drawSymbolMask(
                             g, iDiff, plotLith.getOrder(iCount),
                             parse.parseRockColumnSymbolsUtility.getSymbol(sID, stSymbols),
                             iStartTrack+j*iCellX, iDepth1 + i*iCellY );
			        }
			      }
			    }
		      }
			}
		  }
	    }
	  }
	}
  }

  /** Method drawGrainsizeGrid( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrainsizeGrid( Graphics g, int iStartTrack,  int iWidthTrack )
  {
	int    i,j;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/8;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iY1        = 0;
    int    iY2        = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.black);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.gray);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight);

    g.setColor(Color.lightGray);
    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, LABELSTART+20,
                   i+iStartTrack, PLOT_TITLES+iLogHeight);
      }
    }

    g.setColor(Color.gray);
    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Crypto",      LABELSTART+30, -1*(iStartTrack));
    g2.drawString("Micro",       LABELSTART+30, -1*(iStartTrack + iIncrement));
    g2.drawString("Very Fine",   LABELSTART+20, -1*(iStartTrack + 2*iIncrement));
    g2.drawString("Fine",        LABELSTART+30, -1*(iStartTrack + 3*iIncrement));
    g2.drawString("Medium",      LABELSTART+30, -1*(iStartTrack + 4*iIncrement));
    g2.drawString("Coarse",      LABELSTART+30, -1*(iStartTrack + 5*iIncrement));
    g2.drawString("Very Coarse", LABELSTART+20, -1*(iStartTrack + 6*iIncrement));
    g2.drawString("Extr Coarse", LABELSTART+20, -1*(iStartTrack + 7*iIncrement));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.setColor(Color.black);
    g.drawString("Grainsize",   iStartTrack+iIncrement, LABELSTART+14);

    g.drawLine(iStartTrack, LABELSTART,    iStartTrack, PLOT_TITLES+iLogHeight);
    g.drawLine(iEndTrack,   LABELSTART,    iEndTrack,   PLOT_TITLES+iLogHeight);

    g.drawLine(iStartTrack, LABELSTART,    iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART+20, iEndTrack,   LABELSTART+20);
    g.drawLine(iStartTrack, PLOT_TITLES,   iEndTrack,   PLOT_TITLES);
  }

  /** Method drawGrainsize( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method draw the texture from the Gamma Ray after determining if
   *     it is a carbonate or a siliclastic rock.
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrainsize( Graphics g, int iStartTrack, int iWidthTrack )
  {
	int j,k;
    rockColumnListStruct st         = null;
    double               depth1     = 0.0;
    double               depth2     = 0.0;
    double               diff       = 0;
    double               dRatio     = 0.0;
    int                  iDepth1    = 0;
    int                  iDepth2    = 0;
    int                  iDepth     = 0;

    String               sID        = "";
    int                  iLith      = -1;
    int                  iRGB[]     = { 255, 255, 255 };
    int                  iIncrement = iWidthTrack/8;

    st = stColumn;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (k=0; k<st.iCount; k++)
        {
          depth1 = st.stItem[k].depthStart;
          depth2 = st.stItem[k].depthEnd;

          dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                          (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                          (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs( iDepth2 - iDepth1 );

          if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)      && (iDepth2 < iHeight))
          {
  		    sID    = st.stItem[k].sPrimary;
            iLith  = parse.parseRockColumnSymbolsUtility.getID( sID, stSymbols );

            if (iLith > -1)
            {
              iRGB   = stSymbols.stItem[iLith].iRGB;

              for (j=0; j<8; j++)
              {
				if (st.stItem[k].igrains[j] > -1)
				{
                  g.setColor(new Color(iRGB[0], iRGB[1], iRGB[2]));
                  g.fillRect(iStartTrack+j*iIncrement, iDepth1, iIncrement, iDepth);
				}
			  }
			}
		  }
		}
	  }
	}
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track that was selected
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw(Graphics g, int iSelected, int iStartTrack, int iWidthTrack)
  {
    int iColor = -1;

	if (iSelected == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION)
	{
      iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawGrid( g, iStartTrack, iWidthTrack );
      drawColumn( g, iStartTrack, iWidthTrack );
	}

	if (iSelected == iqstrat.iqstratTracksStruct._ROCK_TEXTURE)
	{
      iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawTextureGrid( g, iStartTrack, iWidthTrack );
      drawColumn( g, iSelected, iStartTrack, iWidthTrack );
	}


	if (iSelected == iqstrat.iqstratTracksStruct._ROCK_GRAIN)
	{
      iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawGrainsize( g, iStartTrack, iWidthTrack );
      drawGrainsizeGrid( g, iStartTrack, iWidthTrack );
	}
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}