/*
 * @lithPlotColorlith.java Version 1.1 09/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.plot;

import java.awt.*;
import java.awt.geom.*;

import las.lasFileDataStruct;
import iqstrat.iqstratShaleListStruct;
import rock.rockFileDataStruct;

/** CLASS lithPlotColorlith
 *  <p> This Class will create the Colorlith Plots.
 *
 *  @version 1.1 09/05/2007
 *  @author  John R. Victorine
 */

public class lithPlotColorlith extends Canvas
{
  // Input Variables

  private boolean bSonic     = false; // Indicator to plot by two-way time
  private int     iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double  depthStart = 0.0;   // Starting Depth
  private double  depthEnd   = 0.0;   // Ending Depth
  private int     iScale     = -1;    // Plot Scale

  // Global Variables

  private int    iColorlith = iqstrat.iqstratTracksStruct._COLORLITH_RU;

  private lasFileDataStruct stLASFileData = null;
  private rockFileDataStruct stRockFileData = null;

  public static final double RHO_MAX = 3.0;
  public static final double RHO_MIN = 2.0;

  // Rhomaa-Umaa Plot Limits

  public static final double UMAA_MAX   = iqstrat.iqstratTracksStruct.UMAA_MAX;
  public static final double UMAA_MIN   = iqstrat.iqstratTracksStruct.UMAA_MIN;

  public static final double RHOMAA_MAX = iqstrat.iqstratTracksStruct.RHOMAA_MAX;
  public static final double RHOMAA_MIN = iqstrat.iqstratTracksStruct.RHOMAA_MIN;

  // DT Plot Limits

  public static final double DTMAA_MAX  = iqstrat.iqstratTracksStruct.DTMAA_MAX;
  public static final double DTMAA_MIN  = iqstrat.iqstratTracksStruct.DTMAA_MIN;

  // Porosity Difference Plot Limits

  public static final double PHI_MAX  = iqstrat.iqstratTracksStruct.PHI_MAX;
  public static final double PHI_MIN  = iqstrat.iqstratTracksStruct.PHI_MIN;

  public static final double DPHI_MAX = iqstrat.iqstratTracksStruct.DPHI_MAX;
  public static final double DPHI_MIN = iqstrat.iqstratTracksStruct.DPHI_MIN;

  // Spectral Gamma Ray Ratio Plot Limits

  public static final double THU_MAX  = iqstrat.iqstratTracksStruct.THU_MAX;
  public static final double THU_MIN  = iqstrat.iqstratTracksStruct.THU_MIN;

  public static final double THK_MAX  = iqstrat.iqstratTracksStruct.THK_MAX;
  public static final double THK_MIN  = iqstrat.iqstratTracksStruct.THK_MIN;

  // Spectral Gamma Ray Plot Limits

  public static final double U_MAX    = iqstrat.iqstratTracksStruct.U_MAX;
  public static final double U_MIN    = iqstrat.iqstratTracksStruct.U_MIN;

  public static final double TH_MAX   = iqstrat.iqstratTracksStruct.TH_MAX;
  public static final double TH_MIN   = iqstrat.iqstratTracksStruct.TH_MIN;

  public static final double K_MAX    = iqstrat.iqstratTracksStruct.K_MAX;
  public static final double K_MIN    = iqstrat.iqstratTracksStruct.K_MIN;

  // Resistivity/Porosity Track Macros

  private static final int _RT   = 0;
  private static final int _PHIT = 1;

  // Resistivity Lithology Track

  private int    iRt         = -1;
  private double dRtMax      = iqstrat.iqstratTracksStruct.MMHO_MAX;
  private double dRtMin      = iqstrat.iqstratTracksStruct.MMHO_MIN;
  private double dataRt[][]  = null; // LAS Conductivity - Non-Linear

  private static final int _MMHOS_M    = iqstrat.iqstratTracksStruct._MMHOS_M;
  private static final int MMHOS_M[][] = iqstrat.iqstratTracksStruct.MMHOS_M;

  // Porosity Lithology Track

  private int    iPhit        = -1;
  private double dPhitMax     = iqstrat.iqstratTracksStruct.PHIT_MAX;
  private double dPhitMin     = iqstrat.iqstratTracksStruct.PHIT_MIN;
  private double dataPhit[][] = null; // LAS Porosity - Non-Linear
  private double dataPhi[][]  = null; // Rock Porosity - Non-Linear

  private static final int _PU    = iqstrat.iqstratTracksStruct._PU;
  private static final int PU[][] = iqstrat.iqstratTracksStruct.PU;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR lithPlotColorlith( int iDataType,
   *                                 double depthStart,
   *                                 double depthEnd,
   *                                 int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data Type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lithPlotColorlith( int iDataType,
                            double depthStart,
                            double depthEnd,
                            int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR lithPlotColorlith( boolean bSonic,
   *                                 int iDataType,
   *                                 double depthStart,
   *                                 double depthEnd,
   *                                 int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data Type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lithPlotColorlith( boolean bSonic,
                            int iDataType,
                            double depthStart,
                            double depthEnd,
                            int iScale )
  {
	this.bSonic    = bSonic;
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stLASFileData  = null;
    stRockFileData = null;

    dataRt   = null; // LAS Conductivity - Non-Linear
    dataPhit = null; // LAS Porosity - Non-Linear
    dataPhi  = null; // Rock Porosity - Non-Linear
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setLASFileData( lasFileDataStruct stLASFileData )
   * <p> This method will set the LAS File Data Structure
   * @param stLASFileData - LAS File Data Structure
   */

  public void setLASFileData( lasFileDataStruct stLASFileData )
  {
    this.stLASFileData = stLASFileData;
  }

  /** Method setRockFileData( rockFileDataStruct stLASFileData )
   * <p> This method will set the Rock File Data Structure
   * @param st - Rock File Data Structure
   */

  public void setRockFileData( rockFileDataStruct st )
  {
    this.stRockFileData = st;
  }

  /** Method setColorlith( int iColorlith )
   * <p> This method will set the colorlith that will be shown
   * @param iColorlith = colorlith identifier
   */

  public void setColorlith( int iColorlith ) { this.iColorlith = iColorlith; }

  /** Method setOHM( int iRt, double dRtMax, double dRtMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iRt    = Identifier representing curve to use in plotting
   * @param dRtMax = the maximum conductivity value
   * @param dRtMin = the minimum conductivity value
   */

  public void setOHM( int iRt, double dRtMax, double dRtMin )
  {
    this.iRt    = iRt;
    this.dRtMax = dRtMax;
    this.dRtMin = dRtMin;
  }

  /** Method setPHI( int iPhit, double dPhitMax, double dPhitMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iPhit    = Identifier representing curve to use in plotting
   * @param dPhitMax = the maximum porosity value
   * @param dPhitMin = the minimum porosity value
   */

  public void setPHI( int iPhit, double dPhitMax, double dPhitMin )
  {
    this.iPhit    = iPhit;
    this.dPhitMax = dPhitMax;
    this.dPhitMin = dPhitMin;
  }

  /** Method setImageData( double dataRt, double dataPhit, double dataPhi )
   * <p> This method will set the Non-Linear Data for the Non-Linear
   *     Resistivity & Porosity Image Tracks
   * @param dataRt   = The Resistivity Data for the Non-Linear Resistivity Track
   * @param dataPhit = The Porosity Data for the Non-Linear LAS-Porosity Track
   * @param dataPhi  = The Porosity Data for the Non-Linear Rock-Porosity Track
   */

  public void setImageData(
                double dataRt[][], double dataPhit[][], double dataPhi[][] )
  {
    this.dataRt   = dataRt;
    this.dataPhit = dataPhit;
    this.dataPhi  = dataPhi;
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGammaTrack( Graphics g, int iStartTrack, int iWidthTrack )
   *  <p> This method will color the zones in the Lithology Track as gray levels.
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Log Lithology Track Selected
   * @param iStartTrack = The starting pixel for the Lithology Track
   * @param iWidthTrack = The width of Lithology Track in pixels
   */

  public void drawGammaTrack( Graphics g,
                              int iSelected,
                              int iStartTrack,
                              int iWidthTrack )
  {
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    int    iGColor  = 0;
    int    iForm    = 0;
    double dRatio   = 0.0;
    double dLog     = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;
    int    iEndTrack = iStartTrack+iWidthTrack;
    double dGammaMin = iqstrat.iqstratShaleStruct.GAMMA_MIN;
    double dGammaMax = iqstrat.iqstratShaleStruct.GAMMA_MAX;
    int    iRows     = 0;
    double depths[]  = null;
    double dCurves[] = null;
    iqstratShaleListStruct stList = null;

    int    iRed      = 255;
    int    iGreen    = 255;
    int    iBlue     = 255;
    double dShaly    = iqstrat.iqstratShaleStruct.SHALY;
    double dShale    = iqstrat.iqstratShaleStruct.SHALE;
    double diff      = 0.0;
    int    iFill     = 0;

    g.setColor(new Color(255, 255, 255));

    if (iSelected == iqstrat.iqstratTracksStruct._LITHOLOGY_GR)
    {
      if (stLASFileData != null)
      {
        iRows   = stLASFileData.iRows;
        if (bSonic)
          depths  = stLASFileData.dTime;
        else
          depths  = stLASFileData.depths;
        dCurves = stLASFileData.dGR;
        stList  = stLASFileData.stList;
      }
    }

    if (iRows > 0)
    {
      for (int i=0; i<iRows-1; i++)
      {
        depth1 = depths[i];
        depth2 = depths[i+1];

        dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                        (depthStart - depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                        (depthStart - depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        iDepth = Math.abs( iDepth2 - iDepth1 );

        dLog = dCurves[i];

        if (stList != null)
        {
          for (int n=0; n<stList.iCount; n++)
          {
            if ((depths[i] >= stList.stItem[n].depthStart) &&
                (depths[i] <= stList.stItem[n].depthEnd))
            {
              dGammaMin = stList.stItem[n].dGammaMin;
              dShaly    = stList.stItem[n].dShaly;
              dShale    = stList.stItem[n].dShale;
              dGammaMax = stList.stItem[n].dGammaMax;
            }
          }
        }

        if (dLog < dGammaMin) dLog = dGammaMin;
        if (dLog > dGammaMax) dLog = dGammaMax;

        iFill = 1;

        if (dLog < 0.1)
        {
          diff   = 0.1 - dGammaMin;
   	      iRed   = 255;
		  iGreen = 255;
		  iBlue  = 255;
		  iFill  = 0;
		}
        else if (dLog < dShale)
        {
		  diff   = dShale - dGammaMin;
		  iRed   = 255;
		  iGreen = (int) ( 255 * (dShale - dLog) / diff );
		  iBlue  = 0;
		}
		else
		{
		  diff   = dGammaMax - dShale;
		  iRed   = (int) ( 255 * (dGammaMax - dLog) / diff );
		  iGreen = 0;
		  iBlue  = 0;
		}

        g.setColor(new Color(iRed, iGreen, iBlue));

        iGColor = (int) (255 * (1.0 - ((dLog-dGammaMin) / (dGammaMax-dGammaMin))));

        if ((iFill == 1) &&
            (iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight)     && (iDepth2 < iHeight))
        {
          iForm = (iWidthTrack * (256-iGColor)) / 255;
          iForm = iWidthTrack - iForm / 2;
          g.fillRect(iEndTrack-iForm, iDepth1, iForm, iDepth);
        }
      }
    }

    // Draw Gamma Ray Grid & Title

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Lithology", LABELSTART+14, -1*(iStartTrack+2*iWidthTrack/3));
    g2.drawString("Gamma Ray", LABELSTART+7, -1*(iStartTrack+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawImageTrack( Graphics g,
   *                         int iSelected,
   *                         int iTrack,
   *                         int iStartTrack,
   *                         int iWidthTrack )
   *  <p> This method will color the zones in the Image Lithology Track with the
   *      visual rock image
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Log Lithology Track Selected
   * @param iTrack      = Image track
   * @param iStartTrack = The starting pixel for the Lithology Track
   * @param iWidthTrack = The width of Lithology Track in pixels
   */

  public void drawImageTrack( Graphics g,
                              int iSelected,
                              int iTrack,
                              int iStartTrack,
                              int iWidthTrack )
  {
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    int    iGColor  = 0;
    double dRatio   = 0.0;
    double dLog     = 0.0;
    double depths[] = null;
    double dCurve[] = null;
    double depth1   = 0.0;
    double depth2   = 0.0;
    int    iEndTrack = iStartTrack+iWidthTrack;
    int    iTool     = -1;
    double dMinimum  = 0.0;
    double dMaximum  = 0.0;
    int    iRows     = 0;
    int    iContinue = 0;
    double dNull     = 0.0; // Null Value of Data

    switch (iTrack)
    {
      case _RT:
        iTool    = iRt;
        dMaximum = dRtMax;
        dMinimum = dRtMin;
        break;

      case _PHIT:
        iTool    =  iPhit;
        dMaximum = dPhitMax;
        dMinimum = dPhitMin;
        break;
    }

    if ((iSelected == iqstrat.iqstratTracksStruct._POROSITY_IMAGER) ||
        (iSelected == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER))
    {
      if (stLASFileData != null)
      {
        iRows  = stLASFileData.iRows;
        if (bSonic)
		  depths = stLASFileData.dTime;
        else
          depths = stLASFileData.depths;
        dCurve = stLASFileData.getData(iTool);
        dNull  = stLASFileData.dNull;

        if (stLASFileData.checkData(iTool))
        {
          iContinue = 1;
        }
      }
    }

    g.setColor(new Color(255, 255, 255));

    if (iTool > -1)
    {
      if (iContinue == 1)
      {
        for (int i=0; i<iRows-1; i++)
        {
          depth1 = depths[i];
          depth2 = depths[i+1];

          dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                          (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                            (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth  = Math.abs( iDepth2 - iDepth1 );

          dLog = dCurve[i];

          if (iTrack == _RT)
          {
            if (dLog > dNull)
              dLog = 1000.0 / dLog;
            else
              dLog = dNull;
          }

          if (dLog < dMinimum) dLog = dMinimum;
          if (dLog > dMaximum) dLog = dMaximum;

          switch (iTrack)
          {
            case _RT:
              iGColor = (int) (((_MMHOS_M-1) * (dLog - dMinimum)) /
                               (dMaximum - dMinimum));
              g.setColor(new Color(MMHOS_M[iGColor][0],
                                   MMHOS_M[iGColor][1],
                                   MMHOS_M[iGColor][2]));
              break;

            case _PHIT:
              iGColor = (int) (((_PU-1) * (dLog - dMinimum)) /
                               (dMaximum - dMinimum));
              g.setColor(new Color(PU[iGColor][0],
                                   PU[iGColor][1],
                                   PU[iGColor][2]));
              break;
          }

          if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 < iHeight))
          {
            if (dLog > dNull)
            {
              g.fillRect(iStartTrack, iDepth1, iWidthTrack, iDepth);
            }
          }
        }
      }
    }

    for (int j=0; j<iWidthTrack; j++)
    {
      switch (iTrack)
      {
        case _RT:
          iGColor = (int) (((_MMHOS_M-1) * j) / iWidthTrack);
          g.setColor(new Color(MMHOS_M[iGColor][0],
                               MMHOS_M[iGColor][1],
                               MMHOS_M[iGColor][2]));
          break;

        case _PHIT:
          iGColor = (int) (((_PU-1) * j) / iWidthTrack);
          g.setColor(new Color(PU[iGColor][0],
                               PU[iGColor][1],
                               PU[iGColor][2]));
          break;
      }
      g.fillRect(iStartTrack+j, PLOT_TITLES-6, 1, 6);
    }

    // Draw Grid & Title

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Linear", LABELSTART+14, -1*(iStartTrack+3*iWidthTrack/4));

    fsb  = new Font("Serif", Font.BOLD, 11);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g2.drawString("color scheme", LABELSTART+1, -1*(iStartTrack+iWidthTrack/2));

    String sTemp = "";
    if (iTool > -1) //&&
//        (iSelected != iqstrat.iqstratTracksStruct._ROCK_POROSITY_IMAGER))
      g2.drawString(las.lasStandardTools.LAS_TOOLS[iTool][1] + sTemp,
                    LABELSTART+14, -1*(iStartTrack));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawNonLinearImageTrack( Graphics g,
   *                                  int iSelected,
   *                                  int iTrack,
   *                                  int iStartTrack,
   *                                  int iWidthTrack )
   *  <p> This method will color the zones in the Image Lithology Track with the
   *      visual rock image
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Log Lithology Track Selected
   * @param iTrack      = Image track
   * @param iStartTrack = The starting pixel for the Lithology Track
   * @param iWidthTrack = The width of Lithology Track in pixels
   */

  public void drawNonLinearImageTrack( Graphics g,
                                       int iSelected,
                                       int iTrack,
                                       int iStartTrack,
                                       int iWidthTrack )
  {
    int    iGColor   = 0;
    int    iTool     = -1;
    int    iDepth1   = 0;
    int    iDepth2   = 0;
    int    iDepth    = 0;
    double dRatio    = 0.0;
    double depth1    = 0.0;
    double depth2    = 0.0;
    int    iEndTrack = iStartTrack+iWidthTrack;
    int    length    = 0;
    double dMinimum  = 0.0;
    double dMaximum  = 0.0;
    double data[][]  = null;
    double dLog      = 0.0;
    int    iContinue = 0;

    switch (iTrack)
    {
      case _RT:
        iTool    = iRt;
        dMaximum = dRtMax;
        dMinimum = dRtMin;
        data     = dataRt;
        break;

      case _PHIT:
        iTool    =  iPhit;
        dMaximum = dPhitMax;
        dMinimum = dPhitMin;
        data     = dataPhit;
        break;
    }

    if (stLASFileData != null)
    {
      if (stLASFileData.checkData(iTool))
        iContinue = 1;
    }

    g.setColor(new Color(255, 255, 255));

    if (iTool > -1)
    {
      if ((iContinue == 1) && (data != null))
      {
        length = data.length;

        for (int i=0; i<length-1; i++)
        {
          depth1 = data[i][0];
          depth2 = data[i+1][0];

          dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                          (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                          (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth  = Math.abs( iDepth2 - iDepth1 );

          dLog = data[i][1];

          if (dLog < dMinimum) dLog = dMinimum;
          if (dLog > dMaximum) dLog = dMaximum;

          iGColor = (int) data[i][2];
          switch (iTrack)
          {
            case _RT:
              g.setColor(new Color( MMHOS_M[iGColor][0],
                                    MMHOS_M[iGColor][1],
                                    MMHOS_M[iGColor][2] ));
              break;

            case _PHIT:
              g.setColor(new Color( PU[iGColor][0],
                                    PU[iGColor][1],
                                    PU[iGColor][2] ));
              break;
          }

          if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 < iHeight))
          {
            if (dLog > stLASFileData.dNull)
            {
              g.fillRect( iStartTrack, iDepth1, iWidthTrack, iDepth );
            }
          }
        }
      }
    }

    for (int j=0; j<iWidthTrack; j++)
    {
      switch (iTrack)
      {
        case _RT:
          iGColor = (int) (((_MMHOS_M-1) * j) / iWidthTrack);
          g.setColor(new Color(MMHOS_M[iGColor][0],
                               MMHOS_M[iGColor][1],
                               MMHOS_M[iGColor][2]));
          break;

        case _PHIT:
          iGColor = (int) (((_PU-1) * j) / iWidthTrack);
          g.setColor(new Color(PU[iGColor][0],
                               PU[iGColor][1],
                               PU[iGColor][2]));
          break;
      }
      g.fillRect(iStartTrack+j, PLOT_TITLES-6, 1, 6);
    }

    // Draw Grid & Title

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Non-linear", LABELSTART+14, -1*(iStartTrack+3*iWidthTrack/4));

    fsb  = new Font("Serif", Font.BOLD, 11);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g2.drawString("color scheme", LABELSTART+1, -1*(iStartTrack+iWidthTrack/2));

    String sTemp = "";
    if (iTool > -1)
      g2.drawString(las.lasStandardTools.LAS_TOOLS[iTool][1] + sTemp,
                    LABELSTART+14, -1*(iStartTrack));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine( iStartTrack, LABELSTART,  iEndTrack,   LABELSTART );
    g.drawLine( iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES );
    g.drawLine( iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight );
    g.drawLine( iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight );
  }

  /** Method drawLithologyTrack( Graphics g,
   *                             int iSelected,
   *                             int iStartTrack,
   *                             int iWidthTrack )
   *  <p> This method will color the zones in the Lithology Track.
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track that was selected
   * @param iStartTrack = The starting pixel for the Lithology Track
   * @param iWidthTrack = The width of Lithology Track in pixels
   */

  public void drawLithologyTrack( Graphics g,
                                  int iSelected,
                                  int iStartTrack,
                                  int iWidthTrack)
  {
    int    iRows     = 0;
    int    iDepth1   = 0;
    int    iDepth2   = 0;
    int    iDepth    = 0;
    int    iRed      = 0;
    int    iGreen    = 0;
    int    iBlue     = 0;
    int    iTemp     = 0;
    int    iForm     = 0;
    double dLog      = 0.0;
    double dRatio    = 0.0;
    int    iEndTrack = iStartTrack+iWidthTrack;

    iqstratShaleListStruct stList = null;

    double dRedMax   = 0.0;
    double dRedMin   = 0.0;
    double dGreenMax = 0.0;
    double dGreenMin = 0.0;
    double dBlueMax  = 0.0;
    double dBlueMin  = 0.0;

    double dNull     = 0.0;
    double depths[]  = null;
    double dRed[]    = null;
    double dGreen[]  = null;
    double dBlue[]   = null;

    String sTitle1   = "";
    String sTitle2   = "";

    switch (iSelected)
    {
      case iqstrat.iqstratTracksStruct._CLRLITH_RU:
        if (stLASFileData != null)
        {
          iRows     = stLASFileData.iRows;
          dNull     = stLASFileData.dNull;
          if (bSonic)
            depths  = stLASFileData.dTime;
          else
            depths  = stLASFileData.depths;

          stList    = stLASFileData.stList;

          dRed      = stLASFileData.dRhomaa;
          dRedMax   = RHOMAA_MAX;
          dRedMin   = RHOMAA_MIN;

          dGreen    = stLASFileData.dUmaa;
          dGreenMax = UMAA_MAX;
          dGreenMin = UMAA_MIN;

          dBlue     = stLASFileData.dGR;
          dBlueMax  = iqstrat.iqstratShaleStruct.GAMMA_MAX;
          dBlueMin  = iqstrat.iqstratShaleStruct.GAMMA_MIN;
        }

        sTitle1   = "Colorlith";
        sTitle2   = "Rhomaa-Umaa";
        break;

      case iqstrat.iqstratTracksStruct._CLRLITH_RN:
        if (stLASFileData != null)
        {
          iRows     = stLASFileData.iRows;
          dNull     = stLASFileData.dNull;
          if (bSonic)
            depths  = stLASFileData.dTime;
          else
            depths  = stLASFileData.depths;

          stList    = stLASFileData.stList;

          dRed      = stLASFileData.dRhomaa;
          dRedMax   = RHOMAA_MAX;
          dRedMin   = RHOMAA_MIN;

          dGreen    = stLASFileData.dNPHI;
          dGreenMax = PHI_MAX;
          dGreenMin = PHI_MIN;

          dBlue     = stLASFileData.dGR;
          dBlueMax  = iqstrat.iqstratShaleStruct.GAMMA_MAX;
          dBlueMin  = iqstrat.iqstratShaleStruct.GAMMA_MIN;
        }

        sTitle1   = "Colorlith";
        sTitle2   = "Rhomaa-NPHI";
        break;

      case iqstrat.iqstratTracksStruct._CLRLITH_RT:
        if (stLASFileData != null)
        {
          iRows     = stLASFileData.iRows;
          dNull     = stLASFileData.dNull;
          if (bSonic)
            depths  = stLASFileData.dTime;
          else
            depths  = stLASFileData.depths;

          stList    = stLASFileData.stList;

          dRed      = stLASFileData.dRhomaa;
          dRedMax   = RHOMAA_MAX;
          dRedMin   = RHOMAA_MIN;

          dGreen    = stLASFileData.dDTmaa;
          dGreenMax = DTMAA_MAX;
          dGreenMin = DTMAA_MIN;

          dBlue     = stLASFileData.dGR;
          dBlueMax  = iqstrat.iqstratShaleStruct.GAMMA_MAX;
          dBlueMin  = iqstrat.iqstratShaleStruct.GAMMA_MIN;
        }

        sTitle1   = "Colorlith";
        sTitle2   = "Rhomaa-DTmaa";
        break;

      case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
        if (stLASFileData != null)
        {
          iRows     = stLASFileData.iRows;
          dNull     = stLASFileData.dNull;
          if (bSonic)
            depths  = stLASFileData.dTime;
          else
            depths  = stLASFileData.depths;

          stList    = stLASFileData.stList;

          dRed      = stLASFileData.dThU;
          dRedMax   = Math.log(THU_MAX)/Math.log(10.0);
          dRedMin   = Math.log(THU_MIN)/Math.log(10.0);

          dGreen    = stLASFileData.dThK;
          dGreenMax = Math.log(THK_MAX)/Math.log(10.0);
          dGreenMin = Math.log(THK_MIN)/Math.log(10.0);

          dBlue     = stLASFileData.dGR;
          dBlueMax  = iqstrat.iqstratShaleStruct.GAMMA_MAX;
          dBlueMin  = iqstrat.iqstratShaleStruct.GAMMA_MIN;
        }

        sTitle1   = "Colorlith";
        sTitle2   = "Th/U-Th/K";
        break;

      case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
        if (stLASFileData != null)
        {
          iRows     = stLASFileData.iRows;
          dNull     = stLASFileData.dNull;
          if (bSonic)
            depths  = stLASFileData.dTime;
          else
            depths  = stLASFileData.depths;

          dRed      = stLASFileData.dK;
          dRedMax   = K_MAX;
          dRedMin   = K_MIN;

          dGreen    = stLASFileData.dTh;
          dGreenMax = TH_MAX;
          dGreenMin = TH_MIN;

          dBlue     = stLASFileData.dU;
          dBlueMax  = U_MAX;
          dBlueMin  = U_MIN;
        }

        sTitle1   = "Colorlith";
        sTitle2   = "K-Th-U Hall";
        break;
    }

    // Start Lithology Track

    g.setColor(new Color(255, 255, 255));

    if (iRows > 0)
    {
      for (int i=0; i<iRows-1; i++)
      {
        if ((dRed[i] > dNull) && (dGreen[i] > dNull))
        {
          dRatio  = (double) iLogHeight * (depths[i]-depthStart) /
                                          (depthEnd - depthStart);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio  = (double) iLogHeight * (depths[i+1]-depthStart) /
                                          (depthEnd - depthStart);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs( iDepth2 - iDepth1 );

          // Set Blue Color Level

          dLog = dBlue[i];

          if (iSelected != iqstrat.iqstratTracksStruct._CLRLITH_HALL)
          {
            if (stList != null)
            {
              for (int n=0; n<stList.iCount; n++)
              {
                if ((depths[i] >= stList.stItem[n].depthStart) &&
                    (depths[i] <= stList.stItem[n].depthEnd))
                {
                  dBlueMin = stList.stItem[n].dGammaMin;
                  dBlueMax = stList.stItem[n].dGammaMax;
                }
              }
            }
          }

          if (dLog < dBlueMin) dLog = dBlueMin;
          if (dLog > dBlueMax) dLog = dBlueMax;

          iBlue = (int) (255 * (dLog-dBlueMin) / (dBlueMax-dBlueMin));

          // Set Red Color Level

          switch (iSelected)
          {
            case iqstrat.iqstratTracksStruct._CLRLITH_RU:
            case iqstrat.iqstratTracksStruct._CLRLITH_RN:
            case iqstrat.iqstratTracksStruct._CLRLITH_RT:
            case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
              dLog = dRed[i];
              break;

            case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
              dLog = Math.log(dRed[i]) / Math.log(10.0);
              break;
          }

          if (dLog > dRedMax) dLog = dRedMax;
          if (dLog < dRedMin) dLog = dRedMin;

          switch (iSelected)
          {
            case iqstrat.iqstratTracksStruct._CLRLITH_RU:
            case iqstrat.iqstratTracksStruct._CLRLITH_RN:
            case iqstrat.iqstratTracksStruct._CLRLITH_RT:
              iRed = (int) (255.0*(1.0 - ((dLog-dRedMin) / (dRedMax-dRedMin))));
              break;

            case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
            case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
              iRed = (int) (255.0 * ((dLog-dRedMin) / (dRedMax-dRedMin)));
              break;
          }

          if (iRed > 255) iRed = 255;
          if (iRed < 0)   iRed = 0;

          // Set Green Color Level

          switch (iSelected)
          {
            case iqstrat.iqstratTracksStruct._CLRLITH_RU:
            case iqstrat.iqstratTracksStruct._CLRLITH_RN:
            case iqstrat.iqstratTracksStruct._CLRLITH_RT:
            case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
              dLog = dGreen[i];
              break;

            case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
              dLog = Math.log(dGreen[i]) / Math.log(10.0);
              break;
          }

          if (dLog > dGreenMax) dLog = dGreenMax;
          if (dLog < dGreenMin) dLog = dGreenMin;

          switch (iSelected)
          {
            case iqstrat.iqstratTracksStruct._CLRLITH_RU:
            case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
            case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
              iGreen = (int)(255.0 * (dLog-dGreenMin) / (dGreenMax-dGreenMin) );
              break;
            case iqstrat.iqstratTracksStruct._CLRLITH_RN:
            case iqstrat.iqstratTracksStruct._CLRLITH_RT:
              iGreen = (int)(255.0 * (1.0 -
                              ((dLog-dGreenMin) / (dGreenMax-dGreenMin))));
              break;
          }

          if (iGreen > 255) iGreen = 255;
          if (iGreen < 0)   iGreen = 0;

          // Set RGB Color


          switch (iSelected)
          {
            case iqstrat.iqstratTracksStruct._CLRLITH_RU:
            case iqstrat.iqstratTracksStruct._CLRLITH_RN:
            case iqstrat.iqstratTracksStruct._CLRLITH_RT:
            case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
              g.setColor(new Color(iRed, iGreen, iBlue));
              break;
            case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
              g.setColor(new Color(iRed, iGreen, 0));
              break;
          }

          // Draw Colorlith Block

          if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 < iHeight))
          {
            if (iSelected == iqstrat.iqstratTracksStruct._CLRLITH_HALL)
            {
              if ((dRed[i] == dNull) || (dGreen[i] == dNull))
                g.setColor(Color.white);

              g.fillRect(iStartTrack, iDepth1, iWidthTrack, iDepth);
            }
            else
            {
              iForm = (iWidthTrack * (iBlue)) / 255;
              iForm = iWidthTrack - iForm / 2;

              if ((dRed[i] == dNull) || (dGreen[i] == dNull) || (dBlue[i] == dNull))
                g.setColor(Color.white);

              g.fillRect(iEndTrack - iForm, iDepth1, iForm, iDepth);
            }
          }
        }
      }
    }

    // Draw Gamma Ray Grid & Title

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString(sTitle1, LABELSTART+14, -1*(iStartTrack+2*iWidthTrack/3));

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g2.drawString(sTitle2, LABELSTART+6, -1*(iStartTrack+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track that was selected
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw(Graphics g, int iSelected, int iStartTrack, int iWidthTrack)
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_LAS;

    if (iSelected == iqstrat.iqstratTracksStruct._LITHOLOGY_GR)
    {
      switch (iSelected)
      {
        case iqstrat.iqstratTracksStruct._LITHOLOGY_GR:
          iColor = iqstrat.iqstratTracksStruct._SRC_LAS;
          break;
      }

      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawGammaTrack(g, iSelected, iStartTrack, iWidthTrack);
    }

    if ((iSelected == iqstrat.iqstratTracksStruct._POROSITY_IMAGER)      ||
        (iSelected == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER)   ||
        (iSelected == iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N)    ||
        (iSelected == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N) )
    {
      switch (iSelected)
      {
        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER:
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER:
        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N:
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N:
          iColor = iqstrat.iqstratTracksStruct._SRC_LAS;
          break;
      }

      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      if (iSelected == iqstrat.iqstratTracksStruct._POROSITY_IMAGER)
        drawImageTrack(g, iSelected, _PHIT, iStartTrack, iWidthTrack);

      if (iSelected == iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N)
        drawNonLinearImageTrack(g, iSelected, _PHIT, iStartTrack, iWidthTrack);

      if (iSelected == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER)
        drawImageTrack(g, iSelected, _RT, iStartTrack, iWidthTrack);

      if (iSelected == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N)
        drawNonLinearImageTrack(g, iSelected, _RT, iStartTrack, iWidthTrack);
    }

    if ((iSelected == iqstrat.iqstratTracksStruct._CLRLITH_RU) ||
        (iSelected == iqstrat.iqstratTracksStruct._CLRLITH_RN) ||
        (iSelected == iqstrat.iqstratTracksStruct._CLRLITH_RT) ||
        (iSelected == iqstrat.iqstratTracksStruct._CLRLITH_SPGR) ||
        (iSelected == iqstrat.iqstratTracksStruct._CLRLITH_HALL) )
    {
      switch (iSelected)
      {
        case iqstrat.iqstratTracksStruct._CLRLITH_RU:
        case iqstrat.iqstratTracksStruct._CLRLITH_RN:
        case iqstrat.iqstratTracksStruct._CLRLITH_RT:
        case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
        case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
          iColor = iqstrat.iqstratTracksStruct._SRC_LAS;
          break;
      }

      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawLithologyTrack(g, iSelected, iStartTrack, iWidthTrack);
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}