/*
 * @lithMathRules.java Version 1.1 07/30/2007
 *
 * Copyright (c) 2006 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.math;

/** Class lithMathRules
 *  <p> This Class will test the log curves and decide what the rock type is.
 *
 *  @version 1.1 07/30/2007
 *  @author  John R. Victorine
 */

public class lithMathRules
{
  public static final double PHI_N4 = -0.04; // -4 %
  public static final double PHI_N2 = -0.02; // -2 %
  public static final double PHI_0  = 0.0;   //  0 %
  public static final double PHI_2  = 0.02;  //  2 %
  public static final double PHI_4  = 0.04;  //  4 %
  public static final double PHI_5  = 0.05;  //  5 %
  public static final double PHI_6  = 0.06;  //  6 %
  public static final double PHI_7  = 0.07;  //  7 %
  public static final double PHI_8  = 0.08;  //  8 %
  public static final double PHI_9  = 0.09;  //  9 %
  public static final double PHI_10 = 0.10;  // 10 %
  public static final double PHI_11 = 0.11;  // 10 %
  public static final double PHI_15 = 0.15;  // 15 %
  public static final double PHI_20 = 0.20;  // 20 %
  public static final double PHI_25 = 0.25;  // 25 %
  public static final double PHI_30 = 0.30;  // 30 %
  public static final double PHI_35 = 0.35;  // 35 %
  public static final double PHI_45 = 0.45;  // 45 %

  public static final double RHO_COAL      = 1.8;
  public static final double RHO_SALT      = 2.03;
  public static final double RHO_GYPSUM    = 2.34;
  public static final double RHO_CLAY      = 2.40;
  public static final double RHO_DOLOMITE  = 2.84;
  public static final double RHO_ANHYDRITE = 2.9;
  public static final double RHO_IGNEOUS   = 2.91;

  public static final double PE_ANHYDRITE  = 4.5;
  public static final double PE_IRONSTONE  = 9.0;
  public static final double PE_IGNEOUS    = 6.0; //4.7;

  /* ======================================================================= *
   * ------------------------------- GET DATA ------------------------------ *
   * ======================================================================= */

  /** Method getLithology()
   * <p> This method will get the Lithology identifier of the rock
   * @param  iH2O       = Indicator for fresh or saline water
   * @param  dShaly     = Start of Shaly gamma ray value     ( 60 API)
   * @param  dShale     = Start of Shale gamma ray value     ( 70 API)
   * @param  dHotShale  = Start of Hot Shale gamma ray value (100 API)
   * @param  dGR        = Gamma Ray             (GR)   Log Value
   * @param  dNPHI      = Neutron Porosity      (NPHI) Log Value
   * @param  dRhob      = Bulk Density          (RHOB) Log Value
   * @param  dPE        = Photoelectric Factor  (PE)   Log Value
   * @param  dDT        = Acoustic Tranist Time (DT)   Log Value
   * @param  dNull      = Log Curve Null Value (NULL)
   * @return iLithology = the lithology identifier of the rock
   */

  public static int getLithology(int    iH2O,
                                 double dShaly,
                                 double dShale,
                                 double dHotShale,
                                 double dGR,
                                 double dNPHI,
                                 double dRhob,
                                 double dPE,
                                 double dDT,
                                 double dNull)
  {
    int    iLithology = lith.lithology.lithologySymbolsStruct._NONE;

	if (dNPHI > dNull)
	{
	  iLithology = getLithology( iH2O, dShaly, dShale, dHotShale,
	                             dGR, dNPHI, dRhob, dPE, dNull );
    }
	else if (dDT > dNull)
	{
	  iLithology = getLithology( iH2O, dShaly, dShale, dHotShale,
	                             dGR, dRhob, dDT, dNull );
	}

    return (iLithology);
  }

  /** Method getLithology()
   * <p> This method will get the Lithology identifier of the rock
   * @param  iH2O       = Indicator for fresh or saline water
   * @param  dShaly     = Start of Shaly gamma ray value     ( 60 API)
   * @param  dShale     = Start of Shale gamma ray value     ( 70 API)
   * @param  dHotShale  = Start of Hot Shale gamma ray value (100 API)
   * @param  dGR        = Gamma Ray             (GR)   Log Value
   * @param  dNPHI      = Neutron Porosity      (NPHI) Log Value
   * @param  dRhob      = Bulk Density          (RHOB) Log Value
   * @param  dPE        = Photoelectric Factor  (PE)   Log Value
   * @param  dNull      = Log Curve Null Value (NULL)
   * @return iLithology = the lithology identifier of the rock
   */

  public static int getLithology(int    iH2O,
                                 double dShaly,
                                 double dShale,
                                 double dHotShale,
                                 double dGR,
                                 double dNPHI,
                                 double dRhob,
                                 double dPE,
                                 double dNull)
  {
    int    iLithology = lith.lithology.lithologySymbolsStruct._NONE;
    double dDPHI      = PHI_0;
    double delta      = PHI_0;

    dDPHI  = math.mathLAS.computePHI( dRhob,
                                      math.mathLAS._DENSITY,
                                      math.mathLAS._CALCITE,
                                      math.mathLAS._FRESH );
    delta = dNPHI - dDPHI;

    // IF GAMMA RAY (API) is GREATER THAN or EQUAL to HOT SHALE (100 API) THEN

    if (dGR >= dHotShale)
    {
	  // ** LITHOLOGY is SHALE **

      iLithology = lith.lithology.lithologySymbolsStruct._SHALE;

      // IF POROSITY DIFFERENCE is LESS THAN or EQUAL to 2% (0.02 PU) AND
      //    NEUTRON POROSITY    is LESS THAN 2% (0.02 PU) THEN
      // ** LITHOLOGY is GRANITE **

      if ((delta <= PHI_2) && (Math.abs(dNPHI) < PHI_2))
          iLithology = lith.lithology.lithologySymbolsStruct._GRANITE;
    }

    // GAMMA RAY (API) is Greater Than or equal to SHALE (70 API)
    //                 but Less Than HOT SHALE (100 API)

    else if ((dGR >= dShale) && (dGR < dHotShale))
    {
	  // ** LITHOLOGY is SHALE **

      iLithology = lith.lithology.lithologySymbolsStruct._SHALE;

      // IF POROSITY DIFFERENCE is GREATER THAN or EQUAL to -4% (-0.04 PU) AND
      //    POROSITY DIFFERENCE is LESS THAN or EQUAL to 4% (0.04 PU) AND
      //    NEUTRON POROSITY    is GREATER THAN or EQUAL to 10% (0.1 PU) AND
      //    NEUTRON POROSITY    is LESS THAN or EQUAL to 30% (0.3 PU) THEN
	  // ** LITHOLOGY is SHALY SANDSTONE (SAND) **

      if ((delta >= PHI_N4) && (delta <=PHI_4) &&
          (Math.abs(dNPHI) >= PHI_10) && (Math.abs(dNPHI) <= PHI_30))
        iLithology = lith.lithology.lithologySymbolsStruct._SHALY_SAND;

      // IF POROSITY DIFFERENCE is LESS THAN 0% (0.0 PU) AND
      //    NEUTRON POROSITY    is LESS THAN 2% (0.02 PU) THEN
      // ** LITHOLOGY is ARKOSE **

      if ((delta < PHI_0) && (Math.abs(dNPHI) < PHI_2))
        iLithology = lith.lithology.lithologySymbolsStruct._ARKOSE;
    }

    // ELSE IF GAMMA RAY (API) is a CLEAN FORMATION (LESS THAN 70 API) THEN

    else
    {
	  // IF POROSITY DIFFERENCE is LESS THAN -4% (-0.04 PU) THEN

      if (delta < PHI_N4)
      {
        // ** LITHOLOGY is SANDSTONE (SAND) **

        iLithology = lith.lithology.lithologySymbolsStruct._SANDSTONE;

        // IF GAMMA RAY (API) IS SHALY (60 API to 70 API) THEN
	    // ** LITHOLOGY is SHALY SANDSTONE (SAND) **

        if ( (dGR >= dShaly) && (dGR < dShale))
          iLithology = lith.lithology.lithologySymbolsStruct._SHALY_SAND;

        // IF BULK DENSITY (GM/CC) is LESS THAN or EQUAL to HALITE (2.03 GM/CC) AND
        //    NEUTRON PORISITY (PU) is LESS THAN or EQUAL to 5.0% (0.05 PU) THEN
        // ** LITHOLOGY is HALITE (SALT) **

        if ((dRhob <= RHO_SALT) && (Math.abs(dNPHI) <= PHI_5))
          iLithology = lith.lithology.lithologySymbolsStruct._HALITE;

        // IF BULK DENSITY (GM/CC) is LESS THAN COAL (1.8 GM/CC) AND
        //    NEUTRON PORISITY (PU) is GREATER THAN 35.0% (0.35 PU) THEN
        // ** LITHOLOGY is COAL **

        if ((dRhob < RHO_COAL) && (dNPHI > PHI_35))
          iLithology = lith.lithology.lithologySymbolsStruct._COAL;
      }

	  // IF POROSITY DIFFERENCE is GREATER THAN or EQUAL to -4% (-0.04 PU) AND
	  //    POROSITY DIFFERENCE is LESS THAN or EQUAL to 4% (0.04 PU) THEN

      else if ((delta >= PHI_N4) && (delta <=PHI_4))
      {
        // ** LITHOLOGY is LIMESTONE **

        iLithology = lith.lithology.lithologySymbolsStruct._LIMESTONE;

        // IF GAMMA RAY (API) IS SHALY (60 API to 70 API) THEN
	    // ** LITHOLOGY is SHALY CARBONATE (LIMESTONE)**

        if ( (dGR >= dShaly) && (dGR < dShale))
          iLithology = lith.lithology.lithologySymbolsStruct._SHALY_CARB;
      }

	  // IF POROSITY DIFFERENCE is GREATER THAN 4% (0.04 PU) THEN

      else if (delta > PHI_4)
      {
        // ** LITHOLOGY is DOLOMITE **

        iLithology = lith.lithology.lithologySymbolsStruct._DOLOMITE;

        // IF GAMMA RAY (API) IS SHALY (60 API to 70 API) THEN
	    // ** LITHOLOGY is SHALY CARBONATE (DOLOMITE)**

        if ((dGR >= dShaly) && (dGR < dShale))
          iLithology = lith.lithology.lithologySymbolsStruct._SHALY_CARB;

        // IF BULK DENSITY (GM/CC) is LESS THAN or EQUAL to GYPSUM (2.35 GM/CC) THEN
        // ** LITHOLOGY is GYPSUM **

        if (dRhob <= lith.math.lithMath._MINERAL[lith.math.lithMath._GYPSUM][0])
          iLithology = lith.lithology.lithologySymbolsStruct._GYPSUM;

        // IF BULK DENSITY (GM/CC) is GREATER THAN CLAY (2.4 GM/CC) AND
        //    NEUTRON POROSITY is GREATER THAN 45% (0.45 PU) THEN
        // ** LITHOLOGY is SCHIST **

        if ((dRhob > RHO_CLAY) && (dNPHI > PHI_45))
          iLithology = lith.lithology.lithologySymbolsStruct._SCHIST;

        // IF BULK DENSITY (GM/CC) is GREATER THAN DOLOMITE (2.84 GM/CC) THEN

        if (dRhob > RHO_DOLOMITE)
        {
          // ** LITHOLOGY is DOLOMITE **

          iLithology = lith.lithology.lithologySymbolsStruct._DOLOMITE;

          // IF GAMMA RAY (API) IS SHALY (60 API to 70 API) THEN
  	      // ** LITHOLOGY is SHALY CARBONATE (DOLOMITE)**

          if ((dGR >= dShaly) && (dGR < dShale))
            iLithology = lith.lithology.lithologySymbolsStruct._SHALY_CARB;
        }

        // IF BULK DENSITY (GM/CC) is GREATER THAN ANHYDRITE (2.9 GM/CC) AND
        //    PHOTOELECTRIC (BARNS/E) is GREATER THAN ANHYDRITE (4.5 BARNS/E) THEN
        // ** LITHOLOGY is ANHYDRITE **

        if (((dRhob > RHO_ANHYDRITE) || (dPE > PE_ANHYDRITE)) &&
            (Math.abs(dNPHI) <= PHI_5))
          iLithology = lith.lithology.lithologySymbolsStruct._ANHYDRITE;

        // IF BULK DENSITY (GM/CC) is GREATER THAN ANHYDRITE (2.9 GM/CC) AND
        //    PHOTOELECTRIC (BARNS/E) is GREATER THAN IRONSTONE (9.0 BARNS/E) THEN
        // ** LITHOLOGY is IRONSTONE **

        if (((dRhob > RHO_ANHYDRITE) || (dPE > PE_IRONSTONE)) &&
            (dNPHI > PHI_15))
          iLithology = lith.lithology.lithologySymbolsStruct._IRONSTONE;

        // IF BULK DENSITY (GM/CC) is GREATER THAN IGNEOUS (2.91 GM/CC) AND
        //    PHOTOELECTRIC (BARNS/E) is GREATER THAN IGNEOUS (6.0 BARNS/E) THEN
        // ** LITHOLOGY is IGNEOUS ROCK **

        if (((dRhob > RHO_IGNEOUS) || (dPE > PE_IGNEOUS)) &&
            (dNPHI > PHI_10))
          iLithology = lith.lithology.lithologySymbolsStruct._IGNEOUS;
      }
    }

    return (iLithology);
  }

  /** Method getLithology()
   * <p> This method will get the Lithology identifier of the rock
   * @param  iH2O       = Indicator for fresh or saline water
   * @param  dShaly     = Start of Shaly gamma ray value     ( 60 API)
   * @param  dShale     = Start of Shale gamma ray value     ( 70 API)
   * @param  dHotShale  = Start of Hot Shale gamma ray value (100 API)
   * @param  dGR        = Gamma Ray             (GR)   Log Value
   * @param  dNPHI      = Neutron Porosity      (NPHI) Log Value
   * @param  dRhob      = Bulk Density          (RHOB) Log Value
   * @param  dPE        = Photoelectric Factor  (PE)   Log Value
   * @param  dNull      = Log Curve Null Value (NULL)
   * @return iLithology = the lithology identifier of the rock
   */

  public static int getLithology(int    iH2O,
                                 double dShaly,
                                 double dShale,
                                 double dHotShale,
                                 double dGR,
                                 double dRhob,
                                 double dDT,
                                 double dNull)
  {
    int    iLithology = lith.lithology.lithologySymbolsStruct._NONE;
    double dDPHI      = PHI_0;
    double dSPHI      = PHI_0;
    double dPHI       = PHI_0;

    dDPHI  = math.mathLAS.computePHI( dRhob,
                                      math.mathLAS._DENSITY,
                                      math.mathLAS._CALCITE,
                                      math.mathLAS._FRESH );

    dSPHI  = math.mathLAS.computePHI( dDT,
                                      math.mathLAS._SONIC,
                                      math.mathLAS._CALCITE,
                                      math.mathLAS._FRESH );

    dPHI   = Math.sqrt(dSPHI*dSPHI + dDPHI*dDPHI);

    // IF GAMMA RAY (API) is GREATER THAN or EQUAL to HOT SHALE (100 API) THEN

    if (dGR >= dHotShale)
    {
	  // ** LITHOLOGY is SHALE **

      iLithology = lith.lithology.lithologySymbolsStruct._SHALE;

      // IF Average POROSITY is LESS THAN 2% (0.02 PU) THEN
      // ** LITHOLOGY is GRANITE **

      if (Math.abs(dPHI) < PHI_2)
          iLithology = lith.lithology.lithologySymbolsStruct._GRANITE;
    }

    // GAMMA RAY (API) is Greater Than or equal to SHALE (70 API)
    //                 but Less Than HOT SHALE (100 API)

    else if ((dGR >= dShale) && (dGR < dHotShale))
    {
	  // ** LITHOLOGY is SHALE **

      iLithology = lith.lithology.lithologySymbolsStruct._SHALE;

      // IF Average POROSITY is LESS THAN 2% (0.02 PU) THEN
      // ** LITHOLOGY is GRANITE **

      if (Math.abs(dPHI) < PHI_2)
        iLithology = lith.lithology.lithologySymbolsStruct._ARKOSE;
    }

    // GAMMA RAY (API) is Greater Than or equal to SHALE (60 API)
    //                 but Less Than SHALE (70 API)

    else if ((dGR >= dShaly) && (dGR < dShale))
    {
	  // ** LITHOLOGY is SHALE **

      iLithology = lith.lithology.lithologySymbolsStruct._SHALY_CARB;

      // IF ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 51.3 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 55.6    THEN

      if ((dDT >= 51.3) && (dDT <= 55.6))
        iLithology = lith.lithology.lithologySymbolsStruct._SHALY_SAND;

      // IF ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 38.5 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 47.6    THEN

      if ((dDT >= 38.5) && (dDT <= 47.6))
        iLithology = lith.lithology.lithologySymbolsStruct._SHALY_CARB;
    }

    // ELSE IF GAMMA RAY (API) is a CLEAN FORMATION (LESS THAN 70 API) THEN

    else
    {
	  // ** LITHOLOGY is LIMESTONE **

      iLithology = lith.lithology.lithologySymbolsStruct._LIMESTONE;

      // IF ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 51.3 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 55.6    THEN

      if ((dDT >= 51.3) && (dDT <= 55.6))
        iLithology = lith.lithology.lithologySymbolsStruct._SANDSTONE;

      // IF ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 43.5 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 47.6    THEN

      if ((dDT >= 43.5) && (dDT <= 47.6))
        iLithology = lith.lithology.lithologySymbolsStruct._LIMESTONE;

      // IF ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 43.5 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 47.6    THEN

      if ((dDT >= 38.5) && (dDT < 43.5))
        iLithology = lith.lithology.lithologySymbolsStruct._DOLOMITE;

      // IF BULK DENSITY (GM/CC) is GREATER THAN ANHYDRITE (2.9 GM/CC)  AND
      //    ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 49.5 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 50.5    AND
      //    AVERAGE POROSITY is LESS THAN and EQUAL to 5% (0.05 PU)     THEN
      // ** LITHOLOGY is ANHYDRITE **

      if ((dRhob > RHO_ANHYDRITE) && (dDT >= 49.5) && (dDT <= 50.5) &&
          (Math.abs(dPHI) <= PHI_5))
        iLithology = lith.lithology.lithologySymbolsStruct._ANHYDRITE;

      // IF BULK DENSITY (GM/CC) is LESS THAN or EQUAL to GYPSUM (2.35 GM/CC) AND
      //    ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 51.5       AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 52.5          AND
      //    AVERAGE POROSITY is GREATER THAN 30% (0.3 PU)                     THEN
      // ** LITHOLOGY is GYPSUM **

      if ((dRhob <= lith.math.lithMath._MINERAL[lith.math.lithMath._GYPSUM][0]) &&
          (dDT >= 51.5) && (dDT <= 52.5) && (Math.abs(dPHI) > PHI_30))
        iLithology = lith.lithology.lithologySymbolsStruct._GYPSUM;

      // IF BULK DENSITY (GM/CC) is LESS THAN or EQUAL to HALITE (2.03 GM/CC) AND
      //    ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 66.5       AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 67.5          AND
      //    AVERAGE PORISITY (PU) is LESS THAN or EQUAL to 5.0% (0.05 PU)     THEN
      // ** LITHOLOGY is HALITE (SALT) **

      if ((dRhob <= RHO_SALT) && (dDT >= 66.5) && (dDT <= 67.5) && (Math.abs(dPHI) <= PHI_5))
        iLithology = lith.lithology.lithologySymbolsStruct._HALITE;

      // IF BULK DENSITY (GM/CC) is LESS THAN COAL (1.8 GM/CC)        AND
      //    ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 90 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 180   AND
      //    AVERAGE PORISITY (PU) is GREATER THAN 35.0% (0.35 PU)     THEN
      // ** LITHOLOGY is COAL **

      if ((dRhob < RHO_COAL) && (dDT >= 90) && (dDT <= 180) && (dPHI > PHI_35))
        iLithology = lith.lithology.lithologySymbolsStruct._COAL;

      // IF BULK DENSITY (GM/CC) is GREATER THAN IGNEOUS (2.91 GM/CC)   AND
      //    ACOUSTIC Transit Time (DT) is GREATER THAN or EQUAL to 39.5 AND
      //    ACOUSTIC Transit Time (DT) is LESS THAN or EQUAL to 49.5    AND
      //    AVERAGE PORISITY (PU) is GREATER THAN 5.0% (0.05 PU)        THEN
      // ** LITHOLOGY is IGNEOUS ROCK **

      if ((dRhob > RHO_IGNEOUS) && (dDT >= 39.5) && (dDT <= 49.5) && (dPHI > PHI_5))
        iLithology = lith.lithology.lithologySymbolsStruct._IGNEOUS;
    }

    return (iLithology);
  }
}