/*
 * @lithMath.java Version 1.1 05/05/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.math;

/** Class lithMath
 *  <p> This Class will perform the basic math for the lithology plots.
 *
 *  @version 1.1 05/05/2008
 *  @author  John R. Victorine
 */

public class lithMath
{
  public static final int _QUARTZ         = math.mathLAS._QUARTZ;    // SILICATE  - Quartz
  public static final int _CALCITE        = math.mathLAS._CALCITE;   // CARBONATE - Calcite
  public static final int _DOLOMITE       = math.mathLAS._DOLOMITE;  // CARBONATE - Dolomite
  public static final int _ANHYDRITE      = 3;  // EVAPORITE - Anhydrite
  public static final int _GYPSUM         = 4;  // EVAPORITE - Gypsum
  public static final int _HALITE         = 5;  // EVAPORITE - Halite
  public static final int _HIGH_Z_CLAY    = 6;  // CLAY      - Illite
  public static final int _LOW_Z_CLAY     = 7;  // CLAY      - Smectite
  public static final int _COAL           = 8;  // ORGANIC   - Coal
  public static final int _FELDSPAR       = 9;  // SILICATE  - Orthoclase
  public static final int _Na_PLAGIOCLASE = 10; // IGNEOUS   - Albite
  public static final int _Ca_PLAGIOCLASE = 11; // IGNEOUS   - Anorthite
  public static final int _ULTRA_MAFIC    = 12; // IGNEOUS   - Ultra Mafic
  public static final int _SIO2           = 13; // IGNEOUS   - Quartz
  public static final int _MICA           = 14; // IGNEOUS   - Muskovite
  public static final int _SHALE          = 15; // SHALE     - Smectite
  public static final int _CaCO3          = 16; // CARBONATE - Calcite

  public static final int _ROWS           = 17; // Total Mineral

  public static final String MINERALS[] =
  {
    "Quartz",
  // Carbonates
    "Calcite",
    "Dolomite",
  //Evaporites
    "Anhydrite",
    "Gypsum",
    "Halite",
  // Clays
    "High Z Clay", // "Illite",
    "Low Z Clay",  // "Smectite"
  // Coal
    "Coal",
  // igneous
    "Feldspar",
    "Anorthite",
    "Ultramafic",
    "QUARTZ No Porosity",
    "Biotite",
  // other
    "SHALE"
  };

  public static final int _RHOB     = 0;   // Bulk Density Log Curve
  public static final int _RHOMAA   = 1;   // Computed Matrix Density
  public static final int _NPHI     = 2;   // Neutron Porosity Log Curve
  public static final int _UMAA     = 3;   // Computed Photoelectric Factor
  public static final int _PHI_DIFF = 4;   // Computed Porosity Difference
  public static final int _GR       = 5;   // Gamma Ray Log Curve
  public static final int _TH       = 6;   // Thorium Log Curve
  public static final int _U        = 7;   // Uranium Log Curve
  public static final int _K        = 8;   // Potassium Log Curve
  public static final int _DTC      = 9;   // Sonic
  public static final int _COLUMNS  = 11;  // Total Columns
/*
  public static final double _MINERAL[][] =
  {
  //  RHOB  Rhomaa NPHI  Umaa    DelPHI   GR     Th    U     K     DT
    { 2.64, 2.65, -2.1,  4.85,   -6.2,    1.0,  0.0,  0.1,  0.0,  50.5 }, // Quartz
    { 2.71, 2.71,  0.0, 13.77,    0.0,   11.0,  0.0,  1.4,  0.0,  48.1 }, // Calcite
    { 2.87, 2.87,  0.5,  9.0,    10.4,    8.0,  0.1,  0.9,  0.0,  44.0 }, // Dolomite
    { 2.98, 2.98, -0.7, 14.9,    15.1,    0.0,  0.0,  0.0,  0.0,  54.0 }, // Anhydrite
    { 2.35, 3.15, 57.6, 15.71,   36.5,    0.0,  0.0,  0.0,  0.0,  52.5 }, // Gypsum
    { 2.03, 2.04, -1.8, 13.3,   -41.6,    0.0,  0.0,  0.0,  0.0,  67.0 }, // Halite
    { 2.77, 3.06, 15.8, 11.7,    19.3,  160.0, 12.0,  4.8,  4.5, 106.0 }, // Illite ( High Z Clay )
    { 2.64, 3.02, 45.1,  6.2,    41.0,  104.0, 19.0,  3.2,  0.1, 211.7 }, // Kaolinite ( Low Z Clay )
    { 1.24, 1.889,60.0,  0.815, -26.0,    0.0,  0.0,  0.0,  0.0, 120.0 }, // Bituminus Coal
    { 2.54, 2.614,-1.1,  7.59,  -11.0,  171.0,  1.1,  0.4, 10.2,  69.0 }, // Orthoclase ( K-Feldspar )
    { 2.58, 2.631,-1.3,  4.491,  -8.9,    8.0,  0.0,  0.0,  0.5,  47.2 }, // Albite ( Na-Plagioclase )
    { 2.74, 2.73, -1.6,  8.7,     0.2,    1.0,  0.0,  0.1,  0.0,  45.0 }, // Anorthite ( Ca-Plagioclase )
    { 3.09, 3.02, 24.0, 13.95,   20.0,   15.0,  2.0,  0.35, 0.54, 49.4 }, // Ultra-Mafic
    { 2.64, 2.65, -2.1,  4.85,   -6.2,    1.0,  0.0,  0.1,  0.0,  50.5 }, // Quartz
    { 2.83, 3.06, 16.5,  7.35,   23.5,  130.0,  0.0,  0.7,  7.8,  47.2 }, // Muscovite
    { 2.12, 3.02, 44.0,  6.0,   15.0,   132.16,17.0, 4.85,  1.6,  55.5 }, // Smectite   - Clay
  };
*/

  public static final double _MINERAL[][] =
  {
  //  RHOB  Rhomaa NPHI  Umaa      DelPHI   GR     Th     U     K     DT
  // (gm/cc)(gm/cc)(PU) (barns/cc)  (PU)   (API)  (ppm) (ppm)  (%) (usec/ft)
  // -- Sedimentary Rocks
    { 2.64,  2.65,  18.0,  4.8,     20.0,  14.83,  2.0,  0.7,  0.08, 55.5 }, // SS  (Quartz)    - Silicate
//    { 2.71,  2.71,   0.1, 13.8,    -10.0,   4.9,   0.0,  0.0,  0.3,  47.6 }, // LM  (Limestone) - Carbonate
    { 2.71,  2.71,   0.1, 13.8,    -10.0,   4.9,   0.0,  0.0,  0.3,  47.5 }, //48.1 }, // LM  (Limestone) - Carbonate
    { 2.85,  2.88,  18.0,  9.0,     20.0,   1.14,  0.0,  0.0,  0.07, 43.5 }, // DOL (Dolomite)  - Carbonate
    { 2.98,  2.98,   1.0, 14.9,    -10.0,   0.0,   0.0,  0.0,  0.0,  50.0 }, // ANH (Anhydrite) - Evaporite
    { 2.35,  3.1,   51.0, 18.38,   -30.0,   0.0,   0.0,  0.0,  0.0,  52.5 }, // GYP (Gypsum)    - Evaporite
    { 2.04,  2.04,  -2.0, 13.8,     40.0,   0.0,   0.0,  0.0,  0.0,  67.0 }, // NaCl (Salt)     - Evaporite
    { 2.52,  3.006, 30.0, 13.0,      3.1, 234.34, 17.5,  6.95, 6.7, 106.0 }, // Illite  (High Z)- Clay
    { 2.12,  3.08,  44.0,  7.0,     15.0, 132.16, 17.0,  4.85, 1.6, 211.7 }, // Smectite(Low Z) - Clay
    { 1.24,  1.5,   60.0,  0.42,    40.0,   0.0,   0.0,  0.0,  0.0, 105.0 }, // Coal            - Organic
  // -- Igneous Rocks
    { 2.54,  2.614, -1.1,  7.59,   -11.0, 171.0,   1.1,  0.4, 10.2,  69.0 }, // Orthoclase ( K-Feldspar )
    { 2.58,  2.631, -1.3,  4.491,   -8.9,   8.0,   0.0,  0.0,  0.5,  47.2 }, // Albite     ( Na-Plagioclase )
    { 2.74,  2.73,  -1.6,  8.7,      0.2,   1.0,   0.0,  0.1,  0.0,  45.0 }, // Anorthite  ( Ca-Plagioclase )
    { 3.09,  3.02,  24.0, 13.95,    20.0,  15.0,   2.0,  0.35, 0.54, 49.4 }, // Ultra-Mafic
    { 2.64,  2.65,  -2.1,  4.85,    -6.2,   1.0,   0.0,  0.1,  0.0,  50.5 }, // Quartz
    { 2.83,  3.06,  16.5,  7.35,    23.5, 130.0,   0.0,  0.7,  7.8,  47.2 }, // Muscovite
  // -- Other Rocks
    { 2.12,  3.02,  44.0,  6.0,     15.0, 132.16, 17.0,  4.85, 1.6,  55.5 }, // Smectite   - Clay
    { 2.71,  2.71,   0.1, 13.8,    -10.0,   0.0,   0.0,  0.0,  0.3,  52.5 }, // Calcite
  };

  // Fluid Constants

  public static final int _FRESH_H2O = math.mathLAS._FRESH;
  public static final int _SALTY_H2O = math.mathLAS._SALT;
  public static final int _OIL_MUD   = math.mathLAS._OIL;

  public static final double RHO_f[]   = {   1.0,   1.1, 0.985 }; // (gm/cc)
  public static final double DT_f[]    = { 189.0, 185.0, 204.5 }; // (usec/ft)
  public static final double U_f[]     = { 0.398, 1.36,  0.0 };   // (barns/cc)

  /* ----------------------------------------------------------------------- *
   * ------- Mineral Order depending on the type of Matrix to create ------- *
   * ----------------------------------------------------------------------- */

  // Dolomite-Quartz-Calcite Triangle
  // -- Rhomaa-Umaa

  public static final int _RU[]      = { _DOLOMITE, _CALCITE, _QUARTZ };
  public static final int _RU_COL[]  = { _RHOMAA, _UMAA };
  public static final int _RU_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._DOLOMITE,  // Blue
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE  // Yellow
  };

  // Dolomite-Quartz-Calcite Triangle
  // -- Rhomaa-NPHI

  public static final int _PD_COL[]   = { _RHOMAA, _NPHI };
  public static final int _PDGR_COL[] = { _RHOMAA, _NPHI, _GR };
//  public static final int _PD_COL[]   = { _RHOMAA, _PHI_DIFF };
//  public static final int _PDGR_COL[] = { _RHOMAA, _PHI_DIFF, _GR };

  // Dolomite-Quartz-Calcite Triangle
  // -- Rhomaa-Umaa

  public static final int _RT_COL[]   = { _RHOMAA, _DTC };
  public static final int _RTGR_COL[] = { _RHOMAA, _DTC, _GR };

  // Shale-Dolomite-Quartz-Calcite Quadrangle
  // -- Rhomaa-NPHI
  public static final int _RUo[]      = { _SHALE,  _DOLOMITE,
                                         _CALCITE, _QUARTZ };
  public static final int _RUo_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._SHALE,     // Light Gray
    lith.lithology.lithologySymbolsStruct._DOLOMITE,  // Blue
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE  // Yellow
  };

  // Low Z Clay-High Z Clay- Silicate Carbonate Quadrangle

  public static final int _RUGR[] = { _LOW_Z_CLAY, _HIGH_Z_CLAY ,
                                      _CALCITE,    _QUARTZ };
  public static final int _RUGR_COL[]  = { _RHOMAA, _UMAA, _GR };
  public static final int _RUGR_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._CLAY,       // Dark Gray
    lith.lithology.lithologySymbolsStruct._SHALE,      // Light Gray
    lith.lithology.lithologySymbolsStruct._LIMESTONE,  // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE   // Yellow
  };

  // High Z Clay-Arkose-Silicate-Carbonate Quadrangle

  public static final int _ARKOSE[] = { _HIGH_Z_CLAY, _CALCITE, _SIO2, _FELDSPAR };
  public static final int _ARKOSE_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._SHALE,     // Light Gray
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE, // Yellow
    lith.lithology.lithologySymbolsStruct._FELDSPAR   // Violet
  };

  public static final int _ARKOSE_RT[] = { _HIGH_Z_CLAY, _SIO2, _FELDSPAR };
  public static final int _ARKOSE_RT_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._SHALE,     // Light Gray
    lith.lithology.lithologySymbolsStruct._SANDSTONE, // Yellow
    lith.lithology.lithologySymbolsStruct._FELDSPAR   // Violet
  };

  // Igneous Rock Mica-Na Plagioclase-Quartz-K Feldspar Quadrangle

  public static final int _GRANITE[] = { _MICA, _Na_PLAGIOCLASE, _SIO2, _FELDSPAR };
  public static final int _GRANITE_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._MUSCOVITE,
    lith.lithology.lithologySymbolsStruct._PLAGIOCLASE,
    lith.lithology.lithologySymbolsStruct._SANDSTONE,
    lith.lithology.lithologySymbolsStruct._FELDSPAR
  };

  // Igneous Rock Ultra Mafic-Ca Plagioclase-Quartz-K Feldspar Quadrangle

  public static final int _IGNEOUS[] = { _ULTRA_MAFIC, _Ca_PLAGIOCLASE, _SIO2, _FELDSPAR };
  public static final int _IGNEOUS_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._ULTRAMAFIC,  // Green,
    lith.lithology.lithologySymbolsStruct._PLAGIOCLASE, // Violet,
    lith.lithology.lithologySymbolsStruct._SANDSTONE,   // Yellow
    lith.lithology.lithologySymbolsStruct._FELDSPAR     // Pink
  };

  // Dolomite-Anydrite-Calcite Triangle

  public static final int _ANHY[]      = { _DOLOMITE, _CALCITE, _ANHYDRITE };
  public static final int _ANHY_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._DOLOMITE,  // Blue
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._ANHYDRITE  // Pink
  };

  // Dolomite-Calcite-Gypsum Triangle

  public static final int _GYP[]      = { _DOLOMITE, _CALCITE, _GYPSUM };
  public static final int _GYP_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._DOLOMITE,  // Blue
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._GYPSUM     // Violet
  };

  // Calcite-Quartz-Halite-Coal Quadrangle

  public static final int _SALT_COAL[] = { _CALCITE, _QUARTZ, _HALITE, _COAL };
  public static final int _SALT_COAL_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE, // Yellow
    lith.lithology.lithologySymbolsStruct._HALITE,    //
    lith.lithology.lithologySymbolsStruct._COAL       // Black
  };

  // Calcite-Quartz-Halite Triangle

  public static final int _SALT[]      = { _CALCITE, _QUARTZ, _HALITE };
  public static final int _SALT_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE, // Yellow
    lith.lithology.lithologySymbolsStruct._HALITE     //
  };

  // Calcite-Quartz-Coal Triangle

  public static final int _BLK_COAL[]      = { _CALCITE, _QUARTZ, _COAL };
  public static final int _BLK_COAL_LITH[] =
  {
    lith.lithology.lithologySymbolsStruct._LIMESTONE, // Cyan
    lith.lithology.lithologySymbolsStruct._SANDSTONE, // Yellow
    lith.lithology.lithologySymbolsStruct._COAL       // Black
  };

  /* ======================================================================= *
   * -------------------------- LOG DATA MATRIX ---------------------------- *
   * ======================================================================= */

  /** Method getLog()
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid = Identifier of type of fluid to use
   * @param  dRHOB = Bulk density log value
   * @param  dPE   = Photoelectric Factor log value
   * @param  dNPHI = Neutron Porosity
   * @param  dGR   = Gammma Ray
   * @return dIn   = the Input matrix for the compositional analysis
   */

  public static double[] getLog( int matrix, int fluid,  double dRHOB,
                                 double dPE, double dNPHI, double dGR )
  {
    double dIn[] = {0.0, 0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    if (dPE > 0.0)
      dIn[1] = math.mathLAS.computeUmaa(fluid, dRHOB, dPE, dPHI);
    else
      dIn[1] = 100.0*dNPHI;
    dIn[2] = dGR;
    dIn[3] = 1.0;

    return (dIn);
  }

  /* ----------------------------------------------------------------------- *
   * ------------------ Dolomite Quartz Calcite Triangle ------------------- *
   * ----------------------------------------------------------------------- */

  /** Method getRU()
   *  { Get RHOMAA-UMAA Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid  = Identifier of type of fluid to use
   * @param  dRHOB  = Bulk density log value
   * @param  dPE    = Photoelectric Factor log value
   * @param  dNPHI  = Neutron Porosity
   * @return dIn    = the Input matrix for the compositional analysis
   */

  public static double[] getRU( int matrix, int fluid,  double dRHOB,
                                double dPE, double dNPHI )
  {
    double dIn[] = {0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = math.mathLAS.computeUmaa(fluid, dRHOB, dPE, dPHI);
    dIn[2] = 1.0;

    return (dIn);
  }

  /** Method getRT()
   *  { Get RHOMAA-DTMAA Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid  = Identifier of type of fluid to use
   * @param  dRHOB  = Bulk density log value
   * @param  dDT    = Acoustic Transit Time value
   * @return dIn    = the Input matrix for the compositional analysis
   */

  public static double[] getRT( int matrix, int fluid,  double dRHOB, double dDT )
  {
    double dIn[] = {0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dSPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dSPHI  = math.mathLAS.computePHI( dDT, math.mathLAS._SONIC,
                                      matrix, fluid );

//    dPHI   = math.mathLAS.computeAvgPHI( dSPHI,  dDPHI );
//    dPHI   = dSPHI; // Using Sonic instead of the Average
    dPHI   = Math.sqrt(dSPHI*dSPHI + dDPHI*dDPHI);

    dIn[0] = math.mathLAS.computeRhomaa( fluid, dRHOB, dPHI);
//    dIn[1] = dDT;
    dIn[1] = math.mathLAS.computeDTMAA( fluid, dDT, dPHI );
    dIn[2] = 1.0;

    return (dIn);
  }

  /** Method getRT()
   *  { Get RHOMAA-UMAA Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid  = Identifier of type of fluid to use
   * @param  dRHOB  = Bulk density log value
   * @param  dDT    = Acoustic Transit Time value
   * @return dIn    = the Input matrix for the compositional analysis
   */

  public static double[] getRT( int matrix, int fluid,  double dRHOB,
                                double dDT, double dNPHI )
  {
    double dIn[] = {0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa( fluid, dRHOB, dPHI);
//    dIn[1] = dDT;
    dIn[1] = math.mathLAS.computeDTMAA( fluid, dDT, dPHI );
    dIn[2] = 1.0;

    return (dIn);
  }

  /** Method getPD()
   * { Get Porosity Difference-Neutron Porosity Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid  = Identifier of type of fluid to use
   * @param  dRHOB  = Bulk density log value
   * @param  dNPHI  = Neutron Porosity
   * @return dIn    = the Input matrix for the compositional analysis
   */

  public static double[] getPD( int matrix, int fluid,
                                double dRHOB, double dNPHI )
  {
    double dIn[] = {0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = 100.0*dNPHI;
    dIn[2] = 1.0;

    return (dIn);
  }

  /* ----------------------------------------------------------------------- *
   * ---------------------- Clay Quartz Calcite Area ----------------------- *
   * ----------------------------------------------------------------------- */

  /** Method getRUGR()
   * { Get RHOMAA-UMAA-Gamma Ray Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid = Identifier of type of fluid to use
   * @param  dRHOB = Bulk density log value
   * @param  dPE   = Photoelectric Factor log value
   * @param  dNPHI = Neutron Porosity
   * @param  dGR   = Gammma Ray
   * @return dIn   = the Input matrix for the compositional analysis
   */

  public static double[] getRUGR( int matrix, int fluid,  double dRHOB,
                                  double dPE, double dNPHI, double dGR )
  {
    double dIn[] = {0.0, 0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = math.mathLAS.computeUmaa(fluid, dRHOB, dPE, dPHI);
    dIn[2] = dGR;
    dIn[3] = 1.0;

    return (dIn);
  }

  /** Method getPDGR()
   * { Get Porosity Difference-Neutron Porosity-Gamma Ray Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid = Identifier of type of fluid to use
   * @param  dRHOB = Bulk density log value
   * @param  dNPHI = Neutron Porosity
   * @param  dGR   = Gammma Ray
   * @return dIn   = the Input matrix for the compositional analysis
   */

  public static double[] getPDGR( int matrix, int fluid,  double dRHOB,
                                  double dNPHI, double dGR )
  {
    double dIn[] = {0.0, 0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = 100.0*dNPHI;
    dIn[2] = dGR;
    dIn[3] = 1.0;

    return (dIn);
  }

  /** Method getRTGR()
   * { Get RHOMAA-UMAA-Gamma Ray Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid = Identifier of type of fluid to use
   * @param  dRHOB = Bulk density log value
   * @param  dDT   = Acoustic Transit Time log value
   * @param  dNPHI = Neutron Porosity
   * @param  dGR   = Gammma Ray
   * @return dIn   = the Input matrix for the compositional analysis
   */

  public static double[] getRTGR( int matrix, int fluid,  double dRHOB,
                                  double dDT, double dNPHI, double dGR )
  {
    double dIn[] = {0.0, 0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dPHI   = math.mathLAS.computeAvgPHI( dNPHI,  dDPHI );

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = math.mathLAS.computeDTMAA( fluid, dDT, dPHI );
//    dIn[1] = dDT;
    dIn[2] = dGR;
    dIn[3] = 1.0;

    return (dIn);
  }

  /** Method getRTGR()
   * { Get RHOMAA-UMAA-Gamma Ray Curves }
   * <p> This method will convert the Log curves to a matrix for the
   *     compositional analysis matrix.
   * @param  matrix = the matrix to use in computing DPHI
   * @param  fluid = Identifier of type of fluid to use
   * @param  dRHOB = Bulk density log value
   * @param  dDT   = Acoustic Transit Time log value
   * @param  dGR   = Gammma Ray
   * @return dIn   = the Input matrix for the compositional analysis
   */

  public static double[] getRTGR( int matrix, int fluid,  double dRHOB,
                                  double dDT, double dGR )
  {
    double dIn[] = {0.0, 0.0, 0.0, 0.0};
    double dDPHI = 0.0;
    double dSPHI = 0.0;
    double dPHI  = 0.0;

    dDPHI  = math.mathLAS.computePHI( dRHOB, math.mathLAS._DENSITY,
                                      matrix, fluid );
    dSPHI  = math.mathLAS.computePHI( dDT, math.mathLAS._SONIC,
                                      matrix, fluid );

//    dPHI   = math.mathLAS.computeAvgPHI( dSPHI,  dDPHI );
//    dPHI   = dSPHI; // Using Sonic instead of the Average
    dPHI   = Math.sqrt(dSPHI*dSPHI + dDPHI*dDPHI);

    dIn[0] = math.mathLAS.computeRhomaa(fluid, dRHOB, dPHI);
    dIn[1] = math.mathLAS.computeDTMAA( fluid, dDT, dPHI );
//    dIn[1] = dDT;
    dIn[2] = dGR;
    dIn[3] = 1.0;

    return (dIn);
  }

  /* ======================================================================= *
   * ---------------------- GET MINERAL N X N MATRIX ----------------------- *
   * ======================================================================= */

  /** Method getMatrix()
   * <p> This method will build the N X N Matrix for the compositional
   *     analysis matrix.
   * @param  iMinerals = the array of minerals
   * @param  iColumns  = data columns
   * @return dMat      = compositional Matrix
   */

  public static double[][] getMatrix(int iMinerals[], int iColumns[])
  {
    int i = 0;
    int j = 0;
    int len = iMinerals.length;
    double dMat[][] = null;

    dMat = new double[len][len];

    // Initialize the Matrix

    for (i=0; i<len; i++)
    {
      for (j=0; j<len; j++)
      {
        dMat[i][j] = 1.0;
      }
    }

    // Build Matrix [ Log Type ] X [ Mineral ]
    // Last Row is Unitary define at initizlization of Matrix

    for (i=0; i<len-1; i++)
    {
      for (j=0; j<len; j++)
      {
//System.out.println(i+" "+j+" "+iColumns[i]+" "+iMinerals[j]);
        dMat[i][j] = _MINERAL[iMinerals[j]][iColumns[i]];
      }
    }

    return (dMat);
  }

  /* ======================================================================= *
   * ----------------------- COMPOSITIONAL ANALYSIS ------------------------ *
   * ======================================================================= */

  /** Method composition()
   * <p> This method will compute the compositional anlaysis of a rock depending
   *     on the data available.
   *        [ % ] = [ L ] / [ M ]
   *
   * @param  dIn    = Array of log variables
   * @param  dMat   = N X N Matrix
   * @return dOut   = The output matrix
   */

  public static double[] composition( double dIn[], double dMat[][])
  {
    double dOut[]   = null;
    double dInv[][] = null;
    int    N        = 0;

    if ((dIn != null) && (dMat != null))
    {
      N    = dIn.length;
      dOut = new double[N];
      dInv = new double[N][N];

      dInv = lith.math.Inverse.invert(dMat);

      for (int i=0; i<N; i++)
      {
        for (int j=0; j<N; j++)
        {
          dOut[i] = dOut[i] + dInv[i][j] * dIn[j];
        }
      }
    }

    return (dOut);
  }

  /** Method computeComposition()
   * <p> This method will compute the percentage of Quartz, Calcite and Dolomite
   *     in the rock
   *     From: http://www.kgs.ku.edu/Gemini/Help/PfEFFER/Pfeffer-theory11.html
   *
   *      D = Dolomite %      Phin = Neutron Porosity
   *      Q = Quartz %        Rhob = Apparent Bulk Density
   *      C = Calcite %       U    = Apparent Photoelectric Factor
   *
   *      Neutron:  5.0 D  - 5.0 Q  + 0 C     + 100 Phi = Phin
   *      Density:  2.87 D + 2.65 Q + 2.71 C  + 1.0 Phi = Rhob
   *      UPhoto:   9.0 D  + 4.79 Q + 13.77 C + 0.5 Phi = U
   *      Unity:    1.0 D  + 1.0  Q + 1.0 C   + 1.0 Phi = 1.0
   *
   *       _                         _   _   _     _     _
   *      |  5.0  -5.0   0.0   100.0  | |  D  |   |  Phin |
   *      |  2.87 2.65   2.71    1.0  | |  Q  | = |  Rhob |
   *      |  9.0  4.79  13.77    0.5  | |  C  |   |  U    |
   *      |  1.0  1.0    1.0     1.0  | | 1.0 |   |  1.0  |
   *       -                         -   -   -     -     -
   *
   * @param  dNPHI = The neutron Porosity
   * @param  dRho  = The Bulk Density   (Rhomaa)
   * @param  dPe   = The Photo Electric (UMaa)
   * @return dComp = Array of Composition percentage
   */

  public static double[] computeComposition( double dNPHI,
                                             double dRho,
                                             double dPe )
  {
    double dIn[]    = { dNPHI, dRho, dPe, 1.0 };
    double dOut[]   = { 0.0,   0.0,  0.0, 0.0 };
    //                Dolomite,  Quartz, Limestone, H20
    double dMat[][] = {{ 1.0,     -2.0,    0.0,   100.0 },  // NPHI
                       { 2.87,     2.65,   2.71,    1.0 },  // RHOMAA
                       { 9.0,      4.79,  13.77,    0.5 },  // UMAA
                       { 1.0,      1.0,    1.0,     1.0 }}; // Unitary
    dMat = lith.math.Inverse.invert(dMat);

    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
      {
        dOut[i] = dOut[i] + dMat[i][j] * dIn[j];
      }
    }

    return (dOut);
  }

  /** Method compositionByCore( double dNPHI, double dRho )
   * <p> This method will compute the percentage of Quartz, Calcite and Dolomite
   *     in the rock
   * @param  dNPHI = The neutron Porosity
   * @param  dRho  = The Bulk Density   (Rhomaa)
   * @return dOut  = Th output matrix (Dolomite, Quartz, Limestone)
   */

  public static double[] compositionByCore( double dNPHI, double dRho )
  {
    double dDPHI    = math.mathLAS.computePHI( dRho,
                                               math.mathLAS._DENSITY,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH );
    double delta    = dNPHI - dDPHI;
    double dRhomaa  = dRho;

    double dIn[]    = { dNPHI, dRhomaa, delta, 1.0 };
    double dOut[]   = { 0.0,   0.0,  0.0, 0.0 };
    //                 Dolomite,  Quartz, Limestone, H20
    double dMat[][] = {{ 1.0,   -2.0,     0.0,     100.0 },  // NPHI
                       { 2.87,   2.65,    2.71,      1.0 },  // RHOMAA
                       { 20.0,   20.0,   -10.0,    100.0 },  // NPHI-DPHI
                       { 1.0,    1.0,     1.0,       1.0 }}; // Unitary

    dMat = lith.math.Inverse.invert(dMat);

    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
      {
        dOut[i] = dOut[i] + dMat[i][j] * dIn[j];
      }
    }

    return (dOut);
  }
  /** Method compositionByRock( double dNPHI, double dRho )
   * <p> This method will compute the percentage of Quartz, Calcite and Dolomite
   *     in the rock
   * @param  dNPHI = The neutron Porosity
   * @param  dRho  = The Bulk Density   (Rhomaa)
   * @return dOut  = Th output matrix (Dolomite, Quartz, Limestone)
   */

  public static double[] compositionByRock( double dNPHI, double dRho )
  {
    double dDPHI    = math.mathLAS.computePHI( dRho,
                                               math.mathLAS._DENSITY,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH );
    double delta    = dNPHI - dDPHI;
    double dRhomaa  = math.mathLAS.computeRhomaa(math.mathLAS._FRESH, dRho, dNPHI);

    double dIn[]    = { dNPHI, dRhomaa, delta, 1.0 };
    double dOut[]   = { 0.0,   0.0,  0.0, 0.0 };
    //                 Dolomite,  Quartz, Limestone, H20
    double dMat[][] = {{ 1.0,   -2.0,     0.0,     100.0 },  // NPHI
                       { 2.87,   2.65,    2.71,      1.0 },  // RHOMAA
                       { 20.0,   20.0,   -10.0,    100.0 },  // NPHI-DPHI
                       { 1.0,    1.0,     1.0,       1.0 }}; // Unitary

    dMat = lith.math.Inverse.invert(dMat);

    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
      {
        dOut[i] = dOut[i] + dMat[i][j] * dIn[j];
      }
    }

    return (dOut);
  }
}