/*
 * @lithologyPlotTrack.java Version 1.1 09/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology.plot;

import java.awt.*;
import java.awt.geom.*;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;
import lith.lithology.plot.lithologyPlotSymbol;

//import rock.rockDataListStruct;
//import rock.rockFileDataStruct;

/** CLASS lithologyPlotTrack
 *  <p> This Class will create the Colorlith Plots.
 *
 *  @version 1.1 09/05/2007
 *  @author  John R. Victorine
 */

public class lithologyPlotTrack extends Canvas
{
  // Input Variables

  private int    iUnits     = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private int    iFocus      = 0;
  private int    iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_RU;

  private lithologyListStruct stLASLithology   = null; // LAS Rock Column (edit)
//  private lithologyListStruct stRockLithology  = null; // Rock Column (edit)

  private lithologySymbolsListStruct stSymbols = null;
  private lithologyPlotSymbol plot             = null;

  // Plot Width Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR lithologyPlotTrack( int iDataType,
   *                                  double depthStart,
   *                                  double depthEnd,
   *                                  int iScale,
   *                                  int iUnits,
   *                                  lithologySymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param iUnits      = The scale factor for building lithology cells
   *  @param stSymbols   = List of Plot Symbols for Lithology
   */

  public lithologyPlotTrack( int iDataType,
                             double depthStart,
                             double depthEnd,
                             int iScale,
                             int iUnits,
                             lithologySymbolsListStruct stSymbols )
  {
    this.iDataType = iDataType;
    this.iUnits    = iUnits;
    this.stSymbols = stSymbols;

    plot = new lithologyPlotSymbol( stSymbols );

    setPlotHeight(iScale, depthEnd, depthStart);

//    buildLASDataStructure();

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stSymbols       = null;

//    stRockLithology = null;
    stLASLithology  = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getRockLithology()
   * <p> This method will get the rock column data
   * @return stRockLithology = Rock Column determined by user from rock data.
   *

  public lithologyListStruct getRockLithology() { return (stRockLithology); }
*/
  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight(int iScale, double dMaximum, double dMinimum)
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight(int iScale, double dMaximum, double dMinimum)
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setMethod( int iColorlith )
   * <p> This method will set the rock column that will be shown
   * @param iColorlith = colorlith identifier
   */

  public void setMethod( int iColorlith ) { this.iColorlith = iColorlith; }

  /** Method setLASRockColumn( lithologyListStruct st )
   * <p> This method will set the las rock column that will be shown
   * @param st = LAS Rock Column determined by user from las file data
   */

  public void setLASRockColumn( lithologyListStruct st )
  {
    this.stLASLithology = st;
  }

  /** Method setRockLithology( lithologyListStruct stRockLithology )
   * <p> This method will set the rock column that will be shown
   * @param stRockLithology = Rock Column determined by user from rock data.
   *

  public void setRockLithology( lithologyListStruct stRockLithology )
  {
    this.stRockLithology = stRockLithology;
  }
*/
  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack,  int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iIncrement = iWidthTrack/10;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iY1        = 0;
    int    iY2        = 0;
    int    iPercent   = 10;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);

    switch (iFocus)
    {
      case 0:
        switch (iColorlith)
        {
          case iqstrat.iqstratTracksStruct._COLORLITH_RU:
            g.drawString("Rhomaa-Umaa",iStartTrack+2*iIncrement, LABELSTART+14);
            g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
            g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
            break;
          case iqstrat.iqstratTracksStruct._COLORLITH_RN:
          case iqstrat.iqstratTracksStruct._COLORLITH_PD:
//            g.drawString("Porosity-Difference",iStartTrack+iIncrement, LABELSTART+14);
            g.drawString("Rhomaa-NPHI",iStartTrack+2*iIncrement, LABELSTART+14);
            g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
            g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
            break;
        }
        break;

//      case 1:
//        g.drawString("Lithology",   iStartTrack+3*iIncrement, LABELSTART+14);
//        g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+28);
//        break;

      case 2:
        g.drawString("Rhomaa-NPHI", iStartTrack+2*iIncrement, LABELSTART+14);
        g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
        g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
        break;

      case 3:
        switch (iColorlith)
        {
          case iqstrat.iqstratTracksStruct._COLORLITH_RU:
            g.drawString("Rhomaa-Umaa",iStartTrack+2*iIncrement, LABELSTART+14);
            g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
            g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
            break;
          case iqstrat.iqstratTracksStruct._COLORLITH_RN:
          case iqstrat.iqstratTracksStruct._COLORLITH_PD:
            g.drawString("Rhomaa-NPHI",iStartTrack+2*iIncrement, LABELSTART+14);
            g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
            g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
            break;
        }
        break;

      case 4:
        g.drawString("Rhomaa-DT", iStartTrack+2*iIncrement, LABELSTART+14);
        g.drawString("Computed", iStartTrack+3*iIncrement, LABELSTART+28);
        g.drawString("Rock Column", iStartTrack+2*iIncrement, LABELSTART+42);
        break;
    }

    for (i=0; i<=4; i++)
    {
      iPercent = 10 + i*20;
      g.drawString("" + iPercent, iStartTrack + 2*i*iIncrement, PLOT_TITLES);
    }
    g.drawString("Percent", iStartTrack + (7*iIncrement)/2, PLOT_TITLES-14);

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawLithology( Graphics g, int iStartTrack,  int iWidthTrack )
   *  <p> This method will draw the rock mineral composition percentage track
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawLithology( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i       = 0;
    int    j       = 0;
    int    k       = 0;
    int    n       = 0;
    int    iPlus   = 1;
    int    iDepth1 = 0;
    int    iDepth2 = 0;
    int    iDepth  = 0;
    int    iCount  = 0;
    int    iGroup  = 0;
    double diff    = 0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dRatio  = 0.0;
    lithologyListStruct st   = stLASLithology; //stLithology;

//    if (iFocus == 1)
//      st = stRockLithology;

//    if (iFocus == 3)
      st = stLASLithology;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        if (st.stItem[0] != null)
        {
          depth1 = st.stItem[0].depthStart;
          depth2 = st.stItem[0].depthEnd;
          diff = depth2 - depth1;
        }

        iPlus = (int) (( 5.0/diff ) *
                       ( iqstrat.iqstratTracksStruct.SCALE[iScale] / 100.0 ));
        if (iPlus < 1) iPlus = 1;

        iCount = -1;
        for (k=0; k<st.iCount; k++)
        {
          if (st.stItem[k] != null)
          {
            depth1 = st.stItem[k].depthStart;
            depth2 = depth1 + diff*iPlus;

            for (i=1; i<iPlus; i++)
            {
              if (k < st.iCount-1)
              {
                k++;
              }
            }

            dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                            (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                            (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = Math.abs( iDepth2 - iDepth1 );

            if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                (iDepth1 < iHeight)      && (iDepth2 < iHeight))
            {
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 =  iDepth2;

              iGroup = iDepth / 5;
              if (iGroup == 0) iGroup = 1;

              for (n=0; n<iGroup; n++)
              {
                iCount++;
                for (j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
                {
                  if (st.stItem[k] != null)
                  {
                    if (st.stItem[k].iLithology != null)
                    {
                      plot.drawCell(g, st.stItem[k].iLithology[j], iCount,
                                    iStartTrack + j * 10, iDepth1 + n * 5);
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }

  /** Method drawAbbreviation( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method draw the abbreviation track
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   *

  public void drawAbbreviation( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int k=0;
    int j=0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dRatio  = 0.0;
    int    iDepth1 = 0;
    int    iDepth2 = 0;
    int    iDepth  = 0;
    int    iRed    = 0;
    int    iGreen  = 0;
    int    iBlue   = 0;
    int    iLith   = 0;
    int    iRange  = 0;
    double diff    = 0;
    int    iPlus   = 0;
    int    iIncrement = iWidthTrack/10;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    lithologyListStruct st = null;
    int    iCount = 0;
    String sAbrev = "";

    Font        fsb  = new Font("Serif", Font.PLAIN, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (iFocus == 1)
        st = stRockLithology;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        if (st.stItem[0] != null)
        {
          depth1 = st.stItem[0].depthStart;
          depth2 = st.stItem[0].depthEnd;
          diff = depth2 - depth1;
        }

        iPlus = (int) (( 5.0/diff ) *
                       ( iqstrat.iqstratTracksStruct.SCALE[iScale] / 100.0 ));
        if (iPlus < 1) iPlus = 1;

        for (k=0; k<st.iCount; k++)
        {
          if (st.stItem[k] != null)
          {
            depth1 = st.stItem[k].depthStart;
            depth2 = depth1 + diff*iPlus;

            for (int i=1; i<iPlus; i++)
            {
              if (k < st.iCount-1)
              {
                k++;
              }
            }

            dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                            (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                            (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = Math.abs( iDepth2 - iDepth1 );

            if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                (iDepth1 < iHeight)      && (iDepth2 < iHeight))
            {
              sAbrev = new String(st.stItem[k].sAbbreviation);
              if (sAbrev.length() == 0)
                sAbrev = new String(
                  lith.lithology.lithologyUtility.getAbbreviation(
                      st.stItem[k].iLithology, stSymbols));

              if (st.stItem[k].sAbbreviation.length() > 11)
                sAbrev = new String(st.stItem[k].sAbbreviation.substring(0,10));

              if (iCount == 0)
              {
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  g.drawString(sAbrev, iStartTrack, iDepth-2); //iDepth1);
                else
                  g.drawString(sAbrev, iStartTrack, iDepth1+6); //iDepth1+8);
              }

              if (iCount == 0)
                iCount = 1;
              else
                iCount = 0;
            }
          }
        }
      }
    }

    g.setColor(Color.black);

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Rock Column", LABELSTART+10,
                  -1*(iStartTrack+2*iWidthTrack/3));
    g2.drawString("Abbreviations", LABELSTART+10,
                  -1*(iStartTrack+iWidthTrack/3));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }
*/
  /** Method drawStreaks( Graphics g, int iStartTrack,  int iWidthTrack )
   * <p> This method draw the Lithology Streaks track
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   *

  public void drawStreaks( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int k=0;
    int j=0;
    int n=0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dRatio  = 0.0;
    int    iDepth1 = 0;
    int    iDepth2 = 0;
    int    iDepth  = 0;
    int    iRed    = 0;
    int    iGreen  = 0;
    int    iBlue   = 0;
    int    iLith   = 0;
    int    iRange  = 0;
    double diff    = 0;
    int    iPlus   = 0;
    int    iIncrement = iWidthTrack/2;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    lithologyListStruct st = null;
    int    iCount = 0;
    int    iGroup = 0;

    g.setColor(Color.black);

    if (iFocus == 1)
//      if (iRockColumn == iqstrat.iqstratTracksStruct._LITHOLOGY_ROCK)
        st = stRockLithology;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        if (st.stItem[0] != null)
        {
          depth1 = st.stItem[0].depthStart;
          depth2 = st.stItem[0].depthEnd;
          diff = depth2 - depth1;
        }

        iPlus = (int) (( 5.0/diff ) *
                       ( iqstrat.iqstratTracksStruct.SCALE[iScale] / 100.0 ));
        if (iPlus < 1) iPlus = 1;

        iCount = -1;
        for (k=0; k<st.iCount; k++)
        {
          if (st.stItem[k] != null)
          {
            depth1 = st.stItem[k].depthStart;
            depth2 = depth1 + diff*iPlus;

            for (int i=1; i<iPlus; i++)
            {
              if (k < st.iCount-1)
              {
                k++;
              }
            }

            dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                            (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                            (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = Math.abs( iDepth2 - iDepth1 );

            if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                (iDepth1 < iHeight)      && (iDepth2 < iHeight))
            {
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 =  iDepth2;

              iGroup = iDepth / 5;
              if (iGroup == 0) iGroup = 1;

              for (n=0; n<iGroup; n++)
              {
                iCount++;
                for (j=0; j<2; j++)
                {
                  plot.drawCell( g, st.stItem[k].iStreaks, iCount,
                                 iStartTrack + j*10, iDepth1 + n*5);
                }
              }
            }
          }
        }
      }
    }

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Rock Column", LABELSTART+10,
                  -1*(iStartTrack+2*iWidthTrack/3));
    g2.drawString("Streaks, lenses", LABELSTART+10, -1*(iStartTrack+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }
*/
  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track that was selected
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw(Graphics g, int iSelected, int iStartTrack, int iWidthTrack)
  {
    int iColor = -1;

    switch (iSelected)
    {
      case iqstrat.iqstratTracksStruct._ROCK_COLUMN:
        iColor = iqstrat.iqstratTracksStruct._SRC_LAS;
        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

        iFocus = 3;//0;
        drawLithology( g, iStartTrack, iWidthTrack );
        drawGrid( g, iStartTrack, iWidthTrack );
        break;

      case iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD:
        iColor = iqstrat.iqstratTracksStruct._SRC_LAS;
        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

        iFocus = 3;//2;
        drawLithology( g, iStartTrack, iWidthTrack );
        drawGrid( g, iStartTrack, iWidthTrack );
        break;

      case iqstrat.iqstratTracksStruct._ROCK_COLUMN_RT:
        iColorlith = iqstrat.iqstratTracksStruct._COLORLITH_RT;
        iColor     = iqstrat.iqstratTracksStruct._SRC_LAS;
        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

        iFocus = 4;
        drawLithology( g, iStartTrack, iWidthTrack );
        drawGrid( g, iStartTrack, iWidthTrack );
        break;

//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
//        iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
//        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
//        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

//        iFocus = 1;
//        drawLithology( g, iStartTrack, iWidthTrack );
//        drawGrid( g, iStartTrack, iWidthTrack );
//        break;
//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY_ABREV:
//        iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
//        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
//        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

//        iFocus = 1;
//        drawAbbreviation( g, iStartTrack, iWidthTrack );
//        break;
//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY_STRK:
//        iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
//        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
//                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
//        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

//        iFocus = 1;
//        drawStreaks( g, iStartTrack, iWidthTrack );
//        break;

    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}