/*
 * @lithologyPlotSymbolLegend.java Version 1.1 02/28/2012
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology.plot;

import java.awt.*;

import lith.lithology.lithologySymbolsListStruct;

/** Class lithologyPlotSymbolLegend
 *  <p> This method will create a legend depending on the Track type that is
 *      available.
 *
 *  @version 1.1 02/28/2012
 *  @author  John R. Victorine
 */

public class lithologyPlotSymbolLegend extends Canvas
{
  public  static final int WIDTH    = 250; //350;
  private static final int TEXT     = 18;
  private static final int ROW      = 25;

  private static final int X_START  = 30;
  private static final int Y_START  = 30;
  private static final int _ROW     = 60;

  private static final int X_DESC   = 50; //50;
  private static final int X_ABBREV = 290; //250;

  private int iWidth  = WIDTH;
  private int iHeight = 950;
  private int TOTAL   = 0;

  private lithologySymbolsListStruct stSymbols = null;

  private static final int _TOTAL         = 13;

  private static final int _SYMBOLS[] =
  {
    lith.lithology.lithologySymbolsStruct._CLAY,
    lith.lithology.lithologySymbolsStruct._SHALE,
    lith.lithology.lithologySymbolsStruct._SANDSTONE,
    lith.lithology.lithologySymbolsStruct._SHALY_SAND,
    lith.lithology.lithologySymbolsStruct._ARKOSE,
    lith.lithology.lithologySymbolsStruct._LIMESTONE,
    lith.lithology.lithologySymbolsStruct._SHALY_CARB,
    lith.lithology.lithologySymbolsStruct._DOLOMITE,
    lith.lithology.lithologySymbolsStruct._SHALY_DOLO,
    lith.lithology.lithologySymbolsStruct._GYPSUM,
    lith.lithology.lithologySymbolsStruct._ANHYDRITE,
    lith.lithology.lithologySymbolsStruct._SALT_GENERAL,
    lith.lithology.lithologySymbolsStruct._COAL
  };

  private static final String SYMBOLS[] =
  {
	"Clay", "Shale", "Sandstone", "Sandstone", "Arkose", "Limestone",
	"Carbonate", "Dolomite", "Dolomite", "Gypsum", "Anhydrite", "Salt", "Coal"
  };

  private static final String SYMBOLS2[] =
  {
	"", "", "", "Shaly", "", "", "Shaly", "", "Shaly", "", "", "", ""
  };

  /** CONSTRUCTOR lithologyPlotSymbolLegend()
   *  <p> This is the constructor for this class.
   *  @param stSymbols = List of Plot Symbols for Lithology
   */

  public lithologyPlotSymbolLegend( lithologySymbolsListStruct stSymbols )
  {
    this.stSymbols = stSymbols;
    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close() { stSymbols = null; }

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getOrder()
   * <p> This method will return the 0 or 1 depending if the number passed in is
   *     even or odd
   * @param  iNo    = the number passed in
   * @return iOrder = 0 or 1 depending if the iNo is even or odd respectively
   */

  public int getOrder( int iNo )
  {
    int iOrder = 0;

    if (iNo > 1)
    {
      while (iNo > 1)
        iNo = iNo - 2;
    }
    iOrder = iNo;

    return (iOrder);
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawCell(Graphics g, int iLithology, int iOrder, int iXo, int iYo)
   * <p> This method will draw the symbol cell.
   *  @param g          = Graphics Handle;
   *  @param iLithology = the lithology identifier
   *  @param iOrder     = the order number (right or left text array to plot)
   *  @param iXo        = initial X starting point on plot
   *  @param iYo        = initial Y starting point on plot
   */

  public void drawCell(Graphics g, int iLithology, int iOrder, int iXo, int iYo)
  {
    drawBlock( g, iLithology, iXo, iYo, 10, 5);
    drawLithologySymbolMask(
        g, 5, getOrder(iOrder),
        lith.lithology.lithologySymbolsUtility.getSymbol(iLithology, stSymbols),
        iXo, iYo );
  }

  /** Method drawBlock( Graphics g,
   *                    int iLithology,
   *                    int iXp,   int iYp,
   *                    int iWide, int iHigh )
   * <p> This method will draw and fill a rectangular block
   *  @param g          = Graphics Handle;
   *  @param iLithology = The Lithology Identifier
   *  @param iXp        = X Starting Position
   *  @param iYp        = Y Starting Position
   *  @param iWide      = Width  of block
   *  @param iHigh      = Height of block
   */

  public void drawBlock( Graphics g,
                         int iLithology,
                         int iXp,   int iYp,
                         int iWide, int iHigh )
  {
    int iColor[] = { 255, 255, 255 };

    if (stSymbols != null)
      iColor = stSymbols.stItem[iLithology].iRGB;

    g.setColor(new Color(iColor[0], iColor[1], iColor[2]));
    g.fillRect(iXp, iYp, iWide, iHigh);
  }

  /** Method drawLithologySymbolMask( Graphics g, int iRows, int iOrder,
   *                                  String symbol[][], int iXo, int iYo )
   * <p> This method will convert the String Array to an image and paint it
   *  @param g      = Graphics Handle;
   *  @param iRows  = total number of rows in the text array
   *  @param iOrder = the order number (right or left text array to plot)
   *  @param symbol = the text array mask
   *  @param iXo    = initial X starting point on plot
   *  @param iYo    = initial Y starting point on plot
   */

  public void drawLithologySymbolMask( Graphics g, int iRows, int iOrder,
                                       String symbol[][], int iXo, int iYo )
  {
    int  i      = 0;
    int  j      = 0;
    int  iRGB[] = null;
    int  iColor = 0;
    char c[]    = null;

    for (i=0; i<iRows; i++)
    {
      if (symbol != null)
        c = symbol[i][iOrder].toCharArray();

      for (j=0; j<c.length; j++)
      {
        if ((stSymbols != null) && (c != null))
          iColor = lith.lithology.lithologySymbolsUtility.getColor(
                     c[j], stSymbols.stForeground );

        if (iColor > -1)
        {
          if (stSymbols != null)
            iRGB = lith.lithology.lithologySymbolsUtility.getRGB(
                     iColor, stSymbols.stForeground );
          g.setColor( new Color(iRGB[0], iRGB[1], iRGB[2]) );
          g.fillRect(iXo+j, iYo+i, 1, 1);
        }
      }
    }
  }

  /** Method draw( Graphics g )
   * <p> This method will draw all the lithology symbols.
   *  @param g  = Graphics Handle;
   *  @param iT = Selected Track
   *  @param iX = X pixel starting point
   *  @param iY = Y pixel starting point
   *  @param iW = Width of plot
   */

  public void draw( Graphics g, int iT, int iX, int iY, int iW )
  {
	int L   = -1; // Lithology order
	int iR  = 1;  // Total Rows
	int row = 0;  // plot row
	int ixo = -1; // actual position of increment x axis
	int iyo = 0;  // actual position of increment y axis

    Font fsb = new Font("Serif", Font.BOLD, 10);
    g.setFont( fsb );

    if ((iT == iqstrat.iqstratTracksStruct._ROCK_COLUMN) ||
        (iT == iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD ))
    {
  	  for (int i=0; i<_TOTAL; i++)
	  {
        for (int p=0; p<stSymbols.iCount; p++)
        {
          if (_SYMBOLS[i] == stSymbols.stItem[p].id)
          {
            L = p;
          }
        }

        if (iW < 50 * _TOTAL)
        {
		  iR = 2;
		  if (i==7)
		  {
			ixo = -1;
			iyo = _ROW;
		  }

  	      ixo++;
		}
		else
		{
		  ixo++;
		}

        for (int j=0; j<4; j++)
        {
          for (int k=0; k<4; k++)
          {
            drawCell( g, L, getOrder(j), X_START+iX+ixo*50+k*10, Y_START+iyo+iY+j*5);
          }
        }

        g.setColor(Color.black);
        g.drawString(SYMBOLS[i],  X_START+iX+ixo*50, Y_START+iyo+iY-5);
        g.drawString(SYMBOLS2[i], X_START+iX+ixo*50, Y_START+iyo+iY-17);
	  }
    }
  }

  /** Method paint()
   *  <p> This method will paint the Scale data.
   *  @param  g      - Graphics Handle;
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 02/28/2012
 *  @author  John Victorine
 */
