/*
 * @lithologyPlotSymbol.java Version 1.1 09/11/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology.plot;

import java.io.*;
import java.lang.*;
//import java.util.*;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import lith.lithology.lithologySymbolsListStruct;

/** Class lithologyPlotSymbol
 *  <p> This method will create the different symbol types for plotting data
 *      values on a plot.
 *
 *  @version 1.1 09/11/2008
 *  @author  John R. Victorine
 */

public class lithologyPlotSymbol extends Canvas implements MouseListener
{
  // Input Variables

  private Observable                 notifier  = null; // Observable Object
  private lithologySymbolsListStruct stSymbols = null; // Lithology Symbols

  // Global Plot Variables

  private int iLithology = 0;
  private int iTotal     = 0;
  private int TOTAL      = 0;

  private int iGroup     = -1;

  public static final int _NO_BACKGROUND = 0;
  public static final int _BACKGROUND    = 1;

  private int iBackground = _BACKGROUND;

  // Panel

  public final static int _LAS        = 0;
  public final static int _ROCK       = 1;
  public final static int _TEXTURE_NO = 2;
  public final static int _TEXTURE    = 3;

  private int iPanel = _ROCK;

  // Lithology Symbol Position

  private int iCount   = 0;
  private int iRow[][] = null;

  // Plot Variables

  public  static final int WIDTH    = 250; //300;//350;
  private static final int TEXT     = 18;
  private static final int ROW      = 25;

  private static final int X_START  = 10;
  private static final int X_DESC   = 50;
  private static final int X_ABBREV = 290;

  private int iWidth  = WIDTH;
  private int iHeight = 950;

  /** CONSTRUCTOR lithologyPlotSymbol( lithologySymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param stSymbols = List of Plot Symbols for Lithology
   */

  public lithologyPlotSymbol( lithologySymbolsListStruct stSymbols )
  {
    this.stSymbols = stSymbols;

    this.setBackground(Color.white);

    if (stSymbols != null)
    {
      if ((stSymbols.stGroups != null) && (stSymbols.iCount > 0))
      {
        TOTAL = stSymbols.stGroups.iGroups + stSymbols.iCount;
        iRow  = new int[stSymbols.iCount][2];
      }
    }

    iHeight = (TOTAL + 5) * 25;
  }

  /** CONSTRUCTOR lithologyPlotSymbol()
   *  <p> This is the constructor for this class.
   *  @param notifier = Observable
   *  @param stSymbols = List of Plot Symbols for Lithology
   */

  public lithologyPlotSymbol( Observable notifier,
                              lithologySymbolsListStruct stSymbols )
  {
    this.notifier  = notifier;
    this.stSymbols = stSymbols;
    this.iPanel    = _ROCK;

    this.setBackground(Color.white);
    addMouseListener(this);

    if (stSymbols != null)
    {
      if ((stSymbols.stGroups != null) && (stSymbols.iCount > 0))
      {
        TOTAL = stSymbols.stGroups.iGroups + stSymbols.iCount;
        iRow  = new int[stSymbols.iCount][2];
      }
    }

    iHeight = (TOTAL + 5) * 25;
  }

  /** CONSTRUCTOR lithologyPlotSymbol()
   *  <p> This is the constructor for this class.
   *  @param notifier  = Observable
   *  @param iPanel    = Type of Data Entry Panel to display
   *  @param stSymbols = List of Plot Symbols for Lithology
   */

  public lithologyPlotSymbol( Observable notifier, int iPanel,
                              lithologySymbolsListStruct stSymbols )
  {
    this.notifier  = notifier;
    this.stSymbols = stSymbols;
    this.iPanel    = iPanel;

    switch (iPanel)
    {
      case _LAS:
      case _ROCK:
        break;
      case _TEXTURE_NO:
        iBackground = _NO_BACKGROUND;
        break;
      case _TEXTURE:
        iBackground = _BACKGROUND;
        break;
    }

    iCount = 0;

    this.setBackground(Color.white);
    addMouseListener(this);

    if (stSymbols != null)
    {
      if ((stSymbols.stGroups != null) && (stSymbols.iCount > 0))
      {
        TOTAL = stSymbols.stGroups.iGroups + stSymbols.iCount;
        iRow  = new int[stSymbols.iCount][2];
      }
    }

    iHeight = (TOTAL + 5) * 25;
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier  = null;
    stSymbols = null;
    iRow      = null;
  }

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getOrder()
   * <p> This method will return the 0 or 1 depending if the number passed in is
   *     even or odd
   * @param  iNo    = the number passed in
   * @return iOrder = 0 or 1 depending if the iNo is even or odd respectively
   */

  public int getOrder( int iNo )
  {
    int iOrder = 0;

    if (iNo > 1)
    {
      while (iNo > 1)
        iNo = iNo - 2;
    }
    iOrder = iNo;

    return (iOrder);
  }

  /** Method getLithID()
   * <p> This method will return the Lithology ID Number
   * @return iLithology = the Lithology ID Number
   */

  public int getLithID() { return (iLithology); }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method setGroup()
   * <p> This method will set the Lithology Group to display
   * @param iGroup = the group id
   */

  public void setGroup( int iGroup )
  {
    this.iGroup = iGroup;

    iTotal = 0;
    iCount = 0;
    iRow   = null;

    if (stSymbols != null)
    {
      if ((stSymbols.stGroups != null) && (stSymbols.iCount > 0))
      {
        if (iGroup == -1)
        {
          TOTAL  = stSymbols.stGroups.iGroups + stSymbols.iCount;
          iRow   = new int[stSymbols.iCount][2];
          iTotal = stSymbols.iCount;
        }
        else
        {
          TOTAL = 1;
          for (int i=0; i<stSymbols.iCount; i++)
          {
            if (stSymbols.stItem[i].iGroup == iGroup)
            {
              TOTAL++;
              iTotal++;
            }
          }

          iRow = new int[iTotal][2];
        }
      }
    }

    iHeight = (TOTAL + 5) * 25;
    repaint();
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  public void mouseMoved(MouseEvent event) {}
  public void mouseEntered(MouseEvent event) {}
  public void mouseExited(MouseEvent event) {}
  public void mousePressed(MouseEvent event) {}
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm     = event.getX();
    int    iYm     = event.getY();

    iLithology = -1;

    if (iRow != null)
    {
      for (int i=0; i<iCount; i++)
      {
        if (iRow[i] != null)
        {
          if ((iYm >= iRow[i][0]) && (iYm < iRow[i][0]+ROW))
          {
            iLithology = iRow[i][1];
          }
        }
      }
    }

    if (iLithology > -1)
    {
      switch ( iPanel )
      {
        case _LAS:
//          if (notifier != null)
//            notifier.notifyObservers(new String("LAS Litholgy Symbol Selected"));
          break;

        case _ROCK:
          if (notifier != null)
            notifier.notifyObservers(new String("Litholgy Symbol Selected"));
          break;
        case _TEXTURE_NO:
        case _TEXTURE:
//          if (notifier != null)
//            notifier.notifyObservers(new String("Texture Symbol Selected"));
          break;
      }
    }
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawCell()
   * <p> This method will draw the symbol cell.
   *  @param g          = Graphics Handle;
   *  @param iLithology = the lithology identifier
   *  @param iOrder     = the order number (right or left text array to plot)
   *  @param iXo        = initial X starting point on plot
   *  @param iYo        = initial Y starting point on plot
   */

  public void drawCell(Graphics g, int iLithology, int iOrder, int iXo, int iYo)
  {
    int iLith = 0;

    for (int i=0; i<stSymbols.iCount; i++)
    {
      if (iLithology == stSymbols.stItem[i].id)
        iLith = i;
    }

    drawBlock( g, iLith, iXo, iYo, 10, 5);
    drawLithologySymbolMask(
        g, 5, getOrder(iOrder),
        lith.lithology.lithologySymbolsUtility.getSymbol(iLith, stSymbols),
        iXo, iYo );
  }

  /** Method drawCell()
   * <p> This method will draw the symbol cell.
   *  @param g          = Graphics Handle;
   *  @param ibg        = Background Color OFF or ON
   *  @param iLithology = the lithology identifier
   *  @param iOrder     = the order number (right or left text array to plot)
   *  @param iXo        = initial X starting point on plot
   *  @param iYo        = initial Y starting point on plot
   */

  public void drawCell(Graphics g, int ibg,
                       int iLithology, int iOrder, int iXo, int iYo)
  {
    int iLith    = 0;
    int iColor[] = { 248, 248, 255 };

    for (int i=0; i<stSymbols.iCount; i++)
    {
      if (iLithology == stSymbols.stItem[i].id)
        iLith = i;
    }

    if (ibg == _BACKGROUND)
    {
      if (stSymbols != null)
        iColor = stSymbols.stItem[iLith].iRGB;
    }

    drawBlock( g, iColor[0], iColor[1], iColor[2], iXo, iYo, 10, 5);

//    drawBlock( g, iLith, iXo, iYo, 10, 5);
    drawLithologySymbolMask(
        g, 5, getOrder(iOrder),
        lith.lithology.lithologySymbolsUtility.getSymbol(iLith, stSymbols),
        iXo, iYo );
  }

  /** Method drawBlock()
   * <p> This method will draw and fill a rectangular block
   *  @param g          = Graphics Handle;
   *  @param iLithology = The Lithology Identifier
   *  @param iXp        = X Starting Position
   *  @param iYp        = Y Starting Position
   *  @param iWide      = Width  of block
   *  @param iHigh      = Height of block
   */

  public void drawBlock( Graphics g,
                         int iLithology,
                         int iXp,   int iYp,
                         int iWide, int iHigh )
  {
    int iColor[] = { 255, 255, 255 };

    if (stSymbols != null)
      iColor = stSymbols.stItem[iLithology].iRGB;

    g.setColor(new Color(iColor[0], iColor[1], iColor[2]));
    g.fillRect(iXp, iYp, iWide, iHigh);
  }

  /** Method drawBlock()
   * <p> This method will draw and fill a rectangular block
   *  @param g          = Graphics Handle;
   *  @param iRed       = Red Color
   *  @param iBlue      = Blue Color
   *  @param iGreen     = Green Color
   *  @param iXp        = X Starting Position
   *  @param iYp        = Y Starting Position
   *  @param iWide      = Width  of block
   *  @param iHigh      = Height of block
   */

  public void drawBlock( Graphics g,
                         int iRed, int iGreen, int iBlue,
                         int iXp,   int iYp,
                         int iWide, int iHigh )
  {
    g.setColor(new Color(iRed, iGreen, iBlue));
    g.fillRect(iXp, iYp, iWide, iHigh);
  }

  /** Method drawLithologySymbolMask()
   * <p> This method will convert the String Array to an image and paint it
   *  @param g      = Graphics Handle;
   *  @param iRows  = total number of rows in the text array
   *  @param iOrder = the order number (right or left text array to plot)
   *  @param symbol = the text array mask
   *  @param iXo    = initial X starting point on plot
   *  @param iYo    = initial Y starting point on plot
   */

  public void drawLithologySymbolMask( Graphics g, int iRows, int iOrder,
                                       String symbol[][], int iXo, int iYo )
  {
    int  i      = 0;
    int  j      = 0;
    int  iRGB[] = null;
    int  iColor = 0;
    char c[]    = null;

    for (i=0; i<iRows; i++)
    {
      if (symbol != null)
        c = symbol[i][iOrder].toCharArray();

      for (j=0; j<c.length; j++)
      {
        if ((stSymbols != null) && (c != null))
          iColor = lith.lithology.lithologySymbolsUtility.getColor(
                     c[j], stSymbols.stForeground );

        if (iColor > -1)
        {
          if (stSymbols != null)
            iRGB = lith.lithology.lithologySymbolsUtility.getRGB(
                     iColor, stSymbols.stForeground );
          g.setColor( new Color(iRGB[0], iRGB[1], iRGB[2]) );
          g.fillRect(iXo+j, iYo+i, 1, 1);
        }
      }
    }
  }

  /** Method drawLithologyGroup( Graphics g, int iStart, int iGroup )
   * <p> This method will draw the Lithology Group
   *  @param  g      - Graphics Handle;
   *  @param  iStart - Starting height of the lithology group
   *  @param  iGroup - Group ID
   *  @return iEnd   - Ending heith of the lithology group
   */

  public int drawLithologyGroup( Graphics g, int iStart, int iGroup )
  {
    int         iEnd  = iStart;
    int         L     = 0;
    int         i     = 0;
    int         j     = 0;
    int         k     = 0;
    String      sText = "";
    int         iLen  = 0;
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );

    if (stSymbols != null)
    {
      if (stSymbols.stGroups != null)
      {
        sText = new String(
            stSymbols.stGroups.sGroups[iGroup][
              lith.lithology.lithologyGroupsStruct._DESCRIPTION]);
      }
    }

    iLen  = sText.length();

    g.setColor(Color.black);
    g.drawString(sText, (WIDTH - 4*iLen)/2, iStart);

    iEnd = iEnd + TEXT;

    for (i=0; i<stSymbols.iCount; i++)
    {
      L = stSymbols.stItem[i].id;

      if (stSymbols.stItem[i].iGroup == iGroup)
      {
        if (iCount < iTotal)
        {
          iRow[iCount][0] = iEnd;
          iRow[iCount][1] = L;
          iCount++;
        }

        for (j=0; j<3; j++)
        {
          for (k=0; k<3; k++)
          {
            drawCell( g, iBackground, L, getOrder(j), X_START+k*10, iEnd+j*5);
          }
        }

        g.setColor(Color.black);
        g.drawString(stSymbols.stItem[i].sName,  X_DESC,   iEnd+10);
//        g.drawString(stSymbols.stItem[i].sAbrev, X_ABBREV, iEnd+10);

        iEnd = iEnd + ROW;
      }
    }

    return (iEnd);
  }

  /** Method draw()
   * <p> This method will draw all the lithology symbols.
   *  @param g - Graphics Handle;
   */

  public void draw(Graphics g)
  {
    int iStart = 10;

    if (stSymbols != null)
    {
      if (stSymbols.stGroups != null)
      {
        if (iGroup == -1)
        {
          for (int i=0; i<stSymbols.stGroups.iGroups; i++)
          {
            iStart = drawLithologyGroup(g, iStart+10, i);
          }
        }
        else
        {
          iStart = drawLithologyGroup(g, iStart+10, iGroup);
        }
      }
    }
  }

  /** Method paint()
   *  <p> This method will paint the Scale data.
   *  @param  g      - Graphics Handle;
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 09/11/2008
 *  @author  John Victorine
 */
