/*
 * @lithologyPlotData.java Version 1.1 04/14/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology.plot;

import java.awt.*;
import java.awt.geom.*;

import lith.lithology.lithologyStruct;
import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.plot.lithologyPlotSymbol;

/** CLASS lithologyPlotData
 *  <p> This Class will allow the user to enter the lithology for the depth
 *      range
 *
 *  @version 1.1 04/14/2011
 *  @author  John R. Victorine
 */

public class lithologyPlotData extends Canvas
{
  private lithologySymbolsListStruct stSymbols = null; // Lithology Symbols
  private lithologyStruct stLithology = null;

  private int iLith = 0;

  private lithologyPlotSymbol plot = null;

  // Data

  public String sAbbreviation = "";
  public int    iStreaks      = 0;
  public int    iLithology[]  = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

  // Plot variables

  private int iWidth    = 220;  // Width of Plot
  private int iHeight   =  80;  // Height of Plot with the Titles

  private static final int Y_TEXT_START    = 0; //10;
  private static final int Y_GRID_START    = 12; //22;
  private static final int Y_GRID_END      = 12; //22; //42;
  private static final int Y_SELECTED      = 50;
  private static final int TEXT_START      = 38;

  private static final int CELL_WIDTH      = 20; //30;   // 3 Lithologies Wide
  private static final int CELL_HEIGHT     = 15; //20;   // 4 Lithologies High

  private static final int LITHOLOGY_START = 0; //10;
  private static final int ABBREV_START    = 315;
  private static final int STREAK_START    = 450;
  private static final int STREAK_END      = 480;

  private int iXBegin   = 0;
  private int iXEnd     = 0;
  private int iYBegin   = 0;
  private int iYEnd     = 0;
  private int iAction   = 0;
  private int iAddBox   = 0;

  private static final int _NONE = 0;
  private static final int _LITH = 1;
  private static final int _STRK = 2;

  private int iTrack = _NONE;

  /** CONSTRUCTOR lithologyPlotData()
   *  <p> This is the constructor for this class.
   *  @param stSymbols = lithology symbols list data structure
   */

   public lithologyPlotData( lithologySymbolsListStruct stSymbols )
   {
     this.stSymbols = stSymbols;

     plot = new lithologyPlotSymbol( stSymbols );

     this.setBackground(Color.white);
   }

   /** Method close()
    * <p> This method will set object to null to force JAVA to reallocate memory
    */

   public void close()
   {
     stSymbols     = null;
     stLithology   = null;
     sAbbreviation = null;
     iLithology    = null;

     if (plot != null)
       plot.close();
     plot = null;

   }

   /* =============================================================== *
    * ------------------------- GET METHODS ------------------------- *
    * =============================================================== */

   /** Method getPlotWidth()
    * <p> This method will return the Plot Width
    * @return iWidth = The Plot Width
    */

   public int getPlotWidth()  { return (iWidth); }

   /** Method getPlotHeight()
    * <p> This method will return the Plot Height
    * @return iHeight = The Plot Height
    */

   public int getPlotHeight() { return (iHeight); }

   /** Method getData()
    * <p> This method will get the lithology data structure
    * @return stLithology = the lithology data structure
    */

   public lithologyStruct getData()
   {
     if (stLithology == null)
     {
       stLithology      = new lithologyStruct();
       stLithology.sKEY = new String( cmn.cmnString.UniqueName() );
     }

     stLithology.iStreaks        = iStreaks;
     stLithology.sAbbreviation = new String(sAbbreviation);

     for (int j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
       stLithology.iLithology[j] = iLithology[j];

     return (stLithology);
   }

   /* =============================================================== *
    * ------------------------- SET METHODS ------------------------- *
    * =============================================================== */

   /** Method setLithology(int iLith)
    * <p> This method will set the lithology that was selected.
    * @param iLith = The idenfier for the lithology selected
    */

   public void setLithology(int iLith)
   {
     this.iLith = iLith;

     repaint();
   }

   /** Method setData(lithologyStruct st)
    * <p> This method will set the rock column that will be shown
    * @param st = Rock Column determined by user from rock data.
    */

   public void setData(lithologyStruct st)
   {
     this.iStreaks = 0;
     this.sAbbreviation = new String("");
     for (int i=0; i<10; i++) this.iLithology[i] = 0;

     this.stLithology   = st;
     if (st != null)
     {
       this.iLithology    = st.iLithology;
       this.iStreaks      = st.iStreaks;
       this.sAbbreviation = new String(st.sAbbreviation);
     }

     repaint();
   }

   /** Method computeArea()
    * <p> This method will fill the area of the Lithology Data selected
    */

   private void computeArea()
   {
     if (iTrack == _STRK)  iStreaks = iLith;
   }

   /* =============================================================== *
    * ------------------------ DRAW METHODS ------------------------- *
    * =============================================================== */

   /** Method drawGrid(Graphics g)
    * <p> This method will draw the depth scale
    * @param g - Graphics Parameter.
    */

   public void drawGrid(Graphics g)
   {
     int         iGroup = CELL_HEIGHT / 5;
     int         iCell  = CELL_WIDTH  / 10;
     int         iCount = 0;
     int         iPer   = 0;
     String      str    = "";
     Font        fsb    = new Font("Serif", Font.BOLD, 12);
     FontMetrics fsbm   = g.getFontMetrics(fsb);

     g.setFont( fsb );
     g.setColor(Color.black);

     g.drawLine(LITHOLOGY_START,                   Y_TEXT_START,
                LITHOLOGY_START + 10 * CELL_WIDTH, Y_TEXT_START);
     g.drawLine(LITHOLOGY_START,                   Y_GRID_START,
                LITHOLOGY_START + 10 * CELL_WIDTH, Y_GRID_START);
     g.drawLine(LITHOLOGY_START,                   Y_GRID_END,
                LITHOLOGY_START + 10 * CELL_WIDTH, Y_GRID_END);

//     g.drawString("Percent",
//                  LITHOLOGY_START + 3*CELL_WIDTH + CELL_WIDTH / 3,
//                  Y_TEXT_START);

//     g.drawString("Abbreviation", ABBREV_START + CELL_WIDTH, Y_TEXT_START);
//     g.drawString("Streaks",      STREAK_START - 5,          Y_TEXT_START);

     for (int j=0; j<lith.lithology.lithologyStruct.TOTAL+1; j++)
     {
       g.drawLine(LITHOLOGY_START + j*CELL_WIDTH, Y_TEXT_START,
                  LITHOLOGY_START + j*CELL_WIDTH, Y_GRID_END);
       iPer = (j+1) * 10;
       if (iPer < 110)
       {
         str = new String("" + iPer);
         g.drawString(str,
                      LITHOLOGY_START + j*CELL_WIDTH + CELL_WIDTH / 5, //CELL_WIDTH / 3,
                      Y_GRID_START);
       }
     }

     g.drawLine(STREAK_START, Y_GRID_START, STREAK_END, Y_GRID_START);
     g.drawLine(STREAK_START, Y_GRID_END,   STREAK_END, Y_GRID_END);

     g.drawLine(STREAK_START, Y_GRID_START, STREAK_START, Y_GRID_END);
     g.drawLine(STREAK_END,   Y_GRID_START, STREAK_END,   Y_GRID_END);

     if (iLith != 0)
     {
       iCount = 0;
       for (int n=0; n<iGroup; n++)
       {
         for (int i=0; i<iCell; i++)
         {
           plot.drawCell( g, iLith, iCount,
                          LITHOLOGY_START + i*10, Y_SELECTED + n*5 );
         }
         iCount++;
       }

       for (int m=0; m<stSymbols.iCount; m++)
       {
         if (iLith == stSymbols.stItem[m].id)
           g.drawString(stSymbols.stItem[m].sName,
                      LITHOLOGY_START + 50, Y_SELECTED + 16);
       }
     }
   }

   /** Method drawLithology(Graphics g)
    * <p> This method will fill one rectangle with the selected litholgy
    * @param g - Graphics Parameter.
    */

   public void drawLithology(Graphics g)
   {
     int    i         = 0;
     int    j         = 0;
     int    m         = 0;
     int    n         = 0;
     int    iGroup    = CELL_HEIGHT / 5;
     int    iCell     = CELL_WIDTH  / 10;
     int    iCount    = 0;
//     int    iFound    = 0;
//     int    iNext     = 0;
//     int    iULith[]  = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
//     String sAbrev    = "";
     Font        fsb  = new Font("Serif", Font.BOLD, 12);
     FontMetrics fsbm = g.getFontMetrics(fsb);

     g.setFont( fsb );

//     for (j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
//     {
//       iFound = 0;
//       if (iLithology[j] != 0)
//       {
//         for (m=0; m<iNext; m++)
//         {
//           if (iLithology[j] == iULith[m]) iFound = 1;
//         }
//
//         if (iFound == 0)
//         {
//           iULith[iNext] = iLithology[j];
//           iNext++;
//         }
//       }
//     }

//     sAbbreviation = new String("");
//     for (m=0; m<iNext; m++)
//     {
//       if (iULith[m] > 0)
//       {
//         if (m > 0) sAbbreviation = new String( sAbbreviation + ",");
//
//         for (int p=0; p<stSymbols.iCount; p++)
//         {
//           if (iULith[m] == stSymbols.stItem[p].id)
//
//           sAbbreviation = new String(
//               sAbbreviation + stSymbols.stItem[p].sAbrev);
//         }
//       }
//     }

     iCount = 0;
     for (n=0; n<iGroup; n++)
     {
       for (j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
       {
         for (i=0; i<iCell; i++)
         {
           plot.drawCell( g,
                          iLithology[j], iCount,
                          LITHOLOGY_START + j*CELL_WIDTH + i*10,
                          Y_GRID_START + n*5 );
         }
       }

//       for (i=0; i<iCell; i++)
//       {
//         plot.drawCell( g, iStreaks, iCount,
//                        STREAK_START + i*10, Y_GRID_START + n*5 );
//       }

       iCount++;
     }

//     sAbrev = new String(sAbbreviation);
//     if (sAbbreviation.length() > 25)
//       sAbrev = new String(sAbrev.substring(0,25));

//     g.setColor(Color.black);
//     g.drawString(sAbrev, ABBREV_START, TEXT_START);
   }

   /** Method paint(Graphics g)
    * <p> This method will paint Porosity & Permeability Plot
    * @param g - Graphics Parameter.
    */

   public void paint(Graphics g)
   {
     g.setColor(Color.white);
     g.fillRect(0, 0, iWidth, iHeight);

     drawLithology(g);
     drawGrid(g);
   }
}