/*
 * @lithologyUtility.java Version 1.1 09/14/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratShaleListStruct;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;

import las.lasFileDataStruct;
import rock.rockDataListStruct;
import rock.rockFileDataStruct;

/** Class lithologyUtility
 *  <p> This Class will provide basic utilities for the Rock Lithology
 *      data structures.
 *
 *  @version 1.1 09/14/2007
 *  @author  John R. Victorine
 */

public class lithologyUtility
{
  /** Method buildNew()
   * <p> This method will create a new Rock Lithology Structure based on the
   *     depth range.
   * @param  iUnit      = Depth Units, 0-Feet and 1-Inches;  Default is Feet
   * @param  depthStart = The Starting Depth of data
   * @param  depthEnd   = The Ending Depth of Data
   * @return st         = The new Rock Lithology List Data Struture
   */

  public static lithologyListStruct buildNew(int    iUnit,
                                             double depthStart,
                                             double depthEnd)
  {
    lithologyListStruct st = null;
    int    iTotal  = 0;
    int    iCount  = 0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dScale  = 0.0;
    int    iTemp   = 0;

    iCount = 1+ (int) (iqstrat.iqstratTracksStruct._SCALE_FACTOR[iUnit] *
                    Math.abs( depthEnd - depthStart ));
    dScale = iqstrat.iqstratTracksStruct._SCALE[iUnit];

    st            = new lithologyListStruct();
    st.iCount     = iCount;
    st.iUnit      = iUnit;
    st.depthStart = depthStart;
    st.depthEnd   = depthEnd;
    st.stItem     = new lithologyStruct[iCount];

    for (int i=0; i<iCount; i++)
    {
      if (iTotal < iCount)
      {
        depth1 = depthStart + i * dScale;
        depth2 = depthStart + (i+1) * dScale;

        st.stItem[iTotal] = new lithologyStruct();
        st.stItem[iTotal].depthStart = getDepth(iUnit, depth1);
        st.stItem[iTotal].depthEnd   = getDepth(iUnit, depth2);
        st.stItem[iTotal].iLithology = new int[10];

        for (int j=0; j<10; j++)
          st.stItem[iTotal].iLithology[j] = 0;

        st.stItem[iTotal].sAbbreviation = new String("");
        st.stItem[iTotal].iStreaks      = 0;

        iTotal++;
      }
    }

    st.iCount = iTotal;

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock lithology data list structure
   * @return  st = the Rock lithology data list structure
   */

  public static lithologyListStruct computeDepthRange(lithologyListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method getUnit()
   * <p> This method will determine the smallest division of the data and set
   *     the spacing accordingly.
   * @param  depth = the depth
   * @return iUnit = The smallest unit
   */

  public static int getUnit(double depth)
  {
    int iUnit = iqstrat.iqstratTracksStruct._SCALE_1_FOOT;

    if (depth <= 0.5)     iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
    if (depth <= 0.16667) iUnit = iqstrat.iqstratTracksStruct._SCALE_2_INCH;
    if (depth <= 0.08333) iUnit = iqstrat.iqstratTracksStruct._SCALE_1_INCH;
    if (depth <= 0.04167) iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_INCH;
    if (depth <= 0.25)    iUnit = iqstrat.iqstratTracksStruct._SCALE_QUARTER_FOOT;
    if (depth <= 0.1)     iUnit = iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT;

    return (iUnit);
  }

  /** Method getUnit()
   * <p> This method will determine the smallest division of the data and set
   *     the spacing accordingly.
   * @param  iUnit = the scale factor the depth is being divided into
   * @param  depth = the unit depth
   * @return iUnit = The smallest unit
   */

  public static int getUnit( int iUnit, double depth )
  {
    int    iTail  = (int) depth;
    int    iTail1 = iTail;
    double dTemp  = 0.0;
    double dTail  = Math.abs( depth - (double) iTail );

    if (dTail == 0.0) dTail = 0.5;

    if (dTail <= 0.5)     iUnit = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT;
    if (dTail <= 0.16667) iUnit = iqstrat.iqstratTracksStruct._SCALE_2_INCH;
    if (dTail <= 0.1)     iUnit = iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT;

    return (iUnit);
  }

  /** Method getDepth()
   * <p> This method will return the depth nearest the scale depth
   * @param  iUnit = the scale factor the depth is being divided into
   * @param  depth = the depth being modified
   * @return depth = the new computed depth
   */

  public static double getDepth(int iUnit, double depth)
  {
    int    iTail  = (int) depth;
    int    iTail1 = iTail;
    double dTemp  = 0.0;
    double dTail  = Math.abs( depth - (double) iTail );
    int    iNeg   = 1;

    if (depth < 0.0)
    {
      iTail1 = (int) Math.abs( depth );
      if (iTail1 == 0) iTail1 = 1;
      iNeg = iTail / iTail1;
    }

    switch(iUnit)
    {
      case iqstrat.iqstratTracksStruct._SCALE_HALF_INCH: // each cell 1/2 in deep
        if (dTail <= .020835)                        dTemp = 0.0;     // 0 in
        if ((dTail > .020835) && (dTail <= .0625))   dTemp = 0.04167; // .5 in
        if ((dTail > .0625)   && (dTail <= .104165)) dTemp = 0.08333; // 1 in
        if ((dTail > .104165) && (dTail <= .145835)) dTemp = 0.125;   // 1.5 in
        if ((dTail > .145835) && (dTail <= .1875))   dTemp = 0.16667; // 2 in
        if ((dTail > .1875)   && (dTail <= .229165)) dTemp = 0.20833; // 2.5 in
        if ((dTail > .229165) && (dTail <= .270835)) dTemp = 0.25;    // 3.0 in
        if ((dTail > .270835) && (dTail <= .3125))   dTemp = 0.29167; // 3.5 in
        if ((dTail > .3125)   && (dTail <= .354165)) dTemp = 0.33333; // 4 in
        if ((dTail > .354165) && (dTail <= .395835)) dTemp = 0.375;   // 4.5 in
        if ((dTail > .395835) && (dTail <= .4375))   dTemp = 0.41667; // 5 in
        if ((dTail > .4375)   && (dTail <= .479165)) dTemp = 0.45833; // 5.5 in
        if ((dTail > .479165) && (dTail <= .520835)) dTemp = 0.5;     // 6.0 in
        if ((dTail > .520835) && (dTail <= .5625))   dTemp = 0.54167; // 6.5 in
        if ((dTail > .5625)   && (dTail <= .604165)) dTemp = 0.58333; // 7.0 in
        if ((dTail > .604165) && (dTail <= .645835)) dTemp = 0.625;   // 7.5 in
        if ((dTail > .645835) && (dTail <= .6875))   dTemp = 0.66667; // 8.0 in
        if ((dTail > .6875)   && (dTail <= .729165)) dTemp = 0.70833; // 8.5 in
        if ((dTail > .729165) && (dTail <= .790835)) dTemp = 0.75;    // 9.0 in
        if ((dTail > .790835) && (dTail <= .8125))   dTemp = 0.79167; // 9.5 in
        if ((dTail > .8125)   && (dTail <= .854165)) dTemp = 0.83333; // 10 in
        if ((dTail > .854165) && (dTail <= .915835)) dTemp = 0.875;   // 10.5 in
        if ((dTail > .915835) && (dTail <= .9375))   dTemp = 0.91667; // 11 in
        if ((dTail > .9375)   && (dTail <= .979165)) dTemp = 0.95833; // 11.5 in
        if (dTail >  .979165)                        dTemp = 1.0;     // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_1_INCH:    // each cell 1 in deep
        if (dTail <= .04167)                       dTemp = 0.0;           // 0 in
        if ((dTail > .04167) && (dTail <= .125))   dTemp = 0.08333; // 1 in
        if ((dTail > .125)   && (dTail <= .20833)) dTemp = 0.16667; // 2 in
        if ((dTail > .20833) && (dTail <= .29167)) dTemp = 0.25;    // 3 in
        if ((dTail > .29167) && (dTail <= .375))   dTemp = 0.33333; // 4 in
        if ((dTail > .375)   && (dTail <= .45833)) dTemp = 0.41667; // 5 in
        if ((dTail > .45833) && (dTail <= .54167)) dTemp = 0.5;     // 6 in
        if ((dTail > .54167) && (dTail <= .625))   dTemp = 0.58333; // 7 in
        if ((dTail > .625)   && (dTail <= .70833)) dTemp = 0.66667; // 8 in
        if ((dTail > .70833) && (dTail <= .79167)) dTemp = 0.75;    // 9 in
        if ((dTail > .79167) && (dTail <= .875))   dTemp = 0.83333; // 10 in
        if ((dTail > .875)   && (dTail <= .95833)) dTemp = 0.91667; // 11 in
        if (dTail >  .95833)                       dTemp = 1.0;     // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_2_INCH:    // each cell 2 in deep
        if (dTail <= .08333)                       dTemp = 0.0;            // 0 in
        if ((dTail > .08333) && (dTail <= .25))    dTemp = 0.16667;  // 2 in
        if ((dTail > .25)    && (dTail <= .41667)) dTemp = 0.33333;  // 4 in
        if ((dTail > .41667) && (dTail <= .58333)) dTemp = 0.5;      // 6 in
        if ((dTail > .58333) && (dTail <= .75))    dTemp = 0.66667;  // 8 in
        if ((dTail > .75)    && (dTail <= .91667)) dTemp = 0.83333;  // 10 in
        if (dTail >  .91667)                       dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_10TH_FOOT:
        if (dTail <= .05)                       dTemp = 0.0;      // 0 ft
        if ((dTail > .05)  && (dTail <= .105))  dTemp = 0.1;      // 3 in
        if ((dTail > .105) && (dTail <= .205))  dTemp = 0.2;      // 3 in
        if ((dTail > .205) && (dTail <= .305))  dTemp = 0.3;      // 3 in
        if ((dTail > .305) && (dTail <= .405))  dTemp = 0.4;      // 3 in
        if ((dTail > .405) && (dTail <= .505))  dTemp = 0.5;      // 3 in
        if ((dTail > .505) && (dTail <= .605))  dTemp = 0.6;      // 3 in
        if ((dTail > .605) && (dTail <= .705))  dTemp = 0.7;      // 3 in
        if ((dTail > .705) && (dTail <= .805))  dTemp = 0.8;      // 3 in
        if ((dTail > .805) && (dTail <= .905))  dTemp = 0.9;      // 3 in
        if (dTail >  .905)                      dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_QUARTER_FOOT:
        if (dTail <= .125)                     dTemp = 0.0;      // 0 in
        if ((dTail > .125) && (dTail <= .375)) dTemp = 0.25;     // 3 in
        if ((dTail > .375) && (dTail <= .625)) dTemp = 0.5;      // 6 in
        if ((dTail > .625) && (dTail <= .875)) dTemp = 0.75;     // 9 in
        if (dTail >  .75)                      dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT: // each cell 1/2 ft deep
        if (dTail <= .25)                     dTemp = 0.0;            // 0 in
        if ((dTail > .25) && (dTail <= .75))  dTemp = 0.5;      // 6 in
        if (dTail >  .75)                     dTemp = 1.0;      // 12 in
        break;
      case iqstrat.iqstratTracksStruct._SCALE_1_FOOT:    // each cell 1 ft deep
        if (dTail <= .5) dTemp = 0.0;
        if (dTail >  .5) dTemp = 1.0;
        break;
    }

//    depth = (double) iTail + dTemp;
    depth = (double) iNeg * (iTail1 + dTemp);

    return (depth);
  }

  /** Method fillEmptySpaces()
   * <p> This method will fill empty spaces within the depth range
   * @param  st = The Rock Lithology List Data Structure
   * @return st = The Rock Lithology List Data Structure
   */

  public static lithologyListStruct fillEmptySpaces(lithologyListStruct st)
  {
    int i = 0;
    int j = 0;
    int iFilled1 = 0;
    int iFilled2 = 0;
    int iFilled3 = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount-2; i++)
      {
        iFilled1 = 0;
        for (j=0; j<10; j++)
        {
          if (st.stItem[i].iLithology[j] !=
                lith.lithology.lithologySymbolsStruct._NONE)
            iFilled1 = 1;
        }

        iFilled2 = 0;
        iFilled3 = 0;
        if (iFilled1 != 0)
        {
          for (j=0; j<10; j++)
          {
            if (st.stItem[i+1].iLithology[j] !=
                  lith.lithology.lithologySymbolsStruct._NONE)
              iFilled2 = 1;
          }

          for (j=0; j<10; j++)
          {
            if (st.stItem[i+2].iLithology[j] !=
                  lith.lithology.lithologySymbolsStruct._NONE)
              iFilled3 = 1;
          }

          if ((iFilled2 == 0) && (iFilled3 == 1))
          {
            for (j=0; j<10; j++)
            {
              st.stItem[i+1].iLithology[j] = st.stItem[i].iLithology[j];
            }

            st.stItem[i+1].sAbbreviation =
                new String(st.stItem[i].sAbbreviation);
            st.stItem[i+1].iStreaks = st.stItem[i].iStreaks;
          }
        }
      }
    }

    return (st);
  }

  /** Method addHeaderData()
   * <p> This method will add the header data to the Lithology data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Lithology List Data Structure
   * @return st       = The Rock Lithology List Data Structure
   */

  public static lithologyListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                  lithologyListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method addRecord( lithologyStruct st, lithologyListStruct stList)
   * <p> This method will add a row to the lithology List data structure
   * @param  st     = Rock Lithology data structure to be added
   * @param  stList = the old Rock Lithology List data structure.
   * @return stList = the new Rock Lithology List data structure.
   */

  public static lithologyListStruct addRecord( lithologyStruct st,
                                               lithologyListStruct stList)
  {
    double depthStart = st.depthStart;
    double depthEnd   = st.depthEnd;

    if ((st != null) && (stList != null))
    {
      for (int i=0; i<stList.iCount; i++)
      {
        if ((depthStart >= stList.stItem[i].depthStart) &&
            (depthStart <  stList.stItem[i].depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
        else if ((depthEnd > stList.stItem[i].depthStart) &&
                 (depthEnd <= stList.stItem[i].depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
        else if ((stList.stItem[i].depthStart >= depthStart) &&
                 (stList.stItem[i].depthEnd <= depthEnd))
        {
          st.depthStart = stList.stItem[i].depthStart;
          st.depthEnd   = stList.stItem[i].depthEnd;
          stList.stItem[i] = copy(st);
        }
      }
    }

    return (stList);
  }

  /** Method add()
   * <p> This method will add a Lithology to an existing list
   * @param  stNew = The Rock Lithology Data Structure
   * @param  st    = The Old Rock Lithology List Data Structure
   * @return st    = The new Rock Lithology List.
   */

  public static lithologyListStruct add( lithologyStruct stNew,
                                         lithologyListStruct st)
  {
    int i = 0;
    int iCount      = 0;
    lithologyListStruct stTemp = new lithologyListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new lithologyStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp.depthStart = st.depthStart;
      stTemp.depthEnd   = st.depthEnd;
      stTemp.iUnit      = st.iUnit;

      if (st.iCount > 0)
        stTemp.stItem = new lithologyStruct[st.iCount+1];

      for (i=0; i<st.iCount; i++)
      {
        stTemp.stItem[iCount] = new lithologyStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (stNew != null)
    {
      stTemp.stItem[iCount] = new lithologyStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new lithologyListStruct();
    st.iCount = iCount;
    st.stItem = new lithologyStruct[st.iCount];

    st.depthStart = stTemp.depthStart;
    st.depthEnd   = stTemp.depthEnd;
    st.iUnit      = stTemp.iUnit;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<st.iCount; i++)
    {
      st.stItem[i] = new lithologyStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Lithology in an existing list
   * @param  stNew = The New Rock Lithology Data Structure
   * @param  st   = The Old Rock Lithology List Data Structure
   * @return st   = The new Rock Lithology List.
   */

  public static lithologyListStruct modify( lithologyStruct stNew,
                                            lithologyListStruct st)
  {
    int i = 0;
    int iCount      = 0;
    lithologyListStruct stTemp = new lithologyListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new lithologyStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp.depthStart = st.depthStart;
      stTemp.depthEnd   = st.depthEnd;
      stTemp.iUnit      = st.iUnit;

      if (st.iCount > 0)
        stTemp.stItem = new lithologyStruct[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(stNew.sKEY))
        {
          stTemp.stItem[iCount] = new lithologyStruct();
          stTemp.stItem[iCount] = copy(stNew);
          iCount++;
        }
        else
        {
          stTemp.stItem[iCount] = new lithologyStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new lithologyListStruct();
      st.iCount = iCount;
      st.stItem = new lithologyStruct[st.iCount];

      st.depthStart = stTemp.depthStart;
      st.depthEnd   = stTemp.depthEnd;
      st.iUnit      = stTemp.iUnit;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<st.iCount; i++)
      {
        st.stItem[i] = new lithologyStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Lithology from an existing list
   * @param  sKEY = Unique Identifier for depth range
   * @param  st   = The Old Rock Lithology List Data Structure
   * @return st   = The new Rock Lithology List.
   */

  public static lithologyListStruct remove(String sKEY, lithologyListStruct st)
  {
    int i = 0;
    int iCount      = 0;
    lithologyListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount < 2)
      {
        st.iCount = 0;
        st.delete();
        st = null;
      }
      else
      {
        stTemp        = new lithologyListStruct();
        stTemp.stItem = new lithologyStruct[st.iCount-1];

        stTemp.depthStart = st.depthStart;
        stTemp.depthEnd   = st.depthEnd;
        stTemp.iUnit      = st.iUnit;

        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i] != null)
          {
            if (!st.stItem[i].sKEY.equals(sKEY))
            {
              stTemp.stItem[iCount] = new lithologyStruct();
              stTemp.stItem[iCount] = copy(st.stItem[i]);
              iCount++;
            }
          }
        }

        st.delete();
        st = null;

        st        = new lithologyListStruct();
        st.iCount = iCount;
        st.stItem = new lithologyStruct[st.iCount];

        st.depthStart = stTemp.depthStart;
        st.depthEnd   = stTemp.depthEnd;
        st.iUnit      = stTemp.iUnit;

        st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
        st.source   = new String( source );    // Source of Data
        st.sCreated = new String( sCreated );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          st.stItem[i] = new lithologyStruct();
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        if (stTemp != null)
          stTemp.delete();
        stTemp = null;
      }
    }

    return (st);
  }

  /** Method copyList( lithologyListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static lithologyListStruct copyList( lithologyListStruct stOld )
  {
    lithologyListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new lithologyListStruct();

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth
      stNew.iUnit      = stOld.iUnit;

      // LAS 3 Dependent Variable

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new lithologyStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static lithologyStruct copy(lithologyStruct stOld)
  {
    lithologyStruct stNew = new lithologyStruct();

    stNew.sKEY        = new String(stOld.sKEY);

    stNew.depthStart  = stOld.depthStart; // Starting Depth
    stNew.depthEnd    = stOld.depthEnd; // Ending Depth

    stNew.dRhomaa     = stOld.dRhomaa;
    stNew.dUmaa       = stOld.dUmaa;
    stNew.dGR         = stOld.dGR;

    for (int i=0; i<lith.lithology.lithologyStruct.TOTAL; i++)
      stNew.iLithology[i] = stOld.iLithology[i];

    stNew.sAbbreviation = new String(stOld.sAbbreviation);
    stNew.iStreaks      = stOld.iStreaks;

    return (stNew);
  }

  /** Method computeLithology()
   * <p> This method will compute the lithology from the digital LAS File.
   *  @param  stLAS      = The LAS File Data Structure
   *  @param  stROCK     = The Rock Data List Data Structure
   *  @param  stSymbols  = Lithology Symbols Data List Structure
   *  @param  st         = The Lithology Data List Structure
   *  @return st         = The Lithology Data List Structure
   */

  public static lithologyListStruct computeLithology(
                                        lasFileDataStruct          stLAS,
                                        rockFileDataStruct         stROCK,
                                        lithologySymbolsListStruct stSymbols,
                                        lithologyListStruct        st)
  {
    int    iRows       = 0;
    int    iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_RU;
    double depthStart  = 0.0;
    double depthEnd    = 0.0;
    double depthStep   = 0.0;
    double dNull       = -999.95;
    double depths[]    = null;
    double dGR[]       = null;
    double dRHO[]      = null;
    double dPHI[]      = null;
    double dPE[]       = null;
    double dTh[]       = null;
    double dK[]        = null;
    double dU[]        = null;
    double dDT[]       = null;

    iqstratShaleListStruct stList = null;

    if (stLAS != null)
    {
      iRows      = stLAS.iRows;
      depthStart = stLAS.depthStart;
      depthEnd   = stLAS.depthEnd;
      depthStep  = stLAS.depthStep;
      dNull      = stLAS.dNull;

      if (stLAS.iPE == -1)
        iColorlith = iqstrat.iqstratTracksStruct._COLORLITH_PD;

      depths    = stLAS.depths;
      dGR       = stLAS.dGR;
      dPHI      = stLAS.dNPHI;
      dRHO      = stLAS.dRHOB;
      dPE       = stLAS.dPE;
      dTh       = stLAS.dTh;
      dU        = stLAS.dU;
      dK        = stLAS.dK;
      dDT       = stLAS.dDT;

      stList    = stLAS.stList;
    }

    if (stROCK != null)
    {
      iRows       = stROCK.iRows;
      depthStart  = stROCK.depthStart;
      depthEnd    = stROCK.depthEnd;
      depthStep   = stROCK.depthStep;
      dNull       = stROCK.dNull;

      iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_PD;

      depths      = stROCK.depths;
      dGR         = stROCK.dGRL;
      dPHI        = stROCK.dPHI;
      dRHO        = rock.rockFileDataUtility.computeBulkDensity( stROCK );
      dTh         = stROCK.dTh;
      dU          = stROCK.dU;
      dK          = stROCK.dK;

      dPE         = new double[iRows];
      dDT         = new double[iRows];
      for (int i=0; i<iRows-1; i++)
      {
        dPE[i] = dNull;
        dDT[i] = dNull;
      }
    }

	st = computeLithology( iColorlith,
	                       depthStart, depthEnd, depthStep, dNull,
	                       iRows, depths, dGR, dRHO, dPHI, dPE, dTh, dK, dU, dDT,
	                       stList, stSymbols, st );

	return (st);
  }

  /** Method computeLithology()
   * <p> This method will compute the lithology from the digital LAS File.
   *  @param  stLAS      = The LAS File Data Structure
   *  @param  stSymbols  = Lithology Symbols Data List Structure
   *  @param  st         = The Lithology Data List Structure
   *  @return st         = The Lithology Data List Structure
   */

  public static lithologyListStruct computeLithology(
                                        lasFileDataStruct          stLAS,
                                        lithologySymbolsListStruct stSymbols,
                                        lithologyListStruct        st)
  {
    int    iRows       = 0;
    int    iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_RU;
    double depthStart  = 0.0;
    double depthEnd    = 0.0;
    double depthStep   = 0.0;
    double dNull       = -999.95;
    double depths[]    = null;
    double dGR[]       = null;
    double dRHO[]      = null;
    double dPHI[]      = null;
    double dPE[]       = null;
    double dTh[]       = null;
    double dK[]        = null;
    double dU[]        = null;
    double dDT[]       = null;

    iqstratShaleListStruct stList = null;

    if (stLAS != null)
    {
      iRows      = stLAS.iRows;
      depthStart = stLAS.depthStart;
      depthEnd   = stLAS.depthEnd;
      depthStep  = stLAS.depthStep;
      dNull      = stLAS.dNull;

      if (stLAS.iPE == -1)
        iColorlith = iqstrat.iqstratTracksStruct._COLORLITH_PD;

      depths    = stLAS.depths;
      dGR       = stLAS.dGR;
      dPHI      = stLAS.dNPHI;
      dRHO      = stLAS.dRHOB;
      dPE       = stLAS.dPE;
      dTh       = stLAS.dTh;
      dU        = stLAS.dU;
      dK        = stLAS.dK;
      dDT       = stLAS.dDT;

      stList    = stLAS.stList;

  	  st = computeLithology( iColorlith,
	                         depthStart, depthEnd, depthStep, dNull,
	                         iRows, depths, dGR, dRHO, dPHI, dPE, dTh, dK, dU, dDT,
	                         stList, stSymbols, st );
    }

	return (st);
  }

  /** Method computeLithology()
   * <p> This method will compute the lithology from the digital LAS File.
   *  @param  iColorlith = Indicator of the type of lithology column to create
   *                       - _COLORLITH_RU = RHOMAA-UMAA-GR
   *                       - _COLORLITH_PD = RHOMAA-NPHI-GR
   *                       - _COLORLITH_RT = RHOMAA-DTMAA-GR
   *  @param  stLAS      = The LAS File Data Structure
   *  @param  stSymbols  = Lithology Symbols Data List Structure
   *  @param  st         = The Lithology Data List Structure
   *  @return st         = The Lithology Data List Structure
   */

  public static lithologyListStruct computeLithology(
	                                    int                        iColorlith,
                                        lasFileDataStruct          stLAS,
                                        lithologySymbolsListStruct stSymbols,
                                        lithologyListStruct        st)
  {
    int    iRows       = 0;
    double depthStart  = 0.0;
    double depthEnd    = 0.0;
    double depthStep   = 0.0;
    double dNull       = -999.95;
    double depths[]    = null;
    double dGR[]       = null;
    double dRHO[]      = null;
    double dPHI[]      = null;
    double dPE[]       = null;
    double dTh[]       = null;
    double dK[]        = null;
    double dU[]        = null;
    double dDT[]       = null;

    iqstratShaleListStruct stList = null;

    if (stLAS != null)
    {
      iRows      = stLAS.iRows;
      depthStart = stLAS.depthStart;
      depthEnd   = stLAS.depthEnd;
      depthStep  = stLAS.depthStep;
      dNull      = stLAS.dNull;

      if (iColorlith == iqstrat.iqstratTracksStruct._COLORLITH_RU)
      {
        if (stLAS.iPE == -1)
          iColorlith = iqstrat.iqstratTracksStruct._COLORLITH_PD;
	  }

      depths    = stLAS.depths;
      dGR       = stLAS.dGR;
      dPHI      = stLAS.dNPHI;
      dRHO      = stLAS.dRHOB;
      dPE       = stLAS.dPE;
      dTh       = stLAS.dTh;
      dU        = stLAS.dU;
      dK        = stLAS.dK;
      dDT       = stLAS.dDT;

      stList    = stLAS.stList;

  	  st = computeLithology( iColorlith,
	                         depthStart, depthEnd, depthStep, dNull,
	                         iRows, depths, dGR, dRHO, dPHI, dPE, dTh, dK, dU, dDT,
	                         stList, stSymbols, st );
    }

	return (st);
  }

  /** Method computeLithology()
   * <p> This method will compute the lithology from the digital LAS File.
   *  @param  stROCK     = The Rock Data List Data Structure
   *  @param  stSymbols  = Lithology Symbols Data List Structure
   *  @param  st         = The Lithology Data List Structure
   *  @return st         = The Lithology Data List Structure
   */

  public static lithologyListStruct computeLithology(
                                        rockFileDataStruct         stROCK,
                                        lithologySymbolsListStruct stSymbols,
                                        lithologyListStruct        st)
  {
    int    iRows       = 0;
    int    iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_RU;
    double depthStart  = 0.0;
    double depthEnd    = 0.0;
    double depthStep   = 0.0;
    double dNull       = -999.95;
    double depths[]    = null;
    double dGR[]       = null;
    double dRHO[]      = null;
    double dPHI[]      = null;
    double dPE[]       = null;
    double dTh[]       = null;
    double dK[]        = null;
    double dU[]        = null;
    double dDT[]       = null;

    iqstratShaleListStruct stList = null;

    if (stROCK != null)
    {
      iRows       = stROCK.iRows;
      depthStart  = stROCK.depthStart;
      depthEnd    = stROCK.depthEnd;
      depthStep   = stROCK.depthStep;
      dNull       = stROCK.dNull;

      iColorlith  = iqstrat.iqstratTracksStruct._COLORLITH_PD;

      depths      = stROCK.depths;
      dGR         = stROCK.dGRL;
      dPHI        = stROCK.dPHI;
      dRHO        = rock.rockFileDataUtility.computeBulkDensity( stROCK );
      dTh         = stROCK.dTh;
      dU          = stROCK.dU;
      dK          = stROCK.dK;

      dPE         = new double[iRows];
      dDT         = new double[iRows];
      for (int i=0; i<iRows-1; i++)
      {
        dPE[i] = dNull;
        dDT[i] = dNull;
      }

  	  st = computeLithology( iColorlith,
	                         depthStart, depthEnd, depthStep, dNull,
	                         iRows, depths, dGR, dRHO, dPHI, dPE, dTh, dK, dU, dDT,
	                         stList, stSymbols, st );
    }

	return (st);
  }

  /** Method computeLithology()
   * <p> This method will compute the lithology from the digital LAS File.
   *  @param  iColorlith = Indicator of the type of lithology column to create
   *                       - _COLORLITH_RU = RHOMAA-UMAA-GR
   *                       - _COLORLITH_PD = RHOMAA-NPHI-GR
   *                       - _COLORLITH_RT = RHOMAA-DTMAA-GR
   *  @param  depthStart = Starting Depth of Data
   *  @param  depthEnd   = Ending Depth of Data
   *  @param  depthStep  = Step Size between data
   *  @param  dNull      = NULL Value - No Data indicator
   *  // Well or Core Data
   *  @param  iRows      = Total Number of Records in the Array
   *  @param  depths     = Array of Depths
   *  @param  dGR        = Array of Gamma Ray Data
   *  @param  dRHO       = Array of Bulk Density Data
   *  @param  dPHI       = Array of Neutron Porosity or Core Porosity Data
   *  @param  dPE        = Array of Photoelectric Factor Data
   *  @param  dTh        = Array of Thorium Data
   *  @param  dK         = Array of Potassium Data
   *  @param  dU         = Array of Uranium Data
   *  @param  dDT        = Array of Acoustic Transit Time Data
   *  @param  stList     = Shale List Data Structure - Override of Gamma Ray Values/Depth Ranges
   *  @param  stSymbols  = Lithology Symbols Data List Structure
   *  @param  st         = The Lithology Data List Structure
   *  @return st         = The Lithology Data List Structure
   */

  public static lithologyListStruct computeLithology(
                      int    iColorlith,
                      double depthStart,
                      double depthEnd,
                      double depthStep,
                      double dNull,
		              int    iRows,
                      double depths[],
                      double dGR[],
                      double dRHO[],
                      double dPHI[],
                      double dPE[],
                      double dTh[],
                      double dK[],
                      double dU[],
                      double dDT[],
                      iqstratShaleListStruct     stList,
                      lithologySymbolsListStruct stSymbols,
                      lithologyListStruct        st)
  {
    int i, j, k, m, n, p;
    int    iTemp       = 0;
    double dSum        = 0.0;
    double dTemp       = 0.0;
    int    iDolomite   = 0;
    int    iLimestone  = 0;
    int    iSand       = 0;
    int    iTotal      = 0;
    int    iCount      = 0;
    int    iStart      = 0;
    int    iEnd        = 0;
    int    iLith       = 0;
    double depth1      = 0.0;
    double depth2      = 0.0;
    double dShaly      = iqstrat.iqstratShaleStruct.SHALY;
    double dShale      = iqstrat.iqstratShaleStruct.SHALE;
    double dHotShale   = iqstrat.iqstratShaleStruct.SHALE_HOT;

    int    iLithology  = 0;
    int    iLithColor  = 0;
    int    iContinue   = 0;

    int    len         = 0;
    int    iMin[]      = null;
    int    iMin_Lith[] = null;
    int    iMin_Col[]  = null;
    double dLith[]     = { 0.0, 0.0, 0.0, 0.0 , 0.0 };
    double dIN[]       = null;
    double dMat[][]    = null;

    if (iRows > 0)
    {
	  // Depending on Log or Core Data Compute Depth Step

	  // Depth Step is 2 Feet
      if (depthStep == 2.0)
      {
        st = buildNew(
            iqstrat.iqstratTracksStruct._SCALE_2_FOOT, depthStart, depthEnd);
      }
	  // Depth Step is 1 Foot
      else if (depthStep == 1.0)
      {
        st = buildNew(
            iqstrat.iqstratTracksStruct._SCALE_1_FOOT, depthStart, depthEnd);
      }
	  // Depth Step is 1/2 Foot
      else
      {
        st = buildNew(
            iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT, depthStart, depthEnd);
      }

      for (i=0; i<iRows-1; i++)
      {
		iContinue = 0;

		if ((iColorlith == iqstrat.iqstratTracksStruct._COLORLITH_RU) ||
			(iColorlith == iqstrat.iqstratTracksStruct._COLORLITH_PD))
		{
          if ((dPHI[i] > dNull) && (dRHO[i] > dNull))
          {
            iContinue = 1;
		  }
		}
        else if (iColorlith == iqstrat.iqstratTracksStruct._COLORLITH_RT)
        {
          if ((dDT[i] > dNull) && (dRHO[i] > dNull))
            iContinue = 1;
		}

        if (iContinue == 1)
        {
          depth1 = depths[i];
          depth2 = depths[i+1];

          // Bin the Data to the Lithology Depth Data
          // IF depth is not a standard step initialize to a standard step

          if (st != null)
          {
            for (k=0; k<st.iCount; k++)
            {
              if (st.stItem[k].depthStart == depth1)
                iLith = k;
              else if (( st.stItem[k].depthStart > depth1 ) &&
                       ( st.stItem[k].depthStart < depth2 ))
                iLith = k;
            }
          }

          // Get the Shale Data Values

          dShaly      = iqstrat.iqstratShaleStruct.SHALY;
          dShale      = iqstrat.iqstratShaleStruct.SHALE;
          dHotShale   = iqstrat.iqstratShaleStruct.SHALE_HOT;
          if (stList != null)
          {
            for (n=0; n<stList.iCount; n++)
            {
              if ((depth1 >= stList.stItem[n].depthStart) &&
                  (depth1 <= stList.stItem[n].depthEnd))
              {
                dShaly    = stList.stItem[n].dShaly;
                dShale    = stList.stItem[n].dShale;
                dHotShale = stList.stItem[n].dHotShale;
              }
            }
          }

          // Determine a General "Lithology" from Log or Core Data
          // -- Bin the minerals

          iLithology = lith.math.lithMathRules.getLithology(
                         math.mathLAS._FRESH,
                         dShaly, dShale,  dHotShale,
                         dGR[i], dPHI[i], dRHO[i], dPE[i], dDT[i],
                         dNull);

          // Load default Lithology into every cell

          for (j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
          {
            st.stItem[iLith].iLithology[j] = iLithology;
          }

          iContinue = 0;

          switch (iColorlith)
          {
            /* ------------------------------------------------------- *
             * ----------------- RHOMAA - UMAA LOGS ------------------ *
             * ------------------------------------------------------- */

            case iqstrat.iqstratTracksStruct._COLORLITH_RU:
              switch (iLithology)
              {
                // ............................................................
                // -- GENERAL SEDIMENTARY LITHOLOGY
				// QUARTZ - CALCITE - DOLOMITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._DOLOMITE:
                case lith.lithology.lithologySymbolsStruct._LIMESTONE:
                case lith.lithology.lithologySymbolsStruct._SANDSTONE:
                case lith.lithology.lithologySymbolsStruct._SHALY_DOLO:
                case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                  iMin      = lith.math.lithMath._RU;
                  iMin_Lith = lith.math.lithMath._RU_LITH;
                  iMin_Col  = lith.math.lithMath._RU_COL;

                  dIN = lith.math.lithMath.getRU(
                          lith.math.lithMath._CALCITE,
                          lith.math.lithMath._FRESH_H2O,
                          dRHO[i], dPE[i], dPHI[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- ARKOSE LITHOLOGY
				// K-FELDSPAR - QUARTZ - CALCITE - ILLITE QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ARKOSE:
                  iMin      = lith.math.lithMath._ARKOSE;
                  iMin_Lith = lith.math.lithMath._ARKOSE_LITH;
                  iMin_Col  = lith.math.lithMath._RUGR_COL;

                  dIN = lith.math.lithMath.getRUGR(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- IGNEOUS ROCKS - GRANITE LITHOLOGY
				// MUSCOVITE - Na-PLAGIOCLASE - QUARTZ, K-FELDSPAR QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._GRANITE:
                  iMin      = lith.math.lithMath._GRANITE;
                  iMin_Lith = lith.math.lithMath._GRANITE_LITH;
                  iMin_Col  = lith.math.lithMath._RUGR_COL;

                  dIN = lith.math.lithMath.getRUGR(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- GENERAL IGNEOUS ROCKS LITHOLOGY
				// ULTRA MAFIC - Ca-PLAGIOCLASE - QUARTZ, K-FELDSPAR QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._IGNEOUS:
                  iMin      = lith.math.lithMath._IGNEOUS;
                  iMin_Lith = lith.math.lithMath._IGNEOUS_LITH;
                  iMin_Col  = lith.math.lithMath._RUGR_COL;

                  dIN = lith.math.lithMath.getRUGR(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- ANHYDRITE LITHOLOGY
				// DOLOMITE - LIMESTONE - ANHYDRITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ANHYDRITE:
                  iMin      = lith.math.lithMath._ANHY;
                  iMin_Lith = lith.math.lithMath._ANHY_LITH;
                  iMin_Col  = lith.math.lithMath._RU_COL;

                  dIN = lith.math.lithMath.getRU(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- GYPSUM LITHOLOGY
				// DOLOMITE - LIMESTONE - GYPSUM TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._GYPSUM:
                  iMin      = lith.math.lithMath._GYP;
                  iMin_Lith = lith.math.lithMath._GYP_LITH;
                  iMin_Col  = lith.math.lithMath._RU_COL;

                  dIN = lith.math.lithMath.getRU(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- SALT & COAL DEFAULT LITHOLOGIES
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._HALITE:
                case lith.lithology.lithologySymbolsStruct._COAL:
                  break;
                // ............................................................
                // -- SHALE/CLAY LITHOLOGY
				// SMECTITE - ILLITE - CALCITE - QUARTZ QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._SHALE:
                  iMin      = lith.math.lithMath._RUGR;
                  iMin_Lith = lith.math.lithMath._RUGR_LITH;
                  iMin_Col  = lith.math.lithMath._RUGR_COL;

                  dIN = lith.math.lithMath.getRUGR(
                      lith.math.lithMath._CALCITE,
                      lith.math.lithMath._FRESH_H2O,
                      dRHO[i], dPE[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
              }
              break;

            /* ------------------------------------------------------- *
             * ---- NEUTRON POROSITY & POROSITY DIFFERENCE LOGS ------ *
             * ------------------------------------------------------- */

            case iqstrat.iqstratTracksStruct._COLORLITH_PD:
              switch (iLithology)
              {
                // ............................................................
                // -- GENERAL SEDIMENTARY LITHOLOGY
				// QUARTZ - CALCITE - DOLOMITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._DOLOMITE:
                case lith.lithology.lithologySymbolsStruct._LIMESTONE:
                case lith.lithology.lithologySymbolsStruct._SANDSTONE:
                case lith.lithology.lithologySymbolsStruct._SHALY_DOLO:
                case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                  iMin      = lith.math.lithMath._RU;
                  iMin_Lith = lith.math.lithMath._RU_LITH;
                  iMin_Col  = lith.math.lithMath._PD_COL;

                  dIN = lith.math.lithMath.getPD( lith.math.lithMath._CALCITE,
                                                  lith.math.lithMath._FRESH_H2O,
                                                  dRHO[i], dPHI[i] );

                  len       = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- ARKOSE LITHOLOGY
				// K-FELDSPAR - QUARTZ - CALCITE - ILLITE QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ARKOSE:
                  iMin      = lith.math.lithMath._ARKOSE;
                  iMin_Lith = lith.math.lithMath._ARKOSE_LITH;
                  iMin_Col  = lith.math.lithMath._PDGR_COL;

                  dIN = lith.math.lithMath.getPDGR(
                          lith.math.lithMath._CALCITE,
                          lith.math.lithMath._FRESH_H2O,
                          dRHO[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- IRONSTONE DEFAULT LITHOLOGIES
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._IRONSTONE:
                  break;
                // ............................................................
                // -- ANHYDRITE LITHOLOGY
				// DOLOMITE - LIMESTONE - ANHYDRITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ANHYDRITE:
                  iMin      = lith.math.lithMath._ANHY;
                  iMin_Lith = lith.math.lithMath._ANHY_LITH;
                  iMin_Col  = lith.math.lithMath._PD_COL;

                  dIN = lith.math.lithMath.getPD( lith.math.lithMath._CALCITE,
                                                  lith.math.lithMath._FRESH_H2O,
                                                  dRHO[i], dPHI[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- GYPSUM LITHOLOGY
				// DOLOMITE - LIMESTONE - GYPSUM TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._GYPSUM:
                  iMin      = lith.math.lithMath._GYP;
                  iMin_Lith = lith.math.lithMath._GYP_LITH;
                  iMin_Col  = lith.math.lithMath._PD_COL;

                  dIN = lith.math.lithMath.getPD( lith.math.lithMath._CALCITE,
                                                  lith.math.lithMath._FRESH_H2O,
                                                  dRHO[i], dPHI[i] );

                  len       = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- SALT & COAL DEFAULT LITHOLOGIES
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._HALITE:
                case lith.lithology.lithologySymbolsStruct._COAL:
                  break;
                // ............................................................
                // -- SHALE/CLAY LITHOLOGY
				// SMECTITE - ILLITE - CALCITE - QUARTZ QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._SHALE:
                  iMin      = lith.math.lithMath._RUGR;
                  iMin_Lith = lith.math.lithMath._RUGR_LITH;
                  iMin_Col  = lith.math.lithMath._PDGR_COL;

                  dIN = lith.math.lithMath.getPDGR(
                          lith.math.lithMath._CALCITE,
                          lith.math.lithMath._FRESH_H2O,
                          dRHO[i], dPHI[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
              }
              break;

            /* ------------------------------------------------------- *
             * ------------------ RHOMAA - DT LOGS ------------------- *
             * ------------------------------------------------------- */

            case iqstrat.iqstratTracksStruct._COLORLITH_RT:
              switch (iLithology)
              {
                // ............................................................
                // -- GENERAL SEDIMENTARY LITHOLOGY
				// QUARTZ - CALCITE - DOLOMITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._DOLOMITE:
                case lith.lithology.lithologySymbolsStruct._LIMESTONE:
                case lith.lithology.lithologySymbolsStruct._SANDSTONE:
                case lith.lithology.lithologySymbolsStruct._SHALY_DOLO:
                case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                  iMin      = lith.math.lithMath._RU;
                  iMin_Lith = lith.math.lithMath._RU_LITH;
                  iMin_Col  = lith.math.lithMath._RT_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRT( lith.math.lithMath._CALCITE,
                                                    lith.math.lithMath._FRESH_H2O,
                                                    dRHO[i], dDT[i], dPHI[i] );
                  else
                    dIN = lith.math.lithMath.getRT( lith.math.lithMath._CALCITE,
                                                    lith.math.lithMath._FRESH_H2O,
                                                    dRHO[i], dDT[i] );

                  len       = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- ARKOSE LITHOLOGY
				// K-FELDSPAR - QUARTZ - CALCITE - ILLITE QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ARKOSE:
/*
                  iMin      = lith.math.lithMath._ARKOSE;
                  iMin_Lith = lith.math.lithMath._ARKOSE_LITH;
                  iMin_Col  = lith.math.lithMath._RTGR_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dPHI[i], dGR[i]);
                  else
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
*/
                  break;
                // ............................................................
                // -- IGNEOUS ROCKS - GRANITE LITHOLOGY
				// MUSCOVITE - Na-PLAGIOCLASE - QUARTZ, K-FELDSPAR QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._GRANITE:
/*
                  iMin      = lith.math.lithMath._GRANITE;
                  iMin_Lith = lith.math.lithMath._GRANITE_LITH;
                  iMin_Col  = lith.math.lithMath._RTGR_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dPHI[i], dGR[i]);
                  else
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
*/
                  break;
                // ............................................................
                // -- GENERAL IGNEOUS ROCKS LITHOLOGY
				// ULTRA MAFIC - Ca-PLAGIOCLASE - QUARTZ, K-FELDSPAR QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._IGNEOUS:

                  iMin      = lith.math.lithMath._IGNEOUS;
                  iMin_Lith = lith.math.lithMath._IGNEOUS_LITH;
                  iMin_Col  = lith.math.lithMath._RTGR_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dPHI[i], dGR[i]);
                  else
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;

                  break;
                // ............................................................
                // -- ANHYDRITE LITHOLOGY
				// DOLOMITE - LIMESTONE - ANHYDRITE TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._ANHYDRITE:
                  iMin      = lith.math.lithMath._ANHY;
                  iMin_Lith = lith.math.lithMath._ANHY_LITH;
                  iMin_Col  = lith.math.lithMath._RT_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRT( lith.math.lithMath._CALCITE,
                                                    lith.math.lithMath._FRESH_H2O,
                                                    dRHO[i], dDT[i], dPHI[i] );
                  else
                    dIN = lith.math.lithMath.getRT( lith.math.lithMath._CALCITE,
                                                    lith.math.lithMath._FRESH_H2O,
                                                    dRHO[i], dDT[i] );

                  len       = iMin.length;
                  iContinue = 1;
                  break;
                // ............................................................
                // -- GYPSUM LITHOLOGY
				// DOLOMITE - LIMESTONE - GYPSUM TRIANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._GYPSUM:
                  break;
                // ............................................................
                // -- SALT & COAL DEFAULT LITHOLOGIES
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._HALITE:
                case lith.lithology.lithologySymbolsStruct._COAL:
                  break;
                // ............................................................
                // -- SHALE/CLAY LITHOLOGY
				// SMECTITE - ILLITE - CALCITE - QUARTZ QUADRANGLE
                // ............................................................
                case lith.lithology.lithologySymbolsStruct._SHALE:
                  iMin      = lith.math.lithMath._RUGR;
                  iMin_Lith = lith.math.lithMath._RUGR_LITH;
                  iMin_Col  = lith.math.lithMath._RTGR_COL;

                  if (dPHI[i] > dNull)
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dPHI[i], dGR[i]);
                  else
                    dIN = lith.math.lithMath.getRTGR(
                            lith.math.lithMath._CALCITE,
                            lith.math.lithMath._FRESH_H2O,
                            dRHO[i], dDT[i], dGR[i]);

                  len = iMin.length;
                  iContinue = 1;
                  break;
              }
              break;
          }

          if (iContinue == 1)
          {
			// Build Lithology Matrix

            dMat  = lith.math.lithMath.getMatrix(iMin, iMin_Col);

            // Compute Mineral Composition

            dLith = lith.math.lithMath.composition( dIN, dMat );

            // Compute the Total Sum of all Mineral Composition

            dSum = 0.0;
            for (j=0; j<len; j++)
            {
              if (dLith[j] < 0.0)
                dLith[j] = 0.0;

              dSum = dSum + dLith[j];
            }

            // FOR EACH Lithology

            for (j=0; j<len; j++)
            {
              // IF First Lithology set to first 10%.

              if (j==0) iStart = 0;

              // Normalize Lithology to the Sum of all Mineral Composition

              dTemp = (10.0 * dLith[j]) / dSum;
              iTemp = (int) dTemp;
              dTemp = dTemp - (double) iTemp;

              // Compute the End 10% value for the lithology

              iEnd = iStart + (int) ((10.0 * dLith[j]) / dSum);

              // Round Cell up to next 10%

              if (dTemp > 0.5)
                iEnd++;

              // IF total cell count is greater than 100% Set to 100%

              if (iEnd > 10)
                iEnd = 10;

              // IF end cell of the last lithology is only 90% set to 100%

              if (j == len-1)
                if (iEnd == 9)
                  iEnd = 10;

              // Retrieve the Mineral Symbols for each lithology

              iLithColor = iMin_Lith[j];

              // If Rock is Shaly or Shale reset Lithology Symbol for
              // General Sedimentary Rock Cells to Shaly

              switch (iLithColor)
              {
                case lith.lithology.lithologySymbolsStruct._DOLOMITE:
                  switch (iLithology)
                  {
                    case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                    case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                    case lith.lithology.lithologySymbolsStruct._SHALE:
                      iLithColor = lith.lithology.lithologySymbolsStruct._SHALY_DOLO;
                      break;
                  }
                  break;
                case lith.lithology.lithologySymbolsStruct._LIMESTONE:
                  switch (iLithology)
                  {
                    case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                    case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                    case lith.lithology.lithologySymbolsStruct._SHALE:
                      iLithColor = lith.lithology.lithologySymbolsStruct._SHALY_CARB;
                      break;
                  }
                  break;
                case lith.lithology.lithologySymbolsStruct._SANDSTONE:
                  switch (iLithology)
                  {
                    case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
                    case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
                    case lith.lithology.lithologySymbolsStruct._SHALE:
                      iLithColor = lith.lithology.lithologySymbolsStruct._SHALY_SAND;
                      break;
                  }
                  break;
              }

              // Load Lithology Symbol into each cell within the Start and
              // End Cells for lithology

              for (k=iStart; k<iEnd; k++)
              {
                st.stItem[iLith].iLithology[k] = iLithColor;
              }

              // Start Next Lithology in Array

              iStart = iEnd;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method setLegend( int iLith[], lithologyListStruct st )
   * <p> This method will set the legend variables
   * @param  iLith = Array of indicators to plot lithology symbol or not
   * @param  st    = Lithology List Data Structure
   * @return iLith = Array of indicators to plot lithology symbol or not
   */

  public static int[] setLegend( int iLith[], lithologyListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int j=0; j<lith.lithology.lithologyStruct.TOTAL; j++)
        {
          if (st.stItem[i].iLithology[j] >
                lith.lithology.lithologySymbolsStruct._NONE)
            iLith[st.stItem[i].iLithology[j]] = 0;
        }
      }
    }

    return (iLith);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static lithologyListStruct bubbleSort(lithologyListStruct st)
  {
    boolean swappedOnPrevRun = true;
    lithologyStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the data structure
   * @param st = The Rock Column Litholog List Data Structure.
   */

  public static void print(lithologyListStruct st)
  {
    System.out.println("-------------------------------------------------");
    if (st != null)
    {
      for (int k=0; k<st.iCount; k++)
      {
        System.out.print(k + " "+ st.stItem[k].sKEY       + " " +
                                  st.stItem[k].depthStart + " " +
                                  st.stItem[k].depthEnd);

        System.out.print(" Abrev:   "+ st.stItem[k].sAbbreviation);
        System.out.print(" Streaks: "+ st.stItem[k].iStreaks);
        System.out.print(" Lithology: ");

        for (int j=0; j<10; j++)
        {
          System.out.print(", "+st.stItem[k].iLithology[j]);
        }
        System.out.println("");
      }
    }
    System.out.println("-------------------------------------------------");
  }
}