/*
 * @lithologySymbolsUtility.java Version 1.1 07/10/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.lithology;

import lith.lithology.lithologyColorsStruct;
import lith.lithology.lithologyGroupsStruct;
import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.lithologySymbolsStruct;
import lith.lithology.lithologyListStruct;

/** Class lithologySymbolsUtility
 *  <p> This Class will provide basic utilities for the Standard Lithology
 *      data structures.
 *
 *  @version 1.1 07/10/2009
 *  @author  John R. Victorine
 */

public class lithologySymbolsUtility
{
  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getColor()
   *  <p> This method will return the color number identifier from the
   *      character that is passed in.
   * @param  c    = Character
   * @param  st   = lithology colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( char c, lithologyColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.colors != null)
      {
        for (int i=1; i<st.colors.length; i++)
        {
          if (c == st.colors[i]) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getColor()
   *  <p> This method will return the color number identifier from the
   *      color name that is passed in.
   * @param  str  = Color Name
   * @param  st   = lithology colors data structure
   * @return iClr = the Color Identifier
   */

  public static int getColor( String str, lithologyColorsStruct st )
  {
    int iClr = -1;

    if (st != null)
    {
      if (st.sColors != null)
      {
        for (int i=1; i<st.sColors.length; i++)
        {
          if (str.equals(st.sColors[i])) iClr = i;
        }
      }
    }

    return (iClr);
  }

  /** Method getRGB()
   * <p> This method will return the RGB Array of colors
   * @param  icolor = The lithology color to return
   * @param  st     = lithology colors data structure
   * @return RGB    = Array of Red Green Blue Color numbers
   */

  public static int[] getRGB( int icolor, lithologyColorsStruct st  )
  {
    int RGB[] = { 255, 255, 255 };

    if (icolor != -1)
    {
      if (st != null)
      {
        if (st.iRGB != null)
        {
          if (icolor < st.iRGB.length)
          {
            for (int j=0; j<3; j++)
              RGB[j] = st.iRGB[icolor][j];
          }
        }
      }
    }

    return ( RGB );
  }

  /** Method getGroupID( int iColumn, String str, lithologyGroupsStruct st )
   *  <p> This method will return the ID number of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  str     = String to comparison string
   * @param  st      = lithology group data structure
   * @return iGroup  = Group id
   */

  public static int getGroupID( int iColumn,
                                String str,
                                lithologyGroupsStruct st )
  {
    int iGroup = -1;

    if (st != null)
    {
      for (int i=0; i<st.iGroups; i++)
      {
        if (st.sGroups[i][iColumn].equals(str))
          iGroup = i;
      }
    }

    return (iGroup);
  }

  /** Method getGroup( int iColumn, String str, lithologyGroupsStruct st )
   *  <p> This method will return the string of the group data
   * @param  iColumn = Column number to retrieve data from Array
   * @param  iGroup  = Group ID
   * @param  st      = lithology group data structure
   * @return sGroup  = Group String
   */

  public static String getGroup( int iColumn,
                                 int iGroup,
                                 lithologyGroupsStruct st )
  {
    String sGroup = "";

    if (st != null)
    {
      if (iGroup > -1)
      {
        sGroup = new String(st.sGroups[iGroup][iColumn]);
      }
    }

    return (sGroup);
  }

  /** Method getColor()
   * <p> This method will retrieve the color for a specific lithology
   * @param  iSymbol = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return iRGB    = array of RGB Colors to color image
   */

  public static int[] getColor( int iSymbol, lithologySymbolsListStruct st )
  {
    int iRGB[] = { 255, 255, 255 };

    if (st != null)
    {
      if (iSymbol > -1)
      {
        for (int j=0; j<3; j++)
          iRGB[j] = st.stItem[iSymbol].iRGB[j];
      }
    }

    return (iRGB);
  }

  /** Method getSymbol()
   * <p> This method will return the symbol
   * @param  iSymbol = Symbol identifier to retrieve symbol mask
   * @param  st      = Lithology Symbols List Data Structure
   * @return symbol  = array of strings to map image
   */

  public static String[][] getSymbol(int iSymbol, lithologySymbolsListStruct st)
  {
    String symbol[][] = null;

    if (st != null)
    {
      if (st.stItem[iSymbol].symbol != null)
      {
        symbol = new String[st.stItem[iSymbol].symbol.length][2];
        for (int i=0; i<st.stItem[iSymbol].symbol.length; i++)
        {
          for (int j=0; j<2; j++)
            symbol[i][j] = st.stItem[iSymbol].symbol[i][j];
        }
      }
    }

    return (symbol);
  }

  /** Method getLithology()
   * <p> This method will check the Lithology key words, phrases to see if any
   *     matches the default lithology key words or phrases and return the row
   *     id if it does.
   *
   * @param  str        = the string passed in.
   * @param  st         = Lithology Symbols List Data Structure
   * @return iLithology = the lithology identifier.
   */

  public static int getLithology( String str, lithologySymbolsListStruct st )
  {
    int     iLithology = -1;
    String  sWord      = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int j=0; j<st.stItem[i].iThesaurus; j++)
        {
          if (sWord.equals(st.stItem[i].sThesaurus[j].toLowerCase()))
            iLithology = st.stItem[i].id;
        }
      }
    }

    return (iLithology);
  }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method setHierarchy( lithologySymbolsListStruct st )
   * <p> This method will set the Hierararchy id from the group number and order
   * @param  st = lithology symbol data list Structure
   * @return st = lithology symbol data list Structure
   */

  public static lithologySymbolsListStruct setHierarchy(
                                            lithologySymbolsListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        st.stItem[i].iHierarchy = 100*st.stItem[i].iGroup + st.stItem[i].iOrder;
      }
    }

    return (st);
  }

  /** Method clearLegend( lithologySymbolsListStruct st )
   * <p> This method will set the legend to the default value
   * @param  st = lithology symbol data list Structure
   * @return st = lithology symbol data list Structure
   */

  public static lithologySymbolsListStruct clearLegend(
                                            lithologySymbolsListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
//        st.stItem[i].iExtend = st.stItem[i].iShow;
        st.stItem[i].iExtend = lith.lithology.lithologySymbolsStruct.OFF;
      }
    }

    return (st);
  }

  /** Method setLegend( lithologyListStruct stList,
   *                    lithologySymbolsListStruct st )
   * <p> This method will set the legend based on the lithologies selected
   *     along with the default legend
   * @param  stList = Lithology List Data Structure
   * @param  st     = Lithology Symbols List Structure
   * @return st     = lithology symbol data list Structure
   */

  public static lithologySymbolsListStruct setLegend(
      lithologyListStruct stList, lithologySymbolsListStruct st )
  {
    int iLith = 0;

    if ((stList != null) && (st != null))
    {
      for (int i=0; i<stList.iCount; i++)
      {
        for (int j=0; j<10; j++)
        {
          if (stList.stItem[i].iLithology[j] != 0)
          {
            iLith = 0;
            for (int k=0; k<st.iCount; k++)
            {
              if (stList.stItem[i].iLithology[j] == st.stItem[k].id)
                iLith = k;
            }
            st.stItem[iLith].iExtend = lith.lithology.lithologySymbolsStruct.ON;
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isLegend()
   * <p> This method will return true it should be included in legend and false
   *     it should not be included in legend
   * @param  iLegend = the id of the lithology
   * @param  st      = lithology symbols list data structure
   * @return bValue  = true or false
   */

  public static boolean isLegend( int iLegend, lithologySymbolsListStruct st )
  {
    boolean bValue = false;

    if (st != null)
    {
      if (st.stItem[iLegend].iExtend == lith.lithology.lithologySymbolsStruct.ON)
        bValue = true;
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( lithologySymbolsListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static lithologySymbolsListStruct copyList(
                                             lithologySymbolsListStruct stOld )
  {
    lithologySymbolsListStruct stNew = null;
    int i = 0;

    if (stOld != null)
    {
      stNew = new lithologySymbolsListStruct();

      // Lithology Colors

      stNew.stForeground = copyColor( stOld.stForeground ); // foreground colors
      stNew.stBackground = copyColor( stOld.stBackground ); // background colors

      // Lithology Groups

      stNew.stGroups = copyGroup( stOld.stGroups ); // Lithology Groups

      // Lithology Symbols

      stNew.iCount = stOld.iCount;
      stNew.stItem = new lithologySymbolsStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static lithologySymbolsStruct copy(lithologySymbolsStruct stOld)
  {
    lithologySymbolsStruct stNew = new lithologySymbolsStruct();
    int i=0;
    int j=0;

    stNew.iHierarchy = stOld.iHierarchy;               // Hierarchy
    stNew.iShow      = stOld.iShow;                    // Default Legend param
    stNew.iExtend    = stOld.iExtend;                  // Extended Legend param
    stNew.id         = stOld.id;                       // ID for Lithology Symbol
    stNew.iOrder     = stOld.iOrder;                   // Order within the group
    stNew.iGroup     = stOld.iGroup;                   // Group Number
    stNew.sGroup     = new String( stOld.sGroup );     // The group name
    stNew.sName      = new String( stOld.sName );      // Name
    stNew.sAbrev     = new String( stOld.sAbrev );     // Abbreviation
    stNew.sMnemonic  = new String( stOld.sMnemonic );  // Mnemonic

    // Background color

    for (i=0; i<3; i++)
      stNew.iRGB[i]   = stOld.iRGB[i];

    // Plot Symbol for lithology default is an empty or None symbol

    stNew.iRows      = stOld.iRows;
    stNew.iColumns   = stOld.iColumns;
    if (stNew.iRows > 0)
    {
      stNew.symbol     = new String[stNew.iRows][2];
      for (i=0; i<stNew.iRows; i++)
      {
        for (j=0; j<2; j++)
        {
          stNew.symbol[i][j] = new String( stOld.symbol[i][j] );
        }
      }
    }

    // Thesaurus for searching and parsing geologist field notes or comments

    stNew.iThesaurus   = stOld.iThesaurus;
    if (stNew.iThesaurus > 0)
    {
      stNew.sThesaurus   = new String[stNew.iThesaurus];
      for (i=0; i<stNew.iThesaurus; i++)
        stNew.sThesaurus[i] = stOld.sThesaurus[i];
    }

    return (stNew);
  }

  /** Method copyGroup( lithologyGroupsStruct stOld )
   * <p> This method will copy the group data structure
   * @param  stOld = The old group data structure
   * @return stNew = The new group data structure
   */

  public static lithologyGroupsStruct copyGroup( lithologyGroupsStruct stOld )
  {
    lithologyGroupsStruct stNew = new lithologyGroupsStruct();

    if (stOld != null)
    {
      stNew.iGroups = stOld.iGroups;
      stNew.sGroups = new String[stNew.iGroups][2];

      for (int i=0; i<stNew.iGroups; i++)
      {
        for (int j=0; j<2; j++)
        {
          stNew.sGroups[i][j] = new String( stOld.sGroups[i][j] );
        }
      }
    }

    return (stNew);
  }

  /** Method copyColor( lithologyGroupsStruct stOld )
   * <p> This method will copy the color data structure
   * @param  stOld = The old color data structure
   * @return stNew = The new color data structure
   */

  public static lithologyColorsStruct copyColor( lithologyColorsStruct stOld )
  {
    lithologyColorsStruct stNew = new lithologyColorsStruct();
    int len = 0;
    int i   = 0;
    int j   = 0;

    if (stOld != null)
    {
      stNew.iColors   = stOld.iColors;    // total number of records

      if (stOld.colors != null)
      {
        len = stOld.colors.length;
        stNew.colors = new char[len]; // Character Symbol

        for (i=0; i<len; i++)
          stNew.colors[i] = stOld.colors[i];
      }

      if (stOld.sColors != null)
      {
        len = stOld.sColors.length;
        stNew.sColors = new String[len]; // Color description

        for (i=0; i<len; i++)
          stNew.sColors[i] = new String( stOld.sColors[i] );
      }

      if (stOld.iRGB != null)
      {
        stNew.iRGB = new int[len][3]; // rgb values

        for (i=0; i<len; i++)
        {
          for (j=0; j<3; j++)
            stNew.iRGB[i][j] = stOld.iRGB[i][j];
        }
      }
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- SORT METHODS ------------------------------ *
   * ======================================================================== */

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = Lithology Symbols data list structure
   * @return st = sorted Lithology Symbols data list structure
   */

  public static lithologySymbolsListStruct bubbleSort(
                                             lithologySymbolsListStruct st )
  {
    boolean swappedOnPrevRun  = true;
    lithologySymbolsStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].iHierarchy > st.stItem[i+1].iHierarchy)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the lithology symbols structure
   * @param  st = Lithology Symbols data list structure
   */

  public static void print( lithologySymbolsListStruct st )
  {
    System.out.println(" ------------ Foreground Colors ------------- \n");
    print( st.stForeground );

    System.out.println("\n ------------ Background Colors ------------- \n");
    print( st.stBackground );

    System.out.println("\n ------------ Lithology Groups ------------- \n");
    print( st.stGroups );

    System.out.println("\n ------------ Lithologies ------------- \n");
    for (int i=0; i<st.iCount; i++)
    {
      print( st.stItem[i] );
    }
  }

  /** Method print()
   * <p> This method will print the color data structure
   * @param  st = Lithology colors data structure
   */

  public static void print( lithologyColorsStruct st )
  {
    int i=0;

    if (st != null)
    {
      if (st.iColors > 0)
      {
        if (st.colors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.colors[i]  + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
        else if (st.sColors != null)
        {
          for (i=0; i<st.iColors; i++)
          {
            System.out.println( st.sColors[i] + " " +
                                st.iRGB[i][0] + " " +
                                st.iRGB[i][1] + " " +
                                st.iRGB[i][2]);
          }
        }
      }
    }
  }

  /** Method print()
   * <p> This method will print the group data structure
   * @param  st = Lithology Groups data structure
   */

  public static void print( lithologyGroupsStruct st )
  {
    if (st != null)
    {
      if (st.iGroups > 0)
      {
        for (int i=0; i<st.iGroups; i++)
        {
          System.out.println( i + " " + st.sGroups[i][0] +
                              " <-> " + st.sGroups[i][1] );
        }
      }
    }
  }

  /** Method print()
   * <p> This method will print the symbols data structure
   * @param  st = Lithology symbols data structure
   */

  public static void print( lithologySymbolsStruct st )
  {
    int i = 0;

    if (st != null)
    {
      System.out.println(st.iHierarchy + " " + // Hierarchy within the list
                         st.id     + " " + // ID for Lithology Symbol
                         st.iOrder + " " + // Order within the group
                         st.iGroup + " " + // Group Number
                         st.sGroup + " " + // The group the lithology belongs
                         st.sName  + " " + // Name or Description of lithology
                         st.sAbrev + " " + // Abbreviation of lithology
                         st.sMnemonic); // Mnemonic of lithology

      System.out.println("Color " +
                         st.iRGB[0] + " " +
                         st.iRGB[1] + " " +
                         st.iRGB[2]);

      if (st.symbol != null)
      {
        for (i=0; i<st.iRows; i++)
        {
          System.out.println(st.symbol[i][0] + " " + st.symbol[i][1]);
        }
      }

      if (st.iThesaurus > 0)
      {
        for (i=0; i<st.iThesaurus; i++)
        {
          System.out.print(st.sThesaurus[i] + " ");
        }
        System.out.println("\n");
      }
    }
  }
}