/*
 * @lithologyEntryFrame.java Version 1.1 04/18/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import iqstrat.iqstratStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;

import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;
import lith.gui.lithSedimentPanel;
import lith.gui.lithMetamorphPanel;
import lith.gui.lithIgneousPanel;

/** Class lithologyEntryFrame
 *  <p> This Frame will allow the user to pick the Lithology & Texture.
 *
 *  @version 1.1 04/18/2011
 *  @author  John R. Victorine
 */

public class lithologyEntryFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable    pNotifier = null; // Calling Observable Object
  private iqstratStruct stStruct  = null; // Global IQSTRAT Data Structure

  // Global Frame Variables

  private Observable notifier = null; // This Observable Object

  private String     sRock    = "";   // Primary Lithology Description
  private String     second   = "";   // Secondary Lithologies
  private int        iTexture = -1;   // Texture Energy Level
  private String     sTexture = "";   // Texture Level Description

  private lithologyStruct    stLithology = null; // Lithology Data Structure

  private lithSedimentPanel  pSediment   = null; // Sedimentary Rock Lithology Panel
  private lithMetamorphPanel pMetamorph  = null; // Metamorphic Rock Lithology Panel
  private lithIgneousPanel   pIgneous    = null; // Igneous Rock Lithology Panel

  // Global Frame Widgets

  private static final int _SEDIMENTARY = 0;
  private static final int _METAMORPHIC = 1;
  private static final int _IGNEOUS     = 2;

  private static final String PANEL[]   = {
    "Sedimentary Rock Lithology",
    "Metamorphic Rock Lithology",
    "Igneous Rock Lithology" };

  private int iPanel = _SEDIMENTARY;

  // -- Diplay Descriptive Data Panel

  private JButton      btnAdd = new JButton();    // Add data to List

  private JPanel       pnlData           = new JPanel();
  private CardLayout   cardLayout        = new CardLayout();
  private TitledBorder titledBorderEntry = null;

  // -- Rock Data Entry Buttons

  private JRadioButton rbSedimentary     = new JRadioButton();
  private JRadioButton rbMetamorphic     = new JRadioButton();
  private JRadioButton rbIgneous         = new JRadioButton();

  // -- Final Lithology/Texture Description

  private JTextArea  txt  = new JTextArea();  // Rock Description

  // -- Menu Items

  private JMenuItem mExit = null; // Exit Profile Plot

  // Dialog Position & Size

  private int iXStart = 0;
  private int iYStart = 0;
  private int iWidth  = 0;
  private int iHeight = 0;

  /** Constructor lithologyEntryFrame()
   *  <p> This is the Constructor for this class.
   *  @param stSymbols = Lithology Symbols List Data Structure
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public lithologyEntryFrame( Observable pNotifier, iqstratStruct stStruct )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;

      jbInit();
      addWindowListener( new lithologyEntryFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop     = new JPanel();
    JPanel pnlCenter  = new JPanel();
    JPanel pnlSouth   = new JPanel();
    JPanel pnlAdd     = new JPanel();

    JScrollPane scrollTxt = new JScrollPane();

    ButtonGroup group = new ButtonGroup();

    titledBorderEntry = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "Sedimentary Rock Lithology");
    titledBorderEntry.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Menu

    JMenuBar    menuBar  = new JMenuBar();
    JMenu       menuFile = new JMenu("Close Dialog");

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Create an Observable

    notifier = new lithologyEntryFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Build Frame

    this.setTitle("Lithology/Texture");
    this.getContentPane().setLayout( new BorderLayout() );

    // Top Panel

    pnlTop.setLayout( new GridLayout() );
    pnlTop.setBackground(Color.white);

    // .. Sedimentary Rocks

    rbSedimentary.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbSedimentary.setBackground(Color.white);
    rbSedimentary.setHorizontalAlignment(SwingConstants.CENTER);
    rbSedimentary.setSelected(true);
    rbSedimentary.setText( "Sedimentary Rocks" );
    rbSedimentary.addActionListener(this);

    // .. Metamorphic Rocks

    rbMetamorphic.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbMetamorphic.setBackground(Color.white);
    rbMetamorphic.setHorizontalAlignment(SwingConstants.CENTER);
    rbMetamorphic.setText( "Metamorphic Rocks" );
    rbMetamorphic.addActionListener(this);

    // .. Igneous Rocks

    rbIgneous.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbIgneous.setBackground(Color.white);
    rbIgneous.setHorizontalAlignment(SwingConstants.CENTER);
    rbIgneous.setText( "Igneous Rocks" );
    rbIgneous.addActionListener(this);

    // Center Panel

    pnlCenter.setLayout( new BorderLayout() );
    pnlCenter.setBackground(Color.white);

    pnlData.setLayout( cardLayout );
    pnlData.setBorder(titledBorderEntry);
    pnlData.setBackground(Color.white);

    pSediment  = new lithSedimentPanel( notifier );
    pMetamorph = new lithMetamorphPanel( notifier );
    pIgneous   = new lithIgneousPanel( notifier );

    // -- Final Lithology/Texture Description

    pnlSouth.setLayout(new BorderLayout());

    txt.setText("");
    txt.setRows(3);
    txt.setFont(new java.awt.Font("Dialog", 0, 14));
    txt.setLineWrap(true);
    txt.setWrapStyleWord(true);
//    txt.addFocusListener(new rockObsEntryFrameFocusAdapter(this));

    // .. Add Buttons to Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    // Attach Widgets to Frame

    this.getContentPane().add( pnlTop,    BorderLayout.NORTH );
    pnlTop.add( rbSedimentary,            null );
    pnlTop.add( rbMetamorphic,            null );
    pnlTop.add( rbIgneous,                null );

    group.add( rbSedimentary );
    group.add( rbMetamorphic );
    group.add( rbIgneous );

    this.getContentPane().add( pnlCenter, BorderLayout.CENTER );
    pnlCenter.add( pnlData,               BorderLayout.CENTER );
    pnlData.add( pSediment,               "SEDIMENTARY" );
    pnlData.add( pMetamorph,              "METAMORPHIC" );
    pnlData.add( pIgneous,                "IGNEOUS" );

    this.getContentPane().add( pnlSouth,  BorderLayout.SOUTH );
    pnlSouth.add( scrollTxt,              BorderLayout.CENTER );
    scrollTxt.getViewport().add( txt,     null );
    pnlSouth.add( pnlAdd,                 BorderLayout.EAST );
    pnlAdd.add( btnAdd,                   null );

    // Display the Frame

    this.setSize(new Dimension(770, 750));

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    int iXPos = (d.width  - this.getSize().width) / 2;
    if (iWidth > 0) iXPos = iXStart + iWidth;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /** METHOD lithologyEntryFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class lithologyEntryFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   *  <p> This method will retrieve the Lithology data Structure
   *  @param stLithology = the lithology data structure
   */

  public lithologyStruct getData()
  {
	lithologyListStruct     st     = null;
    iqstratRemarkStruct     stR    = new iqstratRemarkStruct();
    iqstratRemarkListStruct stTemp = null;

    if (stLithology != null)
      stLithology.delete();
    stLithology = null;

    stR.depthStart =  0.0;
    stR.depthEnd   = 20.0;
    stR.sText      = new String( txt.getText() );

    stTemp = iqstrat.iqstratRemarkUtility.add( stR, stTemp );

    st     = parse.parseMineralsUtility.parse(
                    stTemp,
                    stStruct.stSymbols,
                    stStruct.stParseTexture,
                    stStruct.stMinerals );

    if (st != null)
    {
	  if (st.iCount > 0)
	  {
	    stLithology = lith.lithology.lithologyUtility.copy( st.stItem[0] );
	  }
	}

	return (stLithology);
  }


  /** Method getText()
   *  <p> This method will retrieve the Lithology Description Field
   *  @param the lithology descriptions text
   */

  public String getText() { return (txt.getText()); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setPanel()
   * <p> This method will set the borders of the List and Data Entry Panel
   *  @param iPanel = The type of panel to display
   */

  private void setPanel(int iPanel)
  {
    this.iPanel = iPanel;

	switch (iPanel)
	{
	  case _SEDIMENTARY:
        cardLayout.show(pnlData, "SEDIMENTARY");
        break;
      case _METAMORPHIC:
        cardLayout.show(pnlData, "METAMORPHIC");
        break;
      case _IGNEOUS:
        cardLayout.show(pnlData, "IGNEOUS");
        break;
    }

    titledBorderEntry.setTitle( PANEL[iPanel] );
    pnlData.updateUI();
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add the rock data
   */

  public void add()
  {
    if (pNotifier != null)
      pNotifier.notifyObservers(new String("ADD or MODIFY Lithology Data"));
  }

  /** Method close()
   * <p> This method will cancel the Dialog
   */

  public void close()
  {
  // Input Variables

    pNotifier = null; // Observable Object
    stStruct  = null; // Global IQSTRAT Data Structure

    // Global Frame Variables

    notifier = null; // This Observable Object
    sRock    = null; // Primary Lithology Description
    second   = null; // Secondary Lithologies
    sTexture = null; // Texture Level Description

    if (stLithology != null)
      stLithology.delete();
    stLithology = null;

    if (pSediment != null)
      pSediment.close();
    pSediment  = null; // Sedimentary Rock Lithology Panel

    if (pMetamorph != null)
      pMetamorph.close();
    pMetamorph = null; // Metamorphic Rock Lithology Panel

    if (pIgneous != null)
      pIgneous.close();
    pIgneous   = null; // Igneous Rock Lithology Panel

    // Global Frame Widgets

    // -- Diplay Descriptive Data Panel

    btnAdd = null;    // Add data to List

    pnlData           = null;
    cardLayout        = null;
    titledBorderEntry = null;

    // -- Rock Data Entry Buttons

    rbSedimentary     = null;
    rbMetamorphic     = null;
    rbIgneous         = null;

    // -- Final Lithology/Texture Description

    txt      = null;  // Rock Description

    // -- Menu Items

    mExit    = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == rbSedimentary) { setPanel(_SEDIMENTARY); }
    if (event.getSource() == rbMetamorphic) { setPanel(_METAMORPHIC); }
    if (event.getSource() == rbIgneous)     { setPanel(_IGNEOUS); }

    if (event.getSource() == btnAdd) { add();}

    if (event.getSource() == mExit)  { close(); }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("Sedimentary Lithology Selected"))
    {
      sRock    = new String( pSediment.getLithology() );
      second   = new String( pSediment.getSecondary() );
      iTexture = pSediment.getTexture();
      sTexture = new String( pSediment.getTextureDesc() );

      if (second.length() == 0)
      {
        txt.setText( sRock + ": " +
                     sTexture + " ( Level: " + iTexture + " )");
	  }
	  else
	  {
        txt.setText( sRock + ": " +
                     sTexture + " ( Level: " + iTexture + " ), " +
                     second);
	  }
	}

    if (sArg.equals("Metamorphic Lithology Selected"))
    {
      sRock    = new String( pMetamorph.getLithology() );
      iTexture = pMetamorph.getTexture();
      sTexture = new String( pMetamorph.getTextureDesc() );

      txt.setText( sRock + ": " +
                   sTexture + " ( Level: " + iTexture + " )");
	}

    if (sArg.equals("Igneous Lithology Selected"))
    {
      sRock    = new String( pIgneous.getLithology() );
      iTexture = pIgneous.getTexture();
      sTexture = new String( pIgneous.getTextureDesc() );

      txt.setText( sRock + ": " +
                   sTexture + " ( Level: " + iTexture + " ), " );
	}
  }
}

/** CLASS lithologyEntryFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class lithologyEntryFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 04/18/2011
 *  @author  John Victorine
 */
