/*
 * @lithSedimentsTexturePanel.java Version 1.1 04/19/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class lithSedimentsTexturePanel
 *  <p> This Class will display the sedimentary structures selected
 *
 *  @version 1.1 04/19/2011
 *  @author  John R. Victorine
 */

public class lithSedimentsTexturePanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable notifier = null;     // Panel Observable

  // Global Variables

  public static final String DEFAULT[] = lith.lithConstants.DEFAULT;
  public static final int DEFAULT_COLOR[][] = lith.lithConstants.DEFAULT_COLOR;

  public static final String SIO2[] = lith.lithConstants.SIO2;
  public static final int SIO2_COLOR[][] = lith.lithConstants.SIO2_COLOR;

  public static final String CO3[] = lith.lithConstants.CO3;
  public static final int CO3_COLOR[][] = lith.lithConstants.CO3_COLOR;

  public static final String PYRO[] = lith.lithConstants.PYRO;
  public static final int PYRO_COLOR[][] = lith.lithConstants.PYRO_COLOR;

  public static final int PRIMARY_SIZE[][] = lith.lithConstants.PRIMARY_SIZE;

  private static final int _SAND = 0;
  private static final int _CALC = 1;
  private static final int _DOLO = 2;
  private static final int _PYRO = 3;

  private int    iPanel   = _SAND;   // Sediments Panel
  private int    iRock    = -1;      // Primary Lithology
  private int    iTexture = -1;      // Texture Energy Level
  private String sTexture = "";      // Texture Level Description

  // Global Widgets

  // -- Display Descriptive Data Panel

  private JPanel       pnlData           = new JPanel();
  private CardLayout   cardLayout        = new CardLayout();
  private TitledBorder titledBorderMain  = null;

  // -- Sediment Type

  private JRadioButton rbSand = new JRadioButton();
  private JRadioButton rbCalc = new JRadioButton();
  private JRadioButton rbDolo = new JRadioButton();
  private JRadioButton rbPyro = new JRadioButton();

  // -- Lithology & Texture Widgets

  private JRadioButton rb[]   = null;  // Sediment Size
  private JRadioButton rbP[]  = null;  // Primary Lithologies

  /** Constructor lithSedimentsTexturePanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public lithSedimentsTexturePanel( Observable notifier )
  {
    try
    {
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	int i;
	int    iValue      = 0;
	String sValue      = "";
	int    COLOR[][]   = null;
	String SIZE[]      = null;

    JPanel pnlTop      = new JPanel();
    JPanel pnlSAND     = new JPanel();
    JPanel pnlCALC     = new JPanel();
    JPanel pnlDOLO     = new JPanel();
    JPanel pnlPYRO     = new JPanel();

    JPanel pnlCenter   = new JPanel();
    JPanel pnlWest     = new JPanel();

    ButtonGroup group  = new ButtonGroup();
    ButtonGroup groupS = new ButtonGroup();
    ButtonGroup groupP = new ButtonGroup();

    titledBorderMain = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151)),
        "Sediments");
    titledBorderMain.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSize = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "");
    titledBorderSize.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    this.setBackground(Color.white);
    this.setBorder(titledBorderMain);
    this.setLayout(new BorderLayout());
    this.setPreferredSize(new Dimension(300, 10));

    // Top Panel

    pnlTop.setLayout(new GridLayout(2,2));
    pnlTop.setBackground(new Color( 255, 255, 255 ));
    pnlTop.setBorder(titledBorderSize);

    rbSand.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbSand.setHorizontalAlignment(SwingConstants.CENTER);
    rbSand.setBackground(new Color( 255, 255, 255 ));
    rbSand.setText( "Rock Fragments " );
    rbSand.setSelected(true);
    rbSand.addActionListener(this);

    rbCalc.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbCalc.setHorizontalAlignment(SwingConstants.CENTER);
    rbCalc.setBackground(new Color( 255, 255, 255 ));
    rbCalc.setText( "> 50% Carbonate" );
    rbCalc.addActionListener(this);

    rbDolo.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbDolo.setHorizontalAlignment(SwingConstants.CENTER);
    rbDolo.setBackground(new Color( 255, 255, 255 ));
    rbDolo.setText( "> 50% Dolomite " );
    rbDolo.addActionListener(this);

    rbPyro.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbPyro.setHorizontalAlignment(SwingConstants.CENTER);
    rbPyro.setBackground(new Color( 255, 255, 255 ));
    rbPyro.setText( "Pyroclastic    " );
    rbPyro.addActionListener(this);

    // West Panel

    pnlWest.setLayout(new GridLayout(14,1));
    pnlWest.setBackground(new Color( 255, 255, 255 ));
//    pnlWest.setPreferredSize(new Dimension(175, 10));

    // -- Grain Size

    rb = new JRadioButton[14];

    COLOR = DEFAULT_COLOR;
    SIZE  = DEFAULT;

    for (i=13; i>-1; i--)
    {
	  iValue = i+1;
      sValue = new String("(" + iValue + ") ");
      if (iValue < 10)
        sValue = new String("( " + iValue + ") ");

	  rb[i] = new JRadioButton();
      rb[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      rb[i].setHorizontalAlignment(SwingConstants.LEFT);
      rb[i].setBackground(new Color(COLOR[i][0],COLOR[i][1],COLOR[i][2]));
      rb[i].setText( SIZE[i] );
      if (SIZE[i].length() == 0) { rb[i].setEnabled(false); }
      rb[i].addActionListener(this);

      pnlWest.add( rb[i], null);
      group.add( rb[i] );
	}

    // -- Build Main Lithologies

    rbP = new JRadioButton[lith.lithConstants._PRIMARY];

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = new JRadioButton();
      rbP[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      rbP[i].setHorizontalAlignment(SwingConstants.LEFT);
      rbP[i].setBackground(new Color( 255, 255, 255 ));
      rbP[i].setText( lith.lithConstants.PRIMARY[i][0] );
      rbP[i].addActionListener(this);

      groupP.add( rbP[i] );
    }

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());

    pnlData.setLayout(cardLayout);

    pnlSAND = buildSAND();
    pnlCALC = buildCALC();
    pnlDOLO = buildDOLO();
    pnlPYRO = buildPYRO();

    // Attach Widgets to Panel

    this.add(pnlTop,         BorderLayout.NORTH);
    pnlTop.add(rbSand,       null);
    pnlTop.add(rbCalc,       null);
    pnlTop.add(rbDolo,       null);
    pnlTop.add(rbPyro,       null);

    groupS.add(rbSand);
    groupS.add(rbCalc);
    groupS.add(rbDolo);
    groupS.add(rbPyro);

    this.add(pnlWest,        BorderLayout.WEST);

    this.add(pnlCenter,      BorderLayout.CENTER);
    pnlCenter.add( pnlData,  BorderLayout.CENTER );
    pnlData.add( pnlSAND,    "SAND" );
    pnlData.add( pnlCALC,    "CALC" );
    pnlData.add( pnlDOLO,    "DOLO" );
    pnlData.add( pnlPYRO,    "PYRO" );

    setPanel( _SAND );

  }

  /** Method buildSAND()
   * <p> This method will build the Quartz, rock fragments, feldspar
   *     and other minerals panel
   */

  private JPanel buildSAND()
  {
    JPanel panel     = new JPanel();
    JPanel pnlRud    = new JPanel();
    JPanel pnlAre    = new JPanel();
    JPanel pnlAreL[] = null;
    JPanel pnlArg    = new JPanel();
    JPanel pnlArgL[] = null;

    TitledBorder titledBorderRud = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Rudaceous");
    titledBorderRud.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAre = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Arenaceous");
    titledBorderAre.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderArg = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Argillaceous");
    titledBorderArg.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    // Center Panel

    panel.setLayout(new BorderLayout());

    pnlRud.setBorder(titledBorderRud);
    pnlRud.setPreferredSize(new Dimension(10, 160));
    pnlRud.setBackground(Color.white);
    pnlRud.setLayout(new GridLayout(3,1));

    pnlRud.add(rbP[lith.lithConstants._GRAVEL],         null);
    pnlRud.add(rbP[lith.lithConstants._CONGLOMERATE],   null);
    pnlRud.add(rbP[lith.lithConstants._BRECCIA],        null);

    pnlAre.setBorder(titledBorderAre);
    pnlAre.setBackground(Color.white);
    pnlAre.setLayout(new GridLayout(4,1));

    pnlAreL    = new JPanel[2];

    pnlAreL[1] = new JPanel();
    pnlAreL[1].setLayout(new GridLayout());
    pnlAreL[1].add(rbP[lith.lithConstants._LOAM],       null);
    pnlAreL[1].add(rbP[lith.lithConstants._SOIL],       null);

    pnlAre.add(pnlAreL[1],                              null);

    pnlAreL[0] = new JPanel();
    pnlAreL[0].setLayout(new GridLayout());
    pnlAreL[0].add(rbP[lith.lithConstants._SAND],       null);
    pnlAreL[0].add(rbP[lith.lithConstants._SANDSTONE],  null);

    pnlAre.add(pnlAreL[0],                              null);

    pnlAre.add(rbP[lith.lithConstants._ARKOSE],         null);

    pnlAre.add(rbP[lith.lithConstants._LOESS],          null);

    pnlArg.setBorder(titledBorderArg);
    pnlArg.setPreferredSize(new Dimension(10, 150));
    pnlArg.setBackground(Color.white);
    pnlArg.setLayout(new GridLayout(5,1));

    pnlArgL    = new JPanel[4];

    pnlArgL[0] = new JPanel();
    pnlArgL[0].setLayout(new GridLayout());
    pnlArgL[0].add(rbP[lith.lithConstants._SILT],       null);
    pnlArgL[0].add(rbP[lith.lithConstants._SILTSTONE],  null);

    pnlArg.add(pnlArgL[0],                              null);

    pnlArgL[1] = new JPanel();
    pnlArgL[1].setLayout(new GridLayout());
    pnlArgL[1].add(rbP[lith.lithConstants._CLAY],       null);
    pnlArgL[1].add(rbP[lith.lithConstants._CLAYSTONE],  null);

    pnlArg.add(pnlArgL[1],                              null);

    pnlArgL[2] = new JPanel();
    pnlArgL[2].setLayout(new GridLayout());
    pnlArgL[2].add(rbP[lith.lithConstants._BENTONITE],  null);
    pnlArgL[2].add(rbP[lith.lithConstants._GLAUCONITE], null);

    pnlArg.add(pnlArgL[2],                              null);

    pnlArgL[3] = new JPanel();
    pnlArgL[3].setLayout(new GridLayout());
    pnlArgL[3].add(rbP[lith.lithConstants._MARL],       null);
    pnlArgL[3].add(rbP[lith.lithConstants._MARLSTONE],  null);

    pnlArg.add(pnlArgL[3],                              null);

    pnlArg.add(rbP[lith.lithConstants._SHALE],          null);

    // Attach Widgets

    panel.add(pnlRud,      BorderLayout.NORTH);
    panel.add(pnlAre,      BorderLayout.CENTER);
    panel.add(pnlArg,      BorderLayout.SOUTH);

    return (panel);
  }

  /** Method buildCALC()
   * <p> This method will build the   panel
   */

  private JPanel buildCALC()
  {
    JPanel panel     = new JPanel();
    JPanel pnlRud    = new JPanel();
    JPanel pnlAre    = new JPanel();
    JPanel pnlArg    = new JPanel();

    TitledBorder titledBorderRud = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Rudaceous");
    titledBorderRud.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAre = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Arenaceous");
    titledBorderAre.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderArg = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Argillaceous");
    titledBorderArg.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    // Center Panel

    panel.setLayout(new BorderLayout());

    pnlRud.setBorder(titledBorderRud);
    pnlRud.setPreferredSize(new Dimension(10, 160));
    pnlRud.setBackground(Color.white);
    pnlRud.setLayout(new GridLayout());

    pnlRud.add(rbP[lith.lithConstants._CALCIRUDITE],        null);

    pnlAre.setBorder(titledBorderAre);
    pnlAre.setBackground(Color.white);
    pnlAre.setLayout(new GridLayout());

    pnlAre.add(rbP[lith.lithConstants._CALCARENITE],        null);

    pnlArg.setBorder(titledBorderArg);
    pnlArg.setPreferredSize(new Dimension(10, 150));
    pnlArg.setBackground(Color.white);
    pnlArg.setLayout(new GridLayout());

    pnlArg.add(rbP[lith.lithConstants._CALCILUTITE],        null);

    // Attach Widgets

    panel.add(pnlRud,      BorderLayout.NORTH);
    panel.add(pnlAre,      BorderLayout.CENTER);
    panel.add(pnlArg,      BorderLayout.SOUTH);

    return (panel);
  }

  /** Method buildDOLO()
   * <p> This method will build the   panel
   */

  private JPanel buildDOLO()
  {
    JPanel panel     = new JPanel();
    JPanel pnlRud    = new JPanel();
    JPanel pnlAre    = new JPanel();
    JPanel pnlArg    = new JPanel();

    TitledBorder titledBorderRud = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Rudaceous");
    titledBorderRud.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAre = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Arenaceous");
    titledBorderAre.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderArg = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Argillaceous");
    titledBorderArg.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    // Center Panel

    panel.setLayout(new BorderLayout());

    pnlRud.setBorder(titledBorderRud);
    pnlRud.setPreferredSize(new Dimension(10, 160));
    pnlRud.setBackground(Color.white);
    pnlRud.setLayout(new GridLayout());

    pnlRud.add(rbP[lith.lithConstants._DOLORUDITE],        null);

    pnlAre.setBorder(titledBorderAre);
    pnlAre.setBackground(Color.white);
    pnlAre.setLayout(new GridLayout());

    pnlAre.add(rbP[lith.lithConstants._DOLARENITE],       null);

    pnlArg.setBorder(titledBorderArg);
    pnlArg.setPreferredSize(new Dimension(10, 150));
    pnlArg.setBackground(Color.white);
    pnlArg.setLayout(new GridLayout());

    pnlArg.add(rbP[lith.lithConstants._DOLOLUTITE],        null);

    // Attach Widgets

    panel.add(pnlRud,      BorderLayout.NORTH);
    panel.add(pnlAre,      BorderLayout.CENTER);
    panel.add(pnlArg,      BorderLayout.SOUTH);

    return (panel);
  }

  /** Method buildPYRO()
   * <p> This method will build the   panel
   */

  private JPanel buildPYRO()
  {
    JPanel panel     = new JPanel();
    JPanel pnlRud    = new JPanel();
    JPanel pnlAre    = new JPanel();

    TitledBorder titledBorderRud = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Rudaceous");
    titledBorderRud.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAre = BorderFactory.createTitledBorder(
		       BorderFactory.createEmptyBorder(),
		       "Arenaceous");
    titledBorderAre.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    // Center Panel

    panel.setLayout(new BorderLayout());

    pnlRud.setBorder(titledBorderRud);
    pnlRud.setPreferredSize(new Dimension(10, 160));
    pnlRud.setBackground(Color.white);
    pnlRud.setLayout(new GridLayout(2,1));

    pnlRud.add(rbP[lith.lithConstants._AGGLOMERATE], null);
    pnlRud.add(rbP[lith.lithConstants._VOL_BRECCIA], null);

    pnlAre.setBorder(titledBorderAre);
    pnlAre.setBackground(Color.white);
    pnlAre.setLayout(new GridLayout());

    pnlAre.add(rbP[lith.lithConstants._TUFF],        null);

    // Attach Widgets

    panel.add(pnlRud,      BorderLayout.NORTH);
    panel.add(pnlAre,      BorderLayout.CENTER);

    return (panel);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLithology()
   *  <p> This method will retrieve the Primary Lithology selected
   *  @return iRock = The Primary Lithology
   */

  public int getLithology() { return (iRock); }

  /** Method getTexture()
   *  <p> This method will retrieve the Texture Level selected
   *  @return iTexture = The Texture Level Selected
   */

  public int getTexture() { return (iTexture); }

  /** Method getTextureDesc()
   *  <p> This method will retrieve the Texture Level Description selected
   *  @return sTexture = The Texture Level Description Selected
   */

  public String getTextureDesc() { return (sTexture); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setPanel()
   *  <p> This method will set the panel and enable & disable buttons
   *  @param iPanel = The panel that is set
   */

  private void setPanel(int iPanel)
  {
	int    COLOR[][]   = null;
	String SIZE[]      = null;
	String sTitle      = "";

	this. iPanel = iPanel;

    switch (iPanel)
    {
      case _SAND:
        cardLayout.show(pnlData, "SAND");
        sTitle = new String("Sediments: Quartz, Feldspar, Rock Fragments");

        COLOR = SIO2_COLOR;
        SIZE  = SIO2;
        break;
      case _CALC:
        cardLayout.show(pnlData, "CALC");
        sTitle = new String("Sediments: > 50 % Carbonate-Limestone");

        COLOR = CO3_COLOR;
        SIZE  = CO3;
        break;
      case _DOLO:
        cardLayout.show(pnlData, "DOLO");
        sTitle = new String("Sediments: > 50 % Carbonate-Dolomite");

        COLOR = CO3_COLOR;
        SIZE  = CO3;
        break;
      case _PYRO:
        cardLayout.show(pnlData, "PYRO");
        sTitle = new String("Sediments: > 50 % Volcanic Material");

        COLOR = PYRO_COLOR;
        SIZE  = PYRO;
        break;
      default:
        sTitle = new String("Sediments:");

        COLOR = DEFAULT_COLOR;
        SIZE  = DEFAULT;
        break;

	}

    for (int i=13; i>-1; i--)
    {
      rb[i].setBackground(new Color( COLOR[i][0],
                                     COLOR[i][1],
                                     COLOR[i][2]));
      rb[i].setText( SIZE[i] );
      if (SIZE[i].length() == 0)
      {
		if (i+1 < 14)
          rb[i].setBackground(new Color( COLOR[i+1][0],
                                         COLOR[i+1][1],
                                         COLOR[i+1][2]));
		rb[i].setEnabled(false);
      }
    }

//    titledBorderEntry.setTitle( PANEL[iPanel] );
//    pnlData.updateUI();
    titledBorderMain.setTitle( sTitle );
    this.updateUI();
  }

  /** Method setTextureBTNS()
   *  <p> This method will set the Texture values depending on the seleced
   *      lithology.
   *  @param iRock = is the selected Rock button
   */

  private void setTextureBTNS( int iRock )
  {
	int    COLOR[][]   = null;
	String SIZE[]      = null;

	this.iRock = iRock;

    switch (iPanel)
    {
      case _SAND:
        COLOR = SIO2_COLOR;
        SIZE  = SIO2;
        break;
      case _CALC:
        COLOR = CO3_COLOR;
        SIZE  = CO3;
        break;
      case _DOLO:
        COLOR = CO3_COLOR;
        SIZE  = CO3;
        break;
      case _PYRO:
        COLOR = PYRO_COLOR;
        SIZE  = PYRO;
        break;
      default:
        COLOR = DEFAULT_COLOR;
        SIZE  = DEFAULT;
        break;
	}

    if (iRock > -1)
    {
      for (int i=13; i>-1; i--)
      {
        rb[i].setBackground(new Color( 255, 255, 255 ));
      }

      for (int i=13; i>-1; i--)
      {
	    if ((PRIMARY_SIZE[iRock][i] == lith.lithConstants._ON) ||
	        (PRIMARY_SIZE[iRock][i] == lith.lithConstants._SET))
	    {
          rb[i].setBackground(new Color( COLOR[i][0],
                                         COLOR[i][1],
                                         COLOR[i][2]));
          rb[i].setText( SIZE[i] );
          if (SIZE[i].length() == 0)
          {
  		    if (i+1 < 14)
              rb[i].setBackground(new Color( COLOR[i+1][0],
                                             COLOR[i+1][1],
                                             COLOR[i+1][2]));
	  	    rb[i].setEnabled(false);
          }
	    }
      }

      for (int i=0; i<14; i++)
      {
	    rb[i].setEnabled( false );
	    if ((PRIMARY_SIZE[iRock][i] == lith.lithConstants._ON) ||
	        (PRIMARY_SIZE[iRock][i] == lith.lithConstants._SET))
	    {
          if (SIZE[i].length() > 0)
    	    rb[i].setEnabled( true );
  	      if (PRIMARY_SIZE[iRock][i] == lith.lithConstants._SET)
  	      {
  	        rb[i].setSelected( true );
  	        setTexture( i );
		  }
		}
	  }
    }
  }

  /** Method setTexture()
   *  <p> This method will set the Texture identifier
   *  @param iTexture = is the selected Texture button
   */

  private void setTexture( int iTexture )
  {
	String SIZE[] = null;

    switch (iPanel)
    {
      case _SAND:
        SIZE  = SIO2;
        break;
      case _CALC:
        SIZE  = CO3;
        break;
      case _DOLO:
        SIZE  = CO3;
        break;
      case _PYRO:
        SIZE  = PYRO;
        break;
      default:
        SIZE  = DEFAULT;
        break;
	}

    switch (iRock)
    {
	  case lith.lithConstants._SILT:
	  case lith.lithConstants._SILTSTONE:
	  case lith.lithConstants._CLAY:
	  case lith.lithConstants._CLAYSTONE:
	  case lith.lithConstants._BENTONITE:
	  case lith.lithConstants._GLAUCONITE:
	  case lith.lithConstants._MARL:
	  case lith.lithConstants._MARLSTONE:
	  case lith.lithConstants._SHALE:
        this.sTexture = new String( "" );
	    break;
	  default:
        this.sTexture = new String( SIZE[iTexture] );
	    break;
	}

    this.iTexture = iTexture+1;
  }

  /* ===================================================================== *
   * ----------------------------- CLOSE --------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
	int i=0;

    // Input Variables

    notifier = null; // Panel Observable

    // Global Variables

    sTexture = null; // Texture Level Description

    // Global Widgets

    // -- Display Descriptive Data Panel

    pnlData          = null;
    cardLayout       = null;
    titledBorderMain = null;

    // -- Sediment Type

    rbSand = null;
    rbCalc = null;
    rbDolo = null;
    rbPyro = null;

    // -- Lithology & Texture Widgets

    for (i=0; i<14; i++)
    {
	  rb[i] = null;
	}

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = null;
	}

    rb   = null;  // Sediment Size
    rbP  = null;  // Primary Lithologies
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i=0;

    if (event.getSource() == rbSand) { setPanel(_SAND); }
    if (event.getSource() == rbCalc) { setPanel(_CALC); }
    if (event.getSource() == rbDolo) { setPanel(_DOLO); }
    if (event.getSource() == rbPyro) { setPanel(_PYRO); }

    for (i=0; i<14; i++)
    {
	  if (event.getSource() == rb[i])
	  {
	    setTexture( i );
	  }
	}

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  if (event.getSource() == rbP[i])
	  {
	    setTextureBTNS( i );
	  }
	}

    if (notifier != null)
      notifier.notifyObservers(new String("Primary Sediments Selected"));
  }
}

/*
 *  @version 1.1 04/19/2011
 *  @author  John Victorine
 */
