/*
 * @lithSedimentPanel.java Version 1.1 04/19/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import lith.gui.lithSedimentsTexturePanel;
import lith.gui.lithCarbonateTexturePanel;
import lith.gui.lithChemicalTexturePanel;
import lith.gui.lithOrganicTexturePanel;

/** Class lithSedimentPanel
 *  <p> This Class will display the sedimentary structures selected
 *
 *  @version 1.1 04/19/2011
 *  @author  John R. Victorine
 */

public class lithSedimentPanel extends Panel implements ActionListener, Observer
{
  // Input Variables

  private Observable pNotifier = null;     // Panel Observable

  // Global Variables

  private Observable notifier = null;     // Panel Observable

  private int iPrimary = lith.lithConstants._NONE;

  private int    iRock    = -1;      // Primary Lithology
  private String sRock    = "";      // Primary Lithology Description
  private int    iTexture = -1;      // Texture Energy Level
  private String sTexture = "";      // Texture Level Description
  private String second   = "";      // Secondary Lithologies

  private lithSedimentsTexturePanel pSediments = null; // Sediments
  private lithCarbonateTexturePanel pCarbonate = null; // Carbonate
  private lithChemicalTexturePanel  pChemical  = null; // Chemical
  private lithOrganicTexturePanel   pOrganic   = null; // Organic

  // Global Frame Widgets

  private static final int _SEDIMENTS = 0; // Sediments Texture Scale
  private static final int _CARBONATE = 1; // Carbonate Texture Scale
  private static final int _CHEMICAL  = 2; // Chemical Texture Scale
  private static final int _ORGANIC   = 3; // Allochems Texture Scale

  private int iPanel = _SEDIMENTS;

  // -- Diplay Descriptive Data Panel

  private JPanel       pnlData           = new JPanel();
  private CardLayout   cardLayout        = new CardLayout();
  private TitledBorder titledBorderEntry = null;

  // -- Texture Panel Radiobuttons

  private JRadioButton rbSediments = new JRadioButton();
  private JRadioButton rbCarbonate = new JRadioButton();
  private JRadioButton rbChemical  = new JRadioButton();
  private JRadioButton rbOrganic   = new JRadioButton();

  // -- Lithology Widgets

//  private JRadioButton rbP[]   = null;  // Primary Lithologies
  private JRadioButton rbS[][] = null;  // Secondary Lithologies
  private JLabel       lblS[]  = null;

  /** Constructor lithSedimentPanel()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   */

  public lithSedimentPanel( Observable pNotifier )
  {
    try
    {
      this.pNotifier = pNotifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	int    i,j;
	int    iTotal       = 0;
	int    iCount       = 0;
	String sTemp        = "";

    JPanel pnlNorth     = new JPanel();
    JPanel pnlWest      = new JPanel();
    JPanel pnlCenter    = new JPanel();

    JPanel pnl2nd       = new JPanel();
    JPanel pnl2ndTop    = new JPanel();
    JPanel pnl2ndBtns   = new JPanel();
    JPanel pnl2ndary[]  = null;
    JPanel pnl2ndBtn1[] = null;
    JPanel pnl2ndBtn2[] = null;
    JLabel lbl2nd       = new JLabel();

    JScrollPane scrollP = new JScrollPane();
    JScrollPane scrollS = new JScrollPane();

    ButtonGroup group    = new ButtonGroup();
    ButtonGroup groupS[] = null;

    TitledBorder titledBorder1st = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Primary:");
    titledBorder1st.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorder2nd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Secondary Lithologies:");
    titledBorder2nd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Create an Observable

    notifier = new lithSedimentPanelObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Build Panel

    this.setLayout(new BorderLayout());

    // North Panel

    pnlNorth.setLayout(new GridLayout());
    pnlNorth.setBackground(Color.white);

    // .. Detrital Sedimentary Rocks

    rbSediments.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbSediments.setBackground(Color.white);
    rbSediments.setHorizontalAlignment(SwingConstants.CENTER);
    rbSediments.setSelected(true);
    rbSediments.setText( "Detrital Sediments" );
    rbSediments.addActionListener(this);

    // .. Carbonate Sedimentary Rocks

    rbCarbonate.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbCarbonate.setBackground(Color.white);
    rbCarbonate.setHorizontalAlignment(SwingConstants.CENTER);
    rbCarbonate.setText( "Carbonate Rocks" );
    rbCarbonate.addActionListener(this);

    // .. Metamorphic Rocks

    rbChemical.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbChemical.setBackground(Color.white);
    rbChemical.setHorizontalAlignment(SwingConstants.CENTER);
    rbChemical.setText( "Chemical Rocks" );
    rbChemical.addActionListener(this);

    // .. Igneous Rocks

    rbOrganic.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbOrganic.setBackground(Color.white);
    rbOrganic.setHorizontalAlignment(SwingConstants.CENTER);
    rbOrganic.setText( "Organic Rocks" );
    rbOrganic.addActionListener(this);

    // Texture Panels

    pnlWest.setLayout(new BorderLayout());
    pnlWest.setBackground(Color.white);

    pnlData.setLayout( cardLayout );
    pnlData.setBackground(Color.white);

    pSediments = new lithSedimentsTexturePanel( notifier );
    pCarbonate = new lithCarbonateTexturePanel( notifier );
    pChemical  = new lithChemicalTexturePanel( notifier );
    pOrganic   = new lithOrganicTexturePanel( notifier );

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBackground(Color.white);

    // -- Secondary Lithology

    pnl2nd.setLayout(new BorderLayout());
    pnl2nd.setBackground(Color.white);
    pnl2nd.setBorder(titledBorder2nd);

    lbl2nd.setFont(new java.awt.Font("Monospaced", 2, 11));
    lbl2nd.setBackground(Color.white);
    lbl2nd.setForeground(Color.red);
    lbl2nd.setHorizontalAlignment(SwingConstants.CENTER);
    lbl2nd.setText( "sli=slightly; med=medium; vry=very; ex=extremely" );

    iTotal = lith.lithConstants._SECONDARY;

    pnl2ndBtns.setLayout(new GridLayout(iTotal, 1));
    pnl2ndBtns.setBackground(Color.white);
    pnl2ndBtns.setPreferredSize(new Dimension(10, iTotal*25));

    pnl2ndary  = new JPanel[lith.lithConstants._SECONDARY];
    pnl2ndBtn1 = new JPanel[lith.lithConstants._SECONDARY];
    pnl2ndBtn2 = new JPanel[lith.lithConstants._SECONDARY];
    lblS       = new JLabel[lith.lithConstants._SECONDARY];
    rbS        = new JRadioButton[lith.lithConstants._SECONDARY][5];
    groupS     = new ButtonGroup[lith.lithConstants._SECONDARY];

    for (i=0; i<lith.lithConstants._SECONDARY; i++)
    {
	  pnl2ndary[i] = new JPanel();
      pnl2ndary[i].setLayout(new BorderLayout());
      pnl2ndary[i].setBackground(Color.white);
      pnl2ndary[i].setBorder(BorderFactory.createEtchedBorder());

      pnl2ndBtn1[i] = new JPanel();
      pnl2ndBtn1[i].setBackground(Color.white);
      pnl2ndBtn1[i].setPreferredSize(new Dimension(100, 10));

      pnl2ndBtn2[i] = new JPanel();
      pnl2ndBtn2[i].setBackground(Color.white);
      pnl2ndBtn2[i].setLayout(new GridLayout());

  	  groupS[i] = new ButtonGroup();

	  lblS[i] = new JLabel();
      lblS[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      lblS[i].setBackground(Color.white);
      lblS[i].setHorizontalAlignment(SwingConstants.LEFT);
      lblS[i].setText( lith.lithConstants.SECONDARY[i] );

      for (j=0; j<5; j++)
      {
  	    rbS[i][j] = new JRadioButton();
  	    rbS[i][j].setBackground(Color.white);
        rbS[i][j].setFont(new java.awt.Font("Monospaced", 1, 11));
        rbS[i][j].setHorizontalAlignment(SwingConstants.LEFT);
        if (j == 0) { rbS[i][j].setSelected(true); }
        if (j == 0) { rbS[i][j].setText( "" ); }
        if (j == 1) { rbS[i][j].setText( "sli" ); }
        if (j == 2) { rbS[i][j].setText( "med" ); }
        if (j == 3) { rbS[i][j].setText( "vry" ); }
        if (j == 4) { rbS[i][j].setText( "ex" ); }
        rbS[i][j].addActionListener(this);

        pnl2ndBtn2[i].add(rbS[i][j],     null);
        groupS[i].add( rbS[i][j] );
	  }

      pnl2ndBtns.add( pnl2ndary[i],    null);
      pnl2ndary[i].add( pnl2ndBtn1[i], BorderLayout.EAST );
      pnl2ndBtn1[i].add( lblS[i],      null );
      pnl2ndary[i].add( pnl2ndBtn2[i], BorderLayout.CENTER );
    }

    setPrimary( iPrimary );

    // Attach Widgets to Panel

    this.add(pnlNorth,                    BorderLayout.NORTH );
    pnlNorth.add( rbSediments,            null );
    pnlNorth.add( rbCarbonate,            null );
    pnlNorth.add( rbChemical,             null );
    pnlNorth.add( rbOrganic,              null );

    group.add( rbSediments );
    group.add( rbCarbonate );
    group.add( rbChemical );
    group.add( rbOrganic );

    this.add(pnlWest,                     BorderLayout.WEST );
    pnlWest.add( pnlData,                 BorderLayout.CENTER );
    pnlData.add( pSediments,              "SEDIMENTS" );
    pnlData.add( pCarbonate,              "CARBONATE" );
    pnlData.add( pChemical,               "CHEMICAL" );
    pnlData.add( pOrganic,                "ORGANIC" );

    this.add( pnlCenter,                  BorderLayout.CENTER );
    pnlCenter.add( pnl2nd,                BorderLayout.CENTER );
    pnl2nd.add( pnl2ndTop,                BorderLayout.NORTH );
    pnl2ndTop.add( lbl2nd,                null);

    pnl2nd.add( scrollS,                  BorderLayout.CENTER);
    scrollS.getViewport().add(pnl2ndBtns, null);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLithology()
   *  <p> This method will retrieve the Primary Lithology selected
   *  @return sRock = The Primary Lithology Description
   */

  public String getLithology() { return (sRock); }

  /** Method getSecondary()
   *  <p> This method will retrieve the Secondary Litghologies
   * @return second = The Secondary Lithologies
   */

  public String getSecondary() { return (second); }

  /** Method getTexture()
   *  <p> This method will retrieve the Texture Level selected
   *  @return iTexture = The Texture Level Selected
   */

  public int getTexture() { return (iTexture); }

  /** Method getTextureDesc()
   *  <p> This method will retrieve the Texture Level Description selected
   *  @return sTexture = The Texture Level Description Selected
   */

  public String getTextureDesc() { return (sTexture); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setPanel()
   * <p> This method will set the borders of the List and Data Entry Panel
   *  @param iPanel = The type of panel to display
   */

  private void setPanel(int iPanel)
  {
    this.iPanel = iPanel;

	switch (iPanel)
	{
	  case _SEDIMENTS:
        cardLayout.show(pnlData, "SEDIMENTS");
        break;
      case _CARBONATE:
        cardLayout.show(pnlData, "CARBONATE");
        break;
	  case _CHEMICAL:
        cardLayout.show(pnlData, "CHEMICAL");
        break;
	  case _ORGANIC:
        cardLayout.show(pnlData, "ORGANIC");
        break;
    }

    pnlData.updateUI();
  }

  /** Method setPrimary()
   * <p> This method will enable or disable buttons depending on data
   * @param iPrimary = The primary radio button selected
   */

  private void setPrimary( int iPrimary )
  {
	int i,j;

	this.iPrimary = iPrimary;

    if (iPrimary > -1)
    {
      for (i=0; i<lith.lithConstants._SECONDARY; i++)
      {
	    rbS[i][0].setSelected(true);
	    for (j=0; j<5; j++)
	    {
	      rbS[i][j].setEnabled(false);
	      rbS[i][j].setForeground(Color.lightGray);
	      rbS[i][j].setBackground(Color.white);
	      lblS[i].setForeground(Color.lightGray);
	      lblS[i].setBackground(Color.white);

	      if (lith.lithConstants.ACTION[iPrimary][i] == lith.lithConstants._ON)
	      {
 	        rbS[i][j].setEnabled(true);
  	        rbS[i][j].setForeground(Color.black);
  	        rbS[i][j].setBackground(Color.white);
  	        lblS[i].setForeground(Color.black);
  	        lblS[i].setBackground(Color.white);
  	      }
        }
      }
	}
  }

  /** Method setSecondary()
   * <p> This method will set the secondary lithologies
   */

  private void setSecondary()
  {
	int    i,j;
	String sTemp  = "";
	int    iCount = 0;
    int    iOrder[] = null;
    String sOrder[] = null;
    int    iOrd   = -1;
    String sOrd   = "";
    boolean swappedOnPrevRun = true;

    for (i=0; i<lith.lithConstants._SECONDARY; i++)
    {
      for (j=1; j<5; j++)
      {
   	    if (rbS[i][j].isSelected())
		{
	      iCount++;
	    }
      }
    }

    if (iCount > 0)
    {
	  iOrder = new int[iCount];
	  sOrder = new String[iCount];

      iCount = 0;
      for (i=0; i<lith.lithConstants._SECONDARY; i++)
      {
        for (j=1; j<5; j++)
        {
	  	  if (rbS[i][j].isSelected())
		  {
			iOrder[iCount] = j;
			sOrder[iCount] = new String( lith.lithConstants.SECONDARY[i] );
			iCount++;
		  }
        }
      }
	}

	// Order the Secondary lithology by the amount flag

    while(swappedOnPrevRun)
    {
      // this variable keeps track of whether to continue sorting or exit

      swappedOnPrevRun = false;

      // loop through every element in the array, except for the last one

      for (i=0; i<iCount-1; i++)
      {
        // if current element is less than the next swap the two elements

        if(iOrder[i] < iOrder[i+1])
        {
          // we don't want the loop to end just yet, we're not done

          swappedOnPrevRun = true;

          // store element i in a temporary variable

          iOrd   = iOrder[i];
          sOrd   = new String( sOrder[i] );

          // set element i+1 to where i used to be

          iOrder[i] = iOrder[i+1];
          sOrder[i] = new String( sOrder[i+1] );

          // release the old i from temp into i+1 slot

          iOrder[i+1] = iOrd;
          sOrder[i+1] = new String( sOrd );
        }
	  }
	}

	second = new String("");

    for (i=0; i<iCount; i++)
    {
      switch (iOrder[i])
	  {
	    case 1:
	      sTemp = new String( "slightly "  + sOrder[i] );
	      break;
	    case 2:
	      sTemp = new String( sOrder[i] );
	      break;
	    case 3:
	      sTemp = new String( "very "      + sOrder[i] );
	      break;
	    case 4:
	      sTemp = new String( "extremely " + sOrder[i] );
	      break;
	  }

      if (sTemp.length() > 0)
      {
	    if (second.length() > 0)
	    {
		  second = new String( second + ", " + sTemp );
		}
		else
		{
		  second = new String( sTemp );
		}
	  }
	}

    if (pNotifier != null)
      pNotifier.notifyObservers(new String("Sedimentary Lithology Selected"));
  }

  /* ===================================================================== *
   * ----------------------------- CLOSE --------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    // Input Variables

    pNotifier = null;     // Panel Observable

    // Global Variables

    notifier = null;     // Panel Observable

    sRock    = null;      // Primary Lithology Description
    sTexture = null;      // Texture Level Description
    second   = null;      // Secondary Lithologies

    if (pSediments != null)
      pSediments.close();
    pSediments = null; // Sediments

    if (pCarbonate != null)
      pCarbonate.close();
    pCarbonate = null; // Carbonate

    if (pChemical != null)
      pChemical.close();
    pChemical  = null; // Chemical

    if (pOrganic != null)
      pOrganic.close();
    pOrganic   = null; // Organic

    // Global Frame Widgets

    // -- Diplay Descriptive Data Panel

    pnlData           = null;
    cardLayout        = null;
    titledBorderEntry = null;

    // -- Texture Panel Radiobuttons

    rbSediments = null;
    rbCarbonate = null;
    rbChemical  = null;
    rbOrganic   = null;

    // -- Lithology Widgets

    for (int i=0; i<lith.lithConstants._SECONDARY; i++)
    {
	  lblS[i] = null;

      for (int j=0; j<5; j++)
      {
  	    rbS[i][j] = null;
  	  }
    }

    rbS  = null;  // Secondary Lithologies
    lblS = null;
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i,j;

    if (event.getSource() == rbSediments) { setPanel( _SEDIMENTS ); }
    if (event.getSource() == rbCarbonate) { setPanel( _CARBONATE ); }
    if (event.getSource() == rbChemical)  { setPanel( _CHEMICAL ); }
    if (event.getSource() == rbOrganic)   { setPanel( _ORGANIC ); }

    for (i=0; i<lith.lithConstants._SECONDARY; i++)
    {
      for (j=0; j<5; j++)
      {
	    setSecondary();
      }
    }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String((String) arg);

    if (sArg.equals("Primary Sediments Selected"))
    {
      iRock    = pSediments.getLithology();
      if (iRock > -1)
      {
        sRock    = new String( lith.lithConstants.PRIMARY[iRock][0]);
      }

      iTexture = pSediments.getTexture();
      sTexture = new String( pSediments.getTextureDesc() );

      setPrimary(iRock);
      setSecondary();

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Sedimentary Lithology Selected"));
	}

    if (sArg.equals("Primary Carbonates Selected"))
    {
      iRock    = pCarbonate.getLithology();
      if (iRock > -1)
      {
        sRock    = new String( lith.lithConstants.PRIMARY[iRock][0]);
	  }

      iTexture = pCarbonate.getTexture();
      sTexture = new String( pCarbonate.getTextureDesc() );

      setPrimary(iRock);
      setSecondary();

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Sedimentary Lithology Selected"));
	}

	if (sArg.equals("Primary Chemical Selected"))
	{
      iRock    = pChemical.getLithology();
      if (iRock > -1)
      {
        sRock    = new String( lith.lithConstants.PRIMARY[iRock][0]);
	  }

      iTexture = pChemical.getTexture();
      sTexture = new String( pChemical.getTextureDesc() );

      setPrimary(iRock);
      setSecondary();

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Sedimentary Lithology Selected"));
	}

	if (sArg.equals("Primary Organic Selected"))
	{
      iRock    = pOrganic.getLithology();
      if (iRock > -1)
      {
        sRock    = new String( lith.lithConstants.PRIMARY[iRock][0]);
	  }

      iTexture = pOrganic.getTexture();
      sTexture = new String( pOrganic.getTextureDesc() );

      setPrimary(iRock);
      setSecondary();

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Sedimentary Lithology Selected"));
	}
  }
}

/** CLASS lithSedimentPanelObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class lithSedimentPanelObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 04/19/2011
 *  @author  John Victorine
 */
