/*
 * @lithImageTrackFrame.java Version 1.1 02/05/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import las.lasFileDataStruct;

/** Class lithImageTrackFrame
 *  <p> This Class will provide
 *
 *  @version 1.1 02/05/2009
 *  @author  John R. Victorine
 */

public class lithImageTrackFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable        notifier = null;
  private lasFileDataStruct stLAS    = null;

  // Global Frame Variables

  private int    iTool     = las.lasStandardTools._GR;
  private double dMinimum  =   0.0;
  private double dMaximum  = 150.0;

  // Resistivity/Porosity Track Macros

  public static final int _RT   = 0;
  public static final int _PHIT = 1;

  private int iTrack = _RT;

  // Resistivity Track Variables

  private int    iRt    = las.lasStandardTools._ILD;
  private double dRtMax = iqstrat.iqstratTracksStruct.MMHO_MAX;
  private double dRtMin = iqstrat.iqstratTracksStruct.MMHO_MIN;

  // Porosity Track Variables

  private int    iPhit    = las.lasStandardTools._NPHI;
  private double dPhitMax = iqstrat.iqstratTracksStruct.PHIT_MAX;
  private double dPhitMin = iqstrat.iqstratTracksStruct.PHIT_MIN;

  // Global Frame Widgets

  private JButton btnOk     = new JButton();
  private JButton btnCurves = new JButton();
  private JButton btnCancel = new JButton();

  private JTextField txtDescription = new JTextField();
  private JTextField txtMnemonic    = new JTextField();
  private JTextField txtMinimum     = new JTextField();
  private JTextField txtMaximum     = new JTextField();

  // Curve List Frame Variables

  private JFrame       frame    = null;
  private int          iTotal   = 0;
  private JRadioButton rb[]     = null;

  private JButton      btnClose = null;

  /** Constructor lithImageTrackFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stLAS    = LAS File Data Structure
   */

  public lithImageTrackFrame( Observable notifier, lasFileDataStruct stLAS )
  {
    try
    {
      this.notifier = notifier;
      this.stLAS    = stLAS;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlButtons     = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlNames       = new JPanel();
    JPanel pnlMnemonic    = new JPanel();
    JPanel pnlDescription = new JPanel();
    JPanel pnlLimits      = new JPanel();
    JPanel pnlMin         = new JPanel();
    JPanel pnlMax         = new JPanel();

    TitledBorder titledBorderMnemonic = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Mnemonic:");
    TitledBorder titledBorderDesc = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Description:");
    TitledBorder titledBorderMin = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Minimum:");
    TitledBorder titledBorderMax = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Maximum:");

    titledBorderMnemonic.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderDesc.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderMin.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderMax.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Resistivity & Porosity Image Track");

    // Build Log Curve Radio Button Array

    rb = new JRadioButton[las.lasStandardTools.TOTAL];
    for (int i=0; i<las.lasStandardTools.TOTAL; i++)
    {
      rb[i] = new JRadioButton();
      rb[i].setEnabled(false);
      rb[i].setFont(new java.awt.Font("Dialog", 0, 11));
      rb[i].setText(las.lasStandardTools.LAS_TOOLS[i][1] + " - " +
                    las.lasStandardTools.LAS_TOOLS[i][2]);
      rb[i].addActionListener(this);
    }

    // Curve Data Panel

    pnlCenter.setLayout(new GridLayout(2,1));

    // -- Log Curve Names Panel

    pnlNames.setLayout(new BorderLayout());
    pnlNames.setPreferredSize(new Dimension(110, 50));

    // ---- Log Curve Mnemonic Panel

    pnlMnemonic.setBorder(titledBorderMnemonic);
    pnlMnemonic.setPreferredSize(new Dimension(100, 10));
    pnlMnemonic.setLayout(new BorderLayout());

    txtMnemonic.setEditable(false);
    txtMnemonic.setText(las.lasStandardTools.LAS_TOOLS[0][1]);

    // ---- Log Curve Description Panel

    pnlDescription.setBorder(titledBorderDesc);
    pnlDescription.setLayout(new BorderLayout());

    txtDescription.setEditable(false);
    txtDescription.setText(las.lasStandardTools.LAS_TOOLS[0][2]);

    // -- Log Curve Limits Panel

    pnlLimits.setLayout(new GridLayout());

    // Minimmum Value Panel

    pnlMin.setBorder(titledBorderMin);
    pnlMin.setLayout(new BorderLayout());

    txtMinimum.setText(""+dMinimum);
    txtMinimum.setHorizontalAlignment(SwingConstants.TRAILING);
    txtMinimum.addFocusListener(new lithImageTrackFrameFocusAdapter(this));

    // Maximum Value Panel

    pnlMax.setBorder(titledBorderMax);
    pnlMax.setLayout(new BorderLayout());

    txtMaximum.setText(""+dMaximum);
    txtMaximum.setHorizontalAlignment(SwingConstants.TRAILING);
    txtMaximum.addFocusListener(new lithImageTrackFrameFocusAdapter(this));

    // Global Frame Buttons Panel

    pnlButtons.setBorder(BorderFactory.createEtchedBorder());

    btnOk.setFont(new java.awt.Font("Dialog", 1, 11));
    btnOk.setPreferredSize(new Dimension(75, 25));
    btnOk.setText("Ok");
    btnOk.addActionListener(this);

    btnCurves.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCurves.setText("Select Log Curve");
    btnCurves.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setPreferredSize(new Dimension(75, 25));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // Attach Widgets to Frame

    this.getContentPane().add(pnlButtons, BorderLayout.SOUTH);
    pnlButtons.add(btnOk,                 null);
    pnlButtons.add(btnCurves,             null);
    pnlButtons.add(btnCancel,             null);

    this.getContentPane().add(pnlCenter,  BorderLayout.CENTER);
    pnlCenter.add(pnlNames,               null);
    pnlNames.add(pnlMnemonic,             BorderLayout.WEST);
    pnlMnemonic.add(txtMnemonic,          BorderLayout.CENTER);

    pnlNames.add(pnlDescription,          BorderLayout.CENTER);
    pnlDescription.add(txtDescription,    BorderLayout.CENTER);

    pnlCenter.add(pnlLimits,              null);
    pnlLimits.add(pnlMin,                 null);
    pnlMin.add(txtMinimum,  BorderLayout.CENTER);

    pnlLimits.add(pnlMax,                 null);
    pnlMax.add(txtMaximum, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(450, 200));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(false);
  }

  /** Method panelLogCurves()
   * <p> This method will build the log curve panel in order to change the
   *     type of log curve that is displayed.
   * @return panel = The panel of radio buttons
   */

  private JPanel panelLogCurves()
  {
    int         iCount  = 0;
    int         iStart  = 0;
    int         iEnd    = 0;
    JPanel      panel   = new JPanel();
    JPanel      pnlList = new JPanel();
    JPanel      pnlBTN  = new JPanel();
    JScrollPane scroll  = new JScrollPane();
    ButtonGroup group   = new ButtonGroup();

    JPanel pnl[]        = null;

    switch (iTrack)
    {
      case _RT:
        iStart = las.lasStandardTools._RES;
        iEnd   = las.lasStandardTools._AHT90 + 1;
        iTotal = las.lasStandardTools.TOTAL_RES;
        break;
      case _PHIT:
        iStart = las.lasStandardTools._DPHI;
        iEnd   = las.lasStandardTools._SPHI + 1;
        iTotal = las.lasStandardTools.TOTAL_PHI;
        break;
    }

    panel.setPreferredSize(new Dimension(10, 180));
    panel.setLayout(new BorderLayout());

    // Available Log Curves

    if (iTotal > 0)
    {
      pnlList.setLayout(new GridLayout(iTotal,1));
      pnlList.setSize(100, iTotal*20);

      pnl = new JPanel[las.lasStandardTools.TOTAL];

      for (int i=0; i<las.lasStandardTools.TOTAL; i++)
      {
        pnl[i] = new JPanel();
        pnl[i].setLayout(new BorderLayout());

        if (i == iTool) rb[i].setSelected(true);
        rb[i].setEnabled(false);
        if (stLAS.checkData(i))
          rb[i].setEnabled(true);

        if ((i >= iStart) & (i < iEnd))
        {
          pnl[i].add(rb[i], BorderLayout.WEST);
          pnlList.add(pnl[i], null);
          group.add(rb[i]);
        }
      }

      btnClose = new JButton();
      btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
      btnClose.setPreferredSize(new Dimension(75, 25));
      btnClose.setText("Close");
      btnClose.addActionListener(this);

      panel.add(scroll, BorderLayout.CENTER);
      scroll.getViewport().add(pnlList, null);

      panel.add(pnlBTN,    BorderLayout.SOUTH);
      pnlBTN.add(btnClose, null);
    }

    return (panel);
  }

  /** Method displayCurves()
   * <p> This method will display the log curves for selection
   */

  private void displayCurves()
  {
    if (frame != null)
      frame.dispose();
    frame = null;

    frame = new JFrame();

    switch (iTrack)
    {
      case _RT:
        frame.setTitle("Change Resistivity Curve");
        break;
      case _PHIT:
        frame.setTitle("Change Porosity Curve");
        break;
    }

    JPanel panel = panelLogCurves();
    frame.getContentPane().add(panel, null);

    // Set the Size and Display the Frame

    frame.setSize(200, 250);

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    frame.setLocation((d.width - frame.getSize().width)/2,
                      (d.height - frame.getSize().height)/2);

    frame.setResizable(false);
    frame.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getTrack()
   * <p> This method will return the selected track
   * @return iTrack = the selected track
   */

  public int getTrack() { return (iTrack); }

  /** Method getTool()
   * <p> This method will return the Log Curve identifier
   * @return iTool = the Log Curve identifier
   */

  public int getTool() { return (iTool); }

  /** Method getMinimum()
   * <p> This method will return the minimum value of the track
   * @return dMinimum = the minimum value of the track
   */

  public double getMinimum() { return (dMinimum); }

  /** Method getMaximum()
   * <p> This method will return the maximum value of the track
   * @return dMaximum = the maximum value of the track
   */

  public double getMaximum() { return (dMaximum); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setTrack( int iTrack )
   * <p> this method will set the Image Track Variables needed
   * @param iTrack = Image Track Selected
   */

  public void setTrack( int iTrack )
  {
    this.iTrack = iTrack;

    switch (iTrack)
    {
      case _RT:
        iTool     = iRt;
        dMinimum  = dRtMin;
        dMaximum  = dRtMax;
        break;
      case _PHIT:
        iTool     = iPhit;
        dMinimum  = dPhitMin;
        dMaximum  = dPhitMax;
        break;
    }

    txtMnemonic.setText(las.lasStandardTools.LAS_TOOLS[iTool][1]);
    txtDescription.setText(las.lasStandardTools.LAS_TOOLS[iTool][2]);
    txtMinimum.setText(""+dMinimum);
    txtMaximum.setText(""+dMaximum);
  }

  /** Method setOHM( int iRt, double dRtMax, double dRtMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iRt    = Identifier representing curve to use in plotting
   * @param dRtMax = the maximum conductivity value
   * @param dRtMin = the minimum conductivity value
   */

  public void setOHM( int iRt, double dRtMax, double dRtMin )
  {
    this.iRt    = iRt;
    this.dRtMax = dRtMax;
    this.dRtMin = dRtMin;
  }

  /** Method setPHI( int iPhit, double dPhitMax, double dPhitMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iPhit    = Identifier representing curve to use in plotting
   * @param dPhitMax = the maximum porosity value
   * @param dPhitMin = the minimum porosity value
   */

  public void setPHI( int iPhit, double dPhitMax, double dPhitMin )
  {
    this.iPhit    = iPhit;
    this.dPhitMax = dPhitMax;
    this.dPhitMin = dPhitMin;
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method ok()
   * <p> This method will transfer the data to the variables and send a message
   */

  private void ok()
  {
    switch (iTrack)
    {
      case _RT:
        setOHM( iTool, dMaximum, dMinimum );
        break;
      case _PHIT:
        setPHI( iTool, dMaximum, dMinimum );
        break;
    }

    notifier.notifyObservers(new String("Image Track Changed"));
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier = null;
    stLAS    = null;

    if (frame != null)
      frame.dispose();
    frame = null;

    for (int i=0; i<las.lasStandardTools.TOTAL; i++)
    {
      rb[i] = null;
    }

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    for (int i=0; i<las.lasStandardTools.TOTAL; i++)
    {
      if (event.getSource() == rb[i])
      {
        iTool = i;
        txtMnemonic.setText(las.lasStandardTools.LAS_TOOLS[iTool][1]);
        txtDescription.setText(las.lasStandardTools.LAS_TOOLS[iTool][2]);
      }
    }

    if (event.getSource() == btnCurves) { displayCurves(); }

    if (event.getSource() == btnOk) { ok(); }

    if (event.getSource() == btnCancel) { this.setVisible(false); }

    if (event.getSource() == btnClose)
    {
      btnClose = null;
      frame.dispose();
    }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtMinimum)
    {
      iNumeric = 1;
      sTemp = txtMinimum.getText();
      sMessage = new String("Minimum Value is a Numeric Field");
    }

    if (e.getSource() == txtMaximum)
    {
      iNumeric = 1;
      sTemp = txtMaximum.getText();
      sMessage = new String("Maximum Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtMinimum)
        {
          dMinimum = cmn.cmnString.stringToDouble(txtMinimum.getText());
        }

        if (e.getSource() == txtMaximum)
        {
          dMaximum = cmn.cmnString.stringToDouble(txtMaximum.getText());
        }
      }
    }
  }
}

/** CLASS lithImageTrackFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class lithImageTrackFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  lithImageTrackFrame adaptee;

  lithImageTrackFrameFocusAdapter(lithImageTrackFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 02/05/2009
 *  @author  John Victorine
 */
