/*
 * @lithChemicalTexturePanel.java Version 1.1 04/27/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class lithChemicalTexturePanel
 *  <p> This Class will display the Chemical lithology sedimentary rock selected
 *
 *  @version 1.1 04/27/2011
 *  @author  John R. Victorine
 */

public class lithChemicalTexturePanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable notifier = null;     // Panel Observable

  // Global Variables

  public static final String XLN[] = lith.lithConstants.XLN;
  public static final int XLN_COLOR[][]  = lith.lithConstants.XLN_COLOR;

  public static final String XLN2[] = lith.lithConstants.XLN2;
  public static final int XLN_COLOR2[][] = lith.lithConstants.XLN_COLOR2;

  public static final int PRIMARY_SIZE[][] = lith.lithConstants.PRIMARY_SIZE;

  private int    iRock    = -1;      // Primary Lithology
  private int    iTexture = -1;      // Texture Energy Level
  private String sTexture = "";      // Texture Level Description

  // Global Panel Widgets

  // -- Lithology & Texture Widgets

  private JRadioButton rb[]   = null;  // Sediment Size
  private JRadioButton rbP[]  = null;  // Primary Lithologies

  /** Constructor lithChemicalTexturePanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public lithChemicalTexturePanel( Observable notifier )
  {
    try
    {
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	int i=0;
	int    COLOR[][]   = null;
	String SIZE[]      = null;

    JPanel pnlTop      = new JPanel();

    JPanel pnlCenter   = new JPanel();
    JPanel pnlChemical = new JPanel();
    JPanel pnlWest     = new JPanel();

    ButtonGroup group  = new ButtonGroup();
    ButtonGroup groupP = new ButtonGroup();

    this.setLayout(new BorderLayout());
    this.setBackground(new Color( 255, 255, 255 ));

    // West Panel

    pnlWest.setLayout(new GridLayout(7,1));
    pnlWest.setBackground(new Color( 255, 255, 255 ));

    // -- Grain Size

    rb = new JRadioButton[14];

    COLOR = XLN_COLOR;
    SIZE  = XLN;

    for (i=13; i>-1; i--)
    {
	  rb[i] = new JRadioButton();
      rb[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      rb[i].setHorizontalAlignment(SwingConstants.LEFT);
      rb[i].setBackground(new Color(COLOR[i][0],COLOR[i][1],COLOR[i][2]));
      if ((i==0) || (i==1))
      {
        rb[i].setForeground(new Color( 255, 255, 255 ));
      }
      rb[i].setText( SIZE[i] );
      if (SIZE[i].length() == 0) { rb[i].setEnabled(false); }
      rb[i].addActionListener(this);

      if ((i==0) || (i==1) || (i==2) || (i==3) || (i==5) || (i==7) || (i==9))
      {
        pnlWest.add( rb[i], null);
      }
      group.add( rb[i] );
	}

    // -- Build Main Lithologies

    rbP = new JRadioButton[lith.lithConstants._PRIMARY];

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = new JRadioButton();
      rbP[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      rbP[i].setHorizontalAlignment(SwingConstants.LEFT);
      rbP[i].setBackground(new Color( 255, 255, 255 ));
      rbP[i].setText( lith.lithConstants.PRIMARY[i][0] );
      rbP[i].addActionListener(this);

      groupP.add( rbP[i] );
    }

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBackground(new Color( 255, 255, 255 ));

    pnlChemical = buildPanel();

    // Attach Widgets to Panel

    this.add(pnlWest,            BorderLayout.WEST);

    this.add(pnlCenter,          BorderLayout.CENTER);
    pnlCenter.add(pnlChemical,   BorderLayout.CENTER);
  }

  /** Method buildPanel()
   * <p> This method will build the Chemical Lithology Panel
   */

  private JPanel buildPanel()
  {
	JPanel panel  = new JPanel();
	JPanel pnlCO3 = new JPanel();
	JPanel pnlSO4 = new JPanel();
	JPanel pnlCl  = new JPanel();
	JPanel pnlPO4 = new JPanel();
	JPanel pnlFe  = new JPanel();

    TitledBorder titledBorderCO3 = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Carbonates");
    titledBorderCO3.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSO4 = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Sulfates");
    titledBorderSO4.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderCl = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Chlorides");
    titledBorderCl.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderPO4 = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Phosphates");
    titledBorderPO4.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderFe = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Iron Minerals");
    titledBorderFe.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    panel.setLayout(new GridLayout(5,1));
    panel.setBackground(new Color( 255, 255, 255 ));

    pnlCO3.setLayout(new GridLayout(2,1));
    pnlCO3.setBorder(titledBorderCO3);
    pnlCO3.setBackground(new Color( 255, 255, 255 ));

    pnlCO3.add(rbP[lith.lithConstants._LIMESTONE],  null);
    pnlCO3.add(rbP[lith.lithConstants._DOLOMITE],  null);

    pnlSO4.setLayout(new GridLayout(2,1));
    pnlSO4.setBorder(titledBorderSO4);
    pnlSO4.setBackground(new Color( 255, 255, 255 ));

    pnlSO4.add(rbP[lith.lithConstants._ANHYDRITE],  null);
    pnlSO4.add(rbP[lith.lithConstants._GYPSUM],  null);

    pnlCl.setLayout(new GridLayout(3,1));
    pnlCl.setBorder(titledBorderCl);
    pnlCl.setBackground(new Color( 255, 255, 255 ));

    pnlCl.add(rbP[lith.lithConstants._SALT],  null);
    pnlCl.add(rbP[lith.lithConstants._HALITE],  null);
    pnlCl.add(rbP[lith.lithConstants._SYLVITE],  null);

    pnlPO4.setLayout(new GridLayout());
    pnlPO4.setBorder(titledBorderPO4);
    pnlPO4.setBackground(new Color( 255, 255, 255 ));

    pnlPO4.add(rbP[lith.lithConstants._PHOSPHATE],  null);

    pnlFe.setLayout(new GridLayout(4,1));
    pnlFe.setBorder(titledBorderFe);
    pnlFe.setBackground(new Color( 255, 255, 255 ));

    pnlFe.add(rbP[lith.lithConstants._SIDERITE],  null);
    pnlFe.add(rbP[lith.lithConstants._PYRITE],    null);
    pnlFe.add(rbP[lith.lithConstants._HEMATITE],   null);
    pnlFe.add(rbP[lith.lithConstants._LIMONITE],  null);

	// Attach widgets to panel

	panel.add(pnlCO3, null);
	panel.add(pnlSO4, null);
	panel.add(pnlCl,  null);
	panel.add(pnlPO4, null);
	panel.add(pnlFe,  null);

	return (panel);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLithology()
   *  <p> This method will retrieve the Primary Lithology selected
   *  @return iRock = The Primary Lithology
   */

  public int getLithology() { return (iRock); }

  /** Method getTexture()
   *  <p> This method will retrieve the Texture Level selected
   *  @return iTexture = The Texture Level Selected
   */

  public int getTexture() { return (iTexture); }

  /** Method getTextureDesc()
   *  <p> This method will retrieve the Texture Level Description selected
   *  @return sTexture = The Texture Level Description Selected
   */

  public String getTextureDesc() { return (sTexture); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setTextureBTNS()
   *  <p> This method will set the Texture values depending on the seleced
   *      lithology.
   *  @param iRock = is the selected Rock button
   */

  private void setTextureBTNS( int iRock )
  {
	int    COLOR[][]   = null;
	String SIZE[]      = null;

	this.iRock = iRock;

    switch (iRock)
    {
	  case lith.lithConstants._PYRITE:
	  case lith.lithConstants._LIMONITE:
	  case lith.lithConstants._HEMATITE:
	  case lith.lithConstants._SIDERITE:
	    SIZE  = XLN2;
	    COLOR = XLN_COLOR2;
	    break;
	  default:
	    SIZE  = XLN;
	    COLOR = XLN_COLOR;
	    break;
	}

    if (iRock > -1)
    {
      for (int i=13; i>-1; i--)
      {
	    if ((PRIMARY_SIZE[iRock][i] == lith.lithConstants._ON) ||
	        (PRIMARY_SIZE[iRock][i] == lith.lithConstants._SET))
	    {
          rb[i].setBackground(new Color( COLOR[i][0],
                                         COLOR[i][1],
                                         COLOR[i][2]));
          if (PRIMARY_SIZE[iRock][i] == lith.lithConstants._SET)
          {
  	        rb[i].setSelected( true );
  	        setTexture( i );
	      }
          rb[i].setText( SIZE[i] );
	    }
      }
    }
  }

  /** Method setTexture()
   *  <p> This method will set the Texture identifier
   *  @param iTexture = is the selected Texture button
   */

  private void setTexture( int iTexture )
  {
	String SIZE[] = null;

    switch (iRock)
    {
	  case lith.lithConstants._PYRITE:
	  case lith.lithConstants._LIMONITE:
	  case lith.lithConstants._HEMATITE:
	  case lith.lithConstants._SIDERITE:
	    SIZE = XLN2;
	    break;
	  default:
	    SIZE = XLN;
	    break;
	}

    this.sTexture = new String( SIZE[iTexture] );
    this.iTexture = iTexture+1;
  }

  /* ===================================================================== *
   * ----------------------------- CLOSE --------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
	int i = 0;

    // Input Variables

    notifier = null; // Panel Observable

    // Global Variables

    sTexture = null; // Texture Level Description

    // Global Panel Widgets

    // -- Lithology & Texture Widgets

    for (i=0; i<14; i++)
    {
	  rb[i] = null;
	}

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = null;
	}

    rb   = null;  // Sediment Size
    rbP  = null;  // Primary Lithologies
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i=0;

    for (i=0; i<14; i++)
    {
	  if (event.getSource() == rb[i])
	  {
	    setTexture( i );
	  }
	}

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  if (event.getSource() == rbP[i])
	  {
	    setTextureBTNS( i );
	  }
	}

    if (notifier != null)
      notifier.notifyObservers(new String("Primary Chemical Selected"));
  }
}

/*
 *  @version 1.1 04/27/2011
 *  @author  John Victorine
 */
