/*
 * @lithCarbonateTexturePanel.java Version 1.1 04/26/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package lith.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class lithCarbonateTexturePanel
 *  <p> This Class will display the carbonate textures
 *
 *  @version 1.1 04/26/2011
 *  @author  John R. Victorine
 */

public class lithCarbonateTexturePanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable notifier = null;     // Panel Observable

  // Global Variables

  public static final String FOLK_NAME[][] = lith.lithConstants.FOLK_NAME;

  private static final int _DUNHAM   = 0;
  private static final int _ALLOCHEM = 1;
  private static final int _REPLACE  = 2;

  private int    iPanel   = _DUNHAM;  // Texture Scale

  private static final int _INTRACLASTS = 0;
  private static final int _OOLITES     = 1;
  private static final int _FOSSILS     = 2;
  private static final int _PELLETS     = 3;

  private int iAllochem = _INTRACLASTS;

  private int    iRock    = -1;      // Primary Lithology
  private int    iTexture = -1;      // Texture Energy Level
  private String sTexture = "";      // Texture Level Description

  // Global Widgets

  // -- Display Descriptive Data Panel

  private JPanel       pnlData           = new JPanel();
  private CardLayout   cardLayout        = new CardLayout();
  private TitledBorder titledBorderMain  = null;

  // -- Textures Methods

  private JRadioButton rbDunham   = new JRadioButton();
  private JRadioButton rbAllochem = new JRadioButton();
  private JRadioButton rbReplace  = new JRadioButton();

  // -- Folk's Allochems

  private JRadioButton rbIntra     = new JRadioButton();
  private JRadioButton rbPel       = new JRadioButton();
  private JRadioButton rbBio       = new JRadioButton();
  private JRadioButton rbOO        = new JRadioButton();

  // -- Lithology & Texture Widgets

  private JRadioButton rb[][]  = null;  // Sediment Size A
  private JRadioButton rbP[]  = null;  // Primary Lithologies

  /** Constructor lithCarbonateTexturePanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public lithCarbonateTexturePanel( Observable notifier )
  {
    try
    {
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	int i,j;

    JPanel pnlTop      = new JPanel();
    JPanel pnlDUNHAM   = new JPanel();
    JPanel pnlALLOCHEM = new JPanel();
    JPanel pnlREPLACE  = new JPanel();

    JPanel pnlCenter   = new JPanel();
    JPanel pnlRocks    = new JPanel();

    ButtonGroup group  = new ButtonGroup();
    ButtonGroup groupS = new ButtonGroup();
    ButtonGroup groupP = new ButtonGroup();
    ButtonGroup groupA = new ButtonGroup();

    titledBorderMain = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151)),
        "Carbonate Rocks");

    titledBorderMain.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSize = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "");
    titledBorderSize.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Panel

    this.setBackground(Color.white);
    this.setBorder(titledBorderMain);
    this.setLayout(new BorderLayout());
    this.setPreferredSize(new Dimension(300, 10));

    // Top Panel

    pnlTop.setLayout(new GridLayout(2,2));
    pnlTop.setBackground(new Color( 255, 255, 255 ));
    pnlTop.setBorder(titledBorderSize);

    rbDunham.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbDunham.setHorizontalAlignment(SwingConstants.CENTER);
    rbDunham.setBackground(new Color( 255, 255, 255 ));
    rbDunham.setText( "Dunham          " );
    rbDunham.setSelected(true);
    rbDunham.addActionListener(this);

    rbAllochem.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbAllochem.setHorizontalAlignment(SwingConstants.CENTER);
    rbAllochem.setBackground(new Color( 255, 255, 255 ));
    rbAllochem.setText( "Folk's Allochems" );
    rbAllochem.addActionListener(this);

    rbReplace.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbReplace.setHorizontalAlignment(SwingConstants.CENTER);
    rbReplace.setBackground(new Color( 255, 255, 255 ));
    rbReplace.setText( "Crystalline     " );
    rbReplace.addActionListener(this);

    // -- Grain Size

    rb = new JRadioButton[14][4];

    for (i=13; i>-1; i--)
    {
      for (j=0; j<4; j++)
      {
	    rb[i][j] = new JRadioButton();
        rb[i][j].setFont(new java.awt.Font("Monospaced", 1, 11));
        rb[i][j].setHorizontalAlignment(SwingConstants.LEFT);
        rb[i][j].setBackground(new Color( 255, 255, 255 ));
        rb[i][j].setText( "" );
        rb[i][j].addActionListener(this);

        group.add( rb[i][j] );
      }
	}

    // -- Build Main Lithologies

    rbP = new JRadioButton[lith.lithConstants._PRIMARY];

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = new JRadioButton();
      rbP[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      rbP[i].setHorizontalAlignment(SwingConstants.CENTER);
      rbP[i].setBackground(new Color( 255, 255, 255 ));
      rbP[i].setText( lith.lithConstants.PRIMARY[i][0] );
      rbP[i].addActionListener(this);

      groupP.add( rbP[i] );
    }

    // -- Folk's Allochems

    rbIntra.setBackground(Color.white);
    rbIntra.setFont(new java.awt.Font("Dialog", 1, 11));
    rbIntra.setHorizontalAlignment(SwingConstants.LEFT);
    rbIntra.setBackground(new Color( 255, 255, 255 ));
    rbIntra.setSelected(true);
    rbIntra.setText("Intraclasts");
    rbIntra.addActionListener(this);

    rbPel.setBackground(Color.white);
    rbPel.setFont(new java.awt.Font("Dialog", 1, 11));
    rbPel.setBackground(new Color( 255, 255, 255 ));
    rbPel.setHorizontalAlignment(SwingConstants.LEFT);
    rbPel.setText("Pellets");
    rbPel.addActionListener(this);

    rbBio.setBackground(Color.white);
    rbBio.setFont(new java.awt.Font("Dialog", 1, 11));
    rbBio.setBackground(new Color( 255, 255, 255 ));
    rbBio.setHorizontalAlignment(SwingConstants.LEFT);
    rbBio.setText("Fossils");
    rbBio.addActionListener(this);

    rbOO.setBackground(Color.white);
    rbOO.setFont(new java.awt.Font("Dialog", 1, 11));
    rbOO.setBackground(new Color( 255, 255, 255 ));
    rbOO.setHorizontalAlignment(SwingConstants.LEFT);
    rbOO.setText("Oolites");
    rbOO.addActionListener(this);

    groupA.add(rbIntra);
    groupA.add(rbPel);
    groupA.add(rbBio);
    groupA.add(rbOO);

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());

    pnlRocks.setLayout(new GridLayout());
    pnlRocks.add(rbP[lith.lithConstants._LIMESTONE], null);
    pnlRocks.add(rbP[lith.lithConstants._DOLOMITE],  null);

    pnlData.setLayout(cardLayout);

    pnlDUNHAM   =  buildDUNHAM();
    pnlALLOCHEM = buildALLOCHEM();
    pnlREPLACE  = buildREPLACE();

    // Attach Widgets to Panel

    this.add(pnlTop,          BorderLayout.NORTH);
    pnlTop.add(rbDunham,      null);
    pnlTop.add(rbAllochem,    null);
    pnlTop.add(rbReplace,     null);

    groupS.add(rbDunham);
    groupS.add(rbAllochem);
    groupS.add(rbReplace);

    this.add(pnlCenter,       BorderLayout.CENTER);
    pnlCenter.add( pnlRocks,  BorderLayout.NORTH );

    pnlCenter.add( pnlData,   BorderLayout.CENTER );
    pnlData.add( pnlDUNHAM,   "DUNHAM" );
    pnlData.add( pnlALLOCHEM, "ALLOCHEM" );
    pnlData.add( pnlREPLACE,  "REPLACE" );

    setButtons();
    setPanel( iPanel );
  }

  /** Method buildDUNHAM()
   * <p> This method will build Dunham Carbonate Scale
   */

  private JPanel buildDUNHAM()
  {
	int    COLOR[][]    = null;
    JPanel panel        = new JPanel();

    JPanel pnlCenter    = new JPanel();

    JLabel lblBioherm   = new JLabel();

    JPanel pnlBioherm   = new JPanel();
    JPanel pnlBioC      = new JPanel();
    JPanel pnlBioC1     = new JPanel();
    JPanel pnlBioC2     = new JPanel();

    JPanel pnl1         = new JPanel();

    JPanel pnlGrain     = new JPanel();
    JPanel pnlGrain2    = new JPanel();
    JPanel pnlGrain2a   = new JPanel();
    JPanel pnlGrain21   = new JPanel();
    JPanel pnlGrain22   = new JPanel();

    JLabel lblGrain     = new JLabel();
    JLabel lblGrain2    = new JLabel();

    JPanel pnl2         = new JPanel();

    JPanel pnlMud       = new JPanel();
    JPanel pnlMud2      = new JPanel();
    JPanel pnl2a        = new JPanel();

    JPanel pnlMudMore   = new JPanel();
    JPanel pnlMudMore2  = new JPanel();
    JPanel pnlMudLess   = new JPanel();

    JLabel lblMud       = new JLabel();
    JLabel lblMud2      = new JLabel();
    JLabel lblMud3      = new JLabel();
    JLabel lblMud4      = new JLabel();

    TitledBorder titledBorderDunham = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Dunham Carbonate Scale");
    titledBorderDunham.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    COLOR = lith.lithConstants.LIMESTONE_COLOR;

    // Build Panel

    panel.setLayout(new BorderLayout());
    panel.setBackground(new Color( 255, 255, 255 ));
    panel.setBorder(titledBorderDunham);

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBackground(new Color( 255, 255, 255 ));

    // -- Components bound together during deposition

    pnlBioherm.setLayout(new BorderLayout());
    pnlBioherm.setBackground(new Color( 255, 255, 255 ));

    lblBioherm.setFont(new java.awt.Font("Dialog", 1, 11));
    lblBioherm.setText( "Components bound together during deposition" );
    lblBioherm.setHorizontalAlignment(SwingConstants.CENTER);
    lblBioherm.setBackground(new Color( 255, 255, 255 ));

    pnlBioC.setLayout(new GridLayout());
    pnlBioC.setBackground(new Color( 255, 255, 255 ));

    pnlBioC1.setLayout(new GridLayout());
    pnlBioC1.setBackground(new Color( 255, 255, 255 ));

    rb[11][0].setText( "Boundstone" );
    rb[11][0].setBackground(new Color( COLOR[11][0], COLOR[11][1], COLOR[11][2]));

    pnlBioC1.add(rb[11][0], null);

    pnlBioC2.setLayout(new GridLayout(3,1));
    pnlBioC2.setBackground(new Color( 255, 255, 255 ));

    rb[11][1].setText( "Bafflestone" );
    rb[11][1].setBackground(new Color( COLOR[11][0], COLOR[11][1], COLOR[11][2]));

    rb[10][1].setText( "Bindstone" );
    rb[10][1].setBackground(new Color( COLOR[10][0], COLOR[10][1], COLOR[10][2]));

    rb[9][1].setText( "Framestone" );
    rb[9][1].setBackground(new Color( COLOR[9][0], COLOR[9][1], COLOR[9][2]));

    pnlBioC2.add(rb[11][1], null);
    pnlBioC2.add(rb[10][1], null);
    pnlBioC2.add(rb[9][1],  null);

    // -- Panel 1

    pnl1.setLayout(new BorderLayout());
    pnl1.setBackground(new Color( 255, 255, 255 ));

    // .. Grain Supported

    pnlGrain.setLayout(new BorderLayout());
    pnlGrain.setBackground(new Color( 255, 255, 255 ));

    lblGrain.setFont(new java.awt.Font("Dialog", 1, 11));
    lblGrain.setText( "Grain Supported" );
    lblGrain.setHorizontalAlignment(SwingConstants.CENTER);
    lblGrain.setBackground(new Color( 255, 255, 255 ));

    pnlGrain2.setLayout(new BorderLayout());
    pnlGrain2.setBackground(new Color( 255, 255, 255 ));

    lblGrain2.setFont(new java.awt.Font("Dialog", 0, 11));
    lblGrain2.setText( "Grains > 2 mm" );
    lblGrain2.setHorizontalAlignment(SwingConstants.RIGHT);
    lblGrain2.setBackground(new Color( 255, 255, 255 ));

    pnlGrain2a.setLayout(new GridLayout());
    pnlGrain2a.setBackground(new Color( 255, 255, 255 ));

    pnlGrain21.setLayout(new GridLayout(2,1));
    pnlGrain21.setBackground(new Color( 255, 255, 255 ));

    rb[8][0].setText( "Grainstone" );
    rb[8][0].setBackground(new Color( COLOR[8][0], COLOR[8][1], COLOR[8][2]));

    rb[6][0].setText( "Packstone" );
    rb[6][0].setBackground(new Color( COLOR[6][0], COLOR[6][1], COLOR[6][2]));

    pnlGrain21.add(rb[8][0], null);
    pnlGrain21.add(rb[6][0], null);

    pnlGrain22.setLayout(new GridLayout(2,1));
    pnlGrain22.setBackground(new Color( 255, 255, 255 ));

    rb[8][1].setText( "Rudstone" );
    rb[8][1].setBackground(new Color( COLOR[8][0], COLOR[8][1], COLOR[8][2]));

    rb[6][1].setText( "Rudstone" );
    rb[6][1].setBackground(new Color( COLOR[6][0], COLOR[6][1], COLOR[6][2]));

    pnlGrain22.add(rb[8][1], null);
    pnlGrain22.add(rb[6][1], null);

    // -- Panel 2

    pnl2.setLayout(new BorderLayout());
    pnl2.setBackground(new Color( 255, 255, 255 ));

    // .. Mud Supported

    pnlMud.setLayout(new BorderLayout());
    pnlMud.setBackground(new Color( 255, 255, 255 ));

    pnlMud2.setLayout(new GridLayout(2,1));
    pnlMud2.setBackground(new Color( 255, 255, 255 ));

    lblMud.setFont(new java.awt.Font("Dialog", 1, 11));
    lblMud.setText( "Mud Supported" );
    lblMud.setHorizontalAlignment(SwingConstants.CENTER);
    lblMud.setBackground(new Color( 255, 255, 255 ));

    lblMud2.setFont(new java.awt.Font("Dialog", 0, 11));
    lblMud2.setText( "More than 10% Grains" );
    lblMud2.setHorizontalAlignment(SwingConstants.CENTER);
    lblMud2.setBackground(new Color( 255, 255, 255 ));

    // ..  Panel 2a

    pnl2a.setLayout(new BorderLayout());
    pnl2a.setBackground(new Color( 255, 255, 255 ));

    // .. .. Wackestone - Floatstone

    pnlMudMore.setLayout(new BorderLayout());
    pnlMudMore.setBackground(new Color( 255, 255, 255 ));

    lblMud3.setFont(new java.awt.Font("Dialog", 0, 11));
    lblMud3.setText( "Grains > 2 mm" );
    lblMud3.setHorizontalAlignment(SwingConstants.RIGHT);
    lblMud3.setBackground(new Color( 255, 255, 255 ));

    pnlMudMore2.setLayout(new GridLayout());
    pnlMudMore2.setBackground(new Color( 255, 255, 255 ));

    rb[4][0].setText( "Wackestone" );
    rb[4][0].setBackground(new Color( COLOR[4][0], COLOR[4][1], COLOR[4][2]));

    rb[4][1].setText( "Floatstone" );
    rb[4][1].setBackground(new Color( COLOR[4][0], COLOR[4][1], COLOR[4][2]));

    pnlMudMore2.add(rb[4][0], null);
    pnlMudMore2.add(rb[4][1], null);

    pnlMudLess.setLayout(new GridLayout(2,1));
    pnlMudLess.setBackground(new Color( 255, 255, 255 ));

    lblMud4.setFont(new java.awt.Font("Dialog", 0, 11));
    lblMud4.setText( "Less than 10% Grains" );
    lblMud4.setHorizontalAlignment(SwingConstants.CENTER);
    lblMud4.setBackground(new Color( 255, 255, 255 ));

    rb[2][0].setText( "Mudstone" );
    rb[2][0].setBackground(new Color( COLOR[2][0], COLOR[2][1], COLOR[2][2]));

    pnlMudLess.add(lblMud4,   null);
    pnlMudLess.add(rb[2][0],    null);

    // Attach Widgets to Panel

    panel.add(pnlCenter,        BorderLayout.CENTER);
    pnlCenter.add(pnlBioherm,   BorderLayout.NORTH);
    pnlBioherm.add(lblBioherm,  BorderLayout.NORTH);
    pnlBioherm.add(pnlBioC,     BorderLayout.CENTER);
    pnlBioC.add(pnlBioC1,       null);
    pnlBioC.add(pnlBioC2,       null);

    pnlCenter.add(pnl1,         BorderLayout.CENTER);
    pnl1.add(pnlGrain,          BorderLayout.NORTH);
    pnlGrain.add(lblGrain,      BorderLayout.NORTH);

    pnlGrain.add(pnlGrain2,     BorderLayout.CENTER);
    pnlGrain2.add(lblGrain2,    BorderLayout.NORTH);

    pnlGrain2.add(pnlGrain2a,   BorderLayout.CENTER);
    pnlGrain2a.add(pnlGrain21,   null);
    pnlGrain2a.add(pnlGrain22,   null);

    pnl1.add(pnl2,              BorderLayout.CENTER);
    pnl2.add(pnlMud,            BorderLayout.NORTH);
    pnlMud.add(pnlMud2,         BorderLayout.NORTH);
    pnlMud2.add(lblMud,         null);
    pnlMud2.add(lblMud2,        null);

    pnl2.add(pnl2a,             BorderLayout.CENTER);
    pnl2a.add(pnlMudMore,       BorderLayout.NORTH);
    pnlMudMore.add(lblMud3,     BorderLayout.NORTH);

    pnlMudMore.add(pnlMudMore2, BorderLayout.CENTER);
    pnlMudMore.add(pnlMudLess,  BorderLayout.SOUTH);

    return (panel);
  }

  /** Method buildALLOCHEM()
   * <p> This method will build Folk's Allochems Carbonate Scale
   */

  private JPanel buildALLOCHEM()
  {
	int    COLOR[][]    = null;
    JPanel panel        = new JPanel();
    JPanel pnlTop       = new JPanel();

    JPanel pnlCenter    = new JPanel();

    JLabel lblBioherm   = new JLabel();

    JPanel pnlBioherm   = new JPanel();
    JPanel pnlBioC      = new JPanel();
    JPanel pnlBioC1     = new JPanel();
    JPanel pnlBioC2     = new JPanel();

    JPanel pnl1         = new JPanel();

    JPanel pnlSparite   = new JPanel();
    JPanel pnlSpariteB  = new JPanel();

    JLabel lblSparite   = new JLabel();

    JPanel pnl2         = new JPanel();

    JPanel pnlEqual     = new JPanel();
    JPanel pnlEqualB    = new JPanel();

    JLabel lblEqual     = new JLabel();

    JPanel pnl3         = new JPanel();

    JPanel pnlMicrite   = new JPanel();
    JPanel pnlMicriteB  = new JPanel();

    JLabel lblMicrite   = new JLabel();


    TitledBorder titledBorderDunham = BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Dunham Carbonate Scale");
    titledBorderDunham.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    COLOR = lith.lithConstants.LIMESTONE_COLOR;

    // Build Panel

    panel.setLayout(new BorderLayout());
    panel.setBackground(new Color( 255, 255, 255 ));
    panel.setBorder(titledBorderDunham);

    // Top Panel

    pnlTop.setLayout(new GridLayout(2,2));
    pnlTop.setBackground(new Color( 255, 255, 255 ));

    pnlTop.add(rbIntra, null);
    pnlTop.add(rbPel,   null);
    pnlTop.add(rbBio,   null);
    pnlTop.add(rbOO,    null);

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBackground(new Color( 255, 255, 255 ));

    // -- Components bound together during deposition

    pnlBioherm.setLayout(new BorderLayout());
    pnlBioherm.setBackground(new Color( 255, 255, 255 ));

    lblBioherm.setFont(new java.awt.Font("Dialog", 1, 11));
    lblBioherm.setText( "Undisturbed Bioherm Rocks" );
    lblBioherm.setHorizontalAlignment(SwingConstants.CENTER);
    lblBioherm.setBackground(new Color( 255, 255, 255 ));

    pnlBioC.setLayout(new GridLayout());
    pnlBioC.setBackground(new Color( 255, 255, 255 ));

//    pnlBioC1.setLayout(new GridLayout());
//    pnlBioC1.setBackground(new Color( 255, 255, 255 ));

//    rb[11][0].setText( "Boundstone" );
//    rb[11][0].setBackground(new Color( COLOR[11][0], COLOR[11][1], COLOR[11][2]));

//    pnlBioC1.add(rb[11][0], null);

    pnlBioC2.setLayout(new GridLayout(3,1));
    pnlBioC2.setBackground(new Color( 255, 255, 255 ));

    rb[11][2].setText( "Bafflestone" );
    rb[11][2].setBackground(new Color( COLOR[11][0], COLOR[11][1], COLOR[11][2]));

    rb[10][2].setText( "Bindstone" );
    rb[10][2].setBackground(new Color( COLOR[10][0], COLOR[10][1], COLOR[10][2]));

    rb[9][2].setText( "Framestone" );
    rb[9][2].setBackground(new Color( COLOR[9][0], COLOR[9][1], COLOR[9][2]));

    pnlBioC2.add(rb[11][2], null);
    pnlBioC2.add(rb[10][2], null);
    pnlBioC2.add(rb[9][2],  null);

    // Panel 1

    pnl1.setLayout(new BorderLayout());
    pnl1.setBackground(new Color( 255, 255, 255 ));

    // .. Over 2/3 Sparite Cement

    pnlSparite.setLayout(new BorderLayout());
    pnlSparite.setBackground(new Color( 255, 255, 255 ));

    lblSparite.setFont(new java.awt.Font("Dialog", 1, 11));
    lblSparite.setText( "Over 2/3 Sparite Cement" );
    lblSparite.setHorizontalAlignment(SwingConstants.CENTER);
    lblSparite.setBackground(new Color( 255, 255, 255 ));

    pnlSpariteB.setLayout(new GridLayout(3,1));
    pnlSpariteB.setBackground(new Color( 255, 255, 255 ));

    rb[8][2].setBackground(new Color( COLOR[8][0], COLOR[8][1], COLOR[8][2]));
    rb[7][2].setBackground(new Color( COLOR[7][0], COLOR[7][1], COLOR[7][2]));
    rb[6][2].setBackground(new Color( COLOR[6][0], COLOR[6][1], COLOR[6][2]));

    pnlSpariteB.add(rb[8][2], null);
    pnlSpariteB.add(rb[7][2], null);
    pnlSpariteB.add(rb[6][2], null);

    // Panel 2

    pnl2.setLayout(new BorderLayout());
    pnl2.setBackground(new Color( 255, 255, 255 ));

    // .. Subequal Sparite & Micrite Cement

    pnlEqual.setLayout(new BorderLayout());
    pnlEqual.setBackground(new Color( 255, 255, 255 ));

    lblEqual.setFont(new java.awt.Font("Dialog", 1, 11));
    lblEqual.setText( "Subequal Sparite & Micrite Cement" );
    lblEqual.setHorizontalAlignment(SwingConstants.CENTER);
    lblEqual.setBackground(new Color( 255, 255, 255 ));

    pnlEqualB.setLayout(new GridLayout());
    pnlEqualB.setBackground(new Color( 255, 255, 255 ));

    rb[5][2].setBackground(new Color( COLOR[5][0], COLOR[5][1], COLOR[5][2]));

    pnlEqualB.add(rb[5][2], null);

    // Panel 3

    pnl3.setLayout(new BorderLayout());
    pnl3.setBackground(new Color( 255, 255, 255 ));

    // .. Subequal Sparite & Micrite Cement

    pnlMicrite.setLayout(new BorderLayout());
    pnlMicrite.setBackground(new Color( 255, 255, 255 ));

    lblMicrite.setFont(new java.awt.Font("Dialog", 1, 11));
    lblMicrite.setText( "Over 2/3 Micrite Cement" );
    lblMicrite.setHorizontalAlignment(SwingConstants.CENTER);
    lblMicrite.setBackground(new Color( 255, 255, 255 ));

    pnlMicriteB.setLayout(new GridLayout(5,1));
    pnlMicriteB.setBackground(new Color( 255, 255, 255 ));

    rb[4][2].setBackground(new Color( COLOR[4][0], COLOR[4][1], COLOR[4][2]));
    rb[3][2].setBackground(new Color( COLOR[3][0], COLOR[3][1], COLOR[3][2]));
    rb[2][2].setBackground(new Color( COLOR[2][0], COLOR[2][1], COLOR[2][2]));
    rb[1][2].setBackground(new Color( COLOR[1][0], COLOR[1][1], COLOR[1][2]));
    rb[0][2].setBackground(new Color( COLOR[0][0], COLOR[0][1], COLOR[0][2]));

    pnlMicriteB.add(rb[4][2], null);
    pnlMicriteB.add(rb[3][2], null);
    pnlMicriteB.add(rb[2][2], null);
    pnlMicriteB.add(rb[1][2], null);
    pnlMicriteB.add(rb[0][2], null);

    // Set Allochems Texture Names

    setAllochems( iAllochem );

    // Attach Widgets to Panel

    panel.add(pnlTop,           BorderLayout.NORTH);

    panel.add(pnlCenter,        BorderLayout.CENTER);
    pnlCenter.add(pnlBioherm,   BorderLayout.NORTH);
    pnlBioherm.add(lblBioherm,  BorderLayout.NORTH);
    pnlBioherm.add(pnlBioC,     BorderLayout.CENTER);
//    pnlBioC.add(pnlBioC1,       null);
    pnlBioC.add(pnlBioC2,       null);

    pnlCenter.add(pnl1,         BorderLayout.CENTER);
    pnl1.add(pnlSparite,        BorderLayout.NORTH);
    pnlSparite.add(lblSparite,  BorderLayout.NORTH);
    pnlSparite.add(pnlSpariteB, BorderLayout.CENTER);

    pnl1.add(pnl2,              BorderLayout.CENTER);
    pnl2.add(pnlEqual,          BorderLayout.NORTH);
    pnlEqual.add(lblEqual,      BorderLayout.NORTH);
    pnlEqual.add(pnlEqualB,     BorderLayout.CENTER);

    pnl2.add(pnl3,              BorderLayout.CENTER);
    pnl3.add(pnlMicrite,        BorderLayout.NORTH);
    pnlMicrite.add(lblMicrite,  BorderLayout.NORTH);
    pnlMicrite.add(pnlMicriteB, BorderLayout.CENTER);

    return (panel);
  }

  /** Method buildREPLACE()
   * <p> This method will build Replacement Crystalline Carbonate Scale
   */

  private JPanel buildREPLACE()
  {
	int    COLOR[][]    = null;
    JPanel panel        = new JPanel();

    COLOR = lith.lithConstants.XLN_COLOR;

    panel.setLayout(new BorderLayout());
    panel.setBackground(new Color( 255, 255, 255 ));

    JPanel pnlCenter    = new JPanel();

    pnlCenter.setLayout(new GridLayout(7,1));
    pnlCenter.setBackground(new Color( 255, 255, 255 ));

    rb[9][3].setText("Very Coarse Crystalline");
    rb[9][3].setBackground(new Color( COLOR[9][0], COLOR[9][1], COLOR[9][2]));

    rb[7][3].setText("Coarse Crystalline");
    rb[7][3].setBackground(new Color( COLOR[7][0], COLOR[7][1], COLOR[7][2]));

    rb[5][3].setText("Medium Crystalline");
    rb[5][3].setBackground(new Color( COLOR[5][0], COLOR[5][1], COLOR[5][2]));

    rb[3][3].setText("Fine Crystalline");
    rb[3][3].setBackground(new Color( COLOR[3][0], COLOR[3][1], COLOR[3][2]));

    rb[2][3].setText("Very Fine Crystalline");
    rb[2][3].setBackground(new Color( COLOR[2][0], COLOR[2][1], COLOR[2][2]));

    rb[1][3].setText("Micro Crystalline");
    rb[1][3].setBackground(new Color( COLOR[1][0], COLOR[1][1], COLOR[1][2]));

    rb[0][3].setText("Crypto Crystalline");
    rb[0][3].setBackground(new Color( COLOR[0][0], COLOR[0][1], COLOR[0][2]));

    pnlCenter.add(rb[9][3], null);
    pnlCenter.add(rb[7][3], null);
    pnlCenter.add(rb[5][3], null);
    pnlCenter.add(rb[3][3], null);
    pnlCenter.add(rb[2][3], null);
    pnlCenter.add(rb[1][3], null);
    pnlCenter.add(rb[0][3], null);

    // Attach Widgets to Panel

    panel.add(pnlCenter,        BorderLayout.CENTER);

    return (panel);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLithology()
   *  <p> This method will retrieve the Primary Lithology selected
   *  @return iRock = The Primary Lithology
   */

  public int getLithology() { return (iRock); }

  /** Method getTexture()
   *  <p> This method will retrieve the Texture Level selected
   *  @return iTexture = The Texture Level Selected
   */

  public int getTexture() { return (iTexture); }

  /** Method getTextureDesc()
   *  <p> This method will retrieve the Texture Level Description selected
   *  @return sTexture = The Texture Level Description Selected
   */

  public String getTextureDesc() { return (sTexture); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setPanel()
   *  <p> This method will set the panel and enable & disable buttons
   *  @param iPanel = The panel that is set
   */

  private void setPanel(int iPanel)
  {
	String sTitle = "";

	this. iPanel = iPanel;

    switch (iPanel)
    {
      case _DUNHAM:
        rb[6][0].setSelected(true);

        cardLayout.show(pnlData, "DUNHAM");
        sTitle = new String("Dunham Carbonate Scale");
        break;
      case _ALLOCHEM:
        rb[5][2].setSelected(true);

        cardLayout.show(pnlData, "ALLOCHEM");
        sTitle = new String("Folk's Allochems Carbonate Scale");
        break;
      case _REPLACE:
        rb[5][3].setSelected(true);

        cardLayout.show(pnlData, "REPLACE");
        sTitle = new String("Carbonate Crystalline Scale");
        break;
	}

//    titledBorderMain.setTitle( sTitle );
    this.updateUI();
  }

  /** Method setButtons()
   *  <p> This method will enable or disable the buttons
   */

  private void setButtons()
  {
    rbIntra.setEnabled(false);
    rbPel.setEnabled(false);
    rbBio.setEnabled(false);
    rbOO.setEnabled(false);

    for (int i=0; i<14; i++)
    {
      for (int j=0; j<4; j++)
      {
	    rb[i][j].setEnabled(false);
      }
	}

	if ((rbP[lith.lithConstants._LIMESTONE].isSelected()) ||
	    (rbP[lith.lithConstants._DOLOMITE].isSelected()))
	{
      rbIntra.setEnabled(true);
      rbPel.setEnabled(true);
      rbBio.setEnabled(true);
      rbOO.setEnabled(true);

      for (int i=0; i<14; i++)
      {
        for (int j=0; j<4; j++)
        {
	      rb[i][j].setEnabled(true);
        }
	  }
	}
  }

  /** Method setAllochems()
   *  <p> This method will set the Allochems Texture Name
   *  @param iAllochem = Selected Allochem
   */

  private void setAllochems( int iAllochem )
  {
    this.iAllochem = iAllochem;

    for (int i=0; i<9; i++)
    {
	  rb[i][2].setText( FOLK_NAME[i][iAllochem] );
	  if (rb[i][2].isSelected())
	  {
   	    this.iTexture = i+1;
        this.sTexture = new String( rb[i][2].getText() );
	  }
	}
  }

  /** Method setTexture()
   *  <p> This method will set the Texture identifier
   */

  private void setTexture()
  {
	int i,j;

	switch (iPanel)
	{
      case _DUNHAM:
        for (i=0; i<14; i++)
        {
		  for (j=0; j<2; j++)
		  {
		    if (rb[i][j].isSelected())
		    {
   	          this.iTexture = i+1;
              this.sTexture = new String( rb[i][j].getText() );
			}
		  }
	    }
        break;
      case _ALLOCHEM:
        for (i=0; i<14; i++)
        {
          if (rb[i][2].isSelected())
   	      {
   	        this.iTexture = i+1;
            this.sTexture = new String( rb[i][2].getText() );
		  }
	    }
        break;
      case _REPLACE:
        for (i=0; i<14; i++)
        {
          if (rb[i][3].isSelected())
   	      {
   	        this.iTexture = i+1;
            this.sTexture = new String( rb[i][3].getText() );
		  }
	    }
        break;
	}
  }

  /** Method setTexture()
   *  <p> This method will set the Texture identifier
   *  @param iTexture = is the selected Texture button
   *  @param iLevel   = By Panel and Texture
   *

  private void setTexture( int iTexture, int iLevel )
  {
	this.iTexture = iTexture+1;
    this.sTexture = new String( rb[iTexture][iLevel].getText() );
  }
*/
  /* ===================================================================== *
   * ----------------------------- CLOSE --------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
	int i,j;

    // Input Variables

    notifier = null;     // Panel Observable

    // Global Variables

    sTexture = null;      // Texture Level Description

    // Global Widgets

    // -- Display Descriptive Data Panel

    pnlData          = null;
    cardLayout       = null;
    titledBorderMain = null;

    // -- Textures Methods

    rbDunham    = null;
    rbAllochem  = null;
    rbReplace   = null;

    // -- Folk's Allochems

    rbIntra     = null;
    rbPel       = null;
    rbBio       = null;
    rbOO        = null;

    // -- Lithology & Texture Widgets

    for (i=0; i<14; i++)
    {
      for (j=0; j<4; j++)
      {
	    rb[i][j] = null;
      }
	}

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  rbP[i] = null;
    }

    rb  = null;  // Sediment Size A
    rbP = null;  // Primary Lithologies
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i,j;

    if (event.getSource() == rbDunham)   { setPanel( _DUNHAM ); }
    if (event.getSource() == rbAllochem) { setPanel( _ALLOCHEM ); }
    if (event.getSource() == rbReplace)  { setPanel( _REPLACE ); }

    if (event.getSource() == rbIntra) { setAllochems( _INTRACLASTS ); }
    if (event.getSource() == rbPel)   { setAllochems( _PELLETS ); }
    if (event.getSource() == rbBio)   { setAllochems( _FOSSILS ); }
    if (event.getSource() == rbOO)    { setAllochems( _OOLITES ); }

    setButtons();

    if (iRock > -1) { setTexture(); }

    for (i=0; i<lith.lithConstants._PRIMARY; i++)
    {
	  if (event.getSource() == rbP[i])
	  {
	    iRock = i;
	  }
	}

    if (notifier != null)
      notifier.notifyObservers(new String("Primary Carbonates Selected"));
  }
}

/*
 *  @version 1.1 04/26/2011
 *  @author  John Victorine
 */
