/*
 * @las3Utility.java Version 1.1 01/15/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3Utility
 *  <p> This Class will provide basic utilities for the LAS 3 Data Structure
 *
 *  @version 1.1 01/15/2010
 *  @author  John R. Victorine
 */

public class las3Utility
{
  public static final int _NONE       = las3.las3Constants._NONE;

  // LAS File Format or Data Type Identifier

  public static final int _INTEGER    = las3.las3Struct._INTEGER;
  public static final int _FLOAT      = las3.las3Struct._FLOAT;
  public static final int _DOUBLE     = las3.las3Struct._DOUBLE;
  public static final int _SCIENTIFIC = las3.las3Struct._SCIENTIFIC;
  public static final int _STRING     = las3.las3Struct._STRING;
  public static final int _CHARACTER  = las3.las3Struct._CHARACTER;
  public static final int _ARRAY      = las3.las3Struct._ARRAY;

  public static final String FORMAT[] = las3.las3Struct.FORMAT;

  /** Method isData()
   * <p> This method will determine if the type and array exists in the LAS
   *     Data List Structure.
   * @param  iType  = the Type of Section, "Well", "Log", "Core", etc
   * @param  iArray = the group number, i.e. 1, 2 that represents Core[1], Core[2], etc.
   * @param  st     = the LAS 3 Data List Structure
   * @return bValue = true if the type and array exists and false if it does not
   */

  public static boolean isData( int iType, int iArray, las3ListStruct st )
  {
    boolean bValue = false;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if ((iType == st.stItem[i].iType) && ( iArray == st.stItem[i].iArray))
        {
          bValue = true;
        }
      }
    }

    return (bValue);
  }

  /** Method getData()
   * <p> This method will retrieve the LAS 3 Data Structure depending on the
   *     Data Type and the array.
   * @param  iType  = the Type of Section, "Well", "Log", "Core", etc
   * @param  iArray = the group number, i.e. 1, 2 that represents Core[1], Core[2], etc.
   * @param  st     = the LAS 3 Data List Structure
   * @return stLAS  = the LAS 3 Data Structure
   */

  public static las3Struct getData( int iType, int iArray, las3ListStruct st )
  {
    las3Struct stLAS  = null;
    int        iFound = -1;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if ((iType == st.stItem[i].iType) && ( iArray == st.stItem[i].iArray))
        {
          iFound = i;
        }
      }

      if (iFound == -1)
      {
        stLAS        = new las3Struct();
        stLAS.iType  = iType;
        stLAS.iArray = iArray;
        stLAS.sType  = new String( las3.las3Constants.SECTIONS[iType] );
      }
      else
      {
        stLAS = copy( st.stItem[iFound] );
      }
    }

    return (stLAS);
  }

  /** Method getSectionID()
   * <p> This method will retrieve the Section ID Type Number.
   * @param  str  = the Section Name
   * @return iVal = Section ID
   */

  public static int getSectionID(String str)
  {
    int iVal = -1;

    for (int i=0; i<las3.las3Constants.SECTIONS.length; i++)
    {
      if (str.equals(las3.las3Constants.SECTIONS[i]))
      {
        iVal = i;
      }
    }

    return (iVal);
  }

  /** Method getGroupID()
   * <p> This method will retrieve the Group ID within the Section Type.
   * @param  str  = the Group Name
   * @return iVal = Group ID
   */

  public static int getGroupID(String str)
  {
    int iVal = -1;

    for (int i=0; i<las3.las3Constants.SECTION.length; i++)
    {
      if (str.equals(las3.las3Constants.SECTION[i]))
      {
        iVal = i;
      }
    }

    return (iVal);
  }

  /** Method add()
   * <p> This method will add the LAS File Data to an existing list
   * @param  stBlock = The LAS File Data Block List Data Structure
   * @param  st      = The Old LAS File Data List Data Structure
   * @return st      = The new LAS File Data List.
   */

  public static las3ListStruct add( las3Struct stBlock, las3ListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    double dVersion = 0.0;  // Version Number of LAS File
    String sWrap    = "No"; // Multiple lines per row
    String sDelim   = " ";  // Delimiting Character between data
    las3ListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new las3ListStruct();
    stTemp.stItem = new las3Struct[iRecords];

    if (st != null)
    {
      dVersion = st.dVersion;
      sWrap    = new String( st.sWrap );
      sDelim   = new String( st.sDelim );

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy( st.stItem[i] );
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new las3Struct();
    stTemp.stItem[iCount] = copy( stBlock );
    iCount++;

    stTemp.iCount = iCount;

    st          = new las3ListStruct();
    st.dVersion = dVersion;
    st.sWrap    = new String( sWrap );
    st.sDelim   = new String( sDelim );
    st.stItem   = new las3Struct[iCount];
    st.iCount   = iCount;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy( stTemp.stItem[i] );
    }

    stTemp.delete();

    return (st);
  }

  /** Method removeIQSTRAT()
   * <p> This method will remove any Profile LAS 3 File Data from the LAS 3
   *     File Data List Structure
   * @param  st      = The Old LAS File Data List Data Structure
   * @return st      = The new LAS File Data List.
   */

  public static las3ListStruct removeIQSTRAT( las3ListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    double dVersion = 0.0;  // Version Number of LAS File
    String sWrap    = "No"; // Multiple lines per row
    String sDelim   = " ";  // Delimiting Character between data
    las3ListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount;
    else
      iRecords = 1;

    stTemp        = new las3ListStruct();
    stTemp.stItem = new las3Struct[iRecords];

    if (st != null)
    {
      dVersion = st.dVersion;
      sWrap    = new String( st.sWrap );
      sDelim   = new String( st.sDelim );

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i].iqstrat == -1)
          {
            if (iCount < iRecords)
            {
              stTemp.stItem[iCount] = copy( st.stItem[i] );
              iCount++;
            }
          }
        }
      }

      st.delete();
    }

    stTemp.iCount = iCount;

    st          = new las3ListStruct();
    st.dVersion = dVersion;
    st.sWrap    = new String( sWrap );
    st.sDelim   = new String( sDelim );
    st.stItem   = new las3Struct[iCount];
    st.iCount   = iCount;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy( stTemp.stItem[i] );
    }

    stTemp.delete();

    return (st);
  }

  /** Method remove()
   * <p> This method will remove any Profile LAS 3 File Data from the LAS 3
   *     File Data List Structure
   * @param  st      = The Old LAS File Data List Data Structure
   * @param bLog     = Indicator to identify which Data Sections to
   *                   make available ( required data sections )
   * @return st      = The new LAS File Data List.
   */

  public static las3ListStruct remove( las3ListStruct st, boolean bLog[] )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    double dVersion = 0.0;  // Version Number of LAS File
    String sWrap    = "No"; // Multiple lines per row
    String sDelim   = " ";  // Delimiting Character between data
    las3ListStruct stTemp = null;
    int    iCont    = 0;

    if (st != null)
      iRecords = st.iCount;
    else
      iRecords = 1;

    stTemp        = new las3ListStruct();
    stTemp.stItem = new las3Struct[iRecords];

    if (st != null)
    {
      dVersion = st.dVersion;
      sWrap    = new String( st.sWrap );
      sDelim   = new String( st.sDelim );

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
//System.out.println("???ADD "+i+" "+st.stItem[i].iType+" "+bLog[st.stItem[i].iType]);
		  iCont = 0;
          if ((st.stItem[i].iqstrat == -1) &&
              (st.stItem[i].iType == las3.las3Constants._CORE) &&
              (!bLog[st.stItem[i].iType]))
          {
            iCont = 1;
		  }
          else if ((st.stItem[i].iqstrat == -1) &&
                   (st.stItem[i].iType == las3.las3Constants._TEST) &&
                   (!bLog[st.stItem[i].iType]))
          {
            iCont = 1;
		  }
          else if ((st.stItem[i].iType == las3.las3Constants._PERFORATION) &&
                   (!bLog[st.stItem[i].iType]))
          {
            iCont = 1;
		  }
          else if ((st.stItem[i].iqstrat == -1) &&
                   (st.stItem[i].iType != las3.las3Constants._CORE) &&
                   (st.stItem[i].iType != las3.las3Constants._TEST) &&
                   (st.stItem[i].iType != las3.las3Constants._PERFORATION))
          {
            iCont = 1;
		  }
          else if ((st.stItem[i].iType > las3.las3Constants._IQ_CORE) &&
                   (!bLog[st.stItem[i].iType]))
          {
            iCont = 1;
		  }

          if (iCont == 1)
          {
            if (iCount < iRecords)
            {
//System.out.println("ADD "+i+" "+st.stItem[i].iType+" "+bLog[st.stItem[i].iType]);
              stTemp.stItem[iCount] = copy( st.stItem[i] );
              iCount++;
            }
          }
        }
      }

      st.delete();
    }

    stTemp.iCount = iCount;

    st          = new las3ListStruct();
    st.dVersion = dVersion;
    st.sWrap    = new String( sWrap );
    st.sDelim   = new String( sDelim );
    st.stItem   = new las3Struct[iCount];
    st.iCount   = iCount;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy( stTemp.stItem[i] );
    }

    stTemp.delete();

    return (st);
  }

  /** Method addParameters()
   * <p> This method will add the Parameters line to the LAS 3 Data Structure
   * @param  strs = array of strings holding the LAS Data Line
   * @param  st   = LAS 3 Data Structure
   * @return st   = LAS 3 Data Structure
   */

  public static las3Struct addParameters( String strs[], las3Struct st )
  {
    int i,j;
    int    iCount  = 0;
    String str[][] = null;

    if ((strs != null) && (st != null))
    {
      if (st.sParams == null)
      {
        str = new String[1][st.iParamCols];
      }
      else
      {
        str = new String[st.iParamRows+1][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            str[iCount][j] = new String( st.sParams[i][j] );
          }

          iCount++;
        }
      }

      st.sParams = null;

      for (j=0; j<st.iParamCols; j++)
      {
        str[iCount][j] = new String( strs[j] );
      }
      iCount++;

      st.sParams    = new String[iCount][st.iParamCols];
      st.iParamRows = iCount;

      for (i=0; i<st.iParamRows; i++)
      {
        for (j=0; j<st.iParamCols; j++)
        {
          st.sParams[i][j] = new String( str[i][j] );
        }
      }

      str = null;
    }

    return (st);
  }

  /** Method addCurves()
   * <p> This method will add the Curve line to the LAS 3 Data Structure
   * @param  strs = array of strings holding the LAS Data Line
   * @param  st   = LAS 3 Data Structure
   * @return st   = LAS 3 Data Structure
   */

  public static las3Struct addCurves( String strs[], las3Struct st )
  {
    int i,j;
    int    iCount   = 0;
    String str[][]  = null;
    int    iCurve[] = null;   // Array holding the Curve Format ID
    String sCurve[] = null;   // Array holding the Format of Data
    String sTemp    = "";

    if ((strs != null) && (st != null))
    {
      if (st.sCurves == null)
      {
        str    = new String[1][st.iCurveCols];
        iCurve = new int[1];
        sCurve = new String[1];
      }
      else
      {
        str    = new String[st.iCurveRows+1][st.iCurveCols];
        iCurve = new int[st.iCurveRows+1];
        sCurve = new String[st.iCurveRows+1];

        for (i=0; i<st.iCurveRows; i++)
        {
          iCurve[iCount] = st.iCurve[i];
          sCurve[iCount] = new String( st.sCurve[i] );

          for (j=0; j<st.iCurveCols; j++)
          {
            str[iCount][j] = new String( st.sCurves[i][j] );
          }

          iCount++;
        }
      }

      st.iCurve  = null;
      st.sCurve  = null;
      st.sCurves = null;

      iCurve[iCount] = las3.las3Struct._DOUBLE;
      sCurve[iCount] = new String( strs[las3.las3Struct._FORMAT] );

      if (strs[las3.las3Struct._FORMAT].length() > 0)
      {
        sTemp = new String( strs[las3.las3Struct._FORMAT].substring(0,1) );
        for (j=0; j<las3.las3Struct.FORMAT.length; j++)
        {
          if (sTemp.equals(las3.las3Struct.FORMAT[j]))
          {
            iCurve[iCount] = j;
            sCurve[iCount] = new String( strs[las3.las3Struct._FORMAT] );
          }
        }
      }

      for (j=0; j<st.iCurveCols; j++)
      {
        str[iCount][j] = new String( strs[j] );
      }
      iCount++;

      st.sCurves    = new String[iCount][st.iCurveCols];
      st.select     = new String[iCount][st.iCurveCols];
      st.iCurve     = new int[iCount];
      st.sCurve     = new String[iCount];
      st.iCurveRows = iCount;

      for (i=0; i<st.iCurveRows; i++)
      {
        st.iCurve[i] = iCurve[i];
        st.sCurve[i] = new String( sCurve[i] );

        for (j=0; j<st.iCurveCols; j++)
        {
          st.sCurves[i][j] = new String( str[i][j] );
          st.select[i][j]  = new String( "" );
        }
      }

      iCurve = null;
      sCurve = null;
      str    = null;
    }

    return (st);
  }

  /** Method addIQCurves()
   * <p> This method will add the Selected Curves from the Profile Plot App to
   *     the LAS 3 Data Structure
   * @param  strs = array of strings holding the LAS Data Line
   * @param  st   = LAS 3 Data Structure
   * @return st   = LAS 3 Data Structure
   */

  public static las3Struct addIQCurves( String strs[], las3Struct st )
  {
    int i,j;
    int    iCount   = 0;
    String str[][]  = null;
    String sTemp    = "";

    if ((strs != null) && (st != null))
    {
      if (st.select == null)
      {
        str = new String[1][st.iCurveCols];
      }
      else
      {
        str = new String[st.iSelectRows+1][st.iCurveCols];

        for (i=0; i<st.iSelectRows; i++)
        {
          for (j=0; j<st.iCurveCols; j++)
          {
            str[iCount][j] = new String( st.select[i][j] );
          }

          iCount++;
        }
      }

      st.select = null;

      for (j=0; j<st.iCurveCols; j++)
      {
        str[iCount][j] = new String( strs[j] );
      }
      iCount++;

      st.select      = new String[iCount][st.iCurveCols];
      st.iSelectRows = iCount;

      for (i=0; i<st.iSelectRows; i++)
      {
        for (j=0; j<st.iCurveCols; j++)
        {
          st.select[i][j] = new String( str[i][j] );
        }
      }

      str = null;
    }

    return (st);
  }

  /** Method addData()
   * <p> This method will add the Data line to the LAS 3 Data Structure
   * @param  strs = array of strings holding the LAS Data Line
   * @param  st   = LAS 3 Data Structure
   * @return st   = LAS 3 Data Structure
   */

  public static las3Struct addData( String strs[], las3Struct st )
  {
    int i,j;
    int    iCount  = 0;
    String str[][] = null;

    if ((strs != null) && (st != null))
    {
      if (st.sData == null)
      {
        str = new String[1][st.iColumns];
      }
      else
      {
        str = new String[st.iRows+1][st.iColumns];

        for (i=0; i<st.iRows; i++)
        {
          for (j=0; j<st.iColumns; j++)
          {
            str[iCount][j] = new String( st.sData[i][j] );
          }
          iCount++;
        }

        st.sData = null;
      }

      for (j=0; j<st.iColumns; j++)
      {
        str[iCount][j] = new String( "" );
        if (strs[j] !=null)
          str[iCount][j] = new String( strs[j] );
      }
      iCount++;

      st.sData = new String[iCount][st.iColumns];
      st.iRows = iCount;

      for (i=0; i<st.iRows; i++)
      {
        for (j=0; j<st.iColumns; j++)
        {
          st.sData[i][j] = new String( str[i][j] );
        }
      }

      str = null;
    }

    return (st);
  }

//  double d = Double.parseDouble("7.399999999999985E-5");
//  NumberFormat formatter = new DecimalFormat("###.#####");
//  String f = formatter.format(d);
//  System.out.println(f);// output --> 0.00007

//import java.text.*;
//import java.math.*;

//  NumberFormat formatter = new DecimalFormat();
//  double d = 0.12345;
//  formatter = new DecimalFormat("0.#####E0");
//  System.out.println(formatter.format(d)); // 1.2345E-1

//  formatter = new DecimalFormat("000000E0");
//  System.out.println(formatter.format(d)); // 12345E-6

  /** Method addAssociation()
   * <p> This method will the association data section to the section
   * @param  jSection = the Association ID to be parsed
   * @param  jGroup   = the Type of Data within the Association being modified.
   * @param  jArray   = Array Number of Association Type
   * @param  stList   = The LAS File Data List Data Structure
   * @return st_J     = The LAS File Data Structure.
   */

  public static las3Struct addCurve0( int jSection, int jGroup, int jArray,
                                      las3ListStruct stList )
  {
    int j,k;
    las3Struct st_J      = null;
    String     sOrderd[] = { "", "", "", "", "", "" };
    int        iContinue = 0;

    if (stList != null)
    {
      if (isData( jSection, jArray, stList ))
      {
        st_J = getData( jSection, jArray, stList );
      }
    }

    return (st_J);
  }

  /** Method addAssociation()
   * <p> This method will the association data section to the section
   * @param  iSection = the section ID to be parsed
   * @param  iGroup   = the Type of Data within the section to modify.
   * @param  iArray   = Array Number of Data Type
   * @param  jSection = the Association ID to be parsed
   * @param  jGroup   = the Type of Data within the Association being modified.
   * @param  jArray   = Array Number of Association Type
   * @param  stList   = The LAS File Data List Data Structure
   * @return st_I     = The LAS File Data Structure.
   */

  public static las3Struct addAssociation(
                  int iSection, int iGroup, int iArray,
                  int jSection, int jGroup, int jArray,
                  las3ListStruct stList )
  {
    int j,k;
    las3Struct st_I      = null;
    las3Struct st_J      = null;
    String     sOrderd[] = { "", "", "", "", "", "" };
    int        iContinue = 0;

    if (stList != null)
    {
      st_I = getData( iSection, iArray, stList );

      if (st_I != null)
      {
        if (isData( jSection, jArray, stList ))
        {
          st_J = getData( jSection, jArray, stList );

          if (st_J != null)
          {
            switch (jGroup)
            {
              case las3.las3Constants._PARAMETER:
                if ((st_I.iParamRows == 0) && (st_J.iParamRows > 0))
                {
                  for (j=0; j<st_J.iParamRows; j++)
                  {
                    for (k=0; k<st_J.iParamCols; k++)
                    {
                      sOrderd[k] = st_J.sParams[j][k];
                    }

                    st_I = addParameters(sOrderd, st_I);
                  }
                }
                break;
              case las3.las3Constants._DEFINITION:
                if ((st_I.iCurveRows == 0) && (st_J.iCurveRows > 0))
                {
                  for (j=0; j<st_J.iCurveRows; j++)
                  {
                    for (k=0; k<st_J.iCurveCols; k++)
                    {
                      sOrderd[k] = st_J.sCurves[j][k];
                    }

                    st_I = addCurves(sOrderd, st_I);
                  }
                }
                break;
            }
          }
        }
      }
    }

    return (st_I);
  }

  /** Method modify()
   * <p> This method will modify a Rock LAS 3 File in an existing list
   * @param  stBlock = The LAS 3 File structure that will be replaced
   * @param  st      = The Old LAS File Data List Data Structure
   * @return st      = The new LAS File Data List.
   */

  public static las3ListStruct modify( las3Struct stBlock, las3ListStruct st )
  {
    int i=0;
    double dVersion = 0.0;  // Version Number of LAS File
    String sWrap    = "No"; // Multiple lines per row
    String sDelim   = " ";  // Delimiting Character between data
    las3ListStruct stTemp = null;

    if (st != null)
    {
      stTemp        = new las3ListStruct();
      dVersion      = st.dVersion;
      sWrap         = new String( st.sWrap );
      sDelim        = new String( st.sDelim );
      stTemp.stItem = new las3Struct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if ((stBlock.sType.equals(st.stItem[i].sType)) &&
              (stBlock.iArray == st.stItem[i].iArray))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

      st          = new las3ListStruct();
      st.dVersion = dVersion;
      st.sWrap    = new String( sWrap );
      st.sDelim   = new String( sDelim );
      st.stItem   = new las3Struct[stTemp.iCount];
      st.iCount   = stTemp.iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copyList( las3ListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static las3ListStruct copyList( las3ListStruct stOld )
  {
    las3ListStruct stNew = null;

    if (stOld != null)
    {
      stNew          = new las3ListStruct();

      stNew.dVersion = stOld.dVersion;             // Version Number of LAS File
      stNew.sWrap    = new String( stOld.sWrap );  // Multiple lines per row
      stNew.sDelim   = new String( stOld.sDelim ); // Delimiting Character

      stNew.iCount   = stOld.iCount;
      stNew.stItem   = new las3Struct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method copy( las3Struct stOld )
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original Data Structure
   * @return stNew = The new Data Structure
   */

  public static las3Struct copy( las3Struct stOld )
  {
    int i, j;
    las3Struct stNew = new las3Struct();

    if (stOld != null)
    {
      stNew.iArray     = stOld.iArray;              // Array Number
      stNew.iType      = stOld.iType;               // Type of Data Identifier
      stNew.sType      = new String( stOld.sType ); // Type of Data Set
      stNew.iqstrat    = stOld.iqstrat;

      /* LAS File Information
       */

      stNew.iArrayP    = stOld.iArrayP; // Array Number
      stNew.sWrapP     = new String( stOld.sWrapP );
      stNew.sDelimP    = new String( stOld.sDelimP );
      stNew.iTypeP     = stOld.iTypeP;  // Type of LAS File Retrieval, FILE or URL
      stNew.sDirectory = new String( stOld.sDirectory ); // Directory Path to the LAS File
      stNew.sFilename  = new String( stOld.sFilename );  // Filename of LAS File
      stNew.dVersion   = stOld.dVersion;                 // Version Number of File

      /* Parameters
       */

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns

      if ((stOld.iParamRows > 0) && (stOld.iParamCols > 0))
      {
        if (stOld.sParam != null)
          stNew.sParam = new String[stNew.iParamRows];
        if (stOld.iParam != null)
          stNew.iParam = new int[stNew.iParamRows];

        stNew.sParams = new String[stNew.iParamRows][stNew.iParamCols];

        for (i=0; i<stOld.iParamRows; i++)
        {
          if (stOld.sParam != null)
          {
            if (stOld.sParam[i] != null)
              stNew.sParam[i] = new String(stOld.sParam[i]);
          }
          if (stOld.iParam != null)
            stNew.iParam[i] = stOld.iParam[i];

          for (j=0; j<stOld.iParamCols; j++)
          {
            stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
          }
        }
      }

      /* Curves
       */

      stNew.iCurveRows = stOld.iCurveRows;    // Total number of rows
      stNew.iCurveCols = stOld.iCurveCols;    // Total number of columns

      if ((stOld.iCurveRows > 0) && (stOld.iCurveCols > 0))
      {
        stNew.sCurve  = new String[stNew.iCurveRows];
        stNew.iCurve  = new int[stNew.iCurveRows];

        stNew.sCurves = new String[stNew.iCurveRows][stNew.iCurveCols];
        stNew.select  = new String[stNew.iCurveRows][stNew.iCurveCols];

        for (i=0; i<stOld.iCurveRows; i++)
        {
          if (stOld.sCurve != null)
          {
            if (stOld.sCurve[i] != null)
              stNew.sCurve[i] = new String(stOld.sCurve[i]);
          }
          if (stOld.iCurve != null)
            stNew.iCurve[i] = stOld.iCurve[i];

          for (j=0; j<stOld.iCurveCols; j++)
          {
            stNew.sCurves[i][j] = new String( stOld.sCurves[i][j] );
          }
        }

        stNew.iSelectRows = stOld.iSelectRows;

        for (i=0; i<stOld.iSelectRows; i++)
        {
          for (j=0; j<stOld.iCurveCols; j++)
          {
            if (stOld.select != null)
            {
              if (stOld.select[i][j] != null)
                stNew.select[i][j] = new String(stOld.select[i][j]);
            }
          }
        }
      }

      /* Data
       */

//      stNew.iCount   = stOld.iCount;    // Used for filling Data Array
      stNew.iRows    = stOld.iRows;     // Total Number of Rows
      stNew.iColumns = stOld.iColumns;  // Total Number of Columns

      if ((stOld.iRows > 0) && (stOld.iColumns > 0))
      {
        stNew.sData = new String[stNew.iRows][stNew.iColumns];

        for (i=0; i<stOld.iRows; i++)
        {
          for (j=0; j<stOld.iColumns; j++)
          {
			if (stOld.sData[i][j] != null)
              stNew.sData[i][j] = new String( stOld.sData[i][j] );
            else
              stNew.sData[i][j] = new String( "" );
          }
        }
      }
    }

    return (stNew);
  }

  /** Method copyWOData( las3Struct stOld )
   * <p> This method will copy data from one structure to another ignoring the
   *     data section
   * @param  stOld = The original Data Structure
   * @return stNew = The new Data Structure
   */

  public static las3Struct copyWOData( las3Struct stOld )
  {
    int i, j;
    las3Struct stNew = new las3Struct();

    if (stOld != null)
    {
      stNew.iArray     = stOld.iArray;              // Array Number
      stNew.iType      = stOld.iType;               // Type of Data Identifier
      stNew.sType      = new String( stOld.sType ); // Type of Data Set
      stNew.iqstrat    = stOld.iqstrat;

      // Parameters

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns

      if ((stOld.iParamRows > 0) && (stOld.iParamCols > 0))
      {
        if (stOld.sParam != null)
          stNew.sParam = new String[stNew.iParamRows];
        if (stOld.iParam != null)
          stNew.iParam = new int[stNew.iParamRows];

        stNew.sParams = new String[stNew.iParamRows][stNew.iParamCols];

        for (i=0; i<stOld.iParamRows; i++)
        {
          if (stOld.sParam != null)
          {
            if (stOld.sParam[i] != null)
              stNew.sParam[i] = new String(stOld.sParam[i]);
          }
          if (stOld.iParam != null)
            stNew.iParam[i] = stOld.iParam[i];

          for (j=0; j<stOld.iParamCols; j++)
          {
            stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
          }
        }
      }

      // Curves

      stNew.iCurveRows = stOld.iCurveRows;    // Total number of rows
      stNew.iCurveCols = stOld.iCurveCols;    // Total number of columns

      if ((stOld.iCurveRows > 0) && (stOld.iCurveCols > 0))
      {
        stNew.sCurve  = new String[stNew.iCurveRows];
        stNew.iCurve  = new int[stNew.iCurveRows];

        stNew.sCurves = new String[stNew.iCurveRows][stNew.iCurveCols];
        stNew.select  = new String[stNew.iCurveRows][stNew.iCurveCols];

        for (i=0; i<stOld.iCurveRows; i++)
        {
          if (stOld.sCurve != null)
          {
            if (stOld.sCurve[i] != null)
              stNew.sCurve[i] = new String(stOld.sCurve[i]);
          }
          if (stOld.iCurve != null)
            stNew.iCurve[i] = stOld.iCurve[i];

          for (j=0; j<stOld.iCurveCols; j++)
          {
            stNew.sCurves[i][j] = new String( stOld.sCurves[i][j] );
          }
        }

        stNew.iSelectRows = stOld.iSelectRows;

        for (i=0; i<stOld.iSelectRows; i++)
        {
          for (j=0; j<stOld.iCurveCols; j++)
          {
            if (stOld.select != null)
            {
              if (stOld.select[i][j] != null)
                stNew.select[i][j] = new String(stOld.select[i][j]);
            }
          }
        }
      }
    }

    return (stNew);
  }

  /** Method copyCurves( las3Struct stOld )
   * <p> This method will copy data from one structure to another ignoring the
   *     data & parameter sections
   * @param  stOld = The original Data Structure
   * @param  stNew = The new Data Structure
   * @return stNew = The new Data Structure
   */

  public static las3Struct copyCurves( las3Struct stOld, las3Struct stNew )
  {
    int i, j;

    if ((stOld != null) && (stNew != null))
    {
      stNew.iCurveRows = stOld.iCurveRows;    // Total number of rows
      stNew.iCurveCols = stOld.iCurveCols;    // Total number of columns

      if ((stOld.iCurveRows > 0) && (stOld.iCurveCols > 0))
      {
        stNew.sCurve  = new String[stNew.iCurveRows];
        stNew.iCurve  = new int[stNew.iCurveRows];

        stNew.sCurves = new String[stNew.iCurveRows][stNew.iCurveCols];
        stNew.select  = new String[stNew.iCurveRows][stNew.iCurveCols];

        for (i=0; i<stOld.iCurveRows; i++)
        {
          if (stOld.sCurve != null)
          {
            if (stOld.sCurve[i] != null)
              stNew.sCurve[i] = new String(stOld.sCurve[i]);
          }
          if (stOld.iCurve != null)
            stNew.iCurve[i] = stOld.iCurve[i];

          for (j=0; j<stOld.iCurveCols; j++)
          {
            stNew.sCurves[i][j] = new String( stOld.sCurves[i][j] );
          }
        }

        stNew.iSelectRows = stOld.iSelectRows;

        for (i=0; i<stOld.iSelectRows; i++)
        {
          for (j=0; j<stOld.iCurveCols; j++)
          {
            if (stOld.select != null)
            {
              if (stOld.select[i][j] != null)
                stNew.select[i][j] = new String(stOld.select[i][j]);
            }
          }
        }
      }
    }

    return (stNew);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static las3ListStruct bubbleSort(las3ListStruct st)
  {
    boolean swappedOnPrevRun = true;
    las3Struct stTemp        = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].iType > st.stItem[i+1].iType)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print all the data in the LAS 3 Data List Structure
   * @param st = the LAS 3 Data List Structure
   */

  public static void print( las3ListStruct st )
  {
    if (st != null)
    {
      System.out.println( "VERSION: "    + st.dVersion );
      System.out.println( "WRAP: "       + st.sWrap );
      System.out.println( "DELIMITER: <" + st.sDelim + ">");
      System.out.println("\n");

      for (int i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print()
   * <p> This method will print the data type in the LAS 3 Data List Structure
   * @param st = the LAS 3 Data Structure
   */

  public static void print( las3Struct st )
  {
    int i,j;
    String sObj = "";

    if (st != null)
    {
      System.out.println(
        " ***************************************************************");
      if (st.iArray > 0)
        System.out.println( " ***** " + st.sType.toUpperCase() +
                            "[" + st.iArray + "] ******");
      else if (st.sType != null)
        System.out.println( " ***** " + st.sType.toUpperCase() + " ******");

      if (st.iArrayP          > -1) System.out.println("Array P: " + st.iArrayP);
      if (st.sWrapP.length()  > 0)  System.out.println("Wrap P: "  + st.sWrapP);
      if (st.sDelimP.length() > 0)  System.out.println("Delim P: " + st.sDelimP);

      if (st.sDirectory.length() > 0)   System.out.println("DIR: "  + st.sDirectory);
      if (st.sFilename.length()  > 0)   System.out.println("File: " + st.sFilename);
      if (st.dVersion            > 0.0) System.out.println("Ver: "  + st.dVersion);

      if (st.iParamRows > 0)
      {
        System.out.println(" ***** PARAMETERS ******\n");
        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            sObj = st.sParams[i][j];
            if (sObj.length() > 0)
              System.out.print( las3.las3Struct.LAS_LINE[j] + ": {" + sObj + "} ");
          }
          System.out.println("");
        }
      }

      System.out.println("\n");

      if (st.iCurveRows > 0)
      {
        System.out.println(" ***** DEFINITIONS ******\n");
        for (i=0; i<st.iCurveRows; i++)
        {
          for (j=0; j<st.iCurveCols; j++)
          {
            sObj = st.sCurves[i][j];
            if (sObj.length() > 0)
              System.out.print( las3.las3Struct.LAS_LINE[j] + ": {" + sObj + "} " );
          }
          System.out.println("");
        }
      }
/*
      System.out.println("\n");

      if (st.iRows > 0)
      {
        System.out.println(" ***** DATA ******\n");
        for (i=0; i<st.iRows; i++)
        {
          for (j=0; j<st.iColumns; j++)
          {
            sObj = st.sData[i][j];
            if (j > 0) System.out.print( "; " );
            System.out.print( sObj );
          }
          System.out.println("");
        }
      }
*/
      System.out.println(
        " ***************************************************************\n");
    }
  }
}

/*
 *  @version 1.1 01/15/2010
 *  @author  John Victorine
 */
