/*
 * @las3Map.java Version 1.1 01/27/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import cmn.cmnLASCurveStruct;
import cmn.cmnLASToolsListStruct;

import xml.ReadLASToolsXMLFile;

/** Class las3Map
 *  <p> This Class will provide basic utilities for the LAS 3 Data Structure
 *
 *  @version 1.1 01/27/2010
 *  @author  John R. Victorine
 */

public class las3Map
{
  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  public static final int _LESS_THAN    = -1;
  public static final int _EQUAL_TO     = 0;
  public static final int _GREATER_THAN = 1;

  /** Method getLASCurves()
   * <p> This method will map the LAS Curves from the LAS 3 Data Structure
   *     to the Common LAS Curve List Structure.
   *
   * Note:  This method assumes that the LAS 3 Data Structure will hold the
   *        Curve description for LAS Log Data Curves.
   * @param  stLAS = the LAS 3 Data Structure
   * @return stCMN = Common LAS Curve List Structure.
   */

  public static cmnLASCurveListStruct getLASCurves( las3Struct stLAS )
  {
    cmnLASCurveListStruct stCMN = null;
    String                sTemp = "";

    if (stLAS != null)
    {
      if (stLAS.iCurveRows > 0)
      {
        stCMN        = new cmnLASCurveListStruct();
        stCMN.stItem = new cmnLASCurveStruct[stLAS.iCurveRows];
        stCMN.iCount = stLAS.iCurveRows;

        if (stLAS.sCurves != null)
        {
          for (int i=0; i<stLAS.iCurveRows; i++)
          {
            stCMN.stItem[i] = new cmnLASCurveStruct();

            // Mnemonic Name of Curve

            sTemp = stLAS.sCurves[i][_MNEMONIC];
            if (stLAS.iSelectRows > 0)
              sTemp = stLAS.select[i][_MNEMONIC];

            stCMN.stItem[i].sMnemonic = new String( sTemp );

            // Curve Column Number in LAS File

            stCMN.stItem[i].iCurveNumber = i;

            // Curve Name

            sTemp = stLAS.sCurves[i][_DESCRIPTION];
            if (stLAS.iSelectRows > 0)
              sTemp = stLAS.select[i][_DESCRIPTION];

            stCMN.stItem[i].sCurveName = new String( sTemp );

            // Curve Units

            sTemp = stLAS.sCurves[i][_UNIT];
            if (stLAS.iSelectRows > 0)
              sTemp = stLAS.select[i][_UNIT];

            stCMN.stItem[i].sCurveUnits = new String( sTemp );

            // Description of the Curve

            sTemp = stLAS.sCurves[i][_DESCRIPTION];
            if (stLAS.iSelectRows > 0)
              sTemp = stLAS.select[i][_DESCRIPTION];

            stCMN.stItem[i].sCurveDescription = new String( sTemp );

            if (stLAS.sCurves[i][_ASSOCIATION].length() > 0)
            {
              stCMN.stItem[i].sAssoc = las3.las3Parse.splitDataLine(
                                         stLAS.sCurves[i][_ASSOCIATION], "," );
              if (stCMN.stItem[i].sAssoc != null)
              {
                stCMN.stItem[i].iAssoc = stCMN.stItem[i].sAssoc.length;
              }
            }

            // Other Variables filled by the LAS Standard Tools

            stCMN.stItem[i].sKID          = "0"; // Primary KEY of LAS File
            stCMN.stItem[i].sToolKID      = "0"; // Primary KEY of Tool KID
            stCMN.stItem[i].dMinimumValue = 0.0; // Minimum Value of Curve
            stCMN.stItem[i].dMaximumValue = 0.0; // Maximum Value of Curve
            stCMN.stItem[i].iOrder        = 0;   // Order of Curve (initial sorting)
            stCMN.stItem[i].iTrack        = 0;   // Track the Curve should reside
            stCMN.stItem[i].dToolMinimumValue = 0.0;  // Minimum Curve Tool Value
            stCMN.stItem[i].dToolMaximumValue = 0.0;  // Maximum Curve Tool Value
          }
        }
      }
    }

    return (stCMN);
  }

  /** Method mapToKGS()
   * <p> This method will map the LAS Curves from the LAS File to Standard
   *     KGS Log Curve Types.
   * @param  sPath  = The URL Path to the KGS Standard Tools XML
   * @param  stCMN  = Common LAS Curve List Structure.
   * @return stCMN  = Common LAS Curve List Structure.
   */

  public static cmnLASCurveListStruct mapToKGS(
      String sPath, cmnLASCurveListStruct stCMN )
  {
    cmnLASToolsListStruct stTools = null;  // Standard LAS Curves

    if (stCMN != null)
    {
      ReadLASToolsXMLFile pTools =
          new ReadLASToolsXMLFile( xml.ReadLASToolsXMLFile.URL );
      stTools = pTools.Process( sPath );

      if (stTools != null);
      {
        for (int i=0; i<stCMN.iCount; i++)
        {
          stCMN.stItem[i] = mapToStandardTool(stCMN.stItem[i], stTools);
        }
      }
    }

    return (stCMN);
  }

  /** Method mapToStandardTool()
   * <p> This method will map the Tool to a Standard Tool
   * @param  st      = LAS Curve Data Structure
   * @param  stTools = The KGS Standard Tools List Data Structure
   * @return st      = LAS Curve Data Structure
   */

  private static cmnLASCurveStruct mapToStandardTool(
      cmnLASCurveStruct st, cmnLASToolsListStruct stTools )
  {
    if (stTools != null)
    {
      for (int i=0; i<stTools.iCount; i++)
      {
        if (st.sMnemonic.equals(stTools.stItem[i].sMnemonic))
        {
          st.sToolKID          = new String( stTools.stItem[i].sKEY );
          st.sCurveName        = new String( stTools.stItem[i].sDesc );
          st.sCurveUnits       = new String( stTools.stItem[i].sUnit );
          st.sCurveDescription = new String( stTools.stItem[i].sDesc );
          st.dMinimumValue     = stTools.stItem[i].dMinimum;
          st.dMaximumValue     = stTools.stItem[i].dMaximum;
          st.iTrack            = (int) stTools.stItem[i].dTrack;
        }
        else
        {
          for (int j=0; j<stTools.stItem[i].iOther; j++)
          {
            if (st.sMnemonic.equals(stTools.stItem[i].sOther[j]))
            {
              st.sToolKID          = new String( stTools.stItem[i].sKEY );
              st.sCurveName        = new String( stTools.stItem[i].sDesc );
              st.sCurveUnits       = new String( stTools.stItem[i].sUnit );
              st.sCurveDescription = new String( stTools.stItem[i].sDesc );
              st.dMinimumValue     = stTools.stItem[i].dMinimum;
              st.dMaximumValue     = stTools.stItem[i].dMaximum;
              st.iTrack            = (int) stTools.stItem[i].dTrack;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method mapToKGS()
   * <p> This method will map the Core Data Curves from the LAS File to Standard
   *     KGS Core Curve Types.
   * @param  iType  = The Type of Data being processed
   * @param  stCMN  = Common LAS Curve List Structure.
   * @return stCMN  = Common LAS Curve List Structure.
   */

  public static cmnLASCurveListStruct mapToKGS( int iType,
                                                cmnLASCurveListStruct stCMN )
  {
    int    TOTAL     = 0;
    String sKID      = "";
    String sMnemonic = "";
    String sUnit     = "";
    String sName     = "";
    String sData     = "";
    double dMin      = 0.0;
    double dMax      = 0.0;

    switch (iType)
    {
      case las3.las3Constants._CORE:
        TOTAL = rock.rockStandardTools.TOTAL_DATA;
        break;
      case las3.las3Constants._TOPS:
        TOTAL = horizon.strat.stratStandardTools.TOTAL;
        break;
      case las3.las3Constants._TEST:
        TOTAL = dst.dstStandardTools.TOTAL;
        break;
      case las3.las3Constants._PERFORATION:
        TOTAL = horizon.regions.regionsStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_CONTROL:
        TOTAL = iqstrat.iqstratControlStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_BIO:
        TOTAL = horizon.bio.bioStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_ENV:
        TOTAL = horizon.env.envStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_SEQ_STRAT:
        TOTAL = horizon.seq.seqStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_FLOW:
        TOTAL = pfeffer.pfefferStandardTools.TOTAL_FLOW;
        break;
      case las3.las3Constants._IQ_PFEFFER:
        TOTAL = pfeffer.pfefferStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_GEO_REPORT:
        TOTAL = iqstrat.iqstratRemarkStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_SHALE:
        TOTAL = iqstrat.iqstratShaleStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_IMAGES:
        TOTAL = rock.rockImagesStandardTools.TOTAL;
        break;
      case las3.las3Constants._IQ_BRINE:
        TOTAL = brine.brineStandardTools.TOTAL;
        break;
    }

    if (stCMN != null)
    {
      for (int i=0; i<stCMN.iCount; i++)
      {
        for (int j=0; j<TOTAL; j++)
        {
          switch (iType)
          {
            case las3.las3Constants._CORE:
              sKID      = rock.rockStandardTools.ROCK_TOOLS[j][6];
              sMnemonic = rock.rockStandardTools.ROCK_TOOLS[j][0];
              sUnit     = rock.rockStandardTools.ROCK_TOOLS[j][2];
              sName     = rock.rockStandardTools.ROCK_TOOLS[j][1];
              sData     = rock.rockStandardTools.ROCK_TOOLS[j][4];
              dMin      = 0.0;
              if (cmn.cmnString.isNumeric( sData ))
                dMin    = cmn.cmnString.stringToDouble( sData );
              dMax      = 0.0;
              sData     = rock.rockStandardTools.ROCK_TOOLS[j][5];
              if (cmn.cmnString.isNumeric( sData ))
                dMax    = cmn.cmnString.stringToDouble( sData );
              break;
            case las3.las3Constants._TOPS:
              sKID      = horizon.strat.stratStandardTools.CURVES[j][0];
              sMnemonic = horizon.strat.stratStandardTools.CURVES[j][1];
              sUnit     = horizon.strat.stratStandardTools.CURVES[j][3];
              sName     = horizon.strat.stratStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._TEST:
              sKID      = dst.dstStandardTools.CURVES[i][0];
              sMnemonic = dst.dstStandardTools.CURVES[i][1];
              sUnit     = dst.dstStandardTools.CURVES[i][3];
              sName     = dst.dstStandardTools.CURVES[i][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._PERFORATION:
              sKID      = horizon.regions.regionsStandardTools.CURVES[j][0];
              sMnemonic = horizon.regions.regionsStandardTools.CURVES[j][1];
              sUnit     = horizon.regions.regionsStandardTools.CURVES[j][3];
              sName     = horizon.regions.regionsStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_CONTROL:
              sKID      = iqstrat.iqstratControlStandardTools.CURVES[j][0];
              sMnemonic = iqstrat.iqstratControlStandardTools.CURVES[j][1];
              sUnit     = "";
              sName     = iqstrat.iqstratControlStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_BIO:
              sKID      = horizon.bio.bioStandardTools.CURVES[j][0];
              sMnemonic = horizon.bio.bioStandardTools.CURVES[j][1];
              sUnit     = horizon.bio.bioStandardTools.CURVES[j][3];
              sName     = horizon.bio.bioStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_ENV:
              sKID      = horizon.env.envStandardTools.CURVES[j][0];
              sMnemonic = horizon.env.envStandardTools.CURVES[j][1];
              sUnit     = horizon.env.envStandardTools.CURVES[j][3];
              sName     = horizon.env.envStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_SEQ_STRAT:
              sKID      = horizon.seq.seqStandardTools.CURVES[j][0];
              sMnemonic = horizon.seq.seqStandardTools.CURVES[j][1];
              sUnit     = horizon.seq.seqStandardTools.CURVES[j][3];
              sName     = horizon.seq.seqStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_FLOW:
              sKID      = pfeffer.pfefferStandardTools.P_CURVES[j][0];
              sMnemonic = pfeffer.pfefferStandardTools.P_CURVES[j][1];
              sUnit     = pfeffer.pfefferStandardTools.P_CURVES[j][3];
              sName     = pfeffer.pfefferStandardTools.P_CURVES[j][2];
              sData     = pfeffer.pfefferStandardTools.P_CURVES[j][4];
              dMin      = 0.0;
              if (cmn.cmnString.isNumeric( sData ))
                dMin    = cmn.cmnString.stringToDouble( sData );
              sData     = pfeffer.pfefferStandardTools.P_CURVES[j][5];
              if (cmn.cmnString.isNumeric( sData ))
                dMax    = cmn.cmnString.stringToDouble( sData );
              break;
            case las3.las3Constants._IQ_PFEFFER:
              sKID      = pfeffer.pfefferStandardTools.CURVES[j][0];
              sMnemonic = pfeffer.pfefferStandardTools.CURVES[j][1];
              sUnit     = pfeffer.pfefferStandardTools.CURVES[j][3];
              sName     = pfeffer.pfefferStandardTools.CURVES[j][2];
              sData     = pfeffer.pfefferStandardTools.CURVES[j][4];
              dMin      = 0.0;
              if (cmn.cmnString.isNumeric( sData ))
                dMin    = cmn.cmnString.stringToDouble( sData );
              sData     = pfeffer.pfefferStandardTools.CURVES[j][5];
              if (cmn.cmnString.isNumeric( sData ))
                dMax    = cmn.cmnString.stringToDouble( sData );
              break;
            case las3.las3Constants._IQ_GEO_REPORT:
              sKID      = iqstrat.iqstratRemarkStandardTools.CURVES[j][0];
              sMnemonic = iqstrat.iqstratRemarkStandardTools.CURVES[j][1];
              sUnit     = iqstrat.iqstratRemarkStandardTools.CURVES[j][3];
              sName     = iqstrat.iqstratRemarkStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_SHALE:
              sKID      = iqstrat.iqstratShaleStandardTools.CURVES[j][0];
              sMnemonic = iqstrat.iqstratShaleStandardTools.CURVES[j][1];
              sUnit     = iqstrat.iqstratShaleStandardTools.CURVES[j][3];
              sName     = iqstrat.iqstratShaleStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_IMAGES:
              sKID      = rock.rockImagesStandardTools.CURVES[j][0];
              sMnemonic = rock.rockImagesStandardTools.CURVES[j][1];
              sUnit     = rock.rockImagesStandardTools.CURVES[j][3];
              sName     = rock.rockImagesStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
            case las3.las3Constants._IQ_BRINE:
              sKID      = brine.brineStandardTools.CURVES[j][0];
              sMnemonic = brine.brineStandardTools.CURVES[j][1];
              sUnit     = brine.brineStandardTools.CURVES[j][3];
              sName     = brine.brineStandardTools.CURVES[j][2];
              dMin      = 0.0;
              dMax      = 0.0;
              break;
          }

          if (stCMN.stItem[i].sMnemonic.equals( sMnemonic ))
          {
            stCMN.stItem[i].sToolKID          = new String( sKID );
            stCMN.stItem[i].sCurveName        = new String( sName );
            stCMN.stItem[i].sCurveUnits       = new String( sUnit );
            stCMN.stItem[i].sCurveDescription = new String( sName );
            stCMN.stItem[i].dMinimumValue     = dMin;
            stCMN.stItem[i].dMaximumValue     = dMax;
          }
        }
      }
    }

    return (stCMN);
  }
}

/*
 *  @version 1.1 01/27/2010
 *  @author  John Victorine
 */
