/*
 * @las3LoadWellHeader.java Version 1.1 01/21/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import iqstrat.iqstratHeadersStruct;
import las.lasFileDataStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadWellHeader
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 01/21/2010
 *  @author  John R. Victorine
 */

public class las3LoadWellHeader
{
  public static final int _NONE     = las3.las3Constants._NONE;

  // ~WELL Mnemonics in Well Header

  public static final int _COMPANY    = 0;
  public static final int _WELL       = 1;
  public static final int _FIELD      = 2;
  public static final int _SECTION    = 3;
  public static final int _TOWNSHIP   = 4;
  public static final int _RANGE      = 5;
  public static final int _LOCATION   = 6;
  public static final int _LOCATION_1 = 7;
  public static final int _LOCATION_2 = 8;
  public static final int _STATE      = 9;
  public static final int _PROVINCE   = 10;
  public static final int _COUNTRY    = 11;
  public static final int _COUNTY     = 12;
  public static final int _API        = 13;
  public static final int _UWI        = 14;
  public static final int _LICENSE    = 15;
  public static final int _SERVICE    = 16;
  public static final int _DATE       = 17;
  public static final int _X          = 18;
  public static final int _Y          = 19;
  public static final int _LATI       = 20;
  public static final int _LONG       = 21;
  public static final int _GDAT       = 22;
  public static final int _HZCS       = 23;
  public static final int _UTM        = 24;
  public static final int _STATUS     = 25;

  public static final int _WELL_ROWS  = 26;
  public static final int _WELL_COLS  = 5;
  public static final String WELL[][] = {
  // Mnemonic  Description                    Variations
    { "COMP",  "Company",                     "CN",   "CN2", "COMPANY"  },
    { "WELL",  "Well",                        "LEAS", "WN", "WELL" },
    { "FLD",   "Field",                       "FLD", "", "FIELD" },
    { "SECT",  "Section",                     "SECT2", "SEC", "SECTION" },
    { "TOWN",  "Township",                    "TOWN2", "", "TOWNSHIP" },
    { "RANG",  "Range",                       "RANG2", "", "RANGE" },
    { "LOC",   "Location",                    "LOC",   "", "LOCATION" },
    { "LOC1",  "Location 1",                  "LOC1",  "", "LOCATION 1" },
    { "LOC2",  "Location 2",                  "LOC2",  "", "LOCATION 2" },
    { "STAT",  "State",                       "STAT2", "", "STATE" },
    { "PROV",  "Province",                    "PROV",  "", "PROVINCE" },
    { "CTRY",  "Country",                     "NATI",  "", "COUNTRY" },
    { "CNTY",  "County",                      "COUN",  "", "COUNTY" },
    { "API",   "API-Number",                  "API",   "API-NUMBER", "API NUMBER" },
    { "UWI",   "UNIQUE Well ID",              "UWI",   "", "UNIQUE WELL ID" },
    { "LIC",   "License Number",              "LIC",   "LICENSE", "LICENSE NUMBER" },
    { "SRVC",  "Service Company",             "SRVC",  "SERVICE", "SERVICE COMPANY" },
    { "DATE",  "Service Data",                "DATE2", "", "SERVICE DATE" },
    { "X",     "X or East-West coordinate",   "UTMX",  "", "" },
    { "Y",     "Y or North South coordinate", "UTMY",  "", "" },
    { "LATI",  "Latitude",                    "LAT",   "LAT", "LATITUDE"  },
    { "LONG",  "Longitude",                   "LNG",   "LON", "LONGITUDE"  },
    { "GDAT",  "Geodetic Datum",              "",      "GEODETIC", "GEODETIC DATUM" },
    { "HZCS",  "Horizontal Co-ordinate System", "", "", "" },
    { "UTM",   "UTM LOCATION",                  "", "", "" },
    { "STUS",  "Well Status",                   "", "", "" },
  };

  public static final int _GL = 0;
  public static final int _KB = 1;
  public static final int _DF = 2;
  public static final int _RT = 3;

  public static final int _DEPTHS_ROWS  = 4;
  public static final int _DEPTHS_COLS  = 6;
  public static final String DEPTHS[][] = {
 // Mnemonic  Description           Variations
    { "GL",  "Ground Level",        "EGL", "EGL2", "G.L.","GROUND LEVEL" },
    { "KB",  "Kelly Bushing",       "EKB", "EKB2", "K.B.", "KELLY BUSHING" },
    { "EDF", "Drilling Rig Floor",  "EDF2", "DFE", "D.F.", "DRILLING RIG FLOOR" },
    { "RT",  "Rotary Table",        "ERT", "R.T.", "ROTARY TABLE", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Well Header Data Structure from the
   *     LAS 3 Data List Structure.
   * @param  stLAS = the LAS 3 Data List Structure
   * @return st    = the Well Header Data Structure
   */

  public static iqstratHeadersStruct getData( las3ListStruct stLAS )
  {
    int i,j,k;
    iqstratHeadersStruct st     = new iqstratHeadersStruct();
    las3Struct           stH    = null;
    las3Struct           stP    = null;
    int                  iWell  = -1;
    int                  iParam = -1;

    if (stLAS != null)
    {
      for (i=0; i<stLAS.iCount; i++)
      {
        st.iLAS = 1;
        if (las3.las3Utility.isData( las3.las3Constants._WELL, 0, stLAS ))
          stH = las3.las3Utility.getData( las3.las3Constants._WELL, 0, stLAS );

        if (las3.las3Utility.isData( las3.las3Constants._LOG, 0, stLAS ))
          stP = las3.las3Utility.getData( las3.las3Constants._LOG, 0, stLAS );

        if (las3.las3Utility.isData( las3.las3Constants._CORE, 0, stLAS ))
          st.iCore = 1;

        if (las3.las3Utility.isData( las3.las3Constants._TOPS, 0, stLAS ))
          st.iTops = 1;
      }

      if (stH != null)
      {
        for (i=0; i<stH.iParamRows; i++)
        {
          iWell = _NONE;
          for (j=0; j<_WELL_ROWS; j++)
          {
            for (k=0; k<_WELL_COLS; k++)
            {
              if (stH.sParams[i][_MNEMONIC].toUpperCase().equals(WELL[j][k]))
              {
                iWell = j;

                switch (iWell)
                {
                  case _NONE:
                    break;
                  case _COMPANY:
                    st.sOperator = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _WELL:
                    st.sName     = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _FIELD:
                    st.sField    = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _LOCATION:
                    st.sLocation = new String( stH.sParams[i][_VALUE] );
                    if (st.iSection == 0)
                      st = parseSTR( stH.sParams[i][_VALUE], st );
                    break;
                  case _LOCATION_1:
                    st.sLocation1 = new String( stH.sParams[i][_VALUE] );
                    if (st.iSection == 0)
                      st = parseSTR( stH.sParams[i][_VALUE], st );
                    break;
                  case _LOCATION_2:
                    st.sLocation2 = new String( stH.sParams[i][_VALUE] );
                    if (st.iSection == 0)
                      st = parseSTR( stH.sParams[i][_VALUE], st );
                    break;
                  case _SECTION:
                    st = parseSection( stH.sParams[i][_VALUE], st );
                    break;
                  case _TOWNSHIP:
                    st = parseTownship( stH.sParams[i][_VALUE], st );
                    break;
                  case _RANGE:
                    st = parseRange( stH.sParams[i][_VALUE], st );
                    break;
                  case _STATE:
                    st.state     = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _PROVINCE:
                    st.sProvince = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _COUNTRY:
                    st.sCountry  = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _COUNTY:
                    st.sCounty   = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _API:
                    st.sAPI      = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _UWI:
                    st.sUWI      = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _LICENSE:
                    st.sLicense  = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _SERVICE:
                    st.service   = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _DATE:
                    st.sDate     = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _X:
                    if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                      st.dUTMx =
                          cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                    break;
                  case _Y:
                    if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                      st.dUTMy =
                          cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                    break;
                  case _LATI:
                    st.dLatitude  = parseDegrees( stH.sParams[i][_VALUE] );
                    break;
                  case _LONG:
                    st.dLongitude = parseDegrees( stH.sParams[i][_VALUE] );
                    break;
                  case _GDAT:
                    st.sGDAT      = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _HZCS:
                    st.sHZCS      = new String( stH.sParams[i][_VALUE] );
                    break;
                  case _UTM:
                    if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                      st.dZone =
                          cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                    break;
                  case _STATUS:
                    st.status     = new String( stH.sParams[i][_VALUE] );
                    break;
                }
              }
            }
          }
        }
      }

      if (stP != null)
      {
        for (i=0; i<stP.iParamRows; i++)
        {
          iWell = _NONE;
          for (j=0; j<_WELL_ROWS; j++)
          {
            for (k=0; k<_WELL_COLS; k++)
            {
              if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(WELL[j][k]))
              {
                iWell = j;

                switch (iWell)
                {
                  case _NONE:
                    break;
                  case _X:
                    if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                      st.dUTMx =
                          cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                    break;
                  case _Y:
                    if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                      st.dUTMy =
                          cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                    break;
                  case _LATI:
                    st.dLatitude  = parseDegrees( stP.sParams[i][_VALUE] );
                    break;
                  case _LONG:
                    st.dLongitude = parseDegrees( stP.sParams[i][_VALUE] );
                    break;
                  case _GDAT:
                    st.sGDAT      = new String( stP.sParams[i][_VALUE] );
                    break;
                  case _HZCS:
                    st.sHZCS      = new String( stP.sParams[i][_VALUE] );
                    break;
                  case _UTM:
                    if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                      st.dZone =
                          cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                    break;
                }
              }
            }
          }
        }

        for (i=0; i<stP.iParamRows; i++)
        {
          iParam = _NONE;
          for (j=0; j<_DEPTHS_ROWS; j++)
          {
            for (k=0; k<_DEPTHS_COLS; k++)
            {
              if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(DEPTHS[j][k]))
              {
                iParam = j;

                switch (iParam)
                {
                  case _NONE:
                    break;
                  case _GL:
                    st.dGL = convertMtoF( stP.sParams[i][_VALUE],
                                          stP.sParams[i][_UNIT] );
                    break;
                  case _KB:
                    st.dKB = convertMtoF( stP.sParams[i][_VALUE],
                                          stP.sParams[i][_UNIT] );
                    break;
                  case _DF:
                    st.dDF = convertMtoF( stP.sParams[i][_VALUE],
                                          stP.sParams[i][_UNIT] );
                    break;
                  case _RT:
                    st.dRT = convertMtoF( stP.sParams[i][_VALUE],
                                          stP.sParams[i][_UNIT] );
                    break;
                }
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method parseSTR()
   * <p> This method will parse the section township range from the location
   *     text string:
   *
   *     28-31S-37W (Section-Township-Range)
   * @param  str = location text string
   * @param  st  = Well Header Data Structure
   * @return st  = Well Header Data Structure
   */

  public static iqstratHeadersStruct parseSTR( String str,
                                               iqstratHeadersStruct st )
  {
    String stmp     = "";
    String stmp1    = "";
    String stmp2    = "";
    String tokens[] = null;
    String sDelim   = new String("[ ]+");

    if ((str != null) && (st != null))
    {
      if (str.length() > 0)
      {
        stmp = new String( str.replace('-', ' ') );

        tokens = stmp.split(sDelim);

        if (tokens.length == 3)
        {
          for (int i=0; i<tokens.length; i++)
          {
            if (cmn.cmnString.isNumeric( tokens[i] ))
              st.iSection = cmn.cmnString.stringToInt( tokens[i].trim() );

            stmp = new String( tokens[i].trim() );
            stmp1 = new String( stmp.substring(0, stmp.length()-1));
            stmp2 = new String( stmp.substring(stmp.length()-1, stmp.length()));

            if ((cmn.cmnString.isNumeric( stmp1 )) &&
                ((stmp2.equals("S")) || (stmp2.equals("N"))))
            {
              st.iTownship = cmn.cmnString.stringToInt( stmp1 );
              st.sTownship = new String( stmp2 );
            }

            if ((cmn.cmnString.isNumeric( stmp1 )) &&
                ((stmp2.equals("E")) || (stmp2.equals("W"))))
            {
              st.iRange = cmn.cmnString.stringToInt( stmp1 );
              st.sRange = new String( stmp2 );
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method parseSection()
   * <p> This method will parse the section from the location text string:
   *
   * @param  str = location text string
   * @param  st  = Well Header Data Structure
   * @return st  = Well Header Data Structure
   */

  public static iqstratHeadersStruct parseSection( String str,
                                                   iqstratHeadersStruct st )
  {
    if ((str != null) && (st != null))
    {
      if (str.length() > 0)
      {
        st.iSection = cmn.cmnString.stringToInt( str );
      }
    }

    return (st);
  }

  /** Method parseTownship()
   * <p> This method will parse township from the text string:
   *
   * @param  str = location text string
   * @param  st  = Well Header Data Structure
   * @return st  = Well Header Data Structure
   */

  public static iqstratHeadersStruct parseTownship( String str,
                                                    iqstratHeadersStruct st )
  {
    String stmp1    = "";
    String stmp2    = "";

    if ((str != null) && (st != null))
    {
      if (str.length() > 0)
      {
        stmp1 = new String( str.substring(0, str.length()-1));
        stmp2 = new String( str.substring(str.length()-1, str.length()));

        if ((cmn.cmnString.isNumeric( stmp1 )) &&
            ((stmp2.equals("S")) || (stmp2.equals("N"))))
        {
          st.iTownship = cmn.cmnString.stringToInt( stmp1 );
          st.sTownship = new String( stmp2 );
        }
      }
    }

    return (st);
  }

  /** Method parseSection()
   * <p> This method will parse the range from the text string:
   *
   * @param  str = location text string
   * @param  st  = Well Header Data Structure
   * @return st  = Well Header Data Structure
   */

  public static iqstratHeadersStruct parseRange( String str,
                                                 iqstratHeadersStruct st )
  {
    String stmp1    = "";
    String stmp2    = "";

    if ((str != null) && (st != null))
    {
      if (str.length() > 0)
      {
        stmp1 = new String( str.substring(0, str.length()-1));
        stmp2 = new String( str.substring(str.length()-1, str.length()));

        if ((cmn.cmnString.isNumeric( stmp1 )) &&
            ((stmp2.equals("E")) || (stmp2.equals("W"))))
        {
          st.iRange = cmn.cmnString.stringToInt( stmp1 );
          st.sRange = new String( stmp2 );
        }
      }
    }

    return (st);
  }

  /** Method parseDegrees()
   * <p> This method will parse the latitude or longitude (degrees, minutes,
   *     second) to degrees or get the numeric value
   *
   *   Example: 45.37 12' 58"
   * @param  str    = string to be converted
   * @return dValue = numeric value
   */

  public static double parseDegrees( String str )
  {
    String stmp   = "";
    String str2   = "";
    double dValue = 0.0;
    int    iValue = 0;
    double deg    = 0.0;
    double dMin   = 0.0;
    double dSec   = 0.0;
    String tokens[] = null;
    String sDelim   = new String("[ ]+");

    if (str.length() > 0)
    {
      tokens = (str.trim()).split(sDelim);

      if (tokens.length == 3)
      {
        stmp = new String( tokens[0].trim() );
        str2 = new String( stmp.substring(0, stmp.length()-1));
        if (cmn.cmnString.isNumeric( str2 ))
          deg = cmn.cmnString.stringToDouble( str2.trim() );

        stmp = new String( tokens[1].trim() );
        str2 = new String( stmp.substring(0, stmp.length()-1));
        if (cmn.cmnString.isNumeric( str2 ))
          dMin = cmn.cmnString.stringToDouble( str2.trim() );

        stmp = new String( tokens[2].trim() );
        str2 = new String( stmp.substring(0, stmp.length()-1));
        if (cmn.cmnString.isNumeric( str2 ))
          dSec = cmn.cmnString.stringToDouble( str2.trim() );

        dValue = deg + dMin/60.0 + dSec/3600.0;
      }
      else
      {
        if (cmn.cmnString.isNumeric( str.trim() ))
          dValue = cmn.cmnString.stringToDouble( str.trim() );
      }

      iValue = (int) (dValue * 10000.0);
      dValue = (double) (iValue) / 10000.0;
    }

    return (dValue);
  }

  /** Method convertMtoF()
   * <p> This method will convert meters to feet
   * @param  str    = the text to be converted
   * @param  sUnit  = the unit
   * @return dValue = depth in feet
   */

  public static double convertMtoF(String str, String sUnit)
  {
    double dValue = 0.0;
    double dFeet  = 3.281;
    int    iValue = 0;

    if (str.length() > 0)
    {
      if (cmn.cmnString.isNumeric( str.trim() ))
      {
        dValue = cmn.cmnString.stringToDouble( str.trim() );

        if (sUnit.equals("M"))
          dValue = dFeet * dValue;

        iValue = (int) (dValue * 1000.0);
        dValue = (double) iValue / 1000.0;
      }
    }

    return (dValue);
  }

  /** Method removeChar()
   * <p> This method will remove the Quotes around the String
   * @param  str  = The string to be converted
   * @param  ca   = The character to be stripped from the string
   * @return sNew = The text converted
   */

  public static String removeChar( String str, char ca )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if (c[i] != ca)
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method normalizeDepth()
   * <p> This method will convert meters to feet
   * @param  st  = Well Header Data Structure
   * @param  stL = LAS File Data Structure
   * @return st  = Well Header Data Structure
   */

  public static iqstratHeadersStruct normalizeDepth(
      iqstratHeadersStruct st, lasFileDataStruct stL )
  {
    double diff = 15.0;  // Height of Kelly Bushing above Ground Level in ft

    if ((st != null) && (stL != null))
    {
      if (stL.sPDAT.length() == 0)
      {
        if ((st.dGL > 0.0) && (st.dKB == 0.0))
        {
          st.dKB = st.dGL + diff;
        }
        else if ((st.dGL == 0.0) && (st.dKB > 0.0))
        {
          st.dGL = st.dKB - diff;
        }
        else if ((st.dGL == 0.0) && (st.dDF > 0.0))
        {
          st.dGL = st.dDF - diff;
        }
      }
      else if ((st.dGL == 0.0) && (st.dKB == 0.0) && (st.dDF == 0.0))
      {
        if ((stL.sDREF.equals("KB")) && (stL.sPDAT.equals("GL")))
        {
          st.dGL = stL.dElevation - stL.dAPD;
          st.dKB = stL.dElevation;
        }
        else if ((stL.sDREF.equals("DF")) && (stL.sPDAT.equals("GL")))
        {
          st.dGL = stL.dElevation - stL.dAPD;
          st.dDF = stL.dElevation;
        }
        else if (stL.sDREF.equals("GL"))
        {
          st.dGL = stL.dElevation;
        }
        else if (stL.sDREF.equals("KB"))
        {
          st.dGL = stL.dElevation - stL.dAPD;
          st.dKB = stL.dElevation;
        }
        else if (stL.sDREF.equals("DF"))
        {
          st.dGL = stL.dElevation - stL.dAPD;
          st.dDF = stL.dElevation;
        }
      }

      if (stL.dTDD > 0.0)      // Drillers Depth
        st.depth = stL.dTDD;
      else if (stL.dTLD > 0.0) // Loggers Depth
        st.depth = stL.dTLD;
    }

    return (st);
  }
}

/*
 *  @version 1.1 01/21/2010
 *  @author  John Victorine
 */
