/*
 * @las3LoadTopsData.java Version 1.1 02/12/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadTopsData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 02/12/2010
 *  @author  John R. Victorine
 */

public class las3LoadTopsData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _TOPS        = 1;
  public static final int _TOPDR       = 2;

  public static final int _PARAM_ROWS  = 3;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description                        Variations
    { "IQKGS",  "Profile Web App Saved Data Indicator",  "", "", "" },
    { "TOPS",   "Formation Source",                      "", "", "" },
    { "TOPDR",  "Tops Depth Reference",                  "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Stratigraphic Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stT = the LAS 3 Data Structure
   * @return st  = the Stratigraphic Data List Structure
   */

  public static stratListStruct getData( las3Struct stT )
  {
    int i,j,k;
    int iParam = -1;
    stratListStruct st  = new stratListStruct();

    if (stT != null)
    {
      for (i=0; i<stT.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stT.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _TOPS:
                  st.sTOPS = new String( stT.sParams[i][_VALUE] );
                  break;
                case _TOPDR:
                  st.sTOPDR = new String( stT.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Stratigraphic Data List Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stTops  = the Stratigraphic Data List Structure
   * @return stTops2 = the Stratigraphic Data List Structure
   */

  public static stratListStruct loadData( double dNull,
                                          las3Struct st,
                                          cmnLASCurveListStruct stCMN,
                                          stratListStruct stTops )
  {
    stratListStruct stTops2 = null;
    String          sKEY    = "0";
    int             iTool   = -1;

    if ((st != null) && (stTops != null) && (stCMN != null))
    {
      if (stCMN.iCount > 0)
      {
        stTops2 = horizon.strat.stratUtility.copyList( stTops );

        stTops.delete();
        stTops = null;

        if (stTops2 == null)
          stTops2 = new stratListStruct();

        if (st.iRows > 0)
        {
          stTops2.stItem = new stratStruct[st.iRows];
          stTops2.iCount = st.iRows;

          for (int i=0; i<st.iRows; i++)
          {
            stTops2.stItem[i] = new stratStruct();

            // Unique KID & KEY if the LAS File does not contain a value
            // the value has to be present for the program to function correctly
            // the KID and KEY are used to modify and remove a stratigraphic
            // unit from the list.

            sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
            stTops2.stItem[i].sKID = new String( sKEY );
            stTops2.stItem[i].sKEY = new String( sKEY );

            for (int j=0; j<st.iColumns; j++)
            {
              iTool = -1;
              for (int k=0; k<stCMN.iCount; k++)
              {
                if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                        stCMN.stItem[k].sMnemonic))
                {
                  iTool = k;
                }
              }

              if (iTool > -1)
              {
                stTops2.stItem[i] = addTopsData( dNull,
                                                 stCMN.stItem[iTool].sMnemonic,
                                                 st.sData[i][j],
                                                 stTops2.stItem[i] );
              }
            }
          }
        }
      }
    }

    return (stTops2);
  }

  /** Method addTopsData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Stratigraphic Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Stratigraphic Data Structure
   * @return st        = the Stratigraphic Data Structure
   */

  public static stratStruct addTopsData( double dNULL,
                                         String sMnemonic,
                                         String sData,
                                         stratStruct st )
  {
    int    i     = 0;
    int    iRow  = horizon.strat.stratStandardTools.getRow(
                     horizon.strat.stratStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st !=null)
    {
      switch (iRow)
      {
        // Stratigraphic Unit ID

        case horizon.strat.stratStandardTools._KID: // KGS Primary KEY
          if (!sData.equals(sNull))
            st.sKID = new String(sData);
          break;
        case horizon.strat.stratStandardTools._KEY: // User defined Primary KEY
          if (!sData.equals(sNull))
            st.sKEY = new String(sData);
          break;
        case horizon.strat.stratStandardTools._ID:  // KGS Stratigraphic Unit ID
          if (!sData.equals(sNull))
            st.sid = new String(sData);
          break;

        // Depth

        case horizon.strat.stratStandardTools._TOPT: // Top Depth
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case horizon.strat.stratStandardTools._TOPB: // Base Depth
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Stratigraphic Unit Names, Rank and Level of confidence of selection

        case horizon.strat.stratStandardTools._RANK:     // Rank
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
          {
            sTemp =  new String(sData.toUpperCase());

            st.iRank = horizon.strat.stratStruct._NONE;
            for (i=0; i<horizon.strat.stratStruct._TOTAL_RANK; i++)
            {
              if (sTemp.equals(horizon.strat.stratStruct.RANK[i]))
                st.iRank = i;
            }
          }
          break;
        case horizon.strat.stratStandardTools._LEVEL:    // Goodness
          if (!sData.equals(sNull))
          {
            sTemp =  new String(sData.toUpperCase());
            for (i=0; i<horizon.strat.stratStruct._TOTAL_LEVEL; i++)
            {
              if (sTemp.equals(horizon.strat.stratStruct.LEVEL[i].toUpperCase()))
                st.iLevel = i;
            }
          }
          break;
        case horizon.strat.stratStandardTools._TOPN:     // Unit Name
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sName = new String(sData);
          break;
        case horizon.strat.stratStandardTools._ALT_NAME: // Alternate Name
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sName2 = new String(sData);
          break;

        // Stratigraphic Unit Age

        case horizon.strat.stratStandardTools._EON:   // EON Age
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sEon = new String(sData);
          break;
        case horizon.strat.stratStandardTools._ERA:   // ERA Age
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sEra = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SYS:   // System
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.system = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SER:   // Series
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.series = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SSYS:  // Subsystem
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.subSystem = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SSER:  // Subseries
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.subSeries = new String(sData);
          break;
        case horizon.strat.stratStandardTools._STG:   // Stage
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.stage = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SPGRP: // Super Group
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sprGroup = new String(sData);
          break;
        case horizon.strat.stratStandardTools._GRP:   // Group
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sGroup = new String(sData);
          break;
        case horizon.strat.stratStandardTools._SGRP:  // Subgroup
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.subGroup = new String(sData);
          break;
        case horizon.strat.stratStandardTools._FORM:  // Formation
//          if (!sData.equals(sNull))
          if (!cmn.cmnString.isNumeric(sData))
            st.sFormation = new String(sData);
          break;
      }
    }

    return (st);
  }

  /** Method mergeTopsData()
   * <p> This method will merge a stratigraphic data list Structure with the
   *     new Stratigraphic Data List Structure
   * @param  dNULL  = LAS 3 Null Value
   * @param  st     = Stratigraphic Data List Structure
   * @param  stTops = Merged Stratigraphic Data List Structure
   * @return stTops = Merged Stratigraphic Data List Structure
   */

  public static stratListStruct mergeTopsData( double dNULL,
                                               stratListStruct st,
                                               stratListStruct stTops )
  {
    int i      = 0;
    int iStart = 0;

    if ((st != null) && (stTops != null))
    {
      for (i=0; i<stTops.iCount; i++)
      {
        if (stTops.stItem[i].depthStart != dNULL)
          iStart = i;
      }

      for (i=0; i<st.iCount; i++)
      {
        if (iStart < stTops.iCount)
        {
          stTops.stItem[iStart] =
              horizon.strat.stratUtility.copy( st.stItem[i] );
          iStart++;
        }
      }
    }

    return (stTops);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Stratigraphic Data List Structure.
   * @param  st    = the Stratigraphic Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( stratListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._TOPS;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Stratigraphic Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( stratListStruct st, las3Struct stT )
  {
    int i,j;

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = 3;      // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            break;
          case _TOPS:
            stT.sParams[i][_VALUE] = new String( st.sTOPS );
            break;
          case _TOPDR:
            stT.sParams[i][_VALUE] = new String( st.sTOPDR );
            break;
        }
      }

    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the Stratigraphic Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( stratListStruct st,
                                      las3Struct      stT,
                                      double          dNull )
  {
    int    i,j;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";
    String sNull       = "" + dNull;

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[horizon.strat.stratStandardTools.TOTAL];
      for (i=0; i<horizon.strat.stratStandardTools.TOTAL; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (i=0; i<st.iCount; i++)
      {
        // Stratigraphic Unit ID

//        if (st.stItem[i].sKID.length() > 1)
//          iSelected[horizon.strat.stratStandardTools._KID]++;
//        if (st.stItem[i].sKEY.length() > 1)
//          iSelected[horizon.strat.stratStandardTools._KEY]++;
//        if (st.stItem[i].sid.length() > 1)
//          iSelected[horizon.strat.stratStandardTools._ID]++;

        // Depth

        iSelected[horizon.strat.stratStandardTools._TOPT]++;
        if (st.stItem[i].depthEnd > 0.0) // Base Depth
          iSelected[horizon.strat.stratStandardTools._TOPB]++;

        // Stratigraphic Unit Names, Rank and Level of confidence of selection

        if (st.stItem[i].iRank != horizon.strat.stratStruct._NONE) // Rank
          iSelected[horizon.strat.stratStandardTools._RANK]++;

        iSelected[horizon.strat.stratStandardTools._LEVEL]++;

        if (st.stItem[i].sName.length() > 0) // Unit Name
          iSelected[horizon.strat.stratStandardTools._TOPN]++;

        if (st.stItem[i].sName2.length() > 0) // Alternate Name
          iSelected[horizon.strat.stratStandardTools._ALT_NAME]++;

        // Stratigraphic Unit Age

        if (st.stItem[i].sEon.length() > 0) // EON Age
          iSelected[horizon.strat.stratStandardTools._EON]++;
        if (st.stItem[i].sEra.length() > 0) // ERA Age
          iSelected[horizon.strat.stratStandardTools._ERA]++;
        if (st.stItem[i].system.length() > 0) // System
          iSelected[horizon.strat.stratStandardTools._SYS]++;
        if (st.stItem[i].series.length() > 0) // Series
          iSelected[horizon.strat.stratStandardTools._SER]++;
        if (st.stItem[i].subSystem.length() > 0) // Subsystem
          iSelected[horizon.strat.stratStandardTools._SSYS]++;
        if (st.stItem[i].subSeries.length() > 0) // Subseries
          iSelected[horizon.strat.stratStandardTools._SSER]++;
        if (st.stItem[i].stage.length() > 0)  // Stage
          iSelected[horizon.strat.stratStandardTools._STG]++;
        if (st.stItem[i].sprGroup.length() > 0) // Super Group
          iSelected[horizon.strat.stratStandardTools._SPGRP]++;
        if (st.stItem[i].sGroup.length() > 0) // Group
          iSelected[horizon.strat.stratStandardTools._GRP]++;
        if (st.stItem[i].subGroup.length() > 0) // Subgroup
          iSelected[horizon.strat.stratStandardTools._SGRP]++;
        if (st.stItem[i].sFormation.length() > 0) // Formation
          iSelected[horizon.strat.stratStandardTools._FORM]++;
      }

      // Count the Number of Curves Populated

      for (i=0; i<horizon.strat.stratStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<horizon.strat.stratStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              horizon.strat.stratStandardTools.CURVES[i][
                horizon.strat.stratStandardTools._C_MNEM]);

          stT.sCurves[iCnt][_UNIT] = new String(
              horizon.strat.stratStandardTools.CURVES[i][
                horizon.strat.stratStandardTools._C_UNIT]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              horizon.strat.stratStandardTools.CURVES[i][
                horizon.strat.stratStandardTools._C_NAME]);

          stT.sCurves[iCnt][_FORMAT] = new String("S");
          if (stT.sCurves[iCnt][_UNIT].length() > 0)
            stT.sCurves[iCnt][_FORMAT] = new String("F");

          iCnt++;
        }
      }

      // Build the Data Section

      stT.iRows    = st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Stratigraphic Units Data List Structure

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          switch (iCurves[j])
          {
            // Stratigraphic Unit ID

            case horizon.strat.stratStandardTools._KID: // KGS Primary KEY
              if (st.stItem[i].sKID.length() > 1)
                stT.sData[i][j] = new String(st.stItem[i].sKID);
              break;
            case horizon.strat.stratStandardTools._KEY: // User defined Primary KEY
              if (st.stItem[i].sKEY.length() > 1)
                stT.sData[i][j] = new String(st.stItem[i].sKEY);
              break;
            case horizon.strat.stratStandardTools._ID: // KGS Stratigraphic Unit ID
              if (st.stItem[i].sid.length() > 1)
                stT.sData[i][j] = new String(st.stItem[i].sid);
              break;

              // Depth

            case horizon.strat.stratStandardTools._TOPT: // Top Depth
              stT.sData[i][j] = new String("" + st.stItem[i].depthStart);
              break;
            case horizon.strat.stratStandardTools._TOPB: // Base Depth
              stT.sData[i][j] = new String("" + st.stItem[i].depthEnd);
              break;

              // Stratigraphic Unit Names, Rank and Level of confidence of selection

            case horizon.strat.stratStandardTools._RANK: // Rank
              if (st.stItem[i].iRank != horizon.strat.stratStruct._NONE) // Rank
                stT.sData[i][j] =
                  new String(horizon.strat.stratStruct.RANK[st.stItem[i].iRank]);
              break;
            case horizon.strat.stratStandardTools._LEVEL: // Goodness
              stT.sData[i][j] =
                new String(horizon.strat.stratStruct.LEVEL[st.stItem[i].iLevel]);
              break;
            case horizon.strat.stratStandardTools._TOPN: // Unit Name
              if ((!st.stItem[i].sName.equals("")) &&
                  (!st.stItem[i].sName.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sName);
              break;
            case horizon.strat.stratStandardTools._ALT_NAME: // Alternate Name
              if ((!st.stItem[i].sName2.equals("")) &&
                  (!st.stItem[i].sName2.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sName2);
              break;

              // Stratigraphic Unit Age

            case horizon.strat.stratStandardTools._EON: // EON Age
              if ((!st.stItem[i].sEon.equals("")) &&
                  (!st.stItem[i].sEon.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sEon);
              break;
            case horizon.strat.stratStandardTools._ERA: // ERA Age
              if ((!st.stItem[i].sEra.equals("")) &&
                  (!st.stItem[i].sEra.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sEra);
              break;
            case horizon.strat.stratStandardTools._SYS: // System
              if ((!st.stItem[i].system.equals("")) &&
                  (!st.stItem[i].system.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].system);
              break;
            case horizon.strat.stratStandardTools._SER: // Series
              if ((!st.stItem[i].series.equals("")) &&
                  (!st.stItem[i].series.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].series);
              break;
            case horizon.strat.stratStandardTools._SSYS: // Subsystem
              if ((!st.stItem[i].subSystem.equals("")) &&
                  (!st.stItem[i].subSystem.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].subSystem);
              break;
            case horizon.strat.stratStandardTools._SSER: // Subseries
              if ((!st.stItem[i].subSeries.equals("")) &&
                  (!st.stItem[i].subSeries.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].subSeries);
              break;
            case horizon.strat.stratStandardTools._STG: // Stage
              if ((!st.stItem[i].stage.equals("")) &&
                  (!st.stItem[i].stage.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].stage);
              break;
            case horizon.strat.stratStandardTools._SPGRP: // Super Group
              if ((!st.stItem[i].sprGroup.equals("")) &&
                  (!st.stItem[i].sprGroup.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sprGroup);
              break;
            case horizon.strat.stratStandardTools._GRP: // Group
              if ((!st.stItem[i].sGroup.equals("")) &&
                  (!st.stItem[i].sGroup.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sGroup);
              break;
            case horizon.strat.stratStandardTools._SGRP: // Subgroup
              if ((!st.stItem[i].subGroup.equals("")) &&
                  (!st.stItem[i].subGroup.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].subGroup);
              break;
            case horizon.strat.stratStandardTools._FORM: // Formation
              if ((!st.stItem[i].sFormation.equals("")) &&
                  (!st.stItem[i].sFormation.equals(sNull)))
                stT.sData[i][j] = new String(st.stItem[i].sFormation);
              break;
          }
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 02/12/2010
 *  @author  John Victorine
 */
