/*
 * @las3LoadShaleData.java Version 1.1 04/16/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratShaleListStruct;
import iqstrat.iqstratShaleStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadShaleData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 04/16/2010
 *  @author  John R. Victorine
 */

public class las3LoadShaleData
{
  public static final int _NONE    = las3.las3Constants._NONE;
  // Plot Scale Depth Values
  public static final int _IQ_KGS  = 0;  // Profile Saved Data Indicator
  // Shale Gamma Ray Values for Colorlith tracks
  public static final int _IQSHMIN = 1;  // Gamma Ray Minimum Value (Default)
  public static final int _IQSHLY  = 2;  // Gamma Ray Shaly Value (Default)
  public static final int _IQSH    = 3;  // Gamma Ray Shale Value (Default)
  public static final int _IQSHHOT = 4;  // Gamma Ray Hot Shale Value (Default)
  public static final int _IQSHMAX = 5;  // Gamma Ray Maximum Value (Default)

  public static final int _PARAM_ROWS  = 6;
  public static final int _PARAM_COLS  = 4;
  public static final String PARAM[][] = {
 // Mnemonic  Description                                     Units     Format
    { "IQKGS",   "Profile Web App Saved Data Indicator",       "",        ""  },
    // Shale Gamma Ray Values for Colorlith tracks
    { "IQSHMIN", "Gamma Ray Minimum Value (Default)",          "API",     "F" },
    { "IQSHLY",  "Gamma Ray Shaly Value (Default)",            "API",     "F" },
    { "IQSH",    "Gamma Ray Shale Value (Default)",            "API",     "F" },
    { "IQSHHOT", "Gamma Ray Hot Shale Value (Default)",        "API",     "F" },
    { "IQSHMAX", "Gamma Ray Maximum Value (Default)",          "API",     "F" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  public static final int _SOURCE  = iqstrat.iqstratTracksStruct._SOURCE;
  public static final int _TYPE    = iqstrat.iqstratTracksStruct._TYPE;
  public static final int _DESCR   = iqstrat.iqstratTracksStruct._DESCRIPTION;
  public static final int _STATUS  = iqstrat.iqstratTracksStruct._STATUS;

  /** Method getData()
   * <p> This method will return the Shale Data Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Shale Data Structure
   */

  public static iqstratShaleListStruct getData( las3Struct stP )
  {
    int i,j,k;
    iqstratShaleListStruct st     = new iqstratShaleListStruct();
    int                    iParam = -1;
    int                    iValue = 0;
    String                 sData  = "";

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][0]))
          {
            iParam = j;

            switch (iParam)
            {
              case _NONE:
                break;

                // Required Fields

              case _IQ_KGS:
                st.sKGS = new String("YES");
                break;
                // Shale Gamma Ray Values for Colorlith tracks
              case  _IQSHMIN:  // Gamma Ray Minimum Value (Default)
              case  _IQSHLY:   // Gamma Ray Shaly Value (Default)
              case  _IQSH:     // Gamma Ray Shale Value (Default)
              case  _IQSHHOT:  // Gamma Ray Hot Shale Value (Default)
              case  _IQSHMAX:  // Gamma Ray Maximum Value (Default)
                break;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Shale Data Structure
   * @param  dNull      = LAS 3 Null Value
   * @param  st         = the LAS 3 Data Structure
   * @param  stCMN      = Common LAS Curve List Structure.
   * @param  stShale  = the Shale Data Structure
   * @return stShale2 = the Shale Data Structure
   */

  public static iqstratShaleListStruct loadData( double dNull,
                                               las3Struct st,
                                               cmnLASCurveListStruct stCMN,
                                               iqstratShaleListStruct  stShale )
  {
    iqstratShaleListStruct stShale2 = null;
    String                 sKEY     = "0";
    int                    iTool    = -1;

    if ((st != null) && (stShale != null) && (stCMN != null))
    {
      stShale2 = iqstrat.iqstratShaleUtility.copyList( stShale );

      stShale.delete();
      stShale = null;

      if (stShale2 == null)
        stShale2 = new iqstratShaleListStruct();

      if (st.iRows > 0)
      {
        stShale2.stItem = new iqstratShaleStruct[st.iRows];
        stShale2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stShale2.stItem[i] = new iqstratShaleStruct();

          // Unique KID & KEY if the LAS File does not contain a value
          // the value has to be present for the program to function correctly
          // the KID and KEY are used to modify and remove a Rock Lithology
          // from the list.

          sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
          stShale2.stItem[i].sKEY = new String( sKEY );

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stShale2.stItem[i] = addData( dNull,
                                            stCMN.stItem[iTool].sMnemonic,
                                            st.sData[i][j],
                                            stShale2.stItem[i] );
            }
          }
        }
      }
    }

    return (stShale2);
  }

  /** Method addData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Control Data Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Shale Data Structure
   * @return st        = the Shale Data Structure
   */

  public static iqstratShaleStruct addData( double dNULL,
                                            String sMnemonic,
                                            String sData,
                                            iqstratShaleStruct st )
  {
    int    iRow  = iqstrat.iqstratShaleStandardTools.getRow(
                     iqstrat.iqstratShaleStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st != null)
    {
      switch (iRow)
      {
        case iqstrat.iqstratShaleStandardTools._SHFT: // Top
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SHFB: // Base
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SHMIN:  // Minimum Value
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.dGammaMin = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SHLY:   // Shaly Value
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.dShaly = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SH:     // Shale Value
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.dShale = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SHHOT:  // Hot Shale Value
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.dHotShale = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratShaleStandardTools._SHMAX:  // Maximum Value
          if (!sData.equals(sNull))
          {
            if (cmn.cmnString.isNumeric(sData))
              st.dGammaMax = cmn.cmnString.stringToDouble(sData);
          }
          break;
      }
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Shale Data Structure.
   * @param  st    = the Shale Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( iqstratShaleListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_SHALE;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( stT );        // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( las3Struct stT )
  {
    int i,j;
    String sData = "";

    if (stT != null)
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams     = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_UNIT]        = new String( PARAM[i][2] );
        stT.sParams[i][_FORMAT]      = new String( PARAM[i][3] );

        switch (i)
        {
          case _IQ_KGS:
            break;
            // Shale Gamma Ray Values for Colorlith tracks
          case _IQSHMIN: // Gamma Ray Minimum Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.GAMMA_MIN );
            break;
          case _IQSHLY:  // Gamma Ray Shaly Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALY );
            break;
          case _IQSH:    // Gamma Ray Shale Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALE );
            break;
          case _IQSHHOT: // Gamma Ray Hot Shale Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALE_HOT );
            break;
          case _IQSHMAX: // Gamma Ray Maximum Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.GAMMA_MAX );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the Shale Data Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( iqstratShaleListStruct st,
                                      las3Struct             stT,
                                      double                 dNull )
  {
    int    i,j;
    int    iCnt  = 0;
    String sData = "";

    // Build Curve Definition Section

    stT.iCurveRows = iqstrat.iqstratShaleStandardTools.TOTAL; // Total rows
    stT.iCurveCols = las3.las3Struct._TOTAL;                     // Total cols
    stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

    for (i=0; i<iqstrat.iqstratShaleStandardTools.TOTAL; i++)
    {
      for (j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sCurves[iCnt][j] = new String("");
      }

      stT.sCurves[iCnt][_MNEMONIC] = new String(
          iqstrat.iqstratShaleStandardTools.CURVES[i][
            iqstrat.iqstratShaleStandardTools._C_MNEM] );

      stT.sCurves[iCnt][_UNIT] = new String(
          iqstrat.iqstratShaleStandardTools.CURVES[i][
            iqstrat.iqstratShaleStandardTools._C_UNIT] );

      stT.sCurves[iCnt][_DESCRIPTION] = new String(
          iqstrat.iqstratShaleStandardTools.CURVES[i][
            iqstrat.iqstratShaleStandardTools._C_NAME] );

      stT.sCurves[iCnt][_FORMAT] = new String(
          iqstrat.iqstratShaleStandardTools.CURVES[i][
            iqstrat.iqstratShaleStandardTools._C_FORM] );

      iCnt++;
    }

    // Build the Data Section

    stT.iRows    = st.iCount;       // Total Number of Rows
    stT.iColumns = stT.iCurveRows;     // Total Number of Columns
    stT.sData    = new String[stT.iRows][stT.iColumns];

    // Initialize the Data Column values to Null

    for (i=0; i<st.iCount; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        stT.sData[i][j] = new String("" + dNull);
      }
    }

    // Load the Data Columns from the Shale Data Structure

    for (i=0; i<st.iCount; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        switch (j)
        {
          case iqstrat.iqstratShaleStandardTools._SHFT: // Top
            stT.sData[i][j] = new String( "" + st.stItem[i].depthStart );
            break;
          case iqstrat.iqstratShaleStandardTools._SHFB: // Base
            stT.sData[i][j] = new String( "" + st.stItem[i].depthEnd );
            break;
          case iqstrat.iqstratShaleStandardTools._SHMIN:  // Minimum Value
            stT.sData[i][j] = new String( "" + st.stItem[i].dGammaMin );
            break;
          case iqstrat.iqstratShaleStandardTools._SHLY:   // Shaly Value
            stT.sData[i][j] = new String( "" + st.stItem[i].dShaly );
            break;
          case iqstrat.iqstratShaleStandardTools._SH:     // Shale Value
            stT.sData[i][j] = new String( "" + st.stItem[i].dShale );
            break;
          case iqstrat.iqstratShaleStandardTools._SHHOT:  // Hot Shale Value
            stT.sData[i][j] = new String( "" + st.stItem[i].dHotShale );
            break;
          case iqstrat.iqstratShaleStandardTools._SHMAX:  // Maximum Value
            stT.sData[i][j] = new String( "" + st.stItem[i].dGammaMax );
            break;
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 04/16/2010
 *  @author  John Victorine
 */
