/*
 * @las3LoadSeqData.java Version 1.1 04/12/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import horizon.seq.seqListStruct;
import horizon.seq.seqStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadSeqData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 04/12/2010
 *  @author  John R. Victorine
 */

public class las3LoadSeqData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _SEQ_SRC     = 1;  // Source of Data
  public static final int _SEQ_DATE    = 2;  // Created Date

  public static final int _PARAM_ROWS  = 3;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic        Description                             Variations
    { "IQKGS",     "Profile Web App Saved Data Indicator",  "", "", "" },
    { "SEQ_SRC",   "Source of Data",                        "", "", "" },
    { "SEQ_DATE",  "Created Date",                          "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;


  /** Method getData()
   * <p> This method will return the Sequence Stratigraphy Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Sequence Stratigraphy Data List Structure
   */

  public static seqListStruct getData( las3Struct stP )
  {
    int i,j,k;
    seqListStruct st     = new seqListStruct();
    int               iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _SEQ_SRC:  // Source of Data
                  st.source = new String( stP.sParams[i][_VALUE] );
                  break;
                case _SEQ_DATE:  // Created Date
                  st.sCreated = new String( stP.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Perforation Data List Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stSeq  = the Sequence Stratigraphy Data List Structure
   * @return stSeq2 = the Sequence Stratigraphy Data List Structure
   */

  public static seqListStruct loadData( double dNull,
                                        las3Struct st,
                                        cmnLASCurveListStruct stCMN,
                                        seqListStruct stSeq )
  {
    seqListStruct stSeq2 = null;
    String            sKEY    = "0";
    int               iTool   = -1;

    if ((st != null) && (stSeq != null) && (stCMN != null))
    {
      stSeq2 = horizon.seq.seqUtility.copyList( stSeq );

      stSeq.delete();
      stSeq = null;

      if (stSeq2 == null)
        stSeq2 = new seqListStruct();

      if (st.iRows > 0)
      {
        stSeq2.stItem = new seqStruct[st.iRows];
        stSeq2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stSeq2.stItem[i] = new seqStruct();

          // Unique KID & KEY if the LAS File does not contain a value
          // the value has to be present for the program to function correctly
          // the KID and KEY are used to modify and remove a Sequence Stratigraphy
          // from the list.

          sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
          stSeq2.stItem[i].sKEY = new String( sKEY );

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stSeq2.stItem[i] = addData( dNull,
                                          stCMN.stItem[iTool].sMnemonic,
                                          st.sData[i][j],
                                          stSeq2.stItem[i] );
            }
          }
        }
      }
    }

    return (stSeq2);
  }

  /** Method addData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Sequence Stratigraphy Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Sequence Stratigraphy Data Structure
   * @return st        = the Sequence Stratigraphy Data Structure
   */

  public static seqStruct addData( double dNULL,
                                   String sMnemonic,
                                   String sData,
                                   seqStruct st )
  {
    int    i     = 0;
    int    iRow  = horizon.seq.seqStandardTools.getRow(
                     horizon.seq.seqStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st !=null)
    {
      switch (iRow)
      {
        case horizon.seq.seqStandardTools._SEQFT: // Top
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case horizon.seq.seqStandardTools._SEQFB: // Base
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case horizon.seq.seqStandardTools._SEQMNEM: // Mnemonic
          st.sMnemonic = new String( sData );
          st.abbrev    = new String( sData );
          break;
        case horizon.seq.seqStandardTools._SEQDES: // Description
          st.sequence = new String( sData );
          break;
      }
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Perforation Data List Structure.
   * @param  st    = the Perforation Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( seqListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_SEQ_STRAT;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Perforation Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( seqListStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            stT.sParams[i][_VALUE] = new String( "YES" );
            break;
          case _SEQ_SRC:  // Source of Data
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case _SEQ_DATE:  // Created Date
            if (st.sCreated.length() == 0)
              st.sCreated = new String( cmn.cmnString.getDate() );

            if (st.sCreated.length() > 0)
            {
              sDate = cmn.cmnString.convertDate( st.sCreated );
              stT.sParams[i][_VALUE]  = new String(sDate[0]);
              stT.sParams[i][_FORMAT] = new String(sDate[1]);
            }
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Perforation Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( seqListStruct st,
                                      las3Struct    stT,
                                      double        dNull )
  {
    int    i,j;
    int    iCnt        = 0;
    String sData       = "";

    stT.iCurveRows = horizon.seq.seqStandardTools.TOTAL;
    stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
    stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

    for (i=0; i<horizon.seq.seqStandardTools.TOTAL; i++)
    {
      for (j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sCurves[iCnt][j] = new String("");
      }

      stT.sCurves[iCnt][_MNEMONIC] = new String(
          horizon.seq.seqStandardTools.CURVES[i][
            horizon.seq.seqStandardTools._C_MNEM]);

      stT.sCurves[iCnt][_UNIT] = new String(
          horizon.seq.seqStandardTools.CURVES[i][
            horizon.seq.seqStandardTools._C_UNIT]);

      stT.sCurves[iCnt][_DESCRIPTION] = new String(
          horizon.seq.seqStandardTools.CURVES[i][
            horizon.seq.seqStandardTools._C_NAME]);

      stT.sCurves[iCnt][_FORMAT] = new String(
          horizon.seq.seqStandardTools.CURVES[i][
            horizon.seq.seqStandardTools._C_FORM]);

      iCnt++;
    }

    // Build the Data Section

    stT.iRows    = st.iCount;       // Total Number of Rows
    stT.iColumns = stT.iCurveRows;  // Total Number of Columns ( iCurveRows )
    stT.sData    = new String[stT.iRows][stT.iColumns];

    // Initialize the Data Column values to Null

    for (i=0; i<st.iCount; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        stT.sData[i][j] = new String("" + dNull);
      }
    }

    // Load the Data Columns from the Sequence Stratigraphy Units Data List Structure

    for (i=0; i<st.iCount; i++)
    {
      stT.sData[i][0] = new String( "" + st.stItem[i].depthStart );
      stT.sData[i][1] = new String( "" + st.stItem[i].depthEnd );

      if (st.stItem[i].sMnemonic.length() > 0)
        stT.sData[i][2] = new String( st.stItem[i].sMnemonic );

      if (st.stItem[i].sequence.length() > 0)
        stT.sData[i][3] = new String( st.stItem[i].sequence );
    }

    return (stT);
  }
}

/*
 *  @version 1.1 04/12/2010
 *  @author  John Victorine
 */
