/*
 * @las3LoadRemarksData.java Version 1.1 04/05/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadRemarksData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 04/05/2010
 *  @author  John R. Victorine
 */

public class las3LoadRemarksData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _GEOSTRT     = 1;  // Starting Display Depth
  public static final int _GEOSTOP     = 2;  // Ending Display Depth
  public static final int _GEOSRC      = 3;  // Source
  public static final int _GEOREF      = 4;  // Depth Reference

  public static final int _PARAM_ROWS  = 5;
  public static final int _PARAM_COLS  = 4;
  public static final String PARAM[][] = {
 //  Mnemonic    Description                             Units      Format
    { "IQKGS",   "Profile Web App Saved Data Indicator",  "",        "S" },
    { "GEOSTRT", "Starting Display Depth",                "F",       "F" },
    { "GEOSTOP", "Ending Display Depth",                  "F",       "F" },
    { "GEOSRC",  "Source",                                "",        "S" },
    { "GEOREF",  "Depth Reference",                       "",        "S" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Remarks Data Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Remarks Data Structure
   */

  public static iqstratRemarkListStruct getData( las3Struct stP )
  {
    int i,j,k;
    iqstratRemarkListStruct st     = new iqstratRemarkListStruct();
    int                     iParam = -1;
    int                     iValue = 0;
    String                  sData  = "";

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][0]))
          {
            iParam = j;

            switch (iParam)
            {
              case _NONE:
                break;

                // Required Fields

              case _IQ_KGS:
                st.sKGS = new String("YES");
                break;
              case _GEOSTRT:  // Starting Display Depth
                if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                  st.depthStart =
                      cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                break;
              case _GEOSTOP:  // Ending Display Depth
                if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                  st.depthEnd =
                      cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                break;
              case _GEOSRC:  // Source
                st.source = new String( stP.sParams[i][_VALUE] );
                break;
              case _GEOREF:  // Depth Reference
                st.sRef = new String( stP.sParams[i][_VALUE] );
                break;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Remarks Data List Structure
   * @param  dNull      = LAS 3 Null Value
   * @param  st         = the LAS 3 Data Structure
   * @param  stCMN      = Common LAS Curve List Structure.
   * @param  stRemarks  = the Remarks Data List Structure
   * @return stRemarks2 = the Remarks Data List Structure
   */

  public static iqstratRemarkListStruct loadData( double dNull,
                                            las3Struct st,
                                            cmnLASCurveListStruct stCMN,
                                            iqstratRemarkListStruct stRemarks )
  {
    iqstratRemarkListStruct stRemarks2 = null;
    String                  sKEY       = "0";
    int                     iTool      = -1;

    if ((st != null) && (stRemarks != null) && (stCMN != null))
    {
      stRemarks2 = iqstrat.iqstratRemarkUtility.copyList( stRemarks );

      stRemarks.delete();
      stRemarks = null;

      if (stRemarks2 == null)
        stRemarks2 = new iqstratRemarkListStruct();

      if (st.iRows > 0)
      {
        stRemarks2.stItem = new iqstratRemarkStruct[st.iRows];
        stRemarks2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stRemarks2.stItem[i] = new iqstratRemarkStruct();

          // Unique KEY if the LAS File does not contain a value
          // the value has to be present for the program to function correctly
          // the KEY is used to modify and remove a Remarks from the list.

          sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
          stRemarks2.stItem[i].sKEY = new String( sKEY );

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stRemarks2.stItem[i] = addData( dNull,
                                              stCMN.stItem[iTool].sMnemonic,
                                              st.sData[i][j],
                                              stRemarks2.stItem[i] );
            }
          }
        }
      }
    }

    return (stRemarks2);
  }

  /** Method addData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Remarks Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Remarks Data Structure
   * @return st        = the Remarks Data Structure
   */

  public static iqstratRemarkStruct addData( double dNULL,
                                             String sMnemonic,
                                             String sData,
                                             iqstratRemarkStruct st )
  {
    int    i     = 0;
    int    iRow  = iqstrat.iqstratRemarkStandardTools.getRow(
                     iqstrat.iqstratRemarkStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st !=null)
    {
      switch (iRow)
      {
        case iqstrat.iqstratRemarkStandardTools._GEOTOP: // Top
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratRemarkStandardTools._GEOBASE: // Base
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratRemarkStandardTools._GEOBED: // Bed thickness
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
             st.dThickness = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case iqstrat.iqstratRemarkStandardTools._GEODES: // Bed description
          if (!sData.equals(sNull))
          {
            st.sText = new String(sData);

            if (st.sText.length() < 41)
              st.sNote = new String(sData);
            else
              st.sNote = new String(sData.substring(0, 40));
          }
          break;
      }
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Remarks Data List Structure.
   * @param  st    = the Remarks Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( iqstratRemarkListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_GEO_REPORT;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Perforation Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters(
                             iqstratRemarkListStruct st, las3Struct stT )
  {
    int i,j;

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_UNIT]        = new String( PARAM[i][2] );
        stT.sParams[i][_FORMAT]      = new String( PARAM[i][3] );

        switch (i)
        {
          case _IQ_KGS:
            stT.sParams[i][_VALUE] = new String( "YES" );
            break;
          case _GEOSTRT:  // Starting Display Depth
            stT.sParams[i][_VALUE]  = new String( "" + st.depthStart );
            break;
          case _GEOSTOP:  // Ending Display Depth
            stT.sParams[i][_VALUE]  = new String( "" + st.depthEnd );
            break;
          case _GEOSRC:  // Source
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case _GEOREF:  // Depth Reference
            stT.sParams[i][_VALUE] = new String( st.sRef );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Remarks Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( iqstratRemarkListStruct  st,
                                      las3Struct               stT,
                                      double                   dNull )
  {
    int    i,j;
    int    iCnt        = 0;
    String sData       = "";

    stT.iCurveRows = iqstrat.iqstratRemarkStandardTools.TOTAL;
    stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
    stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

    for (i=0; i<iqstrat.iqstratRemarkStandardTools.TOTAL; i++)
    {
      for (j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sCurves[iCnt][j] = new String("");
      }

      stT.sCurves[iCnt][_MNEMONIC] = new String(
          iqstrat.iqstratRemarkStandardTools.CURVES[i][1]);

      stT.sCurves[iCnt][_UNIT] = new String(
          iqstrat.iqstratRemarkStandardTools.CURVES[i][3]);

      stT.sCurves[iCnt][_DESCRIPTION] = new String(
          iqstrat.iqstratRemarkStandardTools.CURVES[i][2]);

      stT.sCurves[iCnt][_FORMAT] = new String("S");
      if (i<iqstrat.iqstratRemarkStandardTools.TOTAL-1)
        stT.sCurves[iCnt][_FORMAT] = new String("F");

      iCnt++;
    }

    // Build the Data Section

    stT.iRows    = st.iCount;       // Total Number of Rows
    stT.iColumns = stT.iCurveRows;  // Total Number of Columns ( iCurveRows )
    stT.sData    = new String[stT.iRows][stT.iColumns];

    // Initialize the Data Column values to Null

    for (i=0; i<st.iCount; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        stT.sData[i][j] = new String("" + dNull);
      }
    }

    // Load the Data Columns from the Remarks Data List Structure

    for (i=0; i<st.iCount; i++)
    {
      stT.sData[i][0] = new String( "" + st.stItem[i].depthStart );
      stT.sData[i][1] = new String( "" + st.stItem[i].depthEnd );
      if (st.stItem[i].dThickness > 0.0)
        stT.sData[i][2] = new String( "" + st.stItem[i].dThickness );
      if (st.stItem[i].sText.length() > 0)
        stT.sData[i][3] = new String( st.stItem[i].sText );
    }

    return (stT);
  }
}

/*
 *  @version 1.1 04/05/2010
 *  @author  John Victorine
 */
