/*
 * @las3LoadPfefferData.java Version 1.1 03/16/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;

import pfeffer.pfefferDataListStruct;
import pfeffer.pfefferDataStruct;

import las3.las3ListStruct;
import las3.las3Struct;

import las.lasFileDataStruct;
import lith.lithology.lithologySymbolsListStruct;

/** Class las3LoadPfefferData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 03/16/2010
 *  @author  John R. Victorine
 */

public class las3LoadPfefferData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS  = 0;  // Profile Saved Data Indicator
  public static final int _PF_SRC  = 1;  // Source
  public static final int _PF_NEW  = 2;  // Date Created
  public static final int _PF_SAVE = 3;  // Date Saved

  public static final int _PARAM_ROWS  = 4;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 //   Mnemonic     Description                            Units  Format
    { "IQKGS",    "Profile Web App Saved Data Indicator", "",     "", },
    { "PF_SRC",   "PfEFFER Source (Owner, Group, etc.)",  "",     "S", },
    { "PF_NEW",   "Date Created",                         "DATE", "D", },
    { "PF_SAVE",  "Date Last Saved",                      "DATE", "D", },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;


  /** Method getData()
   * <p> This method will return the PfEFFER Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Stratigraphic Data List Structure
   */

  public static pfefferDataListStruct getData( las3Struct stP )
  {
    int i,j,k;
    pfefferDataListStruct st     = new pfefferDataListStruct();
    int                   iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][0]))
          {
            iParam = j;

            switch (iParam)
            {
              case _NONE:
                break;

              // Required Fields

              case _IQ_KGS:  // Profile Saved Data Indicator
                st.sKGS      = new String( "YES" );
                break;
              case _PF_SRC:  // Source
                st.source    = new String( stP.sParams[i][_VALUE] );
                break;
              case _PF_NEW:  // Date Created
                st.sCreated  = new String( stP.sParams[i][_VALUE] );
                break;
              case _PF_SAVE: // Date Saved
                st.savedLast = new String( stP.sParams[i][_VALUE] );
                break;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     PfEFFER Data List Structure
   * @param  dNull    = LAS 3 Null Value
   * @param  st       = the LAS 3 Data Structure
   * @param  stCMN    = Common LAS Curve List Structure.
   * @param  stL      = LAS File Data Structure
   * @param  stS      = Lithology Symbols List Data Structure
   * @param  stFlow   = the PfEFFER Data List Structure
   * @return stFlow2  = the PfEFFER Data List Structure
   */

  public static pfefferDataListStruct loadData( double dNull,
                                                las3Struct st,
                                                cmnLASCurveListStruct stCMN,
                                                lasFileDataStruct stL,
                                                lithologySymbolsListStruct stS,
                                                pfefferDataListStruct stFlow )
  {
    pfefferDataListStruct stFlow2 = null;
    String                sKEY    = "0";
    int                   iTool   = -1;

    if ((st != null) && (stFlow != null) && (stL != null) && (stCMN != null))
    {
      stFlow2 = pfeffer.pfefferDataUtility.copyList( stFlow );

      stFlow.delete();
      stFlow = null;

      if (stFlow2 == null)
        stFlow2 = new pfefferDataListStruct();

      if (st.iRows > 0)
      {
        stFlow2.stItem = new pfefferDataStruct[st.iRows];
        stFlow2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stFlow2.stItem[i] = new pfefferDataStruct();

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stFlow2.stItem[i] = addFlowData( dNull,
                                               stCMN.stItem[iTool].sMnemonic,
                                               st.sData[i][j],
                                               stS,
                                               stFlow2.stItem[i] );
            }
          }
        }
      }
    }

    return (stFlow2);
  }

  /** Method addFlowData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     PfEFFER Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  stS       = Lithology Symbols List Data Structure
   * @param  st        = the PfEFFER Data Structure
   * @return st        = the PfEFFER Data Structure
   */

  public static pfefferDataStruct addFlowData( double dNULL,
                                               String sMnemonic,
                                               String sData,
                                               lithologySymbolsListStruct stS,
                                               pfefferDataStruct st )
  {
    int    i       = 0;
    int    iRow    = pfeffer.pfefferStandardTools.getPRow(
                       pfeffer.pfefferStandardTools._C_MNEM, sMnemonic );
    String sTemp   = "";
    String sNull   = new String(""+dNULL);
    int    iSymbol = -1;

    if (st !=null)
    {
      switch (iRow)
      {
        // PfEFFER Flow Unit/Zone Identifier

        case pfeffer.pfefferStandardTools._KEY: // Unique Identifiery of Zone
          st.sKEY = new String( sData );
          break;
        case pfeffer.pfefferStandardTools._ZONE: // Flow Unit Name
          if (!sData.equals(sNull))
            st.sZone = new String( sData );
          break;
        case pfeffer.pfefferStandardTools._START: // Starting Depth
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._END: // Ending Depth
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // -- Lithology Symbol

        case pfeffer.pfefferStandardTools._ROCK: // Rock Matrix (Lithology)
          iSymbol =
              lith.lithology.lithologySymbolsUtility.getLithology( sData, stS );
          for (int p=0; p<stS.iCount; p++)
          {
            if (iSymbol == stS.stItem[p].id)
            {
              st.iLithology = p;
            }
          }
          break;

        // Water Model Used

        case pfeffer.pfefferStandardTools._WATER: // Water Model
          for (i=0; i<3; i++)
          {
            if (sData.equals(pfeffer.math.pfefferSwModels.WATER_MODEL[i]))
            {
              st.iWaterModel = i;                   // Water Model Identifier
              st.sWaterModel = new String( sData ); // Water Model Used
            }
          }
          break;

        // Archie Equation Parameters

        case pfeffer.pfefferStandardTools._A: // Archie Constant
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.A = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._M: // Archie Cementation Factor
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.M = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._N: // Archie Saturation Exponent
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.N = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._RW: // Formation water resistivity
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.Rw = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._RSH: // Shale resistivity
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.Rsh = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._PHISH: // Shale porosity
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.Phish = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Total Resistivity Log Curve Value

        case pfeffer.pfefferStandardTools._L_Rt: // Log curve - Resistivity
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iRt = i;
            }
          }
          break;

        // Filtrate Resistivity Log Curve Value

        case pfeffer.pfefferStandardTools._L_Rxo: // Log curve - Resistivity
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iRxo = i;
            }
          }
          break;

        // V-Shale Log Curve Values

        case pfeffer.pfefferStandardTools._L_Vsh: // Log curve - Shale Factor
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iVsh = i;
            }
          }
          break;
        case pfeffer.pfefferStandardTools._CLEAN: // Vsh - Clean Formation Value
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dClean = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._SHALE: // Vsh - Shale Value
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dShale = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Total Porosity Log Curve Values

        case pfeffer.pfefferStandardTools._L_PHIt: // Porosity Type
          for (i=0; i<4; i++)
          {
            if (sData.equals(pfeffer.math.pfefferMath.PHI_TYPE[i]))
              st.iPHIt = i;
          }
          break;
        case pfeffer.pfefferStandardTools._L_PHI1: // Log curve - Porosity
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iPHI1 = i;
            }
          }
          break;
        case pfeffer.pfefferStandardTools._L_PHI2: // Log curve - Porosity - Only for Avg
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iPHI2 = i;
            }
          }
          break;

        case pfeffer.pfefferStandardTools._GRAIN: // Rock Matrix Grain Density
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dGrain = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._FLUID: // Fluid Density
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dFluid = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // -- Total Porosity V-Shale Effect

        case pfeffer.pfefferStandardTools._PHI_VSH: // Porosity V-Shale Effect
          st.iPHIVsh = cmn.cmnStruct._NO;
          if (sData.equals("YES"))
            st.iPHIVsh = cmn.cmnStruct._YES;
          break;
        case pfeffer.pfefferStandardTools._PHI_SH:  // 1st Log Curve Value for Shale
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPHIShale1 = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._PHI_SH2: // 2nd Log Curve Value for Shale
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPHIShale2 = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // -- 2nd Porosity Curve ID

        case pfeffer.pfefferStandardTools._L_2ND: // Log curve - Primary Porosity
          for (i=0; i<las.lasStandardTools.TOTAL_LAS; i++)
          {
            if (sData.equals(las.lasStandardTools.LAS_TOOLS[i][1]))
            {
              st.iPHI2nd = i;
            }
          }
          break;
        case pfeffer.pfefferStandardTools._2_GRAIN: // Rock Matrix Grain Density
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dtGrain = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._2_FLUID: // Fluid Density of 2nd Porosity
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dtFluid = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // -- Total Porosity V-Shale Effect

        case pfeffer.pfefferStandardTools._2_VSH: // 2nd Porosity V-Shale Effect
          st.iPHI2ndVsh = cmn.cmnStruct._NO;
          if (sData.equals("YES"))
            st.iPHI2ndVsh = cmn.cmnStruct._YES;
          break;
        case pfeffer.pfefferStandardTools._2_SH: // 2nd Porosity Log Curve Shale Value
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPHI2ndShale = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Cut-Offs

        case pfeffer.pfefferStandardTools._C_PHI: // Porosity Cut Off
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPhiCut = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._C_SW:  // Water Saturation Cut Off
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dSwCut = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._C_VSH: // Fractional Shale Cut Off
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dVshCut = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._C_BVW: // Bulk Volume Water Cut Off
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dBvwCut = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Wyllie-Rose Equation Parameters

        case pfeffer.pfefferStandardTools._P:
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.P = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._Q:
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.Q = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._R:
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.R = cmn.cmnString.stringToDouble(sData);
          }
          break;

        // Volumetrics Data

        case pfeffer.pfefferStandardTools._V_THK: // Columns as Thickness
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dThickness = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._V_FT:  // Oil or Gas Feet
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dHydroCarbon = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._V_PAY: // Amount of Pay in Feet
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPay = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._V_PHI: // Average Porosity
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dPorosity = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case pfeffer.pfefferStandardTools._V_SW:  // Average Water Saturation
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.dSaturation = cmn.cmnString.stringToDouble(sData);
          }
          break;
      }
    }

    return (st);
  }

  /** Method getDepths()
   * <p> This method will return the depth data for the computation table.
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct getDepths(
                    lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int               iRows = 0;
    lasFileDataStruct st    = null;

    if ((stL != null) && (stP != null))
    {
      st    = las.lasFileDataUtility.copy( stL );

      if (stP.depthEnd > 0.0)
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.iRows  = pfeffer.math.pfefferMath.getRowCount(
                           stP.depthStart, stP.depthEnd, st.depthStep );
            stP.depth  = pfeffer.math.pfefferMath.getDepth(
                           stP.depthStart, stP.depthEnd, st.depthStep, st );
            stP.thick = pfeffer.math.pfefferMath.getThickness(
                           stP.depthStart, stP.depthEnd, st.depthStep );
          }

          st.delete();
          st = null;
        }
      }
    }

    return (stP);
  }

  /** Method getRt()
   * <p> This method will return the True Resistivity for the depth range.
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct getRt(
                     lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int               iRows = 0;
    lasFileDataStruct st    = null;

    if ((stL != null) && (stP != null))
    {
      iRows = stP.iRows;
      st    = las.lasFileDataUtility.copy( stL );

      if ((stP.iRt != -1) && (stP.depthEnd > 0.0))
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.Rt = pfeffer.math.pfefferMath.getRt(
                      stP.depthStart, stP.depthEnd, st.depthStep, stP.iRt, st );
            stP.Rtc = pfeffer.math.pfefferMath.getRt(
                      stP.depthStart, stP.depthEnd, st.depthStep, stP.iRt, st );
          }

          st.delete();
          st = null;
        }
      }
    }

    return (stP);
  }

  /** Method getRxo()
   * <p> This method will return the Filtrate Resistivity Rxo for the depth range.
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct getRxo(
                     lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int               iRows = 0;
    lasFileDataStruct st    = null;

    if ((stL != null) && (stP != null))
    {
      iRows = stP.iRows;
      st    = las.lasFileDataUtility.copy( stL );

      if ((stP.iRt != -1) && (stP.depthEnd > 0.0))
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.Rxo = pfeffer.math.pfefferMath.getRt(
                      stP.depthStart, stP.depthEnd, st.depthStep, stP.iRxo, st );
          }

          st.delete();
          st = null;
        }
      }
    }

    return (stP);
  }

  /** Method getVsh()
   * <p> This method will retrieve the Vsh data
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct getVsh(
                   lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int               iRows = 0;
    lasFileDataStruct st    = null;

    if ((stL != null) && (stP != null))
    {
      iRows = stP.iRows;
      st    = las.lasFileDataUtility.copy( stL );

      if ((stP.iVsh != -1) && (stP.depthEnd > 0.0))
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.Vsh = pfeffer.math.pfefferMath.getVsh(
                        stP.depthStart, stP.depthEnd, st.depthStep,
                        stP.iVsh, stP.dClean, stP.dShale, st );
          }

          st.delete();
          st = null;
        }
      }
    }

    return (stP);
  }

  /** Method getPHIt()
   * <p> This method will retrieve the true porosity data
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct getPHIt(
                   lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int               iRows = 0;
    lasFileDataStruct st    = null;

    if ((stL != null) && (stP != null))
    {
      iRows = stP.iRows;
      st    = las.lasFileDataUtility.copy( stL );

      if ((stP.iPHIt != -1) && (stP.depthEnd > 0.0))
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.PHIt = pfeffer.math.pfefferMath.getPHIt(
                           stP.depthStart, stP.depthEnd, st.depthStep,
                           stP.iPHIt,      stP.iPHI1,    stP.iPHI2,
                           stP.dGrain,     stP.dFluid,
                           stP.dPHIShale1, stP.dPHIShale2,
                           stP.iPHIVsh,    stP.Vsh,      st );

            stP.PHI1st = new double[st.iRows];
            stP.PHI2nd = new double[st.iRows];

            for (int i=0; i<st.iRows; i++)
            {
              stP.PHI1st[i] = 0.0;
              stP.PHI2nd[i] = 0.0;
            }
          }

          st.delete();
          st = null;
        }
      }
    }

    stP = get2ndPHI( stL, stP );

    return (stP);
  }

  /** Method get2ndPHI()
   * <p> This method will retrieve the 2nd porosity data
   * @param  stL  = LAS File Data Structure
   * @param  stP  = PfEFFER Data Structure
   * @return stP  = PfEFFER Data Structure
   */

  public static pfefferDataStruct get2ndPHI(
                   lasFileDataStruct stL, pfefferDataStruct stP )
  {
    int    iRows    = 0;
    double data[]   = null;
    double dPHI[]   = null;
    double data2[]  = null;
    double dVsh[]   = null;
    lasFileDataStruct st = null;

    if ((stL != null) && (stP != null))
    {
      iRows = stP.iRows;
      st    = las.lasFileDataUtility.copy( stL );

      if ((stP.iPHI2nd != -1) && (stP.depthEnd > 0.0))
      {
        if (st != null)
        {
          if (st.iRows > 0)
          {
            stP.PHI1st = pfeffer.math.pfefferMath.getPHIt(
                            stP.depthStart,   stP.depthEnd, st.depthStep,
                            pfeffer.math.pfefferMath._SPHI, stP.iPHI2nd, -1,
                            stP.dtGrain,      stP.dtFluid,
                            stP.dPHI2ndShale, 0.0,
                            stP.iPHI2ndVsh,   stP.Vsh, st );

            stP.PHI2nd = pfeffer.math.pfefferMath.get2ndPHI(
                            stP.iRows, stP.PHI1st, stP.PHIt );
          }

          st.delete();
          st = null;
        }
      }
    }

    return (stP);
  }

  /** Method setFlowData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     PfEFFER Data List Structure.
   * @param  st    = the PfEFFER Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @param  stS   = Lithology Symbols List Data Structure
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setFlowData(
     pfefferDataListStruct st, double dNull, lithologySymbolsListStruct stS )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_FLOW;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );  // Add the Required Parameters
      stT        = addCurves( st, stT, dNull, stS );  // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the PfEFFER Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters(
                  pfefferDataListStruct st, las3Struct stT )
  {
    int i,j;
    String sData = "";

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_UNIT]        = new String( PARAM[i][2] );
        stT.sParams[i][_FORMAT]      = new String( PARAM[i][3] );

        switch (i)
        {
          case _IQ_KGS:
            break;
          case _PF_SRC:  // Source
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case _PF_NEW:  // Date Created
            sData = cmn.cmnString.getDate();
            if (st.sCreated.length() > 0)
              sData = st.sCreated;
            stT.sParams[i][_VALUE] = new String( sData );
            break;
          case _PF_SAVE:  // Date Saved
            sData = cmn.cmnString.getDate();
            stT.sParams[i][_VALUE] = new String( sData );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the PfEFFER Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @param  stS   = Lithology Symbols List Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( pfefferDataListStruct st,
                                      las3Struct            stT,
                                      double                dNull,
                                      lithologySymbolsListStruct stS )
  {
    int    i,j;
    int    iCnt  = 0;
    String sData = "";
    String sTemp = "";
    double dTemp = 0.0;

    if ((st != null) && (stT != null))
    {
      // Build Curve Definition Section

      stT.iCurveRows = pfeffer.pfefferStandardTools.TOTAL_FLOW; // Total rows
      stT.iCurveCols = las3.las3Struct._TOTAL;                   // Total cols
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<pfeffer.pfefferStandardTools.TOTAL_FLOW; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sCurves[iCnt][j] = new String("");
        }

        stT.sCurves[iCnt][_MNEMONIC] = new String(
            pfeffer.pfefferStandardTools.P_CURVES[i][1] );
        stT.sCurves[iCnt][_UNIT] = new String(
            pfeffer.pfefferStandardTools.P_CURVES[i][3] );
        stT.sCurves[iCnt][_DESCRIPTION] = new String(
            pfeffer.pfefferStandardTools.P_CURVES[i][2] );

        stT.sCurves[iCnt][_FORMAT] = new String(
            pfeffer.pfefferStandardTools.P_CURVES[i][6] );

        iCnt++;
      }

      // Build the Data Section

      stT.iRows    = st.iCount;          // Total Number of Rows
      stT.iColumns = stT.iCurveRows;     // Total Number of Columns
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Load the Data Columns from the Control Data Structure

      for (j=0; j<st.iCount; j++)
      {
        // Unique Identifiery of Zone
        stT.sData[j][pfeffer.pfefferStandardTools._KEY] =
            new String( st.stItem[j].sKEY );

        // PfEFFER Flow Unit/Zone Identifier

        // -- Flow Unit Name
        sTemp = st.stItem[j].sZone;
        if (sTemp.length() == 0.0) sTemp = "" + dNull;
        stT.sData[j][pfeffer.pfefferStandardTools._ZONE] = new String( sTemp );
        // -- Starting Depth
        stT.sData[j][pfeffer.pfefferStandardTools._START] =
            new String( "" + st.stItem[j].depthStart );
        // -- Ending Depth
        stT.sData[j][pfeffer.pfefferStandardTools._END] =
            new String( "" + st.stItem[j].depthEnd );

        // Lithology Symbol

        stT.sData[j][pfeffer.pfefferStandardTools._ROCK] =
            new String( "" + dNull );
        if ((stS != null) && (st.stItem[j].iLithology != -1))
        {
          stT.sData[j][pfeffer.pfefferStandardTools._ROCK] =
              new String( stS.stItem[st.stItem[j].iLithology].sName );
        }

        // Water Model Used

        stT.sData[j][pfeffer.pfefferStandardTools._WATER] =
            new String( st.stItem[j].sWaterModel );

        // Archie Equation Parameters

        stT.sData[j][pfeffer.pfefferStandardTools._A] =
            new String( "" + st.stItem[j].A );  // Archie Constant
        stT.sData[j][pfeffer.pfefferStandardTools._M] =
            new String( "" + st.stItem[j].M );  // Archie Cementation Factor
        stT.sData[j][pfeffer.pfefferStandardTools._N] =
            new String( "" + st.stItem[j].N );  // Archie Saturation Exponent
        stT.sData[j][pfeffer.pfefferStandardTools._RW] =
            new String( "" + st.stItem[j].Rw ); // Formation water resistivity
        stT.sData[j][pfeffer.pfefferStandardTools._RSH] =
            new String( "" + st.stItem[j].Rsh );   // Shale resistivity
        stT.sData[j][pfeffer.pfefferStandardTools._PHISH] =
            new String( "" + st.stItem[j].Phish ); // Shale porosity

        // Total Resistivity Log Curve Value

        stT.sData[j][pfeffer.pfefferStandardTools._L_Rt] =
            new String( "" + dNull );
        if (st.stItem[j].iRt > -1)
        {
          stT.sData[j][pfeffer.pfefferStandardTools._L_Rt] =
              new String( las.lasStandardTools.LAS_TOOLS[st.stItem[j].iRt][1] );
        }

        // Filtrate Resistivity Log Curve Value

        stT.sData[j][pfeffer.pfefferStandardTools._L_Rxo] =
            new String( "" + dNull );
        if (st.stItem[j].iRxo > -1)
        {
          stT.sData[j][pfeffer.pfefferStandardTools._L_Rxo] =
              new String( las.lasStandardTools.LAS_TOOLS[st.stItem[j].iRxo][1] );
        }

        // V-Shale Log Curve Values

        stT.sData[j][pfeffer.pfefferStandardTools._L_Vsh] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._CLEAN] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._SHALE] =
            new String( "" + dNull );

        if (st.stItem[j].iVsh > -1)
        {
          stT.sData[j][pfeffer.pfefferStandardTools._L_Vsh] =
            new String( las.lasStandardTools.LAS_TOOLS[st.stItem[j].iVsh][1] );
          stT.sData[j][pfeffer.pfefferStandardTools._CLEAN] =
            new String( "" + st.stItem[j].dClean );
          stT.sData[j][pfeffer.pfefferStandardTools._SHALE] =
            new String( "" + st.stItem[j].dShale );
        }

        // Total Porosity Log Curve Values

        stT.sData[j][pfeffer.pfefferStandardTools._L_PHIt] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._L_PHI1] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._L_PHI2] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._GRAIN] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._FLUID] =
            new String( "" + dNull );

        if (st.stItem[j].iPHIt > -1)
        {
          stT.sData[j][pfeffer.pfefferStandardTools._L_PHIt] =
            new String ( pfeffer.math.pfefferMath.PHI_TYPE[st.stItem[j].iPHIt] );

          if (st.stItem[j].iPHI1 > -1)
          {
            stT.sData[j][pfeffer.pfefferStandardTools._L_PHI1] =
              new String(las.lasStandardTools.LAS_TOOLS[st.stItem[j].iPHI1][1]);
          }

          if (st.stItem[j].iPHI2 > -1)
          {
            stT.sData[j][pfeffer.pfefferStandardTools._L_PHI2] =
              new String(las.lasStandardTools.LAS_TOOLS[st.stItem[j].iPHI2][1]);
          }

          stT.sData[j][pfeffer.pfefferStandardTools._GRAIN] =
              new String( "" + st.stItem[j].dGrain );
          stT.sData[j][pfeffer.pfefferStandardTools._FLUID] =
              new String( "" + st.stItem[j].dFluid );
        }

        // -- Total Porosity V-Shale Effect

        stT.sData[j][pfeffer.pfefferStandardTools._PHI_VSH] = new String( "NO" );
        if (st.stItem[j].iPHIVsh == cmn.cmnStruct._YES)
          stT.sData[j][pfeffer.pfefferStandardTools._PHI_VSH] = new String("YES");
        stT.sData[j][pfeffer.pfefferStandardTools._PHI_SH] =
            new String( "" + st.stItem[j].dPHIShale1 );
        stT.sData[j][pfeffer.pfefferStandardTools._PHI_SH2] =
            new String( "" + st.stItem[j].dPHIShale2 );

        // -- 2nd Porosity Curve ID

        stT.sData[j][pfeffer.pfefferStandardTools._L_2ND] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._2_GRAIN] =
            new String( "" + dNull );
        stT.sData[j][pfeffer.pfefferStandardTools._2_FLUID] =
            new String( "" + dNull );

        if (st.stItem[j].iPHI2nd > -1)
        {
          stT.sData[j][pfeffer.pfefferStandardTools._L_2ND] =
            new String(las.lasStandardTools.LAS_TOOLS[st.stItem[j].iPHI2nd][1]);
          stT.sData[j][pfeffer.pfefferStandardTools._2_GRAIN] =
            new String( "" + st.stItem[j].dtGrain );
          stT.sData[j][pfeffer.pfefferStandardTools._2_FLUID] =
            new String( "" + st.stItem[j].dtFluid );
        }

        // -- Total Porosity V-Shale Effect

        stT.sData[j][pfeffer.pfefferStandardTools._2_VSH] = new String( "NO" );
        if (st.stItem[j].iPHI2ndVsh == cmn.cmnStruct._YES)
          stT.sData[j][pfeffer.pfefferStandardTools._2_VSH] = new String("YES");
        stT.sData[j][pfeffer.pfefferStandardTools._2_SH] =
            new String( "" + st.stItem[j].dPHI2ndShale );

        // Cut-Offs

        stT.sData[j][pfeffer.pfefferStandardTools._C_PHI] =
            new String( "" + st.stItem[j].dPhiCut );
        stT.sData[j][pfeffer.pfefferStandardTools._C_SW] =
            new String( "" + st.stItem[j].dSwCut );
        stT.sData[j][pfeffer.pfefferStandardTools._C_VSH] =
            new String( "" + st.stItem[j].dVshCut );
        stT.sData[j][pfeffer.pfefferStandardTools._C_BVW] =
            new String( "" + st.stItem[j].dBvwCut );

        // Wyllie-Rose Equation Parameters

        stT.sData[j][pfeffer.pfefferStandardTools._P] =
            new String( "" + st.stItem[j].P );
        stT.sData[j][pfeffer.pfefferStandardTools._Q] =
            new String( "" + st.stItem[j].Q );
        stT.sData[j][pfeffer.pfefferStandardTools._R] =
            new String( "" + st.stItem[j].R );

        // Volumetrics Data

        stT.sData[j][pfeffer.pfefferStandardTools._V_THK] =
            new String( "" + st.stItem[j].dThickness );
        stT.sData[j][pfeffer.pfefferStandardTools._V_FT] =
            new String( "" + st.stItem[j].dHydroCarbon );
        stT.sData[j][pfeffer.pfefferStandardTools._V_PAY] =
            new String( "" + st.stItem[j].dPay );
        stT.sData[j][pfeffer.pfefferStandardTools._V_PHI] =
            new String( "" + st.stItem[j].dPorosity );
        stT.sData[j][pfeffer.pfefferStandardTools._V_SW] =
            new String( "" + st.stItem[j].dSaturation );
      }
    }

    return (stT);
  }

  /** Method setPfefferData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     PfEFFER Data List Structure.
   * @param  st    = the PfEFFER Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setPfefferData(
     pfefferDataListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_PFEFFER;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addPfParameters( st, stT );  // Add the Required Parameters
      stT        = addPfCurves( st, stT, dNull );  // Add Curves
    }

    return (stT);
  }

  /** Method addPfParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the PfEFFER Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addPfParameters(
                  pfefferDataListStruct st, las3Struct stT )
  {
    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = 1;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (int j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sParams[0][j] = new String("");
      }

      stT.sParams[0][_MNEMONIC]    = new String( PARAM[0][0] );
      stT.sParams[0][_DESCRIPTION] = new String( PARAM[0][1] );
      stT.sParams[0][_UNIT]        = new String( PARAM[0][2] );
      stT.sParams[0][_FORMAT]      = new String( PARAM[0][3] );
    }

    return (stT);
  }

  /** Method addPfCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the PfEFFER Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addPfCurves( pfefferDataListStruct st,
                                        las3Struct            stT,
                                        double                dNull )
  {
    int    i,j,k;
    int    iCnt   = 0;
    String sData  = "";
    String sTemp  = "";
    double dTemp  = 0.0;
    int    iTotal = 0;

    if ((st != null) && (stT != null))
    {
      // Build Curve Definition Section

      stT.iCurveRows = pfeffer.pfefferStandardTools.TOTAL_PF; // Total rows
      stT.iCurveCols = las3.las3Struct._TOTAL;                 // Total cols
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<pfeffer.pfefferStandardTools.TOTAL_PF; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sCurves[iCnt][j] = new String("");
        }

        stT.sCurves[iCnt][_MNEMONIC] = new String(
            pfeffer.pfefferStandardTools.CURVES[i][1] );
        stT.sCurves[iCnt][_UNIT] = new String(
            pfeffer.pfefferStandardTools.CURVES[i][3] );
        stT.sCurves[iCnt][_DESCRIPTION] = new String(
            pfeffer.pfefferStandardTools.CURVES[i][2] );

        stT.sCurves[iCnt][_FORMAT] = new String(
            pfeffer.pfefferStandardTools.CURVES[i][6] );

        iCnt++;
      }

      // Build the Data Section

      iTotal = 0;

      for (j=0; j<st.iCount; j++)
      {
        iTotal = iTotal + st.stItem[j].iRows;
      }

      stT.iRows    = iTotal;             // Total Number of Rows
      stT.iColumns = stT.iCurveRows;     // Total Number of Columns
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Load the Data Columns from the Control Data Structure

      iCnt = 0;

      for (j=0; j<st.iCount; j++)
      {
        for (k=0; k<st.stItem[j].iRows; k++)
        {
          if (iCnt < iTotal)
          {
            // Unique Identifiery of Zone
            stT.sData[iCnt][pfeffer.pfefferStandardTools._FKEY] =
                new String( st.stItem[j].sKEY+"_"+iCnt );

            stT.sData[iCnt][pfeffer.pfefferStandardTools._DEPTH] =
                new String( "" + st.stItem[j].depth[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._THK] =
                new String( "" + st.stItem[j].thick[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._RT] =
                new String( "" + st.stItem[j].Rt[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._PHIT] =
                new String( "" + st.stItem[j].PHIt[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._VSH] =
                new String( "" + st.stItem[j].Vsh[k] );

            stT.sData[iCnt][pfeffer.pfefferStandardTools._PHI1] =
                new String( "" + st.stItem[j].PHI1st[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._PHI2] =
                new String( "" + st.stItem[j].PHI2nd[k] );

            stT.sData[iCnt][pfeffer.pfefferStandardTools._RWA] =
                new String( "" + st.stItem[j].Rwa[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._RO] =
                new String( "" + st.stItem[j].Ro[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._MA] =
                new String( "" + st.stItem[j].Ma[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._SW] =
                new String( "" + st.stItem[j].Sw[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._BVW] =
                new String( "" + st.stItem[j].BVW[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._PAY] =
                new String( "" + st.stItem[j].Pay[k] );

            stT.sData[iCnt][pfeffer.pfefferStandardTools._PHIR] =
                new String( "" + st.stItem[j].PHIr[k] );

            stT.sData[iCnt][pfeffer.pfefferStandardTools._RXO] =
                new String( "" + st.stItem[j].Rxo[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._SXO] =
                new String( "" + st.stItem[j].Sxo[k] );
            stT.sData[iCnt][pfeffer.pfefferStandardTools._BVF] =
                new String( "" + st.stItem[j].BVF[k] );

            iCnt++;
          }
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 03/16/2010
 *  @author  John Victorine
 */
