/*
 * @las3LoadPerfData.java Version 1.1 02/12/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import horizon.regions.regionsListStruct;
import horizon.regions.regionsStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadPerfData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 02/12/2010
 *  @author  John R. Victorine
 */

public class las3LoadPerfData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _PERFTYPE    = 1;

  public static final int _PARAM_ROWS  = 2;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description           Variations
    { "IQKGS",    "Profile Web App Saved Data Indicator",  "", "", "" },
    { "PERFTYPE", "Charge Type",                           "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;


  /** Method getData()
   * <p> This method will return the Stratigraphic Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Stratigraphic Data List Structure
   */

  public static regionsListStruct getData( las3Struct stP )
  {
    int i,j,k;
    regionsListStruct st     = new regionsListStruct();
    int               iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _PERFTYPE:
                  st.sPERFTYPE = new String( stP.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadParams()
   * <p> This method will load other Tops Section Parameters that are not
   *     specifically added to the controlled variables
   * @param  st  = the Perforations Data List Structure
   * @return st  = the Perforations Data List Structure
   *

  public static regionsListStruct loadParams( regionsListStruct st )
  {
    int i,j,k,m;
    int    iCount      = 0;
    int    iParamRows  = 0;      // Total number of rows
    int    iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
    String sParams[][] = null;   // Array holding the Parameter Definitions
    int    iParam      = -1;

    if (st != null)
    {
      if (st.iParamRows > 0)
      {
        sParams = new String[st.iParamRows][iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<iParamCols; j++)
          {
            sParams[i][j] = new String("");
          }
        }

        for (i=0; i<st.iParamRows; i++)
        {
          iParam = _NONE;
          for (j=0; j<_PARAM_ROWS; j++)
          {
            for (k=0; k<_PARAM_COLS; k++)
            {
              if (st.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
              {
                iParam = j;

                switch (iParam)
                {
                  case _NONE:
                    break;

                  // Required Fields

                  case _IQ_KGS:
                  case _PERFTYPE:
                    break;
                  default:
                    break;
                }
              }
            }
          }

          if (iParam == _NONE)
          {
            for (m=0; m<iParamCols; m++)
            {
              if (iCount < st.iParamRows)
              {
                sParams[iCount][m] = new String(st.sParams[i][m]);
              }
            }

            iCount++;
          }
        }

        if (iCount > 0)
        {
          st.sParams    = new String[iCount][iParamCols];
          st.iParamRows = iCount;
          st.iParamCols = las3.las3Struct._TOTAL;

          for (i=0; i<iCount; i++)
          {
            for (j=0; j<iParamCols; j++)
            {
              st.sParams[i][j] = new String( sParams[i][j] );
            }
          }
        }
      }
    }

    return (st);
  }
*/
  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Perforation Data List Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stPerf  = the Perforations Data List Structure
   * @return stPerf2 = the Perforations Data List Structure
   */

  public static regionsListStruct loadData( double dNull,
                                            las3Struct st,
                                            cmnLASCurveListStruct stCMN,
                                            regionsListStruct stPerf )
  {
    regionsListStruct stPerf2 = null;
    String            sKEY    = "0";
    int               iTool   = -1;

    if ((st != null) && (stPerf != null) && (stCMN != null))
    {
      stPerf2 = horizon.regions.regionsUtility.copyList( stPerf );

      stPerf.delete();
      stPerf = null;

      if (stPerf2 == null)
        stPerf2 = new regionsListStruct();

      if (st.iRows > 0)
      {
        stPerf2.stItem = new regionsStruct[st.iRows];
        stPerf2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stPerf2.stItem[i] = new regionsStruct();

          // Unique KID & KEY if the LAS File does not contain a value
          // the value has to be present for the program to function correctly
          // the KID and KEY are used to modify and remove a Perforations
          // from the list.

          sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
          stPerf2.stItem[i].sKID = new String( sKEY );
          stPerf2.stItem[i].sKEY = new String( sKEY );
          stPerf2.stItem[i].sID  =
              new String(horizon.regions.regionsStruct.PERFORATION );
          stPerf2.stItem[i].id   = horizon.regions.regionsStruct._PERFORATION;

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stPerf2.stItem[i] = addPerfData( dNull,
                                               stCMN.stItem[iTool].sMnemonic,
                                               st.sData[i][j],
                                               stPerf2.stItem[i] );
            }
          }
        }
      }
    }

    return (stPerf2);
  }

  /** Method addPerfData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Perforations Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Perforations Data Structure
   * @return st        = the Perforations Data Structure
   */

  public static regionsStruct addPerfData( double dNULL,
                                           String sMnemonic,
                                           String sData,
                                           regionsStruct st )
  {
    int    i     = 0;
    int    iRow  = horizon.regions.regionsStandardTools.getRow(
                     horizon.regions.regionsStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st !=null)
    {
      switch (iRow)
      {
        case horizon.regions.regionsStandardTools._PERFT: // Top
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depth_top = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case horizon.regions.regionsStandardTools._PERFB: // Base
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
            st.depth_base = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case horizon.regions.regionsStandardTools._PERFD: // Shots
          if (!sData.equals(sNull))
          {
           if (cmn.cmnString.isNumeric(sData))
             st.shots = (int) cmn.cmnString.stringToDouble(sData);
          }
          break;
      }
    }

    return (st);
  }

  /** Method mergePerfData()
   * <p> This method will merge a Perforations data list Structure with the
   *     new Perforations Data List Structure
   * @param  dNULL  = LAS 3 Null Value
   * @param  st     = Perforations Data List Structure
   * @param  stPerf = Merged Perforations Data List Structure
   * @return stPerf = Merged Perforations Data List Structure
   */

  public static regionsListStruct mergePerfData( double dNULL,
                                                 regionsListStruct st,
                                                 regionsListStruct stPerf )
  {
    int i      = 0;
    int iStart = 0;

    if ((st != null) && (stPerf != null))
    {
      for (i=0; i<stPerf.iCount; i++)
      {
        if (stPerf.stItem[i].depth_top != dNULL)
          iStart = i;
      }

      for (i=0; i<st.iCount; i++)
      {
        if (iStart < stPerf.iCount)
        {
          stPerf.stItem[iStart] =
              horizon.regions.regionsUtility.copy( st.stItem[i] );
          iStart++;
        }
      }
    }

    return (stPerf);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Perforation Data List Structure.
   * @param  st    = the Perforation Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( regionsListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._PERFORATION;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Perforation Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( regionsListStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            stT.sParams[i][_VALUE] = new String( "YES" );
            break;
          case _PERFTYPE:
            stT.sParams[i][_VALUE] = new String( st.sPERFTYPE );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Perforation Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( regionsListStruct st,
                                      las3Struct         stT,
                                      double             dNull )
  {
    int    i,j;
    int    iCnt        = 0;
    String sData       = "";

    stT.iCurveRows = horizon.regions.regionsStandardTools.TOTAL;
    stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
    stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

    for (i=0; i<horizon.regions.regionsStandardTools.TOTAL; i++)
    {
      for (j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sCurves[iCnt][j] = new String("");
      }

      stT.sCurves[iCnt][_MNEMONIC] = new String(
          horizon.regions.regionsStandardTools.CURVES[i][1]);

      stT.sCurves[iCnt][_UNIT] = new String(
          horizon.regions.regionsStandardTools.CURVES[i][3]);

      stT.sCurves[iCnt][_DESCRIPTION] = new String(
          horizon.regions.regionsStandardTools.CURVES[i][2]);

      stT.sCurves[iCnt][_FORMAT] = new String("F");

      iCnt++;
    }

    // Build the Data Section

    stT.iRows    = st.iCount;       // Total Number of Rows
    stT.iColumns = stT.iCurveRows;  // Total Number of Columns ( iCurveRows )
    stT.sData    = new String[stT.iRows][stT.iColumns];

    // Initialize the Data Column values to Null

    for (i=0; i<st.iCount; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        stT.sData[i][j] = new String("" + dNull);
      }
    }

    // Load the Data Columns from the Stratigraphic Units Data List Structure

    for (i=0; i<st.iCount; i++)
    {
      stT.sData[i][0] = new String("" + st.stItem[i].depth_top );
      stT.sData[i][1] = new String("" + st.stItem[i].depth_base );
      if (st.stItem[i].shots > 0)
        stT.sData[i][2] = new String("" + st.stItem[i].shots );
    }

    return (stT);
  }
}

/*
 *  @version 1.1 02/12/2010
 *  @author  John Victorine
 */
