/*
 * @las3LoadLogData.java Version 1.1 01/22/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import las.lasFileDataStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadLogData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 01/22/2010
 *  @author  John R. Victorine
 */

public class las3LoadLogData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _START_DEPTH = 0;
  public static final int _END_DEPTH   = 1;
  public static final int _STEP_BY     = 2;
  public static final int _NULL_VALUE  = 3;

  public static final int _WELL_ROWS  = 4;
  public static final int _WELL_COLS  = 5;
  public static final String WELL[][] = {
  // Mnemonic  Description                    Variations
    { "STRT",  "First Index Value",           "", "", "" },
    { "STOP",  "Last Index Value",            "", "", "" },
    { "STEP",  "STEP of index",               "", "", "" },
    { "NULL",  "NULL VALUE",                  "", "", "" },
  };

  // ~PARAMETER Mnemonics in Well Header

  public static final int _PDAT  = 0;
  public static final int _APD   = 1;
  public static final int _DREF  = 2;
  public static final int _EREF  = 3;
  public static final int _TDL   = 4;
  public static final int _TDD   = 5;
  public static final int _CSGL  = 6;
  public static final int _CSGD  = 7;
  public static final int _CSGS  = 8;
  public static final int _CSGW  = 9;
  public static final int _BS    = 10;
  public static final int _MUD   = 11;
  public static final int _MUDS  = 12;
  public static final int _MUDD  = 13;
  public static final int _MUDV  = 14;
  public static final int _FL    = 15;
  public static final int _PH    = 16;
  public static final int _RM    = 17;
  public static final int _RMT   = 18;
  public static final int _RMF   = 19;
  public static final int _RMFT  = 20;
  public static final int _RMC   = 21;
  public static final int _RMCT  = 22;
  public static final int _TMAX  = 23;
  public static final int _TIMC  = 24;
  public static final int _TIML  = 25;
  public static final int _T_UNIT = 26;
  public static final int _BASE  = 27;
  public static final int _ENG   = 28;
  public static final int _WIT   = 29;
  public static final int _GL    = 30;
  public static final int _KB    = 31;
  public static final int _DF    = 32;
  public static final int _RT    = 33;

  public static final int _PARAM_STD   = 30;
  public static final int _PARAM_ROWS  = 34;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description                      Variations
    { "PDAT",  "Permanent Data",               "PDAT2", "LMF", "" },
    { "APD",   "Above Permanent Data",         "APD2", "", "" },
    { "DREF",  "Depth Reference (KB,DF,CB)",   "", "", "" },
    { "EREF",  "Elevation of Depth Reference", "", "", "" },

    { "TDL",   "Total Depth Logger",           "", "", ""  },
    { "TDD",   "Total Depth Driller",          "TD",   "DRDP", "" },

    { "CSGL",  "Casing Bottom Logger",         "CBD1", "CASL", ""  },
    { "CSGD",  "Casing Bottom Driller",        "CBL1", "CASD", ""  },
    { "CSGS",  "Casing Size",                  "", "", ""  },
    { "CSGW",  "Casing Weight",                "", "", ""  },
    { "BS",    "Bit Size",                     "", "", ""  },
    { "MUD",   "Mud type",                     "FTYP", "", ""  },
    { "MUDS",  "Mud Source",                   "", "", ""  },
    { "MUDD",  "Mud Density",                  "DFD", "", ""  },
    { "MUDV",  "Mud Viscosity (Funnel)",       "DFV", "", ""  },
    { "FL",    "Fluid Loss",                   "DFL", "", ""  },
    { "PH",    "PH",                           "DFPH", "FPH", ""  },
    { "RM",    "Resistivity of Mud",           "MRS", "", ""  },
    { "RMT",   "Temperature of Mud",           "MST", "MTP", ""  },
    { "RMF",   "Resistivity of Mud Filtrate",  "MFRS", "", ""  },
    { "RMFT",  "Temperature of Mud Filtrate",  "MFTP", "", ""  },
    { "RMC",   "Resistivity of Mud Cake",      "MCRS", "", ""  },
    { "RMCT",  "Temperature of Mud Cake",      "MCTP", "", ""  },
    { "TMAX",  "Maximum Recorded Temp.",       "MRT", "", ""  },
    { "TIMC",  "Date/Time Circulation Stopped", "TNOC", "", ""  },
    { "TIML",  "Date/Time Logger Tagged Bottom", "", "", ""  },
    { "UNIT",  "Logging Unit Number",          "LOGU", "", ""  },
    { "BASE",  "Home Base of Logging Unit",    "", "", ""  },
    { "ENG",   "Recording Engineer",           "RECB", "", ""  },
    { "WIT",   "Witnessed By",                 "", "", ""  },
    { "GL",    "Ground Level",                 "EGL", "EGL2", "G.L." },
    { "KB",    "Kelly Bushing",                "EKB", "EKB2", "K.B." },
    { "EDF",   "Drilling Rig Floor",           "EDF2", "DFE", "D.F." },
    { "RT",    "Rotary Table",                 "ERT", "R.T.", "" },
  };

//  MDEN.      2.71               :Density Matrix
//  MATR.      LIMESTONE          :Neutron Matrix
//  DTMT.      49                 :Delta T Matrix
//  DTFL.                         :Delta T Fluid
//  ELCO.      10000              :Electron Cutoff

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the LAS File Data Structure from the
   *     LAS 3 Data List Structure.
   * @param  stLAS = the LAS 3 Data List Structure
   * @return st    = the LAS File Data Structure
   */

  public static lasFileDataStruct getData( las3ListStruct stLAS )
  {
    int i,j,k;
    lasFileDataStruct st     = new lasFileDataStruct();
    las3Struct        stH    = null;
    las3Struct        stP    = null;
    int               iWell  = -1;
    int               iParam = -1;

    if (stLAS != null)
    {
      for (i=0; i<stLAS.iCount; i++)
      {
        if (las3.las3Utility.isData( las3.las3Constants._WELL, 0, stLAS ))
          stH = las3.las3Utility.getData( las3.las3Constants._WELL, 0, stLAS );

        if (las3.las3Utility.isData( las3.las3Constants._LOG, 0, stLAS ))
          stP = las3.las3Utility.getData( las3.las3Constants._LOG, 0, stLAS );
      }
    }

    if (stH != null)
    {
      for (i=0; i<stH.iParamRows; i++)
      {
        iWell = _NONE;
        for (j=0; j<_WELL_ROWS; j++)
        {
          for (k=0; k<_WELL_COLS; k++)
          {
            if (stH.sParams[i][_MNEMONIC].toUpperCase().equals(WELL[j][k]))
            {
              iWell = j;

              switch (iWell)
              {
                case _NONE:
                  break;
                case _START_DEPTH:
                  if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                    st.depthStart =
                        cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                  break;
                case _END_DEPTH:
                  if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                    st.depthEnd =
                        cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                  break;
                case _STEP_BY:
                  if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                    st.depthStep =
                        cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                  break;
                case _NULL_VALUE:
                  if (cmn.cmnString.isNumeric(stH.sParams[i][_VALUE]))
                    st.dNull =
                        cmn.cmnString.stringToDouble(stH.sParams[i][_VALUE]);
                  break;
              }
            }
          }
        }
      }
    }

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _PDAT:
                  st.sPDAT = new String( stP.sParams[i][_VALUE] );
                  break;
                case _APD:
                  st.dAPD = convertMtoF( stP.sParams[i][_VALUE],
                                         stP.sParams[i][_UNIT] );
                  break;
                case _DREF:
                  st.sDREF = new String( stP.sParams[i][_VALUE] );
                  break;
                case _EREF:
                  st.dElevation = convertMtoF( stP.sParams[i][_VALUE],
                                               stP.sParams[i][_UNIT] );
                  break;

                // Required parameters for AEUB compliance

                case _TDL:
                  if (st.dTLD == 0.0)
                    st.dTLD  = convertMtoF( stP.sParams[i][_VALUE],
                                            stP.sParams[i][_UNIT] );
                  break;
                case _TDD:
                  if (st.dTDD == 0.0)
                    st.dTDD  = convertMtoF( stP.sParams[i][_VALUE],
                                            stP.sParams[i][_UNIT] );
                  break;

                case _CSGL:
                  st.dCSGL = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sCSGL = new String( stP.sParams[i][_UNIT] );
                  break;
                case _CSGD:
                  st.dCSGD = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sCSGD = new String( stP.sParams[i][_UNIT] );
                  break;
                case _CSGS:
                  st.dCSGS = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sCSGS = new String( stP.sParams[i][_UNIT] );
                  break;
                case _CSGW:
                  st.dCSGW = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sCSGW = new String( stP.sParams[i][_UNIT] );
                  break;
                case _BS:
                  st.dBS = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sBS = new String( stP.sParams[i][_UNIT] );
                  break;
                case _MUD:
                  st.sMUD = new String( stP.sParams[i][_VALUE] );
                  break;
                case _MUDS:
                  st.sMUDS = new String( stP.sParams[i][_VALUE] );
                  break;
                case _MUDD:
                  st.dMUDD = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sMUDD = new String( stP.sParams[i][_UNIT] );
                  break;
                case _MUDV:
                  st.dMUDV = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sMUDV = new String( stP.sParams[i][_UNIT] );
                  break;
                case _FL:
                  st.dFL = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sFL = new String( stP.sParams[i][_UNIT] );
                  break;
                case _PH:
                  st.dPH = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _RM:
                  st.dRM = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRM = new String( stP.sParams[i][_UNIT] );
                  break;
                case _RMT:
                  st.dRMT = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRMT = new String( stP.sParams[i][_UNIT] );
                  break;
                case _RMF:
                  st.dRMF = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRMF = new String( stP.sParams[i][_UNIT] );
                  break;
                case _RMFT:
                  st.dRMFT = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRMFT = new String( stP.sParams[i][_UNIT] );
                  break;
                case _RMC:
                  st.dRMC = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRMC = new String( stP.sParams[i][_UNIT] );
                  break;
                case _RMCT:
                  st.dRMCT = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sRMCT = new String( stP.sParams[i][_UNIT] );
                  break;
                case _TMAX:
                  st.dTMAX = stringToDouble( stP.sParams[i][_VALUE] );
                  st.sTMAX = new String( stP.sParams[i][_UNIT] );
                  break;
                case _TIMC:
                  st.sTIMC = new String( stP.sParams[i][_VALUE] );
                  break;
                case _TIML:
                  st.sTIML = new String( stP.sParams[i][_VALUE] );
                  break;
                case _T_UNIT:
                  st.sUNIT = new String( stP.sParams[i][_VALUE] );
                  break;
                case _BASE:
                  st.sBASE = new String( stP.sParams[i][_VALUE] );
                  break;
                case _ENG:
                  st.sENG = new String( stP.sParams[i][_VALUE] );
                  break;
                case _WIT:
                  st.sWIT = new String( stP.sParams[i][_VALUE] );
                  break;
                default:
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method initData()
   * <p> This method will initialize the LAS File Data Structure Lot Data Arrays
   * @param  iRows = Total Number of Rows
   * @param  stLAS = LAS File Data Structure
   * @return stLAS = LAS File Data Structure
   */

  public static lasFileDataStruct initData( int iRows, lasFileDataStruct stLAS )
  {
    int i=0;

    if ((stLAS != null) && (iRows > 0))
    {
      stLAS.iRows     = iRows;                   // Number of Rows

      stLAS.depths    = new double[stLAS.iRows]; // Array of Depth Data Values
      stLAS.dTime     = new double[stLAS.iRows]; // Array of Time Data Values

      stLAS.dGR       = new double[stLAS.iRows]; // Array of Gamma Ray Log Values
      stLAS.dCGR      = new double[stLAS.iRows]; // Array of Gamma Ray minus Uranium Log Values
      stLAS.dSP       = new double[stLAS.iRows]; // Array of Spontaneous Potential Log Values
      stLAS.dCAL      = new double[stLAS.iRows]; // Array of Caliper Log Values

      stLAS.dNPHI     = new double[stLAS.iRows]; // Array of Neutron Porosity Log Values
      stLAS.dPE       = new double[stLAS.iRows]; // Array of PhotoElectric  Values
      stLAS.dRHOB     = new double[stLAS.iRows]; // Array of Bulk Density  Values
      stLAS.dDPHI     = new double[stLAS.iRows]; // Array of Density Porosity Log Values
      stLAS.dRHO      = new double[stLAS.iRows]; // Array of Density Corrrection Log Values

      stLAS.dDT       = new double[stLAS.iRows]; // Array of Sonic Log Values
      stLAS.dSPHI     = new double[stLAS.iRows]; // Array of Sonic Porosity Log Values

      stLAS.dCOND     = new double[stLAS.iRows]; // Array of Conductivity
      stLAS.dCILD     = new double[stLAS.iRows]; // Array of Deep Induction Conductivity
      stLAS.dCILM     = new double[stLAS.iRows]; // Array of Medium Induction Conductivity

      stLAS.dRES      = new double[stLAS.iRows]; // Array of Resistivity
      stLAS.dRMED     = new double[stLAS.iRows]; // Array of Deep Resistivity
      stLAS.dRDEP     = new double[stLAS.iRows]; // Array of Medium Resistivity
      stLAS.dRSHAL    = new double[stLAS.iRows]; // Array of Shallow Resistivity,

      stLAS.dILD      = new double[stLAS.iRows]; // Array of Deep Induction Resistivity
      stLAS.dILM      = new double[stLAS.iRows]; // Array of Medium Induction Resistivity
      stLAS.dSFLU     = new double[stLAS.iRows]; // Array of Spherically Focused Resistivity

      stLAS.dLL       = new double[stLAS.iRows]; // Array of Deep Laterolog Resistivity
      stLAS.dMLL      = new double[stLAS.iRows]; // Array of Micro Laterolog Resistivity
      stLAS.dLL8      = new double[stLAS.iRows]; // Array of Shallow Laterolog Resistivity

      stLAS.dLN       = new double[stLAS.iRows]; // Array of Long Normal Resistivity
      stLAS.dSN       = new double[stLAS.iRows]; // Array of Shallow Normal Resistivity

      stLAS.dMNOR     = new double[stLAS.iRows]; // Array of Micro Normal Resistivity
      stLAS.dMSFL     = new double[stLAS.iRows]; // Array of Micro Spherically Focused Resistivity
      stLAS.dMINV     = new double[stLAS.iRows]; // Array of Micro Inverse Resistivity

      stLAS.dAHT10    = new double[stLAS.iRows]; // Array Induction Resistivity-10
      stLAS.dAHT20    = new double[stLAS.iRows]; // Array Induction Resistivity-20
      stLAS.dAHT30    = new double[stLAS.iRows]; // Array Induction Resistivity-30
      stLAS.dAHT60    = new double[stLAS.iRows]; // Array Induction Resistivity-60
      stLAS.dAHT90    = new double[stLAS.iRows]; // Array Induction Resistivity-90

      stLAS.dTh       = new double[stLAS.iRows]; // Array of Thorium Log Values
      stLAS.dU        = new double[stLAS.iRows]; // Array of Uranium Log Values
      stLAS.dK        = new double[stLAS.iRows]; // Array of Potassium Log Values

      stLAS.dGRN      = new double[stLAS.iRows]; // Gamma Ray Count
      stLAS.dNEUT     = new double[stLAS.iRows]; // Neutron Counts
      stLAS.dPERM     = new double[stLAS.iRows]; // Permeability

      stLAS.dTEMP     = new double[stLAS.iRows];  // Temperature
      stLAS.dTEMPDIFF = new double[stLAS.iRows];  // Temperature Difference

      stLAS.dUmaa     = new double[stLAS.iRows]; // Apparent Volumetric Photoelectric
      stLAS.dRhomaa   = new double[stLAS.iRows]; // Apparent Matrix Grain Density
      stLAS.dDTmaa    = new double[stLAS.iRows]; // Apparent Matrix Grain Acoustic transit time

      stLAS.dNeutron  = new double[stLAS.iRows]; // Neutron Porosity
      stLAS.delta     = new double[stLAS.iRows]; // Neutron - Density Porosity

      stLAS.dThU      = new double[stLAS.iRows]; // Th/U
      stLAS.dThK      = new double[stLAS.iRows]; // Th/K

      stLAS.dLIN_1    = new double[stLAS.iRows]; // Linear Curve 1
      stLAS.dLIN_2    = new double[stLAS.iRows]; // Linear Curve 2
      stLAS.dLIN_3    = new double[stLAS.iRows]; // Linear Curve 3
      stLAS.dLIN_4    = new double[stLAS.iRows]; // Linear Curve 4

      stLAS.dLOG_1    = new double[stLAS.iRows]; // Log Curve 1
      stLAS.dLOG_2    = new double[stLAS.iRows]; // Log Curve 2
      stLAS.dLOG_3    = new double[stLAS.iRows]; // Log Curve 3
      stLAS.dLOG_4    = new double[stLAS.iRows]; // Log Curve 4

      // Zonation Curves

      stLAS.dPHI      = new double[stLAS.iRows]; // Porosity
      stLAS.dGMCC     = new double[stLAS.iRows]; // Grain Density

      stLAS.dRT       = new double[stLAS.iRows]; // Total Resistivity
      stLAS.dRWA      = new double[stLAS.iRows]; // Apparent Water Resistivity
      stLAS.dRO       = new double[stLAS.iRows]; // Water Saturated Rock Resistivity
      stLAS.dRXRT     = new double[stLAS.iRows]; // Quick Look Ratio
      stLAS.dMA       = new double[stLAS.iRows]; // Archie Cementation
      stLAS.dPHIT     = new double[stLAS.iRows]; // Total Porosity
      stLAS.dPHIE     = new double[stLAS.iRows]; // Effective Porosity
      stLAS.dSW       = new double[stLAS.iRows]; // Water Saturation
      stLAS.dBVW      = new double[stLAS.iRows]; // Bulk Volume Water

      stLAS.dATT1     = new double[stLAS.iRows]; // Att 1
      stLAS.dATT2     = new double[stLAS.iRows]; // Att 2
      stLAS.dATT3     = new double[stLAS.iRows]; // Att 3
      stLAS.dATT4     = new double[stLAS.iRows]; // Att 4
      stLAS.dATT5     = new double[stLAS.iRows]; // Att 5
      stLAS.dATT6     = new double[stLAS.iRows]; // Att 6

      stLAS.dVCAL     = new double[stLAS.iRows]; // Calcite Volume Fraction
      stLAS.dVQTZ     = new double[stLAS.iRows]; // Quartz Volume Fraction
      stLAS.dVDOL     = new double[stLAS.iRows]; // Dolomite Volume Fraction
      stLAS.dVSH      = new double[stLAS.iRows]; // Shale Volume Fraction
      stLAS.dVANH     = new double[stLAS.iRows]; // Anhydrite Volume Fraction

      for (i=0; i<stLAS.iRows; i++)
      {
        stLAS.depths[i]    = stLAS.dNull; // Array of Depth Data Values
        stLAS.dTime[i]     = stLAS.dNull;  // Array of Time Data Values

        stLAS.dGR[i]       = stLAS.dNull; // Array of Gamma Ray Log Values
        stLAS.dCGR[i]      = stLAS.dNull; // Array of Gamma Ray minus Uranium Log Values
        stLAS.dSP[i]       = stLAS.dNull; // Array of Spontaneous Potential Log Values
        stLAS.dCAL[i]      = stLAS.dNull; // Array of Caliper Log Values

        stLAS.dNPHI[i]     = stLAS.dNull; // Array of Neutron Porosity Log Values
        stLAS.dPE[i]       = stLAS.dNull; // Array of PhotoElectric  Values
        stLAS.dRHOB[i]     = stLAS.dNull; // Array of Bulk Density  Values
        stLAS.dDPHI[i]     = stLAS.dNull; // Array of Density Porosity Log Values
        stLAS.dRHO[i]      = stLAS.dNull; // Array of Density Corrrection Log Values

        stLAS.dDT[i]       = stLAS.dNull; // Array of Sonic Log Values
        stLAS.dSPHI[i]     = stLAS.dNull; // Array of Sonic Porosity Log Values

        stLAS.dCOND[i]     = stLAS.dNull; // Array of Conductivity
        stLAS.dCILD[i]     = stLAS.dNull; // Array of Deep Induction Conductivity
        stLAS.dCILM[i]     = stLAS.dNull; // Array of Medium Induction Conductivity

        stLAS.dRES[i]      = stLAS.dNull; // Array of Resistivity
        stLAS.dRMED[i]     = stLAS.dNull; // Array of Deep Resistivity
        stLAS.dRDEP[i]     = stLAS.dNull; // Array of Medium Resistivity
        stLAS.dRSHAL[i]    = stLAS.dNull; // Array of Shallow Resistivity,

        stLAS.dILD[i]      = stLAS.dNull; // Array of Deep Induction Resistivity
        stLAS.dILM[i]      = stLAS.dNull; // Array of Medium Induction Resistivity
        stLAS.dSFLU[i]     = stLAS.dNull; // Array of Spherically Focused Resistivity

        stLAS.dLL[i]       = stLAS.dNull; // Array of Deep Laterolog Resistivity
        stLAS.dMLL[i]      = stLAS.dNull; // Array of Micro Laterolog Resistivity
        stLAS.dLL8[i]      = stLAS.dNull; // Array of Shallow Laterolog Resistivity

        stLAS.dLN[i]       = stLAS.dNull; // Array of Long Normal Resistivity
        stLAS.dSN[i]       = stLAS.dNull; // Array of Shallow Normal Resistivity

        stLAS.dMNOR[i]     = stLAS.dNull; // Array of Micro Normal Resistivity
        stLAS.dMSFL[i]     = stLAS.dNull; // Array of Micro Spherically Focused Resistivity
        stLAS.dMINV[i]     = stLAS.dNull; // Array of Micro Inverse Resistivity

        stLAS.dAHT10[i]    = stLAS.dNull; // Array Induction Resistivity-10
        stLAS.dAHT20[i]    = stLAS.dNull; // Array Induction Resistivity-20
        stLAS.dAHT30[i]    = stLAS.dNull; // Array Induction Resistivity-30
        stLAS.dAHT60[i]    = stLAS.dNull; // Array Induction Resistivity-60
        stLAS.dAHT90[i]    = stLAS.dNull; // Array Induction Resistivity-90

        stLAS.dTh[i]       = stLAS.dNull; // Array of Thorium Log Values
        stLAS.dU[i]        = stLAS.dNull; // Array of Uranium Log Values
        stLAS.dK[i]        = stLAS.dNull; // Array of Potassium Log Values

        stLAS.dGRN[i]      = stLAS.dNull; // Gamma Ray Count
        stLAS.dNEUT[i]     = stLAS.dNull; // Neutron Counts

        stLAS.dPERM[i]     = stLAS.dNull; // Permeability

        stLAS.dTEMP[i]     = stLAS.dNull; // Temperature
        stLAS.dTEMPDIFF[i] = stLAS.dNull; // Temperature Difference

        stLAS.dUmaa[i]     = stLAS.dNull; // Apparent Volumetric Photoelectric
        stLAS.dRhomaa[i]   = stLAS.dNull; // Apparent Matrix Grain Density
        stLAS.dDTmaa[i]    = stLAS.dNull; // Apparent Matrix Grain Acoustic transit time

        stLAS.dNeutron[i]  = stLAS.dNull; // Neutron Porosity
        stLAS.delta[i]     = stLAS.dNull; // Neutron - Density Porosity

        stLAS.dLIN_1[i]    = stLAS.dNull; // Linear Curve 1
        stLAS.dLIN_2[i]    = stLAS.dNull; // Linear Curve 2
        stLAS.dLIN_3[i]    = stLAS.dNull; // Linear Curve 3
        stLAS.dLIN_4[i]    = stLAS.dNull; // Linear Curve 4

        stLAS.dLOG_1[i]    = stLAS.dNull; // Log Curve 1
        stLAS.dLOG_2[i]    = stLAS.dNull; // Log Curve 2
        stLAS.dLOG_3[i]    = stLAS.dNull; // Log Curve 3
        stLAS.dLOG_4[i]    = stLAS.dNull; // Log Curve 4

        // Zonation Curves

        stLAS.dPHI[i]      = stLAS.dNull; // Porosity
        stLAS.dGMCC[i]     = stLAS.dNull; // Grain Density

        stLAS.dRT[i]       = stLAS.dNull; // Total Resistivity
        stLAS.dRWA[i]      = stLAS.dNull; // Apparent Water Resistivity
        stLAS.dRO[i]       = stLAS.dNull; // Water Saturated Rock Resistivity
        stLAS.dRXRT[i]     = stLAS.dNull; // Quick Look Ratio
        stLAS.dMA[i]       = stLAS.dNull; // Archie Cementation
        stLAS.dPHIT[i]     = stLAS.dNull; // Total Porosity
        stLAS.dPHIE[i]     = stLAS.dNull; // Effective Porosity
        stLAS.dSW [i]      = stLAS.dNull; // Water Saturation
        stLAS.dBVW[i]      = stLAS.dNull; // Bulk Volume Water

        stLAS.dATT1[i]     = stLAS.dNull; // Att 1
        stLAS.dATT2[i]     = stLAS.dNull; // Att 2
        stLAS.dATT3[i]     = stLAS.dNull; // Att 3
        stLAS.dATT4[i]     = stLAS.dNull; // Att 4
        stLAS.dATT5[i]     = stLAS.dNull; // Att 5
        stLAS.dATT6[i]     = stLAS.dNull; // Att 6

        stLAS.dVCAL[i]     = stLAS.dNull; // Calcite Volume Fraction
        stLAS.dVQTZ[i]     = stLAS.dNull; // Quartz Volume Fraction
        stLAS.dVDOL[i]     = stLAS.dNull; // Dolomite Volume Fraction
        stLAS.dVSH[i]      = stLAS.dNull; // Shale Volume Fraction
        stLAS.dVANH[i]     = stLAS.dNull; // Anhydrite Volume Fraction
      }

      if (stLAS.depthStep > 0.0)
      {
        for (i=0; i<stLAS.iRows; i++)
        {
          stLAS.depths[i] = stLAS.depthStart + i * stLAS.depthStep;
        }
      }
    }

    return (stLAS);
  }

  /** Method addData()
   * <p> This method will add the LAS 3 Data Structure to the LAS Data Structure
   * @param  sData = Array of Log Data From each row
   * @param  iColumns = Total number of Log Curves
   * @param  stC   = LAS File Data Structure - File Curve Order Matched to type
   * @param  stLAS = LAS File Data Structure
   * @return stLAS = LAS File Data Structure
   */

  public static lasFileDataStruct addData( String            sData[],
                                           int               iColumns,
                                           lasFileDataStruct stC,
                                           lasFileDataStruct stLAS )
  {
    int    iRow     = -1;
    double data     = 0.0;
    double dRange[] = { 0.0, 0.0 };
    double diff     = 0.0;

    if (sData != null)
    {
      iRow = -1;
      data = stringToDouble( sData[0], stLAS.dNull );
      for (int i=0; i<stLAS.iRows; i++)
      {
        diff = Math.abs( data - stLAS.depths[i] );
//        if (data == stLAS.depths[i])
        if ( diff < .00001 )
          iRow = i;
      }

      if (iRow > -1)
      {
        for (int j=1; j<iColumns; j++)
        {
          data = stringToDouble(sData[j], stLAS.dNull);

          // Depth Data Values
          if (j == 0)
            stLAS.depths[iRow] = data;

          if (stC.iGR == j) // Gamma Ray Log
          {
            stLAS.dGR[iRow] = data;
//System.out.println(iRow+" "+stLAS.depths[iRow]+" "+"["+data+"],");
		  }
          if (stC.iCGR == j) // Gamma Ray m U Log
            stLAS.dCGR[iRow] = data;
          if (stC.iSP == j)  // Spontaneous Potential
          {
            stLAS.dSP[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dSP, stLAS.dNull);
            stLAS.dSPMin = dRange[0];
            stLAS.dSPMax = dRange[1];
          }
          if (stC.iCAL == j) // Caliper
            stLAS.dCAL[iRow] = data;

          if (stC.iNPHI == j)  // Neutron Porosity
          {
            stLAS.dNPHI[iRow] = data;

            if (stLAS.dNPHI[iRow] > .5)
              stLAS.dPHIMax = .7;
          }
          if (stC.iPE == j) // PhotoElectric
            stLAS.dPE[iRow] = data;
          if (stC.iDPHI == j) // Density Porosity
            stLAS.dDPHI[iRow] = data;
          if (stC.iRHOB == j)  // Bulk Density
          {
            stLAS.dRHOB[iRow] = data;

//            if (stLAS.dRHOB[iRow] > stLAS.dNull)
//            {
//              if (stLAS.dRHOB[iRow] < 2.0)
//              {
//                stLAS.dRhoMin = 0.0;
//                stLAS.dRhoMax = 4.0;
//              }
//            }
          }
          if (stC.iRHO == j) // Density Correction
            stLAS.dRHO[iRow] = data;

          if (stC.iDT == j) // Sonic Log
            stLAS.dDT[iRow] = data;
          if (stC.iSPHI == j) // Sonic Porosity
            stLAS.dSPHI[iRow] = data;

          if (stC.iCOND == j) // Conductivity
            stLAS.dCOND[iRow] = data;
          if (stC.iCILD == j) // Deep Induction Conductivity
            stLAS.dCILD[iRow] = data;
          if (stC.iCILM == j) // Medium Induction Conductivity
            stLAS.dCILM[iRow] = data;

          if (stC.iRES == j) // Resistivity
            stLAS.dRES[iRow] = data;
          if (stC.iRDEP == j) // Deep Resistivity
            stLAS.dRDEP[iRow] = data;
          if (stC.iRMED == j) // Medium Resistivity
            stLAS.dRMED[iRow] = data;
          if (stC.iRSHAL == j) // Shallow Resistivity,
            stLAS.dRSHAL[iRow] = data;

          if (stC.iILD == j) // Deep Induction Resistivity
            stLAS.dILD[iRow] = data;
          if (stC.iILM == j) // Medium Induction Resistivity
            stLAS.dILM[iRow] = data;
          if (stC.iSFLU == j) // Spherically Focused Resistivity
            stLAS.dSFLU[iRow] = data;

          if (stC.iLL == j) // Deep Laterolog Resistivity
            stLAS.dLL[iRow] = data;
          if (stC.iMLL == j) // Micro Laterolog Resistivity
            stLAS.dMLL[iRow] = data;
          if (stC.iLL8 == j) // Shallow Laterolog Resistivity
            stLAS.dLL8[iRow] = data;

          if (stC.iLN == j) // Long Normal Resistivity
            stLAS.dLN[iRow] = data;
          if (stC.iSN == j) // Shallow Normal Resistivity
            stLAS.dSN[iRow] = data;

          if (stC.iMNOR == j) // Micro Normal Resistivity
            stLAS.dMNOR[iRow] = data;
          if (stC.iMSFL == j) // Micro Spherically Focused Resistivity
            stLAS.dMSFL[iRow] = data;
          if (stC.iMINV == j) // Micro Inverse Resistivity
            stLAS.dMINV[iRow] = data;

          if (stC.iAHT10 == j) // Array Induction Resistivity-10
            stLAS.dAHT10[iRow] = data;
          if (stC.iAHT20 == j) // Array Induction Resistivity-20
            stLAS.dAHT20[iRow] = data;
          if (stC.iAHT30 == j) // Array Induction Resistivity-30
            stLAS.dAHT30[iRow] = data;
          if (stC.iAHT60 == j) // Array Induction Resistivity-60
            stLAS.dAHT60[iRow] = data;
          if (stC.iAHT90 == j) // Array Induction Resistivity-90
            stLAS.dAHT90[iRow] = data;

          if (stC.iTh == j) // Thorium
            stLAS.dTh[iRow] = data;
          if (stC.iU == j) // Uranium
            stLAS.dU[iRow] = data;
          if (stC.iK == j) // Potasium
            stLAS.dK[iRow] = data;

          if (stC.iGRN == j) // Gamma Ray Count
            stLAS.dGRN[iRow] = data;
          if (stC.iNEUT == j) // Neutron Counts
            stLAS.dNEUT[iRow] = data;
          if (stC.iPERM == j) // Permeability
            stLAS.dPERM[iRow] = data;

          if (stC.iTEMP == j)  // Temperature
          {
            stLAS.dTEMP[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dTEMP, stLAS.dNull);
            stLAS.dTEMPMin = dRange[0];
            stLAS.dTEMPMax = dRange[1];

            stLAS.dTEMPDIFF = math.mathLAS.computeTempDiff( stLAS.iRows, stLAS.dTEMP, stLAS.dNull );
            stLAS.iTEMPDIFF = 610;
          }

          if (stC.iTEMPDIFF == j)  // Temperature Difference
          {
            stLAS.dTEMPDIFF[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dTEMPDIFF, stLAS.dNull);
            stLAS.dTEMPDIFFMin = dRange[0];
            stLAS.dTEMPDIFFMax = dRange[1];

	        if (stLAS.dTEMPDIFFMin == 0.0)
	          stLAS.dTEMPDIFFMin = -1;
	        if (stLAS.dTEMPDIFFMax == 0.0)
	          stLAS.dTEMPDIFFMax = 1;
          }

          if (stC.iLIN_1 == j)  // Unknown Linear Curve 1
          {
            stLAS.dLIN_1[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dLIN_1, stLAS.dNull);
            stLAS.dLIN_1Min = dRange[0];
            stLAS.dLIN_1Max = dRange[1];
          }
          if (stC.iLIN_2 == j)  // Unknown Linear Curve 2
          {
            stLAS.dLIN_2[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dLIN_2, stLAS.dNull);
            stLAS.dLIN_2Min = dRange[0];
            stLAS.dLIN_2Max = dRange[1];
          }
          if (stC.iLIN_3 == j)  // Unknown Linear Curve 3
          {
            stLAS.dLIN_3[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dLIN_3, stLAS.dNull);
            stLAS.dLIN_3Min = dRange[0];
            stLAS.dLIN_3Max = dRange[1];
          }
          if (stC.iLIN_4 == j)  // Unknown Linear Curve 4
          {
            stLAS.dLIN_4[iRow] = data;
            dRange = math.mathLAS.computeCurveMinMax(stLAS.dLIN_4, stLAS.dNull);
            stLAS.dLIN_4Min = dRange[0];
            stLAS.dLIN_4Max = dRange[1];
          }

          if (stC.iLOG_1 == j)  // Unknown Log Curve 1
          {
            stLAS.dLOG_1[iRow] = data;
            dRange = math.mathLAS.computeLogMinMax(stLAS.dLOG_1, stLAS.dNull);
            stLAS.dLOG_1Min = dRange[0];
            stLAS.dLOG_1Max = dRange[1];
          }
          if (stC.iLOG_2 == j)  // Unknown Log Curve 2
          {
            stLAS.dLOG_2[iRow] = data;
            dRange = math.mathLAS.computeLogMinMax(stLAS.dLOG_2, stLAS.dNull);
            stLAS.dLOG_2Min = dRange[0];
            stLAS.dLOG_2Max = dRange[1];
          }
          if (stC.iLOG_3 == j)  // Unknown Log Curve 3
          {
            stLAS.dLOG_3[iRow] = data;
            dRange = math.mathLAS.computeLogMinMax(stLAS.dLOG_3, stLAS.dNull);
            stLAS.dLOG_3Min = dRange[0];
            stLAS.dLOG_3Max = dRange[1];
          }
          if (stC.iLOG_4 == j)  // Unknown Log Curve 4
          {
            stLAS.dLOG_4[iRow] = data;
            dRange = math.mathLAS.computeLogMinMax(stLAS.dLOG_4, stLAS.dNull);
            stLAS.dLOG_4Min = dRange[0];
            stLAS.dLOG_4Max = dRange[1];
          }

          if (stC.iPHI  == j)   // Porosity
            stLAS.dPHI[iRow]   = data;
          if (stC.iGMCC == j)   // Grain Density
            stLAS.dGMCC[iRow]  = data;

          if (stC.iRT == j)
            stLAS.dRT[iRow]    = data;
          if (stC.iRWA == j)
            stLAS.dRWA[iRow]   = data;
          if (stC.iRO == j)
            stLAS.dRO[iRow]    = data;
          if (stC.iRXRT == j)
          {
            stLAS.dRXRT[iRow]  = data;
		  }
          if (stC.iMA == j)
            stLAS.dMA[iRow]    = data;
          if (stC.iPHIT == j)
            stLAS.dPHIT[iRow]  = data;
          if (stC.iPHIE == j)
            stLAS.dPHIE[iRow]  = data;
          if (stC.iSW == j)
            stLAS.dSW[iRow]    = data;
          if (stC.iBVW == j)
            stLAS.dBVW[iRow]   = data;

          if (stC.iATT1 == j)
            stLAS.dATT1[iRow]  = data;
          if (stC.iATT2 == j)
            stLAS.dATT2[iRow]  = data;
          if (stC.iATT3 == j)
            stLAS.dATT3[iRow]  = data;
          if (stC.iATT4 == j)
            stLAS.dATT4[iRow]  = data;
          if (stC.iATT5 == j)
            stLAS.dATT5[iRow]  = data;
          if (stC.iATT6 == j)
            stLAS.dATT6[iRow]  = data;

          if (stC.iVCAL == j) // Calcite Volume Fraction
            stLAS.dVCAL[iRow]  = data;
          if (stC.iVQTZ == j) // Quartz Volume Fraction
            stLAS.dVQTZ[iRow]  = data;
          if (stC.iVDOL == j) // Dolomite Volume Fraction
            stLAS.dVDOL[iRow]  = data;
          if (stC.iVSH == j) // Shale Volume Fraction
            stLAS.dVSH[iRow]   = data;
          if (stC.iVANH == j) // Anhydrite Volume Fraction
            stLAS.dVANH[iRow]  = data;
        }
      }
    }

    return (stLAS);
  }

  /** Method modifyLogData()
   * <p> This method will check the original log data and modify or compute
   *     other terms depending on the available data
   * @param  st = LAS File Data Structure
   * @return st = LAS File Data Structure
   */

  public static lasFileDataStruct modifyLogData( lasFileDataStruct st )
  {
    int    i        = 0;
    int    idata    = 0;
    double dRange[] = { 0.0, 0.0 };
    int    iTemp    = 0;

    if (st != null)
    {
      // Convert Potassium from decimal to percent

      st.dK = math.mathLAS.convertPotassium( st.iRows, st.dK, st.dNull );

      // Compute Gamma Ray API from Spectral Gamma Ray Curves

      if ((st.iGR == -1) && (st.iTh > -1) && (st.iU > -1) && (st.iK > -1))
      {
        st.dGR = math.mathLAS.computeGR( st.iRows,
                                         st.iGR, st.dGR,
                                         st.iK,  st.dK,
                                         st.iTh, st.dTh,
                                         st.iU,  st.dU,
                                         st.dNull );
        st.iGR = 100;
      }

      // Compute Gamma Ray API from Gamma Ray Counts

      if ((st.iGR == -1) && (st.iGRN  > -1))
      {
        dRange = math.mathLAS.computeCurveMinMax( st.dGRN, st.dNull );
        st.dGRNMinP = (double) ((int) dRange[0]);
        st.dGRNMaxP = (double) ((int) (dRange[1]+1));

        st.dGR = math.mathLAS.computeGR( st.dGRN, st.dNull, 150.0, 0.0 );
        if (st.dGR != null) { st.iGR = 100; }
      }

      // Compute CGR API from Spectral Gamma Ray Curves

      if ((st.iCGR == -1) && (st.iTh > -1) && (st.iK > -1))
      {
        st.dCGR = math.mathLAS.computeCGR( st.iRows,
                                           st.iCGR, st.dCGR,
                                           st.iK,   st.dK,
                                           st.iTh,  st.dTh,
                                           st.dNull );
        st.iCGR = 110;
      }

      // Check to see if the Neutron Porosity in Percent or Fractions.

      st.dNPHI = math.mathLAS.convertPorosity( st.iRows, st.dNPHI, st.dNull );

      // Compute Neutron Porosity from Neutron Counts

      if ((st.iNPHI == -1) && (st.iNEUT  > -1))
      {
        dRange = math.mathLAS.computeCurveMinMax( st.dNEUT, st.dNull );
        st.dNEUTMinP = (double) ((int) dRange[0]);
        st.dNEUTMaxP = (double) ((int) (dRange[1] + 1.0));

        st.dNPHI = math.mathLAS.computeNPHI( st.dNEUT, st.dNull, 0.4, 0.01 );
        if (st.dNPHI != null) { st.iNPHI = 400; }
      }

      // Check to see if the Sonic Porosity in Percent or Fractions.

      st.dSPHI = math.mathLAS.convertPorosity( st.iRows, st.dSPHI, st.dNull );

      // Check to see if the Density Porosity in Percent or Fractions.

      st.dDPHI = math.mathLAS.convertPorosity( st.iRows, st.dDPHI, st.dNull );

      // If the Density Porosity (DPHI) is not present but
      //    the Bulk Density log (RHOB) is present then

      if ((st.iRHOB > -1) && (st.iDPHI == -1))
      {
        st.dDPHI = math.mathLAS.computePHI( st.iRows,
                                            st.dRHOB,
                                            math.mathLAS._DENSITY,
                                            math.mathLAS._CALCITE,
                                            math.mathLAS._FRESH,
                                            st.dNull );
        st.iDPHI = 200;
      }

      // If the Bulk Density log (RHOB) is missing but
      //    the Density Porosity (DPHI) is present then

      else if ((st.iRHOB == -1) && (st.iDPHI > -1))
      {
        st.dRHOB = math.mathLAS.computeLOG( st.iRows,
                                            st.dDPHI,
                                            math.mathLAS._DENSITY,
                                            math.mathLAS._CALCITE,
                                            math.mathLAS._FRESH,
                                            st.dNull );
        st.iRHOB = 210;
/*
        for (i=0; i<st.iRows; i++)
        {
          if (st.dRHOB[i] > st.dNull)
          {
//             if (st.dRHOB[i] < 2.0)
//             {
//               st.dRhoMin = 0.0;
//               st.dRhoMax = 4.0;
//             }
          }
        }
*/
      }

      // If the Sonic Porosity (SPHI) is not present but
      //    the Acoustic Transit Time log (DT) is present then

      if ((st.iDT > -1) && (st.iSPHI == -1))
      {
        st.dSPHI = math.mathLAS.computePHI( st.iRows,
                                            st.dDT,
                                            math.mathLAS._SONIC,
                                            math.mathLAS._CALCITE,
                                            math.mathLAS._FRESH,
                                            st.dNull );
        st.iSPHI = 220;
      }
    }

    // If the Acoustic Transit Time log (DT) is missing but
    //    the Sonic Porosity (SPHI) is present then

    else if ((st.iDT == -1) && (st.iSPHI > -1))
    {
      st.dDT = math.mathLAS.computeLOG( st.iRows,
                                        st.dSPHI,
                                        math.mathLAS._SONIC,
                                        math.mathLAS._CALCITE,
                                        math.mathLAS._FRESH,
                                        st.dNull );
      st.iDT = 210;
    }

    if (st.iTEMP > -1)
    {
      dRange = math.mathLAS.computeCurveMinMax( st.dTEMP, st.dNull );
      st.dTEMPMin = (double) ((int) dRange[0]);
      st.dTEMPMax = (double) ((int) dRange[1]);

      st.dTEMPDIFF = math.mathLAS.computeTempDiff( st.iRows, st.dTEMP, st.dNull );
      st.iTEMPDIFF = 610;
    }

    if (st.iTEMPDIFF > -1)
    {
      dRange = math.mathLAS.computeCurveMinMax( st.dTEMPDIFF, st.dNull );
      st.dTEMPDIFFMin = (double) ((int) dRange[0]);
      st.dTEMPDIFFMax = (double) ((int) dRange[1]);

	  if (st.dTEMPDIFFMin == 0.0)
	    st.dTEMPDIFFMin = -1;
	  if (st.dTEMPDIFFMax == 0.0)
	    st.dTEMPDIFFMax = 1;
    }

    if (st.iRXRT > -1)
    {
      dRange = math.mathLAS.computeCurveMinMax( st.dRXRT, st.dNull );
      st.dRXRTMin = dRange[0];
      st.dRXRTMax = dRange[1];
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     LAS File Data List Structure.
   * @param  st    = the LAS File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( lasFileDataStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._LOG;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
//      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the LAS File Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( lasFileDataStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = las3.las3Parameters._LOG_TOTAL;  // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC] =
            new String( las3.las3Parameters.LOG_PARAMETERS[i][0] );
        stT.sParams[i][_UNIT]     =
            new String( las3.las3Parameters.LOG_PARAMETERS[i][1] );
        stT.sParams[i][_VALUE]    =  new String( "" );
        stT.sParams[i][_DESCRIPTION] =
            new String( las3.las3Parameters.LOG_PARAMETERS[i][3] );
        stT.sParams[i][_FORMAT]      =
            new String( las3.las3Parameters.LOG_PARAMETERS[i][4] );

        switch (i)
        {
            // Required Fields

            case las3.las3Parameters._PDAT:
              stT.sParams[i][_VALUE] = new String( st.sPDAT );
              break;
            case las3.las3Parameters._APD:
              stT.sParams[i][_VALUE] = new String( "" + st.dAPD );
              break;
            case las3.las3Parameters._DREF:
              stT.sParams[i][_VALUE] = new String( st.sDREF );
              break;
            case las3.las3Parameters._EREF:
              stT.sParams[i][_VALUE] = new String( "" + st.dElevation );
              break;

            case las3.las3Parameters._RUN:
              stT.sParams[i][_VALUE] = new String( "1" );
              break;

            // Required parameters for AEUB compliance

            case las3.las3Parameters._TDL:
              if (st.dTLD > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dTLD );
              break;
            case las3.las3Parameters._TDD:
              if (st.dTDD > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dTDD );
              break;

            case las3.las3Parameters._CSGL:
              if (st.dCSGL > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dCSGL );
              break;
            case las3.las3Parameters._CSGD:
              if (st.dCSGD > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dCSGD );
              break;
            case las3.las3Parameters._CSGS:
              if (st.dCSGS > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dCSGS );
              break;
            case las3.las3Parameters._CSGW:
              if (st.dCSGW > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dCSGW );
              break;
            case las3.las3Parameters._BS:
              if (st.dBS > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dBS );
              break;
            case las3.las3Parameters._MUD:
              stT.sParams[i][_VALUE] = new String( st.sMUD );
              break;
            case las3.las3Parameters._MUDS:
              stT.sParams[i][_VALUE] = new String( st.sMUDS );
              break;
            case las3.las3Parameters._MUDD:
              if (st.dMUDD > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dMUDD );
              break;
            case las3.las3Parameters._MUDV:
              if (st.dMUDV > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dMUDV );
              break;
            case las3.las3Parameters._FL:
              if (st.dFL > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dFL );
              break;
            case las3.las3Parameters._PH:
              if (st.dPH > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dPH );
              break;
            case las3.las3Parameters._RM:
              if (st.dRM > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRM );
              break;
            case las3.las3Parameters._RMT:
              if (st.dRMT > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRMT );
              break;
            case las3.las3Parameters._RMF:
              if (st.dRMF > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRMF );
              break;
            case las3.las3Parameters._RMFT:
              if (st.dRMFT > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRMFT );
              break;
            case las3.las3Parameters._RMC:
              if (st.dRMC > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRMC );
              break;
            case las3.las3Parameters._RMCT:
              if (st.dRMCT > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dRMCT );
              break;
            case las3.las3Parameters._TMAX:
              if (st.dTMAX > 0.0)
                stT.sParams[i][_VALUE] = new String( "" + st.dTMAX );
              break;
            case las3.las3Parameters._TIMC:
              stT.sParams[i][_VALUE] = new String( st.sTIMC );
              break;
            case las3.las3Parameters._TIML:
              stT.sParams[i][_VALUE] = new String( st.sTIML );
              break;
            case las3.las3Parameters._T_UNIT:
              stT.sParams[i][_VALUE] = new String( st.sUNIT );
              break;
            case las3.las3Parameters._BASE:
              stT.sParams[i][_VALUE] = new String( st.sBASE );
              break;
            case las3.las3Parameters._ENG:
              stT.sParams[i][_VALUE] = new String( st.sENG );
              break;
            case las3.las3Parameters._WIT:
              stT.sParams[i][_VALUE] = new String( st.sWIT );
              break;
        }
      }
    }

    return (stT);
  }

  /** Method mergeLogData()
   * <p> This method will merge a LAS File Data Structure with the new LAS
   *     File Data Structure
   * @param  st    = LAS File Data Structure
   * @param  stLAS = LAS File Data Structure
   * @return stLAS = LAS File Data Structure
   */

  public static lasFileDataStruct mergeLogData( lasFileDataStruct st,
                                                lasFileDataStruct stLAS )
  {
    double depth  = 0.0;
    double depth1 = 0.0;
    double data1  = 0.0;
    double depth2 = 0.0;
    double data2  = 0.0;
    double dTemp  = 0.0;
    int    iTemp  = 0;

    for (int k=0; k<las.lasStandardTools.TOTAL; k++)
    {
      switch (k)
      {
        case las.lasStandardTools._GR:
          if (st.iGR > -1) stLAS.iGR = st.iGR;
          break;
        case las.lasStandardTools._CGR:
          if (st.iCGR > -1) stLAS.iCGR = st.iCGR;
          break;
        case las.lasStandardTools._SP:
          if (st.iSP > -1) stLAS.iSP = st.iSP;
          break;
        case las.lasStandardTools._CAL:
          if (st.iCAL > -1) stLAS.iCAL = st.iCAL;
          break;

        case las.lasStandardTools._PE:
          if (st.iPE > -1) stLAS.iPE = st.iPE;
          break;
        case las.lasStandardTools._RHOB:
          if (st.iRHOB > -1) stLAS.iRHOB = st.iRHOB;
          break;
        case las.lasStandardTools._DPHI:
          if (st.iDPHI > -1) stLAS.iDPHI = st.iDPHI;
          break;
        case las.lasStandardTools._DRHO:
          if (st.iRHO > -1) stLAS.iRHO = st.iRHO;
          break;
        case las.lasStandardTools._NPHI:
          if (st.iNPHI > -1) stLAS.iNPHI = st.iNPHI;
          break;

        case las.lasStandardTools._DT:
          if (st.iDT > -1) stLAS.iDT = st.iDT;
          break;
        case las.lasStandardTools._SPHI:
          if (st.iSPHI > -1) stLAS.iSPHI = st.iSPHI;
          break;

        case las.lasStandardTools._COND:
          if (st.iCOND > -1) stLAS.iCOND = st.iCOND;
          break;
        case las.lasStandardTools._CILD:
          if (st.iCILD > -1) stLAS.iCILD = st.iCILD;
          break;
        case las.lasStandardTools._CILM:
          if (st.iCILM > -1) stLAS.iCILM = st.iCILM;
          break;

        case las.lasStandardTools._RES:
          if (st.iRES > -1) stLAS.iRES = st.iRES;
          break;
        case las.lasStandardTools._RDEP:
          if (st.iRDEP > -1) stLAS.iRDEP = st.iRDEP;
          break;
        case las.lasStandardTools._RMED:
          if (st.iRMED > -1) stLAS.iRMED = st.iRMED;
          break;
        case las.lasStandardTools._RSHAL:
          if (st.iRSHAL > -1) stLAS.iRSHAL = st.iRSHAL;
          break;

        case las.lasStandardTools._ILD:
          if (st.iILD > -1) stLAS.iILD = st.iILD;
          break;
        case las.lasStandardTools._ILM:
          if (st.iILM > -1) stLAS.iILM = st.iILM;
          break;
        case las.lasStandardTools._SFLU:
          if (st.iSFLU > -1) stLAS.iSFLU = st.iSFLU;
          break;

        case las.lasStandardTools._LL:
          if (st.iLL > -1) stLAS.iLL = st.iLL;
          break;
        case las.lasStandardTools._MLL:
          if (st.iMLL > -1) stLAS.iMLL = st.iMLL;
          break;
        case las.lasStandardTools._LL8:
          if (st.iLL8 > -1) stLAS.iLL8 = st.iLL8;
          break;

        case las.lasStandardTools._LN:
          if (st.iLN > -1) stLAS.iLN = st.iLN;
          break;
        case las.lasStandardTools._SN:
          if (st.iSN > -1) stLAS.iSN = st.iSN;
          break;

        case las.lasStandardTools._MNOR:
          if (st.iMNOR > -1) stLAS.iMNOR = st.iMNOR;
          break;
        case las.lasStandardTools._MSFL:
          if (st.iMSFL > -1) stLAS.iMSFL = st.iMSFL;
          break;
        case las.lasStandardTools._MINV:
          if (st.iMINV > -1) stLAS.iMINV = st.iMINV;
          break;

        case las.lasStandardTools._AHT10:
          if (st.iAHT10 > -1) stLAS.iAHT10 = st.iAHT10;
          break;
        case las.lasStandardTools._AHT20:
          if (st.iAHT20 > -1) stLAS.iAHT20 = st.iAHT20;
          break;
        case las.lasStandardTools._AHT30:
          if (st.iAHT30 > -1) stLAS.iAHT30 = st.iAHT30;
          break;
        case las.lasStandardTools._AHT60:
          if (st.iAHT60 > -1) stLAS.iAHT60 = st.iAHT60;
          break;
        case las.lasStandardTools._AHT90:
          if (st.iAHT90 > -1) stLAS.iAHT90 = st.iAHT90;
          break;

        case las.lasStandardTools._TH:
          if (st.iTh > -1) stLAS.iTh = st.iTh;
          break;
        case las.lasStandardTools._U:
          if (st.iU > -1) stLAS.iU = st.iU;
          break;
        case las.lasStandardTools._K:
          if (st.iK > -1) stLAS.iK = st.iK;
          break;

        case las.lasStandardTools._GRN:
          if (st.iGRN > -1) stLAS.iGRN = st.iGRN;
          break;
        case las.lasStandardTools._NEUT:
          if (st.iNEUT > -1) stLAS.iNEUT = st.iNEUT;
          break;
        case las.lasStandardTools._PERM:
          if (st.iPERM > -1) stLAS.iPERM = st.iPERM;
          break;

        case las.lasStandardTools._TEMP:
          if (st.iTEMP > -1) stLAS.iTEMP = st.iTEMP;
          break;
        case las.lasStandardTools._TEMPDIFF:
          if (st.iTEMPDIFF > -1) stLAS.iTEMPDIFF = st.iTEMPDIFF;
          break;

        case las.lasStandardTools._LIN_1:
          if (st.iLIN_1 > -1)
          {
			stLAS.iLIN_1 = st.iLIN_1;
			stLAS.sUnknown[0][0] = new String( st.sUnknown[0][0] );
		  }
          break;
        case las.lasStandardTools._LIN_2:
          if (st.iLIN_2 > -1)
          {
            stLAS.iLIN_2 = st.iLIN_2;
			stLAS.sUnknown[1][0] = new String( st.sUnknown[1][0] );
		  }
          break;
        case las.lasStandardTools._LIN_3:
          if (st.iLIN_3 > -1)
          {
            stLAS.iLIN_3 = st.iLIN_3;
			stLAS.sUnknown[2][0] = new String( st.sUnknown[2][0] );
		  }
          break;
        case las.lasStandardTools._LIN_4:
          if (st.iLIN_4 > -1)
          {
            stLAS.iLIN_4 = st.iLIN_4;
			stLAS.sUnknown[3][0] = new String( st.sUnknown[3][0] );
		  }
          break;

        case las.lasStandardTools._LOG_1:
          if (st.iLOG_1 > -1)
          {
            stLAS.iLOG_1 = st.iLOG_1;
			stLAS.sUnknown[0][1] = new String( st.sUnknown[0][1] );
		  }
          break;
        case las.lasStandardTools._LOG_2:
          if (st.iLOG_2 > -1)
          {
            stLAS.iLOG_2 = st.iLOG_2;
			stLAS.sUnknown[1][1] = new String( st.sUnknown[1][1] );
		  }
          break;
        case las.lasStandardTools._LOG_3:
          if (st.iLOG_3 > -1)
          {
            stLAS.iLOG_3 = st.iLOG_3;
			stLAS.sUnknown[2][1] = new String( st.sUnknown[2][1] );
		  }
          break;
        case las.lasStandardTools._LOG_4:
          if (st.iLOG_4 > -1)
          {
            stLAS.iLOG_4 = st.iLOG_4;
			stLAS.sUnknown[3][1] = new String( st.sUnknown[3][1] );
		  }
          break;

        case las.lasStandardTools._PHI:
          if (st.iPHI > -1) stLAS.iPHI = st.iPHI;
          break;
        case las.lasStandardTools._GMCC:
          if (st.iGMCC > -1) stLAS.iGMCC = st.iGMCC;
          break;

        case las.lasStandardTools._RTo:
          if (st.iRT > -1) stLAS.iRT = st.iRT;
          break;
        case las.lasStandardTools._RWA:
          if (st.iRWA > -1) stLAS.iRWA = st.iRWA;
          break;
        case las.lasStandardTools._RO:
          if (st.iRO > -1) stLAS.iRO = st.iRO;
          break;
        case las.lasStandardTools._RXRT:
          if (st.iRXRT > -1) stLAS.iRXRT = st.iRXRT;
          break;
        case las.lasStandardTools._MA:
          if (st.iMA > -1) stLAS.iMA = st.iMA;
          break;
        case las.lasStandardTools._PHITo:
          if (st.iPHIT > -1) stLAS.iPHIT = st.iPHIT;
          break;
        case las.lasStandardTools._PHIE:
          if (st.iPHIE > -1) stLAS.iPHIE = st.iPHIE;
          break;
        case las.lasStandardTools._SW:
          if (st.iSW > -1) stLAS.iSW = st.iSW;
          break;
        case las.lasStandardTools._BVW:
          if (st.iBVW > -1) stLAS.iBVW = st.iBVW;
          break;

        case las.lasStandardTools._ATT1:
          if (st.iATT1 > -1) stLAS.iATT1 = st.iATT1;
          break;
        case las.lasStandardTools._ATT2:
          if (st.iATT2 > -1) stLAS.iATT2 = st.iATT2;
          break;
        case las.lasStandardTools._ATT3:
          if (st.iATT3 > -1) stLAS.iATT3 = st.iATT3;
          break;
        case las.lasStandardTools._ATT4:
          if (st.iATT4 > -1) stLAS.iATT4 = st.iATT4;
          break;
        case las.lasStandardTools._ATT5:
          if (st.iATT5 > -1) stLAS.iATT5 = st.iATT5;
          break;
        case las.lasStandardTools._ATT6:
          if (st.iATT6 > -1) stLAS.iATT6 = st.iATT6;
          break;

        case las.lasStandardTools._VCAL:
          if (st.iVCAL > -1) stLAS.iVCAL = st.iVCAL;
          break;
        case las.lasStandardTools._VQTZ:
          if (st.iVQTZ > -1) stLAS.iVQTZ = st.iVQTZ;
          break;
        case las.lasStandardTools._VDOL:
          if (st.iVDOL > -1) stLAS.iVDOL = st.iVDOL;
          break;
        case las.lasStandardTools._VSH:
          if (st.iVSH > -1) stLAS.iVSH = st.iVSH;
          break;
        case las.lasStandardTools._VANH:
          if (st.iVANH > -1) stLAS.iVANH = st.iVANH;
          break;
      }
    }

    if ((st != null) && (stLAS != null))
    {
      for (int j=0; j<stLAS.iRows; j++)
      {
        depth  = stLAS.depths[j];

        for (int i=0; i<st.iRows-1; i++)
        {
          depth1 = st.depths[i];
          depth2 = st.depths[i+1];

          if ((depth >= depth1) && (depth < depth2))
          {
            for (int k=0; k<las.lasStandardTools.TOTAL; k++)
            {
              if (st.checkData(k))
              {
                data1  = st.getRow( k, i );
                data2  = st.getRow( k, i+1 );

                if ((data1 > st.dNull) && (data2 > st.dNull))
                {
                  dTemp = data2 -
                            ((data2-data1) * (depth2-depth)) / (depth2-depth1);

                  iTemp = (int) (dTemp * 1000.0);

                  stLAS.setRow( k, j, (double) iTemp / 1000.0 );
                }
                else
                  stLAS.setRow( k, j, st.dNull );
              }
            }
          }
        }
      }
    }

    return (stLAS);
  }

  /** Method isDepth()
   * <p> This method will determine if the depth is already in the LAS File or
   *     Not
   * @param  depth    = depth being tested
   * @param  iRows    = total number of depths
   * @param  depths   = array of depths in the LAS File Data Structure
   * @return bValue   = true the depth is present, false it is a new depth
   */

  public static boolean isDepth( double depth, int iRows, double depths[] )
  {
    boolean bValue = false;

    for (int i=0; i<iRows; i++)
    {
      if (depth == depths[i])
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back
   * @param  str    = the string to be converted
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str )
  {
    double dValue = 0.0;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back, but uses
   *     the Null Value if it is not a valid number
   * @param  str    = the string to be converted
   * @param  dNull  = the LAS File Null Value
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str, double dNull )
  {
    double dValue = dNull;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

  /** Method convertMtoF()
   * <p> This method will convert meters to feet
   * @param  str    = the text to be converted
   * @param  sUnit  = the unit
   * @return dValue = depth in feet
   */

  public static double convertMtoF(String str, String sUnit)
  {
    double dValue = 0.0;
    double dFeet  = 3.281;
    int    iValue = 0;

    if (str.length() > 0)
    {
      if (cmn.cmnString.isNumeric( str.trim() ))
      {
        dValue = cmn.cmnString.stringToDouble( str.trim() );

        if (sUnit.equals("M"))
          dValue = dFeet * dValue;

        iValue = (int) (dValue * 1000.0);
        dValue = (double) iValue / 1000.0;
      }
    }

    return (dValue);
  }

  /** Method normalizeDepth()
   * <p> This method will convert meters to feet
   * @param  st  = Well Header Data Structure
   * @param  stL = LAS File Data Structure
   * @return st  = Well Header Data Structure
   */

  public static lasFileDataStruct normalizeDepth(
      iqstratHeadersStruct st, lasFileDataStruct stL )
  {
//    double diff = 15.0;  // Height of Kelly Bushing above Ground Level in ft
    double diff = 0.0;  // Height above Ground Level in ft

    if ((st != null) && (stL != null))
    {
      if (stL.sPDAT.length() == 0)
      {
        if ((st.dGL > 0.0) && (st.dKB > 0.0))
        {
          diff = Math.abs(st.dKB - st.dGL);

          stL.sPDAT      = new String( "GL" );  // Permanent Data
          stL.dAPD       = diff;                // Above Permanent Data
          stL.sDREF      = new String( "KB" );  // Depth Reference (KB,DF,CB)
          stL.dElevation = st.dKB;              // "Elevation of Depth Reference"
        }
        else if ((st.dGL > 0.0) && (st.dDF > 0.0))
        {
          diff = Math.abs(st.dDF - st.dGL);

          stL.sPDAT      = new String( "GL" );  // Permanent Data
          stL.dAPD       = diff;                // Above Permanent Data
          stL.sDREF      = new String( "DF" );  // Depth Reference (KB,DF,CB)
          stL.dElevation = st.dDF;              // "Elevation of Depth Reference"
        }
        else if (st.dGL > 0.0)
        {
          stL.sPDAT      = new String( "GL" );  // Permanent Data
          stL.dAPD       = 0.0;                 // Above Permanent Data
          stL.sDREF      = new String( "GL" );  // Depth Reference (KB,DF,CB)
          stL.dElevation = st.dGL;              // "Elevation of Depth Reference"
        }
        else if (st.dKB > 0.0)
        {
          stL.sPDAT      = new String( "GL" );  // Permanent Data
          stL.dAPD       = 15.0;                // Above Permanent Data
          stL.sDREF      = new String( "KB" );  // Depth Reference (KB,DF,CB)
          stL.dElevation = st.dKB - stL.dAPD;   // "Elevation of Depth Reference"
        }
        else if (st.dDF > 0.0)
        {
          stL.sPDAT      = new String( "GL" );  // Permanent Data
          stL.dAPD       = 15.0;                // Above Permanent Data
          stL.sDREF      = new String( "DF" );  // Depth Reference (KB,DF,CB)
          stL.dElevation = st.dDF - stL.dAPD;   // "Elevation of Depth Reference"
        }
      }

      if ((st.depth > 0.0) && (stL.dTLD == 0.0))
        stL.dTLD = st.depth;
    }

    return (stL);
  }

}

/*
 *  @version 1.1 01/22/2010
 *  @author  John Victorine
 */
