/*
 * @las3LoadImageData.java Version 1.1 05/27/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import rock.rockImagesListStruct;
import rock.rockImagesStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadImageData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 05/27/2011
 *  @author  John R. Victorine
 */

public class las3LoadImageData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _IMG_SRC     = 1;  // Source of Data
  public static final int _IMG_DATE    = 2;  // Created Date

  public static final int _PARAM_ROWS  = 3;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic        Description                             Variations
    { "IQKGS",     "Profile Web App Saved Data Indicator",  "", "", "" },
    { "IMG_SRC",   "Source of Data",                        "", "", "" },
    { "IMG_DATE",  "Created Date",                          "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Rock Image Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Rock Image Data List Structure
   */

  public static rockImagesListStruct getData( las3Struct stP )
  {
    int i,j,k;
    rockImagesListStruct st     = new rockImagesListStruct();
    int                  iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _IMG_SRC:  // Source of Data
                  st.sIMAGE_SRC = new String( stP.sParams[i][_VALUE] );
                  break;
                case _IMG_DATE:  // Created Date
                  st.sIMAGE_DATE = new String( stP.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Rock Image Data List Structure
   * @param  dNull  = LAS 3 Null Value
   * @param  st     = the LAS 3 Data Structure
   * @param  stCMN  = Common LAS Curve List Structure.
   * @param  stImg  = the Rock Image Data List Structure
   * @return stImg2 = the Rock Image Data List Structure
   */

  public static rockImagesListStruct loadData( double dNull,
                                               las3Struct st,
                                               cmnLASCurveListStruct stCMN,
                                               rockImagesListStruct  stImg )
  {
    rockImagesListStruct stImg2 = null;
    String               sKEY   = "0";
    int                  iTool  = -1;

    if ((st != null) && (stImg != null) && (stCMN != null))
    {
      stImg2 = rock.rockImagesUtility.copyList( stImg );

      stImg.delete();
      stImg = null;

      if (stImg2 == null)
        stImg2 = new rockImagesListStruct();

      if (st.iRows > 0)
      {
        stImg2.stItem = new rockImagesStruct[st.iRows];
        stImg2.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stImg2.stItem[i] = new rockImagesStruct();

          // Unique KID & KEY if the LAS File does not contain a value
          // the value has to be present for the program to function correctly
          // the KID and KEY are used to modify and remove a Rock Image
          // from the list.

          sKEY = new String( cmn.cmnString.UniqueName() + "_" + i );
          stImg2.stItem[i].sKEY = new String( sKEY );

          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stImg2.stItem[i] = addData( dNull,
                                          stCMN.stItem[iTool].sMnemonic,
                                          st.sData[i][j],
                                          stImg2.stItem[i] );
            }
          }
        }
      }
    }

    return (stImg2);
  }

  /** Method addData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Rock Image Data List Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Rock Image Data Structure
   * @return st        = the Rock Image Data Structure
   */

  public static rockImagesStruct addData( double dNULL,
                                          String sMnemonic,
                                          String sData,
                                          rockImagesStruct st )
  {
    int    i     = 0;
    int    iRow  = rock.rockImagesStandardTools.getRow(
                     rock.rockImagesStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st !=null)
    {
      switch (iRow)
      {
        case rock.rockImagesStandardTools._IMGTOP:  // Start Depth
          if (!sData.equals(sNull))
          {
            st.depthStart = 0.0;
            if (cmn.cmnString.isNumeric(sData))
              st.depthStart = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case rock.rockImagesStandardTools._IMGBASE: // Ending Depth
          if (!sData.equals(sNull))
          {
            st.depthEnd = 0.0;
            if (cmn.cmnString.isNumeric(sData))
              st.depthEnd = cmn.cmnString.stringToDouble(sData);
          }
          break;
        case rock.rockImagesStandardTools._IMGKID:  // KID
          if (!sData.equals(sNull)) { st.sKID = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGLOC:  // Location of File, KGS or PC
          if (!sData.equals(sNull)) { st.sLocation = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGURL:  // URL Path
          if (!sData.equals(sNull)) { st.sURL = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGFORM: // Format of the image File, JPEG, GIF, etc.
          if (!sData.equals(sNull)) { st.sFormat = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGTYPE: // Core Image Type, CORE, SLAB, THIN SECTION
          if (!sData.equals(sNull)) { st.sType = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGDIR:  // Directory Path
          if (!sData.equals(sNull)) { st.sDirectory = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGFILE: // Filename
          if (!sData.equals(sNull)) { st.sFilename = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGYEAR: // Year
          if (!sData.equals(sNull)) { st.sYear = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGTITL: // Title
          if (!sData.equals(sNull)) { st.sTitle = new String(sData); }
          break;
        case rock.rockImagesStandardTools._IMGDES:  // Description
          if (!sData.equals(sNull))
          {
			st.sDescription = new String(sData);
		  }
          break;
        case rock.rockImagesStandardTools._IMGCOM:  // Comments
          if (!sData.equals(sNull))
          {
			st.sComments = new String(sData);
		  }
          break;
        case rock.rockImagesStandardTools._IMGAUTH: // Authors
          if (!sData.equals(sNull)) { st.sAuthors = new String(sData); }
          break;
      }
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Perforation Data List Structure.
   * @param  st    = the Perforation Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( rockImagesListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_IMAGES;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Rock Images Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( rockImagesListStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams    = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            stT.sParams[i][_VALUE] = new String( "YES" );
            break;
          case _IMG_SRC:  // Source of Data
            stT.sParams[i][_VALUE] = new String( st.sIMAGE_SRC );
            break;
          case _IMG_DATE:  // Created Date
            if (st.sIMAGE_DATE.length() == 0)
              st.sIMAGE_DATE = new String( cmn.cmnString.getDate() );

            if (st.sIMAGE_DATE.length() > 0)
            {
              sDate = cmn.cmnString.convertDate( st.sIMAGE_DATE );
              stT.sParams[i][_VALUE]  = new String(sDate[0]);
              stT.sParams[i][_FORMAT] = new String(sDate[1]);
            }
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Rock Images Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( rockImagesListStruct st,
                                      las3Struct           stT,
                                      double               dNull )
  {
    int    i,j,k;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[rock.rockImagesStandardTools.TOTAL];
      for (i=0; i<rock.rockImagesStandardTools.TOTAL; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (i=0; i<rock.rockImagesStandardTools.TOTAL; i++)
      {
        switch (i)
        {
		  // Start Depth
          case rock.rockImagesStandardTools._IMGTOP:
            iSelected[i] = 1;
            break;
          // Ending Depth
          case rock.rockImagesStandardTools._IMGBASE:
            iSelected[i] = 1;
            break;
          // KID
          case rock.rockImagesStandardTools._IMGKID:
            if (st.stItem[i].sKID.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Location of File, KGS or PC
          case rock.rockImagesStandardTools._IMGLOC:
            if (st.stItem[i].sLocation.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // URL Path
          case rock.rockImagesStandardTools._IMGURL:
            if (st.stItem[i].sURL.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Format of the image File, JPEG, GIF, etc.
          case rock.rockImagesStandardTools._IMGFORM:
            if (st.stItem[i].sFormat.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Core Image Type, CORE, SLAB, THIN SECTION
          case rock.rockImagesStandardTools._IMGTYPE:
            if (st.stItem[i].sType.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Directory Path
          case rock.rockImagesStandardTools._IMGDIR:
            if (st.stItem[i].sDirectory.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Filename
          case rock.rockImagesStandardTools._IMGFILE:
            if (st.stItem[i].sFilename.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Year
          case rock.rockImagesStandardTools._IMGYEAR:
            if (st.stItem[i].sYear.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Title
          case rock.rockImagesStandardTools._IMGTITL:
            if (st.stItem[i].sTitle.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Description
          case rock.rockImagesStandardTools._IMGDES:
            if (st.stItem[i].sDescription.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Comments
          case rock.rockImagesStandardTools._IMGCOM:
            if (st.stItem[i].sComments.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
          // Authors
          case rock.rockImagesStandardTools._IMGAUTH:
            if (st.stItem[i].sAuthors.length() > 0)
            {
			  iSelected[i] = 1;
			}
            break;
        }
      }

      // Count the Number of Curves Populated

      for (i=0; i<rock.rockImagesStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<rock.rockImagesStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              rock.rockImagesStandardTools.CURVES[i][1]);

          stT.sCurves[iCnt][_UNIT] = new String(
              rock.rockImagesStandardTools.CURVES[i][3]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              rock.rockImagesStandardTools.CURVES[i][2]);

          stT.sCurves[iCnt][_FORMAT] = new String("F");
          if (i > rock.rockImagesStandardTools._IMGBASE)
          {
            stT.sCurves[iCnt][_FORMAT] = new String("S");
          }

          iCnt++;
        }
      }

      // Build the Data Section

      stT.iRows    = st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Stratigraphic Units Data List Structure

      for (i=0; i<st.iCount; i++)
      {
		j = -1;
        for (k=0; k<rock.rockImagesStandardTools.TOTAL; k++)
//        for (j=0; j<iSum; j++)
        {
          if (iSelected[k] > -1) //if (iSelected[j] > -1)
          {
			j++;
            switch (iCurves[j])
            {
              case rock.rockImagesStandardTools._IMGTOP:  // Start Depth
                stT.sData[i][j] = new String( "" + st.stItem[i].depthStart );
                break;
              case rock.rockImagesStandardTools._IMGBASE: // Ending Depth
                stT.sData[i][j] = new String( "" + st.stItem[i].depthEnd );
                break;
              case rock.rockImagesStandardTools._IMGKID:  // KID
                stT.sData[i][j] = new String( st.stItem[i].sKID );
                break;
              case rock.rockImagesStandardTools._IMGLOC:  // Location of File, KGS or PC
                stT.sData[i][j] = new String( st.stItem[i].sLocation );
                break;
              case rock.rockImagesStandardTools._IMGURL:  // URL Path
                stT.sData[i][j] = new String( st.stItem[i].sURL );
                break;
              case rock.rockImagesStandardTools._IMGFORM: // Format of the image File, JPEG, GIF, etc.
                stT.sData[i][j] = new String( st.stItem[i].sFormat );
                break;
              case rock.rockImagesStandardTools._IMGTYPE: // Core Image Type, CORE, SLAB, THIN SECTION
                stT.sData[i][j] = new String( st.stItem[i].sType );
                break;
              case rock.rockImagesStandardTools._IMGDIR:  // Directory Path
                stT.sData[i][j] = new String( st.stItem[i].sDirectory );
                break;
              case rock.rockImagesStandardTools._IMGFILE: // Filename
                stT.sData[i][j] = new String( st.stItem[i].sFilename );
                break;
              case rock.rockImagesStandardTools._IMGYEAR: // Year
                stT.sData[i][j] = new String( st.stItem[i].sYear );
                break;
              case rock.rockImagesStandardTools._IMGTITL: // Title
                stT.sData[i][j] = new String( st.stItem[i].sTitle );
                break;
              case rock.rockImagesStandardTools._IMGDES:  // Description
                stT.sData[i][j] = new String( st.stItem[i].sDescription );
                break;
              case rock.rockImagesStandardTools._IMGCOM:  // Comments
                stT.sData[i][j] = new String( st.stItem[i].sComments );
                break;
              case rock.rockImagesStandardTools._IMGAUTH: // Authors
                stT.sData[i][j] = new String( st.stItem[i].sAuthors );
                break;
		    }
          }
        }
      }
    }

    return (stT);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back
   * @param  str    = the string to be converted
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str )
  {
    double dValue = 0.0;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back, but uses
   *     the Null Value if it is not a valid number
   * @param  str    = the string to be converted
   * @param  dNull  = the LAS File Null Value
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str, double dNull )
  {
    double dValue = dNull;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }
}

/*
 *  @version 1.1 05/27/2011
 *  @author  John Victorine
 */
