/*
 * @las3LoadEnvData.java Version 1.1 01/31/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import horizon.env.envListStruct;
import horizon.env.envStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadEnvData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 01/31/2012
 *  @author  John R. Victorine
 */

public class las3LoadEnvData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _ENVSRC      = 1;
  public static final int _ENVDR       = 2;

  public static final int _PARAM_ROWS  = 3;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description                        Variations
    { "IQKGS",  "Profile Web App Saved Data Indicator",  "", "", "" },
    { "ENVSRC", "Source",                                "", "", "" },
    { "ENVDR",  "Depth Reference",                       "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Environment Data List Structure
   *     from the LAS 3 Data List Structure.
   * @param  stT = the LAS 3 Data Structure
   * @return st  = the Environment Data List Structure
   */

  public static envListStruct getData( las3Struct stT )
  {
    int i,j,k;
    int iParam = -1;
    envListStruct st  = new envListStruct();

    if (stT != null)
    {
      for (i=0; i<stT.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stT.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _ENVSRC:
                  st.source = new String( stT.sParams[i][_VALUE] );
                  break;
                case _ENVDR:
                  st.sReference = new String( stT.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Environment Data List Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stEnv   = the Environment Data List Structure
   * @return stEnv2  = the Environment Data List Structure
   */

  public static envListStruct loadData( double     dNull,
                                        las3Struct st,
                                        cmnLASCurveListStruct stCMN,
                                        envListStruct         stEnv )
  {
    envListStruct stEnv2 = null;
    envStruct     stE    = null;
    int    iTool         = -1;
    String sMnemonic     = "";
    String sData         = "";
    int    iRow          = -1;
    String sTemp         = "";
    String sNull         = new String(""+dNull); // NULL Value in String Form

    String sKEY          = "0"; // Unique Identifier for Entry
    double depthStart    = 0.0; // Starting Depth
    double depthEnd      = 0.0; // Ending Depth
    double dBed          = 0.0; // Bed Thickness

    // Depositional Environment selected for the depth range

    String sDepEnv       = ""; // Depositional Environment Phrase
    String sDescrip      = ""; // Depositional Environment Descriptions

    int    iType         = -1; // Depositional Environment Model Identifier
    String sType         = ""; // Depositional Environment Model
    String sGroup        = ""; // Type of Depositional Environment
    String sGroup2       = ""; // Group:
    String sGroup3       = ""; // Subgroup

    String sLevel        = ".............."; // Sea Level:
    int    iLevel        = 0;
    char   c[]           = null;


    if ((st != null) && (stEnv != null) && (stCMN != null))
    {
      if (stCMN.iCount > 0)
      {
        stEnv2 = horizon.env.envUtility.copyList( stEnv );

        stEnv.delete();
        stEnv = null;

        if (stEnv2 == null)
          stEnv2 = new envListStruct();

        if (st.iRows > 0)
        {
          for (int i=0; i<st.iRows; i++)
          {
            for (int j=0; j<st.iColumns; j++)
            {
              iTool = -1;
              for (int k=0; k<stCMN.iCount; k++)
              {
                if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                        stCMN.stItem[k].sMnemonic))
                {
                  iTool = k;
                }
              }

              if (iTool > -1)
              {
                sMnemonic = new String( stCMN.stItem[iTool].sMnemonic );
                sData     = new String( st.sData[i][j] );
				iRow      = horizon.env.envStandardTools.getRow(
                     horizon.env.envStandardTools._C_MNEM, sMnemonic );

                switch (iRow)
                {
                  case horizon.env.envStandardTools._STRT:  // Starting Depth
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                      {
                        depthStart = cmn.cmnString.stringToDouble(sData);
                        depthStart = Math.round(depthStart * 100.0) / 100.0;
					  }
					}
                    break;

                  case horizon.env.envStandardTools._END:   // Ending Depth
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        depthEnd = cmn.cmnString.stringToDouble(sData);
                        depthEnd = Math.round(depthEnd * 100.0) / 100.0;
					}
                    break;

                  case horizon.env.envStandardTools._THICK: // Bed Thickness
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dBed = cmn.cmnString.stringToDouble(sData);
					}
                    break;

                  case horizon.env.envStandardTools._ENV:   // Depositional Environment Term
                    if (!sData.equals(sNull))
                      sDepEnv = new String(sData);
                    break;

                  case horizon.env.envStandardTools._MODEL: // Depositional Environment Model
                    if (!sData.equals(sNull))
                      sType = new String(sData);

                    if (sType.toLowerCase().equals("continental"))
                      iType = 0;
                    else if (sType.toLowerCase().equals("transitional"))
                      iType = 1;
                    else if (sType.toLowerCase().equals("marine"))
                      iType = 2;
                    break;

                  case horizon.env.envStandardTools._TYPE:  // Depositional Environment Type
                    if (!sData.equals(sNull))
                      sGroup = new String(sData);
                    break;

                  case horizon.env.envStandardTools._GRP:   // Depositional Environment Group
                    if (!sData.equals(sNull))
                      sGroup2 = new String(sData);
                    break;

                  case horizon.env.envStandardTools._SGRP:  // Depositional Environment Subgroup
                    if (!sData.equals(sNull))
                      sGroup3 = new String(sData);
                    break;

                  case horizon.env.envStandardTools._DESC:  // Description (original text)
                    if (!sData.equals(sNull))
                      sDescrip = new String(sData);
                    break;

                  case horizon.env.envStandardTools._LEVEL: // Sea Level identifer
                    if (!sData.equals(sNull))
                      sLevel = new String(sData);
                    break;
                }
              }
			}

            // Insert variables into bio-stratigraphy data list structure

   	        stE            = new envStruct();
            sKEY           = new String( cmn.cmnString.UniqueName() + "_" + i );
            stE.sKEY       = new String( sKEY );
            stE.depthStart = depthStart;
            stE.depthEnd   = depthStart;
            if (depthEnd > depthStart)
              stE.depthEnd = depthEnd;
            stE.dBed       = dBed;
            if (dBed == 0.0)
            {
			  stE.dBed = (double) ((int) ((stE.depthEnd - stE.depthStart) * 100.0))/100.0;
			}

            // Depositional Environment selected for the depth range

            stE.sDepEnv    = new String(sDepEnv); // Depositional Environment Phrase
            stE.sDescrip   = new String(sDescrip); // Depositional Environment Descriptions

            stE.iType      = iType; // Depositional Environment Type Identifier
            stE.sType      = new String(sType);   // Depositional Environment Type
            stE.sGroup     = new String(sGroup);  // 1st Group: Type of Depositional Environment
            stE.sGroup2    = new String(sGroup2); // 2nd Group: Environment Group
            stE.sGroup3    = new String(sGroup3); // 3rd Group: Environment Subgroup

            c = sLevel.toCharArray();
            for (int p=0; p<horizon.env.envStruct._MAXIMUM; p++)
            {
			  iLevel = 0;
		      if (c[p] == 'X')
		        iLevel = 5;

              switch (p)
              {
                case horizon.env.envStruct._CONTINENTAL:  // Continental
                  stE.icontl = iLevel;
                  break;
                                                          // Coastal
                case horizon.env.envStruct._COASTAL:      // -- Upper
                  stE.icstl = iLevel;
                  break;
                case horizon.env.envStruct._COASTAL_F:    // .. Fresh Water
                  stE.ifrsh = iLevel;
                  break;
                case horizon.env.envStruct._COASTAL_B:    // .. Brackish Water
                  stE.ibrcksh = iLevel;
                  break;
                case horizon.env.envStruct._COASTAL_H:    // .. Hyper saline Water
                  stE.ihyper = iLevel;
                  break;
                                                          // Neritic
                                                          // -- Inner
                case horizon.env.envStruct._NERITIC_I_0:  // .. Shoreface (upper and lower)
                  stE.ishore = iLevel;
                  break;
                case horizon.env.envStruct._NERITIC_I_1:  // .. Offshore transition
                  stE.ioff_tran = iLevel;
                  break;
                case horizon.env.envStruct._NERITIC_I_2:  // .. Offshore
                  stE.ioffshr = iLevel;
                  break;
                case horizon.env.envStruct._NERITIC_M:    // -- Middle
                  stE.ishlf_m = iLevel;
                  break;
                case horizon.env.envStruct._NERITIC_O:    // -- Outer
                  stE.ishlf_o = iLevel;
                  break;
                                              // Bathyal
                case horizon.env.envStruct._BATHYAL_U:    // -- Upper
                  stE.ibthyl_u = iLevel;
                  break;
                case horizon.env.envStruct._BATHYAL_M:    // -- Middle
                  stE.ibthyl_m = iLevel;
                  break;
                case horizon.env.envStruct._BATHYAL_L:    // -- Lower
                  stE.ibthyl_l = iLevel;
                  break;
                case horizon.env.envStruct._ABYSSAL:      // Abyssal
                  stE.iabyssl = iLevel;
                  break;
              }
	        }

 		    stEnv2 = horizon.env.envUtility.add( stE, stEnv2 );

            // Clear Variables for next entry

			if (stE != null)
			  stE.delete();
			stE = null;

            sKEY        = new String( "0" ); // Unique Identifier for Entry
            depthStart  = 0.0; // Starting Depths
            depthEnd    = 0.0; // Ending Depths
            dBed        = 0.0; // Bed Thickness

            // Depositional Environment selected for the depth range

            sDepEnv       = ""; // Depositional Environment Phrase
            sDescrip      = ""; // Depositional Environment Descriptions

            iType         = -1; // Depositional Environment Model Identifier
            sType         = ""; // Depositional Environment Model
            sGroup        = ""; // Type of Depositional Environment
            sGroup2       = ""; // Group:
            sGroup3       = ""; // Subgroup

            sLevel        = ""; // Sea Level:
		  }
        }
      }
    }

    return (stEnv2);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Environment Data List Structure.
   * @param  st    = the Environment Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( envListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_ENV;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Environment Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( envListStruct st, las3Struct stT )
  {
    int i,j;

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = 3;      // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            break;
          case _ENVSRC:
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case _ENVDR:
            stT.sParams[i][_VALUE] = new String( st.sReference );
            break;
        }
      }

    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the Environment Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( envListStruct st,
                                      las3Struct         stT,
                                      double             dNull )
  {
    int    i,j,k,p;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iTotal      = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";
    String sNull       = "" + dNull;
    int    iLevel[]    = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    String sLevel      = "";
    String str         = "";

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[horizon.env.envStandardTools.TOTAL];
      for (i=0; i<horizon.env.envStandardTools.TOTAL; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (i=0; i<st.iCount; i++)
      {
        iSelected[horizon.env.envStandardTools._STRT]++;  // Starting Depth
        iSelected[horizon.env.envStandardTools._END]++;   // Ending Depth
        iSelected[horizon.env.envStandardTools._THICK]++; // Bed Thickness
        iSelected[horizon.env.envStandardTools._ENV]++;   // Depositional Environment Term
        iSelected[horizon.env.envStandardTools._MODEL]++; // Depositional Environment Model
        iSelected[horizon.env.envStandardTools._TYPE]++;  // Depositional Environment Type
        iSelected[horizon.env.envStandardTools._GRP]++;   // Depositional Environment Group
        iSelected[horizon.env.envStandardTools._SGRP]++;  // Depositional Environment Subgroup
        iSelected[horizon.env.envStandardTools._DESC]++;  // Description (original text)
        iSelected[horizon.env.envStandardTools._LEVEL]++; // Sea Level identifer
	  }

      // Count the Number of Curves Populated

      for (i=0; i<horizon.env.envStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<horizon.env.envStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              horizon.env.envStandardTools.CURVES[i][
                horizon.env.envStandardTools._C_MNEM]);

          stT.sCurves[iCnt][_UNIT] = new String(
              horizon.env.envStandardTools.CURVES[i][
                horizon.env.envStandardTools._C_UNIT]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              horizon.env.envStandardTools.CURVES[i][
                horizon.env.envStandardTools._C_NAME]);

          stT.sCurves[iCnt][_FORMAT] = new String("S");
          if (stT.sCurves[iCnt][_UNIT].length() > 0)
            stT.sCurves[iCnt][_FORMAT] = new String("F");

          iCnt++;
        }
      }

      // Build the Data Section

      iTotal       = st.iCount;
      stT.iRows    = st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[iTotal][stT.iColumns]; //[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<iTotal; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Stratigraphic Units Data List Structure

      iCnt = 0;

      for (i=0; i<st.iCount; i++)
      {
	    if (iCnt < iTotal)
		{
          for (j=0; j<iSum; j++)
          {
            switch (iCurves[j])
            {
              case horizon.env.envStandardTools._STRT:  // Starting Depth
                st.stItem[i].depthStart = Math.round(st.stItem[i].depthStart * 100.0) / 100.0;
                stT.sData[iCnt][j] = new String( "" + st.stItem[i].depthStart );
                break;

              case horizon.env.envStandardTools._END:   // Ending Depth
                st.stItem[i].depthEnd = Math.round(st.stItem[i].depthEnd * 100.0) / 100.0;
                stT.sData[iCnt][j] = new String( "" + st.stItem[i].depthEnd );
                break;

              case horizon.env.envStandardTools._THICK: // Bed Thickness
                if (st.stItem[i].dBed == -999.95)
                {
                  st.stItem[i].dBed =
                    (double) ((int) ((st.stItem[i].depthEnd -
                    st.stItem[i].depthStart) * 100.0))/100.0;
			    }
			    else
			    {
                  st.stItem[i].dBed = Math.round(st.stItem[i].dBed * 100.0) / 100.0;
				}

                stT.sData[iCnt][j] = new String( "" + st.stItem[i].dBed );
                break;

              case horizon.env.envStandardTools._ENV:   // Depositional Environment Term
                if (st.stItem[i].sDepEnv.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sDepEnv );
                break;

              case horizon.env.envStandardTools._MODEL: // Depositional Environment Model
                if (st.stItem[i].sType.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sType );
                break;

              case horizon.env.envStandardTools._TYPE:  // Depositional Environment Type
                if (st.stItem[i].sGroup.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sGroup );
                break;

              case horizon.env.envStandardTools._GRP:   // Depositional Environment Group
                if (st.stItem[i].sGroup2.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sGroup2 );
                break;

              case horizon.env.envStandardTools._SGRP:  // Depositional Environment Subgroup
                if (st.stItem[i].sGroup3.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sGroup3 );
                break;

              case horizon.env.envStandardTools._DESC:  // Description (original text)
                if (st.stItem[i].sDescrip.length() > 1)
                  stT.sData[iCnt][j] = new String( st.stItem[i].sDescrip );
                break;

              case horizon.env.envStandardTools._LEVEL: // Sea Level identifer
                iLevel[0] = st.stItem[i].icontl;     // Continental
                iLevel[1] = st.stItem[i].icstl;      // Coastal
                iLevel[2] = st.stItem[i].ifrsh;      // Fresh Water
                iLevel[3] = st.stItem[i].ibrcksh;    // Brackish Water
                iLevel[4] = st.stItem[i].ihyper;     // Brackish Water
                iLevel[5] = st.stItem[i].ishore;     // Shoreface
                iLevel[6] = st.stItem[i].ioff_tran;  // Off shore trasition
                iLevel[7] = st.stItem[i].ioffshr;    // Off Shore
                iLevel[8] = st.stItem[i].ishlf_m;    // Middle Neritic
                iLevel[9] = st.stItem[i].ishlf_o;    // Outer Neritic
                iLevel[10] = st.stItem[i].ibthyl_u;  // Upper Bathyal
                iLevel[11] = st.stItem[i].ibthyl_m;  // Middle Bathyal
                iLevel[12] = st.stItem[i].ibthyl_l;  // Lower Bathyal
                iLevel[13] = st.stItem[i].iabyssl;   // Abyssal


                sLevel = new String("");
                for (p=0; p<14; p++)
                {
                  str = ".";
				  if (iLevel[p] > 0)
                    str = "X";

                  sLevel = new String( sLevel + str );
			    }

                stT.sData[iCnt][j] = new String( sLevel );
                break;
            }
          }

          iCnt++;
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 01/31/2012
 *  @author  John Victorine
 */
