/*
 * @las3LoadDSTData.java Version 1.1 04/23/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import dst.dstListStruct;
import dst.dstStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadDSTData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 04/23/2012
 *  @author  John R. Victorine
 */

public class las3LoadDSTData
{
  public static final int _NONE = las3.las3Constants._NONE;

  public static final int _ID    = dst.dstParameters._ID;    // DST Tab ID       - Main
  public static final int _FORM  = dst.dstParameters._FORM;  // Formation        - Main
  public static final int _STRT  = dst.dstParameters._STRT;  // Starting Depth   - Main
  public static final int _STOP  = dst.dstParameters._STOP;  // Ending Depth     - Main
  public static final int _DEV   = dst.dstParameters._DEV;   // Deviated         - Main
  public static final int _WHIP  = dst.dstParameters._WHIP;  // Whipstock        - Main
  public static final int _SERL  = dst.dstParameters._SERL;  // Serial Number    - Summary
  public static final int _READ  = dst.dstParameters._READ;  // Reading          - Summary
  public static final int _PRES  = dst.dstParameters._PRES;  // Pressure         - Summary
  public static final int _DEPT  = dst.dstParameters._DEPT;  // @Depth           - Summary
  public static final int _CMMNT = dst.dstParameters._CMMNT; // Test Comments    - Summary
  public static final int _HOLE  = dst.dstParameters._HOLE;  // Hole Diameter    - Main
  public static final int _COND  = dst.dstParameters._COND;  // Hole Condition   - Main
  public static final int _MUD   = dst.dstParameters._MUD;   // Mud Type         - Recovery
  public static final int _MUDD  = dst.dstParameters._MUDD;  // Mud Weight       - Recovery
  public static final int _MUDV  = dst.dstParameters._MUDV;  // Viscosity        - Recovery
  public static final int _FL    = dst.dstParameters._FL;    // Water Loss       - Recovery
  public static final int _RM    = dst.dstParameters._RM;    // Resistivity      - Recovery
  public static final int _MSAL  = dst.dstParameters._MSAL;  // Salinity         - Recovery
  public static final int _CAKE  = dst.dstParameters._CAKE;  // Mud Cake         - Recovery
  public static final int _CSHT  = dst.dstParameters._CSHT;  // Cushion Type     - Recovery
  public static final int _CSHL  = dst.dstParameters._CSHL;  // Cushion Length   - Recovery
  public static final int _CSHV  = dst.dstParameters._CSHV;  // Cushion Volume   - Recovery
  public static final int _GCSHT = dst.dstParameters._GCSHT; // Gas Cushion Type - Recovery
  public static final int _GCSHP = dst.dstParameters._GCSHP; // Gas Cushion Pressure - Recovery
  public static final int _OAPI  = dst.dstParameters._OAPI;  // Oil API          - Recovery
  public static final int _WSAL  = dst.dstParameters._WSAL;  // Water Aalinity   - Recovery
  public static final int _CAP   = dst.dstParameters._CAP;   // Capacity         - Summary
  public static final int _STIM  = dst.dstParameters._STIM;  // Start Date-Time  - Main
  public static final int _ETIM  = dst.dstParameters._ETIM;  // End Date-Time    - Main
  public static final int _TIMC  = dst.dstParameters._TIMC;  // Calibration Date - Main
  public static final int _TIMON = dst.dstParameters._TIMON; // Time On Bottom   - Main
  public static final int _TIMOF = dst.dstParameters._TIMOF; // Time Off Bottom  - Main
  public static final int _TYPE  = dst.dstParameters._TYPE;  // Test Type        - Main
  public static final int _UNITa = dst.dstParameters._UNIT;  // Unit Number      - Main
  public static final int _BASE  = dst.dstParameters._BASE;  // DST Company      - Main
  public static final int _ENG   = dst.dstParameters._ENG;   // Tester           - Main

  // -- Quantitative Analysis Panel

  public static final int _m     = dst.dstParameters._m;     // Slope
  public static final int _q     = dst.dstParameters._q;     // Flow Rate
  public static final int _h     = dst.dstParameters._h;     // Thickness
  public static final int _u     = dst.dstParameters._u;     // Viscosity
  public static final int _Pr    = dst.dstParameters._Pr;    // Formation Pressure
  public static final int _degR  = dst.dstParameters._degR;  // Formation Temperature
  public static final int _beta  = dst.dstParameters._beta;  // Formation volume Factor      (FLUID)
  public static final int _Z     = dst.dstParameters._Z;     // Super compressibility Factor (GAS)

  public static final int _KHU   = dst.dstParameters._KHU;   // Transmissibility       (COMPUTED)
  public static final int _KH    = dst.dstParameters._KH;    // Permeability Thickness (COMPUTED)
  public static final int _K     = dst.dstParameters._K;     // Effective Permeability (COMPUTED)

  public static final int _Pif   = dst.dstParameters._Pif;   // Initial Static Reservoir Pressure
  public static final int _Pwf   = dst.dstParameters._Pwf;   // Bottom Hole Flowing Pressure (FFP)
  public static final int _tf    = dst.dstParameters._tf;    // Flow Time
  public static final int _rw    = dst.dstParameters._rw;    // Well Bore radius
  public static final int _re    = dst.dstParameters._re;    // Drainage radius
  public static final int _PHI   = dst.dstParameters._PHI;   // Porosity
  public static final int _c     = dst.dstParameters._c;     // Compressibility
  public static final int _G     = dst.dstParameters._G;     // Specific Gravity of Gas      (GAS)

  public static final int _Pwfc  = dst.dstParameters._Pwfc;  // Bottom Hole Flowing Pressure (COMPUTED) (GAS)

  public static final int _DR    = dst.dstParameters._DR;    // Damage Ratio
  public static final int _S     = dst.dstParameters._S;     // Skin Factor
  public static final int _Ps    = dst.dstParameters._Ps;    // Pressure Drop due to Skin Factor
  public static final int _Pt    = dst.dstParameters._Pt;    // Pressure Drop due to Turbulence  (GAS)

  public static final int _PI    = dst.dstParameters._PI;    // Productive Index
  public static final int _RI    = dst.dstParameters._RI;    // Radius of Investigation

  public static final int _qa    = dst.dstParameters._qa;    // Settled Production Rate (FLUID)
  public static final int _qAOF  = dst.dstParameters._qAOF;  // Absolute Open Flow      (GAS)

  public static final int _ELP1  = dst.dstParameters._ELP1;   // Exclude Points Data Points Below Shut In 1
  public static final int _EUP1  = dst.dstParameters._EUP1;   // Exclude Points Data Points Above Shut In 1

  public static final int _M1    = dst.dstParameters._M1;   // Shut In 1 Slope
  public static final int _P1    = dst.dstParameters._P1;   // Shut In 1 Intercept

  public static final int _ELP2  = dst.dstParameters._ELP2;   // Exclude Points Data Points Below Shut In 1
  public static final int _EUP2  = dst.dstParameters._EUP2;   // Exclude Points Data Points Above Shut In 1

  public static final int _M2    = dst.dstParameters._M2;   // Shut In 1 Slope
  public static final int _P2    = dst.dstParameters._P2;   // Shut In 1 Intercept

  public static final int _RECTL = dst.dstParameters._RECTL; // Recovery Total Length - Recovery
  public static final int _RECTV = dst.dstParameters._RECTV; // Recovery Total Volume - Recovery
  public static final int _RECC  = dst.dstParameters._RECC;  // Recovery Comments     - Recovery
  public static final int _TOTAL = dst.dstParameters._TOTAL;

  public static final int _PARAM_ROWS  = _TOTAL;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = dst.dstParameters.PARAMETERS;

  public static final int _TYPE_ = 0; // Type of DST Type
  public static final int _CNT_R = 1; // Total number of Recovery Pressure Data
  public static final int _CNT_G = 2; // Total number of Gas Rates Data
  public static final int _CNT_1 = 3; // Total number of Shut In 1 Pressure-Temperature-Time Data
  public static final int _CNT_2 = 4; // Total number of Shut In 2 Pressure-Temperature-Time Data
  public static final int _CNT_D = 5; // Total number of Pressure-Temperature-Time Data

  public static final int _PARAM2_ROWS  = 6;
//  public static final int _PARAM2_COLS  = 3;
  public static final String PARAM2[][] =
  {
    { "TYPE_D", "Type of DST Test, FLUID or GAS" },
    { "CNT_R ", "Total number of Recovery Pressure Data" },
    { "CNT_G ", "Total number of Gas Rates Data" },
    { "CNT_1 ", "Total number of Shut In 1 Pressure-Temperature-Time Data" },
    { "CNT_2 ", "Total number of Shut In 2 Pressure-Temperature-Time Data" },
    { "CNT_D ", "Total number of Pressure-Temperature-Time Data" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  // Data Constants

  public static final int _INITIAL_HYDRO = dst.dstStruct._INITIAL_HYDRO; // Initial Hydrostatic
  public static final int _FLOW_1        = dst.dstStruct._FLOW_1;        // Open to Flow 1
  public static final int _SHUT_IN_1     = dst.dstStruct._SHUT_IN_1;     // Shut-in 1
  public static final int _END_SHUT_IN_1 = dst.dstStruct._END_SHUT_IN_1; // End Shut-in 1
  public static final int _FLOW_2        = dst.dstStruct._FLOW_2;        // Open to Flow 2
  public static final int _SHUT_IN_2     = dst.dstStruct._SHUT_IN_2;     // Shut-in 2
  public static final int _END_SHUT_IN_2 = dst.dstStruct._END_SHUT_IN_2; // End Shut-in 2
  public static final int _FINAL_HYDRO   = dst.dstStruct._FINAL_HYDRO;   // Final Hydrostatic

  public static final int _TIME          = dst.dstStruct._TIME;          // Time
  public static final int _PRESSURE      = dst.dstStruct._PRESSURE;      // Pressure
  public static final int _TEMPERATURE   = dst.dstStruct._TEMPERATURE;   // Temperature

  /** Method getData()
   * <p> This method will return the DST Data Structure
   *     from the LAS 3 Data Structure.
   * @param  stT    = the LAS 3 Data Structure
   * @return stData = the DST Data Structure
   */

  public static dstStruct getData( las3Struct stT )
  {
	dstStruct stData = null;

    int i,j;
    int iParam = -1;

    if (stT != null)
    {
      for (i=0; i<stT.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          if (stT.sParams[i][_MNEMONIC].trim().toUpperCase().equals(PARAM[j][0].trim()))
          {
            iParam = j;

            if (stData == null)
            {
			  stData = new dstStruct();
			}

            switch (iParam)
            {
              case _NONE:
                break;

              case _ID:
            	stData.sKEY = new String( cmn.cmnString.UniqueName() );
                stData.sID  = new String( stT.sParams[i][_VALUE] );
                break;
              case _FORM:
                stData.sFormation = new String( stT.sParams[i][_VALUE] );
                break;
              case _STRT:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.depthStart = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _STOP:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.depthEnd = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _DEV:
                if (stT.sParams[i][_VALUE].toUpperCase().equals("NO"))
                  stData.iDeviated = dst.dstStruct._NO;
                else
                  stData.iDeviated = dst.dstStruct._YES;
                break;
              case _WHIP:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dWhipstock = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _SERL:
                stData.serialNo = new String( stT.sParams[i][_VALUE] );
                break;
              case _READ:
                stData.sReading = new String( stT.sParams[i][_VALUE] );
                break;
              case _PRES:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dPressure = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _DEPT:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.depth = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CMMNT:
                stData.sTestRemark = new String( stT.sParams[i][_VALUE] );
                break;
              case _HOLE:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.diameter_hole = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _COND:
                stData.sCondition_hole = new String( stT.sParams[i][_VALUE] );
                break;
              case _MUD:
                stData.sMudType = new String( stT.sParams[i][_VALUE] );
                break;
              case _MUDD:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dMudWeight = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _MUDV:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dViscosity = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _FL:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dWaterLoss = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _RM:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dOHM_M = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _MSAL:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dSalinity = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CAKE:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dFilterCake = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CSHT:
                stData.sCushionType = new String( stT.sParams[i][_VALUE] );
                break;
              case _CSHL:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dCushionLength = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CSHV:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dCushionVolume = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _GCSHT:
                stData.sGasCushionType = new String( stT.sParams[i][_VALUE] );
                break;
              case _GCSHP:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dGasCushionPres = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _OAPI:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dOil_API = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _WSAL:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dWaterSalinity = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CAP:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dCapacity = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _STIM:
                stData.sDateStart = new String( stT.sParams[i][_VALUE] );
                break;
              case _ETIM:
                stData.sDateEnd = new String( stT.sParams[i][_VALUE] );
                break;
              case _TIMC:
                stData.sDateCalib = new String( stT.sParams[i][_VALUE] );
                break;
              case _TIMON:
                stData.sTimeOnBtm = new String( stT.sParams[i][_VALUE] );
                break;
              case _TIMOF:
                stData.sTimeOffBtm = new String( stT.sParams[i][_VALUE] );
                break;
              case _TYPE:
                stData.sTestType = new String( stT.sParams[i][_VALUE] );
                break;
              case _UNITa:
                stData.sUnitNo = new String( stT.sParams[i][_VALUE] );
                break;
              case _BASE:
                stData.sCompany = new String( stT.sParams[i][_VALUE] );
                break;
              case _ENG:
                stData.sTester = new String( stT.sParams[i][_VALUE] );
                break;
              case _RECTL:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dLength = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _RECTV:
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dVolume = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _RECC:
                stData.sComments = new String( stT.sParams[i][_VALUE] );
                break;

              // -- Quantitative Analysis Panel

              case _m:    // Slope
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.m2 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _q:    // Flow Rate
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.q = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _h:    // Thickness
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.h = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _u:    // Viscosity
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.mu = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _Pr:   // Formation Pressure
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dPressure = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _degR: // Formation Temperature
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.degF = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _beta: // Formation volume Factor      (FLUID)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.beta = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _Z:    // Super compressibility Factor (GAS)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.Z = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _KHU:  // Transmissibility       (COMPUTED)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.KHU = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _KH:   // Permeability Thickness (COMPUTED)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.KH = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _K:    // Effective Permeability (COMPUTED)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.K = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _Pif:  // Initial Static Reservoir Pressure
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.P2 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _Pwf:  // Bottom Hole Flowing Pressure (FFP)
                break;
              case _tf:   // Flow Time
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.Tf = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _rw:   // Well Bore radius
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.rw = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
               case _re:   // Drainage radius
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.re = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _PHI:  // Porosity
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.phi = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _c:    // Compressibility
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.c = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _G:    // Specific Gravity of Gas      (GAS)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.G = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _Pwfc: // Bottom Hole Flowing Pressure (COMPUTED) (GAS)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.Pwfc = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _DR:   // Damage Ratio
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.DR = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _S:    // Skin Factor
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.S = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _Ps:   // Pressure Drop due to Skin Factor
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dPs = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _Pt:   // Pressure Drop due to Turbulence  (GAS)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.dPt = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _PI:   // Productive Index
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.PI = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _RI:   // Radius of Investigation
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.RI = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _qa:   // Settled Production Rate (FLUID)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.qs = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _qAOF: // Absolute Open Flow      (GAS)
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.qAOF = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _ELP1: // Exclude Points Data Points Below Shut In 1
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.lpt1 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _EUP1: // Exclude Points Data Points Above Shut In 1
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.upt1 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _M1:   // Shut In 1 Slope
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.m1 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                  break;
                case _P1:   // Shut In 1 Intercept
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.P1 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;

              case _ELP2: // Exclude Points Data Points Below Shut In 2
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.lpt2 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _EUP2: // Exclude Points Data Points Above Shut In 2
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.upt2 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _M2:   // Shut In 2 Slope
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.m2 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _P2:   // Shut In 2 Intercept
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.P2 = cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
	        }
		  }
	    }

        iParam = _NONE;
        for (j=0; j<_PARAM2_ROWS; j++)
        {
          if (stT.sParams[i][_MNEMONIC].trim().toUpperCase().equals(PARAM2[j][0].trim()))
          {
            iParam = j;

            switch (iParam)
            {
              case _NONE:
                break;

              case _TYPE_:  // Type of DST Test
                if (stT.sParams[i][_VALUE].equals("FLUID"))
                  stData.iDSTType = dst.dstStruct._FLUID_DST;
                else
                  stData.iDSTType = dst.dstStruct._GAS_DST;
                break;
              case _CNT_R:  // Total number of Recovery Pressure Data
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.iRecovery = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CNT_G:  // Total number of Gas Rates Data
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.iGasRates = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
              case _CNT_1:  // Total number of Shut In 1 Pressure-Temperature-Time Data
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.iShut_In_1 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                  stData.iData      = stData.iData + stData.iShut_In_1;
                  stData.iShut_In_1 = 0;
                break;
              case _CNT_2:  // Total number of Shut In 2 Pressure-Temperature-Time Data
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.iShut_In_2 = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                  stData.iData      = stData.iData + stData.iShut_In_2;
                  stData.iShut_In_2 = 0;
                break;
              case _CNT_D:  // Total number of Pressure-Temperature-Time Data
                if (cmn.cmnString.isNumeric(stT.sParams[i][_VALUE]))
                  stData.iData = (int) cmn.cmnString.stringToDouble(stT.sParams[i][_VALUE]);
                break;
			}
	      }
	    }
	  }
    }

    if (stData != null)
    {
	  if (stData.iRecovery > 0)
	  {
	    stData.sRecovery = new String[stData.iRecovery][stData.jRecovery];
	    for (i=0; i<stData.iRecovery; i++)
	    {
		  for (j=0; j<stData.jRecovery; j++)
		  {
			if (j==1)
			  stData.sRecovery[i][j] = new String("");
			else
			  stData.sRecovery[i][j] = new String("0.0");
		  }
		}
	  }

	  if (stData.iGasRates > 0)
	  {
	    stData.dGasRates = new double[stData.iGasRates][stData.jGasRates];
	    for (i=0; i<stData.iGasRates; i++)
	    {
		  for (j=0; j<stData.jGasRates; j++)
		  {
			stData.dGasRates[i][j] = 0.0;
		  }
		}
	  }

	  if (stData.iShut_In_1 > 0)
	  {
	    stData.dShut_In_1 = new double[stData.iShut_In_1][3];
	    for (i=0; i<stData.iShut_In_1; i++)
	    {
		  for (j=0; j<3; j++)
		  {
			stData.dShut_In_1[i][j] = 0.0;
		  }
		}
	  }

	  if (stData.iShut_In_2 > 0)
	  {
	    stData.dShut_In_2 = new double[stData.iShut_In_2][3];
	    for (i=0; i<stData.iShut_In_2; i++)
	    {
		  for (j=0; j<3; j++)
		  {
			stData.dShut_In_2[i][j] = 0.0;
		  }
		}
	  }

	  if (stData.iData > 0)
	  {
	    stData.data = new double[stData.iData][3];
	    for (i=0; i<stData.iData; i++)
	    {
		  for (j=0; j<3; j++)
		  {
			stData.data[i][j] = 0.0;
		  }
		}
	  }
	}

    return (stData);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     DST Data Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stData  = the DST Data Structure
   * @return stData  = the DST Data Structure
   */

  public static dstStruct loadData( double                dNull,
                                    las3Struct            st,
                                    cmnLASCurveListStruct stCMN,
                                    dstStruct             stData )
  {
    int    iTool     = -1;
    String sMnemonic = "";
    String sData     = "";
    int    iRow      = -1;
    String sNull     = new String(""+dNull);

    int    iCnt_R    = 0;
    int    iCnt_G    = 0;
    int    iCnt_1    = 0;
    int    iCnt_2    = 0;
    int    iCnt_D    = 0;

    String sType     = "";     // Type of Data
    int    id        = -1;     // Period
    double dTime     = 0.0;    // Time
    double dPres     = 0.0;    // Pressure
    double dTemp     = 0.0;    // Temperature
    double dRate     = 0.0;    // Gas Rate
    double dChoke    = 0.0;    // Choke
    String sLen      = "0.0";  // Length
    String sVol      = "0.0";  // Volume
    String sDesc     = "";     // Description

    if ((st != null) && (stData != null) && (stCMN != null))
    {
      if (stCMN.iCount > 0)
      {
        if (st.iRows > 0)
        {
          for (int i=0; i<st.iRows; i++)
          {
            for (int j=0; j<st.iColumns; j++)
            {
              iTool = -1;
              for (int k=0; k<stCMN.iCount; k++)
              {
                if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                        stCMN.stItem[k].sMnemonic))
                {
                  iTool = k;
                }
              }

              if (iTool > -1)
              {
                sMnemonic = new String( stCMN.stItem[iTool].sMnemonic );
                sData     = new String( st.sData[i][j] );
				iRow      = dst.dstStandardTools.getRow(
                              dst.dstStandardTools._C_MNEM, sMnemonic );

                switch (iRow)
                {
                  case dst.dstStandardTools._T_TYPE:  // Type- Summary, Recovery, etc
                    if (!sData.equals(sNull))
                      sType = new String(sData);
                    break;
                  case dst.dstStandardTools._T_ID:    // Type of Data
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        id = (int) cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_TIME:  // Time
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dTime = cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_PRES:  // Pressure
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dPres = cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_TEMP:  // Temperature
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dTemp = cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_RATE:  // Rate
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dRate = cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_CHOKE: // Choke
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dChoke = cmn.cmnString.stringToDouble(sData);
					}
                    break;
                  case dst.dstStandardTools._T_LEN:   // Length
                    if (!sData.equals(sNull))
                      sLen = new String(sData);
                    break;
                  case dst.dstStandardTools._T_VOL:   // Volume
                    if (!sData.equals(sNull))
                      sVol = new String(sData);
                    break;
                  case dst.dstStandardTools._T_DESC:  // Description
                    if (!sData.equals(sNull))
                      sDesc = new String(sData);
                    break;
				}
			  }
		    }

            // Load Summary Pressure Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[0]))
            {
			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_INITIAL_HYDRO].trim()))
			  {
				stData.dSummary[_INITIAL_HYDRO][_TIME]        = dTime;
				stData.dSummary[_INITIAL_HYDRO][_PRESSURE]    = dPres;
				stData.dSummary[_INITIAL_HYDRO][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_FLOW_1].trim()))
			  {
				stData.dSummary[_FLOW_1][_TIME]        = dTime;
				stData.dSummary[_FLOW_1][_PRESSURE]    = dPres;
				stData.dSummary[_FLOW_1][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_SHUT_IN_1].trim()))
			  {
				stData.dSummary[_SHUT_IN_1][_TIME]        = dTime;
				stData.dSummary[_SHUT_IN_1][_PRESSURE]    = dPres;
				stData.dSummary[_SHUT_IN_1][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_END_SHUT_IN_1].trim()))
			  {
				stData.dSummary[_END_SHUT_IN_1][_TIME]        = dTime;
				stData.dSummary[_END_SHUT_IN_1][_PRESSURE]    = dPres;
				stData.dSummary[_END_SHUT_IN_1][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_FLOW_2].trim()))
			  {
				stData.dSummary[_FLOW_2][_TIME]        = dTime;
				stData.dSummary[_FLOW_2][_PRESSURE]    = dPres;
				stData.dSummary[_FLOW_2][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_SHUT_IN_2].trim()))
			  {
				stData.dSummary[_SHUT_IN_2][_TIME]        = dTime;
				stData.dSummary[_SHUT_IN_2][_PRESSURE]    = dPres;
				stData.dSummary[_SHUT_IN_2][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_END_SHUT_IN_2].trim()))
			  {
				stData.dSummary[_END_SHUT_IN_2][_TIME]        = dTime;
				stData.dSummary[_END_SHUT_IN_2][_PRESSURE]    = dPres;
				stData.dSummary[_END_SHUT_IN_2][_TEMPERATURE] = dTemp;
			  }

			  if (sDesc.equals(dst.dstStruct.ANNOTATION[_FINAL_HYDRO].trim()))
			  {
				stData.dSummary[_FINAL_HYDRO][_TIME]        = dTime;
				stData.dSummary[_FINAL_HYDRO][_PRESSURE]    = dPres;
				stData.dSummary[_FINAL_HYDRO][_TEMPERATURE] = dTemp;
			  }
			}

            // Load Recovery Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[1]))
            {
			  if (stData.iRecovery > 0)
			  {
				if (iCnt_R < stData.iRecovery)
				{
				  stData.sRecovery[iCnt_R][0] = new String( sLen );
				  stData.sRecovery[iCnt_R][1] = new String( sDesc );
				  stData.sRecovery[iCnt_R][2] = new String( sVol );
				  iCnt_R++;
			    }
			  }
			}

            // Load Gas Rate Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[2]))
            {
			  if (stData.iGasRates > 0)
			  {
				if (iCnt_G < stData.iGasRates)
				{

				  stData.dGasRates[iCnt_G][0] = (double) id;
				  stData.dGasRates[iCnt_G][1] = dTime;
				  stData.dGasRates[iCnt_G][2] = dChoke;
				  stData.dGasRates[iCnt_G][3] = dPres;
				  stData.dGasRates[iCnt_G][4] = dRate;
				  iCnt_G++;
			    }
			  }
			}

            // Load Shut In 1 Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[3]))
            {
//			  if (stData.iShut_In_1 > 0)
			  if (stData.iData > 0)
			  {
//				if (iCnt_1 < stData.iShut_In_1)
				if (iCnt_D < stData.iData)
				{
//				  stData.dShut_In_1[iCnt_1][_TIME]        = dTime;
//				  stData.dShut_In_1[iCnt_1][_PRESSURE]    = dPres;
//				  stData.dShut_In_1[iCnt_1][_TEMPERATURE] = dTemp;
//				  iCnt_1++;
				  stData.data[iCnt_D][_TIME]        = dTime + stData.dSummary[_SHUT_IN_1][_TIME];
				  stData.data[iCnt_D][_PRESSURE]    = dPres;
				  stData.data[iCnt_D][_TEMPERATURE] = dTemp;
				  iCnt_D++;
			    }
			  }
			}

            // Load Shut In 2 Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[4]))
            {
//			  if (stData.iShut_In_2 > 0)
			  if (stData.iData > 0)
			  {
//				if (iCnt_2 < stData.iShut_In_2)
				if (iCnt_D < stData.iData)
				{
//				  stData.dShut_In_2[iCnt_2][_TIME]        = dTime;
//				  stData.dShut_In_2[iCnt_2][_PRESSURE]    = dPres;
//				  stData.dShut_In_2[iCnt_2][_TEMPERATURE] = dTemp;
//				  iCnt_2++;
				  stData.data[iCnt_D][_TIME]        = dTime + stData.dSummary[_SHUT_IN_2][_TIME];
				  stData.data[iCnt_D][_PRESSURE]    = dPres;
				  stData.data[iCnt_D][_TEMPERATURE] = dTemp;
				  iCnt_D++;
			    }
			  }
			}

            // Load Raw Data

            if (sType.equals(dst.dstStandardTools.DATA_TYPE[5]))
            {
			  if (stData.iData > 0)
			  {
				if (iCnt_D < stData.iData)
				{
				  stData.data[iCnt_D][_TIME]        = dTime;
				  stData.data[iCnt_D][_PRESSURE]    = dPres;
				  stData.data[iCnt_D][_TEMPERATURE] = dTemp;
				  iCnt_D++;
			    }
			  }
			}

            sType     = new String( "" );    // Type of Data
            id        = -1;     // Period
            dTime     = 0.0;    // Time
            dPres     = 0.0;    // Pressure
            dTemp     = 0.0;    // Temperature
            dRate     = 0.0;    // Gas Rate
            dChoke    = 0.0;    // Choke
            sLen      = new String( "0.0" ); // Length
            sVol      = new String( "0.0" ); // Volume
            sDesc     = new String( "" );    // Description
		  }
	    }
	  }
    }

    stData = dst.dstUtility.parseData( stData );

	return (stData);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     DST Data Structure.
   * @param  st    = the DST Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( dstStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._TEST;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT  = addParameters( st, stT );    // Add the Required Parameters
      stT  = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st     = the DST Data Structure
   * @param  stT    = the LAS 3 File Data Structure
   * @return stT    = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( dstStruct st, las3Struct stT )
  {
    int i,j;
    String sDate[] = { "", "" };
    int icnt = 0;

    if ((st != null) && (stT != null))
    {
      stT.iParamRows = _PARAM_ROWS + _PARAM2_ROWS; // Total number of rows
      stT.iParamCols = las3.las3Struct._TOTAL;     // Total number of columns
      stT.sParams    = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<_PARAM_ROWS; i++)
      {
        stT.sParams[icnt][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[icnt][_UNIT]        = new String( PARAM[i][1] );
        stT.sParams[icnt][_VALUE]       = new String( "" );
        stT.sParams[icnt][_DESCRIPTION] = new String( PARAM[i][3] );
        stT.sParams[icnt][_FORMAT]      = new String( PARAM[i][4] );

        switch (i)
        {
          case _ID:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sID );
            break;
          case _FORM:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sFormation );
            break;
          case _DEV:
            if (st.iDeviated == dst.dstStruct._NO)
              stT.sParams[icnt][_VALUE] = new String("NO");
            else
              stT.sParams[icnt][_VALUE] = new String("YES");
            break;
          case _STRT:
            stT.sParams[icnt][_VALUE] = new String( "" + st.depthStart );
            break;
          case _STOP:
            stT.sParams[icnt][_VALUE] = new String( "" + st.depthEnd );
            break;
          case _WHIP:
            if (st.dWhipstock > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dWhipstock );
            break;
          case _SERL:
            st.serialNo = removeChars( st.serialNo, ':');
            st.serialNo = replaceChars( st.serialNo, '.', ',');
            st.serialNo = replaceChars( st.serialNo, '\n', ' ');
            stT.sParams[icnt][_VALUE] = new String( "" + st.serialNo );
            break;
          case _READ:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sReading );
            break;
          case _PRES:
            if (st.dPressure > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dPressure );
            break;
          case _DEPT:
            if (st.depth > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.depth );
            break;
          case _CMMNT:
            st.sTestRemark = removeChars( st.sTestRemark, ':');
            st.sTestRemark = replaceChars( st.sTestRemark, '.', ',');
            st.sTestRemark = replaceChars( st.sTestRemark, '\n', ' ');
            stT.sParams[icnt][_VALUE] = new String( "" + st.sTestRemark );
            break;
          case _HOLE:
            if (st.diameter_hole > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.diameter_hole );
            break;
          case _COND:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sCondition_hole );
            break;
          case _MUD:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sMudType );
            break;
          case _MUDD:
            if (st.dMudWeight > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dMudWeight );
            break;
          case _MUDV:
            if (st.dViscosity > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dViscosity );
            break;
          case _FL:
            if (st.dWaterLoss > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dWaterLoss );
            break;
          case _RM:
            if (st.dOHM_M > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dOHM_M );
            break;
          case _MSAL:
            if (st.dSalinity > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dSalinity );
            break;
          case _CAKE:
            if (st.dFilterCake > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dFilterCake );
            break;
          case _CSHT:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sCushionType );
            break;
          case _CSHL:
            if (st.dCushionLength > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dCushionLength );
            break;
          case _CSHV:
            if (st.dCushionVolume > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dCushionVolume );
            break;
          case _GCSHT:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sGasCushionType );
            break;
          case _GCSHP:
            if (st.dGasCushionPres > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dGasCushionPres );
            break;
          case _OAPI:
            if (st.dOil_API > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dOil_API );
            break;
          case _WSAL:
            if (st.dWaterSalinity > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dWaterSalinity );
            break;
          case _CAP:
            if (st.dCapacity > 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dCapacity );
            break;
          case _STIM:
            stT.sParams[icnt][_VALUE] = new String( st.sDateStart );
            sDate = cmn.cmnString.convertDate( stT.sParams[icnt][_VALUE] );
            stT.sParams[icnt][_VALUE]  = new String( sDate[0] );
            stT.sParams[icnt][_FORMAT] = new String( sDate[1] );
            break;
          case _ETIM:
            stT.sParams[icnt][_VALUE] = new String( st.sDateEnd );
            sDate = cmn.cmnString.convertDate( stT.sParams[icnt][_VALUE] );
            stT.sParams[icnt][_VALUE]  = new String( sDate[0] );
            stT.sParams[icnt][_FORMAT] = new String( sDate[1] );
            break;
          case _TIMC:
            stT.sParams[icnt][_VALUE] = new String( st.sDateCalib );
            sDate = cmn.cmnString.convertDate( stT.sParams[icnt][_VALUE] );
            stT.sParams[icnt][_VALUE]  = new String( sDate[0] );
            stT.sParams[icnt][_FORMAT] = new String( sDate[1] );
            break;
          case _TIMON:
            stT.sParams[icnt][_VALUE] = new String( st.sTimeOnBtm );
            sDate = cmn.cmnString.convertDate( stT.sParams[icnt][_VALUE] );
            stT.sParams[icnt][_VALUE]  = new String( sDate[0] );
            stT.sParams[icnt][_FORMAT] = new String( sDate[1] );
            break;
          case _TIMOF:
            stT.sParams[icnt][_VALUE] = new String( st.sTimeOffBtm );
            sDate = cmn.cmnString.convertDate( stT.sParams[icnt][_VALUE] );
            stT.sParams[icnt][_VALUE]  = new String( sDate[0] );
            stT.sParams[icnt][_FORMAT] = new String( sDate[1] );
            break;
          case _TYPE:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sTestType );
            break;
          case _UNITa:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sUnitNo );
            break;
          case _BASE:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sCompany );
            break;
          case _ENG:
            stT.sParams[icnt][_VALUE] = new String( "" + st.sTester );
            break;
          case _RECTL:
            stT.sParams[icnt][_VALUE] = new String( "" + st.dLength );
            break;
          case _RECTV:
            stT.sParams[icnt][_VALUE] = new String( "" + st.dVolume );
            break;
          case _RECC:
            st.sComments = removeChars( st.sComments, ':');
            st.sComments = replaceChars( st.sComments, '.', ',');
            st.sComments = replaceChars( st.sComments, '\n', ' ');
            stT.sParams[icnt][_VALUE] = new String( "" + st.sComments );
            break;

          // -- Quantitative Analysis Panel

          case _m:    // Slope
            stT.sParams[icnt][_VALUE] = new String( "" + st.m2 );
            break;
          case _q:    // Flow Rate
            if (st.q != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.q );
            break;
          case _h:    // Thickness
            if (st.h != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.h );
            break;
          case _u:    // Viscosity
            if (st.mu != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.mu );
            break;
          case _Pr:   // Formation Pressure
            if (st.dPressure != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dPressure );
            break;
          case _degR: // Formation Temperature
            if (st.degF != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.degF );
            break;
          case _beta: // Formation volume Factor      (FLUID)
            if (st.beta != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.beta );
            break;
          case _Z:    // Super compressibility Factor (GAS)
            if (st.Z != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.Z );
            break;

          case _KHU:  // Transmissibility       (COMPUTED)
            if (st.KHU != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.KHU );
            break;
          case _KH:   // Permeability Thickness (COMPUTED)
            if (st.KH != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.KH);
            break;
          case _K:    // Effective Permeability (COMPUTED)
            if (st.K != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.K );
            break;

          case _Pif:  // Initial Static Reservoir Pressure
            if (st.P2 != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.P2 );
            break;
          case _Pwf:  // Bottom Hole Flowing Pressure (FFP)
            if (st.dSummary[dst.dstStruct._SHUT_IN_2][1] != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dSummary[dst.dstStruct._SHUT_IN_2][1] );
            break;
          case _tf:   // Flow Time
            if (st.Tf != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.Tf );
            break;
          case _rw:   // Well Bore radius
            if (st.rw != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.rw );
            break;
          case _re:   // Drainage radius
            if (st.re != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.re );
            break;
          case _PHI:  // Porosity
            if (st.phi != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.phi );
            break;
          case _c:    // Compressibility
            if (st.c != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.c );
            break;
          case _G:    // Specific Gravity of Gas      (GAS)
            if (st.G != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.G );
            break;

          case _Pwfc: // Bottom Hole Flowing Pressure (COMPUTED) (GAS)
            if (st.Pwfc != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.Pwfc );
            break;

          case _DR:   // Damage Ratio
            if (st.DR != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.DR );
            break;
          case _S:    // Skin Factor
            if (st.S != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.S );
            break;
          case _Ps:   // Pressure Drop due to Skin Factor
            if (st.dPs != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dPs );
            break;
          case _Pt:   // Pressure Drop due to Turbulence  (GAS)
            if (st.dPt != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.dPt );
            break;

          case _PI:   // Productive Index
            if (st.PI != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.PI );
            break;
          case _RI:   // Radius of Investigation
            if (st.RI != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.RI );
            break;

          case _qa:   // Settled Production Rate (FLUID)
            if (st.qs != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.qs );
            break;
          case _qAOF: // Absolute Open Flow      (GAS)
            if (st.qAOF != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.qAOF );
            break;

          case _ELP1: // Exclude Points Data Points Below Shut In 1
            stT.sParams[icnt][_VALUE] = new String( "" + st.lpt1 );
            break;
          case _EUP1: // Exclude Points Data Points Above Shut In 1
            stT.sParams[icnt][_VALUE] = new String( "" + st.upt1 );
            break;
          case _M1:   // Shut In 1 Slope
            stT.sParams[icnt][_VALUE] = new String( "" + st.m1 );
            break;
          case _P1:   // Shut In 1 Intercept
            if (st.P1 != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.P1 );
            break;

          case _ELP2: // Exclude Points Data Points Below Shut In 2
            stT.sParams[icnt][_VALUE] = new String( "" + st.lpt2 );
            break;
          case _EUP2: // Exclude Points Data Points Above Shut In 2
              stT.sParams[icnt][_VALUE] = new String( "" + st.upt2 );
            break;
          case _M2:   // Shut In 2 Slope
            stT.sParams[icnt][_VALUE] = new String( "" + st.m2 );
            break;
          case _P2:   // Shut In 2 Intercept
            if (st.P2 != 0.0)
              stT.sParams[icnt][_VALUE] = new String( "" + st.P2 );
            break;
        }
        icnt++;
      }

      for (i=0; i<_PARAM2_ROWS; i++)
      {
        stT.sParams[icnt][_MNEMONIC]    = new String( PARAM2[i][0] );
        stT.sParams[icnt][_DESCRIPTION] = new String( PARAM2[i][1] );
        stT.sParams[icnt][_FORMAT]      = new String( "F" );

        switch (i)
        {
          case _TYPE_:  // Type of DST Test
            switch (st.iDSTType)
            {
			  case dst.dstStruct._FLUID_DST:
                stT.sParams[icnt][_VALUE] = new String( "FLUID" );
			    break;
			  case dst.dstStruct._GAS_DST:
                stT.sParams[icnt][_VALUE] = new String( "GAS" );
			    break;
			}
			break;
          case _CNT_R: // Total number of Recovery Pressure Data" },
            stT.sParams[icnt][_VALUE] = new String( "" + st.iRecovery );
            break;
          case _CNT_G: // Total number of Gas Rates Data" },
            stT.sParams[icnt][_VALUE] = new String( "" + st.iGasRates );
            break;
          case _CNT_1: // Total number of Shut In 1 Pressure-Temperature-Time Data" },
//            stT.sParams[icnt][_VALUE] = new String( "" + st.iShut_In_1 );
            stT.sParams[icnt][_VALUE] = new String( "0" );
            break;
          case _CNT_2: // Total number of Shut In 2 Pressure-Temperature-Time Data" },
//            stT.sParams[icnt][_VALUE] = new String( "" + st.iShut_In_2 );
            stT.sParams[icnt][_VALUE] = new String( "0" );
            break;
          case _CNT_D: // Total number of Pressure-Temperature-Time Data" },
            stT.sParams[icnt][_VALUE] = new String( "" + st.iData );
            break;
        }
        icnt++;
      }

    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the DST Data Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( dstStruct  st,
                                      las3Struct stT,
                                      double     dNull )
  {
    int    i,j;
    int    iCnt        = 0;
    String sData       = "";

    stT.iCurveRows = dst.dstStandardTools.TOTAL;
    stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
    stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

    for (i=0; i<dst.dstStandardTools.TOTAL; i++)
    {
      for (j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stT.sCurves[iCnt][j] = new String("");
      }

      stT.sCurves[iCnt][_MNEMONIC] = new String(
          dst.dstStandardTools.CURVES[i][1]);

      stT.sCurves[iCnt][_UNIT] = new String(
          dst.dstStandardTools.CURVES[i][3]);

      stT.sCurves[iCnt][_DESCRIPTION] = new String(
          dst.dstStandardTools.CURVES[i][2]);

      stT.sCurves[iCnt][_FORMAT] = new String(
          dst.dstStandardTools.CURVES[i][4]);

      iCnt++;
    }

    // Build the Data Section

    stT.iRows    = dst.dstStruct._ROWS +
                   st.iRecovery        +
                   st.iGasRates        +
                   st.iData;   // Total Number of Rows
//                   st.iShut_In_1       +
//                   st.iShut_In_2;   // Total Number of Rows
    stT.iColumns = stT.iCurveRows;  // Total Number of Columns ( iCurveRows )
    stT.sData    = new String[stT.iRows][stT.iColumns];

    // Initialize the Data Column values to Null

    for (i=0; i<stT.iRows; i++)
    {
      for (j=0; j<stT.iColumns; j++)
      {
        stT.sData[i][j] = new String("" + dNull);
      }
    }

    iCnt = 0;

    if (st.iRecovery > 0)
    {
      for (j=0; j<st.iRecovery; j++)
      {
	    stT.sData[iCnt][dst.dstStandardTools._T_TYPE] = new String( dst.dstStandardTools.DATA_TYPE[1] );
	    stT.sData[iCnt][dst.dstStandardTools._T_LEN]  = new String( st.sRecovery[j][0] );
	    stT.sData[iCnt][dst.dstStandardTools._T_DESC] = new String( st.sRecovery[j][1] );
	    stT.sData[iCnt][dst.dstStandardTools._T_VOL]  = new String( st.sRecovery[j][2] );
	    iCnt++;
	  }
	}

    if (st.iGasRates > 0)
    {
      for (j=0; j<st.iGasRates; j++)
      {
	    stT.sData[iCnt][dst.dstStandardTools._T_TYPE]  = new String( dst.dstStandardTools.DATA_TYPE[2] );
	    stT.sData[iCnt][dst.dstStandardTools._T_ID]    = new String( "" + (int) st.dGasRates[j][0] );
	    stT.sData[iCnt][dst.dstStandardTools._T_TIME]  = new String( "" + st.dGasRates[j][1] );
	    stT.sData[iCnt][dst.dstStandardTools._T_CHOKE] = new String( "" + st.dGasRates[j][2] );
	    stT.sData[iCnt][dst.dstStandardTools._T_PRES]  = new String( "" + st.dGasRates[j][3] );
	    stT.sData[iCnt][dst.dstStandardTools._T_RATE]  = new String( "" + st.dGasRates[j][4] );
	    stT.sData[iCnt][dst.dstStandardTools._T_DESC]  = new String( dst.dstStandardTools.DATA_TYPE[2] );
	    iCnt++;
      }
    }

    for (j=0; j<dst.dstStruct._ROWS; j++)
    {
      stT.sData[iCnt][dst.dstStandardTools._T_TYPE] = new String( dst.dstStandardTools.DATA_TYPE[0] );
      stT.sData[iCnt][dst.dstStandardTools._T_TIME] = new String( "" + st.dSummary[j][_TIME] );
      stT.sData[iCnt][dst.dstStandardTools._T_PRES] = new String( "" + st.dSummary[j][_PRESSURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_TEMP] = new String( "" + st.dSummary[j][_TEMPERATURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_DESC] = new String( dst.dstStruct.ANNOTATION[j].trim() );
      iCnt++;
    }
/*
    for (j=0; j<st.iShut_In_1; j++)
    {
      stT.sData[iCnt][dst.dstStandardTools._T_TYPE] = new String( dst.dstStandardTools.DATA_TYPE[3] );
      stT.sData[iCnt][dst.dstStandardTools._T_TIME] = new String( "" + st.dShut_In_1[j][_TIME] );
      stT.sData[iCnt][dst.dstStandardTools._T_PRES] = new String( "" + st.dShut_In_1[j][_PRESSURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_TEMP] = new String( "" + st.dShut_In_1[j][_TEMPERATURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_DESC] = new String( dst.dstStandardTools.DATA_TYPE[3] );
      iCnt++;
    }

    for (j=0; j<st.iShut_In_2; j++)
    {
      stT.sData[iCnt][dst.dstStandardTools._T_TYPE] = new String( dst.dstStandardTools.DATA_TYPE[4] );
      stT.sData[iCnt][dst.dstStandardTools._T_TIME] = new String( "" + st.dShut_In_2[j][_TIME] );
      stT.sData[iCnt][dst.dstStandardTools._T_PRES] = new String( "" + st.dShut_In_2[j][_PRESSURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_TEMP] = new String( "" + st.dShut_In_2[j][_TEMPERATURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_DESC] = new String( dst.dstStandardTools.DATA_TYPE[4] );
      iCnt++;
    }
*/
    for (j=0; j<st.iData; j++)
    {
      stT.sData[iCnt][dst.dstStandardTools._T_TYPE] = new String( dst.dstStandardTools.DATA_TYPE[5] );
      stT.sData[iCnt][dst.dstStandardTools._T_TIME] = new String( "" + st.data[j][_TIME] );
      stT.sData[iCnt][dst.dstStandardTools._T_PRES] = new String( "" + st.data[j][_PRESSURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_TEMP] = new String( "" + st.data[j][_TEMPERATURE] );
      stT.sData[iCnt][dst.dstStandardTools._T_DESC] = new String( dst.dstStandardTools.DATA_TYPE[5] );
      iCnt++;
    }

    return (stT);
  }

  /** Method removeChars()
   * <p> This method will remove the Quotes around the String
   * @param  str  = The string to be converted
   * @param  cd   = The character to remove from string
   * @return sNew = The text converted
   */

  public static String removeChars( String str, char cd )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      if (str.length() > 0)
      {
        c = str.toCharArray();

        for (int i=0; i<c.length; i++)
        {
          if (c[i] != cd)
            sNew = new String(sNew + c[i]);
        }
      }
    }

    return (sNew);
  }

  /** Method replaceChars()
   * <p> This method will remove the Quotes around the String
   * @param  str  = The string to be converted
   * @param  cd   = The character to remove from string
   * @param  cr   = The character to replace with
   * @return sNew = The text converted
   */

  public static String replaceChars( String str, char cd, char cr )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      if (str.length() > 0)
      {
        c = str.toCharArray();

        for (int i=0; i<c.length; i++)
        {
          if (c[i] == cd)
            sNew = new String(sNew + cr);
          else
            sNew = new String(sNew + c[i]);
        }
      }
    }

    return (sNew);
  }
}

/*
 *  @version 1.1 04/23/2012
 *  @author  John Victorine
 */
