/*
 * @las3LoadCoreData.java Version 1.1 02/05/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import rock.rockDataListStruct;
import rock.rockDataStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadCoreData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 02/05/2010
 *  @author  John R. Victorine
 */

public class las3LoadCoreData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  // ~PARAMETER Mnemonics in Core Data

  public static final int _SOURCE    = 0;
  public static final int _TYPE      = 1;
  public static final int _RECOVER   = 2;
  public static final int _TOP       = 3;
  public static final int _BASE      = 4;
  public static final int _AMOUNT    = 5;
  public static final int _FORMATION = 6;
  public static final int _DIAMETER  = 7;
  public static final int _COMPANY   = 8;
  public static final int _ANALYZED  = 9;

  public static final int _PARAM_ROWS  = 10;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description                        Variations
    { "C_SRS",  "Core Source",                   "", "", "" }, //   {S}
    { "C_TY",   "Core Type",                     "", "", "" }, //   {S}
    { "C_DT",   "Recovery Date (Date Core Cut)", "", "", "" }, //   {D}
    { "C_TP",   "Core Top Depth",                "", "", "" }, //   {F}
    { "C_BS",   "Core Base Depth",               "", "", "" }, //   {F}
    { "C_RC",   "Recovered Amount (Length)",     "", "", "" }, //   {F}
    { "C_FM",   "Primary Formation Cored",       "", "", "" }, //   {S}
    { "C_DI",   "Core Diameter",                 "", "", "" }, //   {F}
    { "C_AC",   "Analyzing Company",             "", "", "" }, //   {S}
    { "C_AD",   "Analysis Date",                 "", "", "" }  //   {D}
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Rock Data List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP   = the LAS 3 Data Structure
   * @return st    = the Rock Data List Structure
   */

  public static rockDataListStruct getData( las3Struct stP )
  {
    int i,j,k;
    rockDataListStruct st     = new rockDataListStruct();
    int                iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _SOURCE:    // Core Source
                  st.source = new String( stP.sParams[i][_VALUE] );
                  break;
                case _TYPE:      // Core Type
                  st.sType = new String( stP.sParams[i][_VALUE] );
                  break;
                case _RECOVER:   // Recovery Date
                  st.sRecovery = new String( stP.sParams[i][_VALUE] );
                  break;
                case _TOP:       // Core Top Depth
                  st.depthStart = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _BASE:      // Core Base Depth
                  st.depthEnd = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _AMOUNT:    // Recovered Amount
                  st.dLength = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _FORMATION: // Primary Formation
                  st.sFormation = new String( stP.sParams[i][_VALUE] );
                  break;
                case _DIAMETER:  // Core Diameter
                  st.diameter = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _COMPANY:   // Analyzing Company
                  st.sCompany = new String( stP.sParams[i][_VALUE] );
                  break;
                case _ANALYZED:  // Analysis Date
                  st.sAnalyzed = new String( stP.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the Rock Data List
   *     Structure.
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stRock  = the Rock Data List Structure
   * @return stRock2 = the Rock Data List Structure
   */

  public static rockDataListStruct loadData( double dNull,
                                             las3Struct st,
                                             cmnLASCurveListStruct stCMN,
                                             rockDataListStruct    stRock )
  {
    double depthStart  = 0.0;
    double depthEnd    = 0.0;

    if ((st != null) && (stRock != null) && (stCMN != null))
    {
      stRock = findCurves( stCMN, stRock );
      stRock = addCoreData( dNull, st, stRock );

      stRock = rock.rockDataUtility.computeDepthRange(stRock);
      stRock = rock.rockDataUtility.computeBulkDensity(stRock);
      stRock = rock.rockDataUtility.bubbleSort(stRock);
      stRock = rock.rockDataUtility.addBaseDepth( stRock );
      stRock = rock.rockDataUtility.addPorosity( stRock );
    }

    return (stRock);
  }

  /** Method findCurves()
   * <p> This method will identify the Curves and the Core Data Variables.
   * @param  stCurves = The LAS Curve Data List Structure
   * @param  st       = the Rock Data List Structure
   * @return st       = the Rock Data List Structure
   */

  public static rockDataListStruct findCurves( cmnLASCurveListStruct stCurves,
                                               rockDataListStruct st )
  {
    if ((st != null) && (stCurves != null))
    {
      // -- Depth information

      st._TOP      = -1; // Top of Rock Depth
      st._BASE     = -1; // Base of Rock Depth
      st._CORR     = -1; // Depth Correction

      // -- Descriptive fields to define rock

      st._STU      = -1; // Stratigraphic Unit
      st._STN      = -1; // Stratigraphic Name
      st._ENV      = -1; // Environment
      st._LITHO    = -1; // Lithofacies

      // -- Porosity Data

      st._PCORE    = -1; // Porosity whole routine (PU)
      st._PPLUG    = -1; // Porosity Plug routine (PU)
      st._P800     = -1; // Porosity Plug 800 PSI (PU)
      st._PINSI    = -1; // Porosity Plug Insitu (PU)
      st._PEFF     = -1; // Effective Rock Porosity (PU)

      // -- Permeability Data

      st._KMAX     = -1; // Permeability Whole Max (md)
      st._K90      = -1; // Permeability Whole 90 (md)
      st._KVRT     = -1; // Permeability Whole Vertical (md)
      st._KPLG     = -1; // Permeability Plug Routine (md)
      st._KKL      = -1; // Permeability Plug Klinkenberg Routine (md)
      st._KINSI    = -1; // Permeability Plug Insitu (md)
      st._KKLIN    = -1; // Permeability Plug Klinkenberg Insitu (md)
      st._KPVRT    = -1; // Permeability Plug Vertical (md)

      // -- Saturation Data

      st._SOIL     = -1; // Saturation Oil (%)
      st._SW       = -1; // Saturation Water (%)

      // -- Density Data

      st._GMCC     = -1; // Grain Density (gm/cc)
      st._RHOD     = -1; // Density of Rock Dry (gm/cc)
      st._RHOW     = -1; // Density of Rock Wet (gm/cc)

      // -- Archie Constants

      st._MAMB     = -1; // Archie Cementation (M) Ambient
      st._MINSI    = -1; // Archie Cementation (M) Insitu
      st._NAMB     = -1; // Archie Saturation (N) Ambient
      st._NINSI    = -1; // Archie Saturation (N) Insitu

      // -- misc data

      st._LTHCD    = -1; // Lithofacies Code
      st._FRACTURE = -1; // Fractures

      // -- Measured Gamma Ray Data

      st._GR       = -1; // Gamma Ray Value
      st._CGR      = -1; // Gamma Ray minus Uranium Value

      st._TH       = -1; // Thorium Values
      st._U        = -1; // Uranium Values
      st._K        = -1; // Potassium Values

      if (stCurves != null)
      {
        for (int i=0; i<stCurves.iCount; i++)
        {
          // -- Depth information

          // Get Top of Rock Depth Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._TOP][6]))
          {
            if (st._TOP == -1) st._TOP = stCurves.stItem[i].iCurveNumber;
          }

          // Get Base of Rock Depth Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._BASE][6]))
          {
            if (st._BASE == -1) st._BASE = stCurves.stItem[i].iCurveNumber;
          }

          // Get Depth Correction Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._CORR][6]))
          {
            if (st._CORR == -1) st._CORR = stCurves.stItem[i].iCurveNumber;
          }

          // -- Descriptive fields to define rock

          // Get Stratigraphic Unit Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._STU][6]))
          {
            if (st._STU == -1) st._STU = stCurves.stItem[i].iCurveNumber;
          }

          // Get Stratigraphic Name Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._STN][6]))
          {
            if (st._STN == -1) st._STN = stCurves.stItem[i].iCurveNumber;
          }

          // Get Environment Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._ENV][6]))
          {
            if (st._ENV == -1) st._ENV = stCurves.stItem[i].iCurveNumber;
          }

          // Get Lithofacies Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LITHO][6]))
          {
            if (st._LITHO == -1) st._LITHO = stCurves.stItem[i].iCurveNumber;
          }

          // -- Porosity Data

          // Get Porosity whole routine (PU) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._PCORE][6]))
          {
            if (st._PCORE == -1)
            {
              st._PCORE = stCurves.stItem[i].iCurveNumber;
//              st.dNPHI  = st.dPCORE;
            }
          }

          // Get Porosity Plug routine (PU) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._PPLUG][6]))
          {
            if (st._PPLUG == -1)
            {
              st._PPLUG = stCurves.stItem[i].iCurveNumber;
//              st.dNPHI  = st._PPLUG;
            }
          }

          // Get Porosity Plug 800 PSI (PU) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._P800][6]))
          {
            if (st._P800 == -1) st._P800 = stCurves.stItem[i].iCurveNumber;
          }

          // Get Porosity Plug Insitu (PU) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._PINSI][6]))
          {
            if (st._PINSI == -1) st._PINSI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Effective Rock Porosity (PU) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._PEFF][6]))
          {
            if (st._PEFF == -1) st._PEFF = stCurves.stItem[i].iCurveNumber;
          }

          // -- Permeability Data

          // Get Permeability Whole Max (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KMAX][6]))
          {
            if (st._KMAX == -1) st._KMAX = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Whole 90 (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._K90][6]))
          {
            if (st._K90 == -1) st._K90 = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Whole Vertical (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KVRT][6]))
          {
            if (st._KVRT == -1) st._KVRT = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Plug Routine (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KPLG][6]))
          {
            if (st._KPLG == -1) st._KPLG = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Plug Klinkenberg Routine (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KKL][6]))
          {
            if (st._KKL == -1) st._KKL = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Plug Insitu (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KINSI][6]))
          {
            if (st._KINSI == -1) st._KINSI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Plug Klinkenberg Insitu (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KKLIN][6]))
          {
            if (st._KKLIN == -1) st._KKLIN = stCurves.stItem[i].iCurveNumber;
          }

          // Get Permeability Plug Vertical (md) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._KPVRT][6]))
          {
            if (st._KPVRT == -1) st._KPVRT = stCurves.stItem[i].iCurveNumber;
          }

          // -- Saturation Data

          // Get Saturation Oil (%) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._SOIL][6]))
          {
            if (st._SOIL == -1) st._SOIL = stCurves.stItem[i].iCurveNumber;
          }

          // Get Saturation Water (%) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._SW][6]))
          {
            if (st._SW == -1) st._SW = stCurves.stItem[i].iCurveNumber;
          }

          // -- Density Data

          // Get Grain Density (gm/cc) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._GMCC][6]))
          {
            if (st._GMCC == -1) st._GMCC = stCurves.stItem[i].iCurveNumber;
          }

          // Get Density of Rock Dry (gm/cc) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._RHOD][6]))
          {
            if (st._RHOD == -1) st._RHOD = stCurves.stItem[i].iCurveNumber;
          }

          // Get Density of Rock Wet (gm/cc) Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._RHOW][6]))
          {
            if (st._RHOW == -1) st._RHOW = stCurves.stItem[i].iCurveNumber;
          }

          // -- Archie Constants

          // Get Archie Cementation (M) Ambient Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._MAMB][6]))
          {
            if (st._MAMB == -1) st._MAMB = stCurves.stItem[i].iCurveNumber;
          }

          // Get Archie Cementation (M) Insitu Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._MINSI][6]))
          {
            if (st._MINSI == -1) st._MINSI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Archie Saturation (N) Ambient Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._NAMB][6]))
          {
            if (st._NAMB == -1) st._NAMB = stCurves.stItem[i].iCurveNumber;
          }

          // Get Archie Saturation (N) Insitu Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._NINSI][6]))
          {
            if (st._NINSI == -1) st._NINSI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Lithofacies Code Log Curve Number

          // -- misc data

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LTHCD][6]))
          {
            if (st._LTHCD == -1) st._LTHCD = stCurves.stItem[i].iCurveNumber;
          }

          // Get Fractures Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._FRACTURE][6]))
          {
            if (st._FRACTURE == -1) st._FRACTURE = stCurves.stItem[i].iCurveNumber;
          }

          // -- Measured Gamma Ray Data

          // Get Gamma Ray Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._GR][6]))
          {
            if (st._GR == -1) st._GR = stCurves.stItem[i].iCurveNumber;
          }

          // Get Gamma Ray minus Uranium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._CGR][6]))
          {
            if (st._CGR == -1) st._CGR = stCurves.stItem[i].iCurveNumber;
          }


          // Get Thorium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._TH][6]))
          {
            if (st._TH == -1) st._TH = stCurves.stItem[i].iCurveNumber;
          }

          // Get Uranium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._U][6]))
          {
            if (st._U == -1) st._U = stCurves.stItem[i].iCurveNumber;
          }

          // Get Potassium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._K][6]))
          {
            if (st._K == -1) st._K = stCurves.stItem[i].iCurveNumber;
          }

          // Unknown Linear Curves

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LIN_1][6]))
          {
            if (st._LIN_1 == -1)
            {
              st._LIN_1 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[0][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LIN_2][6]))
          {
            if (st._LIN_2 == -1)
            {
			  st._LIN_2 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[1][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LIN_3][6]))
          {
            if (st._LIN_3 == -1)
            {
              st._LIN_3 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[2][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LIN_4][6]))
          {
            if (st._LIN_4 == -1)
            {
              st._LIN_4 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[3][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }

          // Unknown Log Curves

          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LOG_1][6]))
          {
            if (st._LOG_1 == -1)
            {
			  st._LOG_1 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[0][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LOG_2][6]))
          {
            if (st._LOG_2 == -1)
            {
              st._LOG_2 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[1][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LOG_3][6]))
          {
            if (st._LOG_3 == -1)
            {
              st._LOG_3 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[2][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            rock.rockStandardTools.ROCK_TOOLS[rock.rockStandardTools._LOG_4][6]))
          {
            if (st._LOG_4 == -1)
            {
              st._LOG_4 = stCurves.stItem[i].iCurveNumber;
              st.sUnknown[3][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
        }
      }
    }

    return (st);
  }

  /** Method addCoreData()
   * <p> This method will add the LAS 3 Data Structure to
   *     the Rock Data List Structure
   * @param  dNULL  = LAS 3 Null Value
   * @param  st     = LAS 3 Data Structure
   * @param  stRock = Rock Data List Structure
   * @return stRock = Rock Data List Structure
   */

  public static rockDataListStruct addCoreData( double             dNULL,
                                                las3Struct         st,
                                                rockDataListStruct stRock )
  {
    double data = 0.0;
    String str  = "";
    String sKEY = new String( cmn.cmnString.UniqueName() );

    if ((stRock != null) && (st !=null))
    {
      stRock.iCount = st.iRows;  // Number of Rows
      stRock.stItem = new rockDataStruct[st.iRows];

      for (int i=0; i<st.iRows; i++)
      {
        stRock.stItem[i]        = new rockDataStruct();
        stRock.stItem[i].sKEY   = new String( sKEY + "_" + i );

        // Descriptive fields to define rock

        stRock.stItem[i].sUNIT  = "";    // Stratigraphic Unit
        stRock.stItem[i].sNAME  = "";    // Stratigraphic Name
        stRock.stItem[i].sENV   = "";    // DEPOSITIONAL_ENVIRONMENT
        stRock.stItem[i].sLITHO = "";    // LITHOFACIES

        // Depth information

        stRock.stItem[i].dTOP   = dNULL;   // Depth Top of Rock (ft)
        stRock.stItem[i].dBASE  = dNULL;   // Depth Base of Rock (ft)
        stRock.stItem[i].dCORR  = 0.0;   // Correction added to depth (ft)

        // Porosity Data

        stRock.stItem[i].dPCORE = dNULL;   // PHI whole routine (PU)
        stRock.stItem[i].dPPLUG = dNULL;   // PHI Plug routine (PU)
        stRock.stItem[i].dP800  = dNULL;   // PHI Plug 800 PSI (PU)
        stRock.stItem[i].dPINSI = dNULL;   // PHI Plug Insitu (PU)
        stRock.stItem[i].dPEFF  = dNULL;   // Effective Rock PHI (PU)

        // Permeability Data

        stRock.stItem[i].dKMAX  = dNULL;   // K Whole Max (md)
        stRock.stItem[i].dK90   = dNULL;   // K Whole 90 (md)
        stRock.stItem[i].dKVRT  = dNULL;   // K Whole Vertical (md)
        stRock.stItem[i].dKPLG  = dNULL;   // K Plug Routine (md)
        stRock.stItem[i].dKKL   = dNULL;   // K Plug Klinkenberg Routine (md)
        stRock.stItem[i].dKINSI = dNULL;   // K Plug Insitu (md)
        stRock.stItem[i].dKKLIN = dNULL;   // K Plug Klinkenberg Insitu (md)
        stRock.stItem[i].dKPVRT = dNULL;   // K Plug Vertical (md)

        // Saturation Data

        stRock.stItem[i].dSOIL  = dNULL;   // Saturation Oil (%)
        stRock.stItem[i].dSW    = dNULL;   // Saturation Water (%)

        // Density Data

        stRock.stItem[i].dGMCC  = dNULL;   // Grain Density (gm/cc)
        stRock.stItem[i].dRHOD  = dNULL;   // Density of Rock Dry (gm/cc)
        stRock.stItem[i].dRHOW  = dNULL;   // Density of Rock Wet (gm/cc)

        // Archie Constants

        stRock.stItem[i].dMAMB  = dNULL;   // Archie Cementation (M) Ambient
        stRock.stItem[i].dMINSI = dNULL;   // Archie Cementation (M) Insitu
        stRock.stItem[i].dNAMB  = dNULL;   // Archie Saturation (N) Ambient
        stRock.stItem[i].dNINSI = dNULL;   // Archie Saturation (N) Insitu

        // Measured Gamma Ray Data

        stRock.stItem[i].dGR    = dNULL;   // Gamma Ray Value
        stRock.stItem[i].dCGR   = dNULL;   // Gamma Ray minus Uranimum Value

        stRock.stItem[i].dTh    = dNULL;   // Thorium Values
        stRock.stItem[i].dU     = dNULL;   // Uranium Values
        stRock.stItem[i].dK     = dNULL;   // Potassium Values

        stRock.stItem[i].dLIN_1 = dNULL;   // Unknown Linear Curve 1 Values
        stRock.stItem[i].dLIN_2 = dNULL;   // Unknown Linear Curve 2 Values
        stRock.stItem[i].dLIN_3 = dNULL;   // Unknown Linear Curve 3 Values
        stRock.stItem[i].dLIN_4 = dNULL;   // Unknown Linear Curve 4 Values

        stRock.stItem[i].dLOG_1 = dNULL;   // Unknown Log Curve 1 Values
        stRock.stItem[i].dLOG_2 = dNULL;   // Unknown Log Curve 2 Values
        stRock.stItem[i].dLOG_3 = dNULL;   // Unknown Log Curve 3 Values
        stRock.stItem[i].dLOG_4 = dNULL;   // Unknown Log Curve 4 Values

        // misc data

        stRock.stItem[i].iLITH_CD  = 0;  // Lithofacies Code
        stRock.stItem[i].sFracture = ""; // Fractures

        for (int j=0; j<st.iColumns; j++)
        {
          data = dNULL;
          str  = new String( "" + dNULL );

          //if (cmn.cmnString.isNumeric( st.sData[i][j] ))
            data = stringToDouble( st.sData[i][j].trim(), dNULL );

          if (st.sData[i][j].trim().equals(str))
            str = new String( "" );
          else
            str = new String( st.sData[i][j].trim() );

          // -- Depth information

          if (stRock._TOP    == j) // Top of Rock Depth
            stRock.stItem[i].dTOP   = data;
          if (stRock._BASE   == j) // Base of Rock Depth
            stRock.stItem[i].dBASE  = data;
          if (stRock._CORR   == j) // Depth Correction
            stRock.stItem[i].dCORR  = data;

          // -- Descriptive fields to define rock

          if (stRock._STU    == j) // Stratigraphic Unit
            stRock.stItem[i].sUNIT  = new String( str );
          if (stRock._STN    == j) // Stratigraphic Name
            stRock.stItem[i].sNAME  = new String( str );
          if (stRock._ENV    == j) // Environment
            stRock.stItem[i].sENV   = new String( str );
          if (stRock._LITHO  == j) // Lithofacies
            stRock.stItem[i].sLITHO = new String( str );

          // -- Porosity Data

          if (stRock._PCORE  == j) // Porosity whole routine (PU)
            stRock.stItem[i].dPCORE = data;
          if (stRock._PPLUG  == j) // Porosity Plug routine (PU)
            stRock.stItem[i].dPPLUG = data;
          if (stRock._P800   == j) // Porosity Plug 800 PSI (PU)
            stRock.stItem[i].dP800  = data;
          if (stRock._PINSI  == j) // Porosity Plug Insitu (PU)
            stRock.stItem[i].dPINSI = data;
          if (stRock._PEFF   == j) // Effective Rock Porosity (PU)
            stRock.stItem[i].dPEFF  = data;

          // -- Permeability Data

          if (stRock._KMAX   == j) // Permeability Whole Max (md)
            stRock.stItem[i].dKMAX  = data;
          if (stRock._K90    == j) // Permeability Whole 90 (md)
            stRock.stItem[i].dK90   = data;
          if (stRock._KVRT   == j) // Permeability Whole Vertical (md)
            stRock.stItem[i].dKVRT  = data;
          if (stRock._KPLG   == j) // Permeability Plug Routine (md)
            stRock.stItem[i].dKPLG  = data;
          if (stRock._KKL    == j) // Permeability Plug Klinkenberg Routine (md)
            stRock.stItem[i].dKKL   = data;
          if (stRock._KINSI  == j) // Permeability Plug Insitu (md)
            stRock.stItem[i].dKINSI = data;
          if (stRock._KKLIN  == j) // Permeability Plug Klinkenberg Insitu (md)
            stRock.stItem[i].dKKLIN = data;
          if (stRock._KPVRT  == j) // Permeability Plug Vertical (md)
            stRock.stItem[i].dKPVRT = data;

          // -- Saturation Data

          if (stRock._SOIL   == j) // Saturation Oil (%)
            stRock.stItem[i].dSOIL  = data;
          if (stRock._SW     == j) // Saturation Water (%)
            stRock.stItem[i].dSW    = data;

          // -- Density Data

          if (stRock._GMCC   == j) // Grain Density (gm/cc)
            stRock.stItem[i].dGMCC  = data;
          if (stRock._RHOD   == j) // Density of Rock Dry (gm/cc)
            stRock.stItem[i].dRHOD  = data;
          if (stRock._RHOW   == j) // Density of Rock Wet (gm/cc)
            stRock.stItem[i].dRHOW  = data;

          // -- Archie Constants

          if (stRock._MAMB   == j) // Archie Cementation (M) Ambient
            stRock.stItem[i].dMAMB  = data;
          if (stRock._MINSI  == j) // Archie Cementation (M) Insitu
            stRock.stItem[i].dMINSI = data;
          if (stRock._NAMB   == j) // Archie Saturation (N) Ambient
            stRock.stItem[i].dNAMB  = data;
          if (stRock._NINSI  == j) // Archie Saturation (N) Insitu
            stRock.stItem[i].dNINSI = data;

          // -- misc data

          if (stRock._LTHCD  == j) // Lithofacies Code
            stRock.stItem[i].iLITH_CD  = (int) data;
          if (stRock._FRACTURE == j) // Fractures
            stRock.stItem[i].sFracture = new String( str );

          // -- Measured Gamma Ray Data

          if (stRock._GR     == j) // Gamma Ray Value
            stRock.stItem[i].dGR    = data;
          if (stRock._CGR    == j) // Gamma Ray minus Uranium Value
            stRock.stItem[i].dCGR   = data;

          if (stRock._TH     == j) // Thorium Values
            stRock.stItem[i].dTh    = data;
          if (stRock._U      == j) // Uranium Values
            stRock.stItem[i].dU     = data;
          if (stRock._K      == j) // Potassium Values
            stRock.stItem[i].dK     = data;

          if (stRock._LIN_1  == j) // Unknown Linear Curve 1 Values
            stRock.stItem[i].dLIN_1 = data;
          if (stRock._LIN_2  == j) // Unknown Linear Curve 2 Values
            stRock.stItem[i].dLIN_2 = data;
          if (stRock._LIN_3  == j) // Unknown Linear Curve 3 Values
            stRock.stItem[i].dLIN_3 = data;
          if (stRock._LIN_4  == j) // Unknown Linear Curve 4 Values
            stRock.stItem[i].dLIN_4 = data;

          if (stRock._LOG_1  == j) // Unknown Log Curve 1 Values
            stRock.stItem[i].dLOG_1 = data;
          if (stRock._LOG_2  == j) // Unknown Log Curve 2 Values
            stRock.stItem[i].dLOG_2 = data;
          if (stRock._LOG_3  == j) // Unknown Log Curve 3 Values
            stRock.stItem[i].dLOG_3 = data;
          if (stRock._LOG_4  == j) // Unknown Log Curve 4 Values
            stRock.stItem[i].dLOG_4 = data;
        }
      }
    }

    return (stRock);
  }

  /** Method mergeCoreData()
   * <p> This method will merge a Rock Data List Structure with the new Rock
   *     Data List Structure
   * @param  st     = Rock Data List Structure
   * @param  stRock = Merged Rock Data List Structure
   * @return stRock = Merged Rock Data List Structure
   */

  public static rockDataListStruct mergeCoreData( rockDataListStruct st,
                                                  rockDataListStruct stRock )
  {
    int i,j,k;
    int iStart = 0;

    if ((st != null) && (stRock != null))
    {
      for (k=0; k<rock.rockStandardTools.TOTAL_DATA; k++)
      {
        switch (k)
        {
          case rock.rockStandardTools._TOP:
            if (st._TOP > -1) stRock._TOP = st._TOP;
            break;
          case rock.rockStandardTools._BASE:
            if (st._BASE > -1) stRock._BASE = st._BASE;
            break;
          case rock.rockStandardTools._CORR:
            if (st._CORR > -1) stRock._CORR = st._CORR;
            break;

          case rock.rockStandardTools._STU:
            if (st._STU > -1) stRock._STU = st._STU;
            break;
          case rock.rockStandardTools._STN:
            if (st._STN > -1) stRock._STN = st._STN;
            break;
          case rock.rockStandardTools._ENV:
            if (st._ENV > -1) stRock._ENV = st._ENV;
            break;
          case rock.rockStandardTools._LITHO:
            if (st._LITHO > -1) stRock._LITHO = st._LITHO;
            break;

          case rock.rockStandardTools._PCORE:
            if (st._PCORE > -1) stRock._PCORE = st._PCORE;
            break;
          case rock.rockStandardTools._PPLUG:
            if (st._PPLUG > -1) stRock._PPLUG = st._PPLUG;
            break;
          case rock.rockStandardTools._P800:
            if (st._P800 > -1) stRock._P800 = st._P800;
            break;
          case rock.rockStandardTools._PINSI:
            if (st._PINSI > -1) stRock._PINSI = st._PINSI;
            break;
          case rock.rockStandardTools._PEFF:
            if (st._PEFF > -1) stRock._PEFF = st._PEFF;
            break;

          case rock.rockStandardTools._KMAX:
            if (st._KMAX > -1) stRock._KMAX = st._KMAX;
            break;
          case rock.rockStandardTools._K90:
            if (st._K90 > -1) stRock._K90 = st._K90;
            break;
          case rock.rockStandardTools._KVRT:
            if (st._KVRT > -1) stRock._KVRT = st._KVRT;
            break;
          case rock.rockStandardTools._KPLG:
            if (st._KPLG > -1) stRock._KPLG = st._KPLG;
            break;
          case rock.rockStandardTools._KKL:
            if (st._KKL > -1) stRock._KKL = st._KKL;
            break;
          case rock.rockStandardTools._KINSI:
            if (st._KINSI > -1) stRock._KINSI = st._KINSI;
            break;
          case rock.rockStandardTools._KKLIN:
            if (st._KKLIN > -1) stRock._KKLIN = st._KKLIN;
            break;
          case rock.rockStandardTools._KPVRT:
            if (st._KPVRT > -1) stRock._KPVRT = st._KPVRT;
            break;

          case rock.rockStandardTools._SOIL:
            if (st._SOIL > -1) stRock._SOIL = st._SOIL;
            break;
          case rock.rockStandardTools._SW:
            if (st._SW > -1) stRock._SW = st._SW;
            break;

          case rock.rockStandardTools._GMCC:
            if (st._GMCC > -1) stRock._GMCC = st._GMCC;
            break;
          case rock.rockStandardTools._RHOD:
            if (st._RHOD > -1) stRock._RHOD = st._RHOD;
            break;
          case rock.rockStandardTools._RHOW:
            if (st._RHOW > -1) stRock._RHOW = st._RHOW;
            break;

          case rock.rockStandardTools._MAMB:
            if (st._MAMB > -1) stRock._MAMB = st._MAMB;
            break;
          case rock.rockStandardTools._MINSI:
            if (st._MINSI > -1) stRock._MINSI = st._MINSI;
            break;
          case rock.rockStandardTools._NAMB:
            if (st._NAMB > -1) stRock._NAMB = st._NAMB;
            break;
          case rock.rockStandardTools._NINSI:
            if (st._NINSI > -1) stRock._NINSI = st._NINSI;
            break;

          case rock.rockStandardTools._GR:
            if (st._GR > -1) stRock._GR = st._GR;
            break;
          case rock.rockStandardTools._CGR:
            if (st._CGR > -1) stRock._CGR = st._CGR;
            break;

          case rock.rockStandardTools._TH:
            if (st._TH > -1) stRock._TH = st._TH;
            break;
          case rock.rockStandardTools._U:
            if (st._U > -1) stRock._U = st._U;
            break;
          case rock.rockStandardTools._K:
            if (st._K > -1) stRock._K = st._K;
            break;

          case rock.rockStandardTools._LIN_1: // Unknown Linear Curve 1 Values
            if (st._LIN_1 > -1) stRock._LIN_1 = st._LIN_1;
          case rock.rockStandardTools._LIN_2: // Unknown Linear Curve 2 Values
            if (st._LIN_2 > -1) stRock._LIN_2 = st._LIN_2;
          case rock.rockStandardTools._LIN_3: // Unknown Linear Curve 3 Values
            if (st._LIN_3 > -1) stRock._LIN_3 = st._LIN_3;
          case rock.rockStandardTools._LIN_4: // Unknown Linear Curve 4 Values
            if (st._LIN_4 > -1) stRock._LIN_4 = st._LIN_4;

          case rock.rockStandardTools._LOG_1: // Unknown Log Curve 1 Values
            if (st._LOG_1 > -1) stRock._LOG_1 = st._LOG_1;
          case rock.rockStandardTools._LOG_2: // Unknown Log Curve 2 Values
            if (st._LOG_2 > -1) stRock._LOG_2 = st._LOG_2;
          case rock.rockStandardTools._LOG_3: // Unknown Log Curve 3 Values
            if (st._LOG_3 > -1) stRock._LOG_3 = st._LOG_3;
          case rock.rockStandardTools._LOG_4: // Unknown Log Curve 4 Values
            if (st._LOG_4 > -1) stRock._LOG_4 = st._LOG_4;

          case rock.rockStandardTools._LTHCD:
            if (st._LTHCD > -1) stRock._LTHCD = st._LTHCD;
            break;
          case rock.rockStandardTools._FRACTURE:
            if (st._FRACTURE > -1) stRock._FRACTURE = st._FRACTURE;
            break;
        }
      }

      for (i=0; i<stRock.iCount; i++)
      {
        if (stRock.stItem[i].dTOP != rock.rockDataStruct.dNULL)
          iStart = i;
      }

      for (i=0; i<st.iCount; i++)
      {
        if (iStart < stRock.iCount)
        {
          stRock.stItem[iStart] = rock.rockDataUtility.copy( st.stItem[i] );
          iStart++;
        }
      }
    }

    return (stRock);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Rock Data List Structure.
   * @param  st    = the Rock Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( rockDataListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._CORE;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Rock Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( rockDataListStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case  _SOURCE:
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case  _TYPE:
            stT.sParams[i][_VALUE] = new String( st.sType );
            break;
          case  _RECOVER:
            stT.sParams[i][_VALUE]  = new String( st.sRecovery );
            sDate = cmn.cmnString.convertDate( stT.sParams[i][_VALUE] );
            stT.sParams[i][_VALUE]  = new String( sDate[0] );
            stT.sParams[i][_FORMAT] = new String( sDate[1] );
            break;
          case  _TOP:
            stT.sParams[i][_UNIT]   = new String( "F" );
            stT.sParams[i][_VALUE]  = new String( ""+st.depthStart );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _BASE:
            stT.sParams[i][_UNIT]   = new String( "F" );
            stT.sParams[i][_VALUE]  = new String( ""+st.depthEnd );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _AMOUNT:
            stT.sParams[i][_UNIT]   = new String( "F" );
            stT.sParams[i][_VALUE]  = new String( ""+st.dLength );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _FORMATION:
            stT.sParams[i][_VALUE] = new String( st.sFormation );
            break;
          case  _DIAMETER:
            stT.sParams[i][_UNIT]   = new String( "IN" );
            stT.sParams[i][_VALUE]  = new String( ""+st.diameter );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _COMPANY:
            stT.sParams[i][_VALUE] = new String( st.sCompany );
            break;
          case  _ANALYZED:
            stT.sParams[i][_VALUE] = new String( st.sAnalyzed );
            sDate = cmn.cmnString.convertDate( stT.sParams[i][_VALUE] );
            stT.sParams[i][_VALUE]  = new String( sDate[0] );
            stT.sParams[i][_FORMAT] = new String( sDate[1] );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Rock Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( rockDataListStruct st,
                                      las3Struct         stT,
                                      double             dNull )
  {
    int    i,j;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[rock.rockStandardTools.TOTAL_DATA];
      for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
        if (st.checkData(i))
          iSelected[i] = 1;
      }

      // Count the Number of Curves Populated

      for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              rock.rockStandardTools.ROCK_TOOLS[i][0]);

          stT.sCurves[iCnt][_UNIT] = new String(
              rock.rockStandardTools.ROCK_TOOLS[i][2]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              rock.rockStandardTools.ROCK_TOOLS[i][1]);

          stT.sCurves[iCnt][_FORMAT] = new String("F");
          if (((i >= rock.rockStandardTools._STU) &&
               (i <= rock.rockStandardTools._LITHO)) ||
              (i == rock.rockStandardTools._FRACTURE))
          {
            stT.sCurves[iCnt][_FORMAT] = new String("S");
          }
          else if (i == rock.rockStandardTools._LTHCD)
          {
            stT.sCurves[iCnt][_FORMAT] = new String("I");
          }

          iCnt++;
        }
      }

      // Build the Data Section

      stT.iRows    = st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Stratigraphic Units Data List Structure

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          if (((iCurves[j] >= rock.rockStandardTools._STU) &&
               (iCurves[j] <= rock.rockStandardTools._LITHO)) ||
              (iCurves[j] == rock.rockStandardTools._FRACTURE))
          {
            switch (iCurves[j])
            {
              case rock.rockStandardTools._STU:
                stT.sData[i][j] = new String( st.stItem[i].sUNIT );
                break;
              case rock.rockStandardTools._STN:
                stT.sData[i][j] = new String( st.stItem[i].sNAME );
                break;
              case rock.rockStandardTools._ENV:
                stT.sData[i][j] = new String( st.stItem[i].sENV );
                break;
              case rock.rockStandardTools._LITHO:
                stT.sData[i][j] = new String( st.stItem[i].sLITHO );
                break;

              case rock.rockStandardTools._FRACTURE:
                stT.sData[i][j] = new String( st.stItem[i].sFracture );
                break;
            }
          }
          else
            stT.sData[i][j] = new String( "" + st.getData( iCurves[j], i ) );
        }
      }
    }

    return (stT);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back
   * @param  str    = the string to be converted
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str )
  {
    double dValue = 0.0;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back, but uses
   *     the Null Value if it is not a valid number
   * @param  str    = the string to be converted
   * @param  dNull  = the LAS File Null Value
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str, double dNull )
  {
    double dValue = dNull;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

}

/*
 *  @version 1.1 02/05/2010
 *  @author  John Victorine
 */
